(**************************************************************************
QuArK -- Quake Army Knife -- 3D game editor
Copyright (C) 1996-99 Armin Rigo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Contact the author Armin Rigo by e-mail: arigo@planetquake.com
or by mail: Armin Rigo, La Cure, 1854 Leysin, Switzerland.
See also http://www.planetquake.com/quark
**************************************************************************)

unit EdOpenGL;

interface

uses Windows, SysUtils, Classes, GL1, QkObjects, Ed3DFX, qmath, PyMath, PyMath3D;

const
 kDistFarToShort = MinW/65536; { 0.0009765625 }

type
 TGLSceneObject = class(TSceneObject)
 private
   DestWnd: HWnd;
   GLDC: HDC;
   RC: HGLRC;
   ScreenX, ScreenY: Integer;
   CurrentAlpha: LongInt;
   RenderingTextureBuffer: TMemoryStream;
   DoubleBuffered: Boolean;
   procedure LoadCurrentTexture(Tex: PTexture3);
 protected
   function GetInfo(var PW: TPaletteWarning; var VertexSize: Integer) : TBuildMode; override;
   procedure stScalePoly(Texture: PTexture3; var ScaleS, ScaleT: Reel); override;
   procedure stScaleModel(Skin: PTexture3; var ScaleS, ScaleT: Reel); override;
   procedure WriteVertex(PV: PChar; Source: Pointer; const ns,nt: Single; HiRes: Boolean); override;
   procedure RenderPList(PList: PSurfaces; TransparentFaces: Boolean);
   procedure RenderTransparentGL(Transparent: Boolean);
 public
   destructor Destroy; override;
   procedure Init(Wnd: HWnd; nCoord: TCoordinates; const LibName: String;
    var FullScreen, AllowsGDI: Boolean; FOG_DENSITY: Single;
    FOG_COLOR, FrameColor: TColorRef); override;
   procedure SetViewRect(SX, SY: Integer); override;
   procedure Render3DView; override;
   procedure Copy3DView(SX,SY: Integer; DC: HDC); override;
 end;

 {------------------------}

procedure Free3DEditor;

 {------------------------}

implementation

uses Quarkx, QkMapPoly, Setup;

 {------------------------}

procedure Free3DEditor;
begin
 Free3DFXEditor;
 UnloadOpenGl;
end;

procedure FreeOpenGLTexture(Tex: PTexture3);
begin
 if (Tex^.OpenGLName<>0) and Assigned(gl) then
  gl.glDeleteTextures(1, Tex^.OpenGLName);
end;

 {------------------------}

procedure Err(Pos: Integer);  { OpenGL error check }
var
 I, J: Integer;
 S: String;
begin
 S:='';
 for I:=1 to 25 do
  begin
   J:=gl.glGetError;
   if J = GL_NO_ERROR then Break;
   S:=S+' '+IntToStr(J);
  end;
 if S<>'' then
  Raise EErrorFmt(4870, [S, Pos]);
end;

function fRed(Color: TColorRef) : GLclampf;
begin
 fRed:=(Color and $FF) * (1/255.0);
end;

function fGreen(Color: TColorRef) : GLclampf;
begin
 fGreen:=((Color shr 8) and $FF) * (1/255.0);
end;

function fBlue(Color: TColorRef) : GLclampf;
begin
 fBlue:=((Color shr 16) and $FF) * (1/255.0);
end;

function fAlpha(Color: TColorRef) : GLclampf;
begin
 fAlpha:=((Color shr 24) and $FF) * (1/255.0);
end;

 {------------------------}

type
 PVertex3D = ^TVertex3D;
 TVertex3D = record
              st: array[0..1] of Single;
              xyz: vec3_t;
             end;

 {------------------------}

destructor TGLSceneObject.Destroy;
var
 I: Integer;
 NameArray, CurrentName: ^GLuint;
begin
 with TTextureManager.GetInstance do
  begin
   GetMem(NameArray, Textures.Count*SizeOf(GLuint)); try
   CurrentName:=NameArray;
   for I:=0 to Textures.Count-1 do
    with PTexture3(Textures.Objects[I])^ do
     if OpenGLName<>0 then
      begin
       CurrentName^:=OpenGLName;
       Inc(CurrentName);
       OpenGLName:=0;
      end;
   if Assigned(gl) then
    gl.glDeleteTextures((PChar(CurrentName)-PChar(NameArray)) div SizeOf(GLuint), NameArray^);
   finally FreeMem(NameArray); end;
  end;
 if (RC<>0) and Assigned(gl) then
  begin
   gl.wglMakeCurrent(0,0);
   gl.wglDeleteContext(RC);
  end;
 if GLDC<>0 then
  ReleaseDC(DestWnd, GLDC);
 inherited;
end;

procedure TGLSceneObject.Init(Wnd: HWnd; nCoord: TCoordinates; const LibName: String;
    var FullScreen, AllowsGDI: Boolean; FOG_DENSITY: Single; FOG_COLOR, FrameColor: TColorRef);
var
 pfd: TPixelFormatDescriptor;
 pfi: Integer;
 FogColor: array[0..3] of Single;
 FarDistance: Reel;
 Setup: QObject;
 Fog: Boolean;
begin
 if RC<>0 then
  begin
   if Assigned(gl) then
    begin
     gl.wglMakeCurrent(0,0);
     gl.wglDeleteContext(RC);
    end;
   RC:=0;
  end;
 if GLDC<>0 then
  begin
   ReleaseDC(DestWnd, GLDC);
   GLDC:=0;
  end;

 if not OpenGlLoaded and not ReloadOpenGl then
  Raise EErrorFmt(4868, [GetLastError]);

{$IFDEF Debug}
 if not (nCoord is TCameraCoordinates) then
  Raise InternalE('TCameraCoordinates expected');
{$ENDIF}
 FarDistance:=(nCoord as TCameraCoordinates).FarDistance;
 Coord:=nCoord;
 FullScreen:=False;
 TTextureManager.AddScene(Self, False);
 TTextureManager.GetInstance.FFreeTexture:=FreeOpenGLTexture;

 Setup:=SetupSubSet(ssGeneral, 'OpenGL');
 Fog:=Setup.Specifics.Values['Fog']<>'';
 AllowsGDI:=Setup.Specifics.Values['AllowsGDI']<>'';
 GLDC:=GetDC(Wnd);
 if Wnd<>DestWnd then
  begin
   DoubleBuffered:=Setup.Specifics.Values['DoubleBuffer']<>'';
   FillChar(pfd, SizeOf(pfd), 0);
   pfd.nSize:=SizeOf(pfd);
   pfd.nversion:=1;
   pfd.dwflags:=pfd_Support_OpenGl or pfd_Draw_To_Window;
   pfd.iPixelType:=pfd_Type_RGBA;
   if DoubleBuffered then pfd.dwflags:=pfd.dwflags or pfd_DoubleBuffer;
   if Setup.Specifics.Values['SupportsGDI']<>''  then pfd.dwflags:=pfd.dwflags or pfd_Support_GDI;
   pfd.cColorBits:=Round(Setup.GetFloatSpec('ColorBits', 0));
   if pfd.cColorBits<=0 then pfd.cColorBits:=GetDeviceCaps(GLDC, BITSPIXEL);
   pfd.cDepthBits:=Round(Setup.GetFloatSpec('DepthBits', 16));
   pfd.iLayerType:=pfd_Main_Plane;
   pfi:=ChoosePixelFormat(GLDC, @pfd);
   if not SetPixelFormat(GLDC, pfi, @pfd) then
    Raise EErrorFmt(4869, ['SetPixelFormat']);
   DestWnd:=Wnd;
  end;
 RC:=gl.wglCreateContext(GLDC);
 if RC=0 then
  Raise EErrorFmt(4869, ['wglCreateContext']);

  { set up OpenGL }
 gl.wglMakeCurrent(GLDC,RC);
 Err(0);
 FogColor[0]:=fRed(FOG_COLOR);
 FogColor[1]:=fGreen(FOG_COLOR);
 FogColor[2]:=fBlue(FOG_COLOR);
 FogColor[3]:=1;
 gl.glClearColor(FogColor[0], FogColor[1], FogColor[2], FogColor[3]);
// gl.glClearDepth(1);
 gl.glEnable(GL_DEPTH_TEST);
{gl.glDepthFunc(GL_LEQUAL);}
// gl.glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
 gl.glEdgeFlag(0);
 Err(1);

  { set up texture parameters }
(* gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
 gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
{gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
 gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_LINEAR);}
 gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
 gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
{gl.glShadeModel(GL_FLAT);} *)
 gl.glEnable(GL_TEXTURE_2D);
 Err(2);

  { set up fog }
 if Fog then
  begin
   gl.glFogi(GL_FOG_MODE, GL_EXP2);
  {gl.glFogf(GL_FOG_START, FarDistance * kDistFarToShort);
   gl.glFogf(GL_FOG_END, FarDistance);}
   gl.glFogf(GL_FOG_DENSITY, FOG_DENSITY/FarDistance * 100000);
   gl.glFogfv(GL_FOG_COLOR, FogColor);
   gl.glEnable(GL_FOG);
   Err(3);
  end; 
end;

procedure TGLSceneObject.Copy3DView(SX,SY: Integer; DC: HDC);
begin
 if DoubleBuffered then
  Windows.SwapBuffers(GLDC);
end;

procedure TGLSceneObject.SetViewRect(SX, SY: Integer);
begin
 if SX<1 then SX:=1;
 if SY<1 then SY:=1;
 ScreenX:=SX;
 ScreenY:=SY;
end;

function TGLSceneObject.GetInfo(var PW: TPaletteWarning; var VertexSize: Integer) : TBuildMode;
begin
 PW:=Nil;
 VertexSize:=SizeOf(TVertex3D);
 Result:=bmOpenGL;
end;

procedure TGLSceneObject.stScalePoly(Texture: PTexture3; var ScaleS, ScaleT: Reel);
begin
 with Texture^ do
  begin
   ScaleS:=TexW*(1/EchelleTexture);
   ScaleT:=TexH*(-1/EchelleTexture);
  end;
end;

procedure TGLSceneObject.stScaleModel(Skin: PTexture3; var ScaleS, ScaleT: Reel);
begin
 with Skin^ do
  begin
   ScaleS:=1/TexW;
   ScaleT:=1/TexH;
  end;
end;

procedure TGLSceneObject.WriteVertex(PV: PChar; Source: Pointer; const ns,nt: Single; HiRes: Boolean);
begin
 with PVertex3D(PV)^ do
  begin
   if HiRes then
    with PVect(Source)^ do
     begin
      xyz[0]:=X;
      xyz[1]:=Y;
      xyz[2]:=Z;
     end
   else
    xyz:=vec3_p(Source)^;
   st[0]:=ns;
   st[1]:=nt;
  end;
end;

procedure TGLSceneObject.RenderTransparentGL(Transparent: Boolean);
var
 PList: PSurfaces;
 Count: Integer;
 Buffer, BufEnd: ^GLuint;
 BufResident: ^GLboolean;
begin
 if not SolidColors then
  begin
   Count:=0;
   PList:=FListSurfaces;
   while Assigned(PList) do
    begin
     PList^.ok:=(Transparent in PList^.Transparent) and (PList^.Texture^.OpenGLName<>0);
     if PList^.ok then
      Inc(Count);
     PList:=PList^.Next;
    end;
   if Count>0 then
    begin
     GetMem(Buffer, Count*(SizeOf(GLuint)+SizeOf(GLboolean))); try
     BufEnd:=Buffer;
     PList:=FListSurfaces;
     while Assigned(PList) do
      begin
       if PList^.ok then
        begin
         BufEnd^:=PList^.Texture^.OpenGLName;
         Inc(BufEnd);
        end;
       PList:=PList^.Next;
      end;
     PChar(BufResident):=PChar(BufEnd);
     gl.glAreTexturesResident(Count, Buffer^, BufResident^);
     PList:=FListSurfaces;
     while Assigned(PList) do
      begin
       if PList^.ok then
        begin
         PList^.ok:=False;
         if BufResident^<>0 then
          RenderPList(PList, Transparent);
         Inc(BufResident); 
        end;
       PList:=PList^.Next;
      end;
     finally FreeMem(Buffer); end;
    end;
  end;
 PList:=FListSurfaces;
 while Assigned(PList) do
  begin
   if Transparent in PList^.Transparent then
    if SolidColors or not PList^.ok then
     RenderPList(PList, Transparent);
   PList:=PList^.Next;
  end;
end;

procedure TGLSceneObject.Render3DView;
begin
 if not Assigned(gl) then Exit;
{gl.wglMakeCurrent(DC,RC);
 Err(49);}
 gl.glViewport(0, 0, ScreenX, ScreenY);
 Err(50);
 with TCameraCoordinates(Coord) do
  begin
   gl.glMatrixMode(GL_PROJECTION);
   gl.glLoadIdentity;
   gl.gluPerspective(2*VAngleDegrees, ScreenX/ScreenY, FarDistance * kDistFarToShort, FarDistance);
   if PitchAngle<>0 then
    gl.glRotatef(PitchAngle * (180/pi), -1,0,0);
   gl.glRotatef(HorzAngle * (180/pi), 0,-1,0);
   gl.glMatrixMode(GL_MODELVIEW);
   gl.glLoadIdentity;
   gl.glRotatef(120, -1,1,1);
   with Camera do
    gl.glTranslatef(-X, -Y, -Z);
  end;
 Err(51);
 gl.glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT); { clear screen }
 Err(52);
 CurrentAlpha:=0;
 RenderingTextureBuffer:=TMemoryStream.Create; try
 RenderTransparentGL(False);
 Err(53);
 RenderTransparentGL(True);
 Err(54);
 gl.glFlush;
 Err(55);
 finally RenderingTextureBuffer.Free; end;
{gl.wglMakeCurrent(0,0);}
end;

procedure TGLSceneObject.LoadCurrentTexture(Tex: PTexture3);
var
 TexData: PChar;
 MemSize, W, H: Integer;
 Source: PChar;
 PaletteEx: array[0..255] of LongInt;
 BasePalette: Pointer;
begin
 if Tex^.OpenGLName=0 then
  begin
   GetwhForTexture(Tex^.info, W, H);
   MemSize:=W*H*4;
   if RenderingTextureBuffer.Size < MemSize then
    RenderingTextureBuffer.SetSize(MemSize);

   TexData:=RenderingTextureBuffer.Memory;
   Source:=Tex^.info.data;
   BasePalette:=Tex^.GuPalette;
   asm
    push edi
    push esi
    push ebx                  { Indexes in the palette --> RGB colors }
    mov esi, [BasePalette]
    lea edi, [PaletteEx]
    mov ecx, 255
    @Loop1:
     mov eax, [esi+4*ecx]
     bswap eax
     shr eax, 8
     mov [edi+4*ecx], eax
     dec ecx
    jns @Loop1

    mov edi, [TexData]
    mov esi, [Source]
    mov ecx, [MemSize]
    lea ebx, [PaletteEx]
    xor edx, edx
    shr ecx, 2
    @Loop:
     mov dl, [esi]
     inc esi
     mov eax, [ebx+edx*4]
     mov [edi], eax
     add edi, 4
     dec ecx
    jnz @Loop
    pop ebx
    pop esi
    pop edi
   end;
  {gl.gluBuild2DMipmaps(GL_TEXTURE_2D, 3, W, H, GL_RGBA, GL_UNSIGNED_BYTE, TexData^);}
   gl.glGenTextures(1, Tex^.OpenGLName);
   gl.glBindTexture(GL_TEXTURE_2D, Tex^.OpenGLName);
   gl.glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, W, H, 0, GL_RGBA, GL_UNSIGNED_BYTE, TexData^);
   gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
   gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
   gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
   gl.glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  end
 else
  gl.glBindTexture(GL_TEXTURE_2D, Tex^.OpenGLName);
end;

procedure TGLSceneObject.RenderPList(PList: PSurfaces; TransparentFaces: Boolean);
var
 Surf: PSurface3D;
 SurfEnd: PChar;
 PV: PVertex3D;
 NeedTex: Boolean;
 I: Integer;
begin
 NeedTex:=True;
 Surf:=PList^.Surf;
 SurfEnd:=PChar(Surf)+PList^.SurfSize;
 while Surf<SurfEnd do
  with Surf^ do
   begin
    Inc(Surf);
    if ((AlphaColor and $FF000000 = $FF000000) xor TransparentFaces)
    and Coord.PositiveHalf(Normale[0], Normale[1], Normale[2], Dist) then
     begin
      if AlphaColor<>CurrentAlpha then
       begin
        gl.glColor4f(fRed(AlphaColor), fGreen(AlphaColor), fBlue(AlphaColor), fAlpha(AlphaColor));
        CurrentAlpha:=AlphaColor;
       end;
      if NeedTex then
       begin
        LoadCurrentTexture(PList^.Texture);
        NeedTex:=False;
       end;
      PV:=PVertex3D(Surf);
      gl.glBegin(GL_POLYGON);
      for I:=1 to VertexCount do
       begin
        gl.glTexCoord2fv(PV^.st);
        gl.glVertex3fv(PV^.xyz);
        Inc(PV);
       end;
      gl.glEnd;
     end;
    Inc(PVertex3D(Surf), VertexCount);
   end;
 PList^.ok:=True;
end;

 {------------------------}

end.
