(**************************************************************************
QuArK -- Quake Army Knife -- 3D game editor
Copyright (C) 1996-99 Armin Rigo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Contact the author Armin Rigo by e-mail: arigo@planetquake.com
or by mail: Armin Rigo, La Cure, 1854 Leysin, Switzerland.
See also http://www.planetquake.com/quark
**************************************************************************)

unit Duplicator;

interface

uses Windows, SysUtils, Classes, Graphics, QkObjects, QkMapObjects,
     qmath, Python, PyObjects, Quarkx, Setup, PyMath;

type
 TDuplicator = class(TTreeMapEntity)
               private
                 FCache: PyObject;
                 function BuildImages: PyObject;
                 procedure SetCache(nCache: PyObject);
                {function CallDuplicatorMethod(const MethodName: String; args: PyObject) : PyObject;}
               public
                 class function TypeInfo: String; override;
                 property Images: PyObject read BuildImages write SetCache;
                 procedure EtatObjet(var E: TEtatObjet); override;
                 destructor Destroy; override;
                 procedure OpDansScene(Aj: TAjScene; PosRel: Integer); override;
                 procedure ChercheExtremites(var Min, Max: TVect); override;
                 procedure Dessiner; override;
                 function IsExplorerItem(Q: QObject) : TIsExplorerItem; override;
                 function GetFormName : String; override;
                 procedure ListePolyedres(Polyedres, Negatif: TQList; Flags: Integer; Brushes: Integer); override;
                 procedure ListeEntites(Entites: TQList); override;
                 procedure SauverTexte(Negatif: TQList; Texte: TStrings; Flags: Integer; HxStrings: TStrings); override;
                 procedure AddTo3DScene; override;
                 function PyGetAttr(attr: PChar) : PyObject; override;
                {function PySetAttr(attr: PChar; value: PyObject) : Boolean; override;}
               end;

 {------------------------}

implementation

uses QkFileObjects, PyMapView, QkMapPoly, Qk3D;

 {------------------------}

class function TDuplicator.TypeInfo: String;
begin
 TypeInfo:=':d';
end;

procedure TDuplicator.EtatObjet(var E: TEtatObjet);
begin
 inherited;
 E.IndexImage:=iiDuplicator;
end;

destructor TDuplicator.Destroy;
begin
 Py_XDECREF(FCache);
 inherited;
end;

procedure TDuplicator.SetCache(nCache: PyObject);
var
 I, Count: Integer;
 Q: QObject;
begin
 Py_XDECREF(FCache);
 FCache:=Nil;
 if (nCache<>Nil) and (nCache<>Py_None) then
  begin
   Count:=PyObject_Length(nCache);
   if Count<0 then Raise EError(4449);
   for I:=0 to Count-1 do
    begin
     Q:=QkObjFromPyObj(PyList_GetItem(nCache, I));
     if not (Q is TTreeMap) then Raise EError(4449);
     if TTreeMap(Q).FParent = Nil then
      TTreeMap(Q).FParent:=Self;
    end;
   Py_INCREF(nCache);
  end
 else
  nCache:=PyList_New(0);
 FCache:=nCache;
end;

(*function TDuplicator.CallDuplicatorMethod(const MethodName: String; args: PyObject) : PyObject;
var
 dupclass, dupmethod: PyObject;
begin
 Result:=Nil;
 if args=Nil then Exit;
 try
  dupclass:=CallMacro(@PythonObj, 'duplicator');
  if dupclass=Nil then Exit;
  try
   dupmethod:=PyObject_GetAttrString(dupclass, PChar(MethodName));
   if dupmethod=Nil then Exit;
   try
    Result:=PyEval_CallObject(dupmethod, args);
   finally
    Py_DECREF(dupmethod);
   end;
  finally
   Py_DECREF(dupclass);
  end;
 finally
  Py_DECREF(args);
 end;
 PythonCodeEnd;
end;*)

function TDuplicator.BuildImages;
var
 callresult: PyObject;
begin
 if FCache=Nil then
  begin
  {callresult:=CallDuplicatorMethod('buildimages', GetEmptyTuple);}
   callresult:=CallMacro(@PythonObj, 'duplicator');
   try
    Images:=callresult;
   finally Py_XDECREF(callresult); end;
  end;
 Result:=FCache;
end;

procedure TDuplicator.OpDansScene(Aj: TAjScene; PosRel: Integer);
begin
 if FCache<>Nil then
  begin
   Py_DECREF(FCache);
   FCache:=Nil;
  end;
 inherited;
end;

procedure TDuplicator.ChercheExtremites(var Min, Max: TVect);
var
 I: Integer;
begin
 inherited;
 for I:=0 to PyObject_Length(BuildImages)-1 do
  (QkObjFromPyObj(PyList_GetItem(FCache, I)) as TTreeMap).ChercheExtremites(Min, Max);
end;

procedure TDuplicator.Dessiner;
var
 Pts: TPointProj;
 X1, Y1: Integer;
 I: Integer;
 OldPen: HPen;
 C: TColor;
begin
 BuildImages;
 OldPen:=0;
 if Info.PinceauGris <> 0 then
  begin    { if color changes must be made now }
   C:=MapColors({DefaultColor}lcDuplicator);
   CouleurDessin(C);
   OldPen:=Info.PinceauNoir;
   Info.PinceauNoir:=CreatePen(ps_Solid, 0, C);
  end;
 for I:=0 to PyObject_Length(FCache)-1 do
  (QkObjFromPyObj(PyList_GetItem(FCache, I)) as TTreeMap).Dessiner;
 if OldPen<>0 then
  begin
   SelectObject(Info.DC, OldPen);
   DeleteObject(Info.PinceauNoir);
   Info.PinceauNoir:=OldPen;
  end;
 if HasOrigin then
  begin
   Pts:=CCoord.Proj(Origin);
   if not CCoord.CheckVisible(Pts) then Exit;
   if Info.PinceauSelection<>0 then
    SetROP2(Info.DC, Info.BaseR2)
   else
    if Info.ModeAff>0 then
     begin
      if Pts.OffScreen = 0 then
       begin
        SelectObject(Info.DC, Info.PinceauNoir);
        SetROP2(Info.DC, R2_CopyPen);
       end
      else
       begin
        if Info.ModeAff=2 then
         Exit;
        SelectObject(Info.DC, Info.PinceauGris);
        SetROP2(Info.DC, Info.MaskR2);
       end;
     end
    else
    {if Info.ModeAff=0 then}
      begin
       SelectObject(Info.DC, Info.PinceauNoir);
       SetROP2(Info.DC, R2_CopyPen);
      end;
   X1:=Round(Pts.x);
   Y1:=Round(Pts.y);
   MoveToEx(Info.DC, X1-4, Y1-4, Nil);
   LineTo  (Info.DC, X1+5, Y1+5);
   MoveToEx(Info.DC, X1-4, Y1+4, Nil);
   LineTo  (Info.DC, X1+5, Y1-5);
   if Info.PinceauSelection<>0 then
    begin
     SelectObject(Info.DC, Info.PinceauSelection);
     SetROP2(Info.DC, R2_CopyPen);
     Ellipse(Info.DC, X1-7, Y1-7, X1+8, Y1+7);
    end;
  end;
end;

function TDuplicator.IsExplorerItem(Q: QObject) : TIsExplorerItem;
begin
 Result:=ieResult[Q is TTreeMap];
end;

function TDuplicator.GetFormName : String;
begin
 Result:=DefaultForm+TypeInfo;
end;

procedure TDuplicator.ListePolyedres(Polyedres, Negatif: TQList; Flags: Integer; Brushes: Integer);
var
 I: Integer;
begin
 for I:=0 to PyObject_Length(BuildImages)-1 do
  (QkObjFromPyObj(PyList_GetItem(FCache, I)) as TTreeMap).ListePolyedres(Polyedres, Negatif, Flags or soDirectDup, Brushes);
end;

procedure TDuplicator.ListeEntites(Entites: TQList);
var
 I: Integer;
begin
 for I:=0 to PyObject_Length(BuildImages)-1 do
  (QkObjFromPyObj(PyList_GetItem(FCache, I)) as TTreeMap).ListeEntites(Entites);
end;

procedure TDuplicator.SauverTexte(Negatif: TQList; Texte: TStrings; Flags: Integer; HxStrings: TStrings);
var
 I: Integer;
begin
{if (Specifics.Values['out']<>'')
 and (FParent<>Nil) and (TvParent.TvParent=Nil) then
  GlobalWarning(LoadStr1(230));}  { FIXME: do various map tests globally }
 for I:=0 to PyObject_Length(BuildImages)-1 do
  (QkObjFromPyObj(PyList_GetItem(FCache, I)) as TTreeMap).SauverTexte(Negatif, Texte, Flags, HxStrings);
end;

procedure TDuplicator.AddTo3DScene;
var
 Color1: TColorRef;
 I, InvPoly, InvFaces: Integer;
 T: TTreeMap;
begin
 Color1:=CurrentMapView.Scene.BlendColor;
 CurrentMapView.Scene.SetColor(MiddleColor(Color1, MapColors(lcDuplicator), 0.5));
 InvPoly:=0;
 InvFaces:=0;
 for I:=0 to PyObject_Length(BuildImages)-1 do
  begin
   T:=QkObjFromPyObj(PyList_GetItem(FCache, I)) as TTreeMap;
   BuildPolyhedronsNow(T, InvPoly, InvFaces);
   T.AddTo3DScene;
  end;
 CurrentMapView.Scene.SetColor(Color1);
end;

 {------------------------}

function TDuplicator.PyGetAttr(attr: PChar) : PyObject;
begin
 Result:=inherited PyGetAttr(attr);
 if Result<>Nil then Exit;
 case attr[0] of
  'i': if StrComp(attr, 'images') = 0 then
        begin
         Result:=Images;
         Py_INCREF(Result);
         Exit;
        end;
 end;
end;

(*function TDuplicator.PySetAttr(attr: PChar; value: PyObject) : Boolean;
begin
 Result:=inherited PySetAttr(attr, value);
 if not Result then
  case attr[0] of
   'i': if StrComp(attr, 'images') = 0 then
         begin
          Images:=value;
          Result:=True;
          Exit;
         end;
  end;
end;*)

 {------------------------}

initialization
  RegisterQObject(TDuplicator, 'a');
end.
