(**************************************************************************
QuArK -- Quake Army Knife -- 3D game editor
Copyright (C) 1996-99 Armin Rigo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Contact the author Armin Rigo by e-mail: arigo@planetquake.com
or by mail: Armin Rigo, La Cure, 1854 Leysin, Switzerland.
See also http://www.planetquake.com/quark
**************************************************************************)

unit QkSin;

interface

uses Windows, SysUtils, Classes, Graphics, Dialogs, Controls,
     QkObjects, QkFileObjects, QkTextures, QkPak;

type
 QTextureSin = class(QTexture2)
        protected
          procedure Enregistrer(Info: TInfoEnreg1); override;
          procedure Charger(F: TStream; Taille: Integer); override;
        public
          class function TypeInfo: String; override;
          class function CustomParams : Integer; override;
          function BaseGame : Char; override;
          class procedure FileObjectClassInfo(var Info: TFileObjectClassInfo); override;
        end;
 QSinPak = class(QPak)
           public
             class function TypeInfo: String; override;
             class procedure FileObjectClassInfo(var Info: TFileObjectClassInfo); override;
           end;

 {------------------------}

implementation

uses Game, Setup, Quarkx;

type
 TSinHeader = packed record
               Name: array[0..63] of Char;
               Width, Height: LongInt;
               Palette: array[0..255] of record R,G,B,A: Byte; end;
               PalCrc: Word;
               Reserved1: Word;
               Offsets: array[0..3] of LongInt;
               AnimName: array[0..63] of Char;
               Flags, Contents: LongInt;
               Value, direct: Word;
               animtime, nonlit: Single;
               directangle, trans_angle: Word;
               directstyle, translucence, friction, restitution, trans_mag: Single;
               color: array[0..2] of Single;
              end;

 {------------------------}

class function QTextureSin.TypeInfo: String;
begin
 TypeInfo:='.swl';
end;

class procedure QTextureSin.FileObjectClassInfo(var Info: TFileObjectClassInfo);
begin
 inherited;
 Info.NomClasseEnClair:=LoadStr1(5165);
 Info.FileExt:=793;
end;

class function QTextureSin.CustomParams : Integer;
begin
 Result:=4 or cpPalette or cpAnyHeight;
end;

function QTextureSin.BaseGame : Char;
begin
 Result:=mjSin;
end;

procedure QTextureSin.Charger(F: TStream; Taille: Integer);
const
 Spec1 = 'Image1=';
 Spec2 = 'Pal=';
 Spec3 = 'Alpha=';
var
 Header: TSinHeader;
 Q2MipTex: TQ2MipTex;
 Base, I: Integer;
 Lmp: PPaletteLmp;
 Data: String;
 HasAlpha: Boolean;
 P: PChar;
begin
 case ReadFormat of
  1: begin  { as stand-alone file }
      if Taille<=SizeOf(Header) then
       Raise EError(5519);
      Base:=F.Position;
      F.ReadBuffer(Header, SizeOf(Header));

       { reads the palette }
      Data:=Spec2;
      SetLength(Data, Length(Spec2)+SizeOf(TPaletteLmp));
      Lmp:=PPaletteLmp(@Data[Length(Spec2)+1]);
      HasAlpha:=False;
      for I:=Low(Lmp^) to High(Lmp^) do
       with Header.Palette[I] do
        begin
         Lmp^[I,0]:=R;
         Lmp^[I,1]:=G;
         Lmp^[I,2]:=B;
         if A<>0 then
          HasAlpha:=True;
        end;
      SpecificsAdd(Data);  { "Pal=xxxxx" }

      if HasAlpha then
       begin
        Data:=Spec3;
        SetLength(Data, Length(Spec3)+256);
        P:=@Data[Length(Spec3)+1];
        for I:=0 to 255 do
         P[I]:=Chr(Header.Palette[I].A);
        SpecificsAdd(Data);  { "Alpha=xxxx" }
       end;

       { reads misc flags }
      IntSpec['PalCrc']:=Header.PalCrc;
      SpecificsAdd('direct='+IntToStr(Header.direct));
      SetFloatSpec('animtime', Header.animtime);
      SetFloatSpec('nonlit', Header.nonlit);
      SpecificsAdd('directangle='+IntToStr(Header.directangle));
      SpecificsAdd('trans_angle='+IntToStr(Header.trans_angle));
      SetFloatSpec('directstyle', Header.directstyle);
      SetFloatSpec('translucence', Header.translucence);
      SetFloatSpec('friction', Header.friction);
      SetFloatSpec('restitution', Header.restitution);
      SetFloatSpec('trans_mag', Header.trans_mag);
      SetFloatsSpec('color', Header.color);

       { reads the image data }
      Q2MipTex.W:=Header.Width;
      Q2MipTex.H:=Header.Height;
      Q2MipTex.Contents:=Header.Contents;
      Q2MipTex.Flags:=Header.Flags;
      Q2MipTex.Value:=Header.Value;
      Charger1(F, Base, Taille, Q2MipTex, @Header.Offsets,
       Header.Name, Header.AnimName);
     end;
 else inherited;
 end;
end;

procedure QTextureSin.Enregistrer;
var
 Header: TSinHeader;
 I, Taille: Integer;
 Delta: Integer;
 Lmp: PPaletteLmp;
 S: String;
 V: array[1..2] of Single;
begin
 with Info do case Format of
  1: begin  { as stand-alone file }
      FillChar(Header, SizeOf(Header), 0);
      StrPLCopy(Header.Name, GetTexName, SizeOf(Header.Name));
      if not GetFloatsSpec('Size', V) then
       Raise EErrorFmt(5504, ['Size']);
      Header.Width:=Round(V[1]);
      Header.Height:=Round(V[2]);
      LoadPaletteLmp(Lmp);
      for I:=0 to 255 do
       with Header.Palette[I] do
        begin
         R:=Lmp^[I,0];
         G:=Lmp^[I,1];
         B:=Lmp^[I,2];
        end;
      S:=Specifics.Values['Alpha'];
      for I:=1 to Length(S) do
       Header.Palette[I-1].A:=Ord(S[I]);

      Header.PalCrc:=IntSpec['PalCrc'];

      Delta:=SizeOf(Header);
      Taille:=Header.Width*Header.Height;
      for I:=0 to 3 do
       begin
        Header.Offsets[I]:=Delta;
        Inc(Delta, Taille);
        Taille:=Taille div 4;
       end;

      StrPLCopy(Header.AnimName, Specifics.Values['Anim'], SizeOf(Header.AnimName));
      Header.Contents:=StrToIntDef(Specifics.Values['Contents'], 0);
      Header.Flags   :=StrToIntDef(Specifics.Values['Flags'], 0);
      Header.Value   :=StrToIntDef(Specifics.Values['Value'], 0);
      Header.direct  :=StrToIntDef(Specifics.Values['direct'], 0);
      Header.animtime    :=GetFloatSpec('animtime', 0.2);
      Header.nonlit      :=GetFloatSpec('nonlit', 0.0);
      Header.directangle :=StrToIntDef(Specifics.Values['directangle'], 0);
      Header.trans_angle :=StrToIntDef(Specifics.Values['trans_angle'], 0);
      Header.directstyle :=GetFloatSpec('directstyle', 0.0);
      Header.translucence :=GetFloatSpec('translucence', 0.0);
      Header.friction :=GetFloatSpec('friction', 1.0);
      Header.restitution :=GetFloatSpec('restitution', 0.0);
      Header.trans_mag :=GetFloatSpec('trans_mag', 0.0);
      GetFloatsSpec('color', Header.color);

      F.WriteBuffer(Header, SizeOf(Header));
      for I:=0 to 3 do
       begin
        S:=GetTexImage(I);
        F.WriteBuffer(S[1], Length(S));
       end;
     end;
 else inherited;
 end;
end;

 {------------------------}

class function QSinPak.TypeInfo;
begin
 Result:='.sin';
end;

class procedure QSinPak.FileObjectClassInfo(var Info: TFileObjectClassInfo);
begin
 inherited;
 Info.NomClasseEnClair:=LoadStr1(5166);
 Info.FileExt:=794;
end;

 {------------------------}

initialization
  RegisterQObject(QTextureSin, 'k');
  RegisterQObject(QSinPak, 't');
end.
