(**************************************************************************
QuArK -- Quake Army Knife -- 3D game editor
Copyright (C) 1996-99 Armin Rigo

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Contact the author Armin Rigo by e-mail: arigo@planetquake.com
or by mail: Armin Rigo, La Cure, 1854 Leysin, Switzerland.
See also http://www.planetquake.com/quark
**************************************************************************)

unit PyPanels;

interface

uses Windows, Messages, SysUtils, Classes, Controls, Graphics, Forms, ExtCtrls,
     Python, QSplitter, PyControls, QkForm;

type
  TPanelSections = array[TSplitOrientation] of PyObject;
  TQkPanel = class(TCustomPanel)
  private
    Resizer1: TQSplitter;
    procedure ExtraMargins(var Min, Max: Integer);
    procedure SplitterResizedEvt(Sender: TObject; nPosition: Integer);
    procedure SplitterMarginsEvt(Sender: TObject; var nPosition, Min, Max: Integer);
    procedure SectionResizedEvt(Sender: TObject; nPosition: Integer);
    procedure SectionMarginsEvt(Sender: TObject; var nPosition, Min, Max: Integer);
    procedure wmMessageInterne(var Msg: TMessage); message wm_MessageInterne;
  protected
    procedure AlignControls(AControl: TControl; var Rect: TRect); override;
    procedure DragOver(Source: TObject; X,Y: Integer; State: TDragState; var Accept: Boolean); override;
  public
    PanelObject: PyControl;
    Sections: TPanelSections;  { tuple of floats }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DragDrop(Source: TObject; X, Y: Integer); override;
    function SetSize(nSize: Integer; nAlign: TAlign; nParent: TWinControl) : Boolean;
    procedure CreateResizer;
  end;

 {-------------------}

function GetPanelAttr(self: PyObject; attr: PChar) : PyObject; cdecl;
function SetPanelAttr(self: PyObject; attr: PChar; value: PyObject) : Integer; cdecl;

var
 TyPanel_Type: TyTypeObject =
  (ob_refcnt:      1;
   tp_name:        'panel';
   tp_basicsize:   SizeOf(TyControl);
   tp_dealloc:     ControlDestructor;
   tp_getattr:     GetPanelAttr;
   tp_setattr:     SetPanelAttr;
   tp_doc:         'A Delphi Panel.');

 {-------------------}

implementation

uses Quarkx, PyExplorer, PyFormCfg, PyMapView, PyImages,
     PyToolbars, PyForms;

 {-------------------}

constructor TQkPanel.Create;
begin
 inherited;
 Sections[soHorizontal]:=GetEmptyTuple;
 Sections[soVertical]:=GetEmptyTuple;
 PanelObject:=NewControl(TyPanel_Type, Self);
 BevelOuter:=bvNone;
end;

destructor TQkPanel.Destroy;
begin
 PanelObject^.Close;
 Py_DECREF(Sections[soHorizontal]);
 Py_DECREF(Sections[soVertical]);
 inherited;
end;

procedure TQkPanel.wmMessageInterne(var Msg: TMessage);
begin
 case Msg.wParam of
  wp_GetPyControl: Msg.Result:=LongInt(PanelObject);
 else
  if not DefControlMessage(Msg) then
   inherited;
 end;
end;

function TQkPanel.SetSize(nSize: Integer; nAlign: TAlign; nParent: TWinControl) : Boolean;
begin
 Result:=True;
 case nAlign of
  alLeft: SetBounds(0,0, nSize, nParent.ClientHeight);
  alRight: SetBounds(nParent.ClientWidth-nSize,0, nSize, nParent.ClientHeight);
  alTop: SetBounds(0,0, nParent.ClientWidth, nSize);
  alBottom: SetBounds(0,nParent.ClientHeight-nSize, nParent.ClientWidth, nSize);
  alNone:   begin
             BoundsRect:={nParent.ClientRect;} Rect(0,0,0,0);
             Result:=False;
            end;
 else Result:=False;
 end;
end;

procedure TQkPanel.CreateResizer;
begin
 Resizer1.Free;
 Resizer1:=Nil;
 if Align=alNone then Exit;
 Resizer1:=TQSplitter.Create(Self);
 with Resizer1 do
  begin
   case Self.Align of
    alLeft: Left:=Self.Width;
    alRight: Left:=Self.Left-3;
    alTop: Top:=Self.Height;
    alBottom: Top:=Self.Top-3;
   end;
   if Self.Align in [alTop, alBottom] then
    begin
     Orientation:=soHorizontal;
     Cursor:=crVSplit;
     Height:=3;
    end
   else
    Width:=3;
   Align:=Self.Align;
   Parent:=Self.Parent;
   OnMesureMargins:=SplitterMarginsEvt;
   OnResized:=SplitterResizedEvt;
  end;
end;

procedure TQkPanel.ExtraMargins(var Min, Max: Integer);
var
 I: Integer;
 C: TControl;
begin
 Min:=0;
 Max:=0;
 for I:=0 to Parent.ControlCount-1 do
  begin
   C:=Parent.Controls[I];
   if C.Align = Align then
    case Align of
     alLeft:   if C.Left<Left then Inc(Min, C.Width);
     alRight:  if C.Left>Left then Inc(Max, C.Width);
     alTop:    if C.Top<Top then Inc(Min, C.Height);
     alBottom: if C.Top>Top then Inc(Max, C.Height);
    end;
  end;
end;

procedure TQkPanel.SplitterMarginsEvt(Sender: TObject; var nPosition, Min, Max: Integer);
var
 X1, X2: Integer;
begin
 ExtraMargins(X1, X2);
 Inc(Min, X1);
 Dec(Max, X2);
end;

procedure TQkPanel.SplitterResizedEvt(Sender: TObject; nPosition: Integer);
var
 X1, X2: Integer;
begin
 ExtraMargins(X1, X2);
 case Align of
  alLeft:   Width:=nPosition - X1;
  alRight:  Width:=Parent.ClientWidth-nPosition+X2-3;
  alTop:    Height:=nPosition - X1;
  alBottom: Height:=Parent.ClientHeight-nPosition+X2-3;
 end;
end;

function TotalMapViewSurface(F: TControl) : Integer;
var
 I: Integer;
begin
 if F is TPyMapView then
  begin
   Result:=F.Width*F.Height;
   Exit;
  end;
 Result:=0;
 if F is TQkPanel then
  for I:=0 to TQkPanel(F).ControlCount-1 do
   Inc(Result, TotalMapViewSurface(TQkPanel(F).Controls[I]));
end;

procedure TQkPanel.AlignControls(AControl: TControl; var Rect: TRect);
const
 MaxMapViews = 32;
var
 I, J: Integer;
 C: TControl;
 obj: PyControl;
 f1, f2: Double;
 nBounds: TRect;
 W, H: Integer;
 L: array[0..MaxMapViews-1] of record
      Child: TControl;
      Surface: Integer;
    end;
 LCount, K: Integer;
begin
 inherited;
 LCount:=0;
 W:=Rect.Right-Rect.Left;
 H:=Rect.Bottom-Rect.Top;
 for I:=0 to ControlCount-1 do
  begin
   C:=Controls[I];
   if C.Align=alNone then
    if C is TQSplitter then
     with TQSplitter(C) do
      begin
       if Tag>=0 then Continue;
       f1:=PyFloat_AsDouble(PyTuple_GetItem(Sections[Orientation], not Tag));
       if Orientation = soHorizontal then
        SetBounds(Rect.Left, Rect.Top + Round(H*f1) - 3, W, 3)
       else
        SetBounds(Rect.Left + Round(W*f1) - 3, Rect.Top, 3, H);
      end
    else
     begin
      obj:=PyControl(C.Perform(wm_MessageInterne, wp_GetPyControl, 0));
      if obj<>Nil then
       begin
        J:=PyObject_Length(Sections[soVertical]);
        if obj^.SectionX > J then Continue;
        if obj^.SectionX = 0 then
         f1:=0.0
        else
         f1:=PyFloat_AsDouble(PyTuple_GetItem(Sections[soVertical], obj^.SectionX-1));
        if obj^.SectionX = J then
         f2:=-1.0
        else
         f2:=PyFloat_AsDouble(PyTuple_GetItem(Sections[soVertical], obj^.SectionX));
        nBounds.Left:=Rect.Left + Round(W*Abs(f1));
        nBounds.Right:=Rect.Left + Round(W*Abs(f2));
        if f2>=0 then Dec(nBounds.Right, 3);

        J:=PyObject_Length(Sections[soHorizontal]);
        if obj^.SectionY > J then Continue;
        if obj^.SectionY = 0 then
         f1:=0.0
        else
         f1:=PyFloat_AsDouble(PyTuple_GetItem(Sections[soHorizontal], obj^.SectionY-1));
        if obj^.SectionY = J then
         f2:=-1.0
        else
         f2:=PyFloat_AsDouble(PyTuple_GetItem(Sections[soHorizontal], obj^.SectionY));
        nBounds.Top:=Rect.Top + Round(H*Abs(f1));
        nBounds.Bottom:=Rect.Top + Round(H*Abs(f2));
        if f2>=0 then Dec(nBounds.Bottom, 3);

        C.BoundsRect:=nBounds;
        if C is TPyMapView then
         LCount:=LCount+0; 
       end;
     end;
   if LCount<MaxMapViews then
    begin
     J:=TotalMapViewSurface(C);
     if J>0 then
      begin
       K:=LCount;
       while (K>0) and (J<L[K-1].Surface) do
        begin
         L[K]:=L[K-1];
         Dec(K);
        end;
       L[K].Child:=C;
       L[K].Surface:=J;
       Inc(LCount);
      end;
    end;
  end;
 for K:=0 to LCount-1 do
  L[K].Child.SendToBack; 
end;

procedure TQkPanel.SectionResizedEvt(Sender: TObject; nPosition: Integer);
var
 Rect: TRect;
 FullSize: Integer;
 {obj,} obj1: PyObject;
begin
 Rect:=ClientRect;
 AlignControls(Nil, Rect);
 with Sender as TQSplitter do
  begin
   if Orientation=soHorizontal then
    begin
     FullSize:=Rect.Bottom-Rect.Top;
     Dec(nPosition, Rect.Top);
    end
   else
    begin
     FullSize:=Rect.Right-Rect.Left;
     Dec(nPosition, Rect.Left);
    end;
   if FullSize<=3 then Exit;
   Inc(nPosition, 3);
  {obj:=PyTuple_GetItem(Sections[Orientation], not Tag);}
   obj1:=PyFloat_FromDouble(nPosition/FullSize);
   PyTuple_SetItem(Sections[Orientation], not Tag, obj1);
  end;
 Rect:=ClientRect;
 AlignControls(Nil, Rect);
end;

procedure TQkPanel.SectionMarginsEvt(Sender: TObject; var nPosition, Min, Max: Integer);
var
 f1, {f2,} f3: Double;
 I, Count: Integer;
 Base, FullSize: Integer;
 Rect: TRect;
begin
 Rect:=ClientRect;
 AlignControls(Nil, Rect);
 with Sender as TQSplitter do
  begin
   if Orientation=soHorizontal then
    begin
     FullSize:=Rect.Bottom-Rect.Top;
     Base:=Rect.Top;
    end
   else
    begin
     FullSize:=Rect.Right-Rect.Left;
     Base:=Rect.Left;
    end;
   if FullSize<=3 then Exit;
   I:=not Tag;
   Count:=PyObject_Length(Sections[Orientation]);
   if I=0 then
    f1:=0.0
   else
    f1:=Abs(PyFloat_AsDouble(PyTuple_GetItem(Sections[Orientation], I-1)));
  {f2:=PyFloat_AsDouble(PyTuple_GetItem(Sections[Orientation], I));}
   if I+1>=Count then
    f3:=1.0
   else
    f3:=PyFloat_AsDouble(PyTuple_GetItem(Sections[Orientation], I+1));
   Min:=Base + Round(FullSize*f1) + 10;
   Max:=Base + Round(FullSize*f3) - 14;
  end;
end;

procedure TQkPanel.DragOver(Source: TObject; X,Y: Integer; State: TDragState; var Accept: Boolean);
begin
 Accept:=PanelObject.DragOver;
end;

procedure TQkPanel.DragDrop(Source: TObject; X, Y: Integer);
begin
 PanelObject.DragDrop(Source, Self, X,Y);
end;

 {-------------------}

function pControls(self, args: PyObject) : PyObject; cdecl;
var
 I: Integer;
 C: TControl;
 obj: PyControl;
begin
 try
  Result:=PyList_New(0);
  if PyControl(self)^.QkControl<>Nil then
   with PyControl(self)^.QkControl as TQkPanel do
    for I:=0 to ControlCount-1 do
     begin
      C:=Controls[I];
      obj:=PyControl(C.Perform(wm_MessageInterne, wp_GetPyControl, 0));
      if obj<>Nil then
       PyList_Append(Result, obj);
     end;
 except
  EBackToPython;
  Result:=Nil;
 end;
end;

function wNewPanel(self, args: PyObject; nAlign: TAlign) : PyObject;
var
 nSize: Integer;
 Resizeable: PyObject;
begin
 try
  Result:=Nil;
  Resizeable:=Nil;
  if nAlign=alNone then
   nSize:=0
  else
   if not PyArg_ParseTupleX(args, 'i|O', [@nSize, @Resizeable]) then
    Exit;
  with PyControl(self)^ do
   begin
    if QkControl=Nil then
     begin
      PyErr_SetString(QuarkxError, PChar(LoadStr1(4416)));
      Exit;
     end;
    with TQkPanel.Create(QkControl.Owner) do
     begin
      SetSize(nSize, nAlign, QkControl as TQkPanel);
      Align:=nAlign;
      Parent:=TQkPanel(QkControl);
      if (Resizeable=Nil) or PyObject_IsTrue(Resizeable) then
       CreateResizer;
      Result:=PanelObject;
      Py_INCREF(Result);
     end;
   end;
 except
  EBackToPython;
  Result:=Nil;
 end;
end;

function wNewLeftPanel(self, args: PyObject) : PyObject; cdecl;
begin
 Result:=wNewPanel(self, args, alLeft);
end;

function wNewRightPanel(self, args: PyObject) : PyObject; cdecl;
begin
 Result:=wNewPanel(self, args, alRight);
end;

function wNewTopPanel(self, args: PyObject) : PyObject; cdecl;
begin
 Result:=wNewPanel(self, args, alTop);
end;

function wNewBottomPanel(self, args: PyObject) : PyObject; cdecl;
begin
 Result:=wNewPanel(self, args, alBottom);
end;

function wNewFullPanel(self, args: PyObject) : PyObject; cdecl;
begin
 Result:=wNewPanel(self, args, alNone);
end;

function pNewExplorer(self, args: PyObject) : PyObject; cdecl;
begin
 try
  Result:=Nil;
  with PyControl(self)^ do
   begin
    if QkControl=Nil then
     begin
      PyErr_SetString(QuarkxError, PChar(LoadStr1(4416)));
      Exit;
     end;
    with TPythonExplorer.Create(QkControl.Owner) do
     begin
     {Align:=alClient;}
      Parent:=QkControl as TQkPanel;
      Result:=ExplorerObject;
      Py_INCREF(Result);
     end;
   end;
 except
  EBackToPython;
  Result:=Nil;
 end;
end;

function pNewDataForm(self, args: PyObject) : PyObject; cdecl;
var
 F: TPyForm;
begin
 try
  Result:=Nil;
  with PyControl(self)^ do
   begin
    if QkControl=Nil then
     begin
      PyErr_SetString(QuarkxError, PChar(LoadStr1(4416)));
      Exit;
     end;
    with TPyFormCfg.Create(QkControl.Owner) do
     begin
     {Align:=alClient;}
      Parent:=QkControl as TQkPanel;
      Result:=FormCfgObject;
      Py_INCREF(Result);
      F:=GetParentPyForm(QkControl);
      if F<>Nil then
       OnNeedGameInfo:=F.NeedGameInfoEvt;
     end;
   end;
 except
  EBackToPython;
  Result:=Nil;
 end;
end;

function pNewMapView(self, args: PyObject) : PyObject; cdecl;
begin
 try
  Result:=Nil;
  with PyControl(self)^ do
   begin
    if QkControl=Nil then
     begin
      PyErr_SetString(QuarkxError, PChar(LoadStr1(4416)));
      Exit;
     end;
   {if QkControl.Owner is TPyForm then
     TPyForm(QkControl.Owner).NewMapView;}
    with TPyMapView.Create(QkControl.Owner) do
     begin
     {Align:=alClient;}
      Parent:=QkControl as TQkPanel;
      Result:=MapViewObject;
      Py_INCREF(Result);
     end;
   end;
 except
  EBackToPython;
  Result:=Nil;
 end;
end;

function pNewImageCtrl(self, args: PyObject) : PyObject; cdecl;
var
 nImage: PyObject;
begin
 try
  Result:=Nil;
  nImage:=Nil;
  if not PyArg_ParseTupleX(args, '|O', [@nImage]) then
   Exit;
  with PyControl(self)^ do
   begin
    if QkControl=Nil then
     begin
      PyErr_SetString(QuarkxError, PChar(LoadStr1(4416)));
      Exit;
     end;
    with TPyImageControl.Create(QkControl.Owner) do
     begin
      if nImage<>Nil then
       Image1:=nImage;
      Parent:=QkControl as TQkPanel;
      Result:=ImageObject;
      Py_INCREF(Result);
     end;
   end;
 except
  EBackToPython;
  Result:=Nil;
 end;
end;

function pNewBtnPanel(self, args: PyObject) : PyObject; cdecl;
var
 nButtons: PyObject;
begin
 try
  Result:=Nil;
  nButtons:=Nil;
  if not PyArg_ParseTupleX(args, '|O!', [PyList_Type, @nButtons]) then
   Exit;
  with PyControl(self)^ do
   begin
    if QkControl=Nil then
     begin
      PyErr_SetString(QuarkxError, PChar(LoadStr1(4416)));
      Exit;
     end;
    with TQkBtnPanel.Create(QkControl.Owner) do
     begin
     {Align:=alClient;}
      Parent:=QkControl as TQkPanel;
      if nButtons<>Nil then
       SetButtons(nButtons);
      Result:=BtnPanelObject;
      Py_INCREF(Result);
     end;
   end;
 except
  EBackToPython;
  Result:=Nil;
 end;
end;

const
 MethodTable: array[0..10] of TyMethodDef =
  ((ml_name: 'newexplorer';    ml_meth: pNewExplorer;    ml_flags: METH_VARARGS),
   (ml_name: 'newdataform';    ml_meth: pNewDataForm;    ml_flags: METH_VARARGS),
   (ml_name: 'newmapview';     ml_meth: pNewMapView;     ml_flags: METH_VARARGS),
   (ml_name: 'newimagectrl';   ml_meth: pNewImageCtrl;   ml_flags: METH_VARARGS),
   (ml_name: 'newbtnpanel';    ml_meth: pNewBtnPanel;    ml_flags: METH_VARARGS),
   (ml_name: 'newleftpanel';   ml_meth: wNewLeftPanel;   ml_flags: METH_VARARGS),
   (ml_name: 'newrightpanel';  ml_meth: wNewRightPanel;  ml_flags: METH_VARARGS),
   (ml_name: 'newtoppanel';    ml_meth: wNewTopPanel;    ml_flags: METH_VARARGS),
   (ml_name: 'newbottompanel'; ml_meth: wNewBottomPanel; ml_flags: METH_VARARGS),
   (ml_name: 'newpanel';       ml_meth: wNewFullPanel;   ml_flags: METH_VARARGS),
   (ml_name: 'controls';       ml_meth: pControls;       ml_flags: METH_VARARGS));

function GetPanelAttr(self: PyObject; attr: PChar) : PyObject; cdecl;
var
 I: Integer;
begin
 try
  for I:=Low(MethodTable) to High(MethodTable) do
   if StrComp(attr, MethodTable[I].ml_name) = 0 then
    begin
     Result:=PyCFunction_New(MethodTable[I], self);
     Exit;
    end;
  with PyControl(self)^ do
   case attr[0] of
    's': if StrComp(attr, 'size')=0 then
          begin
           if QkControl<>Nil then
            with QkControl as TQkPanel do
             case Align of
              alLeft, alRight: Result:=PyInt_FromLong(Width);
              alTop, alBottom: Result:=PyInt_FromLong(Height);
              else Result:=PyNoResult;
             end
           else
            Result:=PyNoResult;
           Exit;
          end
         else if StrComp(attr, 'sections')=0 then
          begin
           if QkControl<>Nil then
            with QkControl as TQkPanel do
             Result:=Py_BuildValueX('OO', [Sections[soVertical], Sections[soHorizontal]])
           else
            Result:=PyNoResult;
           Exit;
          end;
    'a': if StrComp(attr, 'align')=0 then
          begin
           if QkControl<>Nil then
            with QkControl as TQkPanel do
             case Align of
              alLeft: Result:=PyString_FromString('left');
              alRight: Result:=PyString_FromString('right');
              alTop: Result:=PyString_FromString('top');
              alBottom: Result:=PyString_FromString('bottom');
              alNone: Result:=PyString_FromString('full');
              else Result:=PyNoResult;
             end
           else
            Result:=PyNoResult;
           Exit;
          end;
   end;
  Result:=GetControlAttr(self, attr, 'panel');
 except
  EBackToPython;
  Result:=Nil;
 end;
end;

function SetPanelAttr(self: PyObject; attr: PChar; value: PyObject) : Integer; cdecl;
var
 nS: TPanelSections;
 Orien: TSplitOrientation;
 I: Integer;
 Rect: TRect;
 P: PChar;
begin
 Result:=-1;
 try
  with PyControl(self)^ do
   case attr[0] of
    'a': if (StrComp(attr, 'align')=0) and (QkControl<>Nil)
         and ((QkControl as TQkPanel).Align in [alLeft, alRight, alTop, alBottom]) then
          begin
           P:=PyString_AsString(value);
           if P=Nil then Exit;
           with QkControl as TQkPanel do
            begin
             case P^ of
              'l': Align:=alLeft;
              'r': Align:=alRight;
              't': Align:=alTop;
              'b': Align:=alBottom;
             end;
             if Resizer1<>Nil then
              CreateResizer;
            end;
           Result:=0;
           Exit;
          end;
    's': if StrComp(attr, 'size')=0 then
          begin
           if QkControl<>Nil then
            with QkControl as TQkPanel do
             SetSize(PyInt_AsLong(value), Align, Parent);
           Result:=0;
           Exit;
          end
         else if StrComp(attr, 'sections')=0 then
          begin
           if not PyArg_ParseTupleX(value, 'O!O!', [PyTuple_Type, @nS[soVertical], PyTuple_Type, @nS[soHorizontal]]) then
            Exit;
           for Orien:=Low(Orien) to High(Orien) do
            for I:=0 to PyObject_Length(nS[Orien])-1 do
             if PyTuple_GetItem(nS[Orien],I)^.ob_type <> PyFloat_Type then
              Raise EError(4440);
           if QkControl<>Nil then
            with QkControl as TQkPanel do
             begin
              for I:=ComponentCount-1 downto 0 do
               if Components[I] is TQSplitter then
                with TQSplitter(Components[I]) do
                 if (Align=alNone) and (Tag<0) then
                  Free;
              for Orien:=Low(Orien) to High(Orien) do
               begin
                Py_DECREF(Sections[Orien]);
                Sections[Orien]:=nS[Orien];
                Py_INCREF(nS[Orien]);
                for I:=0 to PyObject_Length(nS[Orien])-1 do
                 if PyFloat_AsDouble(PyTuple_GetItem(nS[Orien],I)) >= 0 then
                  with TQSplitter.Create(QkControl) do
                   begin
                    Parent:=TQkPanel(QkControl);
                    if Orien=soHorizontal then
                     begin
                      Orientation:=soHorizontal;
                      Cursor:=crVSplit;
                     end;
                    Tag:=not I;
                    OnMesureMargins:=SectionMarginsEvt;
                    OnResized:=SectionResizedEvt;
                   end;
               end;
              Rect:=ClientRect;
              AlignControls(Nil, Rect);
             end;
           Result:=0;
           Exit;
          end;
   end;
  Result:=SetControlAttr(self, attr, value);
 except
  EBackToPython;
  Result:=-1;
 end;
end;

end.
