////////////////////////////////////////////////////////////////////
//  File:   isoxddraw.cpp										  //
//  Description: DirectDraw functions       					  //
//  Rev:   A													  //
//  Created : September 1998									  //
//																  //
//  Author: Yannis Deliyannis									  //
//  mail:   yannis@club-internet.fr								  //
//																  //
//  no copyright												  //
////////////////////////////////////////////////////////////////////    


#include "isoxmain.h"
#include "ddutil.cpp"



LPDDraw lpdraw = NULL;


// DDCreate - Initialize and create DirectDraw object and variables /////////////
LPDIRECTDRAW DDCreate(HWND hwndApp)
{
	HRESULT hr;

	// Destroy existing DirectDraw object if this is a restart
	if( lpdraw != NULL )
		DDDestroy();

	// Allocate and initialize DirectDraw object memory
	if( (lpdraw = (LPDDraw)malloc(sizeof(DDraw))) == NULL )
		return NULL;
	memset(lpdraw,0,sizeof(DDraw));

	// Create DirectDraw object
    hr = DirectDrawCreate( NULL, &lpdraw->lpDD, NULL );
	if( hr == DD_OK )
	{
		lpdraw->hwndApp = hwndApp;

		// Initialize defaults
		lpdraw->nCooperation = DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN;
		lpdraw->cx = Screen.Width();
		lpdraw->cy = Screen.Height();
		lpdraw->bbp = Screen.Bpp();
		lpdraw->nBuffers = 2;
	}
	else 
	{
		free(lpdraw);
		lpdraw = NULL;
		OutputDebugString("DDCreate : Couldn't create DirectDraw object.\n" );
		return NULL;
	}
	
	return lpdraw->lpDD;
}

//DDGetLPDirectDraw -///////////////
LPDIRECTDRAW DDGetLPDirectDraw()
{
	return lpdraw->lpDD;
};

// DDCreateSurfaces - Create surfaces according to screen mode /////////////////
static BOOL DDCreateSurfaces()
{
	LPDIRECTDRAW		lpdd = lpdraw->lpDD;
    DDSURFACEDESC       ddsd;
    HRESULT             hr;
    DDSCAPS             ddscaps;

    memset( (VOID *)&ddsd, 0, sizeof( ddsd ) );
    ddsd.dwSize = sizeof( ddsd );
	
	// Fullscreen case 
    if ( lpdraw->nCooperation & DDSCL_FULLSCREEN )
    {
        // create a primary and one back buffer  
        ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
        ddsd.dwBackBufferCount = lpdraw->nBuffers-1;
        ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX;

        hr = lpdd->CreateSurface(&ddsd, &lpdraw->lpDDSPrimary, NULL );
		if( hr != DD_OK )
		{
			OutputDebugString( "DDCreateSurfaces: Couldn't create primary surface...\n" );
            return FALSE;
		}

        // get pointer to back buffer
        ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
        hr = lpdraw->lpDDSPrimary->GetAttachedSurface(&ddscaps, &lpdraw->lpDDSBack );
        if( hr != DD_OK )
		{
			OutputDebugString( "DDCreateSurfaces: Couldn't create backbuffer surface...\n" );
			return FALSE;
		}
		
		// create the offscreen surface by loading bitmap
		
		// load our palette
		lpdraw->lpDDPal = DDLoadPalette(lpdraw->lpDD, "BITTILES");

		// make sure to set the palette before loading bitmaps.
		if (lpdraw->lpDDPal) 
			lpdraw->lpDDSPrimary->SetPalette(lpdraw->lpDDPal);

		lpdraw->lpDDSTiles = DDLoadBitmap (lpdraw->lpDD, "BITTILES", 0, 0);
		if (lpdraw->lpDDSTiles == NULL)
		{
			OutputDebugString( "DDCreateSurfaces (DDLoadBitmap) : Couldn't create Tiles surface...\n" );
			return FALSE;
		}
		
		// Set the color key for this bitmap (white)
		DDSetColorKey (lpdraw->lpDDSTiles, RGB(255,255,255));

		// Create offscreen surface holding board bitmap
		lpdraw->lpDDSBoard = DDLoadBitmap (lpdraw->lpDD, "BITBOARD", 0, 0);
		if (lpdraw->lpDDSBoard == NULL)
		{
			OutputDebugString( "DDCreateSurfaces (DDLoadBitmap) : Couldn't create board surface...\n" );
			return FALSE;
		}
		
		// Set color key for board bitmap (white)
		DDSetColorKey (lpdraw->lpDDSBoard, RGB(255,255,255));
	}
	else
	//  Windowed case
	{
        // create the primary surface
        ddsd.dwFlags = DDSD_CAPS;
        ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

        hr = lpdd->CreateSurface( &ddsd, &lpdraw->lpDDSPrimary, NULL );
        if( hr != DD_OK )
        {
			OutputDebugString("DDCreateSurfaces : Couldn't create primary surface...\n");
			return FALSE;
		}

		// Create backbuffer in offscreen memory
        lpdraw->lpDDSBack = DDCreateSurface( lpdraw->cx, lpdraw->cy, FALSE );
        if( lpdraw->lpDDSBack == NULL )
		{
			OutputDebugString("DDCreateSurfaces : Couldn't create backbuffer...\n");
            return FALSE;
		}
		
		// load our palette
		lpdraw->lpDDPal = DDLoadPalette(lpdraw->lpDD, "BITTILES");

		// make sure to set the palette before loading bitmaps.
		if (lpdraw->lpDDPal) 
			lpdraw->lpDDSPrimary->SetPalette(lpdraw->lpDDPal);

		// Create tile offscreen surface by loading bitmap
		lpdraw->lpDDSTiles = DDLoadBitmap (lpdraw->lpDD, "BITTILES", 0, 0);
		if (lpdraw->lpDDSTiles == NULL)
		{
			OutputDebugString("DDCreateSurfaces (DDLoadBitmap) : Couldn't create tile surface...\n");
			return FALSE;
		}
		
		// Set the color key for this bitmap (white)
		DDSetColorKey (lpdraw->lpDDSTiles, RGB(255,255,255));

		// Create offscreen surface holding board bitmap
		lpdraw->lpDDSBoard = DDLoadBitmap (lpdraw->lpDD, "BITBOARD", 0, 0);
		if (lpdraw->lpDDSBoard == NULL)
		{
			OutputDebugString("DDCreateSurfaces (DDLoadBitmap) : Couldn't create board surface...\n");
			return FALSE;
		}

		DDSetColorKey (lpdraw->lpDDSBoard, RGB(255,255,255));

        // Create a DirectDrawClipper object.
        hr = lpdd->CreateClipper(0, &lpdraw->lpClipper, NULL);
        if( hr != DD_OK )
		{
			OutputDebugString("DDCreateSurfaces : Couldn't create clipper object...\n");
            return FALSE;
		}
        hr = lpdraw->lpClipper->SetHWnd(0, lpdraw->hwndApp);
        if( hr != DD_OK )
		{
			OutputDebugString("DDCreateSurfaces : Couldn't set clipper object's handle window...\n");
            return FALSE;
		}
        hr = lpdraw->lpDDSPrimary->SetClipper(lpdraw->lpClipper);
        if( hr != DD_OK )
		{
			OutputDebugString("DDCreateSurfaces : Couldn't set clipper object...\n");
            return FALSE;
		}
	}

	if (!CreateObjects())
		return FALSE;

	return TRUE;
}


// DDCreateSurface - create a surface as an offscreen surface //////////////////////
LPDIRECTDRAWSURFACE DDCreateSurface(DWORD width, DWORD height, BOOL sysmem )
{
    DDSURFACEDESC			ddsd;
    HRESULT					hr;
    LPDIRECTDRAWSURFACE		lpSurface;

    // fill in surface desc
    memset( &ddsd, 0, sizeof( ddsd ) );
    ddsd.dwSize = sizeof( ddsd );
    ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
    
	if( sysmem )
        ddsd.ddsCaps.dwCaps |= DDSCAPS_SYSTEMMEMORY;

    ddsd.dwHeight = height;
    ddsd.dwWidth = width;

    hr = lpdraw->lpDD->CreateSurface( &ddsd, &lpSurface, NULL );
	if( hr != DD_OK )
	{
		OutputDebugString( "DDCreateSurface: Couldn't create surface...\n" );
		lpSurface = NULL;
	}
	return lpSurface;
}


// DDSetCooperativeLevel - Change cooperative level /////////////////
BOOL DDSetCooperativeLevel(int level)
{
   BOOL fWasEnabled = lpdraw->fEnabled;

	// check if application is enabled and disable it if necessary
   if( fWasEnabled )
      DDDisable();

   // save new cooperation state
   lpdraw->nCooperation = level;

   // check if application is enabled and enable it
   if( fWasEnabled )
      DDEnable();
   return TRUE;
}


// DDGetCooperativeLevel - returns cooperative level /////////////
int DDGetCooperativeLevel()
{
	return lpdraw->nCooperation;
}

// DDSetWidth - Set screen width //////////////////////////////
void DDSetWidth( int w )
{
	lpdraw->cx = w;
}

// DDSetHeight - Set screen height ////////////////////////////
void DDSetHeight( int h )
{
	lpdraw->cy = h;
}

// DDSetBpp - Set Color resolution ////////////////////////////
void DDSetBpp( int bpp )
{
	lpdraw->bbp = bpp;
}

// DDEnable - set cooperative level and display mode ///////////
BOOL DDEnable()
{
	HRESULT hr;

	if( lpdraw == NULL )
		return FALSE;

	hr = lpdraw->lpDD->SetCooperativeLevel( lpdraw->hwndApp, lpdraw->nCooperation );
	if( hr != DD_OK )
	{
		OutputDebugString("DDEnable : Couldn't set cooperative level...\n");
		return FALSE;
	}

	if( lpdraw->nCooperation & DDSCL_EXCLUSIVE )
	{
		// Mode changes allowed when DDSCL_EXCLUSIVE is set
		hr = lpdraw->lpDD->SetDisplayMode( lpdraw->cx, lpdraw->cy, lpdraw->bbp);
		if( hr != DD_OK )
		{
			OutputDebugString("DDEnable : Couldn't set display mode...\n");
			return FALSE;
		}
	}
	else 
	{
		// Not exclusive, use the window as is
        RECT	rcWork;
        RECT	rc;
        HDC		hdc;

        hdc = GetDC(NULL);
        lpdraw->bbp = GetDeviceCaps(hdc, PLANES) * GetDeviceCaps(hdc, BITSPIXEL);
        ReleaseDC(NULL, hdc);

        SetRect(&rc, 0, 0, lpdraw->cx, lpdraw->cy);

        AdjustWindowRectEx(&rc,
            GetWindowStyle(lpdraw->hwndApp),
            GetMenu(lpdraw->hwndApp) != NULL,
            GetWindowExStyle(lpdraw->hwndApp));

        SetWindowPos(lpdraw->hwndApp, NULL, 0, 0, rc.right-rc.left, rc.bottom-rc.top,
            SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE);

        SetWindowPos(lpdraw->hwndApp, HWND_NOTOPMOST, 0, 0, 0, 0,
            SWP_NOSIZE | SWP_NOMOVE | SWP_NOACTIVATE);

        //  make sure our window does not hang outside of the work area
        SystemParametersInfo(SPI_GETWORKAREA, 0, &rcWork, 0);
        GetWindowRect(lpdraw->hwndApp, &rc);
        if (rc.left < rcWork.left) rc.left = rcWork.left;
        if (rc.top  < rcWork.top)  rc.top  = rcWork.top;
        SetWindowPos(lpdraw->hwndApp, NULL, rc.left, rc.top, 0, 0,
            SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);
	}	
	return DDCreateSurfaces();
}


// DDDisable - takes care of DirectDraw cleanup /////////////////
void DDDisable()
{
	if( lpdraw == NULL )
		return;

	if (lpdraw->lpDDSBoard)
	{
		lpdraw->lpDDSBoard->Release();
		lpdraw->lpDDSBoard = NULL;
	}
	
	if (lpdraw->lpDDSTiles)
	{
		lpdraw->lpDDSTiles->Release();
		lpdraw->lpDDSTiles = NULL;
	}
	
	if( lpdraw->lpDDSBack )
    {
        lpdraw->lpDDSBack->Release();
        lpdraw->lpDDSBack = NULL;
    }

    if( lpdraw->lpDDSPrimary )
    {
		lpdraw->lpDDSPrimary->Release();
        lpdraw->lpDDSPrimary = NULL;
    }

	if( lpdraw->lpClipper) 
	{
		lpdraw->lpClipper->Release();
		lpdraw->lpClipper = NULL;
	}

	if ( lpdraw->lpDDPal)
	{
		lpdraw->lpDDPal->Release();
		lpdraw->lpDDPal = NULL;
	}

	if( lpdraw->nCooperation & DDSCL_EXCLUSIVE )
		lpdraw->lpDD->RestoreDisplayMode();

	DestroyObjects();

}


// DDDestroy - takes care of DirectDraw destruction /////////////
void DDDestroy()
{
	if( lpdraw == NULL )
		return;

	if (lpdraw->lpDD != NULL)
		lpdraw->lpDD->Release();

	free(lpdraw);
	lpdraw = NULL;
}


// DDFlip - Flip surfaces //////////////////////////////////////
BOOL DDFlip()
{
	HRESULT hr;

	// Fullscreen case : use Flip to flip surfaces
	if( lpdraw->nCooperation & DDSCL_FULLSCREEN )
	{
		hr = lpdraw->lpDDSPrimary->Flip(NULL, DDFLIP_WAIT);
	}
	else 
	// Windowed case : use Blit to flip surfaces
	{
		hr = lpdraw->lpDDSPrimary->Blt(&gapp.wndRect, lpdraw->lpDDSBack, NULL, DDBLT_WAIT, NULL);
	}
	return hr == DD_OK;
		
	
}

// DDTextOut ////////////////////////////////////////////////
HRESULT DDTextOut( LPDIRECTDRAWSURFACE lpDDSSurface, 
                                char* lpString, 
                                COLORREF BackColor,
                                COLORREF TextColor,
                                int posx,
                                int posy)
{  
    HRESULT ddrval;
    HDC hdc;
     
    ddrval = lpDDSSurface->GetDC( &hdc );
    if SUCCEEDED( ddrval )
    {
        SetBkColor( hdc, BackColor );
        SetTextColor( hdc, TextColor);
        TextOut( hdc, posx, posy, lpString, lstrlen(lpString) );
        lpDDSSurface->ReleaseDC( hdc );
    }
    return ddrval;
}

// BlitBoard (surface)
void BlitBoard (LPDIRECTDRAWSURFACE lpDDSScreen)
{
	DDBLTFX     ddbltfx;
	
	ZeroMemory(&ddbltfx, sizeof(ddbltfx));
	ddbltfx.dwSize = sizeof(ddbltfx);

	lpDDSScreen->Blt(NULL, lpdraw->lpDDSBoard, NULL, DDBLT_DDFX | DDBLT_WAIT | DDBLT_KEYSRC, &ddbltfx); 	
}

