////////////////////////////////////////////////////////////////////
//  File:   isoxmain.cpp										  //
//  Description: This is the main source file					  //
//  Rev:   A													  //
//  Created : September 1998									  //
//																  //
//  Author: Yannis Deliyannis									  //
//		   (Original Mode 13 isometric engine by Jim Adams, 1996) //
//		   (1st DirectX port by Terence Tan, 1997)				  //
//																  //
//  mail:   yannis@club-internet.fr								  //
//																  //
//  This work is not copyrighted. The author is not responsible   //
//	for any damage resulting in its use.						  //
//  You can use it in any way you want but don't forget to give   //
//  credit to the author(s) if you do.  						  //												  //
////////////////////////////////////////////////////////////////////    

#include "isoxmain.h"


#define NAME  "IsometriX"
#define TITLE "IsometriX Engine v1.0"

#define IDM_FILEEXIT 40002


// Variable definitions /////////////////////////////////////////////

static BOOL bActive = FALSE;


// Classes definitions //////////////////////////////////////////////

// Create an instance containing our globals
GApp gapp;			

// declare and initialize screen object
CScreen Screen(640,480,8);

// declare and initialize MIDI music object
CMusic *Music	= NULL;

// declare and initialize sprite objects
CSprite *Player = NULL;
CSprite *Tree	= NULL;


// CreateObjects ///////////////////////////////////////////////////

BOOL CreateObjects()
{
	// Create music object
	Music  = new CMusic(gapp.hwnd);
	if (Music == 0)
	{
		OutputDebugString("CreateObjects : no available memory for Music !\n" );
		return FALSE;
	}

	// Create player sprite object
	Player = new CSprite("BITPLAYER", 54, 70);
	if (Player == 0)
	{
		OutputDebugString("CreateObjects : no available memory for Player !\n" );
		return FALSE;
	}
	Player->m_frame = 34;
	Player->m_layer = 1;
	Player->m_direction = 1;
	Player->SetPosX(Screen.Width()/2);
	Player->SetPosY(Screen.Height()/2);

	// Create other sprite objects
	Tree = new CSprite("BITTREE", 119, 147);

	return TRUE;
}

// DestroyObjects //////////////////////////////////////////////////

void DestroyObjects()
{
	delete Player;
	Player = NULL;

	delete Tree;
	Tree = NULL;

	delete Music;
	Music = NULL;
}



// DrawInit -- Initialize DirectDraw ///////////////////////////////

static BOOL DrawInit()
{
	DDSURFACEDESC dddesc;
	
	// Create DirectDraw object and initialize helper data
	if( (lpdraw->lpDD = DDCreate(gapp.hwnd)) == NULL )
		return FALSE;

	// A palette is required so see if windowed mode is possible
	memset(&dddesc, 0, sizeof(dddesc));
	dddesc.dwSize = sizeof(dddesc);
	dddesc.dwFlags = DDSD_PIXELFORMAT;
	lpdraw->lpDD->GetDisplayMode(&dddesc);

	// Assume we can run windowed
	gapp.bCanRunWindowed = TRUE;

	// if not
	if ((dddesc.ddpfPixelFormat.dwFlags & DDPF_PALETTEINDEXED8) ==0)
	{
		// Fix previous bad assumption
		gapp.bCanRunWindowed = FALSE;

		// Force mode to 640x480x8 in full screen
		DDSetWidth(Screen.Width());
		DDSetHeight(Screen.Height());
		DDSetBpp(Screen.Bpp());
		DDSetCooperativeLevel(DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN);
		
		ShowCursor(FALSE);
		
		ShowWindow(gapp.hwnd, SW_SHOW);
	}
	else
	{
		// Start out windowed
		DDSetCooperativeLevel( DDSCL_NORMAL | DDSCL_NOWINDOWCHANGES );
	}
	
	// Query DirectDraw4 Interface (if DirectX6 installed)////
	/*hr = lpdraw->lpDD->QueryInterface(IID_IDirectDraw4, (LPVOID*)&lpdraw->lpDD4);
	if (hr != DD_OK)
		return FALSE;
	
	// Release old interface 
	lpdraw->lpDD->Release();*/

	if( DDEnable() == FALSE )
		return FALSE;
	
	return TRUE;
}

// WinInit -- //////////////////////////////////////////////////////

static BOOL DoInit (HINSTANCE hInstance, int nCmdShow)
{
	WNDCLASS	wc;
	
	// Set up and register window class
	wc.style = CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc = WindowProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon(hInstance, "ISOXICON");
	wc.hCursor = LoadCursor (NULL, IDC_ARROW);
	wc.hbrBackground = NULL;
	wc.lpszMenuName = NAME;
	wc.lpszClassName = NAME;
	if (!RegisterClass (&wc))
		return FALSE;

	// Create a window
	gapp.hwnd = CreateWindowEx (
		WS_EX_APPWINDOW,
		NAME,
		TITLE,
		WS_VISIBLE | WS_THICKFRAME | WS_SYSMENU | WS_OVERLAPPED,
		0,
		0,
		Screen.Width(),
		Screen.Height(),
		NULL,
		NULL,
		hInstance,
		NULL);
	
	if (!gapp.hwnd)
		return FALSE;

	UpdateWindow(gapp.hwnd);
	
	SetFocus( gapp.hwnd );

	return TRUE;
}


// WindowProc -- ///////////////////////////////////////////////////

long FAR PASCAL WindowProc (HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps;
	HDC			hdc;
	
	switch (message)
	{

		HANDLE_MSG (hWnd, WM_MOUSEMOVE, KeyMouse_OnMouseMove);

	case WM_CREATE:
		break;

	case WM_ACTIVATEAPP:
		bActive = wParam; // Our application is active
		break;

	case WM_SIZE:
	case WM_MOVE:
		GetClientRect(gapp.hwnd, &gapp.wndRect);
		ClientToScreen(gapp.hwnd, (LPPOINT)&gapp.wndRect);
		ClientToScreen(gapp.hwnd, (LPPOINT)&gapp.wndRect+1);
		break;

	case WM_ENTERMENULOOP:
		// Suspend page flipping
		gapp.bFlippingPaused = TRUE;

		// Get GDI's surface back where it's expected !
		lpdraw->lpDD->FlipToGDISurface();

		// Draw menu
		DrawMenuBar(gapp.hwnd);
		RedrawWindow(gapp.hwnd, NULL, NULL, RDW_FRAME);
		break;

	case WM_EXITMENULOOP:
		gapp.bFlippingPaused = FALSE;
		break;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
			// File exit
			case IDM_FILEEXIT:
				PostMessage(hWnd, WM_CLOSE, 0, 0L);
				break;
		}
		break;

	case WM_KEYDOWN:

		switch (wParam)
		{
			// Change scroll speed
			case 0x30:
				Screen.m_scrollSpeed=2;
				break;
			case 0x31:
				Screen.m_scrollSpeed=4;
				break;
			case 0x32:
				Screen.m_scrollSpeed=6;
				break;
			case 0x33:
				Screen.m_scrollSpeed=8;
				break;
			case 0x34:
				Screen.m_scrollSpeed=10;
				break;
			case 0x35:
				Screen.m_scrollSpeed=12;
				break;
			case 0x36:
				Screen.m_scrollSpeed=14;
				break;
			case 0x37:
				Screen.m_scrollSpeed=16;
				break;
			case 0x38:
				Screen.m_scrollSpeed=18;
				break;
			case 0x39:
				Screen.m_scrollSpeed=20;
				break;

			case 0x4D: // M key - plays music on/off
			
				if (Music->m_bPlay)
				{
					Music->Stop();
				}
				else if (!Music->m_bPlay)
				{
					Music->Play("yewtree.mid");
				}
				Music->m_bPlay = !Music->m_bPlay;
				break;

			case 0x46: // F key - displays m_frame rate
				Screen.m_bDisplayFPS = !Screen.m_bDisplayFPS;
				break;

			case 0x44: // D key - display debug vars
				Screen.m_bDisplayDebugVars = !Screen.m_bDisplayDebugVars;
				break;

			case 0x42: // B key - display board
				Screen.m_bBoardVisible = !Screen.m_bBoardVisible;
				break;

			case 0x50: // P key - display sprite
				Player->m_bVisible = !Player->m_bVisible;
				break;
				
			// Escape from program - with ESC or F12
			case VK_ESCAPE:
			case VK_F12:
				PostMessage(hWnd, WM_CLOSE, 0, 0);
				break;
		}
		break;

	case WM_SYSKEYUP:
		switch (wParam)
		{
			case VK_RETURN:
			{
			// ALT+ENTER..switch between fullscreen and windowed mode
			int nCooperation = DDGetCooperativeLevel();
			
			// skip this if we can't run windowed
			if (gapp.bCanRunWindowed == FALSE)
				break;
						
			DDDisable();
			
			if (nCooperation & DDSCL_FULLSCREEN)
			{
				DDSetCooperativeLevel(DDSCL_NORMAL | DDSCL_NOWINDOWCHANGES);
				// mouse cursor visible in windowed mode
				ShowCursor(TRUE);
			}
			else
			{
				DDSetCooperativeLevel(DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN);
				// mouse cursor not visible in fullscreen mode
				ShowCursor(FALSE); 
			}

			DDEnable();
			break;
			}
		}
		break;

	case WM_PAINT:
		hdc = BeginPaint (hWnd, &ps);
		EndPaint(hWnd, &ps);
		return 1;

	case WM_DESTROY:
		DDDisable();
		DDDestroy();
		PostQuitMessage(0);
		break;

	// restarts MIDI music when finished playing
	case MM_MCINOTIFY:
		if (wParam == MCI_NOTIFY_SUCCESSFUL)
			Music->Restart();
		break;
	}

	return DefWindowProc (hWnd, message, wParam, lParam);
}

// WinMain -- Initialization and message loop //////////////////////

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	MSG msg;

	memset((void *)&gapp, 0, sizeof(gapp) );

	// initialize
	if (!DoInit (hInstance, nCmdShow))
		return FALSE;
	
	if( !DrawInit() )
		return FALSE;
	
	// create map
	SetupMap();

	// create objects map
	SetupObjects();

	// Message loop
	while (1)
	{
		if (PeekMessage (&msg, NULL, 0, 0, PM_NOREMOVE))
		{
			if (!GetMessage(&msg, NULL, 0, 0))
				return msg.wParam;
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
		else if (bActive)
		{
			MainLoop();
		}
		else
			WaitMessage();
	}
}


// MainLoop -- The main part ///////////////////////////////////////

void MainLoop()
{
	// used for computing m_frame rate
	DWORD time;
	time = timeGetTime();

	// main loop begins here..... ///

	// check for keyboard input
	CheckKeyboard();
	
	// check to see if map needs wrap up
	CheckMapCoordinates();
	
	// fill back buffer with black color before drawing map
	FillBufferRect(lpdraw->lpDDSBack, NULL, RGB(0,0,0));

	// draw map proper
	DrawMap (lpdraw->lpDDSBack, Screen.m_mapX, Screen.m_mapY, Screen.m_x, Screen.m_y);
	
	// display other things (board, variables, etc).
	if (Screen.m_bBoardVisible)
	{
		BlitBoard (lpdraw->lpDDSBack);
	}

	if (Screen.m_bDisplayFPS)
	{
		GetFrameRate(time);
	}

	if (Screen.m_bDisplayDebugVars)
	{
		DebugVars ();
	}

	// Flip buffers
	if( gapp.bFlippingPaused == FALSE )
		DDFlip();
}

// PlayerNextFrame -- Advance player sprite's m_frame when walking//
void PlayerNextFrame()
{
	switch (Player->m_direction)
	{
	case 4: // west
		if (Player->m_frame < 45 || Player->m_frame >= 54)
			Player->m_frame = 45;

		Player->m_frame++;
		Player->m_bMirroring = FALSE;
		break;
	
	case 6 : // east
		if (Player->m_frame < 45 || Player->m_frame >= 54)
			Player->m_frame = 45;
		
		Player->m_frame++;
		Player->m_bMirroring = TRUE;
		break;
	
	case 2 : // south
		if (Player->m_frame < 23 || Player->m_frame >= 32)
			Player->m_frame = 23;
		
		Player->m_frame++;
		Player->m_bMirroring = FALSE;
		break;
		
	case 8 : // north
		if (Player->m_frame < 1 || Player->m_frame >= 10)
			Player->m_frame = 1;
		
		Player->m_frame++;
		Player->m_bMirroring = FALSE;
		break;
		
	case 1 : // south west
		if (Player->m_frame < 34 || Player->m_frame >= 43)
			Player->m_frame = 34;
	
		Player->m_frame++;
		Player->m_bMirroring = FALSE;
		break;

	case 7 : // north west
		if (Player->m_frame < 12 || Player->m_frame >= 21)
			Player->m_frame = 12;
		
		Player->m_frame++;
		Player->m_bMirroring = FALSE;
		break;

	case 9 : // north east
		if (Player->m_frame < 12 || Player->m_frame >= 21)
			Player->m_frame = 12;
		
		Player->m_frame++;
		Player->m_bMirroring = TRUE;
		break;

	case 3 : // south east
		if (Player->m_frame < 34 || Player->m_frame >= 43)
			Player->m_frame = 34;
		
		Player->m_frame++;
		Player->m_bMirroring = TRUE;
		break;
	}
}

// FillBufferRect //////////////////////////////////////////////////
BOOL FillBufferRect(LPDIRECTDRAWSURFACE lpSurface, LPRECT lprect, COLORREF color)
{
	DDBLTFX ddbltfx;
	HRESULT hr;

	ddbltfx.dwSize = sizeof(ddbltfx);
	ddbltfx.dwFillColor = color;

	hr = lpSurface->Blt(lprect,NULL,NULL,DDBLT_COLORFILL | DDBLT_WAIT, &ddbltfx);

	return hr == DD_OK;
}

// CheckKeyboard ///////////////////////////////////////////////////
void CheckKeyboard()
{	
	// Move map around and update player sprite
	if (GetAsyncKeyState(VK_NUMPAD1) || GetAsyncKeyState(VK_DOWN) && GetAsyncKeyState(VK_LEFT))
	{
	    Screen.m_mapY+=(Screen.m_scrollSpeed*2);
		Player->m_direction = 1;
		PlayerNextFrame();
		Screen.m_x-=(Screen.m_scrollSpeed*2);
		Screen.m_y+=Screen.m_scrollSpeed;
		return;
	}
	
	if (GetAsyncKeyState(VK_NUMPAD7) || GetAsyncKeyState(VK_LEFT) && GetAsyncKeyState(VK_UP))
	{
		Screen.m_mapX-=(Screen.m_scrollSpeed*2);
		Player->m_direction = 7;
		PlayerNextFrame();
		Screen.m_x-=(Screen.m_scrollSpeed*2);
		Screen.m_y-=Screen.m_scrollSpeed;
		return;
	}

	if (GetAsyncKeyState(VK_NUMPAD3) || GetAsyncKeyState(VK_RIGHT) && GetAsyncKeyState(VK_DOWN))
	{
		Screen.m_mapX+=(Screen.m_scrollSpeed*2);
		Player->m_direction = 3;
		PlayerNextFrame();
		Screen.m_x+=(Screen.m_scrollSpeed*2);
		Screen.m_y+=Screen.m_scrollSpeed;
		return;
	}

	if (GetAsyncKeyState(VK_NUMPAD9) || GetAsyncKeyState(VK_RIGHT) && GetAsyncKeyState(VK_UP))
	{
		Screen.m_mapY-=(Screen.m_scrollSpeed*2);
		Player->m_direction = 9;
		PlayerNextFrame();
		Screen.m_x+=(Screen.m_scrollSpeed*2);
		Screen.m_y-=Screen.m_scrollSpeed;
		return;
	}

	if (GetAsyncKeyState(VK_LEFT) || GetAsyncKeyState(VK_NUMPAD4))
	{
		Screen.m_mapX-=Screen.m_scrollSpeed; 
		Screen.m_mapY+=Screen.m_scrollSpeed;
		Player->m_direction = 4;
		PlayerNextFrame();
		Screen.m_x-=(Screen.m_scrollSpeed*2);
		return;
	}
			
	if (GetAsyncKeyState(VK_RIGHT) || GetAsyncKeyState(VK_NUMPAD6))
	{
		Screen.m_mapX+=Screen.m_scrollSpeed;
		Screen.m_mapY-=Screen.m_scrollSpeed;
		Player->m_direction = 6;
		PlayerNextFrame();
		Screen.m_x+=(Screen.m_scrollSpeed*2);
		return;
	}

	if (GetAsyncKeyState(VK_UP) || GetAsyncKeyState(VK_NUMPAD8))
	{
		Screen.m_mapX-=Screen.m_scrollSpeed;
		Screen.m_mapY-=Screen.m_scrollSpeed;
		Player->m_direction = 8;
		PlayerNextFrame();
		Screen.m_y-=Screen.m_scrollSpeed;
		return;
	}
		
	if (GetAsyncKeyState(VK_DOWN) || GetAsyncKeyState(VK_NUMPAD2))
	{
	    Screen.m_mapX+=Screen.m_scrollSpeed;
		Screen.m_mapY+=Screen.m_scrollSpeed;
		Player->m_direction = 2;
		PlayerNextFrame();
		Screen.m_y+=Screen.m_scrollSpeed;
		return;
	}
}
