/*
 *    Example program for the Allegro library, by Grzegorz Adam Hankiewicz.
 *
 *    This program demonstrates how to create custom graphic effects with
 *    the create_color_table function, this time a greyscale effect.
 */


#include <stdio.h>
#include <conio.h>
#include <time.h>

#ifdef __RSXNT__
#include "wrapper.h"
#endif

#include "allegro.h"

/* RGB -> color mapping table. Not needed, but speeds things up */
RGB_MAP rgb_table;

/* greyscale color mapping table */
COLOR_MAP greyscale_table;

PALLETE pal;
BITMAP *background;
BITMAP *temp;


/* Here comes our custom function. It's designed to take the input colors
 * (red, green & blue) and return a greyscale color for it. This way, when
 * any drawing function draws say over green, it draws the greyscale color
 * for green.
 * 'pal' is the palette we are looking in to find the colors.
 * Now, imagine we want to draw a pixel with color A, over color B.
 * Once the table is created, set, and the drawing mode is TRANSLUCENT, then
 * A is the 'x' color passed to the function and B is the 'y' color passed
 * to the function.
 * Since we want a greyscale effect with no matter what A (or 'x') color, we
 * ignore it and use y to look at the palette.
 */

RGB return_grey_color(PALETTE pal, int x, int y)
{
   RGB rgb;
   int c;

   /* first create the greyscale color */
   c = (pal[y].r * 0.3 + pal[y].g * 0.5 + pal[y].b * 0.2);

   /* then assign the greyscale color to our rgb triplet... */
   rgb.r = c;
   rgb.g = c;
   rgb.b = c;

   /* ...and return it */
   return rgb;
}


void generate_background()
{
   int i;

   /* first get some usual colors */
   generate_332_palette(pal);

   /* now remap the first 64 for a perfect greyscale gradient */
   for (i=0; i<64; i++) {
      pal[i].r = i;
      pal[i].g = i;
      pal[i].b = i;
   }

   /* draws some things on the screen using not-greyscale colors */
   for (i=0; i<3000; i++)
      circlefill(background, random()%320, random()%200, random()%25, 64+random()%192);
}


int main(int argc, char **argv)
{
   int x, y;
   int deltax = 1;
   int deltay = 1;

   srandom(time(NULL));

   allegro_init();
   install_keyboard();

   temp = create_bitmap(320, 200);
   background = create_bitmap(320, 200);

   generate_background();

   /* this isn't needed, but it speeds up the color table calculations */
   create_rgb_table(&rgb_table, pal, NULL);
   rgb_map = &rgb_table;

   /* build a color lookup table for greyscale effect */
   create_color_table(&greyscale_table, pal, return_grey_color, NULL);

   /* select the greyscale table */
   color_map = &greyscale_table;

   if (set_gfx_mode(GFX_AUTODETECT, 320, 200, 0, 0) != 0) {
#ifndef __ALLEG_DJGPP_WIN32_RSXNT__
      printf("Sorry, couldn't set graphic mode\n");
#endif
      return 2;
   }
   set_palette(pal);

   /* Look, we have set the drawing mode to TRANS. This makes all the drawing
      functions use the general color_map table, which is _NOT_ translucent,
      since we are using a custom color_map table. */
   drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);

   /* the rest of the example just moves the square through the screen */
   x = y = 50;
   blit(background, temp, 0, 0, 0, 0, 320, 200);
   rectfill(temp, x, y, x+50, y+50, 0);

   acquire_screen();
   blit(temp, screen, 0, 0, 0, 0, 320, 200);
   release_screen();

   while (!keypressed()) {
      x += deltax;
      y += deltay;

      if ((x<1) || (x>320-50))
	 deltax *= -1;

      if ((y<1) || (y>200-50))
	 deltay *= -1;

      blit(background, temp, 0, 0, 0, 0, 320, 200);
      rectfill(temp, x, y, x+50, y+50, 0);
      vsync();
      acquire_screen();
      blit(temp, screen, 0, 0, 0, 0, 320, 200);
      release_screen();
   }

   return 0;
}

