/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *      DirectX Allegro graphics driver, by Stefan Schimanski (1Stein@gmx.de)
 *
 *      See readme.txt for copyright information.
 */


#include <stdlib.h>
#include <string.h>

/*#include "allegro.h"*/
#include "internal.h"
#include "windx.h"


//#define USE_HW_CURSOR


static BITMAP *directx_init(int w, int h, int v_w, int v_h, int color_depth);
static BITMAP *directx_initwin(int w, int h, int v_w, int v_h, int color_depth);
static void directx_exit(BITMAP *b);
static int directx_scroll(int x, int y);
static void directx_vsync();
static void directx_set_pallete_range(PALLETE p, int from, int to, int vsync);
struct BITMAP *directx_create_video_bitmap(int width, int height);
void directx_destroy_video_bitmap(struct BITMAP *bitmap);
int directx_show_video_bitmap(struct BITMAP *bitmap);
int directx_request_video_bitmap(struct BITMAP *bitmap);
int directx_poll_scroll();

int directx_getpixel(struct BITMAP *bmp, int x, int y);
void directx_putpixel(struct BITMAP *bmp, int x, int y, int color);
void directx_vline(struct BITMAP *bmp, int x, int y1, int y2, int color);
void directx_hline(struct BITMAP *bmp, int x1, int y, int x2, int color);
void directx_line(struct BITMAP *bmp, int x1, int y1, int x2, int y2, int color);
void directx_rectfill(struct BITMAP *bmp, int x1, int y1, int x2, int y2, int color);
int  directx_triangle(struct BITMAP *bmp, int x1, int y1, int x2, int y2, int x3, int y3, int color);
void directx_draw_sprite(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_256_sprite(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_sprite_v_flip(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_sprite_h_flip(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_sprite_vh_flip(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_trans_sprite(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_lit_sprite(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y, int color);
void directx_draw_rle_sprite(struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y);
void directx_draw_trans_rle_sprite(struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y);
void directx_draw_lit_rle_sprite(struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y, int color);
void directx_draw_character(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y, int color);
void directx_textout_fixed(struct BITMAP *bmp, void *f, int h, unsigned char *str, int x, int y, int color);
void directx_blit_from_memory(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_blit_to_memory(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_blit_to_self(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_blit_to_self_forward(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_blit_to_self_backward(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_masked_blit(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_clear_to_color(struct BITMAP *bitmap, int color);
void directx_draw_sprite_end(void);
void directx_blit_end(void);
int directx_is_lost(struct BITMAP *bmp);
int directx_acquire(struct BITMAP *bmp);
void directx_release(struct BITMAP *bmp);
void directx_restore(struct BITMAP *bmp);

int directx_set_mouse_sprite(struct BITMAP *sprite, int xfocus, int yfocus);
int directx_show_mouse(struct BITMAP *bmp, int x, int y);
void directx_hide_mouse();
void directx_move_mouse(int x, int y);


GFX_DRIVER gfx_directx = 
{
   "DirectX", 
   "DirectDraw 3", 
   directx_init,
   directx_exit,
   directx_scroll,
   directx_vsync,
   directx_set_pallete_range,
   NULL, // int  (*request_scroll)(int x, int y);
   directx_poll_scroll,
   directx_create_video_bitmap,
   directx_destroy_video_bitmap,
   directx_show_video_bitmap,
   directx_request_video_bitmap,
   NULL, // int  (*set_mouse_sprite)(struct BITMAP *sprite, int xfocus, int yfocus);
   NULL, // int  (*show_mouse)(struct BITMAP *bmp, int x, int y);
   NULL, // void (*hide_mouse)();
   NULL, // void (*move_mouse)(int x, int y);
   NULL, // void (*drawing_mode)();
   0, 0, // physical (not virtual!) screen size
   TRUE, // true if video memory is linear
   0, 0, // bank size, bank granularity
   0,    // video memory size
   0     // physical address of video memory
};


GFX_DRIVER gfx_directxwin = 
{
   "DirectX", 
   "DirectDraw 3", 
   directx_initwin,
   directx_exit,
   directx_scroll,
   directx_vsync,
   directx_set_pallete_range,
   NULL, // int  (*request_scroll)(int x, int y);
   NULL, // int  (*poll_scroll)();
   directx_create_video_bitmap,
   directx_destroy_video_bitmap,
   directx_show_video_bitmap,
   NULL, // int  (*request_video_bitmap)(struct BITMAP *bitmap);
#ifdef USE_HW_CURSOR
   directx_set_mouse_sprite,
   directx_show_mouse,
   directx_hide_mouse,
   directx_move_mouse,
#endif
   NULL,
   NULL,
   NULL,
   NULL,
   NULL, // void (*drawing_mode)();
   0, 0, // physical (not virtual!) screen size
   TRUE, // true if video memory is linear
   0, 0, // bank size, bank granularity
   0,    // video memory size
   0     // physical address of video memory
}; 


struct CDDSurface *screen_surface = NULL;


/********************************************************
 *  Initialize driver									*
 ********************************************************/
static struct BITMAP *directx_init(int w, int h, int v_w, int v_h, int color_depth)
{
	BITMAP *b;

	// Setup graphic driver
	screen_surface = DD_Open(w, h, color_depth, 1);
	if (screen_surface==NULL)
	{
		DD_Close();
		return NULL;
	}
	
	gfx_directx.w = w;
	gfx_directx.h = h;   

	gfx_directx.linear = 1;
	gfx_directx.bank_size = 0;
	gfx_directx.bank_gran = 0;
	gfx_directx.vid_mem = 2000000;    

	// Create screen bitmap
	b = _make_bitmap(w, h, 0, &gfx_directx, color_depth, w);
	if (!b) return NULL;	

	b->extra = (struct CDDSurface *)screen_surface;
	
	// Sets the lock/unlock callbacks
	_screen_vtable.acquire = directx_acquire;
	_screen_vtable.release = directx_release;
	_screen_vtable.restore = directx_restore;

	// Hardware acceleration
	gfx_capabilities = 0;

	_screen_vtable.blit_to_self = directx_blit_to_self;
	_screen_vtable.blit_to_self_forward = directx_blit_to_self;
	_screen_vtable.blit_to_self_backward = directx_blit_to_self;
	_screen_vtable.hw_blit = TRUE;

    gfx_capabilities |= GFX_HW_VRAM_BLIT;
    gfx_capabilities |= GFX_CAN_TRIPLE_BUFFER;

	// Set color palette
	DD_SetPalette((struct CDDColor *)_current_pallete, 0, 255, 0);

	return b;
}


static struct BITMAP *directx_initwin(int w, int h, int v_w, int v_h, int color_depth)
{
	BITMAP *b;

	// Setup graphic driver
	screen_surface = DD_Open(w, h, color_depth, 0);
	if (screen_surface==NULL)
	{
		DD_Close();
		screen_surface = NULL;
		return NULL;
	}

	gfx_directxwin.w = w;
	gfx_directxwin.h = h;   

	gfx_directxwin.linear = 1;
	gfx_directxwin.bank_size = 0;
	gfx_directxwin.bank_gran = 0;
	gfx_directxwin.vid_mem = 2000000;

	// Create screen bitmap
	b = _make_bitmap(w, h, 0, &gfx_directxwin, color_depth, w);
	if (!b) return NULL;

	b->extra = (struct CDDSurface*)screen_surface;

	// Sets the lock/unlock callbacks
	_screen_vtable.acquire = directx_acquire;
	_screen_vtable.release = directx_release;
	_screen_vtable.restore = directx_restore;

	// Hardware acceleration	
	_screen_vtable.blit_to_self = directx_blit_to_self;
	_screen_vtable.blit_to_self_forward = directx_blit_to_self;
	_screen_vtable.blit_to_self_backward = directx_blit_to_self;
	_screen_vtable.hw_blit = TRUE;
	gfx_capabilities |= GFX_HW_VRAM_BLIT;
#ifdef USE_HW_CURSOR
	gfx_capabilities |= GFX_HW_CURSOR;
#endif

	// Set color palette
	DD_SetPalette((struct CDDColor *)_current_pallete, 0, 255, 0);

	return b;
}


/********************************************************
 *  Destruct DirectX									*
 ********************************************************/
static void directx_exit(BITMAP *b)
{		
	DD_Close();
	screen_surface = NULL;
}


/********************************************************
 *  Scroll surface										*
 ********************************************************/
static int directx_scroll(int x, int y)
{
	return -1;
}


/********************************************************
 *  Wait for sync										*
 ********************************************************/
static void directx_vsync()
{
	DD_VSync();
}

/********************************************************
 *  DirectX palette										*
 ********************************************************/
static void directx_set_pallete_range(PALLETE p, int from, int to, int vsync)
{
	DD_SetPalette((struct CDDColor*)p, from, to, vsync);
}


/********************************************************
 * Locks the bitmap memory								*
 ********************************************************/
int directx_acquire(struct BITMAP *b)
{
	int i;
	struct CDDSurface *s = (struct CDDSurface*)(b->extra);

    void *OldBase = b->line[0];

	DD_Lock(s);	
	
    if (b->line[0]!=s->BasePtr)
	{	
		for (i=0; i<b->h; i++)
			b->line[i] = s->BasePtr + i*s->Pitch;

        return (OldBase==NULL)?0:1;
	}

	return s->Lost;
}


void directx_release(struct BITMAP *b)
{
	DD_Unlock((struct CDDSurface*)(b->extra));
}


int directx_is_lost(struct BITMAP *b)
{
	return ((struct CDDSurface*)b->extra)->Lost!=0;
}


void directx_restore(struct BITMAP *bmp)
{
	((struct CDDSurface*)bmp->extra)->Lost = 0;

	DD_Lock((struct CDDSurface*)(bmp->extra));
	DD_Unlock((struct CDDSurface*)(bmp->extra));	
}


struct BITMAP *directx_create_video_bitmap(int width, int height)
{
	BITMAP *bitmap;
	struct CDDSurface *s;
	int y;

	s = DD_CreateSurface(width, height);
	if (s)
	{
		// get memory for structure and line pointers
		bitmap = malloc(sizeof(BITMAP) + (sizeof(char *) * height));
		if (!bitmap)
		{
			DD_DestroySurface(s);
			return NULL;
		}

		bitmap->w = width;
		bitmap->h = height;
		bitmap->clip = TRUE;
		bitmap->cl = 0;
		bitmap->ct = 0;
		bitmap->cr = width;
		bitmap->cb = height;
		bitmap->vtable = &_screen_vtable;
		bitmap->write_bank = _stub_bank_switch;
		bitmap->read_bank = _stub_bank_switch;
		bitmap->dat = NULL;
		bitmap->extra = (void *)s;
		bitmap->x_ofs = 0;
		bitmap->y_ofs = 0;
		bitmap->seg = _dos_ds;
        bitmap->bitmap_id = -1;	
				
		// setup line pointers
		for (y=0; y<height; y++)
			bitmap->line[y] = NULL;

		if (bitmap->vtable->set_clip)
			bitmap->vtable->set_clip(bitmap);

		return bitmap;
	} else return NULL;
}


void directx_destroy_video_bitmap(struct BITMAP *bitmap)
{
	DD_DestroySurface((struct CDDSurface *)bitmap->extra);
	free(bitmap);
}


int directx_show_video_bitmap(struct BITMAP *bitmap)
{
	return DD_ShowSurface((struct CDDSurface *)bitmap->extra, 1);
}


/********************************************************
 *  Setup surface flip              					*
 ********************************************************/
int directx_request_video_bitmap(struct BITMAP *bitmap)
{
    return DD_ShowSurface((struct CDDSurface *)bitmap->extra, 0);
}


/********************************************************
 *  Return flip status		                        	*
 ********************************************************/
int directx_poll_scroll()
{
    return DD_ShowFinished();
}


void directx_blit_to_self(struct BITMAP *source, struct BITMAP *dest, 
						  int source_x, int source_y, int dest_x, int dest_y, 
						  int width, int height)
{
	DD_Blit((struct CDDSurface *)(source->extra), (struct CDDSurface *)(dest->extra), 
			source_x, source_y, dest_x, dest_y, width, height);
}


void directx_blit_to_self_forward(struct BITMAP *source, struct BITMAP *dest, 
								  int source_x, int source_y, int dest_x, int dest_y, 
								  int width, int height)
{
	DD_Blit((struct CDDSurface *)(source->extra), (struct CDDSurface *)(dest->extra), 
			source_x, source_y, dest_x, dest_y, width, height);
}


void directx_blit_to_self_backward(struct BITMAP *source, struct BITMAP *dest, 
								   int source_x, int source_y, int dest_x, int dest_y, 
								   int width, int height)
{
	DD_Blit((struct CDDSurface *)(source->extra), (struct CDDSurface *)(dest->extra), 
			source_x, source_y, dest_x, dest_y, width, height);
}


int directx_set_mouse_sprite(struct BITMAP *sprite, int xfocus, int yfocus)
{
	// TODO: Set new mouse shape
	return 0;
}


int directx_show_mouse(struct BITMAP *bmp, int x, int y)
{
	DI_ShowMouse();
	DI_MoveMouse(x, y);
	return 0;
}


void directx_hide_mouse()
{
	DI_HideMouse();
}


void directx_move_mouse(int x, int y)
{
	DI_MoveMouse(x, y);
}
   

/*int directx_getpixel(struct BITMAP *bmp, int x, int y);
void directx_putpixel(struct BITMAP *bmp, int x, int y, int color);
void directx_vline(struct BITMAP *bmp, int x, int y1, int y2, int color);
void directx_hline(struct BITMAP *bmp, int x1, int y, int x2, int color);
void directx_line(struct BITMAP *bmp, int x1, int y1, int x2, int y2, int color);
void directx_rectfill(struct BITMAP *bmp, int x1, int y1, int x2, int y2, int color);
int  directx_triangle(struct BITMAP *bmp, int x1, int y1, int x2, int y2, int x3, int y3, int color);
void directx_draw_sprite(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_256_sprite(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_sprite_v_flip(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_sprite_h_flip(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_sprite_vh_flip(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_trans_sprite(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y);
void directx_draw_lit_sprite(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y, int color);
void directx_draw_rle_sprite(struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y);
void directx_draw_trans_rle_sprite(struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y);
void directx_draw_lit_rle_sprite(struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y, int color);
void directx_draw_character(struct BITMAP *bmp, struct BITMAP *sprite, int x, int y, int color);
void directx_textout_fixed(struct BITMAP *bmp, void *f, int h, unsigned char *str, int x, int y, int color);
void directx_blit_from_memory(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_blit_to_memory(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_blit_to_self(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_blit_to_self_forward(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_blit_to_self_backward(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_masked_blit(struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height);
void directx_clear_to_color(struct BITMAP *bitmap, int color);
void directx_draw_sprite_end(void);
void directx_blit_end(void);
int directx_is_lost(struct BITMAP *bmp);
void directx_acquire(struct BITMAP *bmp);
void directx_release(struct BITMAP *bmp);*/
