/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *      DirectSound Allegro digital audio driver, by Stefan Schimanski (1Stein@gmx.de)
 *
 *      See readme.txt for copyright information.
 */


#include <stdlib.h>
#include <string.h>
#include <math.h>

/*#include "allegro.h"*/
#include "internal.h"
#include "windx.h"


static int directsound_init(int input, int voices);
static void directsound_exit(int input);
static int directsound_mixer_volume(int volume);
static int directsound_detect(int input);

static char directsound_desc[80] = "not initialised";


DIGI_DRIVER digi_directx =
{
   "DirectSound", 
   directsound_desc,
   0, 0, MIXER_MAX_SFX, MIXER_DEF_SFX,
   directsound_detect,
   directsound_init,
   directsound_exit,
   directsound_mixer_volume,
   _mixer_init_voice,
   _mixer_release_voice,
   _mixer_start_voice,
   _mixer_stop_voice,
   _mixer_loop_voice,
   NULL,
   NULL,
   _mixer_get_position,
   _mixer_set_position,
   _mixer_get_volume,
   _mixer_set_volume,
   _mixer_ramp_volume,
   _mixer_stop_volume_ramp,
   _mixer_get_frequency,
   _mixer_set_frequency,
   _mixer_sweep_frequency,
   _mixer_stop_frequency_sweep,
   _mixer_get_pan,
   _mixer_set_pan,
   _mixer_sweep_pan,
   _mixer_stop_pan_sweep,
   _mixer_set_echo,
   _mixer_set_tremolo,
   _mixer_set_vibrato,
   0, 0,
   NULL, NULL, NULL, NULL, NULL, NULL
};


/********************************************************/

#define PRIMARY

int directsound_buf_size;
int directsound_buf_smp_num;

int ds_freq, ds_16bit, ds_stereo;
struct DS_VOICE *directsound_voice;

#define BASE_BUF_SIZE 4096
#define BUF_DIVISOR 4


/********************************************************
 * Mix some samples                                     *
 ********************************************************/
int next_buffer = 0;
int v = 0;

void directsound_update_mixer()
{
    int SizeA, SizeB;
    void *BufA, *BufB;
    int pos = DS_GetVoicePos(directsound_voice);    
    int i;
    char *p;
    int current_buffer;    

    while (1)
    {
        // Get current play position in buffer blocks
        current_buffer = pos * BUF_DIVISOR / directsound_buf_size;

        // Make linear for better compare
        if (current_buffer<next_buffer) current_buffer += BUF_DIVISOR;

        // Mixer position must be half buffer size behind play
        if (next_buffer+BUF_DIVISOR/2<current_buffer) return;
        
        if (DS_Lock(directsound_voice, directsound_buf_size/BUF_DIVISOR*next_buffer, 
                directsound_buf_size/BUF_DIVISOR, &BufA, &SizeA, &BufB, &SizeB)==0)
        {
            p = (char *)BufA;
            _mix_some_samples(BufA, _dos_ds, TRUE);//FALSE);

            next_buffer++;
            if (next_buffer>=BUF_DIVISOR) next_buffer-=BUF_DIVISOR;
        }

        DS_Unlock(directsound_voice);
    }
        
}

/********************************************************/
static int directsound_timer()
{
    directsound_update_mixer();
}


/********************************************************
 *  Initialize driver									*
 ********************************************************/
static int directsound_init(int input, int voices)
{
	int c = 0;

	// init DirectSound
#ifdef PRIMARY
	if (DS_Init(1)!=0)
        return -1;
#else
    if (DS_Init(0)!=0)
        return -1;
#endif
		  
	digi_directx.voices = voices;

    // Get DirectX capabilities
	DS_GetCaps(&ds_freq, &ds_16bit, &ds_stereo);

    if (DS_IsPrimaryPossible())
    {
        // Create voice
        directsound_voice = DS_CreatePrimaryVoice(ds_16bit, ds_stereo, ds_freq);
        if (directsound_voice==NULL) return -1;

        directsound_buf_size = directsound_voice->Size;
        directsound_buf_smp_num = directsound_buf_size / (ds_16bit?2:1) / (ds_stereo?2:1);    
    } else
    {
        // find buffer size     
        if (ds_freq < 15000)
        {
            ds_freq = 11906;
            directsound_buf_smp_num = BASE_BUF_SIZE*1;
        } else 
        if (ds_freq < 20000)
        {
            ds_freq = 16129;
            directsound_buf_smp_num = BASE_BUF_SIZE*1;
        } else 
        if (ds_freq < 40000)
        {
            ds_freq = 22727;
            directsound_buf_smp_num = BASE_BUF_SIZE*2;
        } else
        {
            ds_freq = 45454;
            directsound_buf_smp_num = BASE_BUF_SIZE*4;
        }
    
        directsound_buf_size = directsound_buf_smp_num * (ds_16bit?2:1) * (ds_stereo?2:1);

        // Create voice
        directsound_voice = DS_CreateVoice(ds_16bit, ds_stereo, ds_freq, directsound_buf_size);
        if (directsound_voice==NULL) return -1;
    }

    
    ds_freq = directsound_voice->Freq;
    ds_16bit = directsound_voice->Is16Bit;
    ds_stereo = directsound_voice->IsStereo;

    // Start mixer
    if (_mixer_init(directsound_buf_smp_num/BUF_DIVISOR*(ds_stereo?2:1),
            ds_freq, ds_stereo, ds_16bit, &digi_directx.voices)!=0)
      return -1;    
        
    next_buffer = 0;
    directsound_update_mixer();

    // Start looping
    DS_AdjustVoice(directsound_voice, 1);
    DS_StartVoice(directsound_voice);

    // Install timer proc
	install_int(directsound_timer, 1000 * directsound_buf_smp_num / 2 / ds_freq /  BUF_DIVISOR);
	
	return 0;
}


/********************************************************
 *  Close DirectX										*
 ********************************************************/
static void directsound_exit(int input)
{	
	// Remove timer proc
	remove_int(directsound_timer);

    // Destroy voice    
    DS_StopVoice(directsound_voice);
    DS_DestroyVoice(directsound_voice);

    _mixer_exit();

	DS_Done();
}


/********************************************************
 *  Set DirectX mixer									*
 ********************************************************/
static int directsound_mixer_volume(int volume)
{
	DS_Volume(volume);
}


/********************************************************
 *  Detect DirectSound									*
 ********************************************************/
static int directsound_detect(int input)
{
	if (input)
      return FALSE;
  
	return DS_Detect();
}
