/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *      DirectSound Allegro digital audio driver, by Stefan Schimanski (1Stein@gmx.de)
 *
 *      See readme.txt for copyright information.
 */

#define INITGUID
#define DIRECTSOUND_VERSION 0x0300

#include <stdlib.h>
#include <string.h>
#include <math.h>

#define WIN32_LEAN_AND_MEAN
#define BITMAP WINDOWS_BITMAP
#include <windows.h>
#include <mmsystem.h>
#include <mmreg.h>
#include <dsound.h>
#include "win/windx.h"
#include "win/winintrn.h"
#undef BITMAP
#undef RGB

/*#include "allegro.h"*/
#include "internal.h"


static int directsound_init(int input, int voices);
static void directsound_exit(int input);
static int directsound_mixer_volume(int volume);
static int directsound_detect(int input);

static void directsound_init_voice(int voice, SAMPLE *sample);
static void directsound_release_voice(int voice);
static void directsound_start_voice(int voice);
static void directsound_stop_voice(int voice);
static void directsound_loop_voice(int voice, int playmode);
static void *directsound_lock_voice(int voice, int start, int end);
static void directsound_unlock_voice(int voice);

static int directsound_get_position(int voice);
static void directsound_set_position(int voice, int position);

static int directsound_get_volume(int voice);
static void directsound_set_volume(int voice, int volume);
static void directsound_ramp_volume(int voice, int time, int endvol);
static void directsound_stop_volume_ramp(int voice);
   
static int directsound_get_frequency(int voice);
static void directsound_set_frequency(int voice, int frequency);
static void directsound_sweep_frequency(int voice, int time, int endfreq);
static void directsound_stop_frequency_sweep(int voice);
   
static int directsound_get_pan(int voice);
static void directsound_set_pan(int voice, int pan);
static void directsound_sweep_pan(int voice, int time, int endpan);
static void directsound_stop_pan_sweep(int voice);

static void directsound_set_echo(int voice, int strength, int delay);
static void directsound_set_tremolo(int voice, int rate, int depth);
static void directsound_set_vibrato(int voice, int rate, int depth);

static char directsound_desc[80] = "not initialised";


DIGI_DRIVER digi_directx =
{
   "DirectSound", 
   directsound_desc,
   0, 0, MIXER_MAX_SFX, MIXER_DEF_SFX,
   directsound_detect,
   directsound_init,
   directsound_exit,
   directsound_mixer_volume,
   directsound_init_voice,
   directsound_release_voice,
   directsound_start_voice,
   directsound_stop_voice,
   directsound_loop_voice,
   directsound_lock_voice,
   directsound_unlock_voice,
   directsound_get_position,
   directsound_set_position,
   directsound_get_volume,
   directsound_set_volume,
   directsound_ramp_volume,
   directsound_stop_volume_ramp,
   directsound_get_frequency,
   directsound_set_frequency,
   directsound_sweep_frequency,
   directsound_stop_frequency_sweep,
   directsound_get_pan,
   directsound_set_pan,
   directsound_sweep_pan,
   directsound_stop_pan_sweep,
   directsound_set_echo,
   directsound_set_tremolo,
   directsound_set_vibrato,
   0, 0,
   NULL, NULL, NULL, NULL, NULL, NULL
};


/********************************************************/


#define DIRECTSOUND_TIMER_PERIOD 50

LPDIRECTSOUND lpDirectSound = NULL;

int ds_freq, ds_bits, ds_stereo;

struct DIRECTSOUND_VOICE
{
    int freq, pan, vol;
    int dfreq, dpan, dvol;
    int target_freq, target_pan, target_vol;
    int looping;
 
    void *lock_buf_a;
    void *lock_buf_b;
    int lock_size_a, lock_size_b;
 
    LPDIRECTSOUNDBUFFER ds_buffer;
};

struct DIRECTSOUND_VOICE *ds_voices;


/********************************************************/

char *ds_err(HRESULT err)
{
    char *s;

    switch (err)
    {
        case DS_OK:                 s="DS_OK"; break;
        case DSERR_ALLOCATED:       s="DSERR_ALLOCATED"; break;
        case DSERR_BADFORMAT:       s="DSERR_BADFORMAT"; break;
        case DSERR_INVALIDPARAM:    s="DSERR_INVALIDPARAM"; break;
        case DSERR_NOAGGREGATION:   s="DSERR_NOAGGREGATION"; break;
        case DSERR_OUTOFMEMORY:     s="DSERR_OUTOFMEMORY"; break;
        case DSERR_UNINITIALIZED:   s="DSERR_UNINITIALIZED"; break;
        case DSERR_UNSUPPORTED:     s="DSERR_UNSUPPORTED"; break;
        default: s="DSERR_UNKNOWN";
    }

    return s;
}


static int directsound_timer()
{
	int c;
	struct DIRECTSOUND_VOICE *v;

	// Check all voices
	for (c=0; c<digi_directx.voices; c++)
	{
		v = ds_voices + c;

		// If active
		if (v->ds_buffer)
		{
			// If changing volume
			if (v->dvol)
			{
				// Change volume
				v->vol += v->dvol;

				// If target reached -> stop change
				if ( (v->dvol>0 && v->vol>v->target_vol) ||
					 (v->dvol<0 && v->vol<v->target_vol)) 
				{
					v->vol = v->target_vol;
					v->dvol = 0;
				}

				// Change DirectSound voice
				directsound_set_volume(c, v->vol);
			}


			// If changing frequency
			if (v->dfreq)
			{
				// Change frequency
				v->freq += v->dfreq;

				// If target reached -> stop change
				if ( (v->dfreq>0 && v->freq>v->target_freq) ||
					 (v->dfreq<0 && v->freq<v->target_freq))
				{
					v->freq = v->target_freq;
					v->dfreq = 0;
				}

				// Change DirectSound voice
				directsound_set_frequency(c, v->freq);
			}

			// If changing pan
			if (v->dpan)
			{
				// Change pan
				v->pan += v->dpan;

				// If target reached -> stop change
				if ( (v->dpan>0 && v->pan>v->target_pan) ||
					 (v->dpan<0 && v->freq<v->target_pan))
				{
					v->pan = v->target_pan;
					v->dpan = 0;
				}

				// Change DirectSound voice
				directsound_set_pan(c, v->pan);
			}

		}
	}

    return 0;
}


/********************************************************
 *  Initialize driver									*
 ********************************************************/
static int directsound_init(int input, int voices)
{
    HRESULT hr;
    DSCAPS dscaps;
    int v;

    digi_directx.voices = voices;
	
    // init DirectSound
    DBEG("directsound_init");

	// Initialize DirectSound interface
    D3("directsound_init - DirectSoundCreate");
	hr = DirectSoundCreate(NULL, &lpDirectSound, NULL);
    if (hr!=DS_OK)
    {
        D4("directsound_init - Can't create DirectSound interface hr=%x", hr);
        goto Error;
    }

	// Set cooperative level
    D3("directsound_init - SetCooperativeLevel");
    hr = lpDirectSound->lpVtbl->SetCooperativeLevel(lpDirectSound, hMainWnd, DSSCL_PRIORITY);
    if (hr==DS_OK)
    {
        D3("directsound_init - Coop level set to PRIORITY");
    } else
    {
        D3("directsound_init - Can't set cooperative level hr=%s", ds_err(hr));

        goto Error;
    }

   	// Get hardware capabilities
    D3("directsound_init - GetCaps");
	dscaps.dwSize = sizeof(DSCAPS);
	hr = lpDirectSound->lpVtbl->GetCaps(lpDirectSound, &dscaps);
	if (hr!=DS_OK)
    {
        D3("directsound_init - Can't get caps hr=%x", hr);
        goto Error;
    }

    if (dscaps.dwFlags & DSCAPS_PRIMARY16BIT)
	    ds_bits=16; else ds_bits=8;

	if (dscaps.dwFlags & DSCAPS_PRIMARYSTEREO)
    	ds_stereo=1; else ds_stereo=0;

	if (dscaps.dwMaxSecondarySampleRate>44000)
        ds_freq=44100; else ds_freq=22050;

    D5("directsound_init: %u bits, %s, %uHz", ds_bits, ds_stereo?"stereo":"mono", ds_freq);


    // Initialize physica voices	
    ds_voices = (struct DIRECTSOUND_VOICE* )malloc(sizeof(struct DIRECTSOUND_VOICE)*voices);
    for (v=0; v<digi_directx.voices; v++)
	{
		ds_voices[v].ds_buffer = NULL;
	}
	
	// Install timer proc
	install_int(directsound_timer, MSEC_TO_TIMER(DIRECTSOUND_TIMER_PERIOD));

    DEND("directsound_init");
	return 0;

Error:
    D3("directsound_init - Error");
    directsound_exit(0);
    return -1;
}

    
    

/********************************************************
 *  Close DirectX										*
 ********************************************************/
static void directsound_exit(int input)
{	
    int v;

    DBEG("directsound_exit");

	// Remove timer proc
	remove_int(directsound_timer);

    // Destroy physical voices
    for (v=0; v<digi_directx.voices; v++)
    {        
        directsound_release_voice(v);
    }

    free(ds_voices);
    ds_voices = NULL;

    // Shutdown DirectSound
    if (lpDirectSound)
	{
		lpDirectSound->lpVtbl->Release(lpDirectSound);
		lpDirectSound = NULL;
	}
	
	DEND("directsound_exit");
}


/********************************************************
 *  Set DirectX mixer									*
 ********************************************************/
static int directsound_mixer_volume(int volume)
{	
    return 0;
}


/********************************************************
 *  Detect DirectSound									*
 ********************************************************/
static int directsound_detect(int input)
{
    HRESULT hr;

	if (input)
      return 0;
  
	DBEG("DS_Detect");

    if (lpDirectSound==NULL)
    {
        D3("DS_Detect - DirectSound not initialiased");

        // Initialize DirectSound interface
        D3("DS_Detect - Creating DirectSound interface");

    	hr = DirectSoundCreate(NULL, &lpDirectSound, NULL);
	    if (hr!=DS_OK)
        {
            DEND("DS_Detect - Creation failed");
            return 0;
        }

        D3("DS_Detect - Creation successfull");

        // Release DirectSound
        D3("DS_Detect - Releasing DirectSound");
        lpDirectSound->lpVtbl->Release(lpDirectSound);
		lpDirectSound = NULL;
    }

    DEND("DS_Detect");
    
    return 1;
}


/********************************************************/
static void directsound_init_voice(int voice, SAMPLE *sample)
{
    PCMWAVEFORMAT pcmwf; 
    DSBUFFERDESC dsbdesc; 
    HRESULT hr;
    LPVOID BufA; 
    DWORD SizeA; 
    LPVOID BufB; 
    DWORD SizeB; 
    LPDIRECTSOUNDBUFFER lpSoundBuf;

    ds_voices[voice].vol = 255;
    ds_voices[voice].pan = 128;
    ds_voices[voice].freq = sample->freq;
    ds_voices[voice].dfreq = 0;
    ds_voices[voice].dpan = 0;
    ds_voices[voice].dvol = 0;
    ds_voices[voice].looping = 0;
    
    // Set up wave format structure. 
    memset(&pcmwf, 0, sizeof(PCMWAVEFORMAT)); 
    pcmwf.wf.wFormatTag = WAVE_FORMAT_PCM; 
    pcmwf.wf.nChannels = sample->stereo?2:1;
    pcmwf.wf.nSamplesPerSec = sample->freq; 
    pcmwf.wBitsPerSample = sample->bits;
    pcmwf.wf.nBlockAlign = sample->bits*(sample->stereo?2:1)/8;
    pcmwf.wf.nAvgBytesPerSec = 
        pcmwf.wf.nSamplesPerSec * pcmwf.wf.nBlockAlign; 
    

    // Set up DSBUFFERDESC structure. 
    memset(&dsbdesc, 0, sizeof(DSBUFFERDESC)); // Zero it out. 
    dsbdesc.dwSize = sizeof(DSBUFFERDESC); 

    // Need default controls (pan, volume, frequency). 
    dsbdesc.dwFlags = DSBCAPS_CTRLDEFAULT; 
    
    // 3-second buffer. 
    dsbdesc.dwBufferBytes = sample->len * (sample->bits/8) * (sample->stereo?2:1);
    dsbdesc.lpwfxFormat = (LPWAVEFORMATEX)&pcmwf; 

    // Create buffer. 
    hr = lpDirectSound->lpVtbl->CreateSoundBuffer(lpDirectSound, 
        &dsbdesc, &lpSoundBuf, NULL); 
    if(hr==DS_OK)
    { 
        ds_voices[voice].ds_buffer = lpSoundBuf;
    } else
    {           
        D3("directsound_init_voice - CreateSoundBuffer");
        D3("directsound_init_voice - failed (%s)", ds_err(hr));
        D3("directsound_init_voice - %d Hz, %s, %d bits", sample->freq, sample->stereo?"stereo":"mono", sample->bits);
        return;
    }

    // Copy sample contents to buffer
    hr = lpSoundBuf->lpVtbl->Lock(lpSoundBuf, 0, 
        sample->len * (sample->bits/8) * (sample->stereo?2:1),
        &BufA, &SizeA, &BufB, &SizeB, 0);        
 
    // If DSERR_BUFFERLOST is returned, restore and retry lock
    if(hr==DSERR_BUFFERLOST)
    { 
        D3("directsound_init_voice - DSERR_BUFFERLOST");
        lpSoundBuf->lpVtbl->Restore(lpSoundBuf);
        hr = lpSoundBuf->lpVtbl->Lock(lpSoundBuf, 0, 
           sample->len * (sample->bits/8) * (sample->stereo?2:1),
            &BufA, &SizeA, &BufB, &SizeB, 0);        
    }
    
    if(hr!=DS_OK)
    {
        D3("directsound_init_voice - Can't lock");
        directsound_release_voice(voice);
        return;
    }

    // Write to pointers. 
    CopyMemory(BufA, sample->data, SizeA);
    if(BufB!=NULL) CopyMemory(BufB, (char*)(sample->data)+SizeA, SizeB); 
         
    // Release the data back to DirectSound. 
    hr = lpSoundBuf->lpVtbl->Unlock(lpSoundBuf, BufA, SizeA, BufB, SizeB);        
    if(hr!=DS_OK)
    { 
        D3("directsound_init_voice - Can't unlock");
    }
}


/********************************************************/
static void directsound_release_voice(int voice)
{
    if (ds_voices[voice].ds_buffer)
    {
        ds_voices[voice].ds_buffer->lpVtbl->Release(ds_voices[voice].ds_buffer);
        ds_voices[voice].ds_buffer = NULL;
    }
}


/********************************************************/
static void directsound_start_voice(int voice)
{    
    if (ds_voices[voice].ds_buffer)
    {
        ds_voices[voice].ds_buffer->lpVtbl->Play(ds_voices[voice].ds_buffer, 
            0, 0, ds_voices[voice].looping?DSBPLAY_LOOPING:0);
    }
}


/********************************************************/
static void directsound_stop_voice(int voice)
{
    if (ds_voices[voice].ds_buffer)
    {
        ds_voices[voice].ds_buffer->lpVtbl->Stop(ds_voices[voice].ds_buffer);
    }
}


/********************************************************/
static void directsound_loop_voice(int voice, int playmode)
{
    switch (playmode)
	{
	    case PLAYMODE_LOOP:
            ds_voices[voice].looping = 1;
	    	break;
    
        case PLAYMODE_PLAY:	
            ds_voices[voice].looping = 0;
	        break;	    
    }
}


/********************************************************/
static void *directsound_lock_voice(int voice, int start, int end)
{
    void *BufA;
    void *BufB;
    int SizeA, SizeB;
    HRESULT hr;
    
    hr = ds_voices[voice].ds_buffer->lpVtbl->Lock(ds_voices[voice].ds_buffer,
            start, end-start, &BufA, &SizeA, &BufB, &SizeB, 0);
    if (hr!=DS_OK)
    {
        D3("directsound_lock_voice - failed (%s)", ds_err(hr));
        return NULL;
    }

    ds_voices[voice].lock_buf_a = BufA;
    ds_voices[voice].lock_size_a = SizeA;
    ds_voices[voice].lock_buf_b = BufB;
    ds_voices[voice].lock_size_b = SizeB;

    if (BufB!=NULL)
    {
        directsound_unlock_voice(voice);
        return NULL;
    }

    return BufA;        
}


/********************************************************/
static void directsound_unlock_voice(int voice)
{
    HRESULT hr;

    if (ds_voices[voice].ds_buffer && ds_voices[voice].lock_buf_a)
    {
        hr = ds_voices[voice].ds_buffer->lpVtbl->Unlock(ds_voices[voice].ds_buffer,
                ds_voices[voice].lock_buf_a,
                ds_voices[voice].lock_size_a,
                ds_voices[voice].lock_buf_b,
                ds_voices[voice].lock_size_b);
                
        if (hr!=DS_OK)
        {
            D3("directsound_unlock_voice - failed (%s)", ds_err(hr));
        }
    }
}


/********************************************************/
static int directsound_get_position(int voice)
{
    HRESULT hr;
    int PlayCursor;
    int WriteCursor;

    if (ds_voices[voice].ds_buffer)
    {
        hr = ds_voices[voice].ds_buffer->lpVtbl->GetCurrentPosition(ds_voices[voice].ds_buffer,
            &PlayCursor, &WriteCursor);
        if (hr!=DS_OK)
        {
            D3("directsound_get_position - failed (%s)", ds_err(hr));
            return 0;
        }

        return PlayCursor;
    } else
        return 0;
}


/********************************************************/
static void directsound_set_position(int voice, int position)
{
    HRESULT hr;

    if (ds_voices[voice].ds_buffer)
    {
        hr = ds_voices[voice].ds_buffer->lpVtbl->SetCurrentPosition(ds_voices[voice].ds_buffer,
            position);
        if (hr!=DS_OK)
        {
            D3("directsound_set_position - failed (%s)", ds_err(hr));            
        }        
    }
}


/********************************************************/
static int directsound_get_volume(int voice)
{   
    return ds_voices[voice].vol;
}


/********************************************************/
static void directsound_set_volume(int voice, int volume)
{  
    ds_voices[voice].vol = volume;
    
    if (ds_voices[voice].ds_buffer)
    {
        int ds_vol = volume*(DSBVOLUME_MAX-DSBVOLUME_MIN)/256;            
        ds_voices[voice].ds_buffer->lpVtbl->SetVolume(ds_voices[voice].ds_buffer, ds_vol);
    }
}


/********************************************************/
static void directsound_ramp_volume(int voice, int time, int endvol)
{
	int d, dvol;

	if (time>0)
	{
		d = endvol - ds_voices[voice].vol;
		dvol = d * DIRECTSOUND_TIMER_PERIOD / time;
	} else dvol = 255;

	ds_voices[voice].target_vol = endvol;
	ds_voices[voice].dvol = dvol;
}


/********************************************************/
static void directsound_stop_volume_ramp(int voice)
{
	ds_voices[voice].dvol = 0;
}


/********************************************************/
static int directsound_get_frequency(int voice)
{
     return ds_voices[voice].freq;
}


/********************************************************/
static void directsound_set_frequency(int voice, int frequency)
{
    ds_voices[voice].freq = frequency;
    
    if (ds_voices[voice].ds_buffer)
    {
        ds_voices[voice].ds_buffer->lpVtbl->SetFrequency(ds_voices[voice].ds_buffer, frequency);
    }
}


/********************************************************/
static void directsound_sweep_frequency(int voice, int time, int endfreq)
{
	int d, dfreq;

	if (time>0)
	{
		d = endfreq - ds_voices[voice].freq;
		dfreq = d * DIRECTSOUND_TIMER_PERIOD / time;
	} else dfreq = 255;

	ds_voices[voice].target_freq = endfreq;
	ds_voices[voice].dfreq = dfreq;
}


/********************************************************/
static void directsound_stop_frequency_sweep(int voice)
{
	ds_voices[voice].dfreq = 0;
}


/********************************************************/
static int directsound_get_pan(int voice)
{
     return ds_voices[voice].pan;    
}


/********************************************************/
static void directsound_set_pan(int voice, int pan)
{
    ds_voices[voice].pan = pan;
    
    if (ds_voices[voice].ds_buffer)
    {
        int ds_pan = ABS(DSBPAN_LEFT)+ABS(DSBPAN_RIGHT) / 256;
        ds_pan += DSBPAN_LEFT;
        ds_voices[voice].ds_buffer->lpVtbl->SetPan(ds_voices[voice].ds_buffer, ds_pan);
    }
}


/********************************************************/
static void directsound_sweep_pan(int voice, int time, int endpan)
{
	int d, dpan;

	if (time>0)
	{
		d = endpan - ds_voices[voice].pan;
		dpan = d * DIRECTSOUND_TIMER_PERIOD / time;
	} else dpan = 255;

	ds_voices[voice].target_pan = endpan;
	ds_voices[voice].dpan = dpan;
}


/********************************************************/
static void directsound_stop_pan_sweep(int voice)
{
	ds_voices[voice].dpan = 0;
}


/********************************************************/
static void directsound_set_echo(int voice, int strength, int delay)
{
}


/********************************************************/
static void directsound_set_tremolo(int voice, int rate, int depth)
{
}


/********************************************************/
static void directsound_set_vibrato(int voice, int rate, int depth)
{
}
