/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 * 
 *      Win32 mouse driver, by Stefan Schimanski (1Stein@gmx.de)
 *
 *      3-button support added by Fabian Nunez.
 *
 *      Mark Wodrich added double-buffered drawing of the mouse pointer and
 *      the set_mouse_sprite_focus() function.
 *
 *      See readme.txt for copyright information.
 */


#ifndef WIN32
#error This file should only be used by the Win32 version of Allegro
#endif

#include <stdlib.h>
#include <string.h>
#include <dos.h>

/*#include "allegro.h"*/
#include "internal.h"
#include "windx.h"


volatile int mouse_x = 0;           /* mouse x pos */
volatile int mouse_y = 0;           /* mouse y pos */
volatile int mouse_b = 0;           /* mouse button state */
volatile int mouse_pos = 0;         /* unified position */

static int _mouse_x = 0;            /* internal position */
static int _mouse_y = 0;
static int _mouse_b = 0;

static int mickeymode = FALSE;      /* which input mode are we using? */
static int ntmode = FALSE;

static int mouse_mx = 0;            /* internal position, in mickeys */
static int mouse_my = 0;

static int mouse_sx = 2;            /* mickey -> pixel scaling factor */
static int mouse_sy = 2;

static int mouse_minx = 0;          /* mouse range */
static int mouse_miny = 0;
static int mouse_maxx = 319;
static int mouse_maxy = 199;

static int mymickey_x = 0;          /* for get_mouse_mickeys() */
static int mymickey_y = 0;
static int mymickey_ox = 0; 
static int mymickey_oy = 0;

int freeze_mouse_flag = FALSE;

void (*mouse_callback)(int flags) = NULL;

int _mouse_installed = FALSE;

int _mouse_x_focus = 1;             /* focus point in mouse sprite */
int _mouse_y_focus = 1;


#define MICKEY_TO_COORD_X(n)        ((n) / mouse_sx)
#define MICKEY_TO_COORD_Y(n)        ((n) / mouse_sy)

#define COORD_TO_MICKEY_X(n)        ((n) * mouse_sx)
#define COORD_TO_MICKEY_Y(n)        ((n) * mouse_sy)


#define CLEAR_MICKEYS()             \
   {                                \
      mymickey_ox = 0;          \
      mymickey_oy = 0;          \
   }


static unsigned char mouse_pointer_data[256] =
{
   1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 2, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 2, 2, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 2, 2, 2, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 2, 2, 2, 2, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 
   1, 2, 2, 2, 2, 2, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 2, 1, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
   1, 2, 1, 0, 1, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
   0, 1, 0, 0, 1, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
   0, 0, 0, 0, 0, 1, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 
   0, 0, 0, 0, 0, 1, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 
   0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0
};


BITMAP *_mouse_pointer = NULL;               /* default mouse pointer */

BITMAP *_mouse_sprite = NULL;                /* mouse pointer */

BITMAP *_mouse_screen = NULL;                /* where to draw the pointer */

static BITMAP *scared_screen = NULL;         /* for unscare_mouse() */

static int got_hw_cursor = FALSE;            /* hardware pointer available? */

static int mx, my;                           /* previous mouse position */
static BITMAP *ms = NULL;                    /* previous screen data */
static BITMAP *mtemp = NULL;                 /* double-buffer drawing area */

int _mouse_width, _mouse_height;             /* size of mouse sprite */

static int mouse_semaphore = FALSE;          /* reentrant interrupt? */



/* draw_mouse_doublebuffer:
 *  Eliminates mouse-cursor flicker by using an off-screen buffer for
 *  updating the cursor, and blitting only the final screen image.
 *  newx and newy contain the new cursor position, and mx and my are 
 *  assumed to contain previous cursor pos. This routine is called if 
 *  mouse cursor is to be erased and redrawn, and the two position overlap.
 */
static void draw_mouse_doublebuffer(int newx, int newy) 
{
   int x1, y1, w, h;

   /* grab bit of screen containing where we are and where we'll be */
   x1 = MIN(mx, newx);
   x1 -= _mouse_x_focus;
   y1 = MIN(my, newy);
   y1 -= _mouse_y_focus;

   /* get width of area */
   w = MAX(mx,newx) - x1 + _mouse_sprite->w+1;
   h = MAX(my,newy) - y1 + _mouse_sprite->h+1;

   /* make new co-ords relative to 'mtemp' bitmap co-ords */
   newx -= _mouse_x_focus+x1;
   newy -= _mouse_y_focus+y1;

   /* save screen image in 'mtemp' */
   blit(_mouse_screen, mtemp, x1, y1, 0, 0, w, h);

   /* blit saved image in 'ms' to corect place in this buffer */
   blit(ms, mtemp, 0, 0, mx-_mouse_x_focus-x1, my-_mouse_y_focus-y1,
	 _mouse_sprite->w, _mouse_sprite->h);

   /* draw mouse at correct place in 'mtemp' */
   blit(mtemp, ms, newx, newy, 0, 0, _mouse_sprite->w, _mouse_sprite->h);
   draw_sprite(mtemp, _mouse_sprite, newx, newy);

   /* blit 'mtemp' to screen */
   blit(mtemp, _mouse_screen, 0, 0, x1, y1, w, h);
}


/* draw_mouse:
 *  Mouse pointer drawing routine. If remove is set, deletes the old mouse
 *  pointer. If add is set, draws a new one.
 */
static void draw_mouse(int remove, int add)
{
   int normal_draw = (remove ^ add); 

   int newmx = mouse_x;
   int newmy = mouse_y;

   int cf = _mouse_screen->clip;
   int cl = _mouse_screen->cl;
   int cr = _mouse_screen->cr;
   int ct = _mouse_screen->ct;
   int cb = _mouse_screen->cb;

   _mouse_screen->clip = TRUE;
   _mouse_screen->cl = _mouse_screen->ct = 0;
   _mouse_screen->cr = _mouse_screen->w;
   _mouse_screen->cb = _mouse_screen->h;

   if (!normal_draw) {
      if ((newmx <= mx-_mouse_width) || (newmx >= mx+_mouse_width) ||
	  (newmy <= my-_mouse_height) || (newmy >= my+_mouse_height))
	 normal_draw = 1;
   }

   acquire_bitmap(_mouse_screen);
   if (normal_draw) {
      if (remove)
	 blit(ms, _mouse_screen, 0, 0, mx-_mouse_x_focus, my-_mouse_y_focus, _mouse_sprite->w, _mouse_sprite->h);

      if (add) {
	 blit(_mouse_screen, ms, newmx-_mouse_x_focus, newmy-_mouse_y_focus, 0, 0, _mouse_sprite->w, _mouse_sprite->h);
	 draw_sprite(_mouse_screen, _mouse_sprite, newmx-_mouse_x_focus, newmy-_mouse_y_focus);
      }
   }
   else 
      draw_mouse_doublebuffer(newmx, newmy);
   release_bitmap(_mouse_screen);

   mx = newmx;
   my = newmy;

   _mouse_screen->clip = cf;
   _mouse_screen->cl = cl;
   _mouse_screen->cr = cr;
   _mouse_screen->ct = ct;
   _mouse_screen->cb = cb;
}


/* update_mouse:
 *  Updates the mouse position variables with new values.
 */
static void update_mouse()
{
   int flags = 0;

   if ((mouse_x != _mouse_x) || 
       (mouse_y != _mouse_y) || 
       (mouse_b != _mouse_b)) {

      if (mouse_callback) {

	 if ((mouse_x != _mouse_x) || (mouse_y != _mouse_y))
	    flags |= MOUSE_FLAG_MOVE;

	 if ((_mouse_b & 1) && !(mouse_b & 1))
	    flags |= MOUSE_FLAG_LEFT_DOWN;
	 else if (!(_mouse_b & 1) && (mouse_b & 1))
	    flags |= MOUSE_FLAG_LEFT_UP;

	 if ((_mouse_b & 2) && !(mouse_b & 2))
	    flags |= MOUSE_FLAG_RIGHT_DOWN;
	 else if (!(_mouse_b & 2) && (mouse_b & 2))
	    flags |= MOUSE_FLAG_RIGHT_UP;

	 if ((_mouse_b & 4) && !(mouse_b & 4))
	    flags |= MOUSE_FLAG_MIDDLE_DOWN;
	 else if (!(_mouse_b & 4) && (mouse_b & 4))
	    flags |= MOUSE_FLAG_MIDDLE_UP;

	 mouse_x = _mouse_x;
	 mouse_y = _mouse_y;
	 mouse_b = _mouse_b;
	 mouse_pos = ((mouse_x & 0xFFFF) << 16) | (mouse_y & 0xFFFF);

	 mouse_callback(flags);
      }
      else {
	 mouse_x = _mouse_x;
	 mouse_y = _mouse_y;
	 mouse_b = _mouse_b;
	 mouse_pos = ((mouse_x & 0xFFFF) << 16) | (mouse_y & 0xFFFF);
      }
   }
}


/* mouse_move:
 *  Do we need to redraw the mouse pointer?
 */
static void mouse_move() 
{
   if (mouse_semaphore)
      return;

   mouse_semaphore = TRUE;
   
   if (!freeze_mouse_flag) {
      update_mouse();

      if ((_mouse_screen) && ((mx != mouse_x) || (my != mouse_y))) {
	 if (gfx_capabilities & GFX_HW_CURSOR)
	    gfx_driver->move_mouse(mx=mouse_x, my=mouse_y);
	 else
	    draw_mouse(TRUE, TRUE);
      }
   }

   mouse_semaphore = FALSE;
}


/* mouseint:
 *  Mouse movement callback for the int 33 mouse driver.
 */

void mouseint(int dx, int dy, int px, int py, int but)
{
   _mouse_b = but;
   
   mymickey_x += dx;
   mymickey_y += dy;

   mymickey_ox = dx;
   mymickey_oy = dy;
   
   mouse_mx = dx;
   mouse_my = dy;

   _mouse_x = px;
   _mouse_y = py;   

   if ((_mouse_x < mouse_minx) || (_mouse_x > mouse_maxx) ||
	  (_mouse_y < mouse_miny) || (_mouse_y > mouse_maxy)) {

      _mouse_x = MID(mouse_minx, _mouse_x, mouse_maxx);
      _mouse_y = MID(mouse_miny, _mouse_y, mouse_maxy);

      mouse_mx = COORD_TO_MICKEY_X(_mouse_x);
      mouse_my = COORD_TO_MICKEY_Y(_mouse_y);

      CLEAR_MICKEYS();
   }
      
   if (!_mouse_screen)
      update_mouse();
}



/* set_hw_cursor:
 *  Tries to download the current mouse sprite to the video driver, for
 *  use as a hardware cursor.
 */
static void set_hw_cursor()
{
   got_hw_cursor = FALSE;

   if ((gfx_driver) && (gfx_driver->set_mouse_sprite))
      if (gfx_driver->set_mouse_sprite(_mouse_sprite, _mouse_x_focus, _mouse_y_focus) == 0)
	 got_hw_cursor = TRUE;
}



/* set_mouse_sprite:
 *  Sets the sprite to be used for the mouse pointer. If the sprite is
 *  NULL, restores the default arrow.
 */
void set_mouse_sprite(struct BITMAP *sprite)
{
   BITMAP *old_mouse_screen = _mouse_screen;

   if (_mouse_screen)
      show_mouse(NULL);

   if (sprite)
      _mouse_sprite = sprite;
   else
      _mouse_sprite = _mouse_pointer;

   lock_bitmap(_mouse_sprite);

   /* make sure the ms bitmap is big enough */
   if ((!ms) || (ms->w < _mouse_sprite->w) || (ms->h < _mouse_sprite->h)) {
      if (ms) {
	 destroy_bitmap(ms);
	 destroy_bitmap(mtemp);
      }
      ms = create_bitmap(_mouse_sprite->w, _mouse_sprite->h);
      lock_bitmap(ms);

      mtemp = create_bitmap(_mouse_sprite->w*2, _mouse_sprite->h*2);
      lock_bitmap(mtemp);
   }

   _mouse_x_focus = _mouse_y_focus = 1;
   _mouse_width = _mouse_sprite->w;
   _mouse_height = _mouse_sprite->h;

   set_hw_cursor();

   if (old_mouse_screen)
      show_mouse(old_mouse_screen);
}



/* set_mouse_sprite_focus:
 *  Sets co-ordinate (x, y) in the sprite to be the mouse location.
 *  Call after set_mouse_sprite(). Doesn't redraw the sprite.
 */
void set_mouse_sprite_focus(int x, int y) 
{
   _mouse_x_focus = x;
   _mouse_y_focus = y;

   set_hw_cursor();
}



/* show_mouse:
 *  Tells Allegro to display a mouse pointer. This only works when the timer 
 *  module is active. The mouse pointer will be drawn onto the bitmap bmp, 
 *  which should normally be the hardware screen. To turn off the mouse 
 *  pointer, which you must do before you draw anything onto the screen, call 
 *  show_mouse(NULL). If you forget to turn off the mouse pointer when 
 *  drawing something, the SVGA bank switching code will become confused and 
 *  will produce garbage all over the screen.
 */
void show_mouse(BITMAP *bmp)
{
   remove_int(mouse_move);

   if (_mouse_screen) {
      if (gfx_capabilities & GFX_HW_CURSOR) {
	 gfx_driver->hide_mouse();
	 gfx_capabilities &= ~GFX_HW_CURSOR;
      }
      else
	 draw_mouse(TRUE, FALSE);
   }

   _mouse_screen = bmp;

   if (bmp) {
      if ((got_hw_cursor) && (bmp->vtable == &_screen_vtable))
	 if (gfx_driver->show_mouse(bmp, mx=mouse_x, my=mouse_y) == 0)
	    gfx_capabilities |= GFX_HW_CURSOR;

     DI_MoveMouse(mouse_x, mouse_y);

      if (!(gfx_capabilities & GFX_HW_CURSOR))
	 draw_mouse(FALSE, TRUE);

      if (_mouse_installed)
	 install_int(mouse_move, 20);
   }
   else {
      if (ntmode) 
	 install_int(mouse_move, 20);
      else
	 update_mouse();
   }
}



/* scare_mouse:
 *  Removes the mouse pointer prior to a drawing operation, if that is
 *  required (ie. noop if the mouse is on a memory bitmap, or a hardware
 *  cursor is in use). This operation can later be reversed by calling
 *  unscare_mouse().
 */
void scare_mouse()
{
   if ((is_same_bitmap(screen ,_mouse_screen)) &&
       (!(gfx_capabilities & GFX_HW_CURSOR))) {
      scared_screen = _mouse_screen;
      show_mouse(NULL);
   }
   else
      scared_screen = NULL;
}



/* unscare_mouse:
 *  Restores the original mouse state, after a call to scare_mouse().
 */
void unscare_mouse()
{
   if (scared_screen)
      show_mouse(scared_screen);

   scared_screen = NULL;
}



/* position_mouse:
 *  Moves the mouse to screen position x, y. This is safe to call even
 *  when a mouse pointer is being displayed.
 */
void position_mouse(int x, int y)
{
   int nowhere = 0;
   BITMAP *old_mouse_screen = _mouse_screen;

   if (_mouse_screen)
      show_mouse(NULL);

   if (!_mouse_installed) {
      mouse_x = _mouse_x = x;
      mouse_y = _mouse_y = y;
   }
   else
   {
      DISABLE();

      mouse_x = _mouse_x = x;
      mouse_y = _mouse_y = y;

      mouse_mx = COORD_TO_MICKEY_X(x);
      mouse_my = COORD_TO_MICKEY_Y(y);

      CLEAR_MICKEYS();

      get_mouse_mickeys(&nowhere, &nowhere);

      DI_MoveMouse(_mouse_x, _mouse_y);

      ENABLE();
   }
   
   if (old_mouse_screen)
      show_mouse(old_mouse_screen);
}



/* set_mouse_range:
 *  Sets the screen area within which the mouse can move. Pass the top left 
 *  corner and the bottom right corner (inclusive). If you don't call this 
 *  function the range defaults to (0, 0, SCREEN_W-1, SCREEN_H-1).
 */
void set_mouse_range(int x1, int y1, int x2, int y2)
{
//   __dpmi_regs r;
   BITMAP *old_mouse_screen = _mouse_screen;

   if (!_mouse_installed)
      return;

   if (_mouse_screen)
      show_mouse(NULL);

   //if (mickeymode) {
      mouse_minx = x1;
      mouse_miny = y1;
      mouse_maxx = x2;
      mouse_maxy = y2;

      DISABLE();

      mouse_x = _mouse_x = MID(mouse_minx, _mouse_x, mouse_maxx);
      mouse_y = _mouse_y = MID(mouse_miny, _mouse_y, mouse_maxy);

      mouse_mx = COORD_TO_MICKEY_X(_mouse_x);
      mouse_my = COORD_TO_MICKEY_Y(_mouse_y);

      CLEAR_MICKEYS();

      ENABLE();
   /*}
   else {
      r.x.ax = 7;
      r.x.cx = x1 * 8;
      r.x.dx = x2 * 8;
      __dpmi_int(0x33, &r);         // set horizontal range

      r.x.ax = 8;
      r.x.cx = y1 * 8;
      r.x.dx = y2 * 8;
      __dpmi_int(0x33, &r);         // set vertical range

      r.x.ax = 3; 
      __dpmi_int(0x33, &r);         // check the position
      mouse_x = _mouse_x = r.x.cx / 8;
      mouse_y = _mouse_y = r.x.dx / 8;
   }*/

   if (old_mouse_screen)
      show_mouse(old_mouse_screen);
}



/* set_mouse_speed:
 *  Sets the mouse speed. Larger values of xspeed and yspeed represent 
 *  slower mouse movement: the default for both is 2.
 */
void set_mouse_speed(int xspeed, int yspeed)
{
//   __dpmi_regs r;

   if (!_mouse_installed)
      return;

//   if (mickeymode) {
      DISABLE();

      mouse_sx = MAX(1, xspeed);
      mouse_sy = MAX(1, yspeed);

      mouse_mx = COORD_TO_MICKEY_X(_mouse_x);
      mouse_my = COORD_TO_MICKEY_Y(_mouse_y);

      CLEAR_MICKEYS();

      ENABLE();
/*   }
   else {
      r.x.ax = 15;
      r.x.cx = xspeed;
      r.x.dx = yspeed;
      __dpmi_int(0x33, &r); 
   }*/
}



/* get_mouse_mickeys:
 *  Measures the mickey count (how far the mouse has moved since the last
 *  call to this function).
 */
void get_mouse_mickeys(int *mickeyx, int *mickeyy)
{
//   __dpmi_regs r;

   if (!_mouse_installed) {
      *mickeyx = 0;
      *mickeyy = 0;
      return;
   }

//   if (mickeymode) {
      DISABLE();

      *mickeyx = mymickey_x;
      *mickeyy = mymickey_y;

      mymickey_x -= *mickeyx;
      mymickey_y -= *mickeyy;

      ENABLE();
/*   }
   else {
      r.x.ax = 11;
      __dpmi_int(0x33, &r); 

      *mickeyx = (signed short)r.x.cx;
      *mickeyy = (signed short)r.x.dx;
   }*/
}



/* _set_mouse_range:
 *  The int33 driver tends to report mouse movements in chunks of 4 or 8
 *  pixels when in an svga mode. So, we increase the mouse range and
 *  sensitivity, and then divide all the values it returns by 8.
 */
void _set_mouse_range()
{
   int i;
   int col;

   if (!gfx_driver || !_mouse_installed)
      return;

   if ((!_mouse_pointer) || 
       ((screen) && (_mouse_pointer) &&
	   (bitmap_color_depth(_mouse_pointer) != bitmap_color_depth(screen)))) {
	   
	   if (_mouse_pointer) {
		   destroy_bitmap(_mouse_pointer);
		   _mouse_pointer = NULL;
	   }
	   
	   if (ms) {
		   destroy_bitmap(ms);
		   ms = NULL;
		   
		   destroy_bitmap(mtemp);
		   mtemp = NULL;
	   }
	   
	   _mouse_pointer = create_bitmap(16, 16);
	   
	   for (i=0; i<256; i++) {
		   if (bitmap_color_depth(_mouse_pointer) == 8) {
			   switch (mouse_pointer_data[i]) {
			   case 1:  _mouse_pointer->line[i/16][i&15] = 16;  break;
			   case 2:  _mouse_pointer->line[i/16][i&15] = 255; break;
			   default: _mouse_pointer->line[i/16][i&15] = 0;   break;
			   }
		   }
		   else {
			   switch (mouse_pointer_data[i]) {
			   case 1:  col = makecol(255, 255, 255);             break;
			   case 2:  col = makecol(0, 0, 0);                   break;
			   default: col = _mouse_pointer->vtable->mask_color; break;
			   }
			   putpixel(_mouse_pointer, i&15, i/16, col);
		   } 
	   }
	   
	   set_mouse_sprite(_mouse_pointer);
   }
   else
	   set_hw_cursor();

   set_mouse_range(0, 0, SCREEN_W-1, SCREEN_H-1);
   set_mouse_speed(2, 2);
   position_mouse(SCREEN_W/2, SCREEN_H/2);
}


/* install_mouse:
 *  Installs the Allegro mouse handler. You must do this before using any
 *  other mouse functions. Return -1 if it can't find a mouse driver,
 *  otherwise the number of buttons on the mouse.
 */
int install_mouse()
{
   int num_buttons;
   int r;

   if (_mouse_installed)
      return -1;

   mickeymode = TRUE;
   ntmode = FALSE;
   
   /*if (gfx_driver==&gfx_directxwin)
      r = DI_InitMouse(mouseint, 0);
   else*/
      r = DI_InitMouse(mouseint, 1);
   if (r!=0) return -1;
   
   num_buttons = 2;
     
   _mouse_installed = TRUE;
   _set_mouse_range();
   _add_exit_func(remove_mouse);

   if (ntmode)
      install_int(mouse_move, 20);   

   return num_buttons;
}



/* remove_mouse:
 *  Removes the mouse handler. You don't normally need to call this, because
 *  allegro_exit() will do it for you.
 */
void remove_mouse()
{
   if (!_mouse_installed)
      return;

   DI_DoneMouse();

   show_mouse(NULL);

   mouse_x = mouse_y = _mouse_x = _mouse_y = 0;
   mouse_b = _mouse_b = 0;
   mouse_pos = 0;
   _mouse_screen = NULL;

   if (_mouse_pointer) {
      destroy_bitmap(_mouse_pointer);
      _mouse_pointer = NULL;
   }

   if (ms) {
      destroy_bitmap(ms);
      ms = NULL;

      destroy_bitmap(mtemp);
      mtemp = NULL;
   }

   _remove_exit_func(remove_mouse);
   _mouse_installed = FALSE;
}

