/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *      Win32 timer interrupt routines, by Stefan Schimanski (1Stein@gmx.de)
 *
 *      See readme.txt for copyright information.*/


#ifndef WIN32
#error This file should only be used by the djgpp version of Allegro
#endif

#include <stdlib.h>
#include <stdio.h>

/*#include "allegro.h"*/
#include "internal.h"


/* Error factor for retrace syncing. Reduce this to spend less time waiting
 * for the retrace, increase it to reduce the chance of missing retraces.
 */

#define VSYNC_MARGIN    1024

#define MAX_TIMERS      16

int _timer_installed = FALSE;

static long timer_delay;                  /* how long between interrupts */
static int timer_semaphore = FALSE;       /* reentrant interrupt? */

volatile int _retrace_hpp_value = -1;     /* to set during next retrace */

static long vsync_counter = BPS_TO_TIMER(70); /* retrace position counter */
static long vsync_speed = BPS_TO_TIMER(70);   /* retrace speed */

static struct {                           /* list of active callbacks */
   void ((*proc)());
   long speed;
   long counter;
} my_int_queue[MAX_TIMERS];

static struct {                           /* list of to-be-added callbacks */
   void ((*proc)());
   long speed;
} waiting_list[MAX_TIMERS];

static int waiting_list_size = 0;


/* timerint:
 *  Hardware level timer interrupt (int 8) handler. Calls whatever user
 *  timer routines are due for dealing with, and calculates how long until
 *  we need another timer tick.
 */
int timerint(int last_delay)
{
   long new_delay = 0x8000;
   int callback[MAX_TIMERS];
   int x;

   timer_delay = last_delay;

   if (!_timer_installed) return MSEC_TO_TIMER(100);

   timer_semaphore = TRUE;

   /* deal with retrace synchronisation */
   vsync_counter -= timer_delay; 
   while (vsync_counter <= 0) {     
	 vsync_counter += vsync_speed;
	 retrace_count++;
	 if (retrace_proc)
	    retrace_proc();
   }
   if (vsync_counter < new_delay)
      new_delay = vsync_counter;

   /* process the user callbacks */
   for (x=0; x<MAX_TIMERS; x++) { 
      callback[x] = FALSE;

      if ((my_int_queue[x].proc) && (my_int_queue[x].speed > 0)) {
	 my_int_queue[x].counter -= timer_delay;
	 if (my_int_queue[x].counter <= 0) {
	    my_int_queue[x].counter += my_int_queue[x].speed;
	    callback[x] = TRUE;
	 }
	 if (my_int_queue[x].counter < new_delay)
	    new_delay = my_int_queue[x].counter;
      }
   }   

    /* finally call the user timer routines */
   for (x=0; x<MAX_TIMERS; x++) {
      if (callback[x]) {
	 my_int_queue[x].proc();
	 if (i_love_bill) {
	    while ((my_int_queue[x].proc) && (my_int_queue[x].counter <= 0)) {
	       my_int_queue[x].counter += my_int_queue[x].speed;
	       my_int_queue[x].proc();
	    }
	 }
      }
   }

   timer_semaphore = FALSE;

   return new_delay;
}



static volatile long rest_count;

static void rest_int()
{
   rest_count--;
}


/* rest_callback:
 *  Waits for time milliseconds.
 */
void rest_callback(long time, void (*callback)())
{
   if (_timer_installed) {
      rest_count = time;

      if (install_int(rest_int, 1) < 0)
	 return;

      do {
	 if (callback)
	    callback();
        WinAllegro_HandleMessages();
      } while (rest_count > 0);

      remove_int(rest_int);
   }        
}



/* rest:
 *  Waits for time milliseconds.
 */
void rest(long time)
{
   rest_callback(time, NULL);
}


/* timer_calibrate_retrace:
 *  Times several vertical retraces, and calibrates the retrace syncing
 *  code accordingly.
 */
static void timer_calibrate_retrace()
{
}



/* timer_simulate_retrace:
 *  Turns retrace simulation on or off, and if turning it on, calibrates
 *  the retrace timer.
 */
void timer_simulate_retrace(int enable)
{
   if (!_timer_installed)
      return;

   _timer_use_retrace = FALSE;
   vsync_counter = vsync_speed = BPS_TO_TIMER(70);
}


/* find_timer_slot:
 *  Searches the list of user timer callbacks for a specified function, 
 *  returning the position at which it was found, or -1 if it isn't there.
 */
static int find_timer_slot(void (*proc)())
{
   int x;

   for (x=0; x<MAX_TIMERS; x++)
      if (my_int_queue[x].proc == proc)
	 return x;

   return -1;
}


/* install_int_ex:
 *  Installs a function into the list of user timers, or if it is already 
 *  installed, adjusts its speed. This function will be called once every 
 *  speed timer ticks. Returns a negative number if there was no room to 
 *  add a new routine (there is only room for eight). Note that your 
 *  routine is called by the Allegro interrupt handler and not directly 
 *  by the processor, so you should return normally rather than using an 
 *  iret. Your interrupt routine must finish quickly, and you should not 
 *  use large amounts of stack or make any calls to the operating system. 
 */
int install_int_ex(void (*proc)(), long speed)
{
   int x;

   if (!_timer_installed) {
      /* we are not alive yet: flag this callback to be started later */
      if (waiting_list_size >= MAX_TIMERS)
	 return -1;

      waiting_list[waiting_list_size].proc = proc;
      waiting_list[waiting_list_size].speed = speed;
      waiting_list_size++;
      return 0;
   }

   x = find_timer_slot(proc);          /* find the handler position */

   if (x < 0)                          /* if not there, find free slot */
      x = find_timer_slot(NULL);

   if (x < 0)                          /* are there any free slots? */
      return -1;

   if (proc != my_int_queue[x].proc) { /* add new entry */
      my_int_queue[x].counter = speed;
      my_int_queue[x].proc = proc; 
   }
   else {                              /* alter speed of existing entry */
      my_int_queue[x].counter -= my_int_queue[x].speed;
      my_int_queue[x].counter += speed;
   }

   my_int_queue[x].speed = speed;

   return 0;
}


/* install_int:
 *  Wrapper for install_int_ex, which takes the speed in milliseconds.
 */
int install_int(void (*proc)(), long speed)
{
   return install_int_ex(proc, MSEC_TO_TIMER(speed));
}


/* remove_int:
 *  Removes a function from the list of user timers.
 */
void remove_int(void (*proc)())
{
   int x = find_timer_slot(proc);

   if (x >= 0) {
      my_int_queue[x].proc = NULL;
      my_int_queue[x].speed = 0;
      my_int_queue[x].counter = 0;
   }
}


/* install_timer:
 *  Installs the timer interrupt handler. You must do this before installing
 *  any user timer routines. You must set up the timer before trying to 
 *  display a mouse pointer or using any of the GUI routines.
 */
int install_timer()
{
   int x;

   if (_timer_installed)
      return -1;

   for (x=0; x<MAX_TIMERS; x++) {
      my_int_queue[x].proc = NULL;
      my_int_queue[x].speed = 0;
      my_int_queue[x].counter = 0;
   }

   timer_delay = 0;

   _add_exit_func(remove_timer);
   _timer_installed = TRUE;

   /* activate any waiting callback functions */
   for (x=0; x<waiting_list_size; x++)
      install_int_ex(waiting_list[x].proc, waiting_list[x].speed);

   waiting_list_size = 0;

   return 0;
}



/* remove_timer:
 *  Removes our timer handler and resets the BIOS clock. You don't normally
 *  need to call this, because allegro_exit() will do it for you.
 */
void remove_timer()
{
   if (!_timer_installed)
      return;

   _remove_exit_func(remove_timer);
   _timer_installed = FALSE;
}


