/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *		DirectDraw access function (windows file), by Stefan Schimanski (1Stein@gmx.de)
 *
 *      See readme.txt for copyright information.
 */


#define DIRECTDRAW_VERSION 0x0300

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <ddraw.h>
#include <stdio.h>
#include "winintrn.h"


/***************************************************/

LPDIRECTDRAW lpDD1 = NULL;
LPDIRECTDRAW2 lpDD = NULL;
DDSURFACEDESC ddsd;
LPDIRECTDRAWPALETTE lpPal = NULL;
LPDIRECTDRAWCLIPPER lpClipper = NULL;
HANDLE hMessageProcessed = NULL;
int DD_Width = 640;
int DD_Height = 480;
int DD_Depth = 8;
int DD_Fullscreen = 0;
int DD_PixelSize = 1;
int DD_BackPitch = 0;
char *DD_BackScreen = NULL;
int PalSet = 0;

CDDSurface *Screen = NULL;
CDDSurface *Front = NULL;

/***************************************************/


PALETTEENTRY PalBuf[256];


/********************************************************
 * Sets primary palette									*
 ********************************************************/
void DD_SetPalette(CDDColor *Pal, int from, int to, int vsync)
{
	if (lpPal != NULL)
	{
		if (!DD_Fullscreen)
		{
			if (from<1) from=1;
			if (to>=254) to=254;
			if (to<from) return;
		}

		for (int n=from; n<=to; n++)
		{
			PalBuf[n].peRed		= Pal[n].r << 2;
			PalBuf[n].peGreen	= Pal[n].g << 2;
			PalBuf[n].peBlue	= Pal[n].b << 2;	
		}
					
		if (vsync) DD_VSync();
		lpPal->SetEntries(0, from, (to-from)+1, PalBuf+from);			
	}
}


void DD_SetCurrentPalette()
{
	if (lpPal)
   {
      lpPal->SetEntries(0, 0, 256, PalBuf);
   }
}


void DD_ExpandGDIPal()
{
	HDC dc = GetDC(hMainWnd);
	SetSystemPaletteUse(dc ,SYSPAL_STATIC);
	RealizePalette(dc);
	ReleaseDC(hMainWnd, dc);
	PostMessage(HWND_BROADCAST, WM_SYSCOLORCHANGE, 0, 0);
}


void DD_ReduceGDIPal()
{
	HDC dc = GetDC(hMainWnd);
	SetSystemPaletteUse(dc ,SYSPAL_NOSTATIC);
	RealizePalette(dc);
	ReleaseDC(hMainWnd, dc);
	PostMessage(HWND_BROADCAST, WM_SYSCOLORCHANGE, 0, 0);
}


/********************************************************
 * Waits for vertical retrace							*
 ********************************************************/
void DD_VSync()
{
	if (lpDD) lpDD->WaitForVerticalBlank(DDWAITVB_BLOCKBEGIN, 0);
}


/********************************************************
 * Initialize Directx									*
 ********************************************************/

void SetStyle(int Windowed)
{
	if (Windowed)
	{
		RECT rect = {0, 0, DD_Width, DD_Height};
	        DWORD dwStyle;

		// Change window attributes
	        dwStyle = GetWindowLong(hMainWnd, GWL_STYLE);
	        dwStyle &= ~WS_POPUP;
	        dwStyle |= WS_OVERLAPPED | WS_SYSMENU | WS_CAPTION;
	        SetWindowLong(hMainWnd, GWL_STYLE, dwStyle);
	
		// Resize the window        
		AdjustWindowRectEx(&rect, GetWindowLong(hMainWnd, GWL_STYLE), 
			GetMenu(hMainWnd) != NULL, GetWindowLong(hMainWnd, GWL_EXSTYLE));

		// Just in case the window was moved off the visible area of the
		// screen.
		SetWindowPos(hMainWnd, NULL, 0, 0, rect.right-rect.left,
	                rect.bottom-rect.top, SWP_NOMOVE | SWP_NOZORDER |
			SWP_NOACTIVATE);

		SetWindowPos(hMainWnd, HWND_NOTOPMOST, 0, 0, 0, 0,
			SWP_NOSIZE | SWP_NOMOVE | SWP_NOACTIVATE);	
	} else
	{
		// Change window attributes
		SetWindowLong(hMainWnd, GWL_STYLE, WS_POPUP | WS_VISIBLE);
		SetWindowPos(hMainWnd, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOACTIVATE |
                SWP_NOMOVE | SWP_NOSIZE | SWP_NOZORDER);		
	}
}


bool IsRightPixelFormat(int bpp, LPDDPIXELFORMAT Format)
{
	D3("#%u %x/%x/%x is %u ?",
		Format->dwRGBBitCount,
		Format->dwRBitMask,
		Format->dwGBitMask,
		Format->dwBBitMask,
		DD_Depth);
	
	switch (bpp)
	{
	case 8:
	case 24:
	case 32:
		return Format->dwRGBBitCount==bpp;

	case 15: // 555
		return (Format->dwRBitMask==0x7c00 &&
			Format->dwGBitMask==0x3e0 &&
			Format->dwBBitMask==0x1f);

	case 16: // 565
		return (Format->dwRBitMask==0xf800 &&
			Format->dwGBitMask==0x7e0 &&
			Format->dwBBitMask==0x1f);

	default: return false;
	}	
}


int DD_TryColorDepth(int Depth)
{
	DBEG("DD_TryColorDepth");

	HRESULT ddrval = lpDD->SetDisplayMode(DD_Width, DD_Height, Depth, 0, 0);
	if (ddrval != DD_OK) 
	{			
		D3("DD_TryColorDepth - SetDisplayMode returned with error");
		D3("DD_TryColorDepth - SetDisplayMode DDSDM_STANDARDVGAMODE");
		ddrval = lpDD->SetDisplayMode(DD_Width, DD_Height, Depth, 0, DDSDM_STANDARDVGAMODE);
		if (ddrval != DD_OK)
		{				
			D3("DD_TryColorDepth - SetDisplayMode with DDSDM_STANDARDVGAMODE error");
			goto Error;
		}
	}
		
	// This is not really needed, except in certain cases. One case
	// is while using MFC. When the desktop is initally at 16bpp, a mode
	// switch to 8bpp somehow causes the MFC window to not fully initialize
	// and a CreateSurface will fail with DDERR_NOEXCLUSIVEMODE. This will
	// ensure that the window is initialized properly after a mode switch.
	ShowWindow(hMainWnd, SW_SHOW);				
		
	// Get video mode
	D3("DD_TryColorDepth - GetDisplayMode");
	ddsd.dwSize = sizeof(ddsd);
	ddrval = lpDD->GetDisplayMode(&ddsd);
	if FAILED(ddrval) goto Error;

	if (!IsRightPixelFormat(DD_Depth, &ddsd.ddpfPixelFormat))
	{
		D3("DD_TryColorDepth - Wrong color depth");
		goto Error;
	}

	return 0;

Error:
	return -1;
}


CDDSurface *DD_Open(int w, int h, int cdepth, int fullscreen)
{
    DBEG("DD_Open");

    DD_Width = w;
	DD_Height = h;
	DD_Depth = cdepth;
	DD_Fullscreen = fullscreen;

	HRESULT ddrval;    

	switch (DD_Depth)
	{
	case 8: 
		DD_PixelSize=1; break;

	case 15: case 16: 
		DD_PixelSize=2; break;

	case 24: 
		DD_PixelSize=3; break;

	case 32: 
		DD_PixelSize=4; break;

	default: goto Error;
	}	

	// Create memory screen
	DD_BackScreen = (char*)malloc(4096);
	DD_BackPitch = 1;
	
	// Call main thread to change video mode	
	Screen = new CDDSurface;

	Screen->Locked = 0;
	Screen->Surface = NULL;
	Screen->Lost = 0;
	Screen->Pitch = DD_BackPitch;
	Screen->BasePtr = DD_BackScreen;

	if (DD_Fullscreen)
	{	
		D3("DD_Open - Fullscreen");

		// Set window style
		SetStyle(0);				
		DI_HideMouse();

		// Set cooperative level
		D3("DD_Open - SetCooperativeLevel");
		ddrval = lpDD->SetCooperativeLevel(hMainWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN);
		if(ddrval != DD_OK) goto Error;				

		// Try first color depth
		D3("DD_Open - SetDisplayMode");
		if (DD_TryColorDepth(DD_Depth)==0) goto CorrectMode;

		if (DD_Depth==16)
		{
			D3("DD_Open - SetDisplayMode, try 15 instead of 16 bpp");
			if (DD_TryColorDepth(15)==0) goto CorrectMode;
		}
				
		if (DD_Depth==15)
		{
			D3("DD_Open - SetDisplayMode, try 16 instead of 15 bpp");
			if (DD_TryColorDepth(16)==0) goto CorrectMode;
		}	

		D3("DD_Open - Can't set specified mode");
		goto Error;

CorrectMode:
		
		// Create primary surface
		ddsd.dwSize = sizeof(ddsd);
		ddsd.dwFlags = DDSD_CAPS;
		ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_VIDEOMEMORY;
	
		D3("DD_Open - Create primary surface");
		ddrval = lpDD->CreateSurface(&ddsd, &Screen->Surface, NULL);
		if (ddrval != DD_OK) goto Error;

		// Work around for Ati Rage Pro cards that minimizes the window after
		// the creation of the primary surface
		ShowWindow(hMainWnd, SW_SHOW);
	} else	
	{
		D3("DD_Open - Windowed mode");

		// Set window style
		SetStyle(1);
		DI_HideMouse();
				
		// Set cooperative level
		D3("DD_Open - SetCooperativeLevel");
		ddrval = lpDD->SetCooperativeLevel(hMainWnd, DDSCL_NORMAL);
		if(ddrval != DD_OK) goto Error;

		// Get video mode
		D3("DD_Open - GetDisplayMode");
		ddsd.dwSize = sizeof(ddsd);
		ddrval = lpDD->GetDisplayMode(&ddsd);
		if FAILED(ddrval) goto Error;

		if (!IsRightPixelFormat(DD_Depth, &ddsd.ddpfPixelFormat))		
		{
			D3("DD_Open - Wrong color depth");
			goto Error;
		}

		// Create primary surface
		ddsd.dwSize = sizeof(ddsd);
		ddsd.dwFlags = DDSD_CAPS;
		ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_VIDEOMEMORY;
	
		D3("DD_Open - Create primary surface");
		ddrval = lpDD->CreateSurface(&ddsd, &Screen->Surface, NULL);
		if (ddrval != DD_OK) goto Error;

		// Work around for Ati Rage Pro cards that minimizes the window after
		// the creation of the primary surface
		ShowWindow(hMainWnd, SW_SHOW);
 	}
	
	// Create palette
	if (DD_Depth==8)
	{	
		DD_ReduceGDIPal();

		// First set up the Windows static entries.
		if (!PalSet)
		{
			PalSet = 1;

			PalBuf[0].peFlags = 0;
			PalBuf[0].peRed = 0;
			PalBuf[0].peGreen = 0;
			PalBuf[0].peBlue = 0;

			PalBuf[255].peFlags = 0;
			PalBuf[255].peRed = 255;
			PalBuf[255].peGreen = 255;
			PalBuf[255].peBlue = 255;

			// Now set up private entries.
			for (int i=1; i < 254; i++)
			{
				PalBuf[i].peFlags = PC_NOCOLLAPSE|PC_RESERVED;
				PalBuf[i].peRed = 64;
				PalBuf[i].peGreen = 64;
				PalBuf[i].peBlue = 64;
			}
		}

		D3("DD_Open - Create palette");
		ddrval = lpDD->CreatePalette(DDPCAPS_8BIT | DDPCAPS_ALLOW256, PalBuf, &lpPal, NULL);
		if (ddrval != DD_OK) goto Error;

		D3("DD_Open - Set palette");
		ddrval = Screen->Surface->SetPalette(lpPal);		
	}
	
	Front = Screen;		
	
    SetEvent(hMessageProcessed);
	DEND("DD_Open");	
	return Screen;
	
Error:
    Screen = NULL;
    
    SetEvent(hMessageProcessed);
	DEND("DD_Open error");    
	return Screen;
}


/********************************************************
 * Creates surface										*
 ********************************************************/
CDDSurface *DD_CreateSurface(int Width, int Height)
{
	DBEG("DD_CreateSurface - start");

	HRESULT ddrval;			
	CDDSurface *s = new CDDSurface;

	s->Locked = 0;
	s->Surface = NULL;
	s->Lost = 0;
	s->Pitch = DD_BackPitch;
	s->BasePtr = DD_BackScreen;

	// Create surface
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH; // | DDSD_PIXELFORMAT;
	ddsd.ddsCaps.dwCaps = DDSCAPS_VIDEOMEMORY;		
	ddsd.dwWidth = Width;
	ddsd.dwHeight = Height;

	/*ddsd.ddpfPixelFormat.dwSize = sizeof(ddsd.ddpfPixelFormat);
	ddsd.ddpfPixelFormat.dwFlags = DDPF_PALETTEINDEXED8 | DDPF_RGB;
	ddsd.ddpfPixelFormat.dwRGBBitCount = 8;
	ddsd.ddpfPixelFormat.dwRBitMask = 0;
	ddsd.ddpfPixelFormat.dwGBitMask = 0;
	ddsd.ddpfPixelFormat.dwBBitMask = 0;*/
	
	D3("DD_CreateSurface - Create surface");
	ddrval = lpDD->CreateSurface(&ddsd, &s->Surface, NULL);
	if (ddrval != DD_OK) goto Error;		

	DEND("DD_CreateSurface - end");
	return s;

Error:
	delete s;
	DEND("DD_CreateSurface - end with error");
	return 0;
}


/********************************************************
 * Destroys surface										*
 ********************************************************/
void DD_DestroySurface(struct CDDSurface *Surface)
{	
	DBEG("DD_DestroySurface - start");
	if (Surface)
	{
		// Detach surface
		D3("DD_DestroySurface - show screen surface");
		if (Front==Surface && Screen!=Surface)
			DD_ShowSurface(Screen, 1);		
		
		// Release surface
		D3("DD_DestroySurface - release surface");
		if (Surface->Surface!=NULL)
			Surface->Surface->Release();		

		// Delete struct
		delete Surface;		
	}
	DEND("DD_DestroySurface - end");
}


/********************************************************
 * Displays surface										*
 ********************************************************/
int DD_ShowSurface(struct CDDSurface *Surface, int Wait)
{
	HRESULT ddrval;
	LPDIRECTDRAWSURFACE a;
	LPDIRECTDRAWSURFACE b;	
    int r=0;

    if (Screen && Surface)
	{
        DBEG("DD_ShowSurface");
        EnterSemaphore();

		// When already active
        if (Surface==Front)
        {
            D3("DD_ShowSurface - Front = Back buffer");
            goto Return;
        }

		b = Surface->Surface;
		a = Front->Surface;
	
        // Surface lost?
        if (a->IsLost())
        {
            D1("DD_ShowSurface - Front surface lost");
            Surface->Lost = 1;
            if (a->Restore()!=DD_OK)
            {
                D1("DD_ShowSurface - Front surface restore failed");
                r = -7;
                goto Return;
            } else
            {
                D1("DD_ShowSurface - Front surface restored");                
            }
        }

        if (b->IsLost())
        {
            D1("DD_ShowSurface - Back surface lost");
            Front->Lost = 1;            
            if (b->Restore()!=DD_OK)
            {
                D1("DD_ShowSurface - Back surface restore failed");
                r = -7;
                goto Return;
            } else
            {
                D1("DD_ShowSurface - Back surface restored");
            }
        }
        
Again:
		// Attach surface
		ddrval = a->AddAttachedSurface(b);
		switch (ddrval)
		{
		case DDERR_CANNOTATTACHSURFACE:	    D1("DD_ShowSurface - DDERR_CANNOTATTACHSURFACE"); r=-2; goto Return;
		case DDERR_GENERIC:					D1("DD_ShowSurface - DDERR_GENERIC"); r=-3; goto Return;
		case DDERR_INVALIDOBJECT:			D1("DD_ShowSurface - DDERR_INVALIDOBJECT"); r=-4; goto Return;
		case DDERR_INVALIDPARAMS:			D1("DD_ShowSurface - DDERR_INVALIDPARAMS"); r=-5; goto Return;
		case DDERR_SURFACEALREADYATTACHED:	D1("DD_ShowSurface - DDERR_SURFACEALREADYATTACHED"); r=-6; goto Return;
		case DDERR_SURFACELOST:				D1("DD_ShowSurface - DDERR_SURFACELOST"); r=-7; goto Return;
		case DDERR_WASSTILLDRAWING:			D1("DD_ShowSurface - DDERR_WASSTILLDRAWING");  r=-8; goto Return;
		}
		        	
		// Show the new surface
        ddrval = a->Flip(b, Wait ? DDFLIP_WAIT : 0);		
		if (ddrval!=DD_OK)
		{
            D1("DD_ShowSurface - Flip failed");
			a->DeleteAttachedSurface(0, b);
            r=-1; goto Return;			
		}

		// Detach old active surface
		ddrval = a->DeleteAttachedSurface(0, b);					
		
		// Swap surfaces
		Surface->Surface = a;
		Front->Surface = b;		
		Front = Surface;

Return:
        LeaveSemaphore();
        DEND("DD_ShowSurface");

		return r;
	} else return 1;
}


/********************************************************
 * Return flip status               					*
 ********************************************************/
int DD_ShowFinished()
{
    if (Front)
    {
        EnterSemaphore();
        int r = ((struct IDirectDrawSurface3 *)Front->Surface)->GetFlipStatus(DDGFS_ISFLIPDONE)==DD_OK;
        LeaveSemaphore();

        return r;
    }
    else
        return -1;
}

/********************************************************
 * Locks surface										*
 ********************************************************/
int DD_LockCore(CDDSurface *Surface)
{	
	HRESULT ddrval;
	LPDIRECTDRAWSURFACE s = Surface->Surface;

    ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = 0;
	ddrval = s->Lock(NULL, &ddsd, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR, NULL);

	if (ddrval==DDERR_SURFACELOST)
	{	
		D1("DD_LockCore - Surface lost");

		s->Restore();
		ddsd.dwSize = sizeof(ddsd);				
		ddsd.dwFlags = 0;
		ddrval = s->Lock(NULL, &ddsd, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR, NULL);

		Surface->Lost = 1;		
	}
	
	if (ddrval!=DD_OK)
	{	
		D1("DD_Lockcore - Lock failed");

		Surface->Lost = 1;
		
		/*
		If the lock doesn't work, a memory buffer is assigned to
		reduce the chance of a crash
		*/
		Surface->BasePtr = DD_BackScreen;
		Surface->Pitch = DD_BackPitch;

		return 1;
	} else
	{
		Surface->BasePtr = (char*)ddsd.lpSurface;
		Surface->Pitch = ddsd.lPitch;				

		if (Surface==Screen && !DD_Fullscreen)
			Surface->BasePtr = Surface->BasePtr + ClientPosX*DD_PixelSize + ClientPosY*Surface->Pitch;
	}

	return 0;
}


void DD_Lock(CDDSurface *Surface)
{   
	if (Surface)
	{	
		EnterSemaphore();
				
		if (Surface->Locked==0)
		{
            WinAllegro_HandleMessages();
			DD_LockCore(Surface);
        }
		
		Surface->Locked++;
	}
}


/********************************************************
 * Unlocks surface										*
 ********************************************************/

void DD_UnlockCore(CDDSurface *Surface)
{
	HRESULT ddrval;

	if (Surface->BasePtr==DD_BackScreen) return;

	LPDIRECTDRAWSURFACE s = Surface->Surface;
	ddrval = s->Unlock(NULL);
	if (FAILED(ddrval))
	{
		D1("DD_UnlockCore - Unlock failed");
	}

#ifdef _DEBUG
	Surface->BasePtr==DD_BackScreen;
	Surface->Pitch==DD_BackPitch;
#endif
}


void DD_Unlock(CDDSurface *Surface)
{
	if (Surface && Surface->Locked)
	{
		Surface->Locked--;

		if (Surface->Locked==0)
		{
			DD_UnlockCore(Surface);
		}

		LeaveSemaphore();
	}
}


/********************************************************
 * Returns whether primary surface was lost				*
 ********************************************************/
int DD_Lost(CDDSurface *Surface)
{
	if (Surface)
	{
		Surface->Lost |= (Surface->Surface->IsLost()==DDERR_SURFACELOST);
		return Surface->Lost;
	} else return 0;
}


/********************************************************
 * Switches to GDI surface				                *
 ********************************************************/
void WinAllegro_SwitchToGDI()
{
    EnterSemaphore();
    if (lpDD) lpDD->FlipToGDISurface();
    LeaveSemaphore();
}


/********************************************************
 * Returns the DirectDraw object                        *
 ********************************************************/
LPDIRECTDRAW WinAllegro_GetDirectDraw()
{
    return lpDD1;
}


LPDIRECTDRAW2 WinAllegro_GetDirectDraw2()
{
    return lpDD;
}


/********************************************************
 * Blit													*
 ********************************************************/
int DD_Blit(struct CDDSurface *Source, struct CDDSurface *Dest, 
			int Source_x, int Source_y, int Dest_x, int Dest_y, int Width, int Height)
{
	HRESULT ddrval;
	RECT SrcRect = { Source_x, Source_y, Source_x+Width, Source_y+Height };

	do
	{
		ddrval = Dest->Surface->BltFast(Dest_x, Dest_y, Source->Surface, 
			&SrcRect, DDBLTFAST_NOCOLORKEY);                       
	} while (ddrval==DDERR_WASSTILLDRAWING);

	return ddrval!=DD_OK;
}


int DD_BlitTrans(struct CDDSurface *Source, struct CDDSurface *Dest, 
			int Source_x, int Source_y, int Dest_x, int Dest_y, int Width, int Height)
{
	HRESULT ddrval;
	RECT SrcRect = { Source_x, Source_y, Source_x+Width, Source_y+Height };

	do
	{
		ddrval = Dest->Surface->BltFast(Dest_x, Dest_y, Source->Surface, 
			&SrcRect, DDBLTFAST_SRCCOLORKEY);                       
	} while (ddrval==DDERR_WASSTILLDRAWING);

	return ddrval!=DD_OK;
}

/*int DD_Fill(struct CDDSurface *Dest, int Dest_x, int Dest_y, int Width, int Height, int Color)
{
	HRESULT ddrval;
	RECT SrcRect = { Source_x, Source_y, Source_x+Width, Source_y+Height };

	if (Source->Locked) DD_UnlockCore(Source);
	if (Dest->Locked) DD_UnlockCore(Dest);

	do
	{
		ddrval = Dest->Surface->BltFast(Dest_x, Dest_y, Source->Surface, 
			&SrcRect, DDBLTFAST_SRCCOLORKEY);                       
	} while (ddrval==DDERR_WASSTILLDRAWING);

	if (Dest->Locked) DD_UnlockCore(Dest);
	if (Source->Locked) DD_Lock(Source);

	return ddrval!=DD_OK;
}*/



/********************************************************
 * Init Directx											*
 ********************************************************/
int DD_Init()
{	
	DBEG("DD_Init - start");

	HRESULT      ddrval;

	// Query DirectDraw interface
	D3("DD_Init - DirectDrawCreate");
	ddrval = DirectDrawCreate(NULL, &lpDD1, NULL);
	if(ddrval != DD_OK) return 1;	

	// Query DirectDraw2 interface
	D3("DD_Init - Query IID_IDirectDraw2 Interface");
	ddrval = lpDD1->QueryInterface(IID_IDirectDraw2, (LPVOID *)&lpDD); 
	if(ddrval != DD_OK) return 1;

    // Create sync event
    hMessageProcessed = CreateEvent(NULL, FALSE, FALSE, NULL);

	DEND("DD_Init - end");
	return 0;
}


/********************************************************
 * Close Directx										*
 ********************************************************/
void DD_Done()
{	
	DBEG("DD_Done - start");
	
    if (hMessageProcessed!=NULL)
    {
        CloseHandle(hMessageProcessed);
        hMessageProcessed = NULL;
    }

	if(lpDD != NULL)
	{	
		D3("DD_Done - Release DirectDraw2");
		lpDD->Release();
		lpDD = NULL;    
	}

	if(lpDD1 != NULL)
	{
		D3("DD_Done - Release DirectDraw");
		lpDD1->Release();
		lpDD1 = NULL;    
	}	

	DEND("DD_Done - end");
}


/********************************************************
 * Close video mode										*
 ********************************************************/
void DD_Close()
{
	DBEG("DD_CloseCore");
	
	if(lpDD != NULL)
	{				
		DI_ShowMouse();

		// Destroy palette
		if (lpPal != NULL)
		{
			D3("DD_CloseCore - Release palette");
			lpPal->Release();
			lpPal = NULL;
		}		

		// Destroy screen surface
		if (Screen!=NULL)
		{
			D3("DD_CloseCore - Destroy primary surface");
			DD_DestroySurface(Screen);
			Screen = NULL;		
			Front = NULL;
		}		

		// Destroy memory screen
		if (DD_BackScreen)
		{
			D3("DD_CloseCore - Destroy back screen");
			free(DD_BackScreen);
			DD_BackScreen = NULL;
		}	

		// Restore video mode
		if (DD_Fullscreen) 
		{
			//SendMessage(hMainWnd, WM_CLOSEDX);
			D3("DD_CloseCore - Set cooperative level to normal");		
			lpDD->SetCooperativeLevel(hMainWnd, DDSCL_NORMAL);  

			D3("DD_CloseCore - Show main window");
			ShowWindow(hMainWnd, SW_SHOW);
		}
	}

	// Restore GDI color management
	D3("DD_CloseCore - Expand GDI palette");
	DD_ExpandGDIPal();

	D3("DD_CloseCore - Set window style");
	SetStyle(1);

    SetEvent(hMessageProcessed);
	
	DEND("DD_CloseCore");
}
	

