/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *		DirectInput functions (windows file), by Stefan Schimanski (1Stein@gmx.de)
 *         Mouse, Keyboard, Joystick
 *
 *      See readme.txt for copyright information.
 */

#define INITGUID
#define DIRECTINPUT_VERSION 0x0300

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <process.h>
#include <dinput.h>
#include <mmsystem.h>
#include <stdio.h>
#include "winintrn.h"


//===============================================================

LPDIRECTINPUT			lpDI = NULL;

int DI_Inited=0;

/****************************************************************************
 *																			*
 * Keyboard functions														*
 *																			*
 ****************************************************************************/

LPDIRECTINPUTDEVICE		lpDIKeyboard = NULL;

const DINPUT_BUFFERSIZE = 256;
DIDEVICEOBJECTDATA KeyBuf[DINPUT_BUFFERSIZE];

long DI_KeyboardEvent=0;
int DI_KeyboardInited=0;

const WAITTIME = 500;
const REPEATTIME = 20;
int KeyboardTimerId;
unsigned char RepeatedScancode = 0xff;
int RepeatTimer;
int (*DI_KeyInt)(int scancode, int foreground)=NULL;
extern "C" int three_fingers_pressed;

int DI_KeyDelay = 200;
int DI_KeyRepeat = 30;

/*
 * key repeat functions
 *   Because DirectInput sends scancodes only once although a key is hold down, a timer
 *   is needed to generate these scancodes in software
 */
void SetRepeatedScancode(unsigned char Scancode)
{
	if (RepeatedScancode==Scancode) return;

	RepeatedScancode = Scancode;		
}


void __stdcall KeyboardTimer(unsigned int,unsigned int,unsigned long,unsigned long,unsigned long)
{
	if (RepeatedScancode==0xff) return;

	int foreground = WinAllegro_IsForeground();
	if (foreground)
	{
		if (RepeatTimer>0) RepeatTimer--; else
		{
			if (RepeatedScancode>=0x80) (*DI_KeyInt)(0xE0, foreground);
			(*DI_KeyInt)(RepeatedScancode & 0x7F, foreground);
		}
	}   	
}


/*
 * HandleKey
 * 
 * Calls Allegro's keyboard interrupt functions
 */
void HandleKey(unsigned char Scancode, int Pressed)
{
	// Generate release code if ctrl-alt-del was pressed
	if (three_fingers_pressed)
	{
		three_fingers_pressed = 0;
		HandleKey(DIK_LALT, 0);
		HandleKey(DIK_RALT, 0);
		HandleKey(DIK_LCONTROL, 0);
		HandleKey(DIK_RCONTROL, 0);
		HandleKey(DIK_DELETE, 0);
	}

	// Handle current scancode
	/*char s[20];
	int sc = Scancode;
	sprintf(s, "HandleKey(%u, %u)", sc, Pressed);
	LOG(s);*/

	int foreground = WinAllegro_IsForeground();
	if (Scancode>=0x80)
	{
		(*DI_KeyInt)(0xE0, foreground);
	}
	
	if (Pressed)
	{
		RepeatTimer = DI_KeyDelay / (1000/REPEATTIME);
		SetRepeatedScancode(Scancode);
		(*DI_KeyInt)(Scancode & 0x7F, foreground); 
	} else 
	{
		if (RepeatedScancode==Scancode) SetRepeatedScancode(0xff);
		(*DI_KeyInt)(Scancode | 0x80, foreground);	
	}
}


/*
 * WinAllegro_GetKeyboardDelay
 */
int WinAllegro_GetKeyboardDelay()
{
    return DI_KeyDelay;
}


/*
 * WinAllegro_GetKeyboardRepeat
 */
int WinAllegro_GetKeyboardRepeat()
{
    return DI_KeyRepeat;
}


/*
 * WinAllegro_SetKeyboardDelay
 */
void WinAllegro_SetKeyboardDelay(int ms)
{
    DI_KeyDelay = ms;
}


/*
 * WinAllegro_SetKeyboardRepeat
 */
void WinAllegro_SetKeyboardRepeat(int ms)
{
    DI_KeyRepeat = ms;

    timeKillEvent(KeyboardTimerId);
   	KeyboardTimerId = timeSetEvent(1000/REPEATTIME, 0, KeyboardTimer, NULL, TIME_PERIODIC);
}


/*
 * DI_PollKeyboard
 *
 * Polling of the DirectInput keyboard buffer
 */
void DI_PollKeyboard()
{
	if (DI_KeyboardInited==0) return;	

	EnterSemaphore();
	while (1 && lpDIKeyboard)
	{
        DWORD Num = DINPUT_BUFFERSIZE;

		HRESULT hr = lpDIKeyboard->GetDeviceData(
                             sizeof(DIDEVICEOBJECTDATA), 
                             KeyBuf,
                             &Num, 0);

		if (hr == DIERR_INPUTLOST) 
		{
            if (DI_AcquireKeyboard()!=0) break;
		} else
		if FAILED(hr)
		{
			break;
		} else
		{
			for (int n=0; n<Num; n++)
			{		
				HandleKey(KeyBuf[n].dwOfs, KeyBuf[n].dwData & 0x80);
			}

			break;
		}
	}        
	LeaveSemaphore();
}


/*
 * DI_DoneKeyboard
 *
 * Shuts down the keyboard driver
 */
void DI_DoneKeyboard()
{
	if (DI_KeyboardInited==0) return;
	DI_KeyboardInited = 0;

	timeKillEvent(KeyboardTimerId);
	RepeatedScancode = 0xff;
	
	if (lpDIKeyboard) 
    {
		lpDIKeyboard->Unacquire();
		lpDIKeyboard->Release();
		lpDIKeyboard = NULL; 
	} 	
}


/*
 * DI_AcquireKeyboard
 *
 * Acquires the DirectInputDevice object and creates the notification event
 */
int DI_AcquireKeyboard()
{
	if (DI_KeyboardInited)
	{
		HRESULT r = lpDIKeyboard->Acquire();
		if FAILED(r) return 1;		

		return 0;
	} else return 1;
}


/*
 * DI_UnAcquireKeyboard
 *
 * Releases the DirectInputDevice object and deletes the notification event
 */
void DI_UnAcquireKeyboard()
{
	if (DI_KeyboardInited)
	{
		lpDIKeyboard->Unacquire();		
	}
}


/*
 * DI_InitKeyboard
 *
 * Initialise the keyboard driver and sets the keyboard handler
 */
int DI_InitKeyboard(int (*keyint)(int scancode, int foreground))
{
	HRESULT	r;
	DIPROPDWORD dipdw =
    {
        // the header
        {
            sizeof(DIPROPDWORD),        // diph.dwSize
            sizeof(DIPROPHEADER),       // diph.dwHeaderSize
            0,                          // diph.dwObj
            DIPH_DEVICE,                // diph.dwHow
        },
        // the data
        DINPUT_BUFFERSIZE,              // dwData
    };
	
	if (DI_KeyboardInited>0) return 0;
	DI_KeyboardInited = 1;
	DI_KeyInt = keyint;

	// Get keyboard device
	r = lpDI->CreateDevice(GUID_SysKeyboard, &lpDIKeyboard, NULL);
	if FAILED(r) goto Error;
	
	// Set data format
	r = lpDIKeyboard->SetDataFormat(&c_dfDIKeyboard); 
	if FAILED(r) goto Error;

	// Set keyboard buffer size
	r = lpDIKeyboard->SetProperty(DIPROP_BUFFERSIZE, &dipdw.diph);
	if FAILED(r) goto Error;
	
	// Set the cooperative level 
	r = lpDIKeyboard->SetCooperativeLevel(hMainWnd,
			DISCL_BACKGROUND | DISCL_NONEXCLUSIVE); 
	if FAILED(r) goto Error;

	// Set event notification
	r = lpDIKeyboard->SetEventNotification((HANDLE)DI_KeyboardEvent);
	if FAILED(r) goto Error;

	// Aquire keyboard device
	if (DI_AcquireKeyboard()) goto Error;

    // Get system repeat default
    SystemParametersInfo(SPI_GETKEYBOARDDELAY, 0, &DI_KeyDelay, 0);
    SystemParametersInfo(SPI_GETKEYBOARDSPEED, 0, &DI_KeyRepeat, 0);

    DI_KeyDelay *= 100;
    DI_KeyRepeat = 1000/DI_KeyRepeat;

	// Create repeat timer
	KeyboardTimerId = timeSetEvent(1000/REPEATTIME, 0, KeyboardTimer, NULL, TIME_PERIODIC);

	// Initialize key array
	SetEvent((HANDLE)DI_KeyboardEvent);
	
	return 0;

Error:
	DI_DoneKeyboard();
	return 1;
}


/****************************************************************************
 * Mouse functions															*
 ****************************************************************************/

LPDIRECTINPUTDEVICE		lpDIMouse = NULL;
int DI_MouseInited=0;
long DI_MouseEvent=0;
int DI_Mickeys=0;
void (*DI_MouseInt)(int dx, int dy, int px, int py, int but)=NULL;
bool MouseHidden = false;
int DI_Mousex = 0;
int DI_Mousey = 0;


/*
 * DI_PollMouse
 *
 * Polls the mouse object for mouse events and calls Allegro's mouse handler
 */
void DI_PollMouse()
{
	if (DI_MouseInited==0) return;

	int Movex=0;
	int Movey=0;
	static int MouseBut=0;

	EnterSemaphore();
	
	int Quit=0;
	while (!Quit && lpDIMouse)
	{
        DIDEVICEOBJECTDATA od;
        DWORD Num = 1;

		HRESULT hr = lpDIMouse->GetDeviceData(
                             sizeof(DIDEVICEOBJECTDATA), 
                             &od,
                             &Num, 0);

		if (hr == DIERR_INPUTLOST) 
		{
            if (DI_AcquireMouse()!=0) break;
			continue;
        } else 
			if (FAILED(hr) || Num==0) break;

		switch (od.dwOfs) 
		{
		case DIMOFS_X: Movex += od.dwData; break;
		case DIMOFS_Y: Movey += od.dwData; break;
        case DIMOFS_BUTTON0:
            if (od.dwData & 0x80) MouseBut |= 1; else MouseBut &= ~1;
            break;
		case DIMOFS_BUTTON1:
            if (od.dwData & 0x80) MouseBut |= 2; else MouseBut &= ~2;
			break;
        }
	}        

	// Get immediate data
	POINT p;
	if (GetCursorPos(&p)!=0)
	{
		int x = p.x - ClientPosX;
		int y = p.y - ClientPosY;
		
		if (x>=0 && y>=0 &&
			x<DD_Width && y<DD_Height)
		{
			DI_Mousex = x;
			DI_Mousey = y;
		}
	}

	(*DI_MouseInt)(Movex, Movey, DI_Mousex, DI_Mousey, MouseBut);
	LeaveSemaphore();
}


/*
 * DI_AcquireMouse
 *
 * Acquires the mouse DirectInputDevice
 */
int DI_AcquireMouse()
{
	if (DI_MouseInited)
	{			
		// Aquire mouse device		
		HRESULT r = lpDIMouse->Acquire();
		if FAILED(r) return 1;

		return 0;
	} else return 1;
}


/*
 * DI_AcquireMouse
 *
 * Releases the mouse DirectInputDevice
 */
void DI_UnAcquireMouse()
{
	if (DI_MouseInited)
	{
		// Unaquire mouse device
		lpDIMouse->Unacquire();
	}
}


/*
 * DI_DoneMouse
 *
 * Shuts down the mouse driver
 */
void DI_DoneMouse()
{
	if (DI_MouseInited==0) return;
	DI_MouseInited = 0;
	
	if (lpDIMouse) 
    {   
		lpDIMouse->Unacquire();
		lpDIMouse->Release();
		lpDIMouse = NULL; 
    } 
}


/*
 * DI_InitMouse
 *
 * Initialises the mouse driver and sets the mouse handler
 */
int DI_InitMouse(void (*MouseInt)(int dx, int dy, int px, int py, int but), int Exclusive)
{
	HRESULT	r;

	DIPROPDWORD dipdw =
    {
        // the header
        {
            sizeof(DIPROPDWORD),        // diph.dwSize
            sizeof(DIPROPHEADER),       // diph.dwHeaderSize
            0,                          // diph.dwObj
            DIPH_DEVICE,                // diph.dwHow
        },
        // the data
        DINPUT_BUFFERSIZE,              // dwData
    };

	if (DI_MouseInited>0) return 0;
	DI_MouseInited = 1;
	DI_MouseInt = MouseInt;

	// Get mouse device
	r = lpDI->CreateDevice(GUID_SysMouse, &lpDIMouse, NULL);
	if FAILED(r) goto Error;
	
	// Set mouse data format
	r = lpDIMouse->SetDataFormat(&c_dfDIMouse); 
	if FAILED(r) goto Error;
	
	// Set mouse cooperative level 
	/*if (Exclusive)
		r = lpDIMouse->SetCooperativeLevel(hMainWnd,
				DISCL_EXCLUSIVE | DISCL_FOREGROUND);
	else
		r = lpDIMouse->SetCooperativeLevel(hMainWnd,
				DISCL_NONEXCLUSIVE | DISCL_BACKGROUND);*/
	r = lpDIMouse->SetCooperativeLevel(hMainWnd,
				DISCL_NONEXCLUSIVE | DISCL_FOREGROUND);
	if FAILED(r) goto Error;

	DI_HideMouse();

	// Set mouse buffer size
	r = lpDIMouse->SetProperty(DIPROP_BUFFERSIZE, &dipdw.diph);
	if FAILED(r) goto Error;

	// Set event notification
	r = lpDIMouse->SetEventNotification((HANDLE)DI_MouseEvent);
	if FAILED(r) goto Error;

	// Aquire mouse device	
	DI_AcquireMouse();

	
	return 0;

Error:
	DI_DoneMouse();
	return 1;
}


/*
 * DI_ShowMouse
 *
 * Shows Windows' mouse pointer
 */
void DI_ShowMouse()
{
	if (MouseHidden)
	{		
		MouseHidden = false;
		DI_UpdateMouseCursor();
	}
}


/*
 * DI_HideMouse
 *
 * Hides Windows' mouse pointer
 */
void DI_HideMouse()
{
	if (!MouseHidden)
	{		
		MouseHidden = true;
		DI_UpdateMouseCursor();
	}
}



/*
 * DI_UpdateMouseCursor
 */
bool PreviousMouse = 1;

void DI_UpdateMouseCursor()
{
	HCURSOR NewCursor;
	//if (PreviousMouse!=MouseHidden || !MouseHidden)
	{
		if (MouseHidden)
		{			
			NewCursor = NULL;
		} else
		{		
			NewCursor = LoadCursor(NULL, IDC_ARROW);
		}

		if (SetCursor(NewCursor)!=NewCursor)
		{
			LOG ("DI_UpdateMouseCursor - Cursor changed");
	
			POINT p;
			GetCursorPos(&p);	
			SetCursorPos(p.x, p.y);
		}		
	}

	PreviousMouse = MouseHidden;
}

/*
 * DI_MoveMouse
 */
void DI_MoveMouse(int x, int y)
{
	SetCursorPos(ClientPosX+x, ClientPosY+y);
}


/****************************************************************************
 * Joystick functions														*
 ****************************************************************************/

int DI_JoystickInited = 0;

int JoysticksNum = 0;
int JoystickAttached = 0;

struct JOYSTICK DI_Joysticks[MAXJOYSTICKS];

/*
 * DI_PollJoystick
 *
 * Poll the joysticks devices
 */
void DI_PollJoystick()
{
	JoystickAttached = 0;

	for (int n=0; n<JoysticksNum; n++)
	{
		MMRESULT r;
		JOYINFOEX js;

		// Get joystick data
		js.dwSize = sizeof(js);
		js.dwFlags = JOY_RETURNALL;

		r = joyGetPosEx(n, &js);
		if (r==JOYERR_NOERROR)
		{
			JoystickAttached++;

			// Get axes
			DI_Joysticks[n].Axes[0] = js.dwXpos;
			DI_Joysticks[n].Axes[1] = js.dwYpos;
			DI_Joysticks[n].Axes[2] = js.dwZpos; // Throttle
			
			// Get hat
			switch (js.dwPOV)
			{
			case JOY_POVLEFT:		DI_Joysticks[n].Hat = 1; break;
			case JOY_POVBACKWARD:	DI_Joysticks[n].Hat = 2; break;
			case JOY_POVRIGHT:		DI_Joysticks[n].Hat = 3; break;
			case JOY_POVFORWARD:	DI_Joysticks[n].Hat = 4; break;
			default:				DI_Joysticks[n].Hat = 0;
			}
				
			// Get buttons
			for (int b=0; b<DI_Joysticks[n].ButtonsNum; b++)
			{
				DI_Joysticks[n].Buttons[b] = ((js.dwButtons & (1 << b))!=0);
			}

			for (; b<MAXBUTTONS; b++)
			{
				DI_Joysticks[n].Buttons[b] = 0;
			}
		} else
		{
			DI_Joysticks[n].Axes[0] = 0;
			DI_Joysticks[n].Axes[1] = 0;
			DI_Joysticks[n].Axes[2] = 0; // Throttle

			DI_Joysticks[n].Hat = 0;

			for (int b=0; b<MAXBUTTONS; b++)
			{
				DI_Joysticks[n].Buttons[b] = 0;
			}
		}
	}
}


/*
 * DI_DoneJoystick
 *
 * Shuts down the jyostick driver
 */
void DI_DoneJoystick()
{
	if (DI_JoystickInited>1) DI_JoystickInited--;
	else
		if (DI_JoystickInited==1)
		{
			DI_JoystickInited--;

			JoysticksNum = 0;
		}
}


/*
 * DI_InitJoystick
 *
 * Initialise joysticks driver
 */
int DI_InitJoystick()
{
	JOYCAPS caps;

	// Initialize vars
	JoysticksNum = 0;	

	// Joystick is initialized
	DI_JoystickInited++;

	JoysticksNum = joyGetNumDevs();

	if (JoysticksNum>MAXJOYSTICKS) 
	{
		JoysticksNum=MAXJOYSTICKS;
		OutputDebugString("Too many joysticks found!");
	}

	// Get joystick calibration
	for (int n=0; n<JoysticksNum; n++)
	{
		int r = joyGetDevCaps(n, &caps, sizeof(caps));		
		if (r==JOYERR_NOERROR)
		{
			DI_Joysticks[n].ButtonsNum = caps.wNumButtons;
			DI_Joysticks[n].AxesNum = caps.wNumAxes;
			
			for (int a=0; a<MAXAXES; a++)
			{
				DI_Joysticks[n].Min[a] = 0;
				DI_Joysticks[n].Max[a] = 0;
			}

			DI_Joysticks[n].Min[0] = caps.wXmin;
			DI_Joysticks[n].Max[0] = caps.wXmax;

			DI_Joysticks[n].Min[1] = caps.wYmin;
			DI_Joysticks[n].Max[1] = caps.wYmax;

			DI_Joysticks[n].Min[2] = caps.wZmin;
			DI_Joysticks[n].Max[2] = caps.wZmax;

			if (caps.wCaps & JOYCAPS_HASPOV)
				DI_Joysticks[n].HasHat = 1;
			else
				DI_Joysticks[n].HasHat = 0;
		} else
		{
			DI_Joysticks[n].ButtonsNum = 0;
			DI_Joysticks[n].AxesNum = 0;
			for (int a=0; a<MAXAXES; a++)
			{
				DI_Joysticks[n].Min[a] = 0;
				DI_Joysticks[n].Max[a] = 0;
			}
		}
	}

	// Poll joysticks
	DI_PollJoystick();

	return !(JoystickAttached>0);
}


/*
 * DI_GetJoysticks
 *
 * Returns a pointer to the joystick structure
 */
struct JOYSTICK *DI_GetJoysticks()
{
	return DI_Joysticks;
}


/****************************************************************************
 * General functions														*
 ****************************************************************************/

/*
 * DI_Done
 *
 * Shuts down the whole DirectInput system
 */
void DI_Done()
{
	if (DI_Inited==0) return;
	DI_Inited--;
	if (DI_Inited>0) return;
	
	if (lpDI)
	{
        lpDI->Release();
        lpDI = NULL; 
	}

	CloseHandle((HANDLE)DI_MouseEvent);
	CloseHandle((HANDLE)DI_KeyboardEvent);
}


/*
 * DI_Init
 *
 * Initialise the DirectInput system
 */
int DI_Init()
{
	HRESULT	r;
	
	DI_Inited++;

	if (DI_Inited>1) return 1;

	// Get Direct Input interface
	r = DirectInputCreate(hInst, DIRECTINPUT_VERSION, &lpDI, NULL); 
	if (FAILED(r)) return 1;

	DI_MouseEvent = (long)CreateEvent(NULL, FALSE, FALSE, NULL);
	DI_KeyboardEvent = (long)CreateEvent(NULL, FALSE, FALSE, NULL);

	return 0;
}
