/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *		Main thread and window proc (windows file), by Stefan Schimanski (1Stein@gmx.de)
 *
 *      See readme.txt for copyright information.
 */


#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <winbase.h>
#include <process.h>
#include <stdio.h>
#include <mmsystem.h>
#include <iostream.h>
#include <fstream.h>
#include "winintrn.h"
#include "resource.h"

/************************************************************************/

int WN_Argc;
char *WN_Argv[256];

// thread priorities
#ifdef _DEBUG
const MainPriority = THREAD_PRIORITY_NORMAL;
const InputPriority = THREAD_PRIORITY_NORMAL;
#else
const MainPriority = THREAD_PRIORITY_ABOVE_NORMAL;
const InputPriority = THREAD_PRIORITY_HIGHEST;
#endif

// Win32 handles
HINSTANCE hInst;					// instance handle
HWND hMainWnd=NULL;					// main window
HANDLE hInputThread;				// thread that polls DirectInput objects
HANDLE hMainThread;					// thread that executes the Allegro code
HPALETTE SavedPal;					// handle to the Win32 palette of the desktop

// Common vars
char CmdLine[256];					// program command line
int ClientPosX=0;					// absolute position of the client area
int ClientPosY=0;
int AppBackground=0;				// set when the program is in background

// Synchronisation objects
HANDLE EndInputThreadEvent;			// event to stop the input thread

// Window proc callbacks
const MaxWndProcCallbacks =  16;	// max. number of callbacks
WNDPROC WndProcCallbacks[MaxWndProcCallbacks]; // pointers to callback functions
int WndProcCallbacksNum = 0;		// Number of callbacks currently registered

/********************************************************
 * Utility functions									*
 ********************************************************/

/*
 * WinAllegro_GetWindowsVersion
 *
 * Returns 0x100 for a WinNT system and 4 for a Windows 95 system
 */
int WinAllegro_GetWindowsVersion()
{
	int v = GetVersion();
	if (v<0x80000000) return 0x100;
	else return 4;
}


/*
 * WinAllegro_DriveExist
 *
 * Returns whether the drive exists
 */
int WinAllegro_DriveExist(int Drive)
{
    return (GetLogicalDrives() & (1 << Drive))!=0;
}


/*
 * WinAllegro_IsForeground
 *
 * Returns whether the application is in foreground
 */
int WinAllegro_IsForeground()
{
	return !AppBackground;
}


/*
 * WinAllegro_GetWindowHandle
 *
 * Returns handle of main window
 */
HWND WinAllegro_GetWindowHandle()
{
	return hMainWnd;
}


/*
 * WinAllegro_SetWindowTitle
 *
 * Set title of main window
 */
void WinAllegro_SetWindowTitle(char *Title)
{
	SetWindowText(hMainWnd, Title);
}


/*
 * WinAllegro_SetWindowIcon
 *
 * Set application icon
 */
void WinAllegro_SetWindowIcon(HICON hIcon)
{
	// TODO: How do I do this?
}


/*
 * WinAllegro_AddWndProcCallback
 *
 * Add callback for main window proc
 */
int WinAllegro_AddWndProcCallback(WNDPROC Callback)
{
	if (WndProcCallbacksNum>=MaxWndProcCallbacks || Callback==NULL) return -1;
	
	WndProcCallbacks[WndProcCallbacksNum] = Callback;
	WndProcCallbacksNum++;
	return 0;
}


/*
 * WinAllegro_RemoveWndProcCallback
 *
 * Remove callback for window proc
 */
void WinAllegro_RemoveWndProcCallback(WNDPROC Callback)
{
	for (int n=0; n<WndProcCallbacksNum; n++)
	{
		if (WndProcCallbacks[n]==Callback) break;
	}

	if (n<WndProcCallbacksNum)
	{
		memmove(WndProcCallbacks[n], WndProcCallbacks[n+1], 
			sizeof(WNDPROC)*(WndProcCallbacksNum-n-1));
		WndProcCallbacksNum--;
	}
}


/*
 * AcquireMouse
 *
 * Posts a message to the message queue to reqcquire the mouse device
 */
void AcquireMouse()
{
	PostMessage(hMainWnd, WM_ACQUIREMOUSE, 0, 0);
}


/*
 * ParseCommandLine
 *
 * Parses the command line of the program and puts the result in Argc and Argv
 */
void ParseCommandLine(char *s)
{
	int n=0;
	strcpy(CmdLine, s);
	s = CmdLine;

	while (*s)
	{			
		while (*s==' ' || *s=='"') 
		{
			*s=0;
			s++;
		}

		if (*s)
		{
			WN_Argv[n] = s;
			n++;
		}

		while (*s && *s!=' ' && *s!='"') s++;
	}

	WN_Argc = n;
}


/*
 * WinAllegro_GetArgc
 */
int WinAllegro_GetArgc()
{
	return WN_Argc;
}


/*
 * WinAllegro_GetArgv
 */
char **WinAllegro_GetArgv()
{
	return WN_Argv;
}


/*
 * ShowApp
 *
 * Show the application in foreground
 */
BOOL ShowApp()
{
	DBEG("ShowApp - start");

	if (AppBackground==1)
	{
		D3("ShowApp - EnterSemaphore");
		EnterSemaphore();

   		// Resume main thread
		AppBackground=0;
        SetThreadPriority((HANDLE)hMainThread, MainPriority);
      
		// Reaquire mouse because it could be lost
		D3("ShowApp - AcquireMouse");
		AcquireMouse();

		// Reaquire keyboard because it could be lost
		D3("ShowApp - AcquireKeyboard");
		DI_AcquireKeyboard();

		// Get new keyboard state
        D3("ShowApp - Ack for new keyboard state");
		PulseEvent((HANDLE)DI_KeyboardEvent);

		// Reduce GDI palette to first and last color
		D3("ShowApp - Restore palette");
		DD_ReduceGDIPal();
		DD_SetCurrentPalette();
       
        // Restore timer
        D3("ShowApp - Restart timer");
        InitTimer();        
        
		D3("ShowApp - LeaveSemaphore");
		LeaveSemaphore();
	} else
	{
		D2("ShowApp - Called not synchroniously with HideApp");
	}

	DEND("ShowApp - end");
	
	return FALSE;
}


/*
 * HideApp
 *
 * Hides the app to go in background
 */
BOOL HideApp()
{
	DBEG("HideApp - start");

	if (AppBackground==0)
	{
		D3("HideApp - EnterSemaphore");
		EnterSemaphore();

        // Stop timer
        D3("HideApp - Stop timer");
        DoneTimer();
	
		// Restore GDI
		D3("HideApp - ExpandGDIPal");
		DD_ExpandGDIPal();
	
		// Suspend main thread
		D3("HideApp - Suspend main thread");		
		AppBackground=1;			    
        SetThreadPriority((HANDLE)hMainThread, THREAD_PRIORITY_NORMAL);

		D3("HideApp - LeaveSemaphore");
		LeaveSemaphore();	
	} else
	{
		D2("HideApp - Called not synchroniously with ShowApp");
	}

	DEND("HideApp - end");

	return FALSE;
}


/*
 * MainWndProc
 *
 * Main windows callback
 */
LRESULT WINAPI MainWndProc(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
	for (int n=0; n<WndProcCallbacksNum; n++)
	{
		(*WndProcCallbacks[n])(hWnd, Msg, wParam, lParam);
	}

	switch (Msg)
	{	
	case WM_CREATE:
		LOG("MainWndProc - WM_CREATE");		
		DD_ReduceGDIPal();
		break;
		
	case WM_DESTROY:
		LOG("MainWndProc - WM_DESTROY");		
		DD_ExpandGDIPal();			
  		PostQuitMessage(0);
		break;

	case WM_CLOSE:
		LOG("MainWndProc - WM_CLOSE");
		return 1;		

	case WM_SYSCHAR:
		return 1;

	case WM_KEYUP:
	case WM_SYSKEYUP:
	case WM_SYSKEYDOWN:    	
		return 1;

	case WM_ACTIVATEAPP:
		if ((BOOL)wParam==TRUE)
		{
			LOG("MainWndProc - WM_ACTIVATEAPP(TRUE)");
			PostMessage(hMainWnd, WM_STOPAPPPAUSE, 0, 0);
		} else 
		{
			LOG("MainWndProc - WM_ACTIVATEAPP(FALSE)");			
			HideApp();
		}
		break;

	case WM_SYSCOMMAND:
		LOG("MainWndProc - WM_SYSCOMMAND");
		switch (wParam & ~0x0F)
		{
		case SC_SCREENSAVE:
		case SC_MONITORPOWER:
			return 0;
		}
		break;

	case WM_STOPAPPPAUSE:
		LOG("MainWndProc - WM_STOPAPPPAUSE");
		ShowApp();
		break;

	case WM_SETCURSOR:
		DI_UpdateMouseCursor();
        return TRUE;		

	case WM_MOVE:
		ClientPosX = (SHORT) LOWORD(lParam);
		ClientPosY = (SHORT) HIWORD(lParam);
		break;

	case WM_ACQUIREMOUSE:
		DI_AcquireMouse();
		break;

	case WM_ERASEBKGND:
        D3("WM_ERASEBKGND");
        return 1;

    case WM_PAINT:
        D3("WM_PAINT");        
        break;

    case WM_NCPAINT:
        D3("WM_NCPAINT");
        break;
	}
		
	return DefWindowProc( hWnd, Msg, wParam, lParam );            	
} 


/*
 * WindowThread
 */
void WinAllegro_CreateMainWindow()
{
	// Register window class
	WNDCLASS wc;

	wc.lpszClassName = "WinAllegroApp";
	wc.lpfnWndProc = MainWndProc;
	wc.style = CS_VREDRAW | CS_HREDRAW;
	wc.hInstance = hInst;
	wc.hIcon = NULL;
	wc.hCursor = NULL;
	wc.hbrBackground = (HBRUSH)( COLOR_WINDOW+1 );
	wc.lpszMenuName = NULL;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	
	RegisterClass( &wc );

	// Create main windows
	LOG("InitApp - CreateWindowEx");
	hMainWnd = CreateWindowEx(
		WS_EX_TOPMOST,
		"WinAllegroApp",
		"WinAllegro Application",		
		WS_POPUP | WS_CAPTION,
		0,
		0,
		320,
		200,
		NULL,
		NULL,
		hInst,
		NULL
	);

	// Show window
	LOG("InitApp - ShowWindow");
	ShowWindow(hMainWnd, SW_SHOWDEFAULT);

	// Message loop
	MSG msg;
    
    while (PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
    {
	    if(!GetMessage( &msg, NULL, 0, 0 ))
	      break;		    

	    TranslateMessage(&msg); 
	    DispatchMessage(&msg);
	}
}


/*
 * WinAllegro_SetWndProc
 */
WNDPROC WinAllegro_SetWndProc(WNDPROC WndProc)
{
    return (WNDPROC)SetWindowLong(hMainWnd, GWL_WNDPROC, (LONG)WndProc);
}


/*
 * WinAllegro_HandleMessages
 */
int WinAllegro_HandleMessages()
{
    MSG msg;

    while (PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE) || AppBackground)
    {        
	    if(!GetMessage( &msg, NULL, 0, 0 ))
	    {
		    return -1;
	    }

	    TranslateMessage(&msg); 
	    DispatchMessage(&msg);

        // Handle messages avoiding time consuming looping with PeekMessage
        while (AppBackground)
        {
            if(!GetMessage( &msg, NULL, 0, 0 ))
    	    {
	    	    return -1;
	        }

	        TranslateMessage(&msg); 
	        DispatchMessage(&msg);
        }
	}

    return 0;
}


/*
 * InputThread
 */
void InputThread(void *pParam)
{
	MSG Msg;

	LOG("InputThread - start");		
	
	// Input loop
	while (TRUE)
	{
		HANDLE Events[3] = 
		{ 						
			(HANDLE)DI_KeyboardEvent,
			(HANDLE)DI_MouseEvent,
			EndInputThreadEvent,
		};					

		int r = WaitForMultipleObjects(3, Events, FALSE, INFINITE);
				
		switch (r)
		{
		case WAIT_OBJECT_0 + 0:
   			DI_PollKeyboard();
			break;

		case WAIT_OBJECT_0 + 1:
			DI_PollMouse();
			break;

		case WAIT_OBJECT_0 + 2:
			LOG("InputThread - End event received");
			goto Quit;

		case WAIT_OBJECT_0 + 3:
			LOG("InputThread - Message queue filled");
			while (PeekMessage(&Msg, NULL, 0, 0, PM_REMOVE))
			{
				LOG("ThreadInput - Message retrieved");
			}
			break;
 
		default:  
			LOG("InputThread - Unknown event");
			goto Quit;
		}
	}

Quit:
	LOG("InputThread - end");

	_endthreadex(0);
}


/*
 * WinAllegro_Init
 */
int WinAllegro_Init(HINSTANCE hInstance)
{   
    if (InitDebug()!=0) return -1;
	
	DBEG("WN_Init - start");

	// Init globals
	hInst = hInstance;
	WndProcCallbacksNum = 0;
	
	// Get main thread handle
	HANDLE hCurrentThread = GetCurrentThread();
	HANDLE hCurrentProcess = GetCurrentProcess();
	DuplicateHandle(hCurrentProcess, hCurrentThread, 
					hCurrentProcess, &hMainThread,
					0, FALSE, DUPLICATE_SAME_ACCESS);
    SetThreadPriority((HANDLE)hMainThread, MainPriority);

	// Parse command line
	D3("InitApp - ParseCommandLine");
	ParseCommandLine(GetCommandLine());

	// Create semaphore
	D3("InitApp - InitSync");
	if (InitSync())
	{
		MessageBox(NULL, 
			"Can't init synchronisation.", 
			"WinAllegro Error",
			MB_OK | MB_ICONERROR);
		return 1;
	}

	// Start window thread
	D3("WN_Init - Create window");
    WinAllegro_CreateMainWindow();
	
	// Window created?    
	if (hMainWnd==NULL)
	{
        D3("WN_Init - Window creation failed");
		return 1;
	}
	
	// Setup DirectInput
	D3("WN_Init - DI_Init");
	if (DI_Init()!=NULL)
	{
		MessageBox(NULL, 
			"Can't setup DirectInput.", 
			"WinAllegro Error",
			MB_OK | MB_ICONERROR);
		return 1;
	}	

	// Start input thread
	D3("WN_Init - Starting input thread");
	EndInputThreadEvent = CreateEvent(NULL, FALSE, FALSE, NULL);	
	hInputThread  = (HANDLE)_beginthread(InputThread, 0, NULL);
	if ((long)hInputThread==-1)
	{
		MessageBox(NULL, 
			"Can't create input thread.", 
			"WinAllegro Error",
			MB_OK | MB_ICONERROR);
		return 1;
	};	
	SetThreadPriority((HANDLE)hInputThread, InputPriority);

	// Intialize DirectDraw
	D3("InitApp - DD_Init");
	if (DD_Init())
	{
		MessageBox(NULL, 
			"To run WinAllegro programs DirectDraw 3.0 or newer is needed.", 
			"WinAllegro Error",
			MB_OK | MB_ICONERROR);
		return 1;		
	}

    // Setup timer
    D3("InitApp - InitTimer");
    if (InitTimer()!=0)
    {
        D1("InitApp - InitTimer failed");
        return 1;
    }

	
	DEND("WN_Init - end");

	return 0;
}


/*
 * WinAllegro_Exit
 */
void WinAllegro_Exit()
{
	DBEG("WN_Exit - start");
	
    // Destroy timer
    DoneTimer();

	// Shutdown DirectDraw
	D3("WN_Exit - Shutdown DirectDraw");
	DD_Done();
	
	// Shutdown DirectInput thread
	D3("WN_Exit - Stop input thread");
	SetEvent(EndInputThreadEvent);
	WaitForSingleObject(hInputThread, INFINITE);

	// Shutdown DirectInput
	D3("WN_Exit - DI_Done");
	DI_Done();
	
	// Destory main window
	D3("WN_Exit - Destroy main window");
    DestroyWindow(hMainWnd);
    hMainWnd = NULL;
		
	// Shutdown synchronisation
	D3("WN_Exit - Shutdown synchronisation");
	DoneSync();

	// Shutdown DebugLog
	DEND("WN_Exit - end");

	// Close still open handles	
	CloseHandle(hInputThread);
	CloseHandle(hMainThread);
	CloseHandle(EndInputThreadEvent);
	
    DoneDebug();
}

