/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *		DirectSound access functions (windows file), by Stefan Schimanski (1Stein@gmx.de)
 *
 *      See readme.txt for copyright information.
 */

#define INITGUID
#define DIRECTSOUND_VERSION 0x0300

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <process.h>
#include <mmsystem.h>
#include <dsound.h>
#include "winintrn.h"


/*****************************************************************************/


LPDIRECTSOUND lpDirectSound = NULL; 

int DS_Frequency  = 44100;
int DS_Stereo		= 0;
int DS_Bits		= 8;
int DS_Primary = 0;


/*
 * DS_Init
 * 
 * Initialise the DirectSound driver
 */
int DS_Init(int aTryPrimary)
{
	HRESULT       hr;
	DSCAPS dscaps; 

    DBEG("DS_Init");

	// Initialize DirectSound interface
    D3("DS_Init - DirectSoundCreate");
	hr = DirectSoundCreate(NULL, &lpDirectSound, NULL);
    if (hr!=DS_OK)
    {
        D4("DS_Init - Can't create DirectSound interface hr=%x", hr);
        goto Error;
    }

	// Set cooperative level
    D3("DS_Init - SetCooperativeLevel");

    if (aTryPrimary)
    {
	    hr = lpDirectSound->SetCooperativeLevel(hMainWnd, DSSCL_WRITEPRIMARY);
        if (hr!=DS_OK)
        {
            D3("DS_Init - WRITEPRIMARY not possible");
            DS_Primary = false;
    
            hr = lpDirectSound->SetCooperativeLevel(hMainWnd, DSSCL_PRIORITY);
            D3("DS_Init - Trying coop level PRIORITY");
        } else
        {
            D3("DS_Init - Coop level set to WRITEPRIMARY");        
            DS_Primary= true;
        }
    } else
    {
        hr = lpDirectSound->SetCooperativeLevel(hMainWnd, DSSCL_PRIORITY);
        if (hr==DS_OK)
        {
            D3("DS_Init - Coop level set to PRIORITY");
        }
    }

	if (hr!=DS_OK)
    {
        char *s;
                
        switch (hr)
        {
        case DSERR_ALLOCATED:       s="DSERR_ALLOCATED"; break;
        case DSERR_INVALIDPARAM:    s="DSERR_INVALIDPARAM"; break;
        case DSERR_UNINITIALIZED:   s="DSERR_UNINITIALIZED"; break;
        case DSERR_UNSUPPORTED:     s="DSERR_UNSUPPORTED"; break;
        default: s="unknown";
        }

        D4("DS_Init - Can't set cooperative level hr=%s", s);

        goto Error;
    }

	// Get hardware capabilities
    D3("DS_Init - GetCaps");
	dscaps.dwSize = sizeof(DSCAPS);
	hr = lpDirectSound->GetCaps(&dscaps);
	if (hr!=DS_OK)
    {
        D4("DS_Init - Can't get caps hr=%x", hr);
        goto Error;
    }

    if (dscaps.dwFlags & DSCAPS_PRIMARY16BIT)
	    DS_Bits=16; else DS_Bits=8;

	if (dscaps.dwFlags & DSCAPS_PRIMARYSTEREO)
    	DS_Stereo=1; else DS_Stereo=0;

	if (dscaps.dwMaxSecondarySampleRate>44000)
        DS_Frequency=44100; else DS_Frequency=22050;

    D5("DS_Caps: %u bits, %s, %uHz", DS_Bits, DS_Stereo?"stereo":"mono", DS_Frequency);
    
    DEND("DS_Init");
	return 0;

Error:
    D3("DS_Init - Error");
	DS_Done();

    DEND("DS_Init");
	return 1;
}


/*
 * DS_Done
 *
 * Shuts down the DirectSound driver
 */
void DS_Done()
{
    DBEG("DS_Done");	
	if (lpDirectSound)
	{
		lpDirectSound->Release();
		lpDirectSound = NULL;
	}
	DEND("DS_Done");
}


/*
 * DS_Detects
 *
 * Returns whether DirectSound can be used to access a sound hardware
 */
int DS_Detect()
{
    DBEG("DS_Detect");

    if (lpDirectSound==NULL)
    {
        D3("DS_Detect - DirectSound not initialiased");

        // Initialize DirectSound interface
        D3("DS_Detect - Creating DirectSound interface");

    	HRESULT hr = DirectSoundCreate(NULL, &lpDirectSound, NULL);
	    if (hr!=DS_OK)
        {
            DEND("DS_Detect - Creation failed");
            return FALSE;
        }

        D3("DS_Detect - Creation successfull");

        // Release DirectSound
        D3("DS_Detect - Releasing DirectSound");
        lpDirectSound->Release();
		lpDirectSound = NULL;
    }

    DEND("DS_Detect");
    
    return TRUE;
}


/*
 * DS_IsPrimaryPossible
 *
 * Returns whether WRITEPRIMARY coop level is set
 */
int DS_IsPrimaryPossible()
{
    return DS_Primary;
}


/*
 * DS_GetCaps
 *
 * Returns capabilities for the installed sound card
 */
void DS_GetCaps(int *aFreq, int *aIs16Bit, int *aIsStereo)
{
	*aFreq	  = DS_Frequency;
	*aIs16Bit  = DS_Bits==16;
	*aIsStereo = DS_Stereo;    
}


/*
 * DS_Lock
 *
 * Locks the buffer memory of a special voice (with position and length)
 * It is possible that DirectSound doesn't return a valid pointer when it is a playing
 * at the moment at this position
 */
int DS_Lock(struct DS_VOICE *aVoice, int aPos, int aLength, void **aBufA, unsigned long *aSizeA, void **aBufB, unsigned long *aSizeB)
{
    DBEG("DS_Lock(%u)", aPos);
	if (aVoice)
	{
		HRESULT hr; 
			
		// Obtain memory address of write block. This will be in two parts
		// if the block wraps around.
		hr = aVoice->SoundBuffer->Lock(aPos, aLength, aBufA, aSizeA, aBufB, aSizeB, 0);
	
		// If DSERR_BUFFERLOST is returned, restore and retry lock. 
		if (hr==DSERR_BUFFERLOST)
		{
            D3("DS_Lock - Buffer lost");
			aVoice->SoundBuffer->Restore();	
			hr = aVoice->SoundBuffer->Lock(aPos, aLength, aBufA, aSizeA, aBufB, aSizeB, 0);
		}

		if (hr!=DS_OK)
        {
            D2("DS_Lock - Lock failed");
            return -1;
        }
		
		aVoice->BufA = *aBufA;
		aVoice->SizeA = *aSizeA;
		aVoice->BufB = *aBufB;
		aVoice->SizeB = *aSizeB;

		return 0;
	}

	return -1;
}


/*
 * DS_Unlock
 *
 * Unlocks a previously locked voice
 */
void DS_Unlock(struct DS_VOICE *aVoice)
{
	if (aVoice)
		aVoice->SoundBuffer->Unlock(aVoice->BufA, aVoice->SizeA, aVoice->BufB, aVoice->SizeB);     
}


/*
 * DS_Volume
 *
 * Set the main volume of the DirectSound interface
 */
void DS_Volume(int aVolume)
{
	/* Not implemented yet */
}


/*
 * DS_VoiceCaps
 *
 * Print voice to to debug output
 */
void DS_PrintVoiceCaps(DSBCAPS &DSBufferCaps)
{
    D5("DS_CreateVoice - Buffer caps");

    D5("  * dwFlags"); 

    if (DSBufferCaps.dwFlags & DSBCAPS_CTRL3D)               
    {
        D5("    - DSBCAPS_CTRL3D");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_CTRLFREQUENCY)
    {
        D5("    - DSBCAPS_CTRLFREQUENCY");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_CTRLPAN)
    {
        D5("    - DSBCAPS_CTRLPAN");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_CTRLVOLUME)
    {
        D5("    - DSBCAPS_CTRLVOLUME");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_GETCURRENTPOSITION2) 
    {
        D5("    - DSBCAPS_GETCURRENTPOSITION2");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_GLOBALFOCUS)          
    {
        D5("    - DSBCAPS_GLOBALFOCUS");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_LOCHARDWARE)
    {
        D5("    - DSBCAPS_LOCHARDWARE");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_LOCSOFTWARE)          
    {
        D5("    - DSBCAPS_LOCSOFTWARE");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_MUTE3DATMAXDISTANCE)  
    {
        D5("    - DSBCAPS_MUTE3DATMAXDISTANCE");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_PRIMARYBUFFER)       
    {
        D5("    - DSBCAPS_PRIMARYBUFFER");
    }

    if (DSBufferCaps.dwFlags & DSBCAPS_STATIC)
    {
        D5("    - DSBCAPS_STATIC");
    }
    
    if (DSBufferCaps.dwFlags & DSBCAPS_STICKYFOCUS)
    {
        D5("    - DSBCAPS_STICKYFOCUS");
    }
 
    D5("  * dwBufferBytes = %u", DSBufferCaps.dwBufferBytes);
    D5("  * dwUnlockTransferRate = %u KB/s", DSBufferCaps.dwUnlockTransferRate);
    D5("  * dwPlayCpuOverhead = %u%%", DSBufferCaps.dwPlayCpuOverhead);
}


/*
 * DS_CreateVoice
 *
 * Creates a voice based on a secondary DirectSoundBuffer
 */
struct DS_VOICE* DS_CreateVoice(int aIs16Bit, int aIsStereo, int aFreq, int Size)
{
    PCMWAVEFORMAT pcmwf;
    DSBUFFERDESC dsbdesc; 
    HRESULT hr;

    DBEG("DS_CreateVoice(%u bits, %s, %u Hz, %u bytes)", aIs16Bit?16:8, aIsStereo?"stereo":"mono", aFreq, Size);

    if (DS_Primary)
    {
        D1("DS_CreateVoice - Can't create voice in primary mode");
        DEND("DS_CreateVoice");
        return NULL;
    }

	// Set up wave format structure. 
    memset(&pcmwf, 0, sizeof(PCMWAVEFORMAT)); 
    pcmwf.wf.wFormatTag = WAVE_FORMAT_PCM; 
	pcmwf.wf.nChannels = (aIsStereo ? 2 : 1);
    pcmwf.wf.nSamplesPerSec = aFreq; 
    pcmwf.wf.nBlockAlign = (aIs16Bit ? 16 : 8) * pcmwf.wf.nChannels / 8; 
    pcmwf.wf.nAvgBytesPerSec = pcmwf.wf.nSamplesPerSec * pcmwf.wf.nBlockAlign; 
    pcmwf.wBitsPerSample = (aIs16Bit ? 16 : 8); 

    // Set up DSBUFFERDESC structure. 
    memset(&dsbdesc, 0, sizeof(DSBUFFERDESC)); // Zero it out. 
    dsbdesc.dwSize = sizeof(DSBUFFERDESC); 	
    dsbdesc.dwFlags = DSBCAPS_CTRLALL;
	dsbdesc.dwBufferBytes = Size;
    dsbdesc.lpwfxFormat = (LPWAVEFORMATEX)&pcmwf; 

	// Create Voice
	DS_VOICE *Voice= new DS_VOICE;	
	Voice->Looping = 0;
	Voice->Playing = 0;
    Voice->Size = Size;
    Voice->Freq = aFreq;
    Voice->Is16Bit = aIs16Bit;
    Voice->IsStereo = aIsStereo;
    
    // Create buffer. 
    hr = lpDirectSound->CreateSoundBuffer(&dsbdesc, &Voice->SoundBuffer, NULL); 
    if (hr!=DS_OK)
	{
        D3("DS_CreateVoice - failed hr=%x", hr);
		delete Voice;
		return NULL; 
	}

    // Check created buffer
    DSBCAPS DSBufferCaps;
    DSBufferCaps.dwSize = sizeof(DSBufferCaps);
    
    hr = Voice->SoundBuffer->GetCaps(&DSBufferCaps);

    if (hr==DS_OK)
    {
        DS_PrintVoiceCaps(DSBufferCaps);
    } else
    {
        D5("DS_CreateVoice - Can't get caps");
    }    
    
    DEND("DS_CreateVoice");
	
	return Voice;
}


/*
 * DS_CreateVoice
 *
 * Creates a voice based on a secondary DirectSoundBuffer
 */
struct DS_VOICE* DS_CreatePrimaryVoice(int aIs16Bit, int aIsStereo, int aFreq)
{
    WAVEFORMATEX pcmwf;
    DSBUFFERDESC dsbdesc; 
    HRESULT hr;
    
    DBEG("DS_CreatePrimaryVoice(%u bits, %s, %u Hz)", aIs16Bit?16:8, aIsStereo?"stereo":"mono", aFreq);

    if (!DS_Primary)
    {
        D1("DS_CreatePrimaryVoice - Can't create primary voice in non primary mode");
        DEND("DS_CreatePrimaryVoice");
        return NULL;
    }

	// Set up wave format structure. 
    memset(&pcmwf, 0, sizeof(PCMWAVEFORMAT)); 
    pcmwf.wFormatTag = WAVE_FORMAT_PCM; 
	pcmwf.nChannels = (aIsStereo ? 2 : 1);
    pcmwf.nSamplesPerSec = aFreq; 
    pcmwf.nBlockAlign = (aIs16Bit ? 16 : 8) * pcmwf.nChannels / 8; 
    pcmwf.nAvgBytesPerSec = pcmwf.nSamplesPerSec * pcmwf.nBlockAlign; 
    pcmwf.wBitsPerSample = (aIs16Bit ? 16 : 8); 
    pcmwf.cbSize = 0;

    // Set up DSBUFFERDESC structure. 
    memset(&dsbdesc, 0, sizeof(DSBUFFERDESC)); // Zero it out. 
    dsbdesc.dwSize = sizeof(DSBUFFERDESC); 	
    dsbdesc.dwFlags = DSBCAPS_PRIMARYBUFFER; //DSBCAPS_GETCURRENTPOSITION2
	dsbdesc.dwBufferBytes = 0;
    dsbdesc.lpwfxFormat = NULL; 

	// Create Voice
	DS_VOICE *Voice= new DS_VOICE;	
	Voice->Looping = 0;
	Voice->Playing = 0;
    Voice->Freq = aFreq;
    Voice->Is16Bit = aIs16Bit;
    Voice->IsStereo = aIsStereo;
    
    // Create buffer. 
    hr = lpDirectSound->CreateSoundBuffer(&dsbdesc, &Voice->SoundBuffer, NULL); 
    if (hr!=DS_OK)
	{
        D3("DS_CreatePrimaryVoice - CreateSoundBuffer failed hr=%x", hr);
		delete Voice;
		return NULL; 
	}

    // Set primary buffer format
    hr = Voice->SoundBuffer->SetFormat(&pcmwf); 
    if (hr!=DS_OK)
    {
        D1("DS_CreatePrimaryVoice - SetFormat - hr=%x", hr);
        Voice->SoundBuffer->Release();
        delete Voice;
        return NULL;
    }

    // Check created buffer
    DSBCAPS DSBufferCaps;
    DSBufferCaps.dwSize = sizeof(DSBufferCaps);
    
    hr = Voice->SoundBuffer->GetCaps(&DSBufferCaps);

    if (hr==DS_OK)
    {
        DS_PrintVoiceCaps(DSBufferCaps);

        Voice->Size = DSBufferCaps.dwBufferBytes;
    } else
    {
        D5("DS_CreateVoice - Can't get caps");
        Voice->SoundBuffer->Release();
        delete Voice;
        return NULL;
    }            
    
    DEND("DS_CreatePrimaryVoice");
	
	return Voice;
}

/*
 * DS_DestroyVoice
 *
 * Destroys a previously created voice
 */
void DS_DestroyVoice(struct DS_VOICE *aVoice)
{
	if (aVoice)
	{
		aVoice->SoundBuffer->Release();
		delete aVoice;
	}
}


/*
 * DS_StartVoice
 *
 * Start playing the voice at the current position (not from the beginning)
 */
void DS_StartVoice(struct DS_VOICE *aVoice)
{
    DBEG("DS_StartVoice");

	if (aVoice)
	{
		aVoice->Playing = 1;

		// Start playing
        HRESULT hr;

		if (aVoice->Looping)
			hr = aVoice->SoundBuffer->Play(0, 0, DSBPLAY_LOOPING);
		else
			hr = aVoice->SoundBuffer->Play(0, 0, 0);

        if (hr!=DS_OK)
        {
            D3("DS_StartVoice - failed hr=%u", hr);
        }
    } else
    {
        D3("DS_StartVoice - NULL pointer passed");
    }

    DEND("DS_StartVoice");    
}


/*
 * DS_StopVoice
 *
 * Stops a playing voice
 */
void DS_StopVoice(struct DS_VOICE *aVoice)
{
	if (aVoice)
	{
		if (aVoice->Playing)
		{
			aVoice->Playing = 0;
			aVoice->SoundBuffer->Stop();
		}
	}
}


/*
 * DS_AdjustVoice
 *
 * Adjust voice parameters
 */
void DS_AdjustVoice(struct DS_VOICE *aVoice, int aLooping)
{
	if (aVoice)
	{
		if (aVoice->Looping!=aLooping)
		{
			aVoice->Looping = aLooping;
			if (aVoice->Playing)
			{
				DS_StartVoice(aVoice);
			}
		}
	}
}


/*
 * DS_GetVoicePos
 *
 * Returns the current position of a playing voice (in samples)
 */
int DS_GetVoicePos(struct DS_VOICE *aVoice)
{
	if (aVoice)
	{
		DWORD Play, Write;

 		aVoice->SoundBuffer->GetCurrentPosition(&Play, &Write);

        D5("DS_GetVoicePos - Play %u, Write %u", Play, Write);

		return Play;
  	} else 
    {
        D3("DS_GetVoicePos - NULL pointer passed");
        return 0;
    }
}


/*
 * DS_SetVoicePos
 *
 * Move the play position of a voice
 */
void DS_SetVoicePos(struct DS_VOICE *aVoice, int aPos)
{
	if (aVoice)
		aVoice->SoundBuffer->SetCurrentPosition(aPos);
}


/*
 * DS_GetVoiceVol
 *
 * Returns the current volume of a voice
 */
int DS_GetVoiceVol(struct DS_VOICE *aVoice)
{
	if (aVoice)
	{
		LONG Vol;
		aVoice->SoundBuffer->GetVolume(&Vol);

		return Vol;
	} else return 0;		
}


/*
 * DS_SetVoiceVol
 *
 * Sets the volume of a voice
 */
void DS_SetVoiceVol(struct DS_VOICE *aVoice, int aVol)
{
	if (aVoice)
		aVoice->SoundBuffer->SetVolume(aVol);
}


/*
 * DS_GetVoiceFreq
 *
 * Returns the frequency of a voice
 */
int DS_GetVoiceFreq(struct DS_VOICE *aVoice)
{
	if (aVoice)
	{
		DWORD Freq;
		aVoice->SoundBuffer->GetFrequency(&Freq);

		return Freq;
	} else return 0;		
}


/*
 * DS_SetVoiceFreq
 *
 * Sets the frequency of a voice (in samples per second)
 */
void DS_SetVoiceFreq(struct DS_VOICE *aVoice, int aFreq)
{
	if (aVoice)
		aVoice->SoundBuffer->SetFrequency(aFreq);
}


/*
 * DS_GetVoicePan
 *
 * Returns the current stereo pan of a voice
 */
int DS_GetVoicePan(struct DS_VOICE *aVoice)
{
	if (aVoice)
	{
		LONG Pan;
		aVoice->SoundBuffer->GetPan(&Pan);

		return Pan;
	} else return 0;		
}


/*
 * DS_SetVoicePan
 *
 * Sets the stereo pan of a voice
 */
void DS_SetVoicePan(struct DS_VOICE *aVoice, int aPan)
{
	if (aVoice)
		aVoice->SoundBuffer->SetPan(aPan);
}