/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *      MCI Allegro digital audio driver, by Stefan Schimanski (1Stein@gmx.de)
 *
 *      See readme.txt for copyright information.
 */


#include <stdlib.h>
#include <string.h>
#include <math.h>


#define WIN32_LEAN_AND_MEAN
#define BITMAP WINDOWS_BITMAP
#include <windows.h>
#include <mmsystem.h>
#include <mmreg.h>
#include "win/windx.h"
#undef BITMAP
#undef RGB

/*#include "allegro.h"*/
#include "internal.h"


static int mci_init(int input, int voices);
static void mci_exit(int input);
static int mci_mixer_volume(int volume);
static int mci_detect(int input);

static char mci_desc[80] = "not initialised";


DIGI_DRIVER digi_mci =
{
   "MCI", 
   mci_desc,
   0, 0, MIXER_MAX_SFX, MIXER_DEF_SFX,
   mci_detect,
   mci_init,
   mci_exit,
   mci_mixer_volume,
   _mixer_init_voice,
   _mixer_release_voice,
   _mixer_start_voice,
   _mixer_stop_voice,
   _mixer_loop_voice,
   NULL,
   NULL,
   _mixer_get_position,
   _mixer_set_position,
   _mixer_get_volume,
   _mixer_set_volume,
   _mixer_ramp_volume,
   _mixer_stop_volume_ramp,
   _mixer_get_frequency,
   _mixer_set_frequency,
   _mixer_sweep_frequency,
   _mixer_stop_frequency_sweep,
   _mixer_get_pan,
   _mixer_set_pan,
   _mixer_sweep_pan,
   _mixer_stop_pan_sweep,
   _mixer_set_echo,
   _mixer_set_tremolo,
   _mixer_set_vibrato,
   0, 0,
   NULL, NULL, NULL, NULL, NULL, NULL
};


/********************************************************/

int mci_buf_size;
int mci_buf_smp_num;

int mci_freq, mci_16bit, mci_stereo;

HWAVEOUT mci_device;
WAVEHDR mci_wave_hdr[3];

char *mci_audio_buffer[3];
int mci_waiting_hdr = 0;
int mci_playing_hdr = 1;
int mci_unused_hdr = 2;
int mci_buffer_played = 0;

#define BASE_BUF_SIZE 4096
 //4096

/********************************************************
 * Mix some samples                                     *
 ********************************************************/
int mci_play_hdr(int hdr)
{
    MMRESULT mm_r;

    DBEG("mci_play_hdr(%d)", hdr);

    // Play header finally
    mm_r = waveOutWrite(mci_device, &(mci_wave_hdr[hdr]), sizeof(WAVEHDR));
    switch (mm_r)
    {
    case MMSYSERR_INVALHANDLE:
        D1("mci_play_buffer write - Specified device handle is invalid.");
        return -1;
    case MMSYSERR_NODRIVER:
        D1("mci_play_buffer write - No device driver is present.");
        return -1;
    case MMSYSERR_NOMEM:
        D1("mci_play_buffer write - Unable to allocate or lock memory.");
        return -1;
    case WAVERR_UNPREPARED:
        D1("mci_play_buffer write - The data block pointed to by the pwh parameter hasnt been prepared.");
        return -1; 
    }
}


int mci_prepare_hdr(int hdr)
{
    MMRESULT mm_r;

    DBEG("mci_prepare_hdr(%d)", hdr);

    _mix_some_samples((unsigned long)(mci_audio_buffer[hdr]), _dos_ds, TRUE);

    // Prepare with new data
    mci_wave_hdr[hdr].lpData = mci_audio_buffer[hdr];
    mci_wave_hdr[hdr].dwBufferLength = mci_buf_size; 
    mci_wave_hdr[hdr].dwBytesRecorded = 0;
    mci_wave_hdr[hdr].dwUser = 0;
    mci_wave_hdr[hdr].dwFlags = 0; 
    mci_wave_hdr[hdr].dwLoops = 0;
    mci_wave_hdr[hdr].lpNext = NULL;
    mci_wave_hdr[hdr].reserved = 0; 

    mm_r = waveOutPrepareHeader(mci_device, &(mci_wave_hdr[hdr]), sizeof(WAVEHDR));
    switch (mm_r)
    {
    case MMSYSERR_INVALHANDLE:
        D1("mci_play_buffer prepare - Specified device handle is invalid.");
        return -1;
    case MMSYSERR_NODRIVER:
        D1("mci_play_buffer prepare - No device driver is present.");
        return -1;
    case MMSYSERR_NOMEM:
        D1("mci_play_buffer prepare - Unable to allocate or lock memory.");
        return -1;
    }
}


int mci_unprepare_hdr(int hdr)
{
    DBEG("mci_unprepare_hdr(%d)", hdr);

    // Unprepare this header in the case it was used before
    waveOutUnprepareHeader(mci_device, &(mci_wave_hdr[hdr]), sizeof(WAVEHDR));
}


void mci_update_mixer()
{
    int t;

    DBEG("mci_update_mixer");

    if (mci_buffer_played)
    {
        mci_buffer_played--;

        D3("mci_update_mixer - mixing next buffer");

        // Switch to the next buffer that isn't playing    
        t = mci_playing_hdr;
        mci_playing_hdr = mci_waiting_hdr;
        mci_waiting_hdr = mci_unused_hdr;
        mci_unused_hdr = t;

        mci_play_hdr(mci_playing_hdr);
        mci_prepare_hdr(mci_waiting_hdr);
        mci_unprepare_hdr(mci_unused_hdr);
    }
}


/********************************************************/
static int mci_timer()
{
    mci_update_mixer();
}


void CALLBACK mci_hdr_done(HWAVEOUT _mci_device, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
{
    D3("mci_hdr_done");
    if (_mci_device==mci_device)
    {        
        if (uMsg==WOM_DONE)
        {
            D3("mci_hdr_done - WOM_DONE");
            mci_buffer_played++;
        }
    }
}


/********************************************************
 *  Initialize driver									*
 ********************************************************/
static int mci_init(int input, int voices)
{
	int c = 0;
    WAVEFORMATEX wave_format;
    MMRESULT mm_r;
	  
	digi_mci.voices = voices;

    mci_freq = 44100;
    mci_16bit = 1;
    mci_stereo = 1;
	
    // Open wave device    
    wave_format.wFormatTag = WAVE_FORMAT_PCM;
    wave_format.nChannels = mci_stereo?2:1;
    wave_format.nSamplesPerSec = mci_freq; 
    wave_format.wBitsPerSample = mci_16bit?16:8;
    wave_format.nBlockAlign = wave_format.nChannels * wave_format.wBitsPerSample / 8 ;
    wave_format.nAvgBytesPerSec = wave_format.nSamplesPerSec * wave_format.nBlockAlign;         
    wave_format.cbSize = 0; 


    mm_r = waveOutOpen(&mci_device, WAVE_MAPPER, &wave_format, (unsigned long)mci_hdr_done, 0, CALLBACK_FUNCTION);
    switch (mm_r)
    {
    case MMSYSERR_ALLOCATED: 
        D1("mci_init - Specified resource is already allocated."); 
        return -1;
    case MMSYSERR_BADDEVICEID: 
        D1("mci_init - Specified device identifier is out of range."); 
        return -1;
    case MMSYSERR_NODRIVER: 
        D1("mci_init - No device driver is present."); 
        return -1;
    case MMSYSERR_NOMEM: 
        D1("mci_init - Unable to allocate or lock memory."); 
        return -1;
    case WAVERR_BADFORMAT: 
        D1("mci_init - Attempted to open with an unsupported waveform-audio format.");
        return -1;
    case WAVERR_SYNC: 
        D1("mci_init - The device is synchronous but waveOutOpen was called without using the WAVE_ALLOWSYNC flag.");
        return -1;
    }
     
    
    if (mci_freq < 15000)
    {
        mci_freq = 11906;
        mci_buf_smp_num = BASE_BUF_SIZE*1;
    } else 
    if (mci_freq < 20000)
    {
        mci_freq = 16129;
        mci_buf_smp_num = BASE_BUF_SIZE*1;
    } else 
    if (mci_freq < 40000)
    {
        mci_freq = 22727;
        mci_buf_smp_num = BASE_BUF_SIZE*2;
    } else
    {
        mci_freq = 45454;
        mci_buf_smp_num = BASE_BUF_SIZE*4;
    }
    
    mci_buf_size = mci_buf_smp_num * (mci_16bit?2:1) * (mci_stereo?2:1);    
    mci_audio_buffer[0] = (char*)malloc(mci_buf_size);
    mci_audio_buffer[1] = (char*)malloc(mci_buf_size);
    mci_audio_buffer[2] = (char*)malloc(mci_buf_size);

    // Start mixer
    if (_mixer_init(mci_buf_smp_num*(mci_stereo?2:1),
            mci_freq, mci_stereo, mci_16bit, &digi_mci.voices)!=0)
      return -1;    
        
    mci_unused_hdr = 2;
    mci_playing_hdr = 1;
    mci_waiting_hdr = 0;
    mci_buffer_played = 0;
    
    mci_prepare_hdr(mci_playing_hdr);
    mci_play_hdr(mci_playing_hdr);
    mci_prepare_hdr(mci_waiting_hdr);
           
    // Install timer proc
	install_int(mci_timer, 1000 * mci_buf_smp_num / 3 / mci_freq);
	
	return 0;
}


/********************************************************
 *  Close DirectX										*
 ********************************************************/
static void mci_exit(int input)
{	
	// Remove timer proc
	remove_int(mci_timer);

    // Destroy voice    
    waveOutReset(mci_device);

    waveOutUnprepareHeader(mci_device, &mci_wave_hdr[0], sizeof(WAVEHDR));    
    waveOutUnprepareHeader(mci_device, &mci_wave_hdr[1], sizeof(WAVEHDR));    

    free(mci_audio_buffer[0]);
    free(mci_audio_buffer[1]);
    free(mci_audio_buffer[2]);

    waveOutClose(mci_device);

    _mixer_exit();

}


/********************************************************
 *  Set DirectX mixer									*
 ********************************************************/
static int mci_mixer_volume(int volume)
{	
}


/********************************************************
 *  Detect DirectSound									*
 ********************************************************/
static int mci_detect(int input)
{
	if (input)
      return FALSE;
  
	return TRUE;
}
