//////////////////////////////////////////////////////////////////////////////////
// $Source: /usr/cvsroot/cdx/src/cdx/cdxinput.cpp,v $
// $Author: pietro $
//
// $Log: cdxinput.cpp,v $
// Revision 2.2  1999/05/20 15:29:02  pietro
// Multiple changes:
// * fixed #include bugs in all .cpp and various .h files
// * fixed all rcsid[] bugs
// * added conditional compile variable CDXINCLUDEALL - when defined,
//   all #include files are included in cdx.h to keep backward compatibility
// * All the libraries are created in ..\..\lib\vc\ directory, library names are
//   cdx.lib/cdxd.lib/cdxdx3.lib/cdxdx3d.lib/cdxadx3.lib/cdxadx3d.lib
//
// Revision 2.1  1999/05/03 21:05:17  MICHAELR
// DIRECTDRAWVERSION >= CDX_DDVER was used instead of the proper DIRECTINPUT_VERSION >= CDX_DIVER
// SetJoystickAbs didn't handle DX3 using #ifdefs correctly
// RunJoystickControlPanel didn't handle DX3 using #ifdefs correctly
//
// Revision 2.0  1999/05/01 13:51:16  bsimser
// Updated revision number to 2.0
//
// Revision 1.1.1.1  1999/05/01 04:10:56  bsimser
// Initial revision to cvs
//
// $Revision: 2.2 $
//////////////////////////////////////////////////////////////////////////////////
#ifdef SAVE_RCSID
static char rcsid[] = "@(#) $Id: cdxinput.cpp,v 2.2 1999/05/20 15:29:02 pietro Exp $";
#endif

#include "CDX.h"
#include "cdxinput.h"

static LPDIRECTINPUTDEVICE lpDID = NULL;

//////////////////////////////////////////////////////////////////////////////////
// Enumerate all the joystick devices on the system.
//////////////////////////////////////////////////////////////////////////////////
BOOL FAR PASCAL EnumJoystick(LPCDIDEVICEINSTANCE pdinst, LPVOID pvRef)
{
	LPDIRECTINPUT pDI = (LPDIRECTINPUT)pvRef;

	if(pDI->CreateDevice(pdinst->guidInstance, &lpDID, NULL) != DI_OK)
		return DIENUM_CONTINUE;

	return DIENUM_STOP;
}

//////////////////////////////////////////////////////////////////////////////////
// Default constructor. Simply initializes the member elements.
//////////////////////////////////////////////////////////////////////////////////
CDXInput::CDXInput(void)
{
	m_lpDI				= NULL;
	m_lpDIDKeyboard		= NULL;
	m_lpDIDMouse		= NULL;

#if DIRECTINPUT_VERSION >= CDX_DIVER
	m_lpDIDJoystick		= NULL;
#endif

	m_bMouse			= FALSE;
	m_bKeyboard			= FALSE;
	m_bJoystick			= FALSE;

	m_mouseSensitivity	= 1.0;
	m_mouseMinX			= 0x80000000;
	m_mouseMinY			= 0x80000000; 
	m_mouseMaxX			= 0x7fffffff;
	m_mouseMaxY			= 0x7fffffff; 

	m_joystickMinX		= 0x80000000;
	m_joystickMinY		= 0x80000000; 
	m_joystickMaxX		= 0x7fffffff;
	m_joystickMaxY		= 0x7fffffff; 

	m_mouseX			= 0;
	m_mouseY			= 0;
	m_mouseFreeX		= 0;
	m_mouseFreeY		= 0;
	m_mouseDeltaX		= 0;
	m_mouseDeltaY		= 0;

	m_joystickX			= 0;
	m_joystickY			= 0;
	m_joystickFreeX		= 0;
	m_joystickFreeY		= 0;
	m_joystickDeltaX	= 0;
	m_joystickDeltaY	= 0;

	// wipe our internal key data
	memset(m_keyStates, 0, sizeof(BYTE) * CDXKEY_NUMKEYS);
	memset(m_keyRaw, 0, sizeof(BYTE) * CDXKEY_NUMKEYS);
	memset(m_keyPressTimes, 0, sizeof(DWORD) * CDXKEY_NUMKEYS);
	memset(m_keyDragStartPositions, 0, sizeof(DWORD) * CDXKEY_NUMKEYS * 2);
}

//////////////////////////////////////////////////////////////////////////////////
// Default destructor. Releases all devices that accquired
//////////////////////////////////////////////////////////////////////////////////
CDXInput::~CDXInput(void)
{
	// make sure we're initialized
	if (!m_lpDI)
		return;
	
	// shutdown keyboard
	if (m_lpDIDKeyboard)
	{
		UnacquireKeyboard();
		m_lpDIDKeyboard->Release();
		m_lpDIDKeyboard = NULL;
	}
	
	// shutdown mouse
	if (m_lpDIDMouse)
	{
		UnacquireMouse();
		m_lpDIDMouse->Release();
		m_lpDIDMouse = NULL;
	}
	
	// shutdown joystick
#if DIRECTINPUT_VERSION >= CDX_DIVER
	if (m_lpDIDJoystick)
	{
		UnacquireJoystick();
		m_lpDIDJoystick->Release();
		m_lpDIDJoystick = NULL;
	}
#endif

	// kill directinput
	m_lpDI->Release();
	m_lpDI = NULL;
}

//////////////////////////////////////////////////////////////////////////////////
// Try to accquire all devices. Use SetActiveDevices() if you do not want 
// some devices.
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXInput::Create(void *hInst, void *hWnd)
{
	HRESULT rval;

	rval = DirectInputCreate((HINSTANCE)hInst, DIRECTINPUT_VERSION, &m_lpDI, NULL);
	if(rval != DI_OK) return FALSE;

	// Create the mouse device
	rval = m_lpDI->CreateDevice(GUID_SysMouse, &m_lpDIDMouse, NULL);
	if(rval == DI_OK)
	{
		m_lpDIDMouse->SetDataFormat(&c_dfDIMouse);
		m_lpDIDMouse->SetCooperativeLevel((HWND)hWnd, DISCL_EXCLUSIVE | DISCL_FOREGROUND);

		AcquireMouse();
		if(!m_bMouse) 
			return FALSE;
	}

	// Create the keyboard device
	rval = m_lpDI->CreateDevice(GUID_SysKeyboard, &m_lpDIDKeyboard, NULL);
	if(rval == DI_OK)
	{
		m_lpDIDKeyboard->SetDataFormat(&c_dfDIKeyboard);
		m_lpDIDKeyboard->SetCooperativeLevel((HWND)hWnd, DISCL_NONEXCLUSIVE | DISCL_FOREGROUND);

		AcquireKeyboard();
		if(!m_bKeyboard) 
			return FALSE;

		// set up DIK_ remap
		InitDIKToCDXKEY();
		InitShiftedKeys();
	}

	// Enumerate the joystick device (DirectX 5 or higher)
#if DIRECTINPUT_VERSION >= CDX_DIVER
	rval = m_lpDI->EnumDevices(DIDEVTYPE_JOYSTICK, EnumJoystick, m_lpDI, DIEDFL_ATTACHEDONLY);
	if(lpDID != NULL)
	{
		rval = lpDID->QueryInterface(IID_IDirectInputDevice2, (LPVOID *)&m_lpDIDJoystick);
		if(rval != DI_OK) return FALSE;

		RELEASE(lpDID);

		m_lpDIDJoystick->SetDataFormat(&c_dfDIJoystick);
		m_lpDIDJoystick->SetCooperativeLevel((HWND)hWnd, DISCL_NONEXCLUSIVE | DISCL_FOREGROUND);

		// Set the X-axis range (-1000 to +1000)
		DIPROPRANGE diprg;
		diprg.diph.dwSize = sizeof(diprg);
		diprg.diph.dwHeaderSize = sizeof(diprg.diph);
		diprg.diph.dwObj = DIJOFS_X;
		diprg.diph.dwHow = DIPH_BYOFFSET;
		diprg.lMin = -1000;
		diprg.lMax = +1000;

		if(m_lpDIDJoystick->SetProperty(DIPROP_RANGE, &diprg.diph) != DI_OK)
			return FALSE;

		// And again for Y-axis range
		diprg.diph.dwObj = DIJOFS_Y;

		if(m_lpDIDJoystick->SetProperty(DIPROP_RANGE, &diprg.diph) != DI_OK)
			return FALSE;

		// Set X axis dead zone to 10%
		DIPROPDWORD dipdw;
		dipdw.diph.dwSize = sizeof(dipdw);
		dipdw.diph.dwHeaderSize = sizeof(dipdw.diph);
		dipdw.diph.dwObj = DIJOFS_X;
		dipdw.diph.dwHow = DIPH_BYOFFSET;
		dipdw.dwData = 1000;

		if(m_lpDIDJoystick->SetProperty(DIPROP_DEADZONE, &dipdw.diph) != DI_OK)
			return FALSE;

		dipdw.diph.dwObj = DIJOFS_Y;

		// Set Y axis dead zone to 10%
		if(m_lpDIDJoystick->SetProperty(DIPROP_DEADZONE, &dipdw.diph) != DI_OK)
			return FALSE;

		AcquireJoystick();
		if(!m_bJoystick) 
			return FALSE;
	}
#endif

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Controls which devices you have accquired.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetActiveDevices(BOOL bMouse, BOOL bKeyboard, BOOL bJoystick)
{
	if (m_bMouse = bMouse)
		AcquireMouse();
	else
		UnacquireMouse();

	if (m_bKeyboard = bKeyboard)
		AcquireKeyboard();
	else 
		UnacquireKeyboard();

#if DIRECTINPUT_VERSION >= CDX_DIVER
	if (m_bJoystick = bJoystick)
		AcquireJoystick();
	else
		UnacquireJoystick();
#endif
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Sets the axis mode of the mouse.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetMouseAbs(void)
{
	DIPROPDWORD dipdw;

	dipdw.diph.dwSize = sizeof(DIPROPDWORD);
	dipdw.diph.dwHeaderSize = sizeof(DIPROPHEADER);
	dipdw.diph.dwObj = 0;
	dipdw.diph.dwHow = DIPH_DEVICE;
	dipdw.dwData = DIPROPAXISMODE_ABS;
	m_lpDIDMouse->SetProperty(DIPROP_AXISMODE, &dipdw.diph);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Sets the axis mode of the joystick.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetJoystickAbs(void)
{
#if DIRECTINPUT_VERSION >= CDX_DIVER
	DIPROPDWORD dipdw;

	dipdw.diph.dwSize = sizeof(DIPROPDWORD);
	dipdw.diph.dwHeaderSize = sizeof(DIPROPHEADER);
	dipdw.diph.dwObj = 0;
	dipdw.diph.dwHow = DIPH_DEVICE;
	dipdw.dwData = DIPROPAXISMODE_ABS;
	m_lpDIDJoystick->SetProperty(DIPROP_AXISMODE, &dipdw.diph);
#endif
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Runs the DirectInput control panel associated with the mouse. If the device does not 
// have a control panel associated with it, the default device control panel is launched.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::RunMouseControlPanel(void *hWnd)
{
	m_lpDIDMouse->RunControlPanel((HWND)hWnd, 0);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Runs the DirectInput control panel associated with the joystick. If the device does not 
// have a control panel associated with it, the default device control panel is launched.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::RunJoystickControlPanel(void *hWnd)
{
#if DIRECTINPUT_VERSION >= CDX_DIVER
	m_lpDIDJoystick->RunControlPanel((HWND)hWnd, 0);
#endif
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Wipes out the internal key data.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::FlushKeyboardData()
{
	memset(m_keyStates, 0, sizeof(BYTE) * CDXKEY_NUMKEYS);
	memset(m_keyRaw, 0, sizeof(BYTE) * CDXKEY_NUMKEYS);
	memset(m_keyPressTimes, 0, sizeof(DWORD) * CDXKEY_NUMKEYS);
	memset(m_keyDragStartPositions, 0, sizeof(DWORD) * CDXKEY_NUMKEYS * 2);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// This restricts the mouse to a defined area.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetMouseLimits(LONG x1, LONG y1, LONG x2, LONG y2)
{
	m_mouseMinX = x1;
	m_mouseMinY = y1;
	m_mouseMaxX = x2;
	m_mouseMaxY = y2;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// This restricts the joystick to a defined area.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetJoystickLimits(LONG x1, LONG y1, LONG x2, LONG y2)
{
	m_joystickMinX = x1;
	m_joystickMinY = y1;
	m_joystickMaxX = x2;
	m_joystickMaxY = y2;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Sets the mouse position. This restricts the position to the physical display.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetMousePos(LONG x, LONG y)
{
	// clamp non-free mouse values to limits
	if ((m_mouseX = x) >= m_mouseMaxX)
		m_mouseX = m_mouseMaxX-1;

	if ((m_mouseY = y) >= m_mouseMaxY)
		m_mouseY = m_mouseMaxY-1;

	if ((m_mouseX = x) <= m_mouseMinX)
		m_mouseX = m_mouseMinX+1;

	if ((m_mouseY = y) <= m_mouseMinY)
		m_mouseY = m_mouseMinY+1;

	// Zero out the Deltas
	m_mouseDeltaX = 0;
	m_mouseDeltaY = 0;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Sets the mouse position. This allows you to set the mouse to anywhere (no limits).
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetMouseFreePos(LONG x, LONG y)
{
	m_mouseFreeX  = x;
	m_mouseFreeY  = y;

	// Zero out the Deltas
	m_mouseDeltaX = 0;
	m_mouseDeltaY = 0;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Sets the joystick position. This restricts the position to the physical display.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetJoystickPos(LONG x, LONG y)
{
	// clamp non-free joystick values to limits
	if ((m_joystickX = x) >= m_joystickMaxX)
		m_joystickX = m_joystickMaxX-1;

	if ((m_joystickY = y) >= m_joystickMaxY)
		m_joystickY = m_joystickMaxY-1;

	if ((m_joystickX = x) <= m_joystickMinX)
		m_joystickX = m_joystickMinX+1;

	if ((m_joystickY = y) <= m_joystickMinY)
		m_joystickY = m_joystickMinY+1;

	// Zero out the Deltas
	m_joystickDeltaX = 0;
	m_joystickDeltaY = 0;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Sets the joystick position. This allows you to set the joystick to anywhere (no limits).
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetJoystickFreePos(LONG x, LONG y)
{
	m_joystickFreeX  = x;
	m_joystickFreeY  = y;

	// Zero out the Deltas
	m_joystickDeltaX = 0;
	m_joystickDeltaY = 0;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Sets the mouse sensitivity factor.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::SetMouseSensitivity(float factor)
{
	m_mouseSensitivity = factor;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Retrieves the current mouse sensitivity factor.
/////////////////////////////////////////////////////////////////////////////////////////////
float CDXInput::GetMouseSensitivity()
{
	return(m_mouseSensitivity);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// This tells you whether or not the given key is in a shifted state.
/////////////////////////////////////////////////////////////////////////////////////////////
BYTE CDXInput::GetShiftedKeyState(BYTE key)
{
	if (m_shiftedKeyStates[key])
		return (m_shiftedKeyStates[key]);

	return(key);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Retrieves the keystate of a given key.
/////////////////////////////////////////////////////////////////////////////////////////////
BYTE CDXInput::GetKeyState(BYTE key)
{
	return(m_keyStates[key]);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Gets raw information about a key.
/////////////////////////////////////////////////////////////////////////////////////////////
BYTE CDXInput::GetKeyRaw(BYTE key)
{
	return(m_keyRaw[key]);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Retrieves the mouse position and stores the x and y values in the passed parameters.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::GetMousePos(LONG* x, LONG* y)
{
	*x = m_mouseX;
	*y = m_mouseY;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Retrieves the free mouse position and stores the x and y values in the passed parameters.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::GetMouseFreePos(LONG* x, LONG* y)
{
	*x = m_mouseFreeX;
	*y = m_mouseFreeY;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Retrieves the mouse deltas and stores the x and y values in the passed parameters.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::GetMouseDeltas(LONG* x, LONG* y)
{
	*x = m_mouseDeltaX;
	*y = m_mouseDeltaY;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Retrieves the joystick position and stores the x and y values in the passed parameters.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::GetJoystickPos(LONG* x, LONG* y)
{
	*x = m_joystickX;
	*y = m_joystickY;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Retrieves the free joystick position and stores the x and y values in the passed parameters.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::GetJoystickFreePos(LONG* x, LONG* y)
{
	*x = m_joystickFreeX;
	*y = m_joystickFreeY;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Retrieves the joystick deltas and stores the x and y values in the passed parameters.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::GetJoystickDeltas(LONG* x, LONG* y)
{
	*x = m_joystickDeltaX;
	*y = m_joystickDeltaY;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Obtains access to the mouse.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::AcquireMouse()
{
	if (m_lpDIDMouse)
	{
		HRESULT err;
		int i;

		m_bMouse = TRUE;
		if ((err = m_lpDIDMouse->Acquire()) == DI_OK)
			return;
		// didn't get it back, try a few more times
		for (i=0;i<50;i++)
		{
			if ((err = m_lpDIDMouse->Acquire()) == DI_OK)
				return;
		}
		// still don't have it, something's up
		m_bMouse = FALSE;
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Releases access to the mouse.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::UnacquireMouse()
{
	m_bMouse = FALSE;
	if (m_lpDIDMouse)
		m_lpDIDMouse->Unacquire();
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Obtains access to the joystick.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::AcquireJoystick()
{
#if DIRECTINPUT_VERSION >= CDX_DIVER
	if (m_lpDIDJoystick)
	{
		HRESULT err;
		int i;

		m_bJoystick = TRUE;
		if ((err = m_lpDIDJoystick->Acquire()) == DI_OK)
			return;
		// didn't get it back, try a few more times
		for (i=0;i<50;i++)
		{
			if ((err = m_lpDIDJoystick->Acquire()) == DI_OK)
				return;
		}
		// still don't have it, something's up
		m_bJoystick = FALSE;
	}
#endif
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Releases access to the joystick.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::UnacquireJoystick()
{
#if DIRECTINPUT_VERSION >= CDX_DIVER
	m_bJoystick = 0;
	if (m_lpDIDJoystick)
		m_lpDIDJoystick->Unacquire();
#endif
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Obtains access to the keyboard.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::AcquireKeyboard()
{
	if (m_lpDIDKeyboard)
	{
		HRESULT err;
		int i;

		m_bKeyboard = TRUE;
		if ((err = m_lpDIDKeyboard->Acquire()) == DI_OK)
		{
			FlushKeyboardData();
			return;
		}
		// didn't get it back, try a few more times
		for (i=0;i<50;i++)
		{
			if ((err = m_lpDIDKeyboard->Acquire()) == DI_OK)
			{
				FlushKeyboardData();
				return;
			}
		}
		// still don't have it, something's up
		m_bKeyboard = FALSE;
	}	
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Releases access to the keyboard.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::UnacquireKeyboard()
{
	FlushKeyboardData();

	m_bKeyboard = FALSE;
	if (m_lpDIDKeyboard)
		m_lpDIDKeyboard->Unacquire();
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Updates all devices. Call this before you check for input.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::Update(void)
{
	DIMOUSESTATE MouseState;
	HRESULT err;
	LONG oldx, oldy;
	BOOL isPressed;
	WORD i;
	DWORD key;

#if DIRECTINPUT_VERSION >= CDX_DIVER
	DIJOYSTATE JoyState;
#endif

	///////////////////////
	//  KEYBOARD
	///////////////////////
	if(1)
	{
		err = m_lpDIDKeyboard->GetDeviceState(256, &m_keyRaw);
		if(err != DI_OK)
		{
			AcquireKeyboard();
			return;
		}
		else
		{
			for(i=0; i<NUMDIKEYS; i++)
			{
				if (key = m_DIKToCDXKEY[i])
				{
					// check current key state (bit 8 flipped on means key is pressed)
					if (m_keyRaw[i] & (BYTE)0x80)
						isPressed = TRUE;
					else
						isPressed = FALSE;

					CheckKeyEvents(key, isPressed);
				}
			}
		}
	}

	///////////////////////
	//  MOUSE
	///////////////////////
	if(m_bMouse)
	{
		err = m_lpDIDMouse->GetDeviceState(sizeof(MouseState), &MouseState);
		if(err != DI_OK)
		{
			AcquireMouse();
			return;
		}
		else
		{
			// get new mouse position status
			oldx = m_mouseFreeX;
			oldy = m_mouseFreeY;
			m_mouseFreeX += MouseState.lX * (long)m_mouseSensitivity;
			m_mouseFreeY += MouseState.lY * (long)m_mouseSensitivity;
			m_mouseX += MouseState.lX * (long)m_mouseSensitivity;
			m_mouseY += MouseState.lY * (long)m_mouseSensitivity;

			// clamp non-free mouse values to limits
			if (m_mouseX >= m_mouseMaxX)
				m_mouseX = m_mouseMaxX-1;

			if (m_mouseY >= m_mouseMaxY)
				m_mouseY = m_mouseMaxY-1;

			if (m_mouseX <= m_mouseMinX)
				m_mouseX = m_mouseMinX;

			if (m_mouseY <= m_mouseMinY)
				m_mouseY = m_mouseMinY;

			m_mouseDeltaX = m_mouseFreeX - oldx;
			m_mouseDeltaY = m_mouseFreeY - oldy;
	
			// check the mouse buttons
			for(i=0; i<3; i++)
			{
				key = CDXKEY_MOUSELEFT + i;

				// check current key state (bit 8 flipped on means key is pressed)
				if (MouseState.rgbButtons[i] & (BYTE)0x80)
					isPressed = TRUE;
				else
					isPressed = FALSE;

				CheckKeyEvents(key, isPressed);
			}
		}
	}

	///////////////////////
	//  JOYSTICK
	///////////////////////
#if DIRECTINPUT_VERSION >= CDX_DIVER
	if(m_bJoystick)
	{
		m_lpDIDJoystick->Poll();

		err = m_lpDIDJoystick->GetDeviceState(sizeof(JoyState), &JoyState);

		if(err != DI_OK)
		{
			m_lpDIDJoystick->Acquire();
			return;
		}
		else
		{
			// get new joystick position status
			oldx			= m_joystickFreeX;
			oldy			= m_joystickFreeY;
			m_joystickFreeX += JoyState.lX * (long)m_mouseSensitivity;
			m_joystickFreeY += JoyState.lY * (long)m_mouseSensitivity;
			m_joystickX		+= JoyState.lX * (long)m_mouseSensitivity;
			m_joystickY		+= JoyState.lY * (long)m_mouseSensitivity;

			// clamp non-free joystick values to limits
			if (m_joystickX >= m_joystickMaxX)
				m_joystickX = m_joystickMaxX-1;

			if (m_joystickY >= m_joystickMaxY)
				m_joystickY = m_joystickMaxY-1;

			if (m_joystickX <= m_joystickMinX)
				m_joystickX = m_joystickMinX;

			if (m_joystickY <= m_joystickMinY)
				m_joystickY = m_joystickMinY;

			m_joystickDeltaX = m_joystickFreeX - oldx;
			m_joystickDeltaY = m_joystickFreeY - oldy;
	
			// check the joystick buttons
			for(i=0; i<10; i++)
			{
				key = CDXKEY_JOYBUTN0 + i;

				// check current key state (bit 8 flipped on means key is pressed)
				if (JoyState.rgbButtons[i] & (BYTE)0x80)
					isPressed = TRUE;
				else
					isPressed = FALSE;

				CheckKeyEvents(key, isPressed);
			}
		}
	}
#endif
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Checks for keyboard events.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::CheckKeyEvents(DWORD key, BOOL isPressed)
{
	if (isPressed)
	{
		if (m_keyStates[key] == CDXKEY_NONE || m_keyStates[key] == CDXKEY_RELEASE)
		{
			// press event (key was up before but down now)
			m_keyStates[key]				= CDXKEY_PRESS;
			m_keyPressTimes[key]			= timeGetTime();
			m_keyDragStartPositions[key][0] = m_mouseFreeX;
			m_keyDragStartPositions[key][1] = m_mouseFreeY;
		}
		else if (m_keyStates[key] == CDXKEY_PRESS)
		{
			// drag event (key is still down)
			m_keyStates[key] = CDXKEY_REPEAT;
		}
	}
	else // Key is not currently pressed
	{
		if (m_keyStates[key] == CDXKEY_REPEAT || m_keyStates[key] == CDXKEY_PRESS)
		{
			// release event (key was down before but up now)
			m_keyStates[key] = CDXKEY_RELEASE;
		}
		else if (m_keyStates[key] == CDXKEY_RELEASE)
		{
			m_keyStates[key] = CDXKEY_NONE;
		}
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Initializes the shifted key states.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::InitShiftedKeys()
{
	WORD i;

	memset(m_shiftedKeyStates, 0, CDXKEY_NUMKEYS * sizeof(BYTE));

	for (i=0;i<26;i++)
		m_shiftedKeyStates[i+'a'] = i+'A';

	m_shiftedKeyStates['0']  = ')';
	m_shiftedKeyStates['1']  = '!';
	m_shiftedKeyStates['2']  = '@';
	m_shiftedKeyStates['3']  = '#';
	m_shiftedKeyStates['4']  = '$';
	m_shiftedKeyStates['5']  = '%';
	m_shiftedKeyStates['6']  = '^';
	m_shiftedKeyStates['7']  = '&';
	m_shiftedKeyStates['8']  = '*';
	m_shiftedKeyStates['9']  = '(';
	m_shiftedKeyStates['-']  = '_';
	m_shiftedKeyStates['=']  = '+';
	m_shiftedKeyStates['[']  = '{';
	m_shiftedKeyStates[']']  = '}';
	m_shiftedKeyStates['\\'] = '|';
	m_shiftedKeyStates[';']  = ':';
	m_shiftedKeyStates['\''] = '"';
	m_shiftedKeyStates[',']  = '<';
	m_shiftedKeyStates['.']  = '>';
	m_shiftedKeyStates['/']  = '?';
	m_shiftedKeyStates['`']  = '~';
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Since we can't guarantee that the constants for these DIKs won't
// change, initialize the remap table here instead of statically doing it.
/////////////////////////////////////////////////////////////////////////////////////////////
void CDXInput::InitDIKToCDXKEY()
{
	// any key mapped to zero means unmapped and should be ignored
	memset(m_DIKToCDXKEY, 0, NUMDIKEYS * sizeof(DWORD));

	m_DIKToCDXKEY[DIK_ESCAPE]		= CDXKEY_ESCAPE;
	m_DIKToCDXKEY[DIK_1]			= '1';
	m_DIKToCDXKEY[DIK_2]			= '2';
	m_DIKToCDXKEY[DIK_3]			= '3';
	m_DIKToCDXKEY[DIK_4]			= '4';
	m_DIKToCDXKEY[DIK_5]			= '5';
	m_DIKToCDXKEY[DIK_6]			= '6';
	m_DIKToCDXKEY[DIK_7]			= '7';
	m_DIKToCDXKEY[DIK_8]			= '8';
	m_DIKToCDXKEY[DIK_9]			= '9';
	m_DIKToCDXKEY[DIK_0]			= '0';
	m_DIKToCDXKEY[DIK_MINUS]		= '-';
	m_DIKToCDXKEY[DIK_EQUALS]		= '=';
	m_DIKToCDXKEY[DIK_BACK]			= CDXKEY_BACKSPACE;
	m_DIKToCDXKEY[DIK_TAB]			= CDXKEY_TAB;
	m_DIKToCDXKEY[DIK_Q]			= 'q';
	m_DIKToCDXKEY[DIK_W]			= 'w';
	m_DIKToCDXKEY[DIK_E]			= 'e';
	m_DIKToCDXKEY[DIK_R]			= 'r';
	m_DIKToCDXKEY[DIK_T]			= 't';
	m_DIKToCDXKEY[DIK_Y]			= 'y';
	m_DIKToCDXKEY[DIK_U]			= 'u';
	m_DIKToCDXKEY[DIK_I]			= 'i';
	m_DIKToCDXKEY[DIK_O]			= 'o';
	m_DIKToCDXKEY[DIK_P]			= 'p';
	m_DIKToCDXKEY[DIK_LBRACKET]		= '[';
	m_DIKToCDXKEY[DIK_RBRACKET]		= ']';
	m_DIKToCDXKEY[DIK_RETURN]		= CDXKEY_ENTER;
	m_DIKToCDXKEY[DIK_LCONTROL]		= CDXKEY_LEFTCTRL;
	m_DIKToCDXKEY[DIK_A]			= 'a';
	m_DIKToCDXKEY[DIK_S]			= 's';
	m_DIKToCDXKEY[DIK_D]			= 'd';
	m_DIKToCDXKEY[DIK_F]			= 'f';
	m_DIKToCDXKEY[DIK_G]			= 'g';
	m_DIKToCDXKEY[DIK_H]			= 'h';
	m_DIKToCDXKEY[DIK_J]			= 'j';
	m_DIKToCDXKEY[DIK_K]			= 'k';
	m_DIKToCDXKEY[DIK_L]			= 'l';
	m_DIKToCDXKEY[DIK_SEMICOLON]	= ';';
	m_DIKToCDXKEY[DIK_APOSTROPHE]	= '\'';
	m_DIKToCDXKEY[DIK_GRAVE]		= '`';
	m_DIKToCDXKEY[DIK_LSHIFT]		= CDXKEY_LEFTSHIFT;
	m_DIKToCDXKEY[DIK_BACKSLASH]	= '\\';
	m_DIKToCDXKEY[DIK_Z]			= 'z';
	m_DIKToCDXKEY[DIK_X]			= 'x';
	m_DIKToCDXKEY[DIK_C]			= 'c';
	m_DIKToCDXKEY[DIK_V]			= 'v';
	m_DIKToCDXKEY[DIK_B]			= 'b';
	m_DIKToCDXKEY[DIK_N]			= 'n';
	m_DIKToCDXKEY[DIK_M]			= 'm';
	m_DIKToCDXKEY[DIK_COMMA]		= ',';
	m_DIKToCDXKEY[DIK_PERIOD]		= '.';
	m_DIKToCDXKEY[DIK_SLASH]		= '/';
	m_DIKToCDXKEY[DIK_RSHIFT]		= CDXKEY_RIGHTSHIFT;
	m_DIKToCDXKEY[DIK_MULTIPLY]		= CDXKEY_NUMSTAR;
	m_DIKToCDXKEY[DIK_LMENU]		= CDXKEY_LEFTALT;
	m_DIKToCDXKEY[DIK_SPACE]		= CDXKEY_SPACE;
	m_DIKToCDXKEY[DIK_CAPITAL]		= CDXKEY_CAPSLOCK;
	m_DIKToCDXKEY[DIK_F1]			= CDXKEY_F1;
	m_DIKToCDXKEY[DIK_F2]			= CDXKEY_F2;
	m_DIKToCDXKEY[DIK_F3]			= CDXKEY_F3;
	m_DIKToCDXKEY[DIK_F4]			= CDXKEY_F4;
	m_DIKToCDXKEY[DIK_F5]			= CDXKEY_F5;
	m_DIKToCDXKEY[DIK_F6]			= CDXKEY_F6;
	m_DIKToCDXKEY[DIK_F7]			= CDXKEY_F7;
	m_DIKToCDXKEY[DIK_F8]			= CDXKEY_F8;
	m_DIKToCDXKEY[DIK_F9]			= CDXKEY_F9;
	m_DIKToCDXKEY[DIK_F10]			= CDXKEY_F10;
	m_DIKToCDXKEY[DIK_NUMLOCK]		= CDXKEY_NUMLOCK;
	m_DIKToCDXKEY[DIK_SCROLL]		= CDXKEY_SCROLLLOCK;
	m_DIKToCDXKEY[DIK_NUMPAD7]		= CDXKEY_NUM7;
	m_DIKToCDXKEY[DIK_NUMPAD8]		= CDXKEY_NUM8;
	m_DIKToCDXKEY[DIK_NUMPAD9]		= CDXKEY_NUM9;
	m_DIKToCDXKEY[DIK_SUBTRACT]		= CDXKEY_NUMMINUS;
	m_DIKToCDXKEY[DIK_NUMPAD4]		= CDXKEY_NUM4;
	m_DIKToCDXKEY[DIK_NUMPAD5]		= CDXKEY_NUM5;
	m_DIKToCDXKEY[DIK_NUMPAD6]		= CDXKEY_NUM6;
	m_DIKToCDXKEY[DIK_ADD]			= CDXKEY_NUMPLUS;
	m_DIKToCDXKEY[DIK_NUMPAD1]		= CDXKEY_NUM1;
	m_DIKToCDXKEY[DIK_NUMPAD2]		= CDXKEY_NUM2;
	m_DIKToCDXKEY[DIK_NUMPAD3]		= CDXKEY_NUM3;
	m_DIKToCDXKEY[DIK_NUMPAD0]		= CDXKEY_NUM0;
	m_DIKToCDXKEY[DIK_DECIMAL]		= CDXKEY_NUMPERIOD;
	m_DIKToCDXKEY[DIK_F11]			= CDXKEY_F11;
	m_DIKToCDXKEY[DIK_F12]			= CDXKEY_F12;
	m_DIKToCDXKEY[DIK_NUMPADENTER] = CDXKEY_NUMENTER;
	m_DIKToCDXKEY[DIK_RCONTROL]		= CDXKEY_RIGHTCTRL;
	m_DIKToCDXKEY[DIK_DIVIDE]		= CDXKEY_NUMSLASH;
	m_DIKToCDXKEY[DIK_SYSRQ]		= CDXKEY_PRINTSCRN;
	m_DIKToCDXKEY[DIK_RMENU]		= CDXKEY_RIGHTALT;
	m_DIKToCDXKEY[DIK_HOME]			= CDXKEY_HOME;
	m_DIKToCDXKEY[DIK_UP]			= CDXKEY_UPARROW;
	m_DIKToCDXKEY[DIK_PRIOR]		= CDXKEY_PGUP;
	m_DIKToCDXKEY[DIK_LEFT]			= CDXKEY_LEFTARROW;
	m_DIKToCDXKEY[DIK_RIGHT]		= CDXKEY_RIGHTARROW;
	m_DIKToCDXKEY[DIK_END]			= CDXKEY_END;
	m_DIKToCDXKEY[DIK_DOWN]			= CDXKEY_DOWNARROW;
	m_DIKToCDXKEY[DIK_NEXT]			= CDXKEY_PGDN;
	m_DIKToCDXKEY[DIK_INSERT]		= CDXKEY_INS;
	m_DIKToCDXKEY[DIK_DELETE]		= CDXKEY_DEL;
	m_DIKToCDXKEY[DIK_LWIN]			= 0;
	m_DIKToCDXKEY[DIK_RWIN]			= 0;
	m_DIKToCDXKEY[DIK_APPS]			= 0;
}
