//////////////////////////////////////////////////////////////////////////////////
// $Source: /usr/cvsroot/cdx/src/cdx/cdxmap.cpp,v $
// $Author: pietro $
//
// $Log: cdxmap.cpp,v $
// Revision 2.1  1999/05/20 15:29:02  pietro
// Multiple changes:
// * fixed #include bugs in all .cpp and various .h files
// * fixed all rcsid[] bugs
// * added conditional compile variable CDXINCLUDEALL - when defined,
//   all #include files are included in cdx.h to keep backward compatibility
// * All the libraries are created in ..\..\lib\vc\ directory, library names are
//   cdx.lib/cdxd.lib/cdxdx3.lib/cdxdx3d.lib/cdxadx3.lib/cdxadx3d.lib
//
// Revision 2.0  1999/05/01 13:51:16  bsimser
// Updated revision number to 2.0
//
// Revision 1.1.1.1  1999/05/01 04:10:56  bsimser
// Initial revision to cvs
//
// $Revision: 2.1 $
//////////////////////////////////////////////////////////////////////////////////
#ifdef SAVE_RCSID
static char rcsid[] = "@(#) $Id: cdxmap.cpp,v 2.1 1999/05/20 15:29:02 pietro Exp $";
#endif

#include "CDX.h"
#include "cdxmap.h"

//////////////////////////////////////////////////////////////////////////////////
// CDXMap Default Constructor
//////////////////////////////////////////////////////////////////////////////////
CDXMap::CDXMap()
{
}

//////////////////////////////////////////////////////////////////////////////////
// CDXMap Constructor
//////////////////////////////////////////////////////////////////////////////////
CDXMap::CDXMap(CDXTile *pTiles, CDXScreen *pScreen)
{
	DATA = NULL;
	Tiles = pTiles;

	m_TileWidth = Tiles->m_BlockWidth;
	m_TileHeight = Tiles->m_BlockHeight;

	m_PosX = 0;
	m_PosY = 0;

	SCREEN_W = pScreen->m_dwPixelWidth;
	SCREEN_H = pScreen->m_dwPixelHeight;

	SCREEN_TW = SCREEN_W / m_TileWidth;
	SCREEN_TH = SCREEN_H / m_TileHeight;
}

//////////////////////////////////////////////////////////////////////////////////
// Default destructor. Destroys the map and frees the memory.
//////////////////////////////////////////////////////////////////////////////////
CDXMap::~CDXMap()
{
	delete [] DATA;
}

//////////////////////////////////////////////////////////////////////////////////
// Creates a new map. Fills in the map with the specified Fill value.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::Create(int Width, int Height, int Fill)
{
	m_Width = Width;
	m_Height = Height;

	m_PixelWidth = m_Width * m_TileWidth;
	m_PixelHeight = m_Height * m_TileHeight;

	SIZE = m_Width * m_Height;
	InitData(SIZE, Fill);
}

//////////////////////////////////////////////////////////////////////////////////
// Initializes the array of data with the specified fill value.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::InitData(int nSIZE, int iFill)
{
	DATA = new CDXMapCell[nSIZE];

	for(int i=0; i<SIZE; i++)
		DATA[i].SetTile(iFill);
}

//////////////////////////////////////////////////////////////////////////////////
// Loads a map from a file.
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXMap::Load(const char *szFilename)
{
	if(szFilename == NULL) return FALSE;

	FILE *fp;

	fp = fopen(szFilename, "rb");
	if(fp == NULL) return FALSE;

	fread(&m_Width, sizeof(int), 1, fp);
	fread(&m_Height, sizeof(int), 1, fp);
	
	SIZE = m_Width * m_Height;
	InitData(SIZE, 0);

	for(int x=0; x<SIZE; x++) {
		DATA[x].Load(fp);
	}

	fclose(fp);

	m_PixelWidth = m_Width * m_TileWidth;
	m_PixelHeight = m_Height * m_TileHeight;

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// Saves a map to a file.
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXMap::Save(const char *szFilename)
{
	FILE *fp;

	fp = fopen(szFilename, "wb");
	if(fp == NULL) return FALSE;

	fwrite(&m_Width, sizeof(int), 1, fp);
	fwrite(&m_Height, sizeof(int), 1, fp);

	SIZE = m_Width * m_Height;

	for(int x=0; x<SIZE; x++) {
		DATA[x].Save(fp);
	}

	fclose(fp);

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// Clears out the map and sets all the map tiles to 0.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::Clear()
{
	for(int i=0; i<SIZE; i++)
		DATA[i].SetTile(0);
}

//////////////////////////////////////////////////////////////////////////////////
// Fills the map with a tile specified by TileNum.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::Fill(int TileNum)
{
  for(int i=0; i<SIZE; i++)
		DATA[i].SetTile(TileNum);
}

//////////////////////////////////////////////////////////////////////////////////
// Set the position in the map. All values are in pixels.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::MoveTo(int PosX, int PosY)
{
	m_PosX = PosX;
	m_PosY = PosY;
}

//////////////////////////////////////////////////////////////////////////////////
// Scroll the map upwards by the offset passed in.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::ScrollUp(int Offset)
{
	m_PosY -= Offset;
	if(m_PosY < 0) m_PosY = 0;
}

//////////////////////////////////////////////////////////////////////////////////
// Scroll the map down by the offset passed in.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::ScrollDown(int Offset)
{
	m_PosY += Offset;
	if(m_PosY > (m_Height - SCREEN_TH) * m_TileHeight)
		m_PosY = (m_Height - SCREEN_TH) * m_TileHeight;
}

//////////////////////////////////////////////////////////////////////////////////
// Scroll the map left by the offset passed in.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::ScrollLeft(int Offset)
{
	m_PosX -= Offset;
	if(m_PosX < 0) m_PosX = 0;
}

//////////////////////////////////////////////////////////////////////////////////
// Scroll the map right by the offset passed in.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::ScrollRight(int Offset)
{
	m_PosX += Offset;
	if(m_PosX > (m_Width - SCREEN_TW) * m_TileWidth)
		m_PosX = (m_Width - SCREEN_TW) * m_TileWidth;
}

//////////////////////////////////////////////////////////////////////////////////
// Scroll the map up by the offset passed in. This wraps around the map.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::WrapScrollUp(int Offset)
{
	if(m_PosY - Offset >= 0) m_PosY -= Offset;
	else m_PosY = (m_Height * m_TileHeight) - Offset + m_PosY;
}

//////////////////////////////////////////////////////////////////////////////////
// Scroll the map down by the offset passed in. This wraps around the map.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::WrapScrollDown(int Offset)
{
	if (m_PosY + Offset < (m_Height * m_TileHeight)) m_PosY += Offset;
	else m_PosY = Offset - (m_Height * m_TileHeight) + m_PosY;
}

//////////////////////////////////////////////////////////////////////////////////
// Scroll the map left by the offset passed in. This wraps around the map.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::WrapScrollLeft(int Offset)
{
	if (m_PosX - Offset >= 0) m_PosX -= Offset;
	else m_PosX = (m_Width * m_TileWidth) - Offset + m_PosX;
}

//////////////////////////////////////////////////////////////////////////////////
// Scroll the map right by the offset passed in. This wraps around the map.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::WrapScrollRight(int Offset)
{
	if (m_PosX + Offset < (m_Width * m_TileWidth)) m_PosX += Offset;
	else m_PosX = Offset - (m_Width * m_TileWidth) + m_PosX;
}

//////////////////////////////////////////////////////////////////////////////////
// Sets the screen tile width and screen tile height. This is the number of 
// tiles drawn vertically and horizontally on screen.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::ScreenTileSize(int Width, int Height)
{
	SCREEN_TW = Width;
	SCREEN_TH = Height;
}

//////////////////////////////////////////////////////////////////////////////////
// This gets the CDXMapCell object from the map at the given MapX and MapY coordinates.
//////////////////////////////////////////////////////////////////////////////////
CDXMapCell *CDXMap::GetCell(int MapX, int MapY)
{
	return &DATA[MapX + (MapY * m_Width)];
}

//////////////////////////////////////////////////////////////////////////////////
// Returns the value of the tile at MapX, MapY. Could be used for tile collision detection.
//////////////////////////////////////////////////////////////////////////////////
int CDXMap::GetTile(int MapX, int MapY)
{
	return DATA[MapX + (MapY * m_Width)].GetTile();
}

//////////////////////////////////////////////////////////////////////////////////
// Sets the value of the tile at MapX, MapY. Could be used for tile collision detection.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::SetTile(int MapX, int MapY, int Tile)
{
	DATA[MapX + (MapY * m_Width)].SetTile(Tile);
}

//////////////////////////////////////////////////////////////////////////////////
// Loads a different set of tiles to the map.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::LoadTiles(CDXTile *pTiles)
{
	Tiles = pTiles;
}

//////////////////////////////////////////////////////////////////////////////////
// Internal function to blt a tile to the screen.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::BltTile(CDXSurface* lpDDS, int xdest, int ydest, int w, int h, int xoff, int yoff)
{
	HRESULT rval;
	RECT src;
	int tile_num;
	int x1, y1;
	int mapx, mapy;

	mapx = ((m_PosX + xdest) % m_PixelWidth) / m_TileWidth;
	mapy = ((m_PosY + ydest) % m_PixelHeight) / m_TileHeight;

	tile_num = DATA[(mapy * m_Width) + mapx].GetTile();

	if(tile_num == 0) return;

	int TILE_SW = Tiles->m_PixelWidth / m_TileWidth;

	x1 = tile_num % TILE_SW;
	x1 = (x1 * m_TileWidth) + xoff;

	y1 = tile_num / TILE_SW;
	y1 = (y1 * m_TileHeight) + yoff;

	src.top = y1;
	src.left = x1;
	src.bottom = y1+h;
	src.right = x1+w;

	rval = lpDDS->m_lpDDS->BltFast(xdest, ydest, Tiles->m_lpDDS, &src, DDBLTFAST_WAIT);
	if(rval == DDERR_SURFACELOST) Tiles->Restore();
}

//////////////////////////////////////////////////////////////////////////////////
// Internal function to blt a tile transparently to the screen.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::BltTileTrans(CDXSurface* lpDDS, int xdest, int ydest, int w, int h, int xoff, int yoff)
{
	HRESULT rval;
	RECT src;
	int tile_num;
	int x1, y1;
	int mapx, mapy;

	mapx = ((m_PosX + xdest) % m_PixelWidth) / m_TileWidth;
	mapy = ((m_PosY + ydest) % m_PixelHeight) / m_TileHeight;

	tile_num = DATA[(mapy * m_Width) + mapx].GetTile();

	if(tile_num == 0) return;

	int TILE_SW = Tiles->m_PixelWidth / m_TileWidth;

	x1 = tile_num % TILE_SW;
	x1 = (x1 * m_TileWidth) + xoff;

	y1 = tile_num / TILE_SW;
	y1 = (y1 * m_TileHeight) + yoff;

	src.top = y1;
	src.left = x1;
	src.bottom = y1+h;
	src.right = x1+w;

	rval = lpDDS->m_lpDDS->BltFast(xdest, ydest, Tiles->m_lpDDS, &src, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);
	if(rval == DDERR_SURFACELOST) Tiles->Restore();
}

//////////////////////////////////////////////////////////////////////////////////
// Draws the map to the surface pointed to by lpDDS.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::Draw(CDXSurface* lpDDS)
{
	int i,j;
	int xoffset, yoffset;
	int xcoord = 0, ycoord = 0;

	xoffset = m_PosX % m_TileWidth;
	yoffset = m_PosY % m_TileHeight;

	// FIRST ROW
	BltTile(lpDDS, 0, 0, m_TileWidth-xoffset, m_TileWidth-yoffset, xoffset, yoffset);

	for(i=0; i<SCREEN_TW-1; i++)
  {
		xcoord += m_TileWidth;
		BltTile(lpDDS, xcoord-xoffset, 0, m_TileWidth, m_TileHeight-yoffset, 0, yoffset);
  }

	xcoord += m_TileWidth;
	BltTile(lpDDS, xcoord-xoffset, 0, xoffset, m_TileHeight-yoffset, 0, yoffset);

	// NEXT X NUMBER OF ROWS
	for(j=0; j<SCREEN_TH-1; j++)
	{
		xcoord = m_TileWidth;
		ycoord += m_TileHeight;
		BltTile(lpDDS, 0, ycoord-yoffset, m_TileWidth-xoffset, m_TileHeight, xoffset, 0);

		for(i=0; i<SCREEN_TW-1; i++)
		{
			BltTile(lpDDS, xcoord-xoffset, ycoord-yoffset, m_TileWidth, m_TileHeight, 0, 0);
			xcoord += m_TileWidth;
		}

		BltTile(lpDDS, xcoord-xoffset, ycoord-yoffset, xoffset, m_TileHeight, 0, 0);
	}

	// LAST ROW
	xcoord = 0;
	ycoord += m_TileHeight;
	BltTile(lpDDS, 0, ycoord-yoffset, m_TileWidth-xoffset, yoffset, xoffset, 0);

	for(i=0; i<SCREEN_TW-1; i++)
	{
		xcoord += m_TileWidth;
		BltTile(lpDDS, xcoord-xoffset, ycoord-yoffset, m_TileWidth, yoffset, 0, 0);
	}

	xcoord += m_TileWidth;
	BltTile(lpDDS, xcoord-xoffset, ycoord-yoffset, xoffset, yoffset, 0, 0);
}

//////////////////////////////////////////////////////////////////////////////////
// Draws the map to the surface pointed to by lpDDS with transparent tiles. If 
// you haven't set the colour key for the tiles nothing will be drawn.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::DrawTrans(CDXSurface* lpDDS)
{
	int i,j;
	int xoffset, yoffset;
	int xcoord = 0, ycoord = 0;

	xoffset = m_PosX % m_TileWidth;
	yoffset = m_PosY % m_TileHeight;

	// FIRST ROW
	BltTileTrans(lpDDS, 0, 0, m_TileWidth-xoffset, m_TileWidth-yoffset, xoffset, yoffset);

	for(i=0; i<SCREEN_TW-1; i++)
  {
		xcoord += m_TileWidth;
		BltTileTrans(lpDDS, xcoord-xoffset, 0, m_TileWidth, m_TileHeight-yoffset, 0, yoffset);
  }

	xcoord += m_TileWidth;
	BltTileTrans(lpDDS, xcoord-xoffset, 0, xoffset, m_TileHeight-yoffset, 0, yoffset);

	// NEXT X NUMBER OF ROWS
	for(j=0; j<SCREEN_TH-1; j++)
	{
		xcoord = m_TileWidth;
		ycoord += m_TileHeight;
		BltTileTrans(lpDDS, 0, ycoord-yoffset, m_TileWidth-xoffset, m_TileHeight, xoffset, 0);

		for(i=0; i<SCREEN_TW-1; i++)
		{
			BltTileTrans(lpDDS, xcoord-xoffset, ycoord-yoffset, m_TileWidth, m_TileHeight, 0, 0);
			xcoord += m_TileWidth;
		}

		BltTileTrans(lpDDS, xcoord-xoffset, ycoord-yoffset, xoffset, m_TileHeight, 0, 0);
	}

	// LAST ROW
	xcoord = 0;
	ycoord += m_TileHeight;
	BltTileTrans(lpDDS, 0, ycoord-yoffset, m_TileWidth-xoffset, yoffset, xoffset, 0);

	for(i=0; i<SCREEN_TW-1; i++)
	{
		xcoord += m_TileWidth;
		BltTileTrans(lpDDS, xcoord-xoffset, ycoord-yoffset, m_TileWidth, yoffset, 0, 0);
	}

	xcoord += m_TileWidth;
	BltTileTrans(lpDDS, xcoord-xoffset, ycoord-yoffset, xoffset, yoffset, 0, 0);
}

//////////////////////////////////////////////////////////////////////////////////
// Draws the map to the surface pointed to by lpDDS. The map is clipped to 
// the rectangle ClipDestRect.
//////////////////////////////////////////////////////////////////////////////////
void CDXMap::DrawClipped(CDXSurface* lpDDS, LPRECT ClipRect)
{
	int xoffset = (m_PosX + ClipRect->left) % m_TileWidth;
	int yoffset = (m_PosY + ClipRect->top) % m_TileWidth;
	int xcoord = 0;
	int ycoord = 0;
	int minwidth = (m_TileWidth - xoffset) < (ClipRect->right - ClipRect->left)
									? (m_TileWidth - xoffset) : (ClipRect->right - ClipRect->left);
	int minheight = (m_TileHeight - yoffset) < (ClipRect->bottom - ClipRect->top)
									 ? (m_TileHeight - yoffset) : (ClipRect->bottom - ClipRect->top);

	// FIRST ROW
	BltTile(lpDDS, ClipRect->left, ClipRect->top, minwidth, minheight, xoffset, yoffset);
	xcoord = ClipRect->left + minwidth;

	while ((xcoord + m_TileWidth) < ClipRect->right)
	{
		BltTile(lpDDS, xcoord, ClipRect->top, m_TileWidth, minheight, 0, yoffset);
		xcoord += m_TileWidth;
	}

	BltTile(lpDDS, xcoord, ClipRect->top, ClipRect->right - xcoord,	minheight, 0, yoffset);

	// NEXT X NUMBER OF ROWS
	ycoord = ClipRect->top + minheight;

	while((ycoord + m_TileHeight) < ClipRect->bottom)
	{
		BltTile(lpDDS, ClipRect->left, ycoord, minwidth, m_TileHeight,	xoffset, 0);
		xcoord = ClipRect->left + minwidth;

		while((xcoord + m_TileWidth) < ClipRect->right)
		{
			BltTile(lpDDS, xcoord, ycoord, m_TileWidth, m_TileHeight, 0, 0);
			xcoord += m_TileWidth;
		}

		BltTile(lpDDS, xcoord, ycoord, ClipRect->right - xcoord,	m_TileHeight, 0, 0);
		ycoord += m_TileHeight;
	}

	// LAST ROW
	BltTile(lpDDS, ClipRect->left, ycoord, minwidth, ClipRect->bottom - ycoord, xoffset, 0);
	xcoord = ClipRect->left + minwidth;

	while((xcoord + m_TileWidth) < ClipRect->right)
	{
		BltTile(lpDDS, xcoord, ycoord, m_TileWidth, ClipRect->bottom - ycoord,	0, 0);
		xcoord += m_TileWidth;
	}

	BltTile(lpDDS, xcoord, ycoord, ClipRect->right - xcoord,	ClipRect->bottom - ycoord, 0, 0);
}
