//////////////////////////////////////////////////////////////////////////////////
// $Source: /usr/cvsroot/cdx/src/cdx/cdxsurface.cpp,v $
// $Author: janni $
//
// $Log: cdxsurface.cpp,v $
// Revision 2.12  1999/05/23 05:21:23  janni
// InitCDXSurface did not initialize all member variables, fixed
//
// Revision 2.11  1999/05/20 21:06:15  janni
// fixed problems in CDXSurface copy constructor
//
// Revision 2.10  1999/05/20 17:03:01  pietro
// hotfix for warning C4244: '=' : conversion... in alpha code..
//
// Revision 2.9  1999/05/20 15:29:03  pietro
// Multiple changes:
// * fixed #include bugs in all .cpp and various .h files
// * fixed all rcsid[] bugs
// * added conditional compile variable CDXINCLUDEALL - when defined,
//   all #include files are included in cdx.h to keep backward compatibility
// * All the libraries are created in ..\..\lib\vc\ directory, library names are
//   cdx.lib/cdxd.lib/cdxdx3.lib/cdxdx3d.lib/cdxadx3.lib/cdxadx3d.lib
//
// Revision 2.8  1999/05/19 04:30:57  pietro
// modified CreateFromBMPi so that the sizeis last one parameter.
//
// Revision 2.7  1999/05/19 04:07:20  pietro
// removed LoadBMP from CDX Resource. Now every object handles his own loading
// via his own interfaces. in cdx surface, use create.
//
// Revision 2.6  1999/05/19 02:28:00  rbt
// Added a copy constructor.
//
// Revision 2.5  1999/05/16 17:45:21  pietro
// Added CDXSurface::CreateFromBMPi.
// Creates a CDXSurface from a 24BPP BMP file.
// The file must be already opened and the position set. (fpos).
// The file will be left opened, but the position will change.
// With this function, you can have many bmp files in one large file.
//
// Revision 2.4  1999/05/11 21:45:55  janni
// changed DELETE macro to SAFEDELETE to get NT compatibility
//
// Revision 2.3  1999/05/11 18:54:15  janni
// added InitCDXSurface function, is called from all  3 constructors to init member variables and fix access violation problems
// added RELEASE( m_lpDDS ) in both Create functions to release allocated surfaces before creating new ones
//
// Revision 2.2  1999/05/10 06:41:28  MICHAELR
// Create was checking for PSD8, but wasn't checking to see if FH == NULL, fgetc failed if so
//
// Revision 2.1  1999/05/03 21:00:39  MICHAELR
// SetColorKey didn't have proper handling of DX3 using #ifdefs for ddsd
//
// Revision 2.0  1999/05/01 13:51:17  bsimser
// Updated revision number to 2.0
//
// Revision 1.1.1.1  1999/05/01 04:10:56  bsimser
// Initial revision to cvs
//
// $Revision: 2.12 $
//////////////////////////////////////////////////////////////////////////////////
#ifdef SAVE_RCSID
static char rcsid[] = "@(#) $Id: cdxsurface.cpp,v 2.12 1999/05/23 05:21:23 janni Exp $";
#endif

#include "CDX.h"
#include "cdxsurface.h"
#include "math.h"

#define MAX_PSD_CHANNELS	24


//////////////////////////////////////////////////////////////////////////////////
// Is called from all constructors to init member variables
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::InitCDXSurface( void )
{
	m_lpDDS = NULL;
	m_DC = NULL;
	m_Font = NULL;
	Screen = NULL;
	m_FontLoaded = FALSE;
	m_pFilename = NULL;
	m_ColorKey = -1;
	DestRect.top = 0;
	DestRect.left = 0;
	DestRect.bottom = 0;
	DestRect.right = 0;
	SrcRect.top = 0;
	SrcRect.left = 0;
	SrcRect.bottom = 0;
	SrcRect.right = 0;
	m_ClipRect.top = 0;
	m_ClipRect.left = 0;
	m_ClipRect.bottom = 0;
	m_ClipRect.right = 0;
	m_MMX = IsMMX();
	m_alphaMask     = NULL;
}


//////////////////////////////////////////////////////////////////////////////////
// Default constructor
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::CDXSurface(void)
{
	InitCDXSurface();
}

//////////////////////////////////////////////////////////////////////////////////
// Creates a empty surface with a size set by Width and Height. memoryType is where
// you want the surface to reside (video or system).
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::CDXSurface(CDXScreen *pScreen, int Width, int Height, BOOL memoryType)
{
	InitCDXSurface();
	Create(pScreen, Width, Height, memoryType);
}

//////////////////////////////////////////////////////////////////////////////////
// Creates a surface holding the bitmap pointed to by szFilename. memoryType is where
// you want the surface to reside (video or system)
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::CDXSurface(CDXScreen *pScreen, const char *szFilename, BOOL memoryType)
{
	InitCDXSurface();
	Create(pScreen, szFilename, memoryType);
}

//////////////////////////////////////////////////////////////////////////////////
// Copy constructor
// Usage: (assuming that pOriginal is a pointer to a CDXSurface)
//	   CDXSurface copy(*pOriginal);
//	   CDXSurface* pCopy = new CDXSurface(*pOriginal);
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::CDXSurface(CDXSurface& aSurface)					
{
	HRESULT rval;

	InitCDXSurface();

	// Copy all member variables
	DestRect		= aSurface.DestRect;
	m_ClipRect		= aSurface.m_ClipRect;
	m_ColorKey		= aSurface.m_ColorKey;
	m_DC			= aSurface.m_DC;			 //!!??!! How do we handle copying of handles !!??!!
	m_DDSD			= aSurface.m_DDSD;		 // All settings including if sysmem or vidmem
	Screen		    = aSurface.Screen;
	m_MMX				= aSurface.m_MMX;
	//sprintf((char*)m_pFilename,"%s", aSurface.m_pFilename);	// Copy the string
	m_pFilename     = aSurface.m_pFilename;
	m_PixelHeight	= aSurface.m_PixelHeight;
	m_PixelWidth	= aSurface.m_PixelWidth;
	m_RGB				= aSurface.m_RGB;
	Screen			= aSurface.Screen;			// Ok to copy pointer here...
	SrcRect			= aSurface.SrcRect;
	if(aSurface.m_alphaMask != NULL)
	{
		m_alphaMask = new BYTE[m_PixelWidth*m_PixelHeight];
		for(int i=0;i<m_PixelWidth*m_PixelHeight;i++)
			m_alphaMask[i] = aSurface.m_alphaMask[i];
	}

	// Copy the font
	LOGFONT FontInfo;
	GetObject( aSurface.m_Font , sizeof( FontInfo ) , &FontInfo );
	m_Font = CreateFontIndirect( &FontInfo );
	m_FontLoaded	= aSurface.m_FontLoaded;

	// Create a dd Surface
	rval = Screen->m_lpDD->CreateSurface(&m_DDSD, &m_lpDDS, NULL);
	if(rval != DD_OK)
	{
		DDError(rval, Screen->m_hWnd, __FILE__, __LINE__);
		return;
	}
	else
	{
		// Add ref to screen dd surface
		Screen->m_lpDD->AddRef();

		// Copy surface content to us by a simple call to Draw
		aSurface.Draw(this);
	}

}



//////////////////////////////////////////////////////////////////////////////////
// Destroys the surface and frees the memory.
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::~CDXSurface()
{
	if(m_Font != NULL) {
		DeleteObject(m_Font);
	}

	if(m_DC != NULL) {
		ReleaseDC();
	}

	if(m_FontLoaded) {
		RemoveFontResource(m_FontName);
	}

	if (m_alphaMask)
		delete [] m_alphaMask;

	RELEASE(m_lpDDS);
}

//////////////////////////////////////////////////////////////////////////////////
// Creates a empty surface with a size set by Width and Height. memoryType is where
// you want the surface to reside (video or system memory)
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::Create(CDXScreen *pScreen, int Width, int Height, BOOL memoryType)
{
	HRESULT rval;

	ZeroMemory(&m_DDSD, sizeof(m_DDSD));
	m_DDSD.dwSize = sizeof(m_DDSD);
	m_DDSD.dwFlags = DDSD_CAPS | DDSD_HEIGHT |DDSD_WIDTH;

	switch(memoryType) {
	    case CDXMEM_SYSTEMONLY:
	        m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
			break;

		case CDXMEM_VIDEOONLY:
			m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY;
			break;
		
        case CDXMEM_VIDTHENSYS:
		default:
	        m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
			break;
	}

	m_DDSD.dwWidth = Width;
	m_DDSD.dwHeight = Height;

	RELEASE( m_lpDDS ); // janni
	rval = pScreen->m_lpDD->CreateSurface(&m_DDSD, &m_lpDDS, NULL);
	if(rval != DD_OK)
	{
		DDError(rval, pScreen->m_hWnd, __FILE__, __LINE__);
		return FALSE;
	}

	m_DC = NULL;
	m_Font = NULL;
	m_pFilename = NULL;
	Screen = pScreen;
	m_ColorKey = -1;

	m_PixelWidth = Width;
	m_PixelHeight = Height;

	DestRect.top = 0;
	DestRect.left = 0;
	DestRect.bottom = m_PixelHeight;
	DestRect.right = m_PixelWidth;

	SrcRect.top = 0;
	SrcRect.left = 0;
	SrcRect.bottom = m_PixelHeight;
	SrcRect.right = m_PixelWidth;

	m_ClipRect.top = 0;
	m_ClipRect.left = 0;
	m_ClipRect.bottom = m_PixelHeight - 1;
	m_ClipRect.right = m_PixelWidth - 1;

	GetRGBFormat(m_lpDDS, &m_RGB);
	m_MMX = IsMMX();
	pScreen->m_lpDD->AddRef();
	m_alphaMask = NULL;
    
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// Creates a surface holding the bitmap pointed to by szFilename.  memoryType is where
// you want the surface to reside (video or system memory)
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::Create(CDXScreen *pScreen, const char *szFilename, BOOL memoryType)
{
	char signature[5];
	FILE *fh;

	if(szFilename == NULL) 
		return FALSE;

	// clear out the sig
	ZeroMemory(signature, sizeof(signature));

	// determine what type of file it is
	fh=fopen(szFilename, "rb");
	
	if (fh != NULL)
	{
		// if the first 4 bytes equal "8BPS" then it's a PSD file
		signature[0]=fgetc(fh);
		signature[1]=fgetc(fh);
		signature[2]=fgetc(fh);
		signature[3]=fgetc(fh);
		signature[4]='\0';
		fclose(fh);
	}
	m_alphaMask=NULL;

	RELEASE( m_lpDDS ); // janni
	if(strcmp(signature, "8BPS") == 0) {
		CreateFromPSD(pScreen, szFilename, memoryType);
	}
	else {
		m_lpDDS = DDLoadSizeBitmap(pScreen->m_lpDD, szFilename, &m_PixelWidth, &m_PixelHeight, memoryType);
	}

	if(m_lpDDS == NULL) 
		return FALSE;

	m_DC = NULL;
	m_Font = NULL;
	m_pFilename = szFilename;
	Screen = pScreen;
	m_ColorKey = 0;

	DestRect.top = 0;
	DestRect.left = 0;
	DestRect.bottom = m_PixelHeight;
	DestRect.right = m_PixelWidth;

	SrcRect.top = 0;
	SrcRect.left = 0;
	SrcRect.bottom = m_PixelHeight;
	SrcRect.right = m_PixelWidth;

	m_ClipRect.top		= 0;
	m_ClipRect.left		= 0;
	m_ClipRect.bottom	= m_PixelHeight - 1;
	m_ClipRect.right	= m_PixelWidth - 1;

	GetRGBFormat(m_lpDDS,&m_RGB);
	m_MMX = IsMMX();
	pScreen->m_lpDD->AddRef();
	
	ZeroMemory(&m_DDSD, sizeof(m_DDSD));
	m_DDSD.dwSize = sizeof(m_DDSD);
	m_DDSD.dwFlags = DDSD_CAPS | DDSD_HEIGHT |DDSD_WIDTH;

	switch (memoryType)
	{
		case CDXMEM_SYSTEMONLY:
			m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
			break;

		case CDXMEM_VIDEOONLY:
			m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY;
			break;

		case CDXMEM_VIDTHENSYS:
		default:
			m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
			break;
	}

	m_DDSD.dwWidth = m_PixelWidth;
	m_DDSD.dwHeight = m_PixelHeight;

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// Loads Bitmap (.bmp) file from resource. This creates new CDXSurface which has
// to be deleted later. Only 24bpp files are supported.
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::Create(CDXScreen *pScreen, CDXResource *r, RESHANDLE i, BYTE memoryType)
{
	r->Seek(i);
	return CreateFromBMPi(pScreen, r->GetfHandle(), memoryType, r->GetSize(i));
}



//////////////////////////////////////////////////////////////////////////////////
// Creates a CDXSurface from a 24BPP BMP file.
// The file must be already opened and the position set. (fpos).
// The file will be left opened, but the position will change.
// With this function, you can have many bmp files in one large file.
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::Create(CDXScreen *pScreen, FILE *fh, UINT size, BYTE memoryType)
{
	return CreateFromBMPi(pScreen, fh, memoryType, size);
}

BOOL CDXSurface::CreateFromBMPi(CDXScreen *pScreen, FILE *fh, BYTE memoryType, UINT size)
{
	BITMAPINFO			bi;
	BITMAPFILEHEADER	bif;
	UINT				DibBitsSize;
	UINT				dwFileStart;
	BYTE				*DibBits;

	//save current position
	dwFileStart = ftell(fh);
	
	//BITMAPFILEHEADER
	if (fread(&bif, sizeof(bif),1,fh)!=1)
		return false;
	if (bif.bfType != 'MB')
		return false;
	 //BITMAPINFOHEADER
	if (fread(&bi, sizeof(BITMAPINFOHEADER),1,fh)!=1)
		return false;

	// Determine the size of the image in the file
	if (size==0)
		DibBitsSize = bif.bfSize - bif.bfOffBits;
	else
		DibBitsSize = size - bif.bfOffBits;
	
	// Allocate memory to hold the image
	if (NULL == (DibBits = (BYTE*) malloc(DibBitsSize)))
		return false;
 
	// Move to the start of DibBits
	fseek(fh, dwFileStart + bif.bfOffBits, SEEK_SET);

	// Read the DibBits into memory
	if(1 != (fread(DibBits, DibBitsSize,1,fh)))
		return false;
 
	m_PixelWidth=bi.bmiHeader.biWidth;
	m_PixelHeight=bi.bmiHeader.biHeight; 

	BOOL retval = Create(pScreen, m_PixelWidth, m_PixelHeight, memoryType);
	if (retval == FALSE)
	{
		free(DibBits);
		return FALSE;
	}
	GetDC();
	int rc = StretchDIBits(
			m_DC,
			0,0,
			m_PixelWidth, m_PixelHeight,
			0,0,
			m_PixelWidth, m_PixelHeight,
			DibBits,
			&bi,
			DIB_RGB_COLORS,SRCCOPY);

	ReleaseDC();
	free(DibBits);
	if (rc == GDI_ERROR)
		return false;
	
	return true;
}



//////////////////////////////////////////////////////////////////////////////////
// Creates a CDXSurface from a PSD file
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::CreateFromPSD(CDXScreen *pScreen, const char* fn, BYTE memType)
{
	FILE *fh;
	WORD channelCount,compressionType;
   	
	// Open the PSD file for reading
	if ( (fh = fopen (fn, "rb")) == NULL)
		return FALSE;

	// First 4 bytes must equal "8BPS"
	char signature[5];
	signature[0] = fgetc(fh);
	signature[1] = fgetc(fh);
	signature[2] = fgetc(fh);
	signature[3] = fgetc(fh);
	signature[4] = '\0';
	if (strcmp(signature,"8BPS") != 0)
		return FALSE;

	// Version must be equal to 1
	if ( Read16(fh) != 1 )
		return FALSE;

	// Skip some more data (Always 0)
	Read32(fh);
	Read16(fh);

	// Read in the number of Channels
	channelCount = Read16(fh);

	// Make sure there's at least one layer, and no more than our
	// constant.
	if ((channelCount < 0) || (channelCount > MAX_PSD_CHANNELS))
		return FALSE;

	// Load in the width and height of the PSD
	m_PixelHeight = Read32(fh);
	m_PixelWidth  = Read32(fh);

	// We can only load in files with 8 bits per channel (24bpp)
	if (Read16(fh) != 8) 
		return FALSE;

	// Make sure the color mode is RGB.
	// Valid options are:
	//   0: Bitmap
	//   1: Grayscale
	//   2: Indexed color
	//   3: RGB color
	//   4: CMYK color
	//   7: Multichannel
	//   8: Duotone
	//   9: Lab color
	if (Read16(fh) != 3)
		return FALSE;

	// Skip the Mode Data.  (It's the palette for indexed color; 
	// other info for other modes.)
	int	ModeDataCount=Read32(fh);
	if (ModeDataCount)
		fseek(fh,ModeDataCount,SEEK_CUR);

	// Skip the image resources.  (resolution, pen tool paths, etc)
	int	ResourceDataCount=Read32(fh);
	if (ResourceDataCount)
		fseek(fh,ResourceDataCount,SEEK_CUR);

	// Skip the reserved data.
	int	ReservedDataCount=Read32(fh);
	if (ReservedDataCount)
		fseek(fh,ReservedDataCount,SEEK_CUR);

	// Find out if the data is compressed. (if it's not RLE, we can't
	// handle it) Known values:
	//   0: no compression
	//   1: RLE compressed
	compressionType = Read16(fh);
	if (compressionType > 1)
		return FALSE;

	// Allocate enough memory for all the pixels (4 bytes each)
	BYTE* PSDPixels = new BYTE[(m_PixelWidth * m_PixelHeight) * 4];

	// Unpack the PSD into the memory we just allocated
	UnPackPSD(fh, PSDPixels, channelCount, compressionType);

	// Close the file
	fclose (fh);

	// Create the DirectX surface
	BOOL retval = Create(pScreen, m_PixelWidth, m_PixelHeight, memType);
	
	if (retval == FALSE)
	{
		delete [] PSDPixels;
		return FALSE;
	}

	// Draw the bitmap on the DirectX surface
	BITMAPINFO BitmapInfo;
	ZeroMemory(&BitmapInfo,sizeof(BitmapInfo));
	BitmapInfo.bmiHeader.biSize     = sizeof(BitmapInfo.bmiHeader);
	BitmapInfo.bmiHeader.biWidth    = m_PixelWidth;
	BitmapInfo.bmiHeader.biHeight   = -m_PixelHeight;
	BitmapInfo.bmiHeader.biPlanes   = 1;
	BitmapInfo.bmiHeader.biBitCount = 32;

	//HDC hdc = 0;
	GetDC();

	int rc = StretchDIBits(m_DC,
					0,
					0,
					m_PixelWidth,
					m_PixelHeight,
					0,
					0,
					m_PixelWidth,
					m_PixelHeight,
					PSDPixels,
					&BitmapInfo,
					DIB_RGB_COLORS,
					SRCCOPY);
	
	ReleaseDC();
	if (rc == GDI_ERROR)
	{
		delete(PSDPixels);
		return FALSE;
	}

	// If we're loading the Alpha Blending Chanel
	if(channelCount > 3)
	{
		m_alphaMask = new BYTE[m_PixelHeight * m_PixelWidth];

		// Go through and load it all in
		for(int x=0; x < m_PixelWidth; x++)
			for(int y=0; y < m_PixelHeight; y++)
			{
				m_alphaMask[(y * m_PixelWidth) + x] = PSDPixels[(((y * m_PixelWidth) + x) * 4) + 3];
			}
	}
	else
	{
		m_alphaMask = NULL;
	}

	// De-allocate the memory we created to load in the pixels
	delete [] PSDPixels;
	
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// Given a file handle this method will unpack a PSD file
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::UnPackPSD(FILE * fh, BYTE* pixels, WORD channelCnt, WORD compression)
{
	int Default[4] = {0,0,0,255};
	int chn[4]     = {2,1,0,3};
	int PixelCount = m_PixelWidth * m_PixelHeight;

	if(compression) 
	{
		fseek(fh, m_PixelHeight * channelCnt * 2, SEEK_CUR);

		for(int c=0; c < 4; c++) 
		{
			int pn = 0;
			int channel = chn[c];

			if(channel >= channelCnt) 
			{
				for(pn=0; pn < PixelCount ;pn++)
				{
					pixels[(pn * 4) + channel] = Default[channel];
				}
			}
			else // Uncompress
			{
				int	count = 0;

				while(count < PixelCount)
				{
					int	len = fgetc(fh);

					if(len == 128) 
					{
					} 
					else if(len < 128) // No RLE  
					{
						len++;
						count += len;

						while(len)
						{
							pixels[(pn * 4) + channel] = fgetc(fh);
							pn++;
							len--;
						}
					}
					else if(len > 128)  // RLE packed
					{
						len ^= 0x0FF;
						len += 2;
						unsigned char val = fgetc(fh);
						count += len;
						while(len) 
						{
							pixels[(pn * 4) + channel] = val;
							pn++;
							len--;
						}
					}
				}
			}
		}
		
	} 
	else 
	{
		for(int c=0; c < 4; c++) 
		{
			int channel = chn[c];
			if (channel > channelCnt) 
			{
				for(int pn=0; pn < PixelCount; pn++)
				{
					pixels[(pn*4) + channel] = Default[channel];
				}
			} 
			else 
			{
				for(int n=0; n < PixelCount; n++) 
				{
					pixels[(n*4) + channel] = fgetc(fh);
				}
			}
		}
	}	
}

//////////////////////////////////////////////////////////////////////////////////
// Reads a 16-bit value from a file an converts it from big endian format
// to little endian format
//////////////////////////////////////////////////////////////////////////////////
unsigned short CDXSurface::Read16(FILE* fh) 
{
	// Read in the two bytes
	int hi=fgetc(fh);
	int lo=fgetc(fh);

	// Return them, joined together
	return lo+(hi<<8);
}

//////////////////////////////////////////////////////////////////////////////////
// Reads a 32-bit value from a file an converts it from big endian format
// to little endian format
//////////////////////////////////////////////////////////////////////////////////
unsigned int CDXSurface::Read32(FILE* fh) 
{
	// Read in the four bytes
	unsigned int b3=fgetc(fh);
	unsigned int b2=fgetc(fh);
	unsigned int b1=fgetc(fh);
	unsigned int b0=fgetc(fh);

	// Shift them, and return them
	return (b3<<24)+(b2<<16)+(b1<<8)+b0;
}

//////////////////////////////////////////////////////////////////////////////////
// This function can be used to load an alpha mask for your surface.
// The file format that this function expects is:
//
// <width><height><alpha values>
//
// width        : must match the width of the surface, 4 bytes long (DWORD)
// height       : must match the height of the surface, 4 bytes long (DWORD)
// alpha values : must be width*height bytes long. One byte value (0..255)
//                for each pixel on the surface.
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::LoadAlphaMask(const char *szFilename)
{
	FILE* fhMask;
	LONG width, height;
	DWORD byteCount;
	DWORD i;

	// check if a mask is already loaded
	if (m_alphaMask != NULL)
		return FALSE;
	
	// Open the alpha mask file for reading
	if ( (fhMask = fopen (szFilename, "rb")) == NULL)
		return FALSE;

	// Read in the width and height
	fread(&width,sizeof(LONG),1,fhMask);
	fread(&height,sizeof(LONG),1,fhMask);

	if ( (width != m_PixelWidth) || (height != m_PixelHeight) )
	{
		fclose(fhMask);
		return FALSE;
	}

	byteCount = width * height;

	// allocate the space for the mask
	if ( (m_alphaMask = new BYTE[byteCount]) == NULL)
	{
		fclose(fhMask);
		return FALSE;
	}

	// read in alpha mask
	for (i=0; i<byteCount; i++)
		m_alphaMask[i] = fgetc(fhMask);

	fclose(fhMask);
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// Does a BLT using a transparent alpha mask.
//
// NOTE: m_alphaMask is a pointer to an array of unsinged char values that
// represents the indiviual alpha values for the bitmap loaded into this
// CDXSurface.  The m_alphaMask array covers the whole surface not just one
// tile or sprite contained on the surface but for all tiles or sprites.
// For example if the surface is 400x200 then m_alphaMask points to an array
// that is also 400x200 BYTEs. If the m_alphaMask is equal to NULL then the
// function will call CDXBLT_Trans() and draw a simple transparent image.
//////////////////////////////////////////////////////////////////////////////////

// there are many warnings - i'll not disable it completely, so we don't forget
// that we need to fix this!
// warning C4244: '=' : conversion from 'unsigned long' to 'unsigned short', possible loss of data
// when removing this hothix, don't forget to remove also the #pragma at the end of alpha code
// hotfixed by pietro 20/MAY/99

#pragma warning( push )	
#pragma warning( once : 4244)

HRESULT CDXSurface::CDXBLT_TransAlphaMask(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect)
{
	int register i,j;
	int height,width;
	BYTE* lpSprite;
	BYTE* lpDest;
	unsigned long dPitch, SpritePitch;
	DWORD sColorKey, doubleColorKey;
	DWORD sTemp,dTemp;
	unsigned long sr,sg,sb,dr,dg,db;
	WORD sbuf,dbuf,abuf;
	DWORD ALPHA;
	DWORD Result;
	BOOL oddWidth = FALSE;
	BYTE* alphaPtr;

	// Point to the alpha channel
	alphaPtr = m_alphaMask;

	// If m_alphaMask = NULL then CDXBLT_TRANS
	if (alphaPtr == NULL)
	{
		CDXBLT_Trans(dest,lDestX, lDestY, srcRect);
		return DDERR_GENERIC;
	}

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	// Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left; 

    // Get the color key for sprite surface
    sColorKey = (DWORD)m_ColorKey;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

	// Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;


	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:
		
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest   += (lDestY * dPitch) + (lDestX * 2);
		alphaPtr += (srcRect.top * m_PixelWidth) + srcRect.left;

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));
		abuf = (WORD)(m_PixelWidth - width);

		doubleColorKey = sColorKey | (sColorKey << 16);

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					ALPHA = *alphaPtr;

					sb = sTemp & 0x1f;
					db = dTemp & 0x1f;
					sg = (sTemp >> 5) & 0x1f;
					dg = (dTemp >> 5) & 0x1f;
					sr = (sTemp >> 10) & 0x1f;
					dr = (dTemp >> 10) & 0x1f;

					*((WORD*)lpDest) = (DWORD)((ALPHA * (sb - db) >> 8) + db |
						((ALPHA * (sg - dg) >> 8) + dg) << 5 |
						((ALPHA * (sr - dr) >> 8) + dr) << 10);


				}

				lpDest   += 2;
				lpSprite += 2;
				alphaPtr++;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != doubleColorKey )
				{
					dTemp = *((DWORD*)lpDest);
					ALPHA = *((WORD*)alphaPtr);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						sb = sTemp & 0x1f;
						db = dTemp & 0x1f;
						sg = (sTemp >> 5) & 0x1f;
						dg = (dTemp >> 5) & 0x1f;
						sr = (sTemp >> 10) & 0x1f;
						dr = (dTemp >> 10) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)(((ALPHA & 0xFF) * (sb - db) >> 8) + db |
							(((ALPHA & 0xFF) * (sg - dg) >> 8) + dg) << 5 |
							(((ALPHA & 0xFF) * (sr - dr) >> 8) + dr) << 10);
											
					
					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						sb = (sTemp >> 16) & 0x1f;
						db = (dTemp >> 16) & 0x1f;
						sg = (sTemp >> 21) & 0x1f;
						dg = (dTemp >> 21) & 0x1f;
						sr = (sTemp >> 26) & 0x1f;
						dr = (dTemp >> 26) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( ((ALPHA >> 8) * (sb - db) >> 8) + db |
							(((ALPHA >> 8) * (sg - dg) >> 8) + dg) << 5 |
							(((ALPHA >> 8) * (sr - dr) >> 8) + dr) << 10 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest   += 4;
				lpSprite += 4;
				alphaPtr += 2;
			}while (--j > 0);

			lpDest   += dbuf;
			lpSprite += sbuf;
			alphaPtr += abuf;
		}while (--i > 0);

		break;


case 16:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest   += (lDestY * dPitch) + (lDestX * 2);
		alphaPtr += (srcRect.top * m_PixelWidth) + srcRect.left;

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));
		abuf = (WORD)(m_PixelWidth - width);

		doubleColorKey = sColorKey | (sColorKey << 16);

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					ALPHA = *alphaPtr;

					sb = sTemp & 0x1f;
					db = dTemp & 0x1f;
					sg = (sTemp >> 5) & 0x3f;
					dg = (dTemp >> 5) & 0x3f;
					sr = (sTemp >> 11) & 0x1f;
					dr = (dTemp >> 11) & 0x1f;

					*((WORD*)lpDest) = (DWORD)((ALPHA * (sb - db) >> 8) + db |
						((ALPHA * (sg - dg) >> 8) + dg) << 5 |
						((ALPHA * (sr - dr) >> 8) + dr) << 11);
				}

				lpDest   += 2;
				lpSprite += 2;
				alphaPtr++;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != doubleColorKey )
				{
					dTemp = *((DWORD*)lpDest);
					ALPHA = *((WORD*)alphaPtr);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						sb = sTemp & 0x1f;
						db = dTemp & 0x1f;
						sg = (sTemp >> 5) & 0x3f;
						dg = (dTemp >> 5) & 0x3f;
						sr = (sTemp >> 11) & 0x1f;
						dr = (dTemp >> 11) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)(((ALPHA & 0xFF) * (sb - db) >> 8) + db |
							(((ALPHA & 0xFF) * (sg - dg) >> 8) + dg) << 5 |
							(((ALPHA & 0xFF) * (sr - dr) >> 8) + dr) << 11);

					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						sb = (sTemp >> 16) & 0x1f;
						db = (dTemp >> 16) & 0x1f;
						sg = (sTemp >> 21) & 0x3f;
						dg = (dTemp >> 21) & 0x3f;
						sr = (sTemp >> 27) & 0x1f;
						dr = (dTemp >> 27) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( ((ALPHA >> 8) * (sb - db) >> 8) + db |
							(((ALPHA >> 8) * (sg - dg) >> 8) + dg) << 5 |
							(((ALPHA >> 8) * (sr - dr) >> 8) + dr) << 11 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest   += 4;
				lpSprite += 4;
				alphaPtr += 2;
			}while (--j > 0);

			lpDest   += dbuf;
			lpSprite += sbuf;
			alphaPtr += abuf;
		}while (--i > 0);

		break;
		
	case 24:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite  += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest    += (lDestY * dPitch) + (lDestX * 3);
		alphaPtr  += (srcRect.top * m_PixelWidth) + srcRect.left;

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));
		abuf = (WORD)(m_PixelWidth - width);

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					ALPHA = *alphaPtr;
					sb = sTemp & 0xFF;
					db = dTemp & 0xFF;
					sg = (sTemp >> 8) & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					sr = (sTemp >> 16) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)((ALPHA * (sb - db) >> 8) + db |
						((ALPHA * (sg - dg) >> 8) + dg) << 8 |
						((ALPHA * (sr - dr) >> 8) + dr) << 16);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest++;
				}
				else
				{
					lpDest += 3;
				}

				lpSprite += 3;
				alphaPtr++;
			}while (--j > 0);

			lpDest   += dbuf;
			lpSprite += sbuf;
			alphaPtr += abuf;
		}while (--i > 0);

		break;

	case 32:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite  += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest    += (lDestY * dPitch) + (lDestX * 4);
		alphaPtr  += (srcRect.top * m_PixelWidth) + srcRect.left;

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));
		abuf = (WORD)(m_PixelWidth - width);

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					ALPHA = *alphaPtr;

					sb = sTemp & 0xFF;
					db = dTemp & 0xFF;
					sg = (sTemp >> 8) & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					sr = (sTemp >> 16) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)((ALPHA * (sb - db) >> 8) + db |
						((ALPHA * (sg - dg) >> 8) + dg) << 8 |
						((ALPHA * (sr - dr) >> 8) + dr) << 16);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest += 2;
				}
				else
				{
					lpDest += 4;
				}

				lpSprite += 4;
				alphaPtr++;
			}while (--j > 0);

			lpDest   += dbuf;
			lpSprite += sbuf;
			alphaPtr += abuf;
		}while (--i > 0);

		break;
	} // End RGB Format switch statement


	UnLock();
	dest->UnLock();

	return DD_OK;
}



//////////////////////////////////////////////////////////////////////////////////
// ValdidateBlt
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::ValidateBlt(CDXSurface* lpCDXS, LONG *lDestX, LONG *lDestY, RECT *srcRect)
{
	RECT destRect;
	RECT tmpRect;

	// Test if srcRect lies somewhere on this surface's clipRect
	if (!(ClipRect(srcRect)))
		return FALSE;

	// From srcRect build the destRect
	destRect.top    = *lDestY;
	destRect.left   = *lDestX;
	destRect.bottom = *lDestY + (srcRect->bottom - srcRect->top);
	destRect.right  = *lDestX + (srcRect->right - srcRect->left);

	// Save off the destRect
	tmpRect.top    = destRect.top;
	tmpRect.left   = destRect.left;
	tmpRect.bottom = destRect.bottom;
	tmpRect.right  = destRect.right;

	// Test if destRect lies somewhere on lpCDXS surface's clipRect
	if (!(lpCDXS->ClipRect(&destRect)))
		return FALSE;

	// Compare the returned destRect to the saved tmpRect
	if (destRect.top != tmpRect.top)
		srcRect->top += destRect.top - tmpRect.top;

	if (destRect.left != tmpRect.left)
		srcRect->left += destRect.left - tmpRect.left;

	if (destRect.bottom != tmpRect.bottom)
		srcRect->bottom += destRect.bottom - tmpRect.bottom;

	if (destRect.right != tmpRect.right)
		srcRect->right += destRect.right - tmpRect.right;

	*lDestX = destRect.left;
	*lDestY = destRect.top;

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// Clips a rectangle specified by Rect to the member variable m_clipRect 
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::ClipRect(RECT *Rect)
{
	// Check if Rect is completely outside of the m_clipRect (Screen Space)
	if (Rect->top >= m_ClipRect.bottom)
		return FALSE;

	if (Rect->left >= m_ClipRect.right)
		return FALSE;

	if (Rect->bottom <= m_ClipRect.top)
		return FALSE;

	if (Rect->right <= m_ClipRect.left)
		return FALSE;

	// Clip rect to the surface's clipRect
	if (Rect->top < m_ClipRect.top)
		Rect->top = m_ClipRect.top;

	if (Rect->left < m_ClipRect.left)
		Rect->left = m_ClipRect.left;

	if (Rect->bottom > m_ClipRect.bottom)
		Rect->bottom = m_ClipRect.bottom;

	if (Rect->right > m_ClipRect.right)
		Rect->right = m_ClipRect.right;

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// Fills out the passed RECT with m_clipRect
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::GetClipRect(RECT *clipRect)
{
	clipRect->top = m_ClipRect.top;
	clipRect->left = m_ClipRect.left;
	clipRect->bottom = m_ClipRect.bottom;
	clipRect->right = m_ClipRect.right;
}

//////////////////////////////////////////////////////////////////////////////////
// Set m_clipRect to the passed in RECT values
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetClipRect(RECT *clipRect)
{
	if ((m_ClipRect.top = clipRect->top) < 0)
		m_ClipRect.top = 0;

	if ((m_ClipRect.left = clipRect->left) < 0)
		m_ClipRect.left = 0;

	if ((m_ClipRect.bottom = clipRect->bottom) > m_PixelHeight)
		m_ClipRect.bottom = m_PixelHeight;

	if ((m_ClipRect.right = clipRect->right) > m_PixelWidth)
		m_ClipRect.right = m_PixelWidth;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_Blk
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_Blk(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;
	
	rval = lpCDXS->m_lpDDS->BltFast(lDestX, lDestY, m_lpDDS, &srcRect, DDBLTFAST_WAIT);

	if(rval == DDERR_SURFACELOST) 
		Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_Trans
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_Trans(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;
	
	rval = lpCDXS->m_lpDDS->BltFast(lDestX, lDestY, m_lpDDS, &srcRect, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);

	if(rval == DDERR_SURFACELOST) 
		Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_BlkHFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkHFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;
	DDBLTFX ddBltFx;
	RECT destRect;
	DWORD Pixel;
	LONG width, height;
	LONG tmpLeft, tmpRight;

	tmpLeft = lDestX;  // equal to or smaller then lDestX
	tmpRight = srcRect.right;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	if (tmpLeft != lDestX)
	{
		srcRect.left -= lDestX - tmpLeft;
		srcRect.right -= lDestX - tmpLeft;
		tmpRight -= lDestX - tmpLeft;
	}

	if (tmpRight != srcRect.right)
	{
		srcRect.left += tmpRight - srcRect.right;
		srcRect.right += tmpRight - srcRect.right;
	}

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORLEFTRIGHT;

	width           = srcRect.right - srcRect.left;
	height          = srcRect.bottom - srcRect.top;
	destRect.top    = lDestY;
	destRect.left   = lDestX;
	destRect.bottom = lDestY + height;
	destRect.right  = lDestX + width;

	rval = lpCDXS->m_lpDDS->Blt(&destRect, m_lpDDS, &srcRect, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
	if (rval == DD_OK)
		return rval;

	for(int j = 0; j < height; j++)
	{
		for(int i = 0; i < width; i++)
		{
			Pixel = GetPixel(srcRect.left + i, srcRect.top + j);
			lpCDXS->PutPixel((destRect.left + width) - i, destRect.top + j, Pixel);
		}
	}
	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_TransHFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransHFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;
	DDBLTFX ddBltFx;
	RECT destRect;
	DWORD Pixel;
	LONG width, height;
	LONG tmpLeft, tmpRight;

	tmpLeft = lDestX;  // equal to or smaller then lDestX
	tmpRight = srcRect.right;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	if (tmpLeft != lDestX)
	{
		srcRect.left -= lDestX - tmpLeft;
		srcRect.right -= lDestX - tmpLeft;
		tmpRight -= lDestX - tmpLeft;
	}

	if (tmpRight != srcRect.right)
	{
		srcRect.left += tmpRight - srcRect.right;
		srcRect.right += tmpRight - srcRect.right;
	}

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORLEFTRIGHT;

	width           = srcRect.right - srcRect.left;
	height          = srcRect.bottom - srcRect.top;
	destRect.top    = lDestY;
	destRect.left   = lDestX;
	destRect.bottom = lDestY + height;
	destRect.right  = lDestX + width;

	rval = lpCDXS->m_lpDDS->Blt(&destRect, m_lpDDS, &srcRect, DDBLT_DDFX | DDBLT_WAIT | DDBLT_KEYSRC, &ddBltFx);
	if (rval == DD_OK)
		return rval;

	for(int j = 0; j < height; j++)
	{
		for(int i = 0; i < width; i++)
		{
			Pixel = GetPixel(srcRect.left + i, srcRect.top + j);
			if (Pixel != m_ColorKey)
				lpCDXS->PutPixel((destRect.left + width) - i, destRect.top + j, Pixel);
		}
	}
	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_BlkVFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkVFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;
	DDBLTFX ddBltFx;
	RECT destRect;
	DWORD Pixel;
	LONG width, height;
	LONG tmpTop, tmpBottom;

	tmpTop = lDestY;  
	tmpBottom = srcRect.bottom;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	if (tmpTop != lDestY)
	{
		srcRect.top -= lDestY - tmpTop;
		srcRect.bottom -= lDestY - tmpTop;
		tmpBottom -= lDestY - tmpTop;
	}

	if (tmpBottom != srcRect.bottom)
	{
		srcRect.top += tmpBottom - srcRect.bottom;
		srcRect.bottom += tmpBottom - srcRect.bottom;
	}

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORUPDOWN;

	width           = srcRect.right - srcRect.left;
	height          = srcRect.bottom - srcRect.top;
	destRect.top    = lDestY;
	destRect.left   = lDestX;
	destRect.bottom = lDestY + height;
	destRect.right  = lDestX + width;

	rval = lpCDXS->m_lpDDS->Blt(&destRect, m_lpDDS, &srcRect, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
	if (rval == DD_OK)
		return rval;

	for(int j = 0; j < height; j++)
	{
		for(int i = 0; i < width; i++)
		{
			Pixel = GetPixel(srcRect.left + i, srcRect.top + j);
			lpCDXS->PutPixel(destRect.left + i, (destRect.top + height) - j, Pixel);
		}
	}
	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_TransVFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransVFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;
	DDBLTFX ddBltFx;
	RECT destRect;
	DWORD Pixel;
	LONG width, height;
	LONG tmpTop, tmpBottom;

	tmpTop = lDestY;  
	tmpBottom = srcRect.bottom;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	if (tmpTop != lDestY)
	{
		srcRect.top -= lDestY - tmpTop;
		srcRect.bottom -= lDestY - tmpTop;
		tmpBottom -= lDestY - tmpTop;
	}

	if (tmpBottom != srcRect.bottom)
	{
		srcRect.top += tmpBottom - srcRect.bottom;
		srcRect.bottom += tmpBottom - srcRect.bottom;
	}

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORUPDOWN;

	width           = srcRect.right - srcRect.left;
	height          = srcRect.bottom - srcRect.top;
	destRect.top    = lDestY;
	destRect.left   = lDestX;
	destRect.bottom = lDestY + height;
	destRect.right  = lDestX + width;

	rval = lpCDXS->m_lpDDS->Blt(&destRect, m_lpDDS, &srcRect, DDBLT_DDFX | DDBLT_WAIT | DDBLT_KEYSRC, &ddBltFx);
	if (rval == DD_OK)
		return rval;

	for(int j = 0; j < height; j++)
	{
		for(int i = 0; i < width; i++)
		{
			Pixel = GetPixel(srcRect.left + i, srcRect.top + j);
			if (Pixel != m_ColorKey)
				lpCDXS->PutPixel(destRect.left + i, (destRect.top + height) - j, Pixel);
		}
	}
	return DD_OK;
}

////////////////////////////////////////////////////////////////////////
// CDXBLT_BlkAlphaFast - 
////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkAlphaFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect)
{
    int register i,j;
	int height,width;
    BYTE* lpSprite;
    BYTE* lpDest;
    unsigned long dPitch, SpritePitch;
    DWORD sTemp,dTemp;
    WORD sbuf,dbuf;
    DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

    // Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

    // Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of the surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;

	// Start RGB Format switch statement
	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);
				dTemp = *((WORD*)lpDest);

				*((WORD*)lpDest) = ((sTemp & 0x7BDE) >> 1) + ((dTemp & 0x7BDE) >> 1);

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				Result = ((sTemp & 0x7BDE) >> 1) + ((dTemp & 0x7BDE) >> 1);
				Result |= ((sTemp & 0x7BDE0000) >> 1) + ((dTemp & 0x7BDE0000) >> 1);

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);
				dTemp = *((WORD*)lpDest);

				*((WORD*)lpDest) = ((sTemp & 0xF7DE) >> 1) + ((dTemp & 0xF7DE) >> 1);

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				Result = ((sTemp & 0xF7DE) >> 1) + ((dTemp & 0xF7DE) >> 1);
				Result |= ((sTemp & 0xF7DE0000) >> 1) + ((dTemp & 0xF7DE0000) >> 1);

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 24:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				Result = ((sTemp & 0xFEFEFE) >> 1) + ((dTemp & 0xFEFEFE) >> 1);

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);

				lpDest++;
				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		
		break;

	case 32:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				Result = ((sTemp & 0xFEFEFE) >> 1) + ((dTemp & 0xFEFEFE) >> 1);

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);

				lpDest += 2;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


    UnLock();
    dest->UnLock();

	return DD_OK;
}

////////////////////////////////////////////////////////////////////////
// CDXBLT_TRANSALPHAFAST - 
////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransAlphaFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect)
{
    int register i,j;
	int height,width;
    BYTE* lpSprite;
    BYTE* lpDest;
    unsigned long dPitch, SpritePitch;
    DWORD sColorKey;
    DWORD sTemp,dTemp;
    WORD sbuf,dbuf;
    DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

    // Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

    // Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of the surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;

	// Get the color key for sprite surface
	sColorKey = (DWORD)m_ColorKey;

	// Start RGB Format switch statement
	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					*((WORD*)lpDest) = ((sTemp & 0x7BDE) >> 1) + ((dTemp & 0x7BDE) >> 1);
				}

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);
				Result = dTemp;

				if ((sTemp & 0xFFFF) != sColorKey)
				{
					Result &= 0xFFFF0000;
					Result |= ((sTemp & 0x7BDE) >> 1) + ((dTemp & 0x7BDE) >> 1);
				}
				if ((sTemp >> 16) != sColorKey)
				{
					Result &= 0xFFFF;
					Result |= ((sTemp & 0x7BDE0000) >> 1) + ((dTemp & 0x7BDE0000) >> 1);
				}

				*((DWORD*)lpDest) = Result;
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					*((WORD*)lpDest) = ((sTemp & 0xF7DE) >> 1) + ((dTemp & 0xF7DE) >> 1);
				}

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);
				Result = dTemp;

				if ((sTemp & 0xFFFF) != sColorKey)
				{
					Result &= 0xFFFF0000;
					Result |= ((sTemp & 0xF7DE) >> 1) + ((dTemp & 0xF7DE) >> 1);
				}
				if ((sTemp >> 16) != sColorKey)
				{
					Result &= 0xFFFF;
					Result |= ((sTemp & 0xF7DE0000) >> 1) + ((dTemp & 0xF7DE0000) >> 1);
				}

				*((DWORD*)lpDest) = Result;
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 24:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp &= 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					Result = ((sTemp & 0xFEFEFE) >> 1) + ((dTemp & 0xFEFEFE) >> 1);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest++;
				}
				else
				{
					lpDest += 3;
				}

				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 32:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp &= 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					Result = ((sTemp & 0xFEFEFE) >> 1) + ((dTemp & 0xFEFEFE) >> 1);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest += 2;
				}
				else
				{
					lpDest += 4;
				}

				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


    UnLock();
    dest->UnLock();

	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_BLKALPHA - 
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkAlpha(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD ALPHA)
{
	int register i,j;
	int height,width;
	BYTE* lpSprite;
	BYTE* lpDest;
	unsigned long dPitch, SpritePitch;
	DWORD sTemp,dTemp;
	unsigned long sr,sg,sb,dr,dg,db;
	WORD sbuf,dbuf;
	DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	// Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left; 

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

	// Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;


	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;
		
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);
				dTemp = *((WORD*)lpDest);
				sb = sTemp & 0x1f;
				db = dTemp & 0x1f;
				sg = (sTemp >> 5) & 0x1f;
				dg = (dTemp >> 5) & 0x1f;
				sr = (sTemp >> 10) & 0x1f;
				dr = (dTemp >> 10) & 0x1f;

				*((WORD*)lpDest) = (WORD)((ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 10);

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				sb = sTemp & 0x1f;
				db = dTemp & 0x1f;
				sg = (sTemp >> 5) & 0x1f;
				dg = (dTemp >> 5) & 0x1f;
				sr = (sTemp >> 10) & 0x1f;
				dr = (dTemp >> 10) & 0x1f;

				Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 10);

				sb = (sTemp >> 16) & 0x1f;
				db = (dTemp >> 16) & 0x1f;
				sg = (sTemp >> 21) & 0x1f;
				dg = (dTemp >> 21) & 0x1f;
				sr = (sTemp >> 26) & 0x1f;
				dr = (dTemp >> 26) & 0x1f;

				Result |= (DWORD)( (ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 10 ) << 16;

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);
				dTemp = *((WORD*)lpDest);

				sb = sTemp & 0x1f;
				db = dTemp & 0x1f;
				sg = (sTemp >> 5) & 0x3f;
				dg = (dTemp >> 5) & 0x3f;
				sr = (sTemp >> 11) & 0x1f;
				dr = (dTemp >> 11) & 0x1f;

				*((WORD*)lpDest) = (WORD)((ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 11);

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				sb = sTemp & 0x1f;
				db = dTemp & 0x1f;
				sg = (sTemp >> 5) & 0x3f;
				dg = (dTemp >> 5) & 0x3f;
				sr = (sTemp >> 11) & 0x1f;
				dr = (dTemp >> 11) & 0x1f;

				Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 11);

				sb = (sTemp >> 16) & 0x1f;
				db = (dTemp >> 16) & 0x1f;
				sg = (sTemp >> 21) & 0x3f;
				dg = (dTemp >> 21) & 0x3f;
				sr = (sTemp >> 27) & 0x1f;
				dr = (dTemp >> 27) & 0x1f;

				Result |= (DWORD)( (ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 11 ) << 16;

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite +=4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 24:
		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				sb = sTemp & 0xFF;
				db = dTemp & 0xFF;
				sg = (sTemp >> 8) & 0xFF;
				dg = (dTemp >> 8) & 0xFF;
				sr = (sTemp >> 16) & 0xFF;
				dr = (dTemp >> 16) & 0xFF;

				Result = (DWORD)(ALPHA * (db - sb) >> 8) + sb |
					(DWORD)((ALPHA * (dg - sg) >> 8) + sg) << 8 |
					(DWORD)((ALPHA * (dr - sr) >> 8) + sr) << 16;

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);

				lpDest++;
				lpSprite += 3;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 32:
		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				sb = sTemp & 0xFF;
				db = dTemp & 0xFF;
				sg = (sTemp >> 8) & 0xFF;
				dg = (dTemp >> 8) & 0xFF;
				sr = (sTemp >> 16) & 0xFF;
				dr = (dTemp >> 16) & 0xFF;

				Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb) |
					(DWORD)((ALPHA * (dg - sg) >> 8) + sg) << 8 |
					(DWORD)((ALPHA * (dr - sr) >> 8) + sr) << 16;

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);

				lpDest += 2;
				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


	UnLock();
	dest->UnLock();

	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_TransAlpha - 
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransAlpha(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD ALPHA)
{
	int register i,j;
	int height,width;
	BYTE* lpSprite;
	BYTE* lpDest;
	unsigned long dPitch, SpritePitch;
	DWORD sColorKey;
	DWORD sTemp,dTemp;
	unsigned long sr,sg,sb,dr,dg,db;
	WORD sbuf,dbuf;
	DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	// Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left; 

    // Get the color key for sprite surface
    sColorKey = (DWORD)m_ColorKey;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

	// Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;


	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;
		
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					sb = sTemp & 0x1f;
					db = dTemp & 0x1f;
					sg = (sTemp >> 5) & 0x1f;
					dg = (dTemp >> 5) & 0x1f;
					sr = (sTemp >> 10) & 0x1f;
					dr = (dTemp >> 10) & 0x1f;

					*((WORD*)lpDest) = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
						((ALPHA * (dg - sg) >> 8) + sg) << 5 |
						((ALPHA * (dr - sr) >> 8) + sr) << 10);
				}

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != ( (DWORD)sColorKey | ((DWORD)sColorKey << 16)) )
				{
					dTemp = *((DWORD*)lpDest);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						sb = sTemp & 0x1f;
						db = dTemp & 0x1f;
						sg = (sTemp >> 5) & 0x1f;
						dg = (dTemp >> 5) & 0x1f;
						sr = (sTemp >> 10) & 0x1f;
						dr = (dTemp >> 10) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)((ALPHA * (db - sb) >> 8) + sb |
							((ALPHA * (dg - sg) >> 8) + sg) << 5 |
							((ALPHA * (dr - sr) >> 8) + sr) << 10);
					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						sb = (sTemp >> 16) & 0x1f;
						db = (dTemp >> 16) & 0x1f;
						sg = (sTemp >> 21) & 0x1f;
						dg = (dTemp >> 21) & 0x1f;
						sr = (sTemp >> 26) & 0x1f;
						dr = (dTemp >> 26) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( (ALPHA * (db - sb) >> 8) + sb |
							((ALPHA * (dg - sg) >> 8) + sg) << 5 |
							((ALPHA * (dr - sr) >> 8) + sr) << 10 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);

		break;

	case 16:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					sb = sTemp & 0x1f;
					db = dTemp & 0x1f;
					sg = (sTemp >> 5) & 0x3f;
					dg = (dTemp >> 5) & 0x3f;
					sr = (sTemp >> 11) & 0x1f;
					dr = (dTemp >> 11) & 0x1f;

					*((WORD*)lpDest) = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
						((ALPHA * (dg - sg) >> 8) + sg) << 5 |
						((ALPHA * (dr - sr) >> 8) + sr) << 11);
				}

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != ( (DWORD)sColorKey | ((DWORD)sColorKey << 16)) )
				{
					dTemp = *((DWORD*)lpDest);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						sb = sTemp & 0x1f;
						db = dTemp & 0x1f;
						sg = (sTemp >> 5) & 0x3f;
						dg = (dTemp >> 5) & 0x3f;
						sr = (sTemp >> 11) & 0x1f;
						dr = (dTemp >> 11) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)((ALPHA * (db - sb) >> 8) + sb |
							((ALPHA * (dg - sg) >> 8) + sg) << 5 |
							((ALPHA * (dr - sr) >> 8) + sr) << 11);
					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						sb = (sTemp >> 16) & 0x1f;
						db = (dTemp >> 16) & 0x1f;
						sg = (sTemp >> 21) & 0x3f;
						dg = (dTemp >> 21) & 0x3f;
						sr = (sTemp >> 27) & 0x1f;
						dr = (dTemp >> 27) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( (ALPHA * (db - sb) >> 8) + sb |
							((ALPHA * (dg - sg) >> 8) + sg) << 5 |
							((ALPHA * (dr - sr) >> 8) + sr) << 11 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest += 4;
				lpSprite +=4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);

		break;

	case 24:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					sb = sTemp & 0xFF;
					db = dTemp & 0xFF;
					sg = (sTemp >> 8) & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					sr = (sTemp >> 16) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
						((ALPHA * (dg - sg) >> 8) + sg) << 8 |
						((ALPHA * (dr - sr) >> 8) + sr) << 16);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest++;
				}
				else
				{
					lpDest += 3;
				}

				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);
		break;

	case 32:
		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					sb = sTemp & 0xFF;
					db = dTemp & 0xFF;
					sg = (sTemp >> 8) & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					sr = (sTemp >> 16) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
						((ALPHA * (dg - sg) >> 8) + sg) << 8 |
						((ALPHA * (dr - sr) >> 8) + sr) << 16);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest += 2;
				}
				else
				{
					lpDest += 4;
				}

				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);
		break;
	} // End RGB Format switch statement


	UnLock();
	dest->UnLock();

	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_BlkShadow - 
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkShadow(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD SHADOW)
{
	int register i,j;
	int height,width;
	BYTE* lpSprite;
	BYTE* lpDest;
	unsigned long dPitch, SpritePitch;
	DWORD dTemp;
	unsigned long dr,dg,db;
	WORD sbuf,dbuf;
	DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	// Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left; 

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

	// Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;


	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;
		
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				dTemp = *((WORD*)lpDest);
				db = dTemp & 0x1f;
				dg = (dTemp >> 5) & 0x1f;
				dr = (dTemp >> 10) & 0x1f;

				*((WORD*)lpDest) = (WORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 10);

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);

				db = dTemp & 0x1f;
				dg = (dTemp >> 5) & 0x1f;
				dr = (dTemp >> 10) & 0x1f;

				Result = (DWORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 10);

				db = (dTemp >> 16) & 0x1f;
				dg = (dTemp >> 21) & 0x1f;
				dr = (dTemp >> 26) & 0x1f;

				Result |= (DWORD)( (SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 10 ) << 16;

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				dTemp = *((WORD*)lpDest);

				db = dTemp & 0x1f;
				dg = (dTemp >> 5) & 0x3f;
				dr = (dTemp >> 11) & 0x1f;

				*((WORD*)lpDest) = (WORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 11);

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);

				db = dTemp & 0x1f;
				dg = (dTemp >> 5) & 0x3f;
				dr = (dTemp >> 11) & 0x1f;

				Result = (DWORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 11);

				db = (dTemp >> 16) & 0x1f;
				dg = (dTemp >> 21) & 0x3f;
				dr = (dTemp >> 27) & 0x1f;

				Result |= (DWORD)( (SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 11 ) << 16;

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite +=4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 24:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);

				db = dTemp & 0xFF;
				dg = (dTemp >> 8) & 0xFF;
				dr = (dTemp >> 16) & 0xFF;

				Result = (DWORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 8 |
					((SHADOW * dr) >> 8) << 16);

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);
				lpDest++;

				lpSprite += 3;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 32:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);

				db = dTemp & 0xFF;
				dg = (dTemp >> 8) & 0xFF;
				dr = (dTemp >> 16) & 0xFF;

				Result = (DWORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 8 |
					((SHADOW * dr) >> 8) << 16);

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);
				lpDest += 2;

				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


	UnLock();
	dest->UnLock();

	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXBLT_TransShadow - 
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransShadow(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD SHADOW)
{
	int register i,j;
	int height,width;
	BYTE* lpSprite;
	BYTE* lpDest;
	unsigned long dPitch, SpritePitch;
	DWORD sColorKey;
	DWORD sTemp,dTemp;
	unsigned long dr,dg,db;
	WORD sbuf,dbuf;
	DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	// Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left; 

    // Get the color key for sprite surface
    sColorKey = (DWORD)m_ColorKey;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

	// Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;


	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;
		
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					db = dTemp & 0x1f;
					dg = (dTemp >> 5) & 0x1f;
					dr = (dTemp >> 10) & 0x1f;

					*((WORD*)lpDest) = (DWORD)( (SHADOW * db) >> 8 |
						((SHADOW * dg) >> 8) << 5 |
						((SHADOW * dr) >> 8) << 10 );
				}

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != ( (DWORD)sColorKey | ((DWORD)sColorKey << 16)) )
				{
					dTemp = *((DWORD*)lpDest);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						db = dTemp & 0x1f;
						dg = (dTemp >> 5) & 0x1f;
						dr = (dTemp >> 10) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)( (SHADOW * db) >> 8 |
							((SHADOW * dg) >> 8) << 5 |
							((SHADOW * dr) >> 8) << 10 );
					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						db = (dTemp >> 16) & 0x1f;
						dg = (dTemp >> 21) & 0x1f;
						dr = (dTemp >> 26) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( (SHADOW * db) >> 8 |
							((SHADOW * dg) >> 8) << 5 |
							((SHADOW * dr) >> 8) << 10 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);

		break;

	case 16:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					db = dTemp & 0x1f;
					dg = (dTemp >> 5) & 0x3f;
					dr = (dTemp >> 11) & 0x1f;

					*((WORD*)lpDest) = (DWORD)( (SHADOW * db) >> 8 |
						((SHADOW * dg) >> 8) << 5 |
						((SHADOW * dr) >> 8) << 11 );
				}

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != ( (DWORD)sColorKey | ((DWORD)sColorKey << 16)) )
				{
					dTemp = *((DWORD*)lpDest);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						db = dTemp & 0x1f;
						dg = (dTemp >> 5) & 0x3f;
						dr = (dTemp >> 11) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)( (SHADOW * db) >> 8 |
							((SHADOW * dg) >> 8) << 5 |
							((SHADOW * dr) >> 8) << 11 );
					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						db = (dTemp >> 16) & 0x1f;
						dg = (dTemp >> 21) & 0x3f;
						dr = (dTemp >> 27) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( (SHADOW * db) >> 8 |
							((SHADOW * dg) >> 8) << 5 |
							((SHADOW * dr) >> 8) << 11 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest += 4;
				lpSprite +=4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);

		break;

	case 24:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				
				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);

					db = dTemp & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)( (SHADOW * db) >> 8 |
						((SHADOW * dg) >> 8) << 8 |
						((SHADOW * dr) >> 8) << 16 );

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest++;
				}
				else
				{
					lpDest += 3;
				}
				
				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);
		break;

	case 32:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				
				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);

					db = dTemp & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)( (SHADOW * db) >> 8 |
						((SHADOW * dg) >> 8) << 8 |
						((SHADOW * dr) >> 8) << 16 );

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest += 2;
				}
				else
				{
					lpDest += 4;
				}
				
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);
		break;
	} // End RGB Format switch statement


	UnLock();
	dest->UnLock();

	return DD_OK;

}

////////////////////////////////////////////////////////////////////////
// CDXBLT_BlkShadowFast - 
////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkShadowFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect)
{
    int register i,j;
	int height,width;
    BYTE* lpSprite;
    BYTE* lpDest;
    unsigned long dPitch, SpritePitch;
    DWORD dTemp;
    WORD sbuf,dbuf;
    DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

    // Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

    // Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of the surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;

	// Start RGB Format switch statement
	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				dTemp = *((WORD*)lpDest);
				*((WORD*)lpDest) = (dTemp & 0x7BDE) >> 1;
				lpDest += 2;
			}

			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);
				Result = (dTemp & 0x7BDE) >> 1;
				Result |= (dTemp & 0x7BDE0000) >> 1;
				*((DWORD*)lpDest) = Result;
				lpDest += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				dTemp = *((WORD*)lpDest);
				*((WORD*)lpDest) = (dTemp & 0xF7DE) >> 1;
				lpDest += 2;
			}

			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);
				Result = (dTemp & 0xF7DE) >> 1;
				Result |= (dTemp & 0xF7DE0000) >> 1;
				*((DWORD*)lpDest) = Result;
				lpDest += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 24:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));
		i = height;
		do
		{
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);
				*((WORD*)lpDest) = (WORD)((dTemp & 0xFEFE) >> 1);
				lpDest += 2;
				*lpDest = (BYTE)(((dTemp >> 16) & 0xFE) >> 1);
				lpDest++;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 32:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));
		i = height;
		do
		{
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);
				*((WORD*)lpDest) = (WORD)((dTemp & 0xFEFE) >> 1);
				lpDest += 2;
				*lpDest = (BYTE)(((dTemp >> 16) & 0xFE) >> 1);
				lpDest += 2;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


    UnLock();
    dest->UnLock();

	return DD_OK;
}

////////////////////////////////////////////////////////////////////////
// CDXBLT_TransShadowFast 
////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransShadowFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect)
{
    int register i,j;
	int height,width;
    BYTE* lpSprite;
    BYTE* lpDest;
    unsigned long dPitch, SpritePitch;
    DWORD sColorKey;
    DWORD sTemp,dTemp;
    WORD sbuf,dbuf;
    DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

    // Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

    // Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of the surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;

	// Get the color key for sprite surface
	sColorKey = (DWORD)m_ColorKey;

	// Start RGB Format switch statement
	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					*((WORD*)lpDest) = (dTemp & 0x7BDE) >> 1;
				}

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);
				Result = dTemp;

				if ((sTemp & 0xFFFF) != sColorKey)
				{
					Result &= 0xFFFF0000;
					Result |= (dTemp & 0x7BDE) >> 1;
				}
				if ((sTemp >> 16) != sColorKey)
				{
					Result &= 0xFFFF;
					Result |= (dTemp & 0x7BDE0000) >> 1;
				}

				*((DWORD*)lpDest) = Result;
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					*((WORD*)lpDest) = (dTemp & 0xF7DE) >> 1;
				}

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);
				Result = dTemp;

				if ((sTemp & 0xFFFF) != sColorKey)
				{
					Result &= 0xFFFF0000;
					Result |= (dTemp & 0xF7DE) >> 1;
				}
				if ((sTemp >> 16) != sColorKey)
				{
					Result &= 0xFFFF;
					Result |= (dTemp & 0xF7DE0000) >> 1;
				}

				*((DWORD*)lpDest) = Result;
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 24:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				
				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);

					*((WORD*)lpDest) = (WORD)((dTemp & 0xFEFE) >> 1);
					lpDest += 2;
					*lpDest = (BYTE)(((dTemp >> 16) & 0xFE) >> 1);
					lpDest++;
				}
				else
				{
					lpDest += 3;
				}

				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 32:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				
				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);

					*((WORD*)lpDest) = (WORD)((dTemp & 0xFEFE) >> 1);
					lpDest += 2;
					*lpDest = (BYTE)(((dTemp >> 16) & 0xFE) >> 1);
					lpDest += 2;
				}
				else
				{
					lpDest += 4;
				}

				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


    UnLock();
    dest->UnLock();

	return DD_OK;
}

#pragma warning( pop )

//////////////////////////////////////////////////////////////////////////////////
// Copies the bitmap from the source surface to the surface pointed to by lpDDest
// Uses the SrcRect and DestRect structures to position the bitmap.  You can shrink 
// and stretch surfaces with this method.
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Draw(CDXSurface* lpDDest)
{
	HRESULT rval;

	rval = lpDDest->m_lpDDS->Blt(&DestRect, m_lpDDS, &SrcRect, DDBLT_WAIT, NULL);

	if(rval == DDERR_SURFACELOST) 
		Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// Copies the bitmap from the source surface to the surface pointed to by lpDDest.
// Uses X and Y to position the bitmap.
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawFast(int X, int Y, CDXSurface* lpDDest)
{
	HRESULT rval;

	rval = lpDDest->m_lpDDS->BltFast(X, Y, m_lpDDS, &SrcRect, DDBLTFAST_WAIT);

	if(rval == DDERR_SURFACELOST) 
		Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// Copies the bitmap from the source surface to the surface pointed to by lpDDest.
// Uses X and Y to position the bitmap and the current source colour key for transparent blit.
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawTrans(int X, int Y, CDXSurface* lpDDest)
{
	HRESULT rval;

	rval = lpDDest->m_lpDDS->BltFast(X, Y, m_lpDDS, &SrcRect, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);
	
	if(rval == DDERR_SURFACELOST) 
		Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// Copies the bitmap from the source surface to the surface pointed to by lpDDest.
// Uses X and Y to position the bitmap and the current source colour key for 
// transparent blit. This function clips the bitmap (if the bitmap spils over 
// the edge of the screen it is cut off, otherwise DirectDraw will not complete 
// the operation). If you must clip the object to a certain portion of the screen 
// you must call Clip manaully.
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawClipped(int X, int Y, CDXSurface* lpDDest, LPRECT ClipRect)
{
	HRESULT rval;
	RECT ModSrc = SrcRect;
	Clip(&X, &Y, &ModSrc, ClipRect);

	rval = lpDDest->m_lpDDS->BltFast(X, Y, m_lpDDS, &ModSrc, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// DrawWindowed
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawWindowed(CDXSurface* lpDDest)
{
	HRESULT rval;
	RECT Window;
	GetClientRect((HWND)Screen->m_hWnd, &Window);
	ClientToScreen((HWND)Screen->m_hWnd, (LPPOINT)&Window);
	ClientToScreen((HWND)Screen->m_hWnd, (LPPOINT)&Window+1);

	rval = lpDDest->m_lpDDS->Blt(&Window, m_lpDDS, NULL, DDBLT_WAIT, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// DrawScaled
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawScaled(int X, int Y, float Factor, CDXSurface* lpDDS)
{
	HRESULT rval;

	DestRect.top = Y;
	DestRect.left = X;
	DestRect.bottom = Y + (int)(m_PixelHeight * Factor);
	DestRect.right = X + (int)(m_PixelWidth * Factor);

	rval = lpDDS->m_lpDDS->Blt(&DestRect, m_lpDDS, &SrcRect, DDBLT_WAIT | DDBLT_KEYSRC, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// DrawScaled
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawScaled(int X, int Y, int Width, int Height, CDXSurface* lpDDS)
{
	HRESULT rval;

	DestRect.top = Y;
	DestRect.left = X;
	DestRect.bottom = Y + (m_PixelHeight * Height);
	DestRect.right = X + (m_PixelWidth * Width);

	rval = lpDDS->m_lpDDS->Blt(&DestRect, m_lpDDS, &SrcRect, DDBLT_WAIT | DDBLT_KEYSRC, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

#define DEG2RAD(x) (x*(float)PI/(float)180)

//////////////////////////////////////////////////////////////////////////////////
// DrawRotated
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::DrawRotated(int X, int Y, double Angle, CDXSurface* Dest)
{
	// NOT IMPLEMENTED
}

//////////////////////////////////////////////////////////////////////////////////
// DrawHFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawHFlip(int X, int Y, CDXSurface* Dest)
{
	DDBLTFX ddBltFx;

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORLEFTRIGHT;
	return m_lpDDS->Blt(NULL, NULL, NULL, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
}

//////////////////////////////////////////////////////////////////////////////////
// DrawVFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawVFlip(int X, int Y, CDXSurface* Dest)
{
	DDBLTFX ddBltFx;

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORUPDOWN;
	return m_lpDDS->Blt(NULL, NULL, NULL, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
}

//////////////////////////////////////////////////////////////////////////////////
// Sets the destination RECT for a Draw operation.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetDest(int t, int l, int b, int r)
{
	DestRect.top	= t;
	DestRect.left	= l;
	DestRect.bottom = b;
	DestRect.right	= r;
}

//////////////////////////////////////////////////////////////////////////////////
// Sets the source for a Draw operation.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetSrc(int t, int l, int b, int r)
{
	SrcRect.top		= t;
	SrcRect.left	= l;
	SrcRect.bottom	= b;
	SrcRect.right	= r;
}

//////////////////////////////////////////////////////////////////////////////////
// Sets the source colour key. This colour will appear transparent when drawn to the screen.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetColorKey(DWORD col)
{
	DDCOLORKEY ddck;

	m_ColorKey = col;
	ddck.dwColorSpaceLowValue = col;
	ddck.dwColorSpaceHighValue = col;

	m_lpDDS->SetColorKey(DDCKEY_SRCBLT, &ddck);
}

//////////////////////////////////////////////////////////////////////////////////
// Sets the source colour key. The color of the pixel in the top, left corner
// will appear transparent when drawn to the screen.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetColorKey(void)
{
	DDCOLORKEY		ddck;
	DWORD			dw = CLR_INVALID;
#if DIRECTDRAW_VERSION >= CDX_DDVER
	DDSURFACEDESC2 ddsd;
#else
	DDSURFACEDESC ddsd;
#endif
	HRESULT			hres;

	ddsd.dwSize = sizeof(ddsd);
	while((hres = m_lpDDS->Lock(NULL, &ddsd, 0, NULL)) == DDERR_WASSTILLDRAWING);

	if(SUCCEEDED(hres)) {
		dw = *(DWORD *)ddsd.lpSurface;
		if(ddsd.ddpfPixelFormat.dwRGBBitCount < 32)
			dw &= (1 << ddsd.ddpfPixelFormat.dwRGBBitCount) - 1;
		m_lpDDS->Unlock(NULL);
	}

	m_ColorKey = dw;
	ddck.dwColorSpaceLowValue = dw;
	ddck.dwColorSpaceHighValue = dw;

	m_lpDDS->SetColorKey(DDCKEY_SRCBLT, &ddck);
}

//////////////////////////////////////////////////////////////////////////////////
// Restores the DirectDrawSurface and bitmap if lost. Called internally if a Draw 
// function fails.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Restore(void)
{
	m_lpDDS->Restore();
	DDReLoadBitmap(m_lpDDS, m_pFilename);
}

//////////////////////////////////////////////////////////////////////////////////
// Locks the surface and provides direct access to the display memory. Should be 
// used before calling graphic functions such as PutPixel, Rect and Line. Remember 
// to call UnLock when you have finished drawing to the display memory. This
// method will properly restore any lost surfaces when called. This way, ALPHA
// and SHADOW functions can be safely used with surfaces in video memory.
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Lock(void)
{
	HRESULT rval;
	
	ZeroMemory(&m_DDSD, sizeof(m_DDSD));
	m_DDSD.dwSize = sizeof(m_DDSD);
	rval = m_lpDDS->Lock(NULL, &m_DDSD, DDLOCK_WAIT, NULL);
	while(rval == DDERR_SURFACELOST) {
		Restore();
		ZeroMemory(&m_DDSD, sizeof(m_DDSD));
		m_DDSD.dwSize = sizeof(m_DDSD);
		rval = m_lpDDS->Lock(NULL, &m_DDSD, DDLOCK_WAIT, NULL);
	}

	return(rval);
}

//////////////////////////////////////////////////////////////////////////////////
// UnLocks the surface and prevents access to display memory. Should be called 
// after you have finished with a Lock function. This method will properly restore 
// any lost surfaces when called. This way, ALPHA and SHADOW functions can be 
// safely used with surfaces in video memory.
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::UnLock(void)
{
	HRESULT rval;
	
	rval = m_lpDDS->Unlock(NULL);
	while(rval == DDERR_SURFACELOST) {
		Restore();
		rval = m_lpDDS->Unlock(NULL);
	}

	return(rval);
}

//////////////////////////////////////////////////////////////////////////////////
// Retieves a handle of the display device context (DC) and stores it in the m_DC 
// data member. The device context is used in GDI functions to draw to the screen.  
// Also used by the TextXY function.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::GetDC()
{
	m_lpDDS->GetDC(&m_DC);
}

//////////////////////////////////////////////////////////////////////////////////
// Releases the device context. Should be called after you have finished with a 
// GetDC function.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::ReleaseDC()
{
	m_lpDDS->ReleaseDC(m_DC);
}

//////////////////////////////////////////////////////////////////////////////////
// Chooses the font to be used by the TextXY function. The FontName string should 
// be a default Windows font name (HINT: look in your fonts directory to find a 
// list of font names eg "Comic Sans MS"). The Width and Height specify the width 
// and height of a single character in pixels. The Attributes member specifies the 
// weight of the font in the range 0 through 1000. For example, 400 is normal and 700 is bold.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::ChangeFont(const char* FontName, int Width, int Height, int Attributes)
{
	if (m_Font != NULL)
		DeleteObject(m_Font);

	m_Font = CreateFont(Height, Width,
		0, 0,
		Attributes,
		FALSE,
		FALSE,
		FALSE,
		ANSI_CHARSET,
		OUT_DEFAULT_PRECIS,
		CLIP_DEFAULT_PRECIS,
		NONANTIALIASED_QUALITY,
		VARIABLE_PITCH,
		FontName);
}

//////////////////////////////////////////////////////////////////////////////////
// Selects the the currently chosen font from ChangeFont. Remember to call the 
// GetDC function before attempting to select the font.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetFont(void)
{
	SelectObject(m_DC, m_Font);
}

//////////////////////////////////////////////////////////////////////////////////
// Draws the text specified by pString at position X,Y in colour Col. The text 
// background is transparent.  Remember to call the GetDC function before 
// attempting to draw text.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::TextXY(int x, int y, COLORREF col, LPCTSTR pString)
{
	SetBkMode(m_DC, TRANSPARENT);
	SetTextColor(m_DC, col);
	::TextOut(m_DC, x, y, pString, strlen(pString));
}

//////////////////////////////////////////////////////////////////////////////////
// Draw word wrapped text on a device context inside a user defined RECT with
// formatting dimensions.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::DrawText(LPCSTR pString, COLORREF col, LPRECT pRect)
{
	SetBkMode(m_DC, TRANSPARENT);
	SetTextColor(m_DC, col);
	
	// call Win32 function
	::DrawText(m_DC, pString, strlen(pString), pRect, DT_WORDBREAK);
}

//////////////////////////////////////////////////////////////////////////////////
// Swaps two integers
//////////////////////////////////////////////////////////////////////////////////
void Swap(int *a, int *b)
{
	int Temp = *a;
	*a = *b;
	*b = Temp;
}

//////////////////////////////////////////////////////////////////////////////////
// Draws a single pixel to the surface at position X,Y in colour Col. Remember to 
// call Lock before calling this function.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::PutPixel(int X, int Y, DWORD Col)
{
    BYTE* Bitmap = (BYTE*)m_DDSD.lpSurface;
    int colorBitDepth = Screen->m_BPP;

	if (X < m_ClipRect.left || X > m_ClipRect.right)
		return;
	if (Y < m_ClipRect.top || Y > m_ClipRect.bottom)
		return;

    switch(colorBitDepth)
    {
    case 8:
			Bitmap += Y * m_DDSD.lPitch + X;
			*Bitmap = (BYTE)Col;
			break;
    case 15:
    case 16:
            Bitmap += Y * m_DDSD.lPitch + X * 2;
            *((WORD*)Bitmap) = (WORD)Col;
            break;
    case 24:
            Bitmap += Y * m_DDSD.lPitch + X * 3;
            *((WORD*)Bitmap) = (WORD)(Col & 0xFFFF);
            Bitmap += 2;
            *Bitmap = (BYTE)((Col >> 16) & 0xFF);
            break;
    case 32:
            Bitmap += Y * m_DDSD.lPitch + X * 4;
			// This would be faster but I'm not sure if it is correct?
			//*((DWORD*)Bitmap) = Col; 
            *((WORD*)Bitmap) = (WORD)(Col & 0xFFFF);
            Bitmap += 2;
            *Bitmap = (BYTE)((Col >> 16) & 0xFF);
            break;
    default:
            break;
    }
}

//////////////////////////////////////////////////////////////////////////////////
// Returns the value of the pixel at position X,Y on the surface. Remember to call 
// Lock before calling this function.
//////////////////////////////////////////////////////////////////////////////////
DWORD CDXSurface::GetPixel(int X, int Y)
{
	DWORD Pixel;
	BYTE* Bitmap = (BYTE*)m_DDSD.lpSurface;

    switch(Screen->m_BPP)
    {
    case 8:
            Bitmap += Y * m_DDSD.lPitch + X;
			Pixel = (DWORD)(*Bitmap);
            return Pixel;
            break;
    case 15:
    case 16:
            Bitmap += Y * m_DDSD.lPitch + X * 2;
			Pixel = (DWORD)(*((WORD*)Bitmap));
            return Pixel;
            break;
    case 24:
            Bitmap += Y * m_DDSD.lPitch + X * 3;
            Pixel = (DWORD)(*((WORD*)Bitmap));
            Bitmap += 2;
			Pixel |= (DWORD)(*Bitmap) << 16;
            return Pixel;
            break;
    case 32:
            Bitmap += Y * m_DDSD.lPitch + X * 4;
            Pixel = *((DWORD*)Bitmap);
            return Pixel;
            break;
	default:
		return NULL;
    }
}

////////////////////////////////////////////////////////////////////////
// Draws a rectangle on the surface with a top-left coordinate at X1,Y1 
// and bottom-right coordinate at X2,Y2 in colour Col. Remember to call 
// Lock before calling this function.
////////////////////////////////////////////////////////////////////////
void CDXSurface::Rect(int X1, int Y1, int X2, int Y2, DWORD Col)
{
	HLine(X1,X2,Y1,Col);
	HLine(X1,X2,Y2,Col);

	VLine(Y1,Y2,X1,Col);
	VLine(Y1,Y2,X2,Col);
}

////////////////////////////////////////////////////////////////////////
// Draws a filled rectangle on the surface with a top-left coordinate at 
// X1,Y1 and bottom-right coordinate at X2,Y2 in colour Col. Remember to call 
// Lock before calling this function.
////////////////////////////////////////////////////////////////////////
void CDXSurface::FillRect(int X1, int Y1, int X2, int Y2, DWORD Col)
{
	DDBLTFX ddbltfx;
	RECT Rect = { X1, Y1, X2, Y2 };
	
	ddbltfx.dwSize = sizeof(ddbltfx);
	ddbltfx.dwFillColor = Col;

	m_lpDDS->Blt(&Rect, NULL, NULL, DDBLT_WAIT | DDBLT_COLORFILL, &ddbltfx);
}

//////////////////////////////////////////////////////////////////////////////////
// Draws a straight line on the surface from X1,Y1 to X2,Y2 in colour Col. Remember 
// to call Lock before calling this function.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Line(int X1, int Y1, int X2, int Y2, DWORD Col)
{
	double xStep, yStep, X, Y;
	int xLength, yLength, xCount, yCount;

	// If the line is horizontal or vertical use the fast version.
	if (X1 == X2)
	{
		VLine(Y1,Y2,X1,Col);
		return;
	}
	else if (Y1 == Y2)
	{
		HLine(X1,X2,Y1,Col);
		return;
	}

	xLength = abs(X2 - X1);
	yLength = abs(Y2 - Y1);

	if(xLength == 0) VLine(Y1, Y2, X1, Col);
	else if(yLength == 0) HLine(X1, X2, Y1, Col);

	else if(xLength > yLength)
	{
		if(X1 > X2)
		{
			Swap(&X1, &X2);
			Swap(&Y1, &Y2);
		}

		yStep = (double)(Y2 - Y1) / (double)(X2 - X1);
		Y = Y1;

		for(xCount = X1; xCount <= X2; xCount++)
		{
			PutPixel(xCount, (int)Y, Col);
			Y += yStep;
		}
	}
	else
	{
		if(Y1 > Y2)
		{
			Swap(&X1, &X2);
			Swap(&Y1, &Y2);
		}

		xStep = (double)(X2 - X1) / (double)(Y2 - Y1);
		X = X1;

		for(yCount = Y1; yCount <= Y2; yCount++)
		{
			PutPixel((int)X, yCount, Col);
			X += xStep;
		}
	}
}

////////////////////////////////////////////////////////////////////////
// Draws a vertical line on the surface from Y1 to Y2 at X in colour Col.
// Remember to call Lock before calling this function.
////////////////////////////////////////////////////////////////////////
void CDXSurface::VLine(int Y1, int Y2, int X, DWORD Col)
{
	// Notice: It is assumed that the surface will be locked prior
	//         to calling this function.  This is done for speed.
    int Length,top,bottom;
	DWORD dwCol;
    BYTE *Pixel = (BYTE*)m_DDSD.lpSurface;

	// Do some quick bounds checking
    if (X < m_ClipRect.left)
        return;

    if (X > m_ClipRect.right)
        return;


	// Determine which is top and bottom by the values
    if (Y1 > Y2)
    {
        top = Y2;
        bottom = Y1 + 1;
    }
    else
    {
        top = Y1;
        bottom = Y2 + 1;
    }

    // Clip the line
    if (top < m_ClipRect.top)
        top = m_ClipRect.top;

    if (bottom > m_ClipRect.bottom)
        bottom = m_ClipRect.bottom;

    Pixel += top * m_DDSD.lPitch;
    Length = bottom - top;

    // Draw the line

	switch (Screen->m_BPP)
	{
	case 8:
		Pixel += X;
		do
		{
			*Pixel = (BYTE)Col;
			Pixel += m_DDSD.lPitch;
			Length--;
		}while (Length > 0);
		break;

	case 15:
	case 16:

		Pixel += X << 1;

		do
		{
			*((WORD*)Pixel) = (WORD)Col;
			Pixel += m_DDSD.lPitch;
			Length--;
		}while(Length > 0);

		break;

	case 24:

		Pixel += X + X + X;

		dwCol = Col & 0xFFFF;
		Col = (Col >> 16) & 0xFF;
		do
		{
			*((WORD*)Pixel) = (WORD)dwCol;
			Pixel += 2;
			*Pixel = (BYTE)Col;
			Pixel += m_DDSD.lPitch - 2;
			Length--;
		}while(Length > 0);

		break;

	case 32:

		Pixel += X << 2;

		dwCol = Col & 0xFFFF;
		Col = (Col >> 16) & 0xFF;
		do
		{
			// This would be faster but I'm not sure if it is correct?
			//*((DWORD*)Pixel) = dwCol;
			//Pixel += m_DDSD.lPitch;
			*((WORD*)Pixel) = (WORD)dwCol;
			Pixel += 2;
			*Pixel = (BYTE)Col;
			Pixel += m_DDSD.lPitch - 2;
			Length--;
		}while(Length > 0);

		break;
	}
}

////////////////////////////////////////////////////////////////////////
// Draws a horizontal line on the surface from X1 to X2 at Y in colour Col.
// Remember to call Lock before calling this function.
////////////////////////////////////////////////////////////////////////
void CDXSurface::HLine(int X1, int X2, int Y, DWORD Col)
{
	// Notice: It is assumed that the surface will be locked prior
	//         to calling this function.  This is done for speed.
    int Length,left,right;
	int i;
    DWORD dwCol;
    BYTE *Pixel = (BYTE*)m_DDSD.lpSurface;

	// Do some quick bounds checking
    if (Y < m_ClipRect.top)
        return;

    if (Y > m_ClipRect.bottom)
        return;


	// Determine which is left and right by value
    if (X1 > X2)
    {
        left = X2;
        right = X1 + 1;
    }
    else
    {
        left = X1;
        right = X2 + 1;
    }

    // Clip the line
    if (left < m_ClipRect.left)
            left = m_ClipRect.left;

    if (right > m_ClipRect.right)
            right = m_ClipRect.right;

	// Calculate the length of the line
    Length = right - left;
    Pixel += Y * m_DDSD.lPitch;

	switch (Screen->m_BPP)
	{
	case 8:
		Pixel += left;
		i = Length % 4;
		Length -= i;
		for (i; i > 0; i--)
		{
			*Pixel = (BYTE)Col;
			Pixel++;
		}
		if (Length > 3)
		{
			dwCol = Col | (Col << 8) | (Col << 16) | (Col << 24);
			do
			{
				*((DWORD*)Pixel) = dwCol;
				Pixel += 4;
				Length -= 4;
			}while(Length > 0);
		}
		break;

	case 15:
	case 16:

		Pixel += left << 1;
		i = Length % 2;
		Length -= i;
		for (i; i > 0; i--)
		{
			*((WORD*)Pixel) = (WORD)Col;
			Pixel += 2;
		}
		if (Length > 1)
		{
			dwCol = Col | (Col << 16);
			do
			{
				*((DWORD*)Pixel) = dwCol;
				Pixel += 4;
				Length -= 2;
			}while(Length > 0);
		}

		break;

	case 24:

		Pixel += left + left + left;

		dwCol = Col & 0xFFFF;
		Col = (Col >> 16) & 0xFF;
		do
		{
			*((WORD*)Pixel) = (WORD)dwCol;
			Pixel += 2;
			*Pixel = (BYTE)Col;
			Pixel++;
			Length--;
		}while(Length > 0);

		break;

	case 32:

		Pixel += left << 2;

		dwCol = Col & 0xFFFF;
		Col = (Col >> 16) & 0xFF;
		do
		{
			// This would be faster but I'm not sure if it is correct?
			//*((DWORD*)Pixel) = dwCol;
			//Pixel += m_DDSD.lPitch + 4;
			*((WORD*)Pixel) = (WORD)dwCol;
			Pixel += 2;
			*Pixel = (BYTE)Col;
			Pixel += 2;
			Length--;
		}while(Length > 0);

		break;
	}
}
//////////////////////////////////////////////////////////////////////////////////
// Draws a circle on the surface at position X,Y with Radius in colour Col.
// Remember to call Lock before calling this function.
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Circle(int X, int Y, int Radius, DWORD Col)
{
	// Thank Micheal Abrash for this routine.
	int majorAxis, minorAxis;
	double radiusSqrd = (double)Radius * Radius;

	majorAxis = 0;
	minorAxis = Radius;

	do
	{
		PutPixel(X+majorAxis, Y-minorAxis, Col);
		PutPixel(X-majorAxis, Y-minorAxis, Col);
		PutPixel(X+majorAxis, Y+minorAxis, Col);
		PutPixel(X-majorAxis, Y+minorAxis, Col);
		PutPixel(X+minorAxis, Y-majorAxis, Col);
		PutPixel(X-minorAxis, Y-majorAxis, Col);
		PutPixel(X+minorAxis, Y+majorAxis, Col);
		PutPixel(X-minorAxis, Y+majorAxis, Col);

		majorAxis++;
		minorAxis = (int)(sqrt(radiusSqrd - ((double)majorAxis * majorAxis)) + 0.5);

	}while (majorAxis <= minorAxis);

}
////////////////////////////////////////////////////////////////////////
// Draws a filled circle on the surface at position X,Y with Radius in colour Col.
// Remember to call Lock before calling this function.
////////////////////////////////////////////////////////////////////////
void CDXSurface::FillCircle(int X, int Y, int Radius, DWORD Col)
{
    int Y1,dx;
    int top, bot;
	double R;

    R = (double)Radius * Radius;

	// Perform a little bounds checking
    if ((top = Y - Radius) < m_ClipRect.top)
        top = m_ClipRect.top;
    else if (top > m_ClipRect.bottom)
        return;

    if ((bot = Y + Radius) > m_ClipRect.bottom)
        bot = m_ClipRect.bottom;
    else if (bot < m_ClipRect.top )
        return;

    for(Y1=top; Y1 < bot; Y1++)
    {
        dx = (int)(sqrt(R - ((double)(Y-Y1)*(Y-Y1))) + 0.5);
		HLine(X-dx, X+dx, Y1, Col);
    }
}

//////////////////////////////////////////////////////////////////////////////////
// Fills the surface with the specified colour
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Fill(DWORD FillColor)
{
	DDBLTFX ddBltFx;

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwFillColor = FillColor;
	m_lpDDS->Blt(NULL, NULL, NULL, DDBLT_WAIT | DDBLT_COLORFILL, &ddBltFx);
}

//////////////////////////////////////////////////////////////////////////////////
// Clips a destination rectange and modifys X,Y coords appropriatly
//////////////////////////////////////////////////////////////////////////////////
void Clip(int *DestX, int *DestY, RECT *SrcRect, RECT *DestRect)
{
	// If it's partly off the right side of the screen
	if(*DestX + (SrcRect->right - SrcRect->left) > DestRect->right)
		SrcRect->right -= *DestX + (SrcRect->right-SrcRect->left) - DestRect->right;

	// Partly off the left side of the screen
	if(*DestX < DestRect->left)
	{
		SrcRect->left += DestRect->left - *DestX;
		*DestX = DestRect->left;
	}

	// Partly off the top of the screen
	if(*DestY < DestRect->top)
	{
		SrcRect->top += DestRect->top - *DestY;
		*DestY = DestRect->top;
	}

	// If it's partly off the bottom side of the screen
	if(*DestY + (SrcRect->bottom - SrcRect->top) > DestRect->bottom)
	SrcRect->bottom -= ((SrcRect->bottom-SrcRect->top)+*DestY) - DestRect->bottom;

	return;
}
