/******************************************
GLUTRAD
Copyright, 1998, Colbeck Desktop Solutions Ltd.
contact: colbeck@cix.co.uk
******************************************/
#include <windows.h>
#include <stdio.h>
#include <GL/glut.h>
#include <math.h>
#include "objects.h"

int res, hres;  // window/hemicube resolution and half-resolution

double cos_angle; // the cosine of the maximum allowable 'sharp' angle (non smoothed)
double brightness; // scaling factor for colour intensity
int step; // higher numbers may speed up convergence, but make the program less interactive

double near_plane, far_plane, camera_fov;
TVector camera_pos, camera_lookat, camera_up;

double convergence, converged; // stop processing when this limit is reached
double EnergyLimit; // total energy in scene
int count=0;

enum {STEP, AUTO, STOP, AALIAS, SAVE}; // menu ids
enum {LIT_MODEL=1, INDEX_MODEL}; // display lists

GLfloat viewanglex, viewanglez, orgviewanglex, orgviewanglez;
int startx, starty; // used for dragging

TVector lightpos; // details of the emmitting patch and the necessary vectors for the hemicube camera
TVector lightdir;
TVector lookat;
TVector lightup;
double lightarea;
TColour emission;
int emmitter, patch; // which object and which triangle are emmiting

TObject* object; // array of objects
int ObjectCount;

double *top_ff, *side_ff; // heaps of memory for the pre-computed form factor table
GLubyte *Pixels; // the display buffer

void RenderLitModel(); // some forward declarations of functions
void Radiate();
extern void AliasPerspective(int model); // in jitter.cpp

// Radiosity functions /////////////////////////////////////////////////////////////////////////
void Step() // the main radiosity step
{
	Radiate();	 // distribute the emmitter's unshot energy
	object[emmitter].T[patch]->unshot.Set(0.0, 0.0, 0.0); // reset the unshot energy	
	if(count%step==0)
	{
		printf("Convergence %2.5f\t\tStep %d\n", convergence, count);
		RenderLitModel(); // update the view
		glutPostRedisplay();
	}
	count++;
}

void HemicubeFaceCam(double* ff, int yres) // put a camera where the light is and point it at a hemicube face
{
	glMatrixMode(GL_PROJECTION); // position the camera
	glLoadIdentity();
	gluPerspective(90.0, 1, near_plane, far_plane);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(lightpos.x+lightdir.x/1000, lightpos.y+lightdir.y/1000, lightpos.z+lightdir.z/1000, 
			  lookat.x, lookat.y, lookat.z, 
			  lightup.x, lightup.y, lightup.z); 

	glCallList(INDEX_MODEL);	// draw light's view with numbered faces
	glPixelStorei(GL_PACK_ALIGNMENT, 1);
	glReadPixels(0, 0, res, res, GL_RGB, GL_UNSIGNED_BYTE, Pixels); // read the buffer
	int o, g, b, x, y, ix, iy, index, i=0;
	for(iy=0; iy<yres; iy++) // count coloured pixels
	{
		y = (iy<hres ? iy : hres-1-(iy%hres))*hres;
		for(ix=0; ix<res; ix++, i+=3)
		{
			b=Pixels[i+2];
			if(b) // if a colour is present...
			{	
				o=Pixels[i]; g=Pixels[i+1]; 
				index= (g<<8) + b -1;
				x = ix<hres ? ix : hres-1-(ix%hres);
				object[o].T[index]->ff += ff[y+x]; // ...add the delta formfactor to the associated patch
			}
		}
	}
}

void Radiate() // shoot a single patch's energy
{
	int o, p;
	double tot, clampff, max_emit=0;

	TVector vec, u, v; // u and v are tangent vectors to lightdir
	do // vec is randomly generated and so causes a random rotation of the hemicube around its normal (lightdir)
	{
		vec.x = (float)rand()/RAND_MAX; vec.y = (float)rand()/RAND_MAX; vec.z = (float)rand()/RAND_MAX;
		u=lightdir.cross(vec); // compute tangent vector 
	} 
	while(u.Normalise()==0.0);	
	v=lightdir.cross(u); // compute the other tangent

	lightup=u;					// calculate hemicube camera vectors
	lookat=lightpos+lightdir;
	HemicubeFaceCam(top_ff, res);	// do top face

	lightup = lightdir;	
	lookat=lightpos+u;
	HemicubeFaceCam(side_ff, hres);	// do side faces
	
	lookat=lightpos-u;
	HemicubeFaceCam(side_ff, hres);
	
	lookat=lightpos+v;
	HemicubeFaceCam(side_ff, hres);

	lookat=lightpos-v;
	HemicubeFaceCam(side_ff, hres);
	
	for(o=0; o<ObjectCount; o++)
	{
		for(p=0; p<object[o].TriCount; p++) // calculate each patch's unshot energy
		{
			clampff=min(object[o].T[p]->ff, 1.0); // clamp the form factor to 1
			object[o].T[p]->unshot.r += (emission.r * clampff * object[o].reflectance.r);
			object[o].T[p]->unshot.g += (emission.g * clampff * object[o].reflectance.g);
			object[o].T[p]->unshot.b += (emission.b * clampff * object[o].reflectance.b);
			object[o].T[p]->energy.r += object[o].T[p]->unshot.r; 
			object[o].T[p]->energy.g += object[o].T[p]->unshot.g; 
			object[o].T[p]->energy.b += object[o].T[p]->unshot.b;

			object[o].T[p]->ff=0.0; // set form factor to zero, ready for next iteration
			tot=object[o].T[p]->area*object[o].T[p]->unshot.Sum(); // find next potential emmitter
			if(tot>max_emit)
			{
				emmitter=o; patch=p;
				max_emit=tot;
			}
		}
	}
	convergence=max_emit/EnergyLimit;
	if(convergence<converged) // have we finished ?
		glutIdleFunc(0);
	lightpos=object[emmitter].T[patch]->centre; // set up the new emmitter
	lightdir=object[emmitter].T[patch]->N;
	lightarea=object[emmitter].T[patch]->area;
	emission.Set(object[emmitter].T[patch]->unshot);
}

// preparatory functions /////////////////////////////////////////////////////////////////
void RenderColourIndexModel() // to calculate what a light sees, place a camera at the light position and render the scene
{
	int o, i;
	int index=0; // each triangle has an index number for a colour
	DWORD r, g, b;
	glNewList(INDEX_MODEL, GL_COMPILE);
		glClearColor(0, 0, 0, 0); // clear background to maximum intensity
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glShadeModel(GL_FLAT);
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		glBegin(GL_TRIANGLES);
		for(o=0; o<ObjectCount; o++)
		{
			index=1;
			for(i=0; i<object[o].TriCount; i++)
			{
				r=o; g=index>>8; b=index-(g<<8); // derive r, g, b values from index
				glColor3f(float(r)/255, float(g)/255, float(b)/255);
				index++;
				glVertex3f(object[o].T[i]->V[0]->v.x, object[o].T[i]->V[0]->v.y, object[o].T[i]->V[0]->v.z);
				glVertex3f(object[o].T[i]->V[1]->v.x, object[o].T[i]->V[1]->v.y, object[o].T[i]->V[1]->v.z);
				glVertex3f(object[o].T[i]->V[2]->v.x, object[o].T[i]->V[2]->v.y, object[o].T[i]->V[2]->v.z);
			}
		}
		glEnd();
		glEndList();
}

void CalculateVertexColor(TColour& c, TVertex* v, TTriangle* tri) // called three thines for each triangle
{
	TTriangleLink *tl; // each vertex has a linked list of triangles to which it belongs
	c.Set(tri->energy); // set the vertex colour to be that of the triangle in question

	double dot; // we're going to calculate the angle between triangles that share this vertex...
	TVector n;  // ...if its a small angle, blend their colours at this vertex
	n=tri->N;

	int count=1;
	tl=v->user_head;
	while(tl)
	{
		if(tl->t!=tri)
		{
			dot = tl->t->N.dot(n); // find angle between the triangle normal and the neighbour triangle's normal
			if (dot > cos_angle) 
			{
				count++; // keep track of how many triangles were blending over
				c.r += tl->t->energy.r; c.g += tl->t->energy.g; c.b += tl->t->energy.b;
			}
		}
		tl=tl->next;
	}
	c.r/=count; c.g/=count; c.b/=count; // average the energy
	c.Scale(brightness); 
	c.ConvertToColour(); // transform energy value to a colour
}

void RenderLitModel()
{
	TColour c0, c1, c2;
	int o, i;
	glNewList(LIT_MODEL, GL_COMPILE);
		glShadeModel(GL_SMOOTH);
		glBegin(GL_TRIANGLES);
		for(o=0; o<ObjectCount; o++)
		{
			for(i=0; i<object[o].TriCount; i++)
			{
				// this is a bit computationally intensive, but we're compiling a display list so its only done once
				CalculateVertexColor(c0, object[o].T[i]->V[0], object[o].T[i]); // work out the colour of each vertex
				CalculateVertexColor(c1, object[o].T[i]->V[1], object[o].T[i]);
				CalculateVertexColor(c2, object[o].T[i]->V[2], object[o].T[i]);

				glColor3f(c0.r, c0.g, c0.b);
				glVertex3f(object[o].T[i]->V[0]->v.x, object[o].T[i]->V[0]->v.y, object[o].T[i]->V[0]->v.z);

				glColor3f(c1.r, c1.g, c1.b);
				glVertex3f(object[o].T[i]->V[1]->v.x, object[o].T[i]->V[1]->v.y, object[o].T[i]->V[1]->v.z);

				glColor3f(c2.r, c2.g, c2.b);
				glVertex3f(object[o].T[i]->V[2]->v.x, object[o].T[i]->V[2]->v.y, object[o].T[i]->V[2]->v.z);
			}
		}
		glEnd();
	glEndList();
}

void CalculateTopFF() // pre-compute form factor parameters for 1/4 the top face of a hemicube
{
	int ix, iy;
	double x, y;
	double* ptr;

	top_ff=new double[hres*hres]; // the top face is symmetrical, so only store one quarter
	ptr = top_ff;
	for(iy=0; iy<hres; iy++)
	{
		y=(hres-((double)iy+0.5))/hres;
		for(ix=0; ix<hres; ix++)
		{
			x=(hres-((double)ix+0.5))/hres;
			*ptr++ = 1.0 / (3.1415927 * (x*x+y*y+1.0)*(x*x+y*y+1.0) * hres*hres);
		}
	}
}

void CalculateSideFF() // pre-compute form factor parameters for 1/2 the size face of a hemicube
{
	int ix, iy;
	double x, y;
	double* ptr;

	side_ff=new double[hres*hres];
	ptr = side_ff;

	for(iy=0; iy<hres; iy++)
	{
		y=(hres-(double(iy)+0.5))/hres;
       	for (ix=0; ix<hres; ix++)
       	{
       		x=(hres-(double(ix)+0.5))/hres;
        	*ptr++ = y / (3.1415927 * (x*x+y*y+1.0)*(x*x+y*y+1.0) * hres*hres);
       	}
    }
}

// read a single object and its details...Windows only I'm afraid
void ReadObject(int index, const char* inifile)
{
	char section[20], entry[128];
	wsprintf(section, "object_%d", index);
	int tris=GetPrivateProfileInt(section, "triangles", 0, inifile);
	object[index].T=new TTriangle*[tris];
	object[index].V=new TVertex*[tris*3]; // over approximation, we really need a dynamic array (MFC?)

	GetPrivateProfileString(section, "file", "", entry, 128, inifile);
	printf("Opening: %s\n", entry);
	if(object[index].FileOpenRaw(entry)==0)
		printf("--------------- FAILED ---------------\n");
	object[index].reflectance.r=(double)GetPrivateProfileInt(section, "reflectance_r", 255, inifile)/255;
	object[index].reflectance.g=(double)GetPrivateProfileInt(section, "reflectance_g", 255, inifile)/255;
	object[index].reflectance.b=(double)GetPrivateProfileInt(section, "reflectance_b", 255, inifile)/255;

	object[index].emission.r=(double)GetPrivateProfileInt(section, "emission_r", 255, inifile)/255;
	object[index].emission.g=(double)GetPrivateProfileInt(section, "emission_g", 255, inifile)/255;
	object[index].emission.b=(double)GetPrivateProfileInt(section, "emission_b", 255, inifile)/255;
}

// read the settings and objects from the ini-file
void ReadIniFile(const char* inifile)
{
	printf("Reading: %s\n", inifile);
	char entry[128];
	res=GetPrivateProfileInt("settings", "resolution", 256, inifile);
	hres=res/2; // half res
	GetPrivateProfileString("settings", "brightness", "2.5", entry, 128, inifile);
	brightness=atof(entry);
	step=GetPrivateProfileInt("settings", "step", 2, inifile);
	GetPrivateProfileString("settings", "near", "0.1", entry, 128, inifile);
	near_plane=atof(entry);
	GetPrivateProfileString("settings", "far", "40.0", entry, 128, inifile);
	far_plane=atof(entry);
	double angle=GetPrivateProfileInt("settings", "smooth_angle", 2, inifile);
	cos_angle=cos(angle * 3.1415927 / 180.0); // the cosine of the maximum allowable 'sharp' angle (non smoothed)
	GetPrivateProfileString("settings", "stop_at", "0.0001", entry, 128, inifile);
	converged=atof(entry);

	GetPrivateProfileString("camera", "pos_x", "0.0", entry, 128, inifile);
	camera_pos.x=atof(entry);
	GetPrivateProfileString("camera", "pos_y", "-20.0", entry, 128, inifile);
	camera_pos.y=atof(entry);
	GetPrivateProfileString("camera", "pos_z", "0.0", entry, 128, inifile);
	camera_pos.z=atof(entry);

	GetPrivateProfileString("camera", "lookat_x", "0.0", entry, 128, inifile);
	camera_lookat.x=atof(entry);
	GetPrivateProfileString("camera", "lookat_y", "0.0", entry, 128, inifile);
	camera_lookat.y=atof(entry);
	GetPrivateProfileString("camera", "lookat_z", "0.0", entry, 128, inifile);
	camera_lookat.z=atof(entry);

	GetPrivateProfileString("camera", "up_x", "0.0", entry, 128, inifile);
	camera_up.x=atof(entry);
	GetPrivateProfileString("camera", "up_y", "0.0", entry, 128, inifile);
	camera_up.y=atof(entry);
	GetPrivateProfileString("camera", "up_z", "1.0", entry, 128, inifile);
	camera_up.z=atof(entry);

	GetPrivateProfileString("camera", "fov", "45.0", entry, 128, inifile);
	camera_fov=atof(entry);
	
	ObjectCount=GetPrivateProfileInt("settings", "objects", 0, inifile);
	object=new TObject[ObjectCount];
	for(int i=0; i<ObjectCount; i++)
		ReadObject(i, inifile);
}
// GLUT callbacks /////////////////////////////////////////////////////////////////////////
void redraw(void)
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glMatrixMode(GL_PROJECTION); // put camera in the correct position
	glLoadIdentity();
	gluPerspective(camera_fov, 1, near_plane, far_plane);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(camera_pos.x, camera_pos.y, camera_pos.z, camera_lookat.x, camera_lookat.y, camera_lookat.z, camera_up.x, camera_up.y, camera_up.z); 
	glRotatef(viewanglex, 1.f, 0.f, 0.f);
	glRotatef(viewanglez, 0.f, 0.f, 1.f);
	glCallList(LIT_MODEL);
	glutSwapBuffers();
}

void special(int key, int x, int y)
{
	switch (key) 
	{
	case GLUT_KEY_LEFT:
		viewanglez += 10.f; glutPostRedisplay(); break;
	case GLUT_KEY_RIGHT:
		viewanglez -= 10.f; glutPostRedisplay(); break;
	case GLUT_KEY_UP:
		viewanglex -= 10.f; glutPostRedisplay(); break;
	case GLUT_KEY_DOWN:
		viewanglex += 10.f; glutPostRedisplay(); break;
	break;
	}
}

void key(unsigned char key, int x, int y)
{
	switch (key) 
	{
	case ' ':
		Step(); 
		printf("Convergence %2.5f\t\tStep %d\n", convergence, count);
		RenderLitModel(); // update the view
		glutPostRedisplay();
		break;
	case '\033':
		exit(0);
	}
}

void mouse(int button, int state, int x, int y)
{
	if (state == GLUT_UP) 
	{
		viewanglex=orgviewanglex-(starty-y); viewanglez=orgviewanglez-(startx-x);
	} 
	else // GLUT_DOWN 
	{              
		startx = x; starty = y;
		orgviewanglex=viewanglex; orgviewanglez=viewanglez;
	}
}

void motion(int x, int y)
{
	viewanglex=orgviewanglex-(starty-y);
	viewanglez=orgviewanglez-(startx-x);
	glutPostRedisplay();
}

void menu(int sel)
{
	if(sel==AUTO)
		glutIdleFunc(Step);
	if(sel==STOP)
		glutIdleFunc(0);
	if(sel==STEP)
	{
		Step(); 
		printf("Convergence %2.5f\t\tStep %d\n", convergence, count);
		RenderLitModel(); // update the view
	}
	if(sel==AALIAS)
	{
		AliasPerspective(LIT_MODEL);
		glutSwapBuffers();
		return;
	}
	if(sel==SAVE)
	{
		for(int i=0; i<ObjectCount; i++)
			object[i].Save6d();
	}
	glutPostRedisplay();
}

void reshape(int w, int h)
{
// dummy to prevent the viewport from resizing
}

// start here /////////////////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	HDC dc=GetDC(0); // desktop dc
	int bpp=GetDeviceCaps(dc, BITSPIXEL);
	ReleaseDC(0, dc);
	if(bpp<24)
	{
		MessageBox(0, "Please change to 24 or 32 bit colour depth", "Warning", MB_TASKMODAL);
		return 0;
	}

	glutInit(&argc, argv);

	if(argc==1)
		ReadIniFile(".\\glutrad.ini");
	else
		ReadIniFile(argv[1]);

	glutInitWindowSize(res, res);
	glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE);
	glutCreateWindow("GLUT_RAD");
	glutDisplayFunc(redraw);
	glutKeyboardFunc(key);
	glutSpecialFunc(special);
	glutMouseFunc(mouse);
	glutMotionFunc(motion);
	glutIdleFunc(Step);
	glutReshapeFunc(reshape);

	glutCreateMenu(menu);
	glutAddMenuEntry("Auto", AUTO);
	glutAddMenuEntry("Stop", STOP);
	glutAddMenuEntry("Step", STEP);
	glutAddMenuEntry("Anti-Alias", AALIAS);
	glutAddMenuEntry("Save6d", SAVE);
	glutAttachMenu(GLUT_RIGHT_BUTTON);

	glEnable(GL_CULL_FACE);
	glEnable(GL_DEPTH_TEST);

	viewanglex=viewanglez=0;

	Pixels = new GLubyte[res * res * 4];
	CalculateTopFF();
	CalculateSideFF();
	
	RenderColourIndexModel();

	EnergyLimit=0.0;
	int o, p, emmitter, patch;
	double tot, max_emit=0;
	for(o=0; o<ObjectCount; o++) // calculate total energy in the scene and get ready to shoot it
	{
		for(p=0; p<object[o].TriCount; p++)
		{
			object[o].T[p]->emission.Set(object[o].emission);
			EnergyLimit+=object[o].T[p]->emission.Sum()*object[o].T[p]->area;
			object[o].T[p]->energy=object[o].T[p]->unshot=object[o].T[p]->emission;
			object[o].T[p]->ff=0.0; // set form factors to zero, ready for next iteration
			tot=object[o].T[p]->area*object[o].T[p]->unshot.Sum();
			if(tot>max_emit)
			{
				emmitter=o; patch=p;
				max_emit=tot;
			}
		}
	}
	lightpos=object[emmitter].T[patch]->centre; // set up the new emmitter
	lightdir=object[emmitter].T[patch]->N;
	lightarea=object[emmitter].T[patch]->area;
	emission.Set(object[emmitter].T[patch]->unshot);

	glutMainLoop(); // the action starts with the Step function above

	delete[] top_ff;
	delete[] side_ff;
	delete[] object;
	delete Pixels; // goodness, this program eats memory
	return 0; 
}
