/*------------------------------------------------------------------
TEST

Sample class to demonstrate DirectDraw ...

By: Jose Carlos Ramos - 1996
---------------------------------------------------------------------*/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <ddraw.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
LPDIRECTDRAWPALETTE lpDDPal;
// the MAPSHIFT constant controls the size of the map. The map dimentions
// are (2^MAPSHIFT)x(2^MAPSHIFT). So if you set MAPSHIFT equal to 8, you
// will get a 256x256 map.
#define  MAPSHIFT    9              // 2^MAPSHIFT = size of map

// Macros dependant on MAPSHIFT. They change with the size of the map. They
// should not be changed manually. 
#define  MAPSIZE     (1<<MAPSHIFT)           // size of one side of the map
#define  BYTE(a)     (a & (MAPSIZE-1))       // put the range within MAPSIZE
#define  INDEX(x,y)  ((BYTE(y) << MAPSHIFT) + BYTE(x)) // index calculation 
#define  fINDEX(x,y) (INDEX(x >> 8, y >> 8)) // fixed point 24.8
#define  fpINDEX(x,y) (INDEX(x >> 16, y >> 16)) // fixed point 16.16

// A couple of miscelanous constants
#define  NUMSMOOTH   3     // smooth the map 3 times
#define  FARDEPTH    80    // number of depth steps
#define  NEARDEPTH   0     // do not change this number....
#define  NUMSTEPS    FARDEPTH-NEARDEPTH
#define  HEIGHTMUL   25    // perspective multiplication factor
#define  SLOPE       1     // do not change this number....
#define RENDERWIDTH  320
#define RENDERHEIGHT 240
typedef struct tview    // a "view" on the voxel map
{                       // uses 3 points to define a view
  int x1, y1, x2, y2;   // a center or viewer position and a point to the
  int cx, cy;           // left and a point to the right. The side points are
  int z;
  int viewangle;
} tview;                // used to define the viewer rotation and where to
                        // trace on the height map

tview mainview;  // current view
#define  RANDOM(a)   (rand() % a) // a simple random function


typedef struct hrec     // struct to store a height and a color for a vertical
{                       // line for the rendering process.
    int y;
    int c;
} hrec;

typedef struct               // 4 byte division for union use
{
    char b1, b2, b3, b4;
} _bytes;

typedef union register32_8  // simulate a byte accessable 32-bit register
{
    int   dword;
     _bytes bytes; 
} register32_8;

typedef float matrix[3*3];

char *heightmap;          // height feild for the voxels
char *colormap;           // colors for the voxels
char *vpage;              // a virtual screen page
hrec *buf1;               // buffer for storing the "top" part
hrec *buf2;               // buffer for storing the "bottom" part
int   divtable[512];      // a 65536/x table
int   sine[256];          // sin lookup table
int   cosine[256];        // cos lookup table
int   rolltable[RENDERWIDTH];
matrix viewer;

void createmaps();
void createfractalmap(int x1, int y1, int x2, int y2);
char newcolor(int mc, int n, int dvd);
void cleanup();
void smoothmap();
void makecolormap();
void buildtables();
void drawscreen(tview *v);
void gouraudbuffers();
void makepalette();
void setrgb(char n, char r, char g, char b);
void setview(int speed, int viewangle, int moveangle, tview *v);
short int initmouse(); 
void readmickey(short int *x, short int *y);
void setrolltable(int x);
void drawmarkers(short int x, short int y);
void initmatrix(matrix a);



HINSTANCE gb_hInstance;
BOOL gb_bActive;
int gb_nCmdShow;

long FAR PASCAL WindowProc( HWND hWnd, UINT message,WPARAM wParam, LPARAM lParam );
HWND AppWnd;
#define NAME "TEST"
#define TITLE "** TEST DDraw **"
/*------------------------------------------------------------------

Class CUseDdraw - declaration
Shows some basic DirectDraw features

--------------------------------------------------------------------*/
class CUseDdraw
{
public:
static LPDIRECTDRAW lpDD;
static LPDIRECTDRAWSURFACE lpDDSPrimary;
static LPDIRECTDRAWSURFACE lpDDSBack;

BOOL Start();
void Finish();

HRESULT RestoreAll();

BOOL ok;

public:
CUseDdraw();
~CUseDdraw();

void Update();

BOOL Ok()
{
return ok;
}

DDSURFACEDESC LockBackBuf();
void UnLockBackBuf();
};

/*------------------------------------------------------------------

Class CUseDdraw - Implementation
Shows some basic DirectDraw features

--------------------------------------------------------------------*/

LPDIRECTDRAW CUseDdraw::lpDD=NULL;
LPDIRECTDRAWSURFACE CUseDdraw::lpDDSPrimary=NULL;
LPDIRECTDRAWSURFACE CUseDdraw::lpDDSBack=NULL;

// Constructor - Calls Start() to init. DDraw
CUseDdraw::CUseDdraw()
{
ok = Start();
}
// Destructor - Calls Finish() to do the cleanup
CUseDdraw::~CUseDdraw()
{
Finish();
}
CUseDdraw *pUseDD;


void vpagecopy(char *src)
{

    int i,j;
	DDSURFACEDESC ddsd;
	char * start_src = src;
	ddsd = pUseDD->LockBackBuf();
//	MessageBox(AppWnd,"Locked","OK",MB_OK);
    
	unsigned char *pBuff = (unsigned char *) ddsd.lpSurface;
	DWORD lPitch = ddsd.lPitch;
	if(!pBuff)
	{
//	MessageBox(AppWnd,"pBuff = NULL","ERROR",MB_OK | MB_ICONERROR);
	return;
	};
//	MessageBox(AppWnd,"Start copying","OK",MB_OK);
    
	// copy vscreen to real screen row by row,col by col because of pitch != renderwidth.
	for (j=0;j<RENDERHEIGHT;j++)
	{
	 pBuff = (unsigned char*)ddsd.lpSurface+j*lPitch;		
	 src   = (char*)(start_src+j*RENDERWIDTH);
	 // Copy src to pBuff.
	 for (i=0;i<RENDERWIDTH;i++)
	 {
      // calculate addresses
	  pBuff++;
	  src++;
	  *pBuff = *src;	  
	  }
	};
  
 pUseDD->UnLockBackBuf();
 //memset(src,0,RENDERWIDTH*RENDERHEIGHT);
 
//  clear
 src = start_src;	
 for (i=0;i<RENDERWIDTH*RENDERHEIGHT;i++)
    *(src++) = 0;
 
};


int height;
int angle, lasta;
int speed;
    
void initengine()
{
    tview *v;
    int st, et;      // time counters for FPS calculations
    int frame;       // frame counter for FPS calculations
    int actionflag;  // mouse available flag 
    short int x;     // mouse mickey "x" counter
    short int y;     // mouse mickey "y" counter
    int quit;
	createmaps();    // build the terrain and color maps
	buildtables();   // build sin/cos and div tables
	
    frame = 0;
    //st = clock();    // starting time counter
	// initialize view
    mainview.cy = 0;
    mainview.cx = 128*256;
    mainview.viewangle = 0;
    setview(0, 0, 0, &mainview);
	
    actionflag = 1;
    angle = 0;
    height = 128*256;
    speed = 0;
    lasta = 0;
    quit = 0;
    initmatrix(viewer);
    setview(speed, angle, angle, &mainview);
    setrolltable(0);
	
	if (height > 1000*256) height = 1000*256;
	if (height < ((int)heightmap[fINDEX(mainview.cx, mainview.cy)] << 8))
	height = ((int)heightmap[fINDEX(mainview.cx, mainview.cy)]) << 8;
	mainview.z = height;

	/*
    while (!quit)
    {
      lasta = angle;
      readmouse(&x, &y);
      angle += (x - 160) / 8;
      height += (99 - (int)(y))*16;
      if (height > 1000*256) height = 1000*256;
      if (height < ((int)heightmap[fINDEX(mainview.cx, mainview.cy)] << 8))
        height = ((int)heightmap[fINDEX(mainview.cx, mainview.cy)]) << 8;
      mainview.z = height;

      if (kbhit())
      {
        switch (getch())
        {
         case '+' : speed += 16;
                    if (speed > 512) speed = 512;
                    break;
         case '-' : speed -= 16;
                    if (speed < 0) speed = 0;
                    break;
         case 27 : quit = 1;
        }
      }
      setview(speed, angle, angle, &mainview);
      setrolltable(angle - lasta);
      drawscreen(&mainview);  // render the image to the virtual screen
      drawmarkers(x, y);

      while (!(inp(0x3da) & 8));  // wait for vertical retrace

      vpagecopy(vpage);       // copy the virtual screen to video memory
      frame++;
    }
    et = clock();
    vmode(0x03);
    printf("%5.2f frames per second\n\n", ((float)(frame*CLOCKS_PER_SEC))/(et-st));
    cleanup();
    printf("Heightfield engine by Alex Chalfin\n");
    printf("email: achalfin@uceng.uc.edu\n");
	*/
	MessageBox(AppWnd,"End of initengine","OK",MB_OK);
    
}

void initmatrix(matrix m)
{
  m[0] = 1; m[1] = 0; m[2] = 0;
  m[3] = 0; m[4] = 1; m[5] = 0;
  m[6] = 0; m[7] = 0; m[8] = 1;
}

void matmul(matrix m1, matrix m2)
{
  int i,j,k;
  matrix temp;

  for (i = 0; i < 3; i++)
    for (j = 0; j < 3; j++)
      for (k = 0, temp[i*3+j] = 0; k < 3; k++)
        temp[i*3+j] += m1[i*3+k] * m2[k*3+j];
  for (i = 0; i < 9; i++)
    m1[i] = temp[i];
}

void rotate(matrix o, float xr, float yr, float zr)
{
  matrix xrot, yrot, zrot;

  xrot[0] = 1; xrot[1] = 0;       xrot[2] = 0;
  xrot[3] = 0; xrot[4] = cos(xr); xrot[5] = -sin(xr);
  xrot[6] = 0; xrot[7] = sin(xr); xrot[8] = cos(xr);
  yrot[0] = cos(yr);  yrot[1] = 0; yrot[2] = sin(yr);
  yrot[3] = 0;        yrot[4] = 1; yrot[5] = 0;
  yrot[6] = -sin(yr); yrot[7] = 0; yrot[8] = cos(yr);
  zrot[0] = cos(zr);  zrot[1] = -sin(zr); zrot[2] = 0;
  zrot[3] = sin(zr);  zrot[4] = cos(zr);  zrot[5] = 0;
  zrot[6] = 0;        zrot[7] = 0;        zrot[8] = 1;

  /* concatenate matricies with orientation matrix */
  matmul(o, xrot);
  matmul(o, yrot);
  matmul(o, zrot);
}


void drawmarkers(short int x, short int y)
{
  int i;
  for (i = 0; i < 4; i++)
  {
    vpage[y*RENDERWIDTH+i] = 255;
    vpage[x+i*RENDERWIDTH] = 255;
  }
}
void drawmap()
{
	for (int i=0;i<MAPSIZE/4;i++)
     for (int j=0;j<MAPSIZE/4;j++)
	  vpage[i+j*RENDERWIDTH] = heightmap[INDEX(i*4,j*4)];
	vpage[(mainview.cx>>8)/4+(mainview.cy >> 8)/4*RENDERWIDTH] =255;
	
}
void setrolltable(int x)
{
  int i;
  int y1, y2, dy;

  if (x > 32) x = 32;
  if (x < -32) x = -32;
  dy = sine[(x & 255)];
  y1 = 99 << 8;
  y2 = 99 << 8;

  for (i = 0; i < 160; i++)
  {
    rolltable[160+i] = y1 >> 8;
    rolltable[159-i] = y2 >> 8;
    y1 -= dy;
    y2 += dy;
  }
  
}

void setview(int speed, int viewangle, int moveangle, tview *v)
// sets the viewpoint parameters
// in:  speed - the speed you are traveling in 24.8 fixed point
//      viewangle - the angle to look. range 0..255
//      moveangle - the angle to move. range 0..255
//      viewer - height of the viewer
//      v - tview to set parameters of
{
  int index;

  // update the center position based on speed and move angle
  moveangle &= 255;
  v->cx += ((speed*sine[moveangle]) >> 8);
  v->cy += ((speed*cosine[moveangle]) >> 8);

  // set viewing angle
  viewangle &= 255;
  v->viewangle = viewangle;
  v->y1 = (FARDEPTH * cosine[viewangle]) + (SLOPE*FARDEPTH * sine[viewangle]) + v->cy;
  v->x1 = (FARDEPTH * sine[viewangle])   - (SLOPE*FARDEPTH * cosine[viewangle]) + v->cx;

  v->y2 = (FARDEPTH * cosine[viewangle]) - (SLOPE*FARDEPTH * sine[viewangle]) + v->cy;
  v->x2 = (FARDEPTH * sine[viewangle])   + (SLOPE*FARDEPTH * cosine[viewangle]) + v->cx;
}
void verticallines();
void drawscreen(tview *v)
// This is the main drawing routine.
// It takes a TVIEW structure as a parameter and renders the scene for you.
// It will render at any rotation (depends on the TVIEW passed in)
// it does not do view tilting (easily added).
// Based on 24.8 fixed point and does proper map repeating when necessary
{
    int x,y,i;                             // general counters
    int xstep, xval, ystep, yval;          // constant z interpolation vars
    int viewer, per;                       // perspective vars
    int lxval1, lxval2, lyval1, lyval2;
    int lxstep1, lxstep2, lystep1, lystep2;
    static int leest[RENDERWIDTH];         // interpolation index list
    int index, oldi;                       // map location and old map location
    hrec *tbuf;                            // buf record. for pointer walks
    int depth;
    int dirlength;
    int cx, cy;
    int val;
    int cosA, sinA;
    int roty, roty2;
    int snapx, snapy;
    int dx, dx2, dx3;
    int d1, d2;
    int i1, i2, i3, i4;

    viewer = v->z + (20 << 8);   // set appropriate viewing position

    cx = v->cx;
    cy = v->cy;

    cosA = cosine[(-v->viewangle) & 0xff];
    sinA = sine[(-v->viewangle) & 0xff];

    // set up map traversing vars
    lxval1 = v->x1;
    lxval2 = v->x2;
    lxstep1 = (cx - lxval1) / NUMSTEPS;
    lxstep2 = (cx - lxval2) / NUMSTEPS;

    lyval1 = v->y1;
    lyval2 = v->y2;
    lystep1 = (cy - lyval1) / NUMSTEPS;
    lystep2 = (cy - lyval2) / NUMSTEPS;

    // calculate the adjusted projection factor based on sub-voxel
    // location (fractional portion for location)
    snapy = lyval1 & 0xffffff00;
    roty = (((snapy - cy) * cosA) + ((lxval1 - cx) * sinA)) >> 8;
    roty2 = (((lyval1 - cy) * cosA) + ((lxval1 - cx) * sinA)) >> 8;
    dirlength = roty2 - roty;

    lxval1 += (lxstep1*dirlength) >> 8;
    lxval2 += (lxstep2*dirlength) >> 8;
    lyval1 += (lystep1*dirlength) >> 8;
    lyval2 += (lystep2*dirlength) >> 8;

    depth = ((NUMSTEPS+1) << 8) - dirlength;

    // do the first line of the rendering. i.e. furthest from the viewer
    xval = lxval1 << 8;
    xstep = ((lxval2 - lxval1) << 8) / RENDERWIDTH;
    yval = lyval1 << 8;
    ystep = ((lyval2 - lyval1) << 8) / RENDERWIDTH;
    per = (HEIGHTMUL << 16) / depth;
    index = 0;
    tbuf = buf1;
    if (abs(xstep) > abs(ystep))
    {
        oldi = (xval - 65536) >> 16;
        for (x = 0; x < RENDERWIDTH; x++)
        {
          if ((oldi != (xval >> 16)) || (x == (RENDERWIDTH-1)))
          {
            i1 = fpINDEX(xval+65536, yval);
            i2 = fpINDEX(xval-65536, yval);
            i3 = fpINDEX(xval, yval+65536);
            i4 = fpINDEX(xval, yval-65536);
            val = (heightmap[i1] + heightmap[i2] +
                   heightmap[i3] + heightmap[i4]) << 6;
            tbuf->y = rolltable[x] - (((val - viewer)*per) >> 16);
            tbuf->c = (colormap[i1] + colormap[i2] + colormap[i3] + colormap[i4]) >> 2;
            leest[index++] = x;
            oldi = xval >> 16;
          }
          xval += xstep;
          yval += ystep;
          tbuf++;
        }
    }
    else
    {
        oldi = (yval - 65536) >> 16;
        for (x = 0; x < RENDERWIDTH; x++)
        {
          if ((oldi != (yval >> 16)) || (x == (RENDERWIDTH-1)))
          {
            i1 = fpINDEX(xval+65536, yval);
            i2 = fpINDEX(xval-65536, yval);
            i3 = fpINDEX(xval, yval+65536);
            i4 = fpINDEX(xval, yval-65536);
            val = (heightmap[i1] + heightmap[i2] +
                   heightmap[i3] + heightmap[i4]) << 6;
            tbuf->y = rolltable[x] - (((val - viewer)*per) >> 16);
            tbuf->c = (colormap[i1] + colormap[i2] + colormap[i3] + colormap[i4]) >> 2;

            leest[index++] = x;
            oldi = yval >> 16;
          }
          xval += xstep;
          yval += ystep;
          tbuf++;
        }
    }

    for (x = 0; x <= (index - 2); x++)
    {
      xval = buf1[leest[x]].y << 8;
      yval = buf1[leest[x]].c << 8;
      per = divtable[leest[x+1]-leest[x]];
      xstep = ((buf1[leest[x+1]].y - buf1[leest[x]].y) * per) >> 8;
      ystep = ((buf1[leest[x+1]].c - buf1[leest[x]].c) * per) >> 8;
      tbuf = &buf1[leest[x]+1];
      for (i = leest[x]+1; i < leest[x+1]; i++)
      {
         xval+=xstep;
         yval+=ystep;
         tbuf->y = xval >> 8;
         tbuf->c = yval >> 8;
         tbuf++;
      }
    }
        



    lxval1 += lxstep1;
    lxval2 += lxstep2;
    lyval1 += lystep1;
    lyval2 += lystep2;
    depth -= 256;

  // do the intermediate lines

    for (y = (FARDEPTH - 1); y > NEARDEPTH; y--)
    {
        xval = lxval1 << 8;
        xstep = ((lxval2 - lxval1) << 8) / RENDERWIDTH;
        yval = lyval1 << 8;
        ystep = ((lyval2 - lyval1) << 8) / RENDERWIDTH;
        tbuf = buf2;
        index = 0;  // initialize interpolation index
        per = (HEIGHTMUL << 16) / depth;
        if (abs(xstep) > abs(ystep))
        {
            oldi = (xval - 65536) >> 16;
            for (x = 0; x < RENDERWIDTH; x++)
            {
              if ((oldi != (xval >> 16)) || (x == (RENDERWIDTH-1)))
              {
                i1 = fpINDEX(xval+65536, yval);
                i2 = fpINDEX(xval-65536, yval);
                i3 = fpINDEX(xval, yval+65536);
                i4 = fpINDEX(xval, yval-65536);
                val = (heightmap[i1] + heightmap[i2] +
                       heightmap[i3] + heightmap[i4]) << 6;
                tbuf->y = rolltable[x] - (((val - viewer)*per) >> 16);
                tbuf->c = (colormap[i1] + colormap[i2] + colormap[i3] + colormap[i4]) >> 2;
                leest[index++] = x;
                oldi = xval >> 16;
              }
              xval += xstep;
              yval += ystep;
              tbuf++;
            }
        }
        else
        {
            oldi = (yval - 65536) >> 16;
            for (x = 0; x < RENDERWIDTH; x++)
            {
              if ((oldi != (yval >> 16)) || (x == (RENDERWIDTH-1)))
              {
                i1 = fpINDEX(xval+65536, yval);
                i2 = fpINDEX(xval-65536, yval);
                i3 = fpINDEX(xval, yval+65536);
                i4 = fpINDEX(xval, yval-65536);
                val = (heightmap[i1] + heightmap[i2] +
                       heightmap[i3] + heightmap[i4]) << 6;
                tbuf->y = rolltable[x] - (((val - viewer)*per) >> 16);
                tbuf->c = (colormap[i1] + colormap[i2] + colormap[i3] + colormap[i4]) >> 2;
                leest[index++] = x;
                oldi = yval >> 16;
              }
              xval += xstep;
              yval += ystep;
              tbuf++;
            }
        }

        // fix the left side according to sub voxel errors

        dx = leest[2] - leest[1];  // get approximate screen delta
        dx2 = leest[1] - leest[0]; // get location of incorrect data
        dx3 = dx - dx2;
        if ((dx3 > 0) && (dx > 0))
        {
          d1 = ((buf2[leest[1]].y - buf2[leest[0]].y) * divtable[dx]) >> 8;
          d2 = ((buf2[leest[1]].c - buf2[leest[0]].c) * divtable[dx]) >> 8;
          buf2[0].y =  buf2[0].y + ((dx3*d1) >> 8);
          buf2[0].c =  buf2[0].c + ((dx3*d2) >> 8);
        }

        // fix the right side according to sub voxel errors

        dx = leest[2] - leest[1];  // get approximate screen delta
        dx2 = leest[index-1] - leest[index-2]; // get actual delta
        dx3 = dx - dx2;
        if ((dx > 0) && (dx2 > 0) && (dx3 >= 0))
        {
          xval += (xstep * dx3);
          yval += (ystep * dx3);
          i = fpINDEX(xval, yval);
          val = (heightmap[fpINDEX(xval+65536, yval)] +
                 heightmap[fpINDEX(xval-65536, yval)] +
                 heightmap[fpINDEX(xval, yval+65536)] +
                 heightmap[fpINDEX(xval, yval-65536)]) << 6;
          buf2[leest[index-1]].y = rolltable[leest[index-1]] - (((val - viewer)*per) >> 16);
          buf2[leest[index-1]].c = colormap[i];
  
          d1 = ((buf2[leest[index-1]].y - buf2[leest[index-2]].y) * divtable[dx]) >> 8;
          d2 = ((buf2[leest[index-1]].c - buf2[leest[index-2]].c) * divtable[dx]) >> 8;
          buf2[leest[index-1]].y = buf2[leest[index-2]].y + ((dx2*d1) >> 8);
          buf2[leest[index-1]].c = buf2[leest[index-2]].c + ((dx2*d2) >> 8);
        }

        for (x = 0; x <= (index - 2); x++)
        {
          xval = buf2[leest[x]].y << 8;
          yval = buf2[leest[x]].c << 8;
          per = divtable[leest[x+1]-leest[x]];
          xstep = ((buf2[leest[x+1]].y - buf2[leest[x]].y) * per) >> 8;
          ystep = ((buf2[leest[x+1]].c - buf2[leest[x]].c) * per) >> 8;
          tbuf = &buf2[leest[x]+1];

          for (i = leest[x]+1; i < leest[x+1]; i++)
          {
             xval+=xstep;
             yval+=ystep;
             tbuf->y = xval >> 8;
             tbuf->c = yval >> 8;
             tbuf++;
          }
        }
        

// draw the vertical bars
        verticallines();

// swap the buffers
        tbuf = buf1;
        buf1 = buf2;
        buf2 = tbuf;

        lxval1 += lxstep1;
        lyval1 += lystep1;
        lxval2 += lxstep2;
        lyval2 += lystep2;
        depth -= 256;
    }

   // do the closest row

   xval = lxval1;
   xstep = (lxval2 - lxval1) / RENDERWIDTH;
   yval = lyval1;
   ystep = (lyval2 - lyval1) / RENDERWIDTH;

   tbuf = buf2;
   if (abs(xstep) > abs(ystep))
   {
     oldi = (xval - 256) >> 8;
     for (x = 0; x < RENDERWIDTH; x++)
     {
       if ((oldi != (xval >> 8)) || (x == (RENDERWIDTH-1)))
       {
         i = fINDEX(xval, yval);
         tbuf->c = colormap[i];
         leest[index++] = x;
         oldi = xval >> 8;
       }
       tbuf->y = RENDERHEIGHT-1;
       xval += xstep;
       yval += ystep;
       tbuf++;
     }
   }
   else
   {
     oldi = (yval - 256) >> 8;
     for (x = 0; x < RENDERWIDTH; x++)
     {
       if ((oldi != (yval >> 8)) || (x == (RENDERWIDTH-1)))
         {
           i = fINDEX(xval, yval);
           tbuf->c = colormap[i];
           leest[index++] = x;
           oldi = yval >> 8;
         }
       tbuf->y = RENDERHEIGHT-1;
       xval += xstep;
       yval += ystep;
       tbuf++;
     }
   }

   dx = leest[2] - leest[1];  // get approximate screen delta
   dx2 = leest[1] - leest[0]; // get location of incorrect data
   dx3 = dx - dx2;
   if ((dx3 > 0) && (dx > 0))
   {
     d2 = ((buf2[leest[1]].c - buf2[leest[0]].c) * divtable[dx]) >> 8;
     buf2[0].c =  buf2[0].c + ((dx3*d2) >> 8);
   }

   dx = leest[2] - leest[1];  // get approximate screen delta
   dx2 = leest[index-1] - leest[index-2]; // get actual delta
   dx3 = dx - dx2;
   if ((dx > 0) && (dx2 > 0) && (dx3 >= 0))
   {
     xval += (xstep * dx3);
     yval += (ystep * dx3);
     i = fpINDEX(xval, yval);
     buf2[leest[index-1]].c = colormap[i];
     d2 = ((buf2[leest[index-1]].c - buf2[leest[index-2]].c) * divtable[dx]) >> 8;
     buf2[leest[index-1]].c = buf2[leest[index-2]].c + ((dx2*d2) >> 8);
   }


   for (x = 0; x <= (index - 2); x++)
   {
     yval = buf2[leest[x]].c << 8;
     per = divtable[leest[x+1]-leest[x]];
     ystep = ((buf2[leest[x+1]].c - buf2[leest[x]].c) * per) >> 8;
     tbuf = &buf2[leest[x]+1];

     for (i = leest[x]+1; i < leest[x+1]; i++)
     {
       yval+=ystep;
       tbuf->c = yval >> 8;
       tbuf++;
     }
   }
   verticallines();
    
}

void verticallines()
// draws the vertical lines between two "buffers"
{
    int x, y;
    int cstep;
    int y1, y2;
    register32_8 cval;

    for (x = 0; x < RENDERWIDTH; x++)
    {
        if (buf1[x].y < buf2[x].y)
        {
            y1 = buf1[x].y;
            y2 = buf2[x].y;
            cval.dword = buf1[x].c << 8;
            cstep = ((buf2[x].c - buf1[x].c) * divtable[y2-y1]) >> 8;
            if (y2 > RENDERHEIGHT-1) y2 = RENDERHEIGHT-1;
            if (y1 < 0)
            {
              cval.dword += cstep*(-y1);
              y1 = 0;
            }
            y1 = ( (int)(vpage) + (y1*RENDERWIDTH) + x);
            y2 = ( (int)(vpage) + (y2*RENDERWIDTH) + x);
            for (y = y1; y <= y2; y+=RENDERWIDTH)
            {
                *(char *)(y) = cval.bytes.b2;
                cval.dword+=cstep;
            }
        }
    }
}

unsigned char pal[384] = {
  0,0,0,48,48,48,1,0,43,1,3,43,2,5,44,2,7,44,3,9,45,4,11,46,5,13,47,6,15,48,
  7,17,49,8,19,50,9,21,51,10,22,52,11,24,52,12,26,54,13,28,54,14,30,56,15,32,
  56,16,34,58,17,34,58,17,36,58,18,38,60,19,40,60,20,42,62,21,44,62,10,31,0,
  11,31,0,11,31,1,11,32,1,12,32,1,12,32,2,12,33,2,13,33,2,14,33,3,15,33,3,15,
  34,3,15,34,4,15,35,4,16,35,4,16,35,5,16,36,5,17,36,5,17,36,6,18,37,6,18,38,
  7,19,38,8,20,39,8,20,40,9,21,40,10,22,41,10,22,42,11,23,42,12,24,43,12,24,
  44,13,25,44,14,25,45,14,26,46,15,27,46,16,27,47,17,28,47,18,28,48,19,29,49,
  19,30,49,20,30,50,21,31,51,21,32,51,22,32,52,23,33,53,23,34,53,24,34,54,25,
  35,55,25,36,55,26,36,56,27,37,57,27,38,57,27,39,57,27,41,57,27,42,57,27,43,
  57,27,44,57,27,45,57,27,46,57,27,47,57,27,49,57,27,50,57,27,51,57,27,52,57,
  27,53,57,27,55,57,27,56,57,27,57,57,27,58,57,27,58,57,26,58,57,25,58,57,24,
  58,56,23,58,55,22,58,54,20,58,53,19,58,51,18,58,50,17,58,50,16,58,49,15,58,
  48,14,58,47,13,58,46,12,58,45,11,58,44,11,58,44,10,58,43,10,58,42,9,57,41,
  8,57,40,8,56,39,7,56,38,6,55,37,5,55,35,4,54,33,4,54,31,2,32,32,32,63,63,63,
  63,63,63,63,63,63,63,63,63,48,48,48,63,63,63,63,63,63 };

void makepalette(PALETTEENTRY * pe)
{
   int i,j;
   
   for (i = 1; i < 64; i++)
   {
	   pe[i].peRed =i/2;
	   pe[i].peGreen =64+i;	
	   pe[i].peBlue =0;
   };
   for (i = 64; i < 128; i++)
   {
	   pe[i].peRed =i/2;
	   pe[i].peGreen =128+((i-64)*2);
	   pe[i].peBlue =0;
	   
   }
   
   /*
   for (i=0;i<128;i++)
   {
		pe[i].peRed   = pal[i*3]*4;
		pe[i].peGreen = pal[i*3+1]*4;
		pe[i].peBlue  = pal[i*3+2]*4;
   }
	   
   for (i=0;i<256;i++)
   {
	   pe[i].peRed*=1.0;
	   pe[i].peGreen*=1.0;
	   pe[i].peBlue*=1.0;

	
   };
   */
   pe[255].peRed =255;pe[255].peGreen =255;pe[255].peBlue =255;
   
}

void buildtables()
{
  int i;

  for (i = 1; i < 512; i++)
    divtable[i] = 65536/i;
  for (i = 0; i < 256; i++)
  {
    sine[i] = (sin(i*2*3.141592/256.0)*256);
    cosine[i] = (cos(i*2*3.141592/256.0)*256);
  }
}


void createmaps()
{
    int i;
    srand(time(NULL));
	// allocate necessary memory
    heightmap = (char *)malloc(MAPSIZE*MAPSIZE);
    colormap = (char *)malloc(MAPSIZE*MAPSIZE);
    vpage = (char *)malloc(RENDERWIDTH*RENDERHEIGHT);
    buf1 = (hrec *)malloc(RENDERWIDTH * sizeof(hrec));
    buf2 = (hrec *)malloc(RENDERWIDTH * sizeof(hrec));
    
    // clear all memory
    memset(heightmap, 0, MAPSIZE*MAPSIZE);
    memset(vpage, 0, RENDERWIDTH*RENDERHEIGHT);

//    printf("Creating %dx%d fractal terrain\n", MAPSIZE, MAPSIZE);
    heightmap[0] = (rand() % 128) + 64;   // initialize starting point on map
    createfractalmap(0, 0, MAPSIZE, MAPSIZE);

    //printf("Smooting terrain\n");
    for (i = 0; i < NUMSMOOTH; i++)
      smoothmap();

    makecolormap();
}

void cleanup()
// free allocated memory
{
    free(heightmap);
    free(colormap);
    free(vpage);
    free(buf1);
    free(buf2);
}

void createfractalmap(int x1, int y1, int x2, int y2)
// recursive fractal terrain builder
{
    int p1, p2, p3, p4;
    int xn, yn, dxy;

    if (((x2-x1) < 2) && ((y2-y1) < 2))  // make sure their is something to do
      return;

    p1 = heightmap[INDEX(x1,y1)];
    p2 = heightmap[INDEX(x1,y2)];
    p3 = heightmap[INDEX(x2,y1)];
    p4 = heightmap[INDEX(x2,y2)];

    xn = (x2+x1) >> 1;
    yn = (y2+y1) >> 1;
    dxy = 5*(x2 - x1 + y2 - y1) / 3;

    if (heightmap[INDEX(xn,y1)] == 0)
      heightmap[INDEX(xn,y1)] = newcolor(p1+p3, dxy, 2);
    if (heightmap[INDEX(x1,yn)] == 0)
      heightmap[INDEX(x1,yn)] = newcolor(p2+p4, dxy, 2);
    if (heightmap[INDEX(x2,yn)] == 0)
      heightmap[INDEX(x2,yn)] = newcolor(p3+p4, dxy, 2);
    if (heightmap[INDEX(xn,y2)] == 0)
      heightmap[INDEX(xn,y2)] = newcolor(p1+p2, dxy, 2);
    heightmap[INDEX(xn,yn)] = newcolor(p1+p2+p3+p4, dxy, 4);
    createfractalmap(x1, y1, xn, yn);
    createfractalmap(xn, y1, x2, yn);
    createfractalmap(x1, yn, xn, y2);
    createfractalmap(xn, yn, x2, y2);
}

char newcolor(int mc, int n, int dvd)
{
    int loc;

    loc = (mc + n - RANDOM(n*2)) / dvd - 1;
    if (loc > 255)
      loc = 255;
    if (loc < 10)
      loc = 10;
    return(loc);
}

void smoothmap()
// smooths the map. Gives better appearence
{
    int x,y;

    for (x = 0; x < MAPSIZE; x++)
      for (y = 0; y < MAPSIZE; y++)
        heightmap[INDEX(x,y)] = (heightmap[INDEX(x-1,y-1)] +
                                 heightmap[INDEX(x-1,y+1)] +
                                 heightmap[INDEX(x+1,y-1)] +
                                 heightmap[INDEX(x+1,y+1)] +
                                 heightmap[INDEX(x,  y-1)] +
                                 heightmap[INDEX(x,  y+1)] +
                                 heightmap[INDEX(x-1,y)] +
                                 heightmap[INDEX(x,y)] +
                                 heightmap[INDEX(x+1,y)]) / 9;
}

void makecolormap()
// builds a color map based on the height map
// attempts to add some color shift
{
    int x,y,temp;

    for (x = 0; x < MAPSIZE; x++)
     for (y = 0; y < MAPSIZE; y++)
     {
         temp = heightmap[INDEX(x,y)] >> 1;
         if (heightmap[INDEX(x-1,y)] < heightmap[INDEX(x,y)])
         {
            temp -= 8*(heightmap[INDEX(x,y)] - heightmap[INDEX(x-1, y)]);
            if (temp < 2)
               temp = 2;
         }
         else
         {
             if (heightmap[INDEX(x-1,y)] > heightmap[INDEX(x,y)])
             {
                 temp += 3*(heightmap[INDEX(x-1,y)] - heightmap[INDEX(x,y)]);
                 if (temp > 127)
                   temp = 127;
             }
         }
         colormap[INDEX(x,y)] = temp;
     }
}


// Start() - Create the window that will be used to atach the
// DirectDraw object. Create the DD object and the surface.
BOOL CUseDdraw::Start()
{
WNDCLASS wc;
DDSURFACEDESC ddsd;
DDSCAPS ddscaps;
HRESULT ddrval;

// Create the Window class and the Window ...
wc.style = CS_HREDRAW | CS_VREDRAW;
wc.lpfnWndProc = WindowProc;
wc.cbClsExtra = 0;
wc.cbWndExtra = 0;
wc.hInstance = gb_hInstance;
wc.hIcon = LoadIcon( gb_hInstance, IDI_APPLICATION );
wc.hCursor = LoadCursor( NULL, IDC_ARROW );
wc.hbrBackground = GetStockObject(BLACK_BRUSH);
wc.lpszMenuName = NAME;
wc.lpszClassName = NAME;
if(!RegisterClass( &wc ))
return FALSE;

DWORD exStyle = 0;
DWORD style = WS_POPUP;

HWND hwnd = CreateWindowEx(exStyle,NAME,TITLE,
style,0,0,
GetSystemMetrics(SM_CXSCREEN),
GetSystemMetrics(SM_CYSCREEN),
NULL,NULL, gb_hInstance,NULL );

if( !hwnd ) return FALSE;
AppWnd = hwnd;
ShowWindow( hwnd, gb_nCmdShow );
UpdateWindow( hwnd );
SetFocus( hwnd );

// Create DD Object
ddrval = DirectDrawCreate( NULL, &lpDD, NULL );
if( ddrval != DD_OK )
return FALSE;

ddrval = lpDD->SetCooperativeLevel( hwnd, DDSCL_ALLOWMODEX | DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN);
if( ddrval != DD_OK )
return FALSE;

ddrval = lpDD->SetDisplayMode( RENDERWIDTH,RENDERHEIGHT, 8);

if(ddrval != DD_OK)
return FALSE;

// Create the primary surface with a back buffer
ddsd.dwSize = sizeof( ddsd );
ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
DDSCAPS_FLIP |
DDSCAPS_COMPLEX;
ddsd.dwBackBufferCount = 1;
ddrval = lpDD->CreateSurface( &ddsd, &lpDDSPrimary, NULL );
if( ddrval != DD_OK )
return FALSE;

ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
ddrval = lpDDSPrimary->GetAttachedSurface(&ddscaps, &lpDDSBack);
if( ddrval != DD_OK )
return FALSE;
MessageBox(AppWnd,"Now entering initengine()","OK",MB_OK);
initengine();	
MessageBox(AppWnd,"Now exited initengine()","OK",MB_OK);

PALETTEENTRY pe[256];
makepalette(pe);
ddrval = lpDD->CreatePalette(DDPCAPS_8BIT,pe,&lpDDPal,NULL);
if (ddrval != DD_OK)
 return FALSE;
lpDDSPrimary->SetPalette(lpDDPal);
return TRUE;

}

// Finish() - Cleanup function
void CUseDdraw::Finish()
{
if( lpDD != NULL )
{
if( lpDDSPrimary != NULL )
{
lpDDSPrimary->Release();
lpDDSPrimary = NULL;
}
if (lpDDPal != NULL)
{
lpDDPal->Release();
lpDDPal = NULL;
}
lpDD->Release();
lpDD = NULL;

}
}

// RestoreAll() - Restores the surface if it is lost ...
HRESULT CUseDdraw::RestoreAll()
{
HRESULT ddrval;
ddrval = lpDDSPrimary->Restore();
return ddrval;
}

// LockBackBuf() - Lock the back surface, returns a pointer
// to the back buffer, UnLockBackBuf() must be called after !
DDSURFACEDESC CUseDdraw::LockBackBuf()
{
DDSURFACEDESC ddsd;
ddsd.dwSize = sizeof(ddsd);
HRESULT ret=DDERR_WASSTILLDRAWING;
while (ret== DDERR_WASSTILLDRAWING)
ret=lpDDSBack->Lock(NULL, &ddsd, 0, NULL);
if (ret != DD_OK)
 MessageBox(AppWnd,"FAILED LOCK","ERROR",MB_OK);
	
 
return ddsd;
}

// UnLockBackBuf() - Unlocks the surface previous locked with
// LockBackBuf()
void CUseDdraw::UnLockBackBuf()
{
lpDDSBack->Unlock(NULL);
}

// Update() - Flip the surfaces and update the screen
void CUseDdraw::Update()
{
RECT r={0,0,RENDERWIDTH,RENDERHEIGHT};
lpDDSPrimary->BltFast(0,0,lpDDSBack,&r,DDBLTFAST_NOCOLORKEY);

while( 1 )
{
HRESULT ddrval = lpDDSPrimary->Flip( NULL, 0 );
if( ddrval == DD_OK )
break;

if( ddrval == DDERR_SURFACELOST )
{
ddrval = RestoreAll();
if( ddrval != DD_OK )
break;

}
if( ddrval != DDERR_WASSTILLDRAWING )
break;

}

}

/*-------------------------------------------------------------------

WindowProc -
Window procedure used by the Window created in CUseDdraw::Start()

---------------------------------------------------------------------*/

long FAR PASCAL WindowProc( HWND hWnd, UINT message,
WPARAM wParam, LPARAM lParam )
{
switch( message )
{
case WM_ACTIVATEAPP:
gb_bActive = wParam;
break;

case WM_SETCURSOR:
SetCursor(NULL);
return TRUE;

case WM_CREATE:
break;

case WM_KEYDOWN:
switch( wParam )
{
case VK_ESCAPE:
case VK_F12:
PostMessage(hWnd, WM_CLOSE, 0, 0);
break;
}
break;

case WM_DESTROY:
PostQuitMessage( 0 );
break;

}
return DefWindowProc(hWnd, message, wParam, lParam);
}

/*-------------------------------------------------------------------

WinMain

---------------------------------------------------------------------*/

int PASCAL WinMain( HINSTANCE hInstance, HINSTANCE ,LPSTR , int nCmdShow)
{
MSG msg;
gb_hInstance = hInstance;
gb_nCmdShow = nCmdShow;


pUseDD = new CUseDdraw();
if(!pUseDD)
return 255;

if(!pUseDD->Ok())
return 255;

srand( (unsigned)time( NULL ) );

while( 1 )
{
if( PeekMessage( &msg, NULL, 0, 0, PM_NOREMOVE ) )
{
if( !GetMessage( &msg, NULL, 0, 0 ) )
{
return msg.wParam;
}
TranslateMessage(&msg);
DispatchMessage(&msg);
}
else if( gb_bActive )
{
// clear screen
	  DDBLTFX ddbltfx;
	  ddbltfx.dwSize = sizeof(ddbltfx);
	  ddbltfx.dwFillColor = 0;
	  pUseDD->lpDDSBack->Blt(NULL,NULL,NULL,DDBLT_COLORFILL | DDBLT_WAIT,&ddbltfx);
	  POINT p;
	  GetCursorPos(&p);
			 
	  lasta = angle;
// Get the pointer and do the drawing ...
	  angle += (p.x - (RENDERWIDTH/2)) / 8 * 0.5;
	  
	  if (GetAsyncKeyState(VK_ADD) & 0x80000000)
		speed+=16;
	  else if (GetAsyncKeyState(VK_SUBTRACT) & 0x80000000)
	    speed-=16;
      height += ((RENDERHEIGHT/2-1) - (int)(p.y))*16;
      if (height > 1000*256) height = 1000*256;
      if (height < ((int)heightmap[fINDEX(mainview.cx, mainview.cy)] << 8))
        height = ((int)heightmap[fINDEX(mainview.cx, mainview.cy)]) << 8;
      mainview.z = height;
	  //speed = 64;
      setview(speed, angle, angle, &mainview);
      setrolltable(angle - lasta);

	  drawscreen(&mainview);
	  drawmap();
	  drawmarkers(p.x,p.y);
	  vpagecopy(vpage);
// Free the pointer and update the screen ...
pUseDD->Update();
}
else
WaitMessage();

}
delete pUseDD;
return 0;
}
