/*CLAX.C**********************************************************************

   Clax: Portable keyframing library
         see inclosed LICENSE.TXT for licensing terms.

         for documentation, refer to CLAX.TXT

   author           : Borzom
   file created     : 17/04/97
   file description : main clax routines

   revision history :
     v0.10 (17/04/97) Borzom: Initial version.

   notes            :
     the library itself, if it has bugs, its probably here :)

*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "clax.h"
#include "claxi.h"

#include "D3DUTIL.H"
#include "CALCOBJ.H"
#include "MATRIX.H"


/*****************************************************************************
  copyright, format table, etc...
*****************************************************************************/

char    clax_version[]   = "clax version 0.10(beta)";
char    clax_copyright[] = "copyright (c) 1997 borzom";

// current active scene
c_SCENE  *clax_scene;

// current active camera
c_CAMERA *clax_camera;

// curreng flags
int32     clax_flags;

struct {

	char   *name;						// file extension
	int32 (*load_mesh)   (FILE *f);		// loads mesh
	int32 (*load_motion) (FILE *f);		// loads motion
	int32 (*save_scene)  (FILE *f);		// saves the whole scene

} clax_drivers[] = {
	{ "3DS", clax_load_mesh_3DS, clax_load_motion_3DS, NULL },
	//{ "CLX", clax_load_mesh_CLX, clax_load_motion_CLX, clax_save_CLX }
};

/****************************************************************************
  internal functions
*****************************************************************************/

/*
  calc_normals: calculates face/vertex normals.
*/
static void calc_objnormals (c_OBJECT *obj)
{
	c_VECTOR	a, b, normal;
	int32		i, j, num;

	// face normals
	for (i = 0; i < obj->numfaces; i++) {
		vec_sub (&obj->faces[i].pa->vert, &obj->faces[i].pb->vert, &a);
		vec_sub (&obj->faces[i].pb->vert, &obj->faces[i].pc->vert, &b);

		vec_cross (&a, &b, &normal);
		vec_normalize (&normal, &obj->faces[i].norm);
	}

	// vertex normals
	for (i = 0; i < obj->numverts; i++) { 
		num = 0;
		vec_zero (&normal);

		//  vertex normal شǴ ̽ normal ؼ ȴ.
		for (j = 0; j < obj->numfaces; j++) {

			if (obj->faces[j].a == i || 
				obj->faces[j].b == i || 
				obj->faces[j].c == i) {

				vec_add (&normal, &obj->faces[j].norm, &normal);
				num++;
			}
		}

		if (num)
			vec_scale (&normal, 1.0f / (float)num, &normal);

		vec_normalize (&normal, &obj->vertices[i].norm);
	}
}


/*
  calc_normals: calculates face/vertex normals.
*/
/*
static void recalc_objnormals (c_OBJECT *obj)
{
	c_VECTOR  a, b, normal;
	int32     i, j, num;

	// face normals
	for (i = 0; i < obj->numfaces; i++) { 
		vec_sub (&obj->faces[i].pa->pvert, &obj->faces[i].pb->pvert, &a);
		vec_sub (&obj->faces[i].pb->pvert, &obj->faces[i].pc->pvert, &b);
		vec_cross (&a, &b, &normal);
		vec_normalize (&normal, &obj->faces[i].pnorm);
	}

	// vertex normals
	for (i = 0; i < obj->numverts; i++) { 
		num = 0;
		vec_zero (&normal);

		for (j = 0; j < obj->numfaces; j++) {
			if (obj->faces[j].a == i ||
				obj->faces[j].b == i ||
				obj->faces[j].c == i) {
				vec_add (&normal, &obj->faces[j].pnorm, &normal);
				num++;
			}
		}

		if (num) 
			vec_scale (&normal, 1.0 / (float)num, &normal);

		vec_normalize (&normal, &obj->vertices[i].pnorm);
	}
}
*/


/*
  calc_normals: calculates face/vertex normals.
*/
static void calc_normals ()
{
	w_NODE *node;

	for (node = clax_scene->world; node; node = node->next)
		if (node->type == clax_obj_object)
			calc_objnormals ((c_OBJECT *)node->object);
}


/*
  calc_bbox: calculate bounding boxes for objects.
*/
/*
static void calc_bbox ()
{
	w_NODE   *node;
	c_OBJECT *obj;
	int       i;

	for (node = clax_scene->world; node; node = node->next) {

		if (node->type == clax_obj_object) {
			obj = (c_OBJECT *)node->object;

			vec_copy (&obj->vertices[0].vert, &obj->bbox.min);
			vec_copy (&obj->vertices[0].vert, &obj->bbox.max);

			// vertex sort Ͽ ڽ Ѵ.
			for (i = 1; i < obj->numverts; i++) {

				if (obj->vertices[i].vert.x < obj->bbox.min.x)
					obj->bbox.min.x = obj->vertices[i].vert.x;

				if (obj->vertices[i].vert.y < obj->bbox.min.y)
					obj->bbox.min.y = obj->vertices[i].vert.y;

				if (obj->vertices[i].vert.z < obj->bbox.min.z)
					obj->bbox.min.z = obj->vertices[i].vert.z;

				if (obj->vertices[i].vert.x > obj->bbox.max.x)
					obj->bbox.max.x = obj->vertices[i].vert.x;

				if (obj->vertices[i].vert.y > obj->bbox.max.y)
					obj->bbox.max.y = obj->vertices[i].vert.y;

				if (obj->vertices[i].vert.z > obj->bbox.max.z)
					obj->bbox.max.z = obj->vertices[i].vert.z;
			}
		}

	}
}
*/


//  transform all world (see notes in the documentation).
static void do_transform ()
{
	w_NODE		*node;
	//w_NODE	*from, *to;
	c_OBJECT	*obj;
	//c_OBJECT	*o1, *o2;
	//c_VERTEX	*v1, *v2;
	// c_FACE   *f1, *f2;
	//c_VECTOR  vec;
	c_MATRIX	objmat, normat;
	//float     alpha;
	int32		i;

	for (node = clax_scene->world; node; node = node->next) {

		if (node->type == clax_obj_object) {

			obj = (c_OBJECT *)node->object;

			if (clax_camera) {
				// ī޶  
				continue;
				// mat_mul (clax_camera->matrix, obj->matrix, objmat);
			}
			else 
				// ׹ Ʈ  ó
				mat_copy (obj->matrix, objmat);

			mat_normalize (obj->matrix, normat);

			// ο  ó
			if ((clax_flags & clax_domorph) && (obj->flags & clax_obj_morph)) {

				continue;
				/*
				clax_byid (obj->morph.from, &from);
				clax_byid (obj->morph.to, &to);

				o1 = (c_OBJECT *)from->object;
				o2 = (c_OBJECT *)to->object;

				v1 = o1->vertices;
				v2 = o2->vertices;

				f1 = o1->faces;
				f2 = o2->faces;

				alpha = obj->morph.alpha;

				vec_lerp (&o1->bbox.min, &o2->bbox.min, alpha, &vec);
				mat_mulvec (obj->matrix, &vec, &obj->pbbox.min);

				vec_lerp (&o1->bbox.max, &o2->bbox.max, alpha, &vec);
				mat_mulvec (obj->matrix, &vec, &obj->pbbox.max);

				for (i = 0; i < obj->numverts; i++) {
					vec_lerp (&v1[i].vert, &v2[i].vert, alpha, &vec);
					mat_mulvec (objmat, &vec, &obj->vertices[i].pvert);
				}

				if ((clax_flags & clax_normlerp) && (clax_flags & clax_calcnorm)) {

					for (i = 0; i < obj->numverts; i++) {
						vec_lerp (&v1[i].norm, &v2[i].norm, alpha, &vec);
						mat_mulnorm (normat, &vec, &obj->vertices[i].pnorm);
					}

					for (i = 0; i < obj->numfaces; i++) {
						vec_lerp (&f1[i].norm, &f2[i].norm, alpha, &vec);
						mat_mulnorm (normat, &vec, &obj->faces[i].pnorm);
					}

				} 
				else if ((clax_flags & clax_normcalc) && (clax_flags & clax_calcnorm)) {
					recalc_objnormals (obj);
				}
				else if (clax_flags & clax_calcnorm) {

					for (i = 0; i < obj->numverts; i++) {
						mat_mulnorm (normat, &obj->vertices[i].norm,
									&obj->vertices[i].pnorm);
					}

					for (i = 0; i < obj->numfaces; i++) {
						mat_mulnorm (normat, &obj->faces[i].norm, &obj->faces[i].pnorm);
					}

				}
				*/
			}

			else {
				//    ٿ ڽ .
				//mat_mulvec (obj->matrix, &obj->bbox.min, &obj->pbbox.min);
				//mat_mulvec (obj->matrix, &obj->bbox.max, &obj->pbbox.max);

				for (i = 0; i < obj->numverts; i++) {

					mat_mulvec (objmat, &obj->vertices[i].vert, &obj->vertices[i].pvert);

					obj->lpD3DVertex[i].x = obj->vertices[i].pvert.x;
					obj->lpD3DVertex[i].y = obj->vertices[i].pvert.y;
					obj->lpD3DVertex[i].z = obj->vertices[i].pvert.z;
				}

				//if (clax_flags & clax_calcnorm) {

					for (i = 0; i < obj->numverts; i++) {

						mat_mulnorm (normat, &obj->vertices[i].norm, 
									 &obj->vertices[i].pnorm);

						obj->lpD3DVertex[i].nx = obj->vertices[i].pnorm.x;
						obj->lpD3DVertex[i].ny = obj->vertices[i].pnorm.y;
						obj->lpD3DVertex[i].nz = obj->vertices[i].pnorm.z;
					}

					/*
					for (i = 0; i < obj->numfaces; i++) {
						mat_mulnorm (normat, &obj->faces[i].norm, &obj->faces[i].pnorm);
					}
					*/
				//}
			}
		}
	}
}


//	deallocated memory used by track.
static void clax_free_track (t_TRACK *track)
{
	t_KEY *key;

	if (!track) 
		return;

	for (key = track->keys; key; key = key->next)
		free (key);

	free (track);
}


//	non-case sensitive string compare.
static int32 strucmp (char *s1, char *s2)
{
	int32 diff = 0;

	do {
		diff += (toupper (*s1) - toupper (*s2));
	} while (*s1++ && *s2++);

	return diff;
}


/*
  clax_insidebox: collision detection (point inside bounding box).
*/
/*
int clax_insidebox (c_BOUNDBOX *a, c_VECTOR *b)
{

  if( b->x > a->min.x && b->y > a->min.y && b->z > a->min.z &&
      b->x < a->max.x && b->y < a->max.y && b->z < a->max.z) return 1;
  return 0;
}
*/


//  print_track: output track data to stdout.
static void print_track (t_TRACK *track, char *n, int32 type, FILE *fp)
{
	t_KEY	*keys;
	w_NODE	*node;

	if (!track) 
		return;

#ifdef CLAX_DEBUG

	fprintf (fp, "%s %3d, frames:%8.3f, flags: ", n, track->numkeys, track->frames);

	if (track->flags == clax_track_repeat)
		fprintf (fp, "repeat\n"); 
	else if (track->flags == clax_track_loop)
		fprintf (fp, "loop\n"); 
	else
		fprintf (fp, "none\n");


	switch (type) {
		case 0: // float
			for(keys = track->keys; keys; keys = keys->next)
				fprintf(fp, "    frame: %9.3f  || value: %9.3f\n", keys->frame, 
																	keys->val._float);
			break;

		case 1: // vector
			for(keys = track->keys; keys; keys = keys->next)
				fprintf(fp, "    frame: %9.3f  || x: %9.3f, y: %9.3f, z: %9.3f\n",
						keys->frame,
						keys->val._vect.x, keys->val._vect.y, keys->val._vect.z);
			break;

		case 2: // rgb color
			for(keys = track->keys; keys; keys = keys->next)
				fprintf(fp, "    frame: %9.3f  || r: %9.3f, g: %9.3f, b: %9.3f\n",
						keys->frame,
						keys->val._vect.x, keys->val._vect.y, keys->val._vect.z);
			break;

		case 3: // quaternion
			for(keys = track->keys; keys; keys = keys->next)
				fprintf(fp, "    frame: %9.3f  || x: %9.3f, y: %9.3f, z: %9.3f, w: %9.3f\n",
						keys->frame,
						keys->val._quat.x, keys->val._quat.y, keys->val._quat.z,
						keys->val._quat.w);
			break;

		case 4: // morph
			for(keys = track->keys; keys; keys = keys->next) {
				clax_byid (keys->val._int, &node);

				fprintf (fp, "    frame: %9.3f  || morph to: %s\n",
						keys->frame, ((c_OBJECT *)node->object)->name);
			}
			break;

		case 5: // hide
			for(keys = track->keys; keys; keys = keys->next) {
				fprintf (fp, "    frame: %9.3f  || hidden: ", keys->frame);
				if (keys->val._int) 
					fprintf (fp, "yes\n"); 
				else 
					fprintf (fp, "no\n");
			}
	}
//#elif
	//if (form) {} // to skip the warning
	//if (type) {} // to skip the warning
#endif
}


/*
  print_map: output map data to stdout.
*/
static void print_map (c_MAP *map, char *n, FILE *fp)
{
	fprintf (fp, "  %s", n);
	fprintf (fp, "    filename:  %s\n", map->file);
	fprintf (fp, "    flags:     ");

	if (map->flags & clax_map_mirror) 
		fprintf (fp, "mirror ");

	if (map->flags & clax_map_negative) 
		fprintf (fp, "negative ");

	fprintf (fp, "\n  1/u scale: %9.3f\n", map->U_scale);
	fprintf (fp, "    1/v scale: %9.3f\n", map->V_scale);
	fprintf (fp, "    u offset:  %9.3f\n", map->U_offset);
	fprintf (fp, "    v offset:  %9.3f\n", map->V_offset);
	fprintf (fp, "    angle:     %9.3f\n", map->rot_angle);
}


/*****************************************************************************
  clax library (initialization, error handling)
*****************************************************************************/

//	clax initialization.
int32 clax_init (int32 flags)
{
	clax_scene  = NULL;
	clax_camera = NULL;
	clax_flags  = flags;

	return clax_err_ok;
}


//	clax deinitialization.
int32 clax_done ()
{
	clax_scene = NULL;
	clax_flags = 0;
	return clax_err_ok;
}


//	return error string.
char *clax_geterror (int32 code)
{
	switch (code) {
		case clax_err_nomem:
			return "not enough memory";

		case clax_err_nofile:
			return "file not found";

		case clax_err_badfile:
			return "corrupted file";

		case clax_err_badver:
			return "unsupported version";

		case clax_err_badformat: 
			return "unsupported format";

		case clax_err_badframe:  
			return "invalid frame number";

		case clax_err_badname:   
			return "invalid object name";

		case clax_err_noframes:  
			return "no frames in keyframer";

		case clax_err_notloaded: 
			return "scene not loaded";

		case clax_err_nullptr:   
			return "null pointer assignment";

		case clax_err_invparam:  
			return "invalid parameter";

		case clax_err_spline:    
			return "less than 2 keys in spline";

		case clax_err_singular:  
			return "cannot inverse singular matrix";

		case clax_err_badid:     
			return "bad object id";

		case clax_err_exist:     
			return "object already exist";

		case clax_err_undefined:
		default:
			return "internal error";
	}
}


/*****************************************************************************
  clax library (time and world handling)
*****************************************************************************/

//  return number of frames.
int32 clax_getframes (float *start, float *end)
{
	if (!clax_scene) 
		return clax_err_notloaded;

	if (!clax_scene->keyframer) 
		return clax_err_notloaded;

	if (clax_scene->f_end - clax_scene->f_start == 0) 
		return clax_err_noframes;

	*start = clax_scene->f_start;
	*end = clax_scene->f_end;
	
	return clax_err_ok;
}


//	set current frame number.
int32 clax_setframe (float frame)
{
	if (!clax_scene) 
		return clax_err_notloaded;

	if (!clax_scene->keyframer) 
		return clax_err_notloaded;

	if (clax_scene->f_end - clax_scene->f_start == 0) 
		return clax_err_noframes;

	if (clax_scene->f_start <= frame && clax_scene->f_end > frame) {
		clax_scene->f_current = frame;
		return clax_err_ok;
	} 
	else 
		return clax_err_badframe;
}


/*
  clax_getframe: get current frame number.
*/
/*
int32 clax_getframe (float *frame)
{
  if (!clax_scene) return clax_err_notloaded;
  if (!clax_scene->keyframer) return clax_err_notloaded;
  if (clax_scene->f_end - clax_scene->f_start == 0) return clax_err_noframes;
  *frame = clax_scene->f_current;
  return clax_err_ok;
}
*/


//	set active scene.
int32 clax_setactive_scene (c_SCENE *scene)
{
	if (!scene) 
		return clax_err_nullptr;

	clax_scene = scene;
	return clax_err_ok;
}


/*
  clax_setactive_camera: set active camera.
*/
/*
int32 clax_setactive_camera (c_CAMERA *cam)
{
	clax_camera = cam;
	return clax_err_ok;
}
*/


/*
  clax_getactive_scene: get active scene.
*/
/*
int32 clax_getactive_scene (c_SCENE **scene)
{

  *scene = clax_scene;
  return clax_err_ok;
}
*/


/*
  clax_getactive_camera: get active camera.
*/
/*
int32 clax_getactive_camera (c_CAMERA **camera)
{
  *camera = clax_camera;
  return clax_err_ok;
}
*/


// find object by name (world tree).
int32 clax_byname (char *name, w_NODE **node)
{
	// clax_scene->world : Ʈ  ּҸ  ִ.
	if (!clax_scene || !clax_scene->world) 
		return clax_err_notloaded;

	// ϴ ̸ ã Ʈ ּҸ  Ѵ.
	for (*node = clax_scene->world; *node; *node = (*node)->next) {

		switch ((*node)->type) {

			case clax_obj_camera:
				if (strcmp (((c_CAMERA *)((*node)->object))->name, name) == 0)
					return clax_err_ok; 
				break;

			case clax_obj_object:
				if (strcmp (((c_OBJECT *)((*node)->object))->name, name) == 0)
					return clax_err_ok; 
				break;

			case clax_obj_light:
				if (strcmp ( ((c_LIGHT *)((*node)->object))->name, name ) == 0)
					return clax_err_ok; 
					break;

			case clax_obj_material:
				if (strcmp (((c_MATERIAL *)((*node)->object))->name, name) == 0)
					return clax_err_ok; 
				break;

			case clax_obj_ambient:
				if (strcmp (((c_AMBIENT *)((*node)->object))->name, name) == 0)
					return clax_err_ok;
		}
	}

	return clax_err_nullptr;
}


//  find object by id (world tree).
int32 clax_byid (int32 id, w_NODE **node)
{
	if (!clax_scene || !clax_scene->world) 
		return clax_err_notloaded;

	for (*node = clax_scene->world; *node; *node = (*node)->next) {

		switch ((*node)->type) {

			case clax_obj_camera:
				if (((c_CAMERA *)((*node)->object))->id == id)
					return clax_err_ok;
				break;

			case clax_obj_object:
				if (((c_OBJECT *)((*node)->object))->id == id) 
					return clax_err_ok;
				break;

			case clax_obj_light:
				if (((c_LIGHT *)((*node)->object))->id == id) 
					return clax_err_ok;
				break;

			case clax_obj_material:
				if (((c_MATERIAL *)((*node)->object))->id == id) 
					return clax_err_ok;
				break;

			case clax_obj_ambient:
				if (((c_AMBIENT *)((*node)->object))->id == id) 
					return clax_err_ok;
		}
	}

	return clax_err_nullptr;
}


/*
  clax_findfirst: finds first node with attribute "attr" (world tree).
*/
/*
int32 clax_findfirst (int32 attr, w_NODE **node)
{

  if (!clax_scene || !clax_scene->world) return clax_err_notloaded;
  for (*node = clax_scene->world; *node; *node = (*node)->next)
    if ((*node)->type & attr) return clax_err_ok;
  return clax_err_ok; // return NULL
}
*/


/*
  clax_findnext: finds next node with attribute "attr" (world tree).
*/
/*
int32 clax_findnext (int32 attr, w_NODE **node)
{
  if (!clax_scene || !clax_scene->world) return clax_err_notloaded;
  for (*node = (*node)->next; *node; *node = (*node)->next)
    if ((*node)->type & attr) return clax_err_ok;
  return clax_err_ok; // return NULL
}
*/


/*****************************************************************************
  clax library (world/keyframer constructors)
*****************************************************************************/
//  add object to world list.
int32 clax_add_world (int32 type, void *obj)
{
	w_NODE *node;

	// clax_scene ü ҴǾ  ʴٸ ...
	if (!clax_scene) 
		return clax_err_notloaded;

	if ((node = (w_NODE *)malloc (sizeof (w_NODE))) == NULL)
		return clax_err_nomem;

	node->type = type;
	node->object = obj;
	node->next = NULL;

	// ũ Ʈ .
	if (!clax_scene->world) {
		node->prev = NULL;
		clax_scene->world = node;
		clax_scene->wtail = node;
	} 
	else {
	    node->prev = clax_scene->wtail;
		clax_scene->wtail->next = node;
		clax_scene->wtail = node;
	}

	return clax_err_ok;
}


//	add track to keyframer list.
int32 clax_add_track (int32 type, int32 id, int32 parent, void *track, void *obj)
{
	k_NODE *node, *pnode;

	if (!clax_scene) 
		return clax_err_notloaded;

	if ((node = (k_NODE*)malloc (sizeof (k_NODE))) == NULL)
		return clax_err_nomem;

	node->type	  = type;
	node->id	  = id;
	node->track	  = track;
	node->brother = NULL;
	node->child   = NULL;
	node->next    = NULL;
	node->object  = obj;

	if ( !clax_scene->keyframer ) {
		// Ʈ  .
		node->prev = NULL;
		clax_scene->keyframer = node;
		clax_scene->ktail = node;
	}
	else {
		node->prev = clax_scene->ktail;

		clax_scene->ktail->next = node;
		// ο Tail() ּ
		clax_scene->ktail = node;
	}

	// update hierarchy tree
	// Ʈ  Ѵ.
	if (parent != -1) { 

		for (pnode = clax_scene->keyframer; pnode; pnode = pnode->next) {

			// θ object Ѵ.
			if (pnode->id == parent) {
				node->parent = pnode;

				// θ ڽ Ʈ Ѵ.
				if (pnode->child == NULL)
					pnode->child = node;
				// ̹ ڽ   Ʈ Ѵ.
				else {
					node->brother = pnode->child;
					pnode->child = node;
				}
			}

		}

	}
	else 
		node->parent = NULL;

	return clax_err_ok;
}


// assign a track to keyframer node.
int32 clax_set_track (int32 type, int32 id, t_TRACK *track)
{
	// Ű  ͸ Ų.
	k_NODE *node = clax_scene->keyframer;
	void   *obj;

	if (!clax_scene || !clax_scene->keyframer)
		return clax_err_notloaded;

	while (node && node->id != id)
		node = node->next;

	if (!node) 
		return clax_err_undefined;

	obj = node->track;

	switch (node->type) {

		// Camera Track
		case clax_track_camera:
			switch (type) {
				case clax_key_pos:
					((t_CAMERA *)obj)->pos = track;
					break;

				case clax_key_fov: 
					((t_CAMERA *)obj)->fov = track; 
					break;

				case clax_key_roll:
					((t_CAMERA *)obj)->roll = track;
			}
			break;

		// Camera Target Track
		case clax_track_cameratgt:
			switch (type) {
				case clax_key_pos: 
					((t_CAMERATGT *)obj)->pos = track;
			}
			break;

		// Light Track
		case clax_track_light:

			switch (type) {
				case clax_key_pos:
					((t_LIGHT *)obj)->pos = track;
					break;

				case clax_key_color:
					((t_LIGHT *)obj)->color = track;
			}
			break;

		case clax_track_lighttgt:
			switch (type) {
				case clax_key_pos:
					((t_LIGHTTGT *)obj)->pos = track;
			}
			break;

		case clax_track_spotlight:
			switch (type) {
				case clax_key_pos:   
					((t_SPOTLIGHT *)obj)->pos = track; 
					break;

				case clax_key_color: 
					((t_SPOTLIGHT *)obj)->color = track; 
					break;

				case clax_key_roll:  
					((t_SPOTLIGHT *)obj)->roll = track;
			}
			break;

		// Object Track
		case clax_track_object:
			switch (type) {
				case clax_key_pos:
					((t_OBJECT *)obj)->translate = track;
					break;

				case clax_key_scale:  
					((t_OBJECT *)obj)->scale = track;
					break;

				case clax_key_rotate:
					((t_OBJECT *)obj)->rotate = track;
					break;

				case clax_key_morph:  
					((t_OBJECT *)obj)->morph = track; 
					break;

				case clax_key_hide:   
					((t_OBJECT *)obj)->hide = track;
			}
			break;

		case clax_track_ambient:
			switch (type) {
				case clax_key_color: 
					((t_AMBIENT *)obj)->color = track;
			}
	}

	return clax_err_ok;
}


/*****************************************************************************
  clax library (scene load/save/free functions)
*****************************************************************************/

// allocates memory for a new scene.
int32 clax_alloc_scene (c_SCENE **scene)
{
	// Ʈ  ҴѴ.
	if ((*scene = (c_SCENE *)malloc (sizeof (c_SCENE))) == NULL)
		return clax_err_nomem;

	(*scene)->f_start = 0;
	(*scene)->f_end = 0;
	(*scene)->f_current = 0;

	(*scene)->world = NULL;
	(*scene)->wtail = NULL;
	(*scene)->keyframer = NULL;
	(*scene)->ktail = NULL;
	
	D3DUtil_SetIdentityMatrix( (*scene)->Matrix );

	return clax_err_ok;
}


//	loads mesh data from file "filename" into scene "scene".
int32 clax_load_world (char *filename, c_SCENE *scene)
{
	int32   (*loader)(FILE *f) = NULL;			// Լ  
	c_SCENE *old_scene = clax_scene;
	FILE    *f;
	char    *s;
	int32    i, error;

	// Ȯڿ  д Լ Ѵ.
	for(i = 0; i < sizeof (clax_drivers) / sizeof (clax_drivers[0]); i++) {
		s = strchr (filename, '.') + 1;

		if(strucmp (s, clax_drivers[i].name) == 0)
		   loader = clax_drivers[i].load_mesh;
	}

	//  α׷  ʴ Ȯ ...
	if (!loader) 
		return clax_err_badformat;

	// ȭ  ϸ ...
	if ((f = fopen (filename, "rb")) == NULL) 
		return clax_err_nofile;

	clax_setactive_scene (scene);
	error = loader (f);
	fclose (f);

	if (error) {
		clax_setactive_scene (old_scene);
		clax_free_world (scene);
		return error;
	}

	// Ʈ üũڽ Ѵ.
	// calc_bbox ();

	// 븻 ͸  ʿ䰡 ִٸ...
	//if (clax_flags & clax_calcnorm) 
	//	calc_normals ();

	clax_setactive_scene (old_scene);
	return clax_err_ok;
}


//	loads motion data from file "filename" into scene "scene".
int32 clax_load_motion (char *filename, c_SCENE *scene)
{
	int32    (*loader)(FILE *f) = NULL;
	c_SCENE  *old_scene = clax_scene;
	FILE     *f;
	char     *s;
	int32     i, error;

	// Ʈ   ...
	if (!scene->world) 
		return clax_err_notloaded;

	for (i = 0; i < sizeof (clax_drivers) / sizeof (clax_drivers[0]); i++) {
		s = strchr (filename, '.') + 1;

		if (strucmp (s, clax_drivers[i].name) == 0)
			loader = clax_drivers[i].load_motion;
	}

	if (!loader) 
		return clax_err_badformat;

	if ((f = fopen (filename, "rb")) == NULL) 
		return clax_err_nofile;

	clax_setactive_scene (scene);
	error = loader (f);
	fclose (f);

	if (error) {
		clax_setactive_scene (old_scene);
		clax_free_motion (scene);
		return error;
	}

	clax_setframe (0);
	clax_update ();
	clax_setactive_scene (old_scene);

	return clax_err_ok;
}


/*
  clax_load_scene: loads mesh and keyframer data from file "filename"
                   into scene "scene".
*/
/*
int32 clax_load_scene (char *filename, c_SCENE *scene)
{

  int32 error;

  if ((error = clax_load_world (filename, scene)) != clax_err_ok)
    return error;
  return clax_load_motion (filename, scene);
}
*/


/*
  clax_save_scene: saves scene "scene" to filename "filename".
*/
/*
int32 clax_save_scene (char *filename, c_SCENE *scene)
{
  int32    (*loader)(FILE *f) = NULL;
  c_SCENE  *old_scene = clax_scene;
  FILE     *f;
  char     *s;
  int32     i, error;

  if (!scene || !scene->world) return clax_err_notloaded;
  for (i = 0; i < sizeof (clax_drivers) / sizeof (clax_drivers[0]); i++) {
    s = strchr (filename, '.') + 1;
    if (strucmp (s, clax_drivers[i].name) == 0)
      loader = clax_drivers[i].save_scene;
  }
  if (!loader) return clax_err_badformat;
  if ((f = fopen (filename, "wb")) == NULL) return clax_err_undefined;
  clax_setactive_scene (scene);
  error = loader (f);
  fclose (f);
  clax_setactive_scene (old_scene);
  if (error) return error;
  return clax_err_ok;
}
*/


//	release all memory used by world.
int32 clax_free_world (c_SCENE *scene)
{
	w_NODE     *node, *next;
	c_LIGHT    *light;
	c_CAMERA   *cam;
	c_OBJECT   *obj;
	c_MATERIAL *mat;
	c_AMBIENT  *amb;

	if (!scene || !scene->world) 
		return clax_err_nullptr;

	for (node = scene->world; node; node = next) {

		switch (node->type) {

			case clax_obj_light:
				light = (c_LIGHT *)node->object;
				free (light->name);
				break;

			case clax_obj_camera:
				cam = (c_CAMERA *)node->object;
				free (cam->name);
				break;

			case clax_obj_object:
				obj = (c_OBJECT *)node->object;
				free (obj->name);
				free (obj->vertices);
				free (obj->faces);
				free (obj->lpD3DVertex);

				LPFACEMATGROUP Group, Temp;

				Group = obj->lpFaceMatGroup;

				while ( Group ) {
					
					Temp = Group;
					Group = Group->lpNext;

					free(Temp);
				}
				obj->lpFaceMatGroup = NULL;
				break;

			// material ü Ѵ.
			case clax_obj_material:
				mat = (c_MATERIAL *)node->object;
				free (mat->name);
				free (mat->texture.file);
				free (mat->bump.file);
				free (mat->reflection.file);

				mat->lpD3DMaterial->Release();
				mat->lpD3DMaterial = NULL;
				break;

			case clax_obj_ambient:
				amb = (c_AMBIENT *)node->object;
				free (amb->name);
	    }

		next = node->next;
		free (node->object);
		free (node);
	}

	scene->world = NULL;
	scene->wtail = NULL;

	return clax_err_ok;
}


//	release all memory used by keyframer.
int32 clax_free_motion (c_SCENE *scene)
{
	k_NODE      *node, *next;
	t_CAMERA    *cam;
	t_CAMERATGT *camtgt;
	t_LIGHT     *light;
	t_LIGHTTGT  *litgt;
	t_SPOTLIGHT *spotl;
	t_OBJECT    *obj;
	t_AMBIENT   *amb;

	if (!scene || !scene->keyframer) 
		return clax_err_nullptr;

	for (node = scene->keyframer; node; node = next) {

		switch (node->type) {
			case clax_track_camera:
				cam = (t_CAMERA *)node->track;
				clax_free_track (cam->pos);
				clax_free_track (cam->fov);
				clax_free_track (cam->roll);
				break;

			case clax_track_cameratgt:
				camtgt = (t_CAMERATGT *)node->track;
				clax_free_track (camtgt->pos);
				break;

			case clax_track_light:
				light = (t_LIGHT *)node->track;
				clax_free_track (light->pos);
				clax_free_track (light->color);
				break;

			case clax_track_spotlight:
				spotl = (t_SPOTLIGHT *)node->track;
				clax_free_track (spotl->pos);
				clax_free_track (spotl->color);
				clax_free_track (spotl->roll);
				break;

			case clax_track_lighttgt:
				litgt = (t_LIGHTTGT *)node->track;
				clax_free_track (litgt->pos);
				break;

			case clax_track_object:
				obj = (t_OBJECT *)node->track;
				clax_free_track (obj->translate);
				clax_free_track (obj->scale);
				clax_free_track (obj->rotate);
				clax_free_track (obj->morph);
				clax_free_track (obj->hide);
				break;

			case clax_track_ambient:
				amb = (t_AMBIENT *)node->track;
				clax_free_track (amb->color);
		}

		next = node->next;
		free (node->track);
		free (node);
	}

	scene->keyframer = NULL;
	scene->ktail = NULL;
	return clax_err_ok;
}


//  release all memory used by meshes.
/*
int32 clax_free_mesh (c_SCENE *scene)
{
	w_NODE     *node;
	c_OBJECT   *obj;

	if (!scene || !scene->world || (clax_flags & clax_transform)) 
		return clax_err_nullptr;

	for (node = scene->world; node; node = node->next) {

		switch (node->type) {
			case clax_obj_object:
				obj = (c_OBJECT *)node->object;
				free (obj->vertices); 
				obj->vertices = NULL;

				free (obj->faces);    
				obj->faces = NULL;
				break;
		}

	}

	return clax_err_ok;
}
*/


// 𵨸 Űִϸ̼ Ʈ Ѵ.
int32 clax_free_scene (c_SCENE *scene)
{
	int32 error;

	if ((error = clax_free_world (scene)) != clax_err_ok) 
		return error;

	return clax_free_motion (scene);
}


/*****************************************************************************
  clax library (debugging)
*****************************************************************************/

//	output world information to stdout.
void clax_print_world ()
{
	w_NODE      *node;
	c_CAMERA    *cam;
	c_LIGHT     *light;
	c_OBJECT    *obj;
	c_MATERIAL  *mat;
	c_AMBIENT   *amb;
	int32        i;
	FILE		*fp;

	fp = fopen( "WORLD.TXT", "wt" );

	if (!clax_scene) {
		fprintf (fp, "error: scene not loaded.\n");
		return;
	}

	if (!clax_scene->world) {
		fprintf (fp, "error: null world tree.\n");
		return;
	}

	fprintf (fp, "<<start of world tree>>\n\n");

	for (node = clax_scene->world; node; node = node->next) {

		switch (node->type) {

			case clax_obj_camera:
				cam = (c_CAMERA *)node->object;
				fprintf (fp, "<<camera>> name: %s, id: %d\n", cam->name, cam->id);
				fprintf (fp, "  parents:     %d, %d\n", cam->parent1, cam->parent2);
				fprintf (fp, "  position:    x: %9.3f, y: %9.3f, z: %9.3f\n",
						 cam->pos.x, cam->pos.y, cam->pos.z);

				fprintf (fp, "  target:      x: %9.3f, y: %9.3f, z: %9.3f\n",
						 cam->target.x, cam->target.y, cam->target.z);

				fprintf (fp, "  fov:            %9.3f\n", cam->fov);
				fprintf (fp, "  roll:           %9.3f\n\n", cam->roll);
				break;

			case clax_obj_light:
				light = (c_LIGHT *)node->object;

				if (light->flags == clax_light_omni) {
					fprintf (fp, "<<omni light>> name: %s, id: %d\n", light->name, light->id);
					fprintf (fp, "  parent:      %d\n", light->parent1);

					fprintf (fp, "  position:    x: %9.3f, y: %9.3f, z: %9.3f\n",
							 light->pos.x, light->pos.y, light->pos.z);

					fprintf (fp, "  color:       r: %9.3f, g: %9.3f, b: %9.3f\n\n",
							 light->color.r, light->color.g, light->color.b);
				} 
				else {
					fprintf (fp, "<<spot light>> name: %s, id: %d\n", light->name, light->id);
					fprintf (fp, "  parents:     %d, %d\n", light->parent1, light->parent2);

					fprintf (fp, "  position:    x: %9.3f, y: %9.3f, z: %9.3f\n",
							 light->pos.x, light->pos.y, light->pos.z);

					fprintf (fp, "  target:      x: %9.3f, y: %9.3f, z: %9.3f\n",
							 light->target.x, light->target.y, light->target.z);

					fprintf (fp, "  color:       r: %9.3f, g: %9.3f, b: %9.3f\n",
							 light->color.r, light->color.g, light->color.b);

					fprintf (fp, "  roll:           %9.3f\n", light->roll);
					fprintf (fp, "  hotspot:        %9.3f\n", light->hotspot);
					fprintf (fp, "  falloff:        %9.3f\n\n", light->falloff);
				}
				break;

			case clax_obj_object:
				obj = (c_OBJECT *)node->object;

				fprintf (fp, "<<object>> name: %s, id: %d\n", obj->name, obj->id);
				fprintf (fp, "  parent:      %d\n", obj->parent);
				fprintf (fp, "  vertices:    %d, faces: %d\n", obj->numverts, obj->numfaces);

				fprintf (fp,"  pivot:       x: %9.3f, y: %9.3f, z: %9.3f\n",
						 obj->pivot.x, obj->pivot.y, obj->pivot.z);

				fprintf (fp, "  translate:   x: %9.3f, y: %9.3f, z: %9.3f\n",
						 obj->translate.x, obj->translate.y, obj->translate.z);

				fprintf (fp, "  scale:       x: %9.3f, y: %9.3f, z: %9.3f\n",
						 obj->scale.x, obj->scale.y, obj->scale.z);

				fprintf (fp, "  rotate:      x: %9.3f, y: %9.3f, z: %9.3f, w: %9.3f\n",
						 obj->rotate.x, obj->rotate.y, obj->rotate.z, obj->rotate.w);

				fprintf (fp, "  flags:       ");

				if (obj->flags) {
					if (obj->flags & clax_obj_hidden) 
						fprintf (fp, "hidden ");

					if (obj->flags & clax_obj_chidden) 
						fprintf (fp, "display-hidden ");

					if (obj->flags & clax_obj_dummy) 
						fprintf (fp, "dummy ");

					if (obj->flags & clax_obj_morph) 
						fprintf (fp, "morph ");
				}
				else 
					fprintf (fp, "none");

				fprintf(fp, "\n  matrix:\n");

				fprintf(fp, "    xx: %9.3f xy: %9.3f xz: %9.3f xw: %9.3f\n",
						 obj->matrix[X][X], obj->matrix[X][Y],
						 obj->matrix[X][Z], obj->matrix[X][W]);

				fprintf(fp, "    yx: %9.3f yy: %9.3f yz: %9.3f yw: %9.3f\n",
						 obj->matrix[Y][X], obj->matrix[Y][Y],
						 obj->matrix[Y][Z], obj->matrix[Y][W]);

				fprintf(fp, "    zx: %9.3f zy: %9.3f zz: %9.3f zw: %9.3f\n",
						 obj->matrix[Z][X], obj->matrix[Z][Y],
						 obj->matrix[Z][Z], obj->matrix[Z][W]);

				fprintf(fp, "\n");

#ifdef CLAX_DEBUG

				fprintf (fp, "  vertex list:\n");

				for(i = 0; i < obj->numverts; i++) {
					fprintf (fp, "    x: %9.3f, y: %9.3f, z: %9.3f, u: %4.3f, v: %4.3f\n",
							 obj->vertices[i].vert.x,
							 obj->vertices[i].vert.y,
							 obj->vertices[i].vert.z,
							 obj->vertices[i].u, obj->vertices[i].v);
				}
				fprintf (fp, "  face list:\n");

				/*
				for(i = 0; i < obj->numfaces; i++) {
					fprintf (fp, "    a: %3d b: %3d c: %3d, flags %3d, material: %3d\n",
							 obj->faces[i].a, obj->faces[i].b, obj->faces[i].c,
							 obj->faces[i].flags, obj->faces[i].mat);
				}
				*/
				for(i = 0; i < obj->numfaces; i++) {
					fprintf ( fp, "    a: %3d b: %3d c: %3d, flags %3d\n",
							  obj->faces[i].a, obj->faces[i].b, obj->faces[i].c,
							  obj->faces[i].flags );
				}
#endif
				fprintf (fp, "\n");
				break;

			case clax_obj_material:
				mat = (c_MATERIAL *)node->object;

				fprintf(fp, "<<material>> name: %s, id: %d\n", mat->name, mat->id);
				fprintf(fp, "  shading:     ");

				switch (mat->shading) {
					case clax_mat_flat: 
						fprintf (fp, "flat"); 
						break;

					case clax_mat_gouraud: 
						fprintf (fp, "gouraud");
						break;

					case clax_mat_phong: 
						fprintf (fp, "phong"); 
						break;

					case clax_mat_metal: 
						fprintf (fp, "metal");
				}
				fprintf (fp, "\n  flags:       ");

				if (mat->flags) {
					if (mat->flags & clax_mat_twosided) 
						fprintf (fp, "twosided ");

					if (mat->flags & clax_mat_soften) 
						fprintf (fp, "soften ");

					if (mat->flags & clax_mat_wire) 
						fprintf (fp, "wireframe ");

					if (mat->flags & clax_mat_transadd) 
						fprintf (fp, "transparency add ");
				}
				else 
					fprintf(fp, "none ");

				fprintf(fp, "\n  ambient:            r: %9.3f, g: %9.3f, b: %9.3f\n",
						mat->Material.ambient.r, 
						mat->Material.ambient.g, 
						mat->Material.ambient.b);

				fprintf(fp, "  diffuse:            r: %9.3f, g: %9.3f, b: %9.3f\n",
						mat->Material.diffuse.r, 
						mat->Material.diffuse.g, 
						mat->Material.diffuse.b);

				fprintf(fp, "  specular:           r: %9.3f, g: %9.3f, b: %9.3f\n",
						mat->Material.specular.r, 
						mat->Material.specular.g, 
						mat->Material.specular.b);

				//fprintf(fp, "  shininess:             %9.3f\n", mat->shininess);
				fprintf(fp, "  shininess:             %9.3f\n", mat->Material.power);

				fprintf(fp, "  shininess strength:    %9.3f\n", mat->shin_strength);
				fprintf(fp, "  transparency:          %9.3f\n", mat->transparency);
				fprintf(fp, "  trans. falloff:        %9.3f\n", mat->trans_falloff);
				fprintf(fp, "  reflection blur:       %9.3f\n", mat->refblur);
				fprintf(fp, "  self illumination:     %9.3f\n", mat->self_illum);

				print_map(&mat->texture, "texture map:\n", fp);
				print_map(&mat->bump, "bump map:\n", fp);
				print_map(&mat->reflection, "reflection map:\n", fp);

				fprintf(fp, "\n");
				break;

			case clax_obj_ambient:
				amb = (c_AMBIENT *)node->object;
				fprintf (fp, "<<ambient>> name: %s, id: %d\n", amb->name, amb->id);
				fprintf (fp, "  color:       r: %9.3f, g: %9.3f, b: %9.3f\n\n",
						 amb->color.r, amb->color.g, amb->color.b);
		}
	}

	fprintf (fp, "<<end of world tree>>\n");
	fclose(fp);
}


//	output keyframer information to stdout.
void clax_print_keyframer ()
{
	k_NODE      *node;
	t_CAMERA    *cam;
	t_CAMERATGT *tcam;
	t_LIGHT     *light;
	t_LIGHTTGT  *tlight;
	t_SPOTLIGHT *spotl;
	t_OBJECT    *obj;
	t_AMBIENT   *amb;
	FILE		*fp;

	fp = fopen( "KEY.TXT", "wt" );

	if (!clax_scene) {
		fprintf (fp, "error: scene not loaded.\n");
		return;
	}

	if (!clax_scene->keyframer) {
		fprintf (fp, "error: null keyframer tree.\n");
		return;
	}

	fprintf (fp, "<<start of keyframer tree>>\n\n");
	fprintf (fp, "  start frame: %f\n", clax_scene->f_start);
	fprintf (fp, "  end frame:   %f\n\n", clax_scene->f_end);

	for (node = clax_scene->keyframer; node; node = node->next) {

		switch (node->type) {

			case clax_track_camera:
				cam = (t_CAMERA *)node->track;
				fprintf (fp, "<<camera source>> id: %d, name: %s\n", node->id,
						((c_CAMERA *)node->object)->name);

				print_track (cam->pos,  "  position keys:", 1, fp);
				print_track (cam->roll, "  roll keys:    ", 0, fp);
				print_track (cam->fov,  "  fov keys:     ", 0, fp);

				fprintf (fp, "\n");
				break;

			case clax_track_cameratgt:
				tcam = (t_CAMERATGT *)node->track;
				fprintf (fp, "<<camera target>> id: %d, name: %s\n", node->id,
						((c_CAMERA *)node->object)->name);

				print_track (tcam->pos, "  position keys:", 1, fp);
				fprintf (fp, "\n");
				break;

			case clax_track_light:
				light = (t_LIGHT *)node->track;
				fprintf (fp, "<<light source>> id: %d, name: %s\n", node->id,
						((c_LIGHT *)node->object)->name);

				print_track (light->pos,   "  position keys:", 1, fp);
				print_track (light->color, "  color keys:   ", 2, fp);
				fprintf (fp, "\n");
				break;

			case clax_track_spotlight:
				spotl = (t_SPOTLIGHT *)node->track;
				fprintf (fp, "<<spotlight source>> id: %d, name: %s\n", node->id,
						((c_LIGHT *)node->object)->name);

				print_track (spotl->pos,   "  position keys:", 1, fp);
				print_track (spotl->color, "  color keys:   ", 2, fp);
				print_track (spotl->roll,  "  roll keys:    ", 0, fp);
				fprintf (fp, "\n");
				break;

			case clax_track_lighttgt:
				tlight = (t_LIGHTTGT *)node->track;
				fprintf (fp, "<<light target>> id: %d, name: %s\n", node->id,
						((c_LIGHT *)node->object)->name);

				print_track (tlight->pos, "  position keys:", 1, fp);
				fprintf (fp, "\n");
				break;

			case clax_track_object:
				obj = (t_OBJECT *)node->track;
				fprintf (fp, "<<object track>> id: %d, name: %s\n", node->id,
						((c_OBJECT *)node->object)->name);

				print_track (obj->translate, "  position keys:", 1, fp);
				print_track (obj->rotate,    "  rotation keys:", 3, fp);
				print_track (obj->scale,     "  scale keys:   ", 1, fp);
				print_track (obj->morph,     "  morph keys:   ", 4, fp);
				print_track (obj->hide,      "  hide keys:    ", 5, fp);
				fprintf (fp, "\n");
				break;

			case clax_track_ambient:
				amb = (t_AMBIENT *)node->track;
				fprintf (fp ,"<<ambient track>> id: %d, name: %s\n", node->id,
						((c_AMBIENT *)node->object)->name);

				print_track (amb->color, "  color keys:   ", 2, fp);
				fprintf (fp, "\n");
		}
	}

	fprintf (fp, "<<end of keyframer tree>>\n");
	fclose(fp);
}


/*****************************************************************************
  clax library (track handling)
*****************************************************************************/

/*
  clax_getkey_float: return float key at frame "frame".
*/
/*
int32 clax_getkey_float (t_TRACK *track, float frame, float *out)
{
	t_KEY *keys;

	if (frame < 0.0) 
		return clax_err_badframe;

	if (!track || !track->keys) 
		return clax_err_nullptr;

	keys = track->keys;

	if (track->flags != 0) 
		frame = fmod (frame, track->frames);

	if (!keys->next || frame < keys->frame) {
		*out = keys->val._float;
		return clax_err_ok;
	}

	// more than one key, spline interpolation
	return spline_getkey_float (track, frame, out);
}
*/


//	return vector key at frame "frame".
int32 clax_getkey_vect (t_TRACK *track, float frame, c_VECTOR *out)
{
	t_KEY *keys;

	if (frame < 0.0f) 
		return clax_err_badframe;

	if (!track || !track->keys) 
		return clax_err_nullptr;

	keys = track->keys;

	if (track->flags != 0)
		frame = (float)fmod (frame, track->frames);

	if (!keys->next || frame < keys->frame) {
		vec_copy (&keys->val._vect, out);
		return clax_err_ok;
	}

	// more than one key, spline interpolation
	return spline_getkey_vect (track, frame, out);
}


//  return rgb key at frame "frame".
int32 clax_getkey_rgb (t_TRACK *track, float frame, c_RGB *out)
{
	c_VECTOR *vect = (c_VECTOR *)out;

	return spline_getkey_vect (track, frame, vect);
}


//	return quaternion key at frame "frame".
int32 clax_getkey_quat (t_TRACK *track, float frame, c_QUAT *out)
{
	t_KEY *keys;
	float  alpha;

	if (frame < 0.0f) 
		return clax_err_badframe;

	if (!track || !track->keys) 
		return clax_err_nullptr;

	keys = track->keys;

	if (track->flags != 0) 
		frame = (float)fmod (frame, track->frames);

	//if (clax_flags & clax_slerp) {

		if (frame < track->last->frame) 
			keys = track->keys; 
		else
			keys = track->last;

		while (keys->next && frame > keys->next->frame) 
			keys = keys->next;

		track->last = keys;

		if (!keys->next || frame < keys->frame) {
			qt_copy (&keys->qa, out);
			return clax_err_ok;
		}

		alpha = (frame - keys->frame) / (keys->next->frame - keys->frame);
		alpha = spline_ease (alpha, keys->easefrom, keys->next->easeto);
		qt_slerp (&keys->qa, &keys->next->qa, 0, alpha, out);

		return clax_err_ok;
	//} 
	/*
	else {
		if (!keys->next || frame < keys->frame) {
			qt_copy (&keys->qa, out);
			return clax_err_ok;
		}

		return spline_getkey_quat (track, frame, out);
	}
	*/
}


//  return hide key at frame "frame".
int32 clax_getkey_hide (t_TRACK *track, float frame, int32 *out)
{
	t_KEY *keys;

	if (frame < 0.0f) 
		return clax_err_badframe;

	if (!track || !track->keys) {
		*out = 0;
		return clax_err_nullptr;
	}

	keys = track->keys;

	if (track->flags != 0) 
		frame = (float)fmod (frame, track->frames);

	if (frame < keys->frame) {
		*out = 0;
		return clax_err_ok;
	}

	if (frame < track->last->frame) 
		keys = track->keys; 
	else
		keys = track->last;

	while (keys->next && frame > keys->next->frame) 
		keys = keys->next;

	track->last = keys;
	*out = keys->val._int;

	return clax_err_ok;
}


//	return morph key at frame "frame".
int32 clax_getkey_morph (t_TRACK *track, float frame, c_MORPH *out)
{
	t_KEY *keys;
	float  alpha;

	if (frame < 0.0f)
		return clax_err_badframe;

	if (!track || !track->keys)
		return clax_err_nullptr;

	if (frame < track->last->frame)
		keys = track->keys;
	else
		keys = track->last;

	while (keys->next && frame > keys->next->frame) 
		keys = keys->next;

	track->last = keys;

	if (!keys->next || frame < keys->frame) {
		out->from  = keys->prev->val._int;
		out->to	   = keys->val._int;
		out->alpha = 1.0f;
		return clax_err_ok;
	}

	out->from = keys->val._int;
	out->to   = keys->next->val._int;
	alpha     = (frame - keys->frame) / (keys->next->frame - keys->frame);

	out->alpha = spline_ease (alpha, keys->easefrom, keys->next->easeto);
	return clax_err_ok;
}


/*****************************************************************************
  clax library (keyframer)
*****************************************************************************/

/*
  clax_collide: collision detection (using bounding box).
*/
/*
int32 clax_collide (w_NODE *a, w_NODE *b, int32 *result)
{

  c_OBJECT *o, *obj;
  c_CAMERA *cam;
  c_LIGHT  *light;
  int32     type;

  if (a->type != clax_obj_object && b->type != clax_obj_object)
    return clax_err_invparam;
  if (a->type == clax_obj_object) {
    o = (c_OBJECT *)a->object;
    obj = (c_OBJECT *)b->object;
    cam = (c_CAMERA *)b->object;
    light = (c_LIGHT *)b->object;
    type = b->type;
  } else {
    o = (c_OBJECT *)b->object;
    obj = (c_OBJECT *)a->object;
    cam = (c_CAMERA *)a->object;
    light = (c_LIGHT *)a->object;
    type = a->type;
  }
  switch (type) {
    case clax_obj_object:
      *result = (clax_insidebox (&o->pbbox, &obj->pbbox.min) ||
                 clax_insidebox (&o->pbbox, &obj->pbbox.max) ||
                 clax_insidebox (&obj->pbbox, &o->pbbox.min) ||
                 clax_insidebox (&obj->pbbox, &o->pbbox.max));
      return clax_err_ok;
    case clax_obj_camera:
      *result = (clax_insidebox (&o->pbbox, &cam->pos) ||
                 clax_insidebox (&o->pbbox, &cam->target));
      return clax_err_ok;
    case clax_obj_light:
      if (light->flags == clax_light_spot)
        *result = (clax_insidebox (&o->pbbox, &light->pos) ||
                   clax_insidebox (&o->pbbox, &light->target));
      else
        *result = (clax_insidebox (&o->pbbox, &light->pos));
      return clax_err_ok;
    case clax_obj_material:
    case clax_obj_ambient: return clax_err_invparam;
  }
  return clax_err_undefined;
}
*/


// update all keyframer data.
int32 clax_update()
{
	k_NODE      *node, *child;
	//c_CAMERA    *cam;
	//t_CAMERA    *tcam;
	//t_CAMERATGT *tcamt;
	//c_LIGHT     *light;
	//t_LIGHT     *tlight;
	//t_SPOTLIGHT *tspotl;
	//t_LIGHTTGT  *tlightt;
	c_AMBIENT   *amb;
	t_AMBIENT   *tamb;
	c_OBJECT    *obj,*cobj;
	t_OBJECT    *tobj;
	
	c_MATRIX     c, d;
	int32        hidden;
	float        frame = clax_scene->f_current;


	if (!clax_scene) 
		return clax_err_notloaded;

	if (!clax_scene->world || !clax_scene->keyframer)
		return clax_err_notloaded;

	// update objects
	for (node = clax_scene->keyframer; node; node = node->next) {

		switch (node->type) {
			//	Camera Track
			//case clax_track_camera:
			//	cam = (c_CAMERA *)node->object;
			//	tcam = (t_CAMERA *)node->track;
			//	clax_getkey_vect (tcam->pos, frame, &cam->pos);
			//	clax_getkey_float (tcam->fov, frame, &cam->fov);
			//	clax_getkey_float (tcam->roll, frame, &cam->roll);
			//	break;

			//case clax_track_cameratgt:
			//	cam = (c_CAMERA *)node->object;
			//	tcamt = (t_CAMERATGT *)node->track;
			//	clax_getkey_vect (tcamt->pos, frame, &cam->target);
			//	break;
			//

			/*
			case clax_track_light:
				light = (c_LIGHT *)node->object;
				tlight = (t_LIGHT *)node->track;
				clax_getkey_vect (tlight->pos, frame, &light->pos);
				clax_getkey_rgb (tlight->color, frame, &light->color);
				break;
			*/

			/*
			case clax_track_spotlight:
				light = (c_LIGHT *)node->object;
				tspotl = (t_SPOTLIGHT *)node->track;
				clax_getkey_vect (tspotl->pos, frame, &light->pos);
				clax_getkey_rgb (tspotl->color, frame, &light->color);
				clax_getkey_float (tspotl->roll, frame, &light->roll);
				break;

			case clax_track_lighttgt:
				light = (c_LIGHT *)node->object;
				tlightt = (t_LIGHTTGT *)node->track;
				clax_getkey_vect (tlightt->pos, frame, &light->target);
				break;
			*/

			// Object Track
			case clax_track_object:
				obj = (c_OBJECT *)node->object;
				tobj = (t_OBJECT *)node->track;

				clax_getkey_vect (tobj->translate, frame, &obj->translate);
				clax_getkey_vect (tobj->scale, frame, &obj->scale);
				clax_getkey_quat (tobj->rotate, frame, &obj->rotate);
				clax_getkey_hide (tobj->hide, frame, &hidden);

				if (clax_getkey_morph (tobj->morph, frame, &obj->morph) == clax_err_ok)
					obj->flags |= clax_obj_morph;
				else 
					obj->flags &= ~clax_obj_morph;

				if (hidden) 
					obj->flags |= clax_obj_hidden;
				else
					obj->flags &= ~clax_obj_hidden;

				// ִϸ̼  Ѵ.
				qt_invmatrix (&obj->rotate, c);
				mat_setscale (&obj->scale, d);

				c[X][W] = obj->translate.x;
				c[Y][W] = obj->translate.y;
				c[Z][W] = obj->translate.z;
				mat_mul (c, d, obj->matrix);
				break;

			case clax_track_ambient:
				amb = (c_AMBIENT *)node->object;
				tamb = (t_AMBIENT *)node->track;
				clax_getkey_rgb (tamb->color, frame, &amb->color);
		}
	}

	// update hierarchy
	if (clax_flags & clax_hierarchy) {

		for (node = clax_scene->keyframer; node; node = node->next) {

			if (node->type == clax_track_object) {
				for (child = node->child; child; child = child->brother) {

					if (child->type == clax_track_object) {
						obj = (c_OBJECT *)node->object;
						cobj = (c_OBJECT *)child->object;

						mat_mul (obj->matrix, cobj->matrix, cobj->matrix);
					}

					//
					//if (child->type == clax_track_camera) {
					//	obj = (c_OBJECT *)node->object;
					//	cam = (c_CAMERA *)child->object;
					//	mat_mulvec (obj->matrix, &cam->pos, &cam->pos);
					//}

					//if (child->type == clax_track_cameratgt) {
					//	obj = (c_OBJECT *)node->object;
					//	cam = (c_CAMERA *)child->object;
					//	mat_mulvec (obj->matrix, &cam->target, &cam->target);
					//}
					//

					/*
					if (child->type == clax_track_light) {
						obj = (c_OBJECT *)node->object;
						light = (c_LIGHT *)child->object;
						mat_mulvec (obj->matrix, &light->pos, &light->pos);
					}

					if (child->type == clax_track_spotlight) {
						obj = (c_OBJECT *)node->object;
						light = (c_LIGHT *)child->object;
						mat_mulvec (obj->matrix, &light->pos, &light->pos);
					}

					if (child->type == clax_track_lighttgt) {
						obj = (c_OBJECT *)node->object;
						light = (c_LIGHT *)child->object;
						mat_mulvec (obj->matrix, &light->target, &light->target);
					}
					*/
				}
			}

			//
			//if (node->type == clax_track_camera) {
			//	for (child = node->child; child; child = child->brother) {

			//		if (child->type == clax_track_light) {
			//			cam = (c_CAMERA *)node->object;
			//			light = (c_LIGHT *)child->object;
			//			vec_add (&cam->pos, &light->pos, &light->pos);
			//		}

			//		if (child->type == clax_track_spotlight) {
			//			cam = (c_CAMERA *)node->object;
			//			light = (c_LIGHT *)child->object;
			//			vec_add (&cam->pos, &light->pos, &light->pos);
			//		}
			//	}
			//}

		}
	}

	// update camera matrices
	//
	//for (node = clax_scene->keyframer; node; node = node->next) {

	//    if (node->type == clax_track_camera) {
	//		cam = (c_CAMERA *)node->object;
	//		cam_update (cam);
	//	}
	//}

	// do transformation if neccesary
	if (clax_flags & clax_transform) 
		do_transform ();

	return clax_err_ok;
}
