/* CLAX.H**********************************************************************

    Clax: Portable keyframing library, version 0.10
          see inclosed LICENSE.TXT for licensing terms.

          for documentation, refer to CLAX.TXT

    author           : BoRZoM
    file created     : 16/04/97
    file description : clax include file

	Revision by		 : Jeon Min Woo

*****************************************************************************/

#ifndef _CLAX_H_
#define _CLAX_H_

#define D3D_OVERLOADS
#include <d3d.h>

#define CLAX_VERSION 0.10

#ifndef M_PI
#define M_PI 3.14159265359f
#endif

// 1e-6f = 0   Ҽڸ 0 6° ڸ 1 Ѵ.
// , 0.000001 (epsilon)  Ǵ ̴.
// if( fLength < 1e-6f )
#define	EPSILON 1.0e-6
#define X		0
#define Y		1
#define Z		2
#define W		3

/*****************************************************************************
  type definitions
*****************************************************************************/

typedef signed short int16;
typedef signed long  int32;

/*****************************************************************************
  error codes
*****************************************************************************/

// error codes
enum clax_error_ {

	clax_err_ok        = 0,                // no error                       
	clax_err_nomem     = 1,                // not enough memory              
	clax_err_nofile    = 2,                // file not found                 
	clax_err_badfile   = 3,                // corrupted file                 
	clax_err_badver    = 4,                // unsupported version            
	clax_err_badformat = 5,                // unsupported format             
	clax_err_badframe  = 6,                // illegal frame number           
	clax_err_badname   = 7,                // illegal object name            
	clax_err_noframes  = 8,                // no keyframer data              
	clax_err_notloaded = 9,                // no scene loaded                
	clax_err_nullptr   = 10,               // null pointer assignment        
	clax_err_invparam  = 11,               // invalid parameter              
	clax_err_spline    = 12,               // less than 2 keys in spline     
	clax_err_singular  = 13,               // cannot inverse singular matrix 
	clax_err_badid     = 14,               // bad object id number           
	clax_err_exist     = 15,               // object already exist           
	clax_err_undefined = 256               // internal error                 

};

/*****************************************************************************
  track/object types
*****************************************************************************/

// object types
enum clax_object_ {

	clax_obj_camera    = 1,                // camera (c_CAMERA)     
	clax_obj_light     = 2,                // light (c_LIGHT)       
	clax_obj_object    = 4,                // object (c_OBJECT)     
	clax_obj_material  = 8,                // material (c_MATERIAL) 
	clax_obj_ambient   = 16                // ambient (c_AMBIENT)

};


// track types
enum clax_track_ {

	clax_track_camera    = 1,              // camera track        
	clax_track_cameratgt = 2,              // camera target track 
	clax_track_light     = 4,              // light track         
	clax_track_spotlight = 8,              // spotlight track     
	clax_track_lighttgt  = 16,             // light target track  
	clax_track_object    = 32,             // object track        
	clax_track_ambient   = 64              // ambient track       

};


// key types
enum clax_key_ {

	clax_key_pos      = 1,                 // position track 
	clax_key_rotate   = 2,                 // rotation track 
	clax_key_scale    = 3,                 // scale track    
	clax_key_fov      = 4,                 // fov track      
	clax_key_roll     = 5,                 // roll track     
	clax_key_color    = 6,                 // color track    
	clax_key_morph    = 7,                 // morph track    
	clax_key_hide     = 8,                 // hide track

};

/*****************************************************************************
  flags
*****************************************************************************/

// clax flags
enum clax_flags_ {

	clax_hierarchy  = 1,                   // hierarchical transformations 
	clax_calcnorm   = 2,                   // calculate normals            
	clax_domorph    = 4,                   // internal object morph        
	clax_transform  = 8,                   // internal transformations     
	clax_slerp      = 16,                  // quaternion spherical interp. 
	clax_normlerp   = 32,                  // morph: lerp normals          
	clax_normcalc   = 64                   // morph: recalculate normals

};


enum clax_mat_flags_ { /* clax material flags */
  clax_mat_twosided = 1,                 /* two sided           */
  clax_mat_soften   = 2,                 /* soften              */
  clax_mat_wire     = 4,                 /* wireframe rendering */
  clax_mat_transadd = 8                  /* transparency add    */
};

enum clax_map_flags_ { /* clax map flags */
  clax_map_mirror   = 2,                 /* mirror   */
  clax_map_negative = 8                  /* negative */
};

enum clax_shade_flags_ { /* clax material shading */
	clax_mat_flat    = 1,                  /* flat shading    */
	clax_mat_gouraud = 2,                  /* gouraud shading */
	clax_mat_phong   = 3,                  /* phong shading   */
	clax_mat_metal   = 4                   /* metal shading   */
};


// clax face flags
enum clax_face_flags_ { 

	clax_face_wrapU   = 1,                 // face has texture wrapping (u)
	clax_face_wrapV   = 2,                 // face has texture wrapping (v)
	clax_face_visible = 4                  // visible flag (backface cull)

};


// clax object flags
enum clax_obj_flags_ { 

	clax_obj_hidden  = 1,                   // object is hidden
	clax_obj_chidden = 2,                   // object is always hidden
	clax_obj_dummy   = 4,                   // object is dummy
	clax_obj_morph   = 8                    // object is morphing

};


// clax light flags
enum clax_light_flags_ {

	clax_light_omni = 1,                    // light is omni
	clax_light_spot = 2                     // light is spotlight

};


enum clax_track_flags_ { /* clax track flags */
	clax_track_repeat = 1,                  /* track repeat */
	clax_track_loop   = 2                   /* track loop   */
};

/*****************************************************************************
  world structures
*****************************************************************************/

//typedef float c_MATRIX[4][4];
typedef float c_MATRIX[3][4];

typedef struct _c_VECTOR {

	float	x, y, z;

} c_VECTOR;


typedef struct _c_RGB {

	float	r, g, b;

} c_RGB;


// quaternion
typedef struct _c_QUAT { 

	float w, x, y, z;                      // quaternion (w,[x,y,z])

} c_QUAT;


// bounding box
typedef struct _c_BOUNDBOX {

	D3DVECTOR min;                         // bounding box 
	D3DVECTOR max;                         // bounding box 

} c_BOUNDBOX;


// morph struct
typedef struct _c_MORPH { 

	int32	from, to;                        // morph: from/to object 
	float	alpha;                           // morph stage (0 -> 1.0)

} c_MORPH;


// map struct
typedef struct _c_MAP { 

	char	*file;                           // map filename   
	int32	flags;                           // map flags      
	float	U_scale, V_scale;                // 1/U, 1/V scale 
	float	U_offset, V_offset;              // U, V offset    
	float	rot_angle;                       // rotation angle 

} c_MAP;


// material struct
typedef struct _c_MATERIAL { 

	char				*name;                           // material name        
	int32				id;                              // material id          
	int32				shading, flags;                  // shading, flags       

	float				shin_strength;                   // snininess strength   
	float				transparency;                    // transparency         
	float				trans_falloff;                   // transparency falloff 
	float				refblur;                         // reflection blur      
	float				self_illum;                      // self illuminance     

	c_MAP				texture;                         // texture map          
	c_MAP				bump;                            // bump map             
	c_MAP				reflection;                      // reflection map       

	LPDIRECT3DMATERIAL3	lpD3DMaterial;
	D3DMATERIAL			Material;
	D3DMATERIALHANDLE	Handle;

} c_MATERIAL;


// vertex struct
typedef struct _c_VERTEX { 

	c_VECTOR	vert, pvert;                 // vertex
	c_VECTOR	norm, pnorm;                 // vertex normal
	float		u, v;                        // texture coordinates

} c_VERTEX;


// face struct
typedef struct _c_FACE { 

	WORD		a, b, c;                      // vertices of triangle
	int32		flags;                        // face flags: clax_face_*
	// int32		mat;                      // ace material
	c_VERTEX	*pa, *pb, *pc;                // pointers to vertices
	// c_MATERIAL	*pmat;                    // pointer to material
	c_VECTOR	norm, pnorm;                  // face normal

} c_FACE;


typedef struct _FACEMATGROUP {

	char			szName[100];
	WORD			NumTriangles;
	LPWORD			lpTriangles;
	c_MATERIAL		*lpMaterial;
	_FACEMATGROUP	*lpNext;		// pointer to next trimesh mat group

} FACEMATGROUP, *LPFACEMATGROUP;


// light struct
typedef struct _c_LIGHT { 

	char			*name;                        // light name
	int32			id, parent1, parent2;         // object id, parents
	int32			flags;                        // light flags

	//D3DVECTOR		pos, target;                  // light position
	c_VECTOR		pos, target;                  // light position

	float			roll;                         // roll (spotlight)
	float			hotspot, falloff;             // hotspot, falloff
	D3DCOLORVALUE	color;                        // light color

} c_LIGHT;


// camera struct
typedef struct _c_CAMERA {

	char		*name;                        // camera name
	int32		id, parent1, parent2;         // object id, parent

	D3DVECTOR	pos, target;                  // source, target vectors
	//c_VECTOR	pos, target;                  // source, target vectors

	float		Lens, fov, roll;              // field of view, roll
	float		sizeX, sizeY;                 // screen size (set by user)
	float		AspectRatio;                  // aspect ratio (set by user)
	float		perspX, perspY;               // perspective multipliers
	D3DMATRIX	matrix;                       // camera matrix

} c_CAMERA;

/*
typedef struct _CAMERA {

	VECTOR		dvPosition;       	// Position in world space 
	VECTOR		dvDirection;    	// Direction in world space 

    float		bank;				// Camera bank ( rotation angle )
    float		lens;				// Camera lens

	float		nearPlane;			// Camera near plane
	float		farPlane;			// Camera far plane

	char    	szName[100];
	_CAMERA     *lpNext;

} CAMERA, *LPCAMERA;
*/


// object struct
typedef struct _c_OBJECT {

	char				*name;                    // object name
	int32				id, parent;               // object id, object parent

	int32				numverts;                 // number of vertices
	WORD				numfaces;                 // number of faces

	WORD				NumMatFaces;
	int32				flags;                    // object flags: clax_obj_*
	c_VERTEX			*vertices;                // object vertices
	c_FACE				*faces;                   // object faces

	LPD3DVERTEX			lpD3DVertex;
	LPFACEMATGROUP		lpFaceMatGroup;
	LPWORD				lpMatFace;

	c_VECTOR			pivot;                     // object pivot point
	c_VECTOR			translate;                 // object translation vector
	c_VECTOR			scale;                     // object scale vector

	c_BOUNDBOX			bbox, pbbox;               // object bounding box
	c_QUAT				rotate;                    // object rotation quaternion
	c_MORPH				morph;                     // object morph
	c_MATRIX			matrix;                    // object keyframer matrix

} c_OBJECT;


// ambient struct
typedef struct _c_AMBIENT {

	char			*name;                 // ambient name
	int32			id;                    // ambient id
	c_RGB			color;                 // ambient color
	D3DCOLORVALUE	D3DCol;

} c_AMBIENT;


/*****************************************************************************
  keyframer structures
*****************************************************************************/

// key data union
typedef union _t_KDATA {

	int32		_int;                         // boolean/object id
	float		_float;                       // float
	c_VECTOR	_vect;                        // vector
	c_QUAT		_quat;                        // quaternion

} t_KDATA;


// key struct
typedef struct _t_KEY { 

	float			frame;                  // current frame
	float			tens, bias, cont;       // tension, bias, continuity
	float			easeto, easefrom;       // ease to, ease from
	t_KDATA			val;                    // the interpolated values
	float			dsa, dsb, dsc, dsd;
	float			dda, ddb, ddc, ddd;
	c_QUAT			ds, dd;
	c_QUAT			qa;
	struct _t_KEY	*next, *prev;

} t_KEY;


// track struct
typedef struct _t_TRACK {

	int32		flags;                          // track flags
	int32		numkeys;                        // number of keys
	float		frames;                         // number of frames

	t_KEY		*keys;                          // the track
	t_KEY		*last;                          // pointer to last used key

} t_TRACK;


// camera track
typedef struct _t_CAMERA { 

	t_TRACK		*pos;                          // position
	t_TRACK		*fov, *roll;                   // field of view, roll

} t_CAMERA;


// camera target track
typedef struct _t_CAMERATGT { 

	t_TRACK		*pos;                          // position

} t_CAMERATGT;


// light track
typedef struct _t_LIGHT { 

	t_TRACK		*pos;                         // position
	t_TRACK		*color;                       // color

} t_LIGHT;


typedef struct _t_LIGHTTGT { /* light target track */
  t_TRACK *pos;                          /* position */
} t_LIGHTTGT;

typedef struct _t_SPOTLIGHT { /* spotlight track */
  t_TRACK *pos;                          /* position */
  t_TRACK *color;                        /* color    */
  t_TRACK *roll;                         /* roll     */
  t_TRACK *hotspot;                      /* hotspot  */
  t_TRACK *falloff;                      /* falloff  */
} t_SPOTLIGHT;


// object track
typedef struct _t_OBJECT {

	t_TRACK		*translate;                    // position
	t_TRACK		*scale;                        // scale    
	t_TRACK		*rotate;                       // rotation 
	t_TRACK		*morph;                        // morph    
	t_TRACK		*hide;                         // hide

} t_OBJECT;


// ambient track
typedef struct _t_AMBIENT { 

	t_TRACK		*color;					// color

} t_AMBIENT;


// keyframer node
typedef struct _k_NODE {

	int32			type;                   // track type
	int32			id;                     // track id
	void			*track, *object;        // track / object pointer

	struct _k_NODE	*parent;                // parent node            
	struct _k_NODE	*child, *brother;       // hierarchy tree
	struct _k_NODE	*next, *prev;           // next/previous node

} k_NODE;


// world node
typedef struct _w_NODE { 

	int32				type;               // object type
	void				*object;            // object
	struct	_w_NODE		*next, *prev;		// next node

} w_NODE;


// scene (world, keyframer)
typedef struct _c_SCENE {

	float		f_start, f_end, f_current;    // start/end/current frame
	w_NODE		*world, *wtail;               // world
	k_NODE		*keyframer, *ktail;           // keyframer

	D3DMATRIX	Matrix;

} c_SCENE;

/*****************************************************************************
  externals
*****************************************************************************/

/*
#ifdef __cplusplus
extern "C" {
#endif
*/

extern char clax_version[];
extern char clax_copyright[];

/*****************************************************************************
  library functions (clax api)
*****************************************************************************/

int32 clax_init (int32 flags);
/*
	initializes clax, and does some internal stuff.
	returns error codes: clax_ok, clax_nomem, clax_undefined.

	note: this function must be called once, before accessing any other
		  clax routines.

	see also: clax_flag_*
*/

  int32 clax_done ();
/*
  int32 clax_done ()
  deinitializes clax, and does some internal stuff.
  returns error codes: clax_ok, clax_undefined.

  note: this function must be called once you exit your program.
*/

// loads scene from supported fileformat (not keyframer).
// returns error codes: clax_ok, clax_nomem, clax_badfile, clax_undefined.
int32 clax_load_world (char *filename, c_SCENE *scene);


int32 clax_load_motion (char *filename, c_SCENE *scene);
/*
	loads keyframer from supported fileformat (not scene).
	returns error codes: clax_ok, clax_nomem, clax_badfile, clax_undefined.

	note: clax_loadscene must be called before this.
*/

/*
	sets active scene "scene".
	returns error codes: clax_ok, clax_nullptr, clax_undefined.

	note: this function must be called right after loading, and before
		  using any other clax functions.
*/
int32 clax_setactive_scene (c_SCENE *scene);


void cam_lens_fov (float lens, float *fov);
int32 clax_getactive_scene (c_SCENE **scene);
int32 clax_getactive_camera (c_CAMERA **camera);
int32 clax_byname (char *name, w_NODE **node);
int32 clax_byid (int32 id, w_NODE **node);
int32 clax_findfirst (int32 attr, w_NODE **node);
int32 clax_findnext (int32 attr, w_NODE **node);
int32 clax_add_world (int32 type, void *obj);
int32 clax_add_track (int32 type, int32 id, int32 parent, void *track, void *obj);
int32 clax_free_world (c_SCENE *scene);
int32 clax_free_motion (c_SCENE *scene);
int32 clax_getkey_float (t_TRACK *track, float frame, float *out);
int32 clax_getkey_vect (t_TRACK *track, float frame, c_VECTOR *out);
int32 clax_getkey_quat (t_TRACK *track, float frame, c_QUAT *out);
int32 clax_alloc_scene (c_SCENE **scene);
int32 spline_getkey_quat (t_TRACK *track, float frame, c_QUAT *out);
int32 clax_set_track (int32 type, int32 id, t_TRACK *track);
int32 clax_update ();
int32 clax_free_scene (c_SCENE *scene);
void qt_identity (c_QUAT *out);
void qt_zero (c_QUAT *out);
void mat_zero (c_MATRIX out);
void vec_zero (c_VECTOR *out);
int32 clax_load_scene (char *filename, c_SCENE *scene);
void qt_scale (c_QUAT *a, float s, c_QUAT *out);
void qt_print (c_QUAT *a);
void vec_print (D3DVECTOR *a);

void mat_print (D3DMATRIX a);
int32 mat_invscale (c_MATRIX &a, c_MATRIX &out);
void mat_mulnorm (c_MATRIX a, c_VECTOR *b, c_VECTOR *out);

void vec_midpoint (D3DVECTOR *a, D3DVECTOR *b, D3DVECTOR *out);
int32 clax_free_mesh (c_SCENE *scene);
void clax_print_world ();
void clax_print_keyframer ();
void qt_rescale (c_QUAT *a, float s, c_QUAT *out);
void vec_rescale (D3DVECTOR *a, float s, D3DVECTOR *out);
int32 clax_getkey_hide (t_TRACK *track, float frame, int32 *out);
int32 clax_getkey_morph (t_TRACK *track, float frame, c_MORPH *out);
int32 clax_setactive_camera (c_CAMERA *cam);
int32 clax_save_scene (char *filename, c_SCENE *scene);
int32 mat_normalize (c_MATRIX a, c_MATRIX out);
int32 qt_equal (c_QUAT *a, c_QUAT *b);
void mat_toeuler (D3DMATRIX mat, D3DVECTOR *out);
void mat_pretrans (D3DVECTOR *v, D3DMATRIX mat, D3DMATRIX out);
float spline_ease (float t, float a, float b);
void mat_transpose (D3DMATRIX a, D3DMATRIX out);
int32 clax_collide (w_NODE *a, w_NODE *b, int32 *result);
float qt_length (c_QUAT *a);
float qt_dot (c_QUAT *a, c_QUAT *b);
float qt_dotunit (c_QUAT *a, c_QUAT *b);
void qt_negate (c_QUAT *a, c_QUAT *out);
void qt_inverse (c_QUAT *a, c_QUAT *out);
void qt_exp (c_QUAT *a, c_QUAT *out);
void qt_log (c_QUAT *a, c_QUAT *out);
void qt_lndif (c_QUAT *a, c_QUAT *b, c_QUAT *out);
void qt_slerpl (c_QUAT *a, c_QUAT *b, float spin, float alpha, c_QUAT *out);
int32 spline_initrot (t_TRACK *track);


void SetCamera( LPDIRECT3DDEVICE3 lpDev, w_NODE *World );


char *clax_geterror (int32 code);
//  returns an error string for given error code.

int32 clax_getframes (float *start, float *end);
/*
	returns the start and end frame number in keyframer to start/end.
	returns error codes: clax_ok, clax_noframes, clax_notloaded, clax_undefined.
*/

int32 clax_setframe (float frame);
//	sets current frame number in the keyframer.
//	returns error codes: clax_ok, clax_badframe, clax_noframes, clax_notloaded,
//						 clax_undefined.

int32 clax_getframe (float *frame);
//	returns current frame number in the keyframer.
//	returns error codes: clax_ok, clax_noframes, clax_notloaded, clax_undefined.

void cam_update (c_CAMERA *cam);
/*
	creates a matrix from camera "cam", giving result in "cam->mat".
	also returns perspective multipliers in "cam->perspX", "cam->perspY".

	note: dont forget to fill c_CAMERA.sizeX, c_CAMERA.sizeY, c_CAMERA.aspectratio.
*/

/*****************************************************************************
  library functions (matrix routines)
*****************************************************************************/

// returns identity matrix in "out".
void mat_identity ( c_MATRIX &out );

void mat_swap (c_MATRIX a);

// copies matrix "a" to matrix "out".
void mat_copy (c_MATRIX a, c_MATRIX out);

  void mat_add (D3DMATRIX a, D3DMATRIX b, D3DMATRIX out);
/*
  void mat_add (D3DMATRIX a, D3DMATRIX b, D3DMATRIX out)
  matrix addition, [out] = [a]+[b].
*/

  void mat_sub (D3DMATRIX a, D3DMATRIX b, D3DMATRIX out);
/*
  void mat_sub (D3DMATRIX a, D3DMATRIX b, D3DMATRIX out)
  matrix substraction, [out]=[a]-[b].
*/

//  matrix multiplication, [out] = [a] * [b].
void mat_mul (c_MATRIX a, c_MATRIX b, c_MATRIX out);

//  calculates inverse matrix of "a", giving result in "out".
//  returns error codes: clax_ok, clax_singular, clax_undefined.
int32 mat_inverse (c_MATRIX a, c_MATRIX &out);


void mat_settrans (D3DVECTOR *v, D3DMATRIX out);
// creates translation matrix "out" from vector "v".

// creates a scale matrix "out" from vector "v".
void mat_setscale (c_VECTOR *v, c_MATRIX &out);


// void mat_rotateX (float ang, D3DMATRIX out);
/*
  void mat_rotateX (float ang, D3DMATRIX out)
  creates rotation matrix around X axis "ang" degrees.
*/

// void mat_rotateY (float ang, D3DMATRIX out);
/*
  void mat_rotateY (float ang, D3DMATRIX out)
  creates rotation matrix around Y axis "ang" degrees.
*/

// void mat_rotateZ (float ang, D3DMATRIX out);
/*
  void mat_rotateZ (float ang, D3DMATRIX out)
  creates rotation matrix around Z axis "ang" degrees.
*/

// multiplies vector "b" by matrix "a", giving result in "out".
void mat_mulvec (c_MATRIX a, c_VECTOR *b, c_VECTOR *out);

D3DMATRIX RotateXMatrix( const float rads );
D3DMATRIX RotateYMatrix( const float rads );
D3DMATRIX RotateZMatrix( const float rads );
D3DMATRIX MatrixMult( const D3DMATRIX &a, const D3DMATRIX &b );


/*****************************************************************************
  library functions (vector routines)
*****************************************************************************/

// create vector, out = [x,y,z].
void vec_make (float x, float y, float z, c_VECTOR *out);

// vector copy, out = a.
void vec_copy (c_VECTOR *a, c_VECTOR *out);

// vector addition, out = a + b.
void vec_add (c_VECTOR *a, c_VECTOR *b, c_VECTOR *out);

// vector substraction, out = a - b.
void vec_sub (c_VECTOR *a, c_VECTOR *b, c_VECTOR *out);
  
// vector multiplication, out = a * b;
void vec_mul (D3DVECTOR *a, D3DVECTOR *b, D3DVECTOR *out);

// scales vector "a" to a new length, giving result in "out".
void vec_scale (c_VECTOR *a, float s, c_VECTOR *out);

// negates vector "a", giving result in "out".
void vec_negate (D3DVECTOR *a, D3DVECTOR *out);

// compares two vector "a" and "b".
int32 vec_equal (D3DVECTOR *a, D3DVECTOR *b);

// computes vector length.
float vec_length (D3DVECTOR *a);

// computes distance between two vectors "a" and "b".
float vec_distance (D3DVECTOR *a, D3DVECTOR *b);

// computes dot product of two vectors "a" and "b".
float vec_dot (D3DVECTOR *a, D3DVECTOR *b);

// computes cross product of two vectors "a" and "b", giving result in "out".
void vec_cross (c_VECTOR *a, c_VECTOR *b, c_VECTOR *out);

// liner interpolation of vectors.
// interpolates vectors "a" and "b", with interpolation parameter
// "alpha" (range 0 - 1.0) giving result in "out".
void vec_lerp (D3DVECTOR *a, D3DVECTOR *b, float alpha, D3DVECTOR *out);

// computes linear combination of two vectors "a" and "b", with
// scalar "as" and "bs", giving result in "out".
void vec_combine (D3DVECTOR *a, D3DVECTOR *b, float as, float bs, D3DVECTOR *out);

// computes normalized vector "a", giving result in "out".
void vec_normalize (c_VECTOR *a, c_VECTOR *out);


/*****************************************************************************
  library functions (quaternion routines)
*****************************************************************************/

  void qt_fromang (float ang, float x, float y, float z, c_QUAT *out);
/*
  void qt_fromang (float ang, float x, float y, float z, c_QUAT *out)
  computes quaternion from [angle,axis] representation, giving result
  in "out".
*/

  void qt_toang (c_QUAT *a, float *ang, float *x, float *y, float *z);
/*
  void qt_toang (c_QUAT *a, float *ang, float *x, float *y, float *z)
  converts quaternion to [angle,axis] representation, giving result
  in "out".
*/

void qt_make (float w, float x, float y, float z, c_QUAT *out);
void qt_swap (c_QUAT *a);

  void qt_copy (c_QUAT *a, c_QUAT *out);
/*
  void qt_copy (c_QUAT *a, c_QUAT *out)
  quaternion copy, out = a.
*/

  void qt_add (c_QUAT *a, c_QUAT *b, c_QUAT *out);
/*
  void qt_add (c_QUAT *a, c_QUAT *b, c_QUAT *out)
  quaternion addition, out = a+b.
*/

  void qt_sub (c_QUAT *a, c_QUAT *b, c_QUAT *out);
/*
  void qt_sub (c_QUAT *a, c_QUAT *b, c_QUAT *out)
  quaternion substraction, out = a-b.
*/

  void qt_mul (c_QUAT *a, c_QUAT *b, c_QUAT *out);
/*
  void qt_mul (c_QUAT a, c_QUAT b, c_QUAT *out)
  multiplies quaternion "a" by quaternion "b", giving result in "out".
*/

  void qt_div (c_QUAT *a, c_QUAT *b, c_QUAT *out);
/*
  void qt_div (c_QUAT *a, c_QUAT *b, c_QUAT *out)
  quaternion division, out = a/b.
*/

  void qt_square (c_QUAT *a, c_QUAT *out);
/*
  void qt_square (c_QUAT *a, c_QUAT *out)
  quaternion square, out = a^2.
*/

  void qt_sqrt (c_QUAT *a, c_QUAT *out);
/*
  void qt_sqrt (c_QUAT *a, c_QUAT *out)
  quaternion square root, out = sqrt (a).
*/

  void qt_normalize (c_QUAT *a, c_QUAT *out);
/*
  void qt_normalize (c_QUAT *a, c_QUAT *out)
  normalizes quaternion "a", giving result in "out".
*/

  void qt_negate (c_QUAT *a, c_QUAT *out);
/*
  void qt_negate (c_QUAT *a, c_QUAT *out)
  forms multiplicative inverse of quaternion "a", giving result in "out".
*/

void qt_slerp (c_QUAT *a, c_QUAT *b, float spin, float alpha, c_QUAT *out);
/*
  void qt_slerp (c_QUAT *a, c_QUAT *b, float alpha, c_QUAT *out)
  spherical liner interpolation of quaternions.
  interpolates quaterions "a" and "b", with interpolation parameter
  "alpha" (range 0 - 1.0) giving result in "out".
*/

// creates a matrix from quaternion "a", giving result in "mat".
void qt_matrix (c_QUAT *a, c_MATRIX mat);

// creates an inverse matrix from quaternion "q", giving result in "mat".
void qt_invmatrix (c_QUAT *a, c_MATRIX mat);
 

  void qt_frommat (D3DMATRIX mat, c_QUAT *out);
/*
  void qt_frommat (D3DMATRIX mat, c_QUAT *out)
  converts a rotation matrix "mat" to quaternion, giving result in "out".
*/

/*****************************************************************************
  library functions (spline routines)
*****************************************************************************/

//  initializes spline interpolation, with an array of keys in "keys", given
//  number of keys in "numkeys".
//  returns error codes: clax_ok, clax_spline, clax_undefined.

//  see also: t_KEY structure
int32 spline_init (t_TRACK *track);

//  returns one-dimensional interpolated value for "frame", giving
//  result in "out".
//  returns error codes: clax_ok, clax_badframe, clax_undefined.
//  the value interpolated is t_KEY.a
int32 spline_getkey_float (t_TRACK *track, float frame, float *out);

//  returns three-dimensional interpolated value for "frame", giving
//  result in "out".
//  returns error codes: clax_ok, clax_badframe, clax_undefined.
//  the values interpolated are t_KEY.a, t_KEY.b, t_KEY.c
int32 spline_getkey_vect (t_TRACK *track, float frame, c_VECTOR *out);


/*
#ifdef __cplusplus
}
#endif
*/

#endif
