/*FILE_3DS.C******************************************************************

   Clax: Portable keyframing library
         see inclosed LICENSE.TXT for licensing terms.

         for documentation, refer to CLAX.TXT

   author           : Borzom
   file created     : 04/05/97
   file description : 3D studio 4.0 driver (world + motion)

   revision history :
     ----- (xx/xx/96) Jare:   Initial version (3dsrdr.c).
     v0.10 (04/05/97) Borzom: Rewritten to fit clax structure.
           (10/05/97) Borzom: Added version check.
           (11/05/97) Borzom: Added color track and dummy name chunks.
           (22/05/97) Borzom: Added more material chunks.

   Revisioned by            : Jeon Min Woo

*****************************************************************************/

#include <stdlib.h>
#include <malloc.h>
#include <stdio.h>

#include "clax.h"
#include "claxi.h"
#include "CALCOBJ.H"
#include "D3DUTIL.H"

/*****************************************************************************
  chunks/readers definitions, structures
*****************************************************************************/


// Chunk ID
enum clax_3ds_chunks_ { 

	CHUNK_RGBF         = 0x0010, 
	CHUNK_RGBB         = 0x0011,
	CHUNK_PRJ          = 0xC23D, 
	CHUNK_MLI          = 0x3DAA,
	CHUNK_MAIN         = 0x4D4D, 
	CHUNK_OBJMESH      = 0x3D3D,
	CHUNK_BKGCOLOR     = 0x1200, 
	CHUNK_AMBCOLOR     = 0x2100,
	CHUNK_OBJBLOCK     = 0x4000, 
	CHUNK_TRIMESH      = 0x4100,
	CHUNK_VERTLIST     = 0x4110, 
	CHUNK_VERTFLAGS    = 0x4111,
	CHUNK_FACELIST     = 0x4120, 
	CHUNK_FACEMAT      = 0x4130,
	CHUNK_MAPLIST      = 0x4140, 
	CHUNK_SMOOLIST     = 0x4150,
	CHUNK_TRMATRIX     = 0x4160, 
	CHUNK_MESHCOLOR    = 0x4165,
	CHUNK_TXTINFO      = 0x4170, 
	CHUNK_LIGHT        = 0x4600,
	CHUNK_SPOTLIGHT    = 0x4610, 
	CHUNK_CAMERA       = 0x4700,
	CHUNK_HIERARCHY    = 0x4F00, 
	CHUNK_VIEWPORT     = 0x7001,
	CHUNK_MATERIAL     = 0xAFFF, 
	CHUNK_MATNAME      = 0xA000,
	CHUNK_AMBIENT      = 0xA010, 
	CHUNK_DIFFUSE      = 0xA020,
	CHUNK_SPECULAR     = 0xA030, 
	CHUNK_SHININESS    = 0xA040,
	CHUNK_TEXTURE      = 0xA200,
	CHUNK_BUMPMAP      = 0xA230, 
	CHUNK_REFLECTION   = 0xA220,
	CHUNK_MAPFILE      = 0xA300, 
	CHUNK_MAPFLAGS     = 0xA351,
	CHUNK_MAPUSCALE    = 0xA354, 
	CHUNK_MAPVSCALE    = 0xA356,
	CHUNK_MAPUOFFSET   = 0xA358, 
	CHUNK_MAPVOFFSET   = 0xA35A,
	CHUNK_KEYFRAMER    = 0xB000, 
	CHUNK_AMBIENTKEY   = 0xB001,
	CHUNK_TRACKINFO    = 0xB002, 
	CHUNK_TRACKOBJNAME = 0xB010,
	CHUNK_TRACKPIVOT   = 0xB013, 
	CHUNK_TRACKPOS     = 0xB020,
	CHUNK_TRACKROTATE  = 0xB021, 
	CHUNK_TRACKSCALE   = 0xB022,
	CHUNK_TRACKMORPH   = 0xB026, 
	CHUNK_TRACKHIDE    = 0xB029,
	CHUNK_OBJNUMBER    = 0xB030, 
	CHUNK_TRACKCAMERA  = 0xB003,
	CHUNK_TRACKFOV     = 0xB023, 
	CHUNK_TRACKROLL    = 0xB024,
	CHUNK_TRACKCAMTGT  = 0xB004, 
	CHUNK_TRACKLIGHT   = 0xB005,
	CHUNK_TRACKLIGTGT  = 0xB006, 
	CHUNK_TRACKSPOTL   = 0xB007,
	CHUNK_TRACKCOLOR   = 0xB025, 
	CHUNK_FRAMES       = 0xB008,
	CHUNK_DUMMYNAME    = 0xB011, 
	CHUNK_MAPROTANGLE  = 0xA35C,
	CHUNK_SHINSTRENGTH = 0xA041,
	CHUNK_TRANSPARENCY = 0xA050, 
	CHUNK_TRANSFALLOFF = 0xA052,
	CHUNK_REFBLUR      = 0xA053, 
	CHUNK_SELFILLUM    = 0xA084,
	CHUNK_TWOSIDED     = 0xA081, 
	CHUNK_TRANSADD     = 0xA083,
	CHUNK_WIREON       = 0xA085, 
	CHUNK_SOFTEN       = 0xA08C,
	CHUNK_MATTYPE      = 0xA100, 
	CHUNK_AMOUNTOF     = 0x0030

};


// 3DS chunk structure
typedef struct { 

	WORD	chunk_id;                // chunk id (clax_3ds_chunks_)
	DWORD	chunk_size;              // chunk length

} c_CHUNK;


// Chunk reader list
typedef struct { 
	WORD	id;							// chunk id
	int		sub;                        // has subchunks
	int		(*func) (FILE *f);          // reader function

	char	*Str;						// Debug info

} c_LISTKEY, c_LISTWORLD;


c_AMBIENT	*MainAmbient = NULL;


static int read_NULL         (FILE *f);		// (skip chunk)
static int ReadAmbientCol	 (FILE *f);
static int read_RGBF         (FILE *f);		// RGB float
static int read_RGBB         (FILE *f);		// RGB byte
static int read_AMOUNTOF     (FILE *f);		// Amount of
static int read_ASCIIZ       (FILE *f);		// ASCIIZ string
static int read_TRIMESH      (FILE *f);		// Triangular mesh
static int read_VERTLIST     (FILE *f);		// Vertex list
static int read_FACELIST     (FILE *f);		// Face list
static int read_FACEMAT      (FILE *f);		// Face material
static int read_MAPLIST      (FILE *f); /* Mapping list            */
static int read_TRMATRIX     (FILE *f);		// Transformation matrix
static int read_LIGHT        (FILE *f);		// Light
static int read_SPOTLIGHT    (FILE *f); /* Spotlight               */
static int read_CAMERA       (FILE *f);		// Camera
static int read_MATERIAL     (FILE *f);		// Material
static int read_MATNAME      (FILE *f);		// Material name
static int read_FRAMES       (FILE *f);		// Number of frames
static int read_OBJNUMBER    (FILE *f);		// Object number
static int read_TRACKOBJNAME (FILE *f);		// Track object name
static int read_DUMMYNAME    (FILE *f); /* Dummy object name       */
static int read_TRACKPIVOT   (FILE *f);		// Track pivot point
static int read_TRACKPOS     (FILE *f);		// Track position
static int read_TRACKCOLOR   (FILE *f);		// Track color
static int read_TRACKROT     (FILE *f);		// Track rotation
static int read_TRACKSCALE   (FILE *f);		// Track scale
static int read_TRACKFOV     (FILE *f);		// Track fov
static int read_TRACKROLL    (FILE *f);		// Track roll
static int read_TRACKMORPH   (FILE *f);		// Track morph
static int read_TRACKHIDE    (FILE *f);		// Track hide
static int read_MATTYPE      (FILE *f); /* Material: type          */
static int read_MATTWOSIDED  (FILE *f); /* Material: two sided     */
static int read_MATSOFTEN    (FILE *f); /* Material: soften        */
static int read_MATWIRE      (FILE *f); /* Material: wire          */
static int read_MATTRANSADD  (FILE *f); /* Material: transparency  */
static int read_MAPFLAGS     (FILE *f); /* Map flags               */
static int read_MAPFILE      (FILE *f); /* Map file                */
static int read_MAPUSCALE    (FILE *f); /* Map 1/U scale           */
static int read_MAPVSCALE    (FILE *f); /* Map 1/V scale           */
static int read_MAPUOFFSET   (FILE *f); /* Map U offset            */
static int read_MAPVOFFSET   (FILE *f); /* Map V offset            */
static int read_MAPROTANGLE  (FILE *f); /* Map rotation angle      */


// World definition chunks
/*
static c_LISTWORLD world_chunks[] = { 
  {CHUNK_RGBF,         0, read_RGBF},
  {CHUNK_RGBB,         0, read_RGBB},
  {CHUNK_AMOUNTOF,     0, read_AMOUNTOF},
  {CHUNK_PRJ,          1, read_NULL},
  {CHUNK_MLI,          1, read_NULL},
  {CHUNK_MAIN,         1, read_NULL},
  {CHUNK_OBJMESH,      1, read_NULL},
  {CHUNK_BKGCOLOR,     1, read_NULL},
  {CHUNK_AMBCOLOR,     1, read_NULL},
  {CHUNK_OBJBLOCK,     1, read_ASCIIZ},
  {CHUNK_TRIMESH,      1, read_TRIMESH},
  {CHUNK_VERTLIST,     0, read_VERTLIST},
  {CHUNK_VERTFLAGS,    0, read_NULL},
  {CHUNK_FACELIST,     1, read_FACELIST},
  {CHUNK_MESHCOLOR,    0, read_NULL},
  {CHUNK_FACEMAT,      0, read_FACEMAT},
  {CHUNK_MAPLIST,      0, read_MAPLIST},
  {CHUNK_TXTINFO,      0, read_NULL},
  {CHUNK_SMOOLIST,     0, read_NULL},
  {CHUNK_TRMATRIX,     0, read_TRMATRIX},
  {CHUNK_LIGHT,        1, read_LIGHT},
  {CHUNK_SPOTLIGHT,    0, read_SPOTLIGHT},
  {CHUNK_CAMERA,       0, read_CAMERA},
  {CHUNK_HIERARCHY,    1, read_NULL},
  {CHUNK_VIEWPORT,     0, read_NULL},
  {CHUNK_MATERIAL,     1, read_MATERIAL},
  {CHUNK_MATNAME,      0, read_MATNAME},
  {CHUNK_AMBIENT,      1, read_NULL},
  {CHUNK_DIFFUSE,      1, read_NULL},
  {CHUNK_SPECULAR,     1, read_NULL},
  {CHUNK_TEXTURE,      1, read_NULL},
  {CHUNK_BUMPMAP,      1, read_NULL},
  {CHUNK_REFLECTION,   1, read_NULL},
  {CHUNK_MAPFILE,      0, read_MAPFILE},
  {CHUNK_MAPFLAGS,     0, read_MAPFLAGS},
  {CHUNK_MAPUSCALE,    0, read_MAPUSCALE},
  {CHUNK_MAPVSCALE,    0, read_MAPVSCALE},
  {CHUNK_MAPUOFFSET,   0, read_MAPUOFFSET},
  {CHUNK_MAPVOFFSET,   0, read_MAPVOFFSET},
  {CHUNK_MAPROTANGLE,  0, read_MAPROTANGLE},
  {CHUNK_SHININESS,    1, read_NULL},
  {CHUNK_SHINSTRENGTH, 1, read_NULL},
  {CHUNK_TRANSPARENCY, 1, read_NULL},
  {CHUNK_TRANSFALLOFF, 1, read_NULL},
  {CHUNK_REFBLUR,      1, read_NULL},
  {CHUNK_SELFILLUM,    1, read_NULL},
  {CHUNK_TWOSIDED,     0, read_MATTWOSIDED},
  {CHUNK_TRANSADD,     0, read_MATTRANSADD},
  {CHUNK_WIREON,       0, read_MATWIRE},
  {CHUNK_SOFTEN,       0, read_MATSOFTEN},
  {CHUNK_MATTYPE,      0, read_MATTYPE}
};
*/
static c_LISTWORLD world_chunks[] = { 

	{CHUNK_RGBF,         0, read_RGBF,		"RGB float" },
	{CHUNK_RGBB,         0, read_RGBB,		"RGB byte"  },
	{CHUNK_AMOUNTOF,     0, read_AMOUNTOF,	"Amount of" },
	{CHUNK_PRJ,          1, read_NULL,		"Skip PRJ..."   },
	{CHUNK_MLI,          1, read_NULL,		"Skip MLI..."   },
	{CHUNK_MAIN,         1, read_NULL,		"Skip Main..."   },
	{CHUNK_OBJMESH,      1, read_NULL,		"Skip ObjectMesh..." },
	{CHUNK_BKGCOLOR,     1, read_NULL,		"Skip BackColor..."  },

	//{CHUNK_AMBCOLOR,     1, read_NULL,	"Skip Ambientcolor..." },
	{CHUNK_AMBCOLOR,     1, ReadAmbientCol,	"Ambientcolor..." },

	{CHUNK_OBJBLOCK,     1, read_ASCIIZ,	"ASCIIZ string"  },
	{CHUNK_TRIMESH,      1, read_TRIMESH,	"Triangular mesh" },
	{CHUNK_VERTLIST,     0, read_VERTLIST,	"Vertex list" },
	{CHUNK_VERTFLAGS,    0, read_NULL,		"Skip VertFlags..." },
	{CHUNK_FACELIST,     1, read_FACELIST,	"Face list" },
	{CHUNK_MESHCOLOR,    0, read_NULL,		"Skip MeshColor..." },
	{CHUNK_FACEMAT,      0, read_FACEMAT,	"Face Material" },
	{CHUNK_MAPLIST,      0, read_MAPLIST,	"Mapping list" },
	{CHUNK_TXTINFO,      0, read_NULL,		"Skip TextureInfo..." },
	{CHUNK_SMOOLIST,     0, read_NULL,		"Skip SmoothList..." },
	{CHUNK_TRMATRIX,     0, read_TRMATRIX,	"Transformation matrix" },
	{CHUNK_LIGHT,        1, read_LIGHT,		"Light..." },
	{CHUNK_SPOTLIGHT,    0, read_SPOTLIGHT, "Spotlight" },
	{CHUNK_CAMERA,       0, read_CAMERA,	"Camera" },
	{CHUNK_HIERARCHY,    1, read_NULL,		"Skip Hierarchy..." },
	{CHUNK_VIEWPORT,     0, read_NULL,		"Skip Viewport..." },
	{CHUNK_MATERIAL,     1, read_MATERIAL,	"Material" },
	{CHUNK_MATNAME,      0, read_MATNAME,	"Material Name" },
	{CHUNK_AMBIENT,      1, read_NULL,		"Skip Ambient..." },
	{CHUNK_DIFFUSE,      1, read_NULL,		"Skip Diffuse..." },
	{CHUNK_SPECULAR,     1, read_NULL,		"Skip Specular..." },
	{CHUNK_SHININESS,    1, read_NULL,		"Skip Shininess..." },
	{CHUNK_TEXTURE,      1, read_NULL,		"Skip Texture..." },
	{CHUNK_BUMPMAP,      1, read_NULL,		"Skip Bumpmap..." },
	{CHUNK_REFLECTION,   1, read_NULL,		"Skip Reflection..." },
	{CHUNK_MAPFILE,      0, read_MAPFILE,	"Map File" },
	{CHUNK_MAPFLAGS,     0, read_MAPFLAGS,	"Map Flags" },
	{CHUNK_MAPUSCALE,    0, read_MAPUSCALE,	"Map 1/U Scale" },
	{CHUNK_MAPVSCALE,    0, read_MAPVSCALE,	"Map 1/V Scale" },
	{CHUNK_MAPUOFFSET,   0, read_MAPUOFFSET,"Map U Offset" },
	{CHUNK_MAPVOFFSET,   0, read_MAPVOFFSET,"Map V Offset" },
	{CHUNK_MAPROTANGLE,  0, read_MAPROTANGLE,"Map rotation angle" },
	{CHUNK_SHINSTRENGTH, 1, read_NULL,		"Skip Strength..."  },
	{CHUNK_TRANSPARENCY, 1, read_NULL,		"Skip Transparency..."  },
	{CHUNK_TRANSFALLOFF, 1, read_NULL,		"Skip TransFallOff..."  },
	{CHUNK_REFBLUR,      1, read_NULL,		"Skip RefBlur..."  },
	{CHUNK_SELFILLUM,    1, read_NULL,		"Skip SelfIllum..." },
	{CHUNK_TWOSIDED,     0, read_MATTWOSIDED,"TwoSide..." },
	{CHUNK_TRANSADD,     0, read_MATTRANSADD,"TransAdd..." },
	{CHUNK_WIREON,       0, read_MATWIRE,	"WireON..." },
	{CHUNK_SOFTEN,       0, read_MATSOFTEN, "Soften..." },
	{CHUNK_MATTYPE,      0, read_MATTYPE,	"Material Type..." }

};


// Keyframer chunks
static c_LISTKEY key_chunks[] = {
	{CHUNK_MAIN,         1, read_NULL,			"Skip Main..." },
	{CHUNK_KEYFRAMER,    1, read_NULL,			"Skip KeyFramer..." },
	{CHUNK_AMBIENTKEY,   1, read_NULL,			"Skip AmbientKey..." },
	{CHUNK_TRACKINFO,    1, read_NULL,			"Skip TrackInfo..." },
	{CHUNK_FRAMES,       0, read_FRAMES,		"Frames"},
	{CHUNK_TRACKOBJNAME, 0, read_TRACKOBJNAME,	"TrackObjectName" },
	{CHUNK_DUMMYNAME,    0, read_DUMMYNAME,		"DummyName" },
	{CHUNK_TRACKPIVOT,   0, read_TRACKPIVOT,	"TrackPivot" },
	{CHUNK_TRACKPOS,     0, read_TRACKPOS,		"TrackPosition" },
	{CHUNK_TRACKCOLOR,   0, read_TRACKCOLOR,	"TrackColor" },
	{CHUNK_TRACKROTATE,  0, read_TRACKROT,		"TrackRotate" },
	{CHUNK_TRACKSCALE,   0, read_TRACKSCALE,	"TrackScale" },
	{CHUNK_TRACKMORPH,   0, read_TRACKMORPH,	"TrackMorph" },
	{CHUNK_TRACKHIDE,    0, read_TRACKHIDE,		"TrackHide" },
	{CHUNK_OBJNUMBER,    0, read_OBJNUMBER,		"ObjectNumber" },
	{CHUNK_TRACKCAMERA,  1, read_NULL,			"Skip TrackCamera..." },
	{CHUNK_TRACKCAMTGT,  1, read_NULL,			"Skip TrackCameraTarget..." },
	{CHUNK_TRACKLIGHT,   1, read_NULL,			"Skip TrackLight..." },
	{CHUNK_TRACKLIGTGT,  1, read_NULL,			"Skip TrackLightTarget..." },
	{CHUNK_TRACKSPOTL,   1, read_NULL,			"Skip TrackSpotL..." },
	{CHUNK_TRACKFOV,     0, read_TRACKFOV,		"Track FOV" },
	{CHUNK_TRACKROLL,    0, read_TRACKROLL,		"Track Roll" }
};

static int   c_chunk_last;       // parent chunk
static int   c_chunk_prev;       // previous chunk
static int   c_chunk_curr;       // current chunk
static int   c_id;               // current id
static char  c_string[64];       // current name
static void *c_node;             // current node
BOOL		IsObject = FALSE;

static int		SubCount = 0;
static FILE	*fp;


/*****************************************************************************
  internal functions
*****************************************************************************/

//	swap y/z in vector.
static void vec_swap (c_VECTOR *a)
{
	float tmp;

#ifdef CLAX_SWAP_YZ
	tmp  = a->y;
	a->y = a->z;
	a->z = tmp;
#endif
}



void PrintTab( int Count )
{
	int Num;

	for(Num = 0; Num < Count; Num++) {
		fprintf( fp, "\t" );
	}
}


//	copies asciiz string with memory allocation. 
//	i know, i could use strdup(), but its not ANSI C.
static char *strcopy (char *src)
{
	char *temp;

	if ((temp = (char *)malloc (strlen (src) + 1)) == NULL) 
		return NULL;

	strcpy (temp, src);
	return temp;
}


//	reset material map to default;
static void clear_map (c_MAP *map)
{
	map->file = NULL;
	map->flags = 0;
	map->U_scale = 0.0;
	map->V_scale = 0.0;
	map->U_offset = 0.0;
	map->V_offset = 0.0;
	map->rot_angle = 0.0;
}


//	reset material to default.
static void InitMaterial (c_MATERIAL *mat)
{
	mat->shading = 0;
	mat->flags = 0;

	clear_map (&mat->texture);
	clear_map (&mat->bump);
	clear_map (&mat->reflection);

	D3DUtil_InitMaterial( mat->Material, 1.0f, 1.0f, 1.0f );
}


//	allocate memory for a track.
static t_TRACK *alloc_track ()
{
	t_TRACK		*track;

	track = (t_TRACK*) malloc(sizeof (t_TRACK));
	track->keys = NULL;
	track->last = NULL;
	track->flags = 0;
	track->frames = 0.0f;
	track->numkeys = 0;

	return track;
}


//	add a key to track.
static int add_key (t_TRACK *track, t_KEY *key, int frame)
{
	if (track == NULL || key == NULL) 
		return clax_err_nullptr;

	key->frame = (float)frame;
	key->next = NULL;

	if (track->keys == NULL) {
		key->prev = NULL;
		track->keys = key;
	}
	else {
		key->prev = track->last;
		track->last->next = key;
	}

	track->frames = key->frame;
	track->last = key;

	track->numkeys++;

	return clax_err_ok;
}



/*****************************************************************************
  chunk readers (world)
*****************************************************************************/

//	"dummy" chunk reader.
static int read_NULL (FILE *f)
{
	// to skip the warning
	if (f) {}

	return clax_err_ok;
}



static int ReadAmbientCol(FILE *f)
{
	// object ambient
	//if(c_chunk_last == CHUNK_MATNAME)
	//	return clax_err_ok;
	
	MainAmbient = (c_AMBIENT *)malloc(sizeof(c_AMBIENT));
	
	fread( &MainAmbient->color.r, sizeof(float), 1, f );
	fread( &MainAmbient->color.g, sizeof(float), 1, f );
	fread( &MainAmbient->color.b, sizeof(float), 1, f );

	MainAmbient->D3DCol.r = MainAmbient->color.r;
	MainAmbient->D3DCol.g = MainAmbient->color.g;
	MainAmbient->D3DCol.b = MainAmbient->color.b;
	MainAmbient->D3DCol.a = 1.0f;
	
	return clax_err_ok;
}


// RGB float reader.
static int read_RGBF (FILE *f)
{
	c_MATERIAL *mat = (c_MATERIAL *)c_node;
	D3DCOLORVALUE	*rgb = NULL;

	switch (c_chunk_last) {
		case CHUNK_LIGHT:
			rgb = &(((c_LIGHT *)c_node)->color);
			break;

		case CHUNK_AMBIENT:
			rgb = &(mat->Material.ambient);
			break;

		case CHUNK_DIFFUSE:
		    rgb = &(mat->Material.diffuse); 
			break;

		case CHUNK_SPECULAR:
			rgb = &(mat->Material.specular); 
			break;
	}

	if (rgb) {

		fread( &rgb->r, sizeof(float), 1, f );
		fread( &rgb->g, sizeof(float), 1, f );
		fread( &rgb->b, sizeof(float), 1, f );

		rgb->a = 1.0f;
	}

	return clax_err_ok;
}


// RGB Byte reader.
static int read_RGBB (FILE *f)
{
	c_MATERIAL	*mat = (c_MATERIAL *)c_node;
	D3DCOLORVALUE	*rgb = NULL;
	BYTE			 c[3];

	switch (c_chunk_last) {
		case CHUNK_LIGHT:
			rgb = &(((c_LIGHT *)c_node)->color); 
			break;

		case CHUNK_AMBIENT:
			rgb = &(mat->Material.dcvAmbient);
			break;

		case CHUNK_DIFFUSE:
			rgb = &(mat->Material.dcvDiffuse);
			break;

		case CHUNK_SPECULAR:
			rgb = &(mat->Material.dcvSpecular);
			break;
	}

	if (fread (c, sizeof (c), 1, f) != 1)
		return clax_err_badfile;

	// Ǽ 0 ~ 1  ȯ ش.
	if (rgb) {
		rgb->r = (float)c[0] / 255.0f;
		rgb->g = (float)c[1] / 255.0f;
		rgb->b = (float)c[2] / 255.0f;

		rgb->a = 1.0f;
	}

	return clax_err_ok;
}


// "amount of" reader.
static int read_AMOUNTOF (FILE *f)
{
	c_MATERIAL *mat = (c_MATERIAL *)c_node;
	float      *fl = NULL;
	WORD        w;

	switch (c_chunk_last) {

		case CHUNK_SHININESS:
			fl = &(mat->Material.dvPower);
			break;

		case CHUNK_SHINSTRENGTH:
			fl = &(mat->shin_strength);
			break;

		case CHUNK_TRANSPARENCY: 
			fl = &(mat->transparency); 
			break;

		case CHUNK_TRANSFALLOFF: 
			fl = &(mat->trans_falloff);
			break;

		case CHUNK_REFBLUR:
			fl = &(mat->refblur);
			break;

		case CHUNK_SELFILLUM:
			fl = &(mat->self_illum);

	}

	if (fread (&w, sizeof (w), 1, f) != 1)
		return clax_err_badfile;
	
	if (fl)
		*fl = (float)w / 100.0f;

	return clax_err_ok;
}


// ASCIIZ string reader.
static int read_ASCIIZ (FILE *f)
{
	// Ͱ  ̸() Ű Ѵ.
	char *s = c_string;
	int   c;

	while ((c = fgetc (f)) != EOF && c != '\0') 
		*s++ = (char)c;

	if (c == EOF) 
		return clax_err_badfile;

	*s = '\0';
	return clax_err_ok;
}


// Triangular mesh reader.
static int read_TRIMESH (FILE *f)
{
	c_OBJECT *obj;

	// to skip the warning
	if (f) {}

	if ((obj = (c_OBJECT *)malloc (sizeof (c_OBJECT))) == NULL)
		return clax_err_nomem;

	if ((obj->name = strcopy (c_string)) == NULL)
		return clax_err_nomem;

	obj->id = c_id++;
	// Ʈ θ  ʴ´.
	obj->parent = -1;
	obj->flags = 0;

	vec_zero (&obj->pivot);
	vec_zero (&obj->translate);
	vec_zero (&obj->scale);
	qt_zero (&obj->rotate);
	mat_zero (obj->matrix);

	obj->lpFaceMatGroup = NULL;

	c_node = obj;
	IsObject = TRUE;

	clax_add_world (clax_obj_object, obj);
	return clax_err_ok;
}



static int read_VERTLIST (FILE *f)
{
	c_OBJECT	*obj = (c_OBJECT *)c_node;
	c_VERTEX	*v;
	float		c[3];
	WORD		nv;

	// vertex  д´.
	if (fread (&nv, sizeof (nv), 1, f) != 1) 
		return clax_err_badfile;

	// vertex  ŭ ͸ ҴѴ.
	if ((v = (c_VERTEX *)malloc (nv * sizeof (c_VERTEX))) == NULL)
		return clax_err_nomem;

	// Ʈ Ͱ Ű...
	obj->vertices = v;
	obj->numverts = nv;

	while (nv-- > 0) {

		if (fread (c, sizeof (c), 1, f) != 1) 
			return clax_err_badfile;

		vec_make (c[0], c[1], c[2], &v->vert);
		vec_swap (&v->vert);

		// ؽ ǥ Ѵ.
		v->u = 0.0;
		v->v = 0.0;

		v++;
	}

	return clax_err_ok;
}


// Face list reader.
/*
static int read_FACELIST (FILE *f)
{
	c_OBJECT *obj = (c_OBJECT *)c_node;
	c_FACE   *fc;
	WORD      nv, flags;

	// Face   ´.
	if (fread (&nv, sizeof (nv), 1, f) != 1) 
		return clax_err_badfile;

	if ((fc = (c_FACE *)malloc (nv * sizeof (c_FACE))) == NULL)
		return clax_err_nomem;

	obj->numfaces = nv;
	obj->faces = fc;

	while (nv-- > 0) {

		fread( &fc->c, sizeof(WORD), 1, f );
		fread( &fc->b, sizeof(WORD), 1, f );
		fread( &fc->a, sizeof(WORD), 1, f );

		if (fread (&flags, sizeof (flags), 1, f) != 1) 
			return clax_err_badfile;

		fc->pa = &obj->vertices[ fc->a ];
		fc->pb = &obj->vertices[ fc->b ];
		fc->pc = &obj->vertices[ fc->c ];

		fc->flags = 0;
		// fc->mat = 0;

		if (flags & 0x08) 
			fc->flags |= clax_face_wrapU;

		if (flags & 0x10) 
			fc->flags |= clax_face_wrapV;

		fc++;
	}

	return clax_err_ok;
}
*/
static int read_FACELIST (FILE *f)
{
	c_OBJECT *obj = (c_OBJECT *)c_node;
	c_FACE   *fc;
	WORD      c[3];
	WORD      nv, flags;

	// Face   ´.
	if (fread (&nv, sizeof (nv), 1, f) != 1) 
		return clax_err_badfile;

	if ((fc = (c_FACE *)malloc (nv * sizeof (c_FACE))) == NULL)
		return clax_err_nomem;

	obj->numfaces = nv;
	obj->faces = fc;

	while (nv-- > 0) {
		if (fread (c, sizeof (c), 1, f) != 1) 
			return clax_err_badfile;

		if (fread (&flags, sizeof (flags), 1, f) != 1) 
			return clax_err_badfile;

		fc->c = c[0];
		fc->b = c[1];
		fc->a = c[2];

		fc->pa = &obj->vertices[ c[2] ];
		fc->pb = &obj->vertices[ c[1] ];
		fc->pc = &obj->vertices[ c[0] ];

		fc->flags = 0;
		//fc->mat = 0;

		if (flags & 0x08) 
			fc->flags |= clax_face_wrapU;

		if (flags & 0x10) 
			fc->flags |= clax_face_wrapV;

		fc++;
	}

	return clax_err_ok;
}



static int read_FACEMAT (FILE *f)
{
	LPFACEMATGROUP	Ptr, Group = ((c_OBJECT*)c_node)->lpFaceMatGroup;
	LPFACEMATGROUP	TempGroup;
	WORD			i, FaceNum, Face;

	// Ʈ  ּҸ Ѵ.
	Ptr = Group;

	// ̽ material ̸ д´.
	if (read_ASCIIZ (f)) 
		return clax_err_badfile;

	fread( &FaceNum, sizeof(WORD), 1, f );

	TempGroup = (LPFACEMATGROUP) calloc(1, sizeof(FACEMATGROUP) );
	strcpy( TempGroup->szName, c_string );

	TempGroup->NumTriangles = FaceNum;
    //TempGroup->lpTriangles  = new WORD [FaceNum];
	TempGroup->lpTriangles  = (LPWORD) malloc( FaceNum * sizeof(WORD) );

	for ( i = 0; i < FaceNum; i++ ) {
        fread( &Face, sizeof(WORD), 1, f );
        TempGroup->lpTriangles[i] = Face;
	}


	if ( !Group )
		((c_OBJECT*)c_node)->lpFaceMatGroup = TempGroup;
    else {
		while( Group->lpNext )
			Group = Group->lpNext;
		
		Group->lpNext = TempGroup;
		((c_OBJECT*)c_node)->lpFaceMatGroup = Ptr;
	}

	return clax_err_ok;
}


//  Map list reader.
static int read_MAPLIST (FILE *f)
{
	// Ʈ vertext list Ű...
	c_VERTEX *v = ((c_OBJECT *)c_node)->vertices;
	WORD      nv;

	// vertex  u,v  ´.
	if (fread (&nv, sizeof (nv), 1, f) != 1) 
		return clax_err_badfile;

	while (nv-- > 0) {

		// vertex u,v Ѵ.
		fread( &v->u, sizeof(float), 1, f);
		fread( &v->v, sizeof(float), 1, f);

		v++;
	}

	return clax_err_ok;
}



static int read_TRMATRIX (FILE *f)
{
	c_OBJECT *obj = (c_OBJECT *)c_node;
	c_VERTEX *v = obj->vertices;
	c_VECTOR  piv;
	c_MATRIX  mat;
	float     pivot[3];
	int       i, j;

	//  ʱȭ Ѵ.
	mat_identity (mat);

	for(i = 0; i < 3; i++) {
		for(j = 0; j < 3; j++) {

			if (fread (&mat[i][j], sizeof (float), 1, f) != 1)
				return clax_err_badfile;
		}
	}

	if (fread (pivot, sizeof (pivot), 1, f) != 1) 
		return clax_err_badfile;

	vec_make (pivot[0], pivot[1], pivot[2], &piv);
	vec_swap (&piv);

	//  ȿ y,z ٲش.
	mat_swap (mat);

	//  Ѵ.
	mat_invscale (mat, mat);

	for (i = 0; i < obj->numverts; i++) {

		vec_sub (&v->vert, &piv, &v->vert);
		mat_mulvec (mat, &v->vert, &v->vert);
		v++;

	}

	return clax_err_ok;
}


//  Light reader.
static int read_LIGHT (FILE *f)
{
	c_LIGHT *light;

	// light ü ŭ ޸𸮸 ҴѴ.
	if ((light = (c_LIGHT *)malloc (sizeof (c_LIGHT))) == NULL)
		return clax_err_nomem;

	if ((light->name = strcopy (c_string)) == NULL) 
		return clax_err_nomem;

	light->id = c_id++;
	// light type   ش. omni = ü 
	light->flags = clax_light_omni;

	fread( &light->pos.x, sizeof(D3DVALUE), 1, f );
	fread( &light->pos.z, sizeof(D3DVALUE), 1, f );
	fread( &light->pos.y, sizeof(D3DVALUE), 1, f );

	c_node = light;

	clax_add_world (clax_obj_light, light);
	return clax_err_ok;
}



static int read_SPOTLIGHT (FILE *f)
{
/*
  read_SPOTLIGHT: Spot light reader.
*/
  float   c[5];
  c_LIGHT *light = (c_LIGHT *)c_node;

  if (fread (c, sizeof (c), 1, f) != 1) return clax_err_badfile;
  light->target.x = c[0];
  light->target.y = c[1];
  light->target.z = c[2];
  light->hotspot = c[3];
  light->falloff = c[4];
  light->flags = clax_light_spot;
  light->roll = 0.0;
  vec_swap (&light->target);
  return clax_err_ok;
}


// Camera reader.
static int read_CAMERA (FILE *f)
{
	c_CAMERA	*cam;

	// ī޶ ü ޸𸮸 ҴѴ.
	if ((cam = (c_CAMERA *)malloc (sizeof (c_CAMERA))) == NULL)
		return clax_err_nomem;

	if ((cam->name = strcopy (c_string)) == NULL) 
		return clax_err_nomem;

	cam->id = c_id++;

	// ī޶ ġ д´.
	fread( &cam->pos.x, sizeof(float), 1, f );
	fread( &cam->pos.z, sizeof(float), 1, f );
	fread( &cam->pos.y, sizeof(float), 1, f );

	// ī޶ ǥ д´.
	fread( &cam->target.x, sizeof(float), 1, f );
	fread( &cam->target.z, sizeof(float), 1, f );
	fread( &cam->target.y, sizeof(float), 1, f );
	
	// ī޶ ȸ д´.
	fread( &cam->roll, sizeof(float), 1, f );

	// Lens о FOV  Ѵ.
	fread( &cam->Lens, sizeof(float), 1, f );
	cam_lens_fov (cam->Lens, &cam->fov);

	c_node = cam;
	clax_add_world (clax_obj_camera, cam);

	return clax_err_ok;
}


// Material reader.
static int read_MATERIAL (FILE *f)
{
	c_MATERIAL *mat;

	// to skip the warning
	if (f) {}

	// material ü ŭ ޸𸮸 ҴѴ.
	if ((mat = (c_MATERIAL *)malloc (sizeof (c_MATERIAL))) == NULL)
		return clax_err_nomem;

	InitMaterial (mat);

	// material ID  δ.
	mat->id = c_id++;
	c_node = mat;

	// Ʈ Ʈ ߰Ѵ.
	clax_add_world (clax_obj_material, mat);
	return clax_err_ok;
}


// Material name reader.
static int read_MATNAME (FILE *f)
{
	//  material 带 ´.
	c_MATERIAL *mat = (c_MATERIAL *)c_node;

	if (read_ASCIIZ (f)) 
		return clax_err_badfile;

	//  Ʈ  ̸ ´.
	if ((mat->name = strcopy (c_string)) == NULL) 
		return clax_err_nomem;

	return clax_err_ok;
}


// Material type reader.
static int read_MATTYPE (FILE *f)
{
	c_MATERIAL *mat = (c_MATERIAL *)c_node;
	WORD	   type;

	if (fread (&type, sizeof (type), 1, f) != 1)
		return clax_err_badfile;

	mat->shading = type;
	return clax_err_ok;
}


/*
  read_MATTWOSIDED: Material two sided reader.
*/
static int read_MATTWOSIDED (FILE *f)
{
	c_MATERIAL *mat = (c_MATERIAL *)c_node;

	/* to skip the warning */
	if (f) {}
	mat->flags |= clax_mat_twosided;
	return clax_err_ok;
}



static int read_MATSOFTEN (FILE *f)
{
/*
  read_MATSOFTEN: Material soften reader.
*/
  c_MATERIAL *mat = (c_MATERIAL *)c_node;

  if (f) {} /* to skip the warning */
  mat->flags |= clax_mat_soften;
  return clax_err_ok;
}

static int read_MATWIRE (FILE *f)
{
/*
  read_MATWIRE: Material wireframe reader.
*/
  c_MATERIAL *mat = (c_MATERIAL *)c_node;

  if (f) {} /* to skip the warning */
  mat->flags |= clax_mat_wire;
  return clax_err_ok;
}

static int read_MATTRANSADD (FILE *f)
{
/*
  read_MATTRANSADD: Material transparency add reader.
*/
  c_MATERIAL *mat = (c_MATERIAL *)c_node;

  if (f) {} /* to skip the warning */
  mat->flags |= clax_mat_transadd;
  return clax_err_ok;
}


/*
  read_MAPFILE: MAP file reader.
*/
static int read_MAPFILE (FILE *f)
{
	c_MATERIAL *mat = (c_MATERIAL *)c_node;
	c_MAP      *map = NULL;

	// mapping  ̸ е Ѵ.
	if (read_ASCIIZ (f)) 
		return clax_err_badfile;

	switch (c_chunk_last) {
		case CHUNK_TEXTURE:
			map = &(mat->texture); 
			break;

		case CHUNK_BUMPMAP:
			map = &(mat->bump);
			break;

		// reflection map : ݻǴ ؽ ̹
		case CHUNK_REFLECTION:
			map = &(mat->reflection);
	}

	if (map)
	    if ((map->file = strcopy (c_string)) == NULL)
			return clax_err_nomem;

	return clax_err_ok;
}


/*
  read_MAPFLAGS: MAP flags reader.
*/
static int read_MAPFLAGS (FILE *f)
{
	c_MATERIAL *mat = (c_MATERIAL *)c_node;
	c_MAP      *map = NULL;
	WORD        flags;

	if (fread (&flags, sizeof (flags), 1, f) != 1) 
		return clax_err_badfile;

	switch (c_chunk_last) {
		case CHUNK_TEXTURE: 
			map = &(mat->texture); 
			break;

		case CHUNK_BUMPMAP: 
			map = &(mat->bump); 
			break;

		case CHUNK_REFLECTION: 
			map = &(mat->reflection);
	}

	if (map) 
		map->flags = flags;

	return clax_err_ok;
}



static int read_MAPUSCALE (FILE *f)
{
/*
  read_MAPUSCALE: MAP U scale reader.
*/
  c_MATERIAL *mat = (c_MATERIAL *)c_node;
  c_MAP      *map = NULL;
  float       U;

  if (fread (&U, sizeof (U), 1, f) != 1) return clax_err_badfile;
  switch (c_chunk_last) {
    case CHUNK_TEXTURE: map = &(mat->texture); break;
    case CHUNK_BUMPMAP: map = &(mat->bump); break;
    case CHUNK_REFLECTION: map = &(mat->reflection);
  }
  if (map) map->U_scale = U;
  return clax_err_ok;
}

static int read_MAPVSCALE (FILE *f)
{
/*
  read_MAPUSCALE: MAP U scale reader.
*/
  c_MATERIAL *mat = (c_MATERIAL *)c_node;
  c_MAP      *map = NULL;
  float       V;

  if (fread (&V, sizeof (V), 1, f) != 1) return clax_err_badfile;
  switch (c_chunk_last) {
    case CHUNK_TEXTURE: map = &(mat->texture); break;
    case CHUNK_BUMPMAP: map = &(mat->bump); break;
    case CHUNK_REFLECTION: map = &(mat->reflection);
  }
  if (map) map->V_scale = V;
  return clax_err_ok;
}

static int read_MAPUOFFSET (FILE *f)
{
/*
  read_MAPUSCALE: MAP U offset reader.
*/
  c_MATERIAL *mat = (c_MATERIAL *)c_node;
  c_MAP      *map = NULL;
  float       U;

  if (fread (&U, sizeof (U), 1, f) != 1) return clax_err_badfile;
  switch (c_chunk_last) {
    case CHUNK_TEXTURE: map = &(mat->texture); break;
    case CHUNK_BUMPMAP: map = &(mat->bump); break;
    case CHUNK_REFLECTION: map = &(mat->reflection);
  }
  if (map) map->U_offset = U;
  return clax_err_ok;
}

static int read_MAPVOFFSET (FILE *f)
{
/*
  read_MAPUSCALE: MAP V offset reader.
*/
  c_MATERIAL *mat = (c_MATERIAL *)c_node;
  c_MAP      *map = NULL;
  float       V;

  if (fread (&V, sizeof (V), 1, f) != 1) return clax_err_badfile;
  switch (c_chunk_last) {
    case CHUNK_TEXTURE: map = &(mat->texture); break;
    case CHUNK_BUMPMAP: map = &(mat->bump); break;
    case CHUNK_REFLECTION: map = &(mat->reflection);
  }
  if (map) map->V_offset = V;
  return clax_err_ok;
}

static int read_MAPROTANGLE (FILE *f)
{
/*
  read_MAPUSCALE: MAP rotation angle reader.
*/
  c_MATERIAL *mat = (c_MATERIAL *)c_node;
  c_MAP      *map = NULL;
  float       angle;

  if (fread (&angle, sizeof (angle), 1, f) != 1) return clax_err_badfile;
  switch (c_chunk_last) {
    case CHUNK_TEXTURE: map = &(mat->texture); break;
    case CHUNK_BUMPMAP: map = &(mat->bump); break;
    case CHUNK_REFLECTION: map = &(mat->reflection);
  }
  if (map) map->rot_angle = angle;
  return clax_err_ok;
}


/*****************************************************************************
  chunk readers (keyframer)
*****************************************************************************/

//	Frames reader.
//  Ű  ó  Ѵ.
static int read_FRAMES (FILE *f)
{
	DWORD	c[2];

	if (fread (c, sizeof (c), 1, f) != 1) 
		return clax_err_badfile;

	clax_scene->f_start = (float)c[0];
	clax_scene->f_end = (float)c[1];

	return clax_err_ok;
}


// Object number reader. (3DS 4.0+)
static int read_OBJNUMBER (FILE *f)
{
	WORD	n;

	if (fread (&n, sizeof (n), 1, f) != 1) 
		return clax_err_badfile;

	c_id = n;

#ifdef	CLAX_DEBUG
	PrintTab( SubCount );
	fprintf( fp, "%d\n", n );
#endif

	return clax_err_ok;
}



static int read_DUMMYNAME (FILE *f)
{
/*
  read_DUMMYNAME: Dummy object name reader.
*/
  c_OBJECT *obj = (c_OBJECT *)c_node;

  if (read_ASCIIZ (f)) return clax_err_badfile;
  if ((obj->name = strcopy (c_string)) == NULL) return clax_err_nomem;
  return clax_err_ok;
}


//	Track object name reader.
static int read_TRACKOBJNAME (FILE *f)
{
	w_NODE    *node;
	k_NODE    *pnode;
	c_OBJECT  *obj = NULL;
	c_LIGHT   *light = NULL;
	c_CAMERA  *cam = NULL;
	c_AMBIENT *amb = NULL;
	void      *track;
	WORD       flags[2];
	sword      parent;
	int        wparent = -1;
	
	// for 3DS 3.0 compatibility
	if (c_chunk_prev != CHUNK_OBJNUMBER)
		c_id++;

	if (read_ASCIIZ (f)) 
		return clax_err_badfile;

#ifdef	CLAX_DEBUG
	PrintTab( SubCount );
	fprintf( fp, "%s\n", c_string );
#endif

	// Ambient Object...
	if (strcmp (c_string, "$AMBIENT$") == 0) {

		if ((amb = (c_AMBIENT *)malloc (sizeof (c_AMBIENT))) == NULL)
			return clax_err_nomem;

		if ((amb->name = strcopy (c_string)) == NULL) 
			return clax_err_nomem;

		amb->id = 1024 + c_id;

		vec_zero ((c_VECTOR *)&amb->color);
		clax_add_world (clax_obj_ambient, amb);

	}
	// ӽ Ʈ(Dummy)...
	else if (strcmp (c_string, "$$$DUMMY") == 0) {

		if ((obj = (c_OBJECT *)malloc (sizeof (c_OBJECT))) == NULL)
			return clax_err_nomem;

		obj->id = 1024 + c_id;
		obj->flags = clax_obj_dummy;
		obj->numverts = 0;
		obj->numfaces = 0;
		obj->vertices = NULL;
		obj->faces = NULL;
		vec_zero (&obj->translate);
		vec_zero (&obj->scale);
		qt_zero (&obj->rotate);
		clax_add_world (clax_obj_object, obj);
	}
	else {
		// ڿ Ʈ ˻Ѵ.
		clax_byname (c_string, &node);

		// ã ϸ  Ѵ.
		if (!node) 
			return clax_err_undefined;

		obj = (c_OBJECT *)node->object;
		cam = (c_CAMERA *)node->object;
		light = (c_LIGHT *)node->object;
	}

	if (fread (flags, sizeof (flags), 1, f) != 1) 
		return clax_err_badfile;

	if (fread (&parent, sizeof (parent), 1, f) != 1) 
		return clax_err_badfile;

#ifdef	CLAX_DEBUG
	PrintTab( SubCount );
	fprintf( fp, "Parent Num : %d\n", parent );
#endif

	// θ Ʈ id ã´.
	if (parent != -1) {

		for (pnode = clax_scene->keyframer; pnode; pnode = pnode->next) {

			if (pnode->id == parent)
				wparent = ((c_OBJECT *)pnode->object)->id;

		}

	}

	switch(c_chunk_last) {

		// object Ʈ
		case CHUNK_TRACKINFO:
			obj->parent = wparent;

			if (flags[0] & 0x800) 
				// Ʈ ׻ .
				obj->flags |= clax_obj_chidden;

			if ((track = malloc (sizeof (t_OBJECT))) == NULL) 
				return clax_err_nomem;

			memset (track, 0, sizeof (t_OBJECT));
			clax_add_track (clax_track_object, c_id, parent, track, obj);
			c_node = obj;
			break;

		// ī޶ Ű
		case CHUNK_TRACKCAMERA:
			cam->parent1 = wparent;
			if ((track = malloc (sizeof (t_CAMERA))) == NULL) 
				return clax_err_nomem;

			memset (track, 0, sizeof (t_CAMERA));
			clax_add_track (clax_track_camera, c_id, parent, track, cam);
			break;

		// Camera Target KeyFrame
		case CHUNK_TRACKCAMTGT:
			cam->parent2 = wparent;
			if ((track = malloc (sizeof (t_CAMERATGT))) == NULL)
				return clax_err_nomem;

			memset (track, 0, sizeof (t_CAMERATGT));
			clax_add_track (clax_track_cameratgt, c_id, parent, track, cam);
			break;

		// Light KeyFrame
		case CHUNK_TRACKLIGHT:

			light->parent1 = wparent;

			if ((track = malloc (sizeof (t_LIGHT))) == NULL) 
				return clax_err_nomem;

			memset (track, 0, sizeof (t_LIGHT));
			clax_add_track (clax_track_light, c_id, parent, track, light);
			break;

		case CHUNK_TRACKSPOTL:
			light->parent1 = wparent;
			if ((track = malloc (sizeof (t_SPOTLIGHT))) == NULL)
				return clax_err_nomem;
			memset (track, 0, sizeof (t_SPOTLIGHT));
			clax_add_track (clax_track_spotlight, c_id, parent, track, light);
			break;

		case CHUNK_TRACKLIGTGT:
			light->parent2 = wparent;
			if ((track = malloc (sizeof (t_LIGHTTGT))) == NULL)
				return clax_err_nomem;
			memset (track, 0, sizeof (t_LIGHTTGT));
			clax_add_track (clax_track_lighttgt, c_id, parent, track, light);
			break;

		case CHUNK_AMBIENTKEY:
			if ((track = malloc (sizeof (t_AMBIENT))) == NULL) 
				return clax_err_nomem;

			memset (track, 0, sizeof (t_AMBIENT));
			clax_add_track (clax_track_ambient, c_id, parent, track, amb);
	}

	return clax_err_ok;
}


//	Track pivot point reader.
static int read_TRACKPIVOT (FILE *f)
{
	c_OBJECT *obj = (c_OBJECT *)c_node;
	int       i;

	fread( &obj->pivot.x, sizeof(float), 1, f );
	fread( &obj->pivot.z, sizeof(float), 1, f );
	fread( &obj->pivot.y, sizeof(float), 1, f );

	for (i = 0; i < obj->numverts; i++)
	     vec_sub (&obj->vertices[i].vert, &obj->pivot, &obj->vertices[i].vert);

	return clax_err_ok;
}


//	Key flags/spline reader.
static int read_KFLAGS (FILE *f, WORD *nFrame, t_KEY *key)
{
	WORD	unknown, flags;
	int		i;
	float	dat;

	key->tens	  = 0.0f;
	key->cont	  = 0.0f;
	key->bias	  = 0.0f;
	key->easeto	  = 0.0f;
	key->easefrom = 0.0f;

	if (fread (nFrame, sizeof (WORD), 1, f) != 1) 
		return clax_err_badfile;

	if (fread (&unknown, sizeof (WORD), 1, f) != 1) 
		return clax_err_badfile;

	if (fread (&flags, sizeof (WORD), 1, f) != 1) 
		return clax_err_badfile;

	// flags & 1, flags & 2, flags & 4, flags & 8........ flags & 65536
	for (i = 0; i < 16; i++) {

		if (flags & (1 << i)) {

			if (fread (&dat, sizeof (dat), 1, f) != 1) 
				return clax_err_badfile;

			switch (i) {
				case 0: 
					key->tens = dat;
					break;

				case 1: 
					key->cont = dat; 
					break;

				case 2: 
					key->bias = dat; 
					break;

				case 3: 
					key->easeto = dat; 
					break;

				case 4: 
					key->easefrom = dat;
			}
		}
	}

	return clax_err_ok;
}


// Track flags reader.
static int read_TFLAGS (FILE *f, t_TRACK *track, WORD *n)
{
	WORD flags[7];

	if (fread (flags, sizeof (flags), 1, f) != 1)
		return clax_err_badfile;

#ifdef	CLAX_DEBUG
	PrintTab( SubCount );
	fprintf( fp, "TrackFlags:\n" );
#endif


	if ((flags[0] & 0x02) == 0x02)
		track->flags = clax_track_repeat;

	if ((flags[0] & 0x03) == 0x03)
		track->flags = clax_track_loop;

	//   Ѵ.
	*n = flags[5];

	return clax_err_ok;
}


// Track position reader.
static int read_TRACKPOS (FILE *f)
{
	t_TRACK		*track;
	t_KEY		*key;
	WORD		n, nFrame;

	//  ü ҴѴ.
	track = alloc_track();

	//    ´. (Լ ȿ ÷׸ о Ѵ.)
	if (read_TFLAGS (f, track, &n) != 0)
		return clax_err_badfile;

	while (n-- > 0) {
	    if ((key = (t_KEY *)malloc (sizeof (t_KEY))) == NULL)
			return clax_err_nomem;

		if (read_KFLAGS (f, &nFrame, key))
			return clax_err_badfile;

		fread( &key->val._vect.x, sizeof(float), 1, f );
		fread( &key->val._vect.z, sizeof(float), 1, f );
		fread( &key->val._vect.y, sizeof(float), 1, f );

		add_key (track, key, nFrame);
	}

	spline_init (track);
	clax_set_track (clax_key_pos, c_id, track);

	return clax_err_ok;
}


// Track color reader.
static int read_TRACKCOLOR (FILE *f)
{
	t_TRACK		*track;
	t_KEY		*key;
	WORD		n, nFrame;

	track = alloc_track();

	if (read_TFLAGS (f, track, &n) != 0) 
		return clax_err_badfile;

	while (n-- > 0) {
		if ((key = (t_KEY *)malloc (sizeof (t_KEY))) == NULL)
			return clax_err_nomem;

		if (read_KFLAGS (f, &nFrame, key)) 
			return clax_err_badfile;

		fread( &key->val._vect.x, sizeof(float), 1, f );
		fread( &key->val._vect.z, sizeof(float), 1, f );
		fread( &key->val._vect.y, sizeof(float), 1, f );

		add_key (track, key, nFrame);
	}

	spline_init (track);
	clax_set_track (clax_key_color, c_id, track);

	return clax_err_ok;
}


// Track rotation reader.
static int read_TRACKROT (FILE *f)
{
	t_TRACK *track;
	t_KEY   *key;
	c_QUAT   q, old;
	float    pos[4];
	WORD     keys,n, nf;
	//int		 angle = 0;
	float	 angle = 0.0f;

	track = alloc_track();
	qt_identity (&old);

	if (read_TFLAGS (f, track, &n) != 0) 
		return clax_err_badfile;

	keys = n;
	while (n-- > 0) {
		if ((key = (t_KEY *)malloc (sizeof (t_KEY))) == NULL)
			return clax_err_nomem;

		if (read_KFLAGS (f, &nf, key)) 
			return clax_err_badfile;

		if (fread (pos, sizeof(pos), 1, f) != 1) 
			return clax_err_badfile;

		qt_fromang (pos[0], pos[1], pos[2], pos[3], &q);
		qt_swap (&q);

		// !!! FIX !!! I SAID ANGLE IS ABSOLUTE!!!!!!!!!
		//if (keys == n - 1)
		//	angle = pos[0]; 
		//else 
		//	angle += pos[0];

		angle += pos[0];

		qt_make (angle, pos[1], pos[2], pos[3], &key->val._quat);
		qt_swap (&key->val._quat);

		qt_mul (&q, &old, &old);
		qt_copy (&old, &key->qa);

		add_key (track, key, nf);
	}

	spline_initrot (track);
	clax_set_track (clax_key_rotate, c_id, track);
	return clax_err_ok;
}


// Track scale reader.
static int read_TRACKSCALE (FILE *f)
{
	t_TRACK	*track;
	t_KEY	*key;
	WORD	n, nf;

	track = alloc_track();
	if (read_TFLAGS (f, track, &n) != 0) 
		return clax_err_badfile;

	while (n-- > 0) {
		if ((key = (t_KEY *)malloc (sizeof (t_KEY))) == NULL)
			return clax_err_nomem;

		if (read_KFLAGS (f, &nf, key)) 
			return clax_err_badfile;

		fread( &key->val._vect.x, sizeof(float), 1, f );
		fread( &key->val._vect.z, sizeof(float), 1, f );
		fread( &key->val._vect.y, sizeof(float), 1, f );

		add_key (track, key, nf);
	}

	spline_init (track);
	clax_set_track (clax_key_scale, c_id, track);
	return clax_err_ok;
}


/*
  read_TRACKFOV: Track FOV reader.
*/
static int read_TRACKFOV (FILE *f)
{
	t_TRACK *track;
	t_KEY *key;
	WORD  n, nf;
	float fov;

	track = alloc_track();
	if (read_TFLAGS (f, track, &n) != 0)
		return clax_err_badfile;

	while (n-- > 0) {
		if ((key = (t_KEY *)malloc (sizeof (t_KEY))) == NULL)
			return clax_err_nomem;

		if (read_KFLAGS (f, &nf, key)) 
			return clax_err_badfile;

		if (fread (&fov, sizeof (fov), 1, f) != 1) 
			return clax_err_badfile;

		key->val._float = fov;
		add_key (track, key, nf);
	}

	spline_init (track);
	clax_set_track (clax_key_fov, c_id, track);
	return clax_err_ok;
}


/*
  read_TRACKROLL: Track ROLL reader.
*/
static int read_TRACKROLL (FILE *f)
{
	t_TRACK *track;
	t_KEY *key;
	WORD   n, nf;
	float  roll;

	track = alloc_track();
	if (read_TFLAGS (f, track, &n) != 0) 
		return clax_err_badfile;

	while (n-- > 0) {
		if ((key = (t_KEY *)malloc (sizeof (t_KEY))) == NULL)
			return clax_err_nomem;

		if (read_KFLAGS (f, &nf, key)) 
			return clax_err_badfile;

		if (fread(&roll, sizeof(roll), 1, f) != 1) 
			return clax_err_badfile;

		key->val._float = roll;
		add_key (track, key, nf);
	}

	spline_init (track);
	clax_set_track (clax_key_roll, c_id, track);
	return clax_err_ok;
}


/*
  read_TRACKMORPH: Track morph reader.
*/
static int read_TRACKMORPH (FILE *f)
{
  t_TRACK *track;
  t_KEY  *key;
  w_NODE *node;
  WORD    n, nf;

  track = alloc_track();
  if (read_TFLAGS (f, track, &n) != 0) return clax_err_badfile;
  while (n-- > 0) {
    if ((key = (t_KEY *)malloc (sizeof (t_KEY))) == NULL)
      return clax_err_nomem;
    if (read_KFLAGS (f, &nf, key)) return clax_err_badfile;
    if (read_ASCIIZ (f)) return clax_err_badfile;
    clax_byname (c_string, &node);
    if (!node) return clax_err_undefined;
    key->val._int = ((c_OBJECT *)node->object)->id;
    add_key (track, key, nf);
  }
  clax_set_track (clax_key_morph, c_id, track);
  return clax_err_ok;
}


/*
  read_TRACKHIDE: Track hide reader.
*/
static int read_TRACKHIDE (FILE *f)
{
  t_TRACK *track;
  t_KEY *key;
  WORD   unknown[2];
  WORD   n, nf;
  int    hide = 0;

  track = alloc_track();
  if (read_TFLAGS (f, track, &n) != 0) return clax_err_badfile;
  while (n-- > 0) {
    if ((key = (t_KEY *)malloc (sizeof (t_KEY))) == NULL)
      return clax_err_nomem;
    if (fread (&nf, sizeof (nf), 1, f) != 1) return clax_err_badfile;
    if (fread (unknown, sizeof (WORD), 2, f) != 2) return clax_err_badfile;
    key->val._int = (hide ^= 1);
    add_key (track, key, nf);
  }
  clax_set_track (clax_key_hide, c_id, track);
  return clax_err_ok;
}


// Chunk reader.
static int read_CHUNK (FILE *f, c_CHUNK *h)
{
	if (fread (&h->chunk_id, sizeof (WORD), 1, f) != 1)
		return clax_err_badfile;

	if (fread (&h->chunk_size, sizeof (DWORD), 1, f) != 1)
		return clax_err_badfile;

	return clax_err_ok;
}



/*****************************************************************************
  chunk readers control
*****************************************************************************/

// Recursive chunk reader (world).
static int ChunkReaderWorld (FILE *f, long p, WORD parent)
{
	c_CHUNK h;
	long    pc;
	int     n, i, error;

	c_chunk_last = parent;

	while ((pc = ftell (f)) < p) {

		if (read_CHUNK (f, &h) != 0)
			return clax_err_badfile;

		//  chunk id Ѵ.
		c_chunk_curr = h.chunk_id;
		n = -1;

		// ǵ chunk Ʈ ؼ id Ѵ.
		for(i = 0; i < sizeof (world_chunks) / sizeof (world_chunks[0]); i++) {
			if (h.chunk_id == world_chunks[i].id) {
				n = i;

#ifdef	CLAX_DEBUG
				//   Ѵ.
				PrintTab( SubCount );
				fprintf(fp, "SUB %d : %s\n", SubCount, world_chunks[i].Str);
#endif
				break;
			}
		}

		if (n < 0) 
			// Ʈ  chunk id ŭ  ͸ skipѴ.
			fseek (f, pc + h.chunk_size, SEEK_SET);
		else {
			pc = pc + h.chunk_size;
			// شǴ Լ ȣѴ.
			if ((error = world_chunks[n].func (f)) != clax_err_ok)
				return error;

			// ӵ chunk б ...
			if (world_chunks[n].sub) {

				SubCount++;

				if ((error = ChunkReaderWorld (f, pc, h.chunk_id)) != clax_err_ok)
					return error;
			}

			fseek (f, pc, SEEK_SET);
			c_chunk_prev = h.chunk_id;
		}

		// д ߿   ...
		if (ferror (f)) 
			return clax_err_badfile;
	}


	// Ʈ normal, material Ѵ.
	if( IsObject == TRUE && SubCount == 3) {
		IsObject = FALSE;

		CalcObject( (c_OBJECT*)c_node );

#ifdef	CLAX_DEBUG

		PrintTab( SubCount );
		fprintf(fp, "Calc Object!\n");

#endif
	}

	if( SubCount > 0 ) {
		SubCount--;

#ifdef	CLAX_DEBUG
		PrintTab( SubCount );
		fprintf(fp, "SUB %d end!", SubCount);
#endif
	}

#ifdef	CLAX_DEBUG
	fprintf(fp, "\n" );
#endif

	return clax_err_ok;
}


// Recursive chunk reader (keyframer).
static int ChunkReaderKey (FILE *f, long p, WORD parent)
{
	c_CHUNK h;
	long    pc;
	int     n, i, error;

	c_chunk_last = parent;

	while ((pc = ftell (f)) < p) {

		if (read_CHUNK (f, &h) != 0) 
			return clax_err_badfile;

		c_chunk_curr = h.chunk_id;
		n = -1;

		for (i = 0; i < sizeof (key_chunks) / sizeof (key_chunks[0]); i++) {

			if (h.chunk_id == key_chunks[i].id) {
				n = i;

#ifdef	CLAX_DEBUG
				PrintTab( SubCount );
				fprintf(fp, "SUB %d : %s\n", SubCount, key_chunks[i].Str);
#endif
				break;
			}

		}

	    if (n < 0) 
			fseek (f, pc + h.chunk_size, SEEK_SET);
		else {
			pc = pc + h.chunk_size;

			if ((error = key_chunks[n].func (f)) != 0) 
				return error;

			if (key_chunks[n].sub) {

				SubCount++;

				if ((error = ChunkReaderKey (f, pc, h.chunk_id)) != 0)
					return error;
			}

			fseek (f, pc, SEEK_SET);
			c_chunk_prev = h.chunk_id;
		}

		if (ferror (f)) 
			return clax_err_badfile;
	}


	if( SubCount > 0 ) {
		SubCount--;

	#ifdef	CLAX_DEBUG
		PrintTab( SubCount );
		fprintf(fp, "SUB %d end!", SubCount);
	#endif
	}

#ifdef	CLAX_DEBUG
	fprintf(fp, "\n" );
#endif

	return clax_err_ok;
}


/*****************************************************************************
  world/motion load routines
*****************************************************************************/

//	loads mesh data from 3ds file "filename" into scene "scene".
int32 clax_load_mesh_3DS (FILE *f)
{
	BYTE		version;
	long		length;
	int			Result;
	c_MATERIAL *MatObj;

	c_id = 0;
	fseek (f, 0, SEEK_END);
	// ȭ  ˾Ƴ.
	length = ftell (f);
	fseek (f, 28L, SEEK_SET);

	if (fread (&version, sizeof (BYTE), 1, f) != 1)
		return clax_err_badfile;

	// 3DS 3.0+ supported
	if (version < 2) 
		return clax_err_badver; 

	fseek (f, 0, SEEK_SET);

	
	// material ü ŭ ޸𸮸 ҴѴ.
	// (Default Material Ѵ.)
	if ( (MatObj = (c_MATERIAL *)malloc (sizeof(c_MATERIAL))) == NULL )
		return clax_err_nomem;

	InitMaterial (MatObj);
	MatObj->name = strcopy ("DEFAULT");

	// ID -1 δ.
	MatObj->id = -1;

	// Ʈ Ʈ ߰Ѵ.
	clax_add_world (clax_obj_material, MatObj);

	SubCount = 0;

#ifdef	CLAX_DEBUG
	fp = fopen( "RESULT3DS.TXT", "wt" );
#endif

	Result = ChunkReaderWorld (f, length, 0);

#ifdef	CLAX_DEBUG
	fclose(fp);
#endif

	return Result;
}


// loads motion data from 3ds file "filename" into scene "scene".
int32 clax_load_motion_3DS (FILE *f)
{
	BYTE	version;
	long	length;
	int		Result;

	c_id = -1;
	fseek (f, 0, SEEK_END);
	length = ftell (f);
	fseek (f, 28L, SEEK_SET);

	if (fread (&version, sizeof (BYTE), 1, f) != 1)
		return clax_err_badfile;

	// 3DS 3.0+ supported
	if (version < 2) 
		return clax_err_badver;

	fseek (f, 0, SEEK_SET);

	SubCount = 0;

#ifdef	CLAX_DEBUG
	fp = fopen("KEYFORM.TXT", "wt");
#endif

	Result = ChunkReaderKey(f, length, 0);

#ifdef	CLAX_DEBUG
	fclose(fp);
#endif

	return Result;
}
