/*MATRIX.C********************************************************************

   Clax: Portable keyframing library
         see inclosed LICENSE.TXT for licensing terms.

         for documentation, refer to CLAX.TXT

   author           : Borzom
   file created     : 17/04/97
   file description : matrix math

   revision history :
     v0.10 (17/04/97) Borzom: Initial version.

   notes            :
     ( xx , xy , xz , xw ) clax 3x4 matrix structure.
     ( yx , yy , yz , yw )
     ( zx , zy , zz , zw )

*****************************************************************************/

#include <math.h>
#include <stdio.h>
#include "clax.h"
#include "claxi.h"


static c_MATRIX Tidentity = {
	{ 1, 0, 0, 0 },
	{ 0, 1, 0, 0 },
	{ 0, 0, 1, 0 }
};


// clear matrix.
void mat_zero (c_MATRIX out)
{
	int16 i, j;

	for(i = 0; i < 3; i++)
		for (j = 0; j < 4; j++)
			out[i][j] = 0.0f;
}
/*
D3DMATRIX ZeroMatrix()
{
    D3DMATRIX	m;

    m(0,0) = 0.0f;	m(1,0) = 0.0f;	m(2,0) = 0.0f;	m(3,0) = 0.0f;
    m(0,1) = 0.0f;	m(1,1) = 0.0f;	m(2,1) = 0.0f;	m(3,1) = 0.0f;
    m(0,2) = 0.0f;	m(1,2) = 0.0f;	m(2,2) = 0.0f;	m(3,2) = 0.0f;
    m(0,3) = 0.0f;	m(1,3) = 0.0f;	m(2,3) = 0.0f;	m(3,3) = 0.0f;

    return m;
}
*/


//  make identity matrix.
void mat_identity (c_MATRIX &out)
{
	int16 i, j;

	for (i = 0; i < 3; i++)
		for (j = 0; j < 4; j++)
			out[i][j] = Tidentity[i][j];
}


//  swap y/z in matrix.
void mat_swap ( c_MATRIX a)
{
	int   i;
	float tmp;

#ifdef CLAX_SWAP_YZ

	// swap columns
	for (i = 0; i < 3; i++) {
		tmp = a[i][Y];
		a[i][Y] = a[i][Z];
		a[i][Z] = tmp;
	}

	// swap rows
	for (i = 0; i < 4; i++) {
	    tmp = a[Y][i];
	    a[Y][i] = a[Z][i];
		a[Z][i] = tmp;
	}
#endif
}


//	matrix copy.
void mat_copy (c_MATRIX a, c_MATRIX out)
{
	int	 i, j;

	for(i = 0; i < 3; i++)
		for(j = 0; j < 4; j++)
			out[i][j] = a[i][j];
}



void mat_print (c_MATRIX a)
{
/*
  mat_print: print matrix on stdout.
*/
  printf ("xx: %9.3f xy: %9.3f xz: %9.3f xw: %9.3f\n",
          a[X][X], a[X][Y], a[X][Z], a[X][W]);
  printf ("yx: %9.3f yy: %9.3f yz: %9.3f yw: %9.3f\n",
          a[Y][X], a[Y][Y], a[Y][Z], a[Y][W]);
  printf ("zx: %9.3f zy: %9.3f zz: %9.3f zw: %9.3f\n",
          a[Z][X], a[Z][Y], a[Z][Z], a[Z][W]);
}

void mat_add (c_MATRIX a, c_MATRIX b, c_MATRIX out)
{
/*
  mat_add: matrix addition.
*/
  int16    i, j;
  c_MATRIX temp;
  for (i = 0; i < 3; i++)
    for (j = 0; j < 4; j++)
      temp[i][j] = a[i][j]+b[i][j];
  mat_copy (temp, out);
}

void mat_sub (c_MATRIX a, c_MATRIX b, c_MATRIX out)
{
/*
  mat_sub: matrix substraction.
*/
  int16    i, j;
  c_MATRIX temp;

  for (i = 0; i < 3; i++)
    for (j = 0; j < 4; j++)
      temp[i][j] = a[i][j] - b[i][j];
  mat_copy (temp, out);
}


//	matrix multiplication.
void mat_mul (c_MATRIX a, c_MATRIX b, c_MATRIX out)
{
	c_MATRIX temp;

	temp[X][X] = a[X][X] * b[X][X] + a[X][Y] * b[Y][X] + a[X][Z] * b[Z][X];
	temp[X][Y] = a[X][X] * b[X][Y] + a[X][Y] * b[Y][Y] + a[X][Z] * b[Z][Y];
	temp[X][Z] = a[X][X] * b[X][Z] + a[X][Y] * b[Y][Z] + a[X][Z] * b[Z][Z];
	temp[X][W] = a[X][X] * b[X][W] + a[X][Y] * b[Y][W] + a[X][Z] * b[Z][W] + a[X][W];

	temp[Y][X] = a[Y][X] * b[X][X] + a[Y][Y] * b[Y][X] + a[Y][Z] * b[Z][X];
	temp[Y][Y] = a[Y][X] * b[X][Y] + a[Y][Y] * b[Y][Y] + a[Y][Z] * b[Z][Y];
	temp[Y][Z] = a[Y][X] * b[X][Z] + a[Y][Y] * b[Y][Z] + a[Y][Z] * b[Z][Z];
	temp[Y][W] = a[Y][X] * b[X][W] + a[Y][Y] * b[Y][W] + a[Y][Z] * b[Z][W] + a[Y][W];
 
	temp[Z][X] = a[Z][X] * b[X][X] + a[Z][Y] * b[Y][X] + a[Z][Z] * b[Z][X];
	temp[Z][Y] = a[Z][X] * b[X][Y] + a[Z][Y] * b[Y][Y] + a[Z][Z] * b[Z][Y];
	temp[Z][Z] = a[Z][X] * b[X][Z] + a[Z][Y] * b[Y][Z] + a[Z][Z] * b[Z][Z];
	temp[Z][W] = a[Z][X] * b[X][W] + a[Z][Y] * b[Y][W] + a[Z][Z] * b[Z][W] + a[Z][W];
	
	mat_copy (temp, out);
}


void mat_transpose (c_MATRIX a, c_MATRIX out)
{
/*
  mat_transpose: transpose matrix.
*/
  int      i, j;
  c_MATRIX temp;

  for (i = 0; i < 3; i++)
    for (j = 0; j < 3; j++)
      temp[i][j] = a[j][i];
  for (i = 0; i < 3; i++)
    temp
	[i][W] = a[i][W];
  mat_copy (temp, out);
}


//  inverse matrix calculation (non-singular).
int32 mat_inverse (c_MATRIX a, c_MATRIX &out)
{
	int16    i, j;
	float    scale;
	c_MATRIX temp;

	for (i = 0; i < 3; i++) {
		scale = a[i][X] * a[i][X] + a[i][Y] * a[i][Y] + a[i][Z] * a[i][Z];

		if (scale == 0.0f) {
			mat_identity (out);
			return clax_err_singular;
		}

		scale = 1.0f / scale;
		for (j = 0; j < 3; j++) 
			temp[i][j] = a[j][i] * scale;

		temp[i][W] = -(temp[i][X] * a[i][W] + temp[i][Y] * a[Y][W] + temp[i][Z] * a[Z][W]);
	}

	mat_copy (temp, out);
	return clax_err_ok;
}


//  inverse matrix scale.
int32 mat_invscale (c_MATRIX &a, c_MATRIX &out)
{
	int	     i, j;
	float    scale;
	c_MATRIX temp;

	for (i = 0; i < 3; i++) {
		scale = a[i][X] * a[i][X] + a[i][Y] * a[i][Y] + a[i][Z] * a[i][Z];

		if (scale == 0.0f) {
			mat_identity (out);
			return clax_err_singular;
		}

		scale = 1.0f / scale;

		for (j = 0; j < 3; j++) {
			temp[i][j] = a[i][j] * scale;
		}

		temp[i][W] = a[i][W];
	}

	mat_copy (temp, out);
	return clax_err_ok;
}


//  normalize matrix.
int32 mat_normalize (c_MATRIX a, c_MATRIX out)
{
	int			i, j;
	float		len;
	c_MATRIX	temp;

	for (i = 0; i < 3; i++) {
		len = (float)sqrt (a[i][X] * a[i][X] + a[i][Y] * a[i][Y] + a[i][Z] * a[i][Z]);

		if (len != 0.0f) 
			len = 1.0f / len; 
		else 
			len = 1.0f;

		for (j = 0; j < 3; j++)
			temp[i][j] = a[i][j] * len;

		temp[i][W] = a[i][W];
	}

	mat_copy (temp, out);
	return clax_err_ok;
}



void mat_toeuler (c_MATRIX mat, c_VECTOR *out)
{
/*
  mat_toeuler: convert rotation matrix to euler angles.
*/
   float siny, cosy;

	siny = mat[Z][X];
	cosy = (float)sqrt (1.0 - siny*siny);

	if (mat[X][X] < 0.0f && mat[Z][Z] < 0.0f) 
		cosy = -cosy;

	if (cosy != 0.0) {
		out->x = (float)atan2 (mat[Z][Y] / cosy, mat[Z][Z] / cosy);
		out->y = (float)atan2 (siny, cosy);
		out->z = (float)atan2 (mat[Y][X] / cosy, mat[X][X] / cosy);
	} 
	else {
		out->x = (float)atan2 (-mat[Y][Z], mat[Y][Y]);
		out->y = (float)asin (siny);
		out->z = 0.0f;
   }
}


/*
  mat_pretrans: create a pre-translation matrix.
*/
void mat_pretrans (c_VECTOR *v, c_MATRIX mat, c_MATRIX out)
{
	c_MATRIX temp;
	int      i, j;

	for (i = 0; i < 3; i++)
		for (j = 0; j < 3; j++)
			temp[i][j] = mat[i][j];

	temp[X][W] = mat[X][X]*v->x + mat[X][Y]*v->y + mat[X][Z]*v->z + mat[X][W];
	temp[Y][W] = mat[Y][X]*v->x + mat[Y][Y]*v->y + mat[Y][Z]*v->z + mat[Y][W];
	temp[Z][W] = mat[Z][X]*v->x + mat[Z][Y]*v->y + mat[Z][Z]*v->z + mat[Z][W];

	mat_copy (temp, out);
}


/*
  mat_settrans: create a translation matrix.
*/
/*
void mat_settrans (c_VECTOR *v, c_MATRIX out)
{

  mat_identity (out);
  out[X][W] = v->x;
  out[Y][W] = v->y;
  out[Z][W] = v->z;
}
*/


//	create a scale matrix.
void mat_setscale (c_VECTOR *v, c_MATRIX &out)
{
	mat_identity (out);

	out[0][0] = v->x;
	out[1][1] = v->y;
	out[2][2] = v->z;
}


/*
  mat_rotateX: create rotation matrix around X axis.
*/
/*
void mat_rotateX (float ang, c_MATRIX out)
{
	float sinx, cosx;

	mat_identity (out);
	sinx = (float)sin (ang);
	cosx = (float)cos (ang);
	out[Y][Y] = cosx;
	out[Y][Z] = sinx;
	out[Z][Y] = -sinx;
	out[Z][Z] = cosx;


  // y, z ٲ  . (3dsǥ ٸ ...)
    m(0,0) = 1.0f; m(1,0) = 0.0f;   m(2,0) = 0.0f;   m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = cosine; m(2,1) = sine;   m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = -sine;  m(2,2) = cosine; m(3,2) = 0.0f;
    m(0,3) = 0.0f; m(1,3) = 0.0f;   m(2,3) = 0.0f;   m(3,3) = 1.0f;
}
*/


/*
  mat_rotateY: create rotation matrix around Y axis.
*/
/*
void mat_rotateY (float ang, c_MATRIX out)
{
	float siny, cosy;

	mat_identity (out);
	siny = (float)sin (ang);
	cosy = (float)cos (ang);
	out[X][X] =  cosy;
	out[X][Z] = -siny;
	out[Z][X] =  siny;
	out[Z][Z] =  cosy;
}
*/


/*
  mat_rotateZ: create rotation matrix around Z axis.
*/
/*
void mat_rotateZ (float ang, c_MATRIX out)
{
	float sinz, cosz;

	mat_identity (out);
	sinz = (float)sin (ang);
	cosz = (float)cos (ang);
	out[X][X] =  cosz;
	out[X][Y] =  sinz;
	out[Y][X] = -sinz;
	out[Y][Y] =  cosz;
}
*/


//	multiply a vector by matrix (out = [a] * b)
void mat_mulvec (c_MATRIX a, c_VECTOR *b, c_VECTOR *out)
{
	c_VECTOR	temp;

	temp.x = b->x * a[X][X] + b->y * a[X][Y] + b->z * a[X][Z] + a[X][W];
	temp.y = b->x * a[Y][X] + b->y * a[Y][Y] + b->z * a[Y][Z] + a[Y][W];
	temp.z = b->x * a[Z][X] + b->y * a[Z][Y] + b->z * a[Z][Z] + a[Z][W];

	vec_copy (&temp, out);
}


//	multiply a normal by matrix (out = [a] * b)
void mat_mulnorm (c_MATRIX a, c_VECTOR *b, c_VECTOR *out)
{
	c_VECTOR temp;

	temp.x = b->x * a[X][X] + b->y * a[X][Y] + b->z * a[X][Z];
	temp.y = b->x * a[Y][X] + b->y * a[Y][Y] + b->z * a[Y][Z];
	temp.z = b->x * a[Z][X] + b->y * a[Z][Y] + b->z * a[Z][Z];

	vec_copy (&temp, out);
}


// Rotate matrix about X axis
D3DMATRIX RotateXMatrix( const float rads )
{
	float	cosine, sine;
	D3DMATRIX   m;

	cosine = (float) cos(rads);
    sine   = (float) sin(rads);

	// y, z ٲ  . (3dsǥ ٸ ...)
    m(0,0) = 1.0f; m(1,0) = 0.0f;   m(2,0) = 0.0f;   m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = cosine; m(2,1) = sine;   m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = -sine;  m(2,2) = cosine; m(3,2) = 0.0f;
    m(0,3) = 0.0f; m(1,3) = 0.0f;   m(2,3) = 0.0f;   m(3,3) = 1.0f;

    return m;
}



// Rotate matrix about Y axis
D3DMATRIX RotateYMatrix( const float rads )
{
	float	cosine, sine;

	cosine = (float) cos(rads);
    sine   = (float) sin(rads);

    D3DMATRIX   m;

	// y, z ٲ  . (3dsǥ ٸ ...)
    m(0,0) = cosine; m(1,0) = 0.0f; m(2,0) = -sine;  m(3,0) = 0.0f;
    m(0,1) = 0.0f;   m(1,1) = 1.0f; m(2,1) = 0.0f;   m(3,1) = 0.0f;
    m(0,2) = sine;   m(1,2) = 0.0f; m(2,2) = cosine; m(3,2) = 0.0f;
    m(0,3) = 0.0f;   m(1,3) = 0.0f; m(2,3) = 0.0f;   m(3,3) = 1.0f;

    return m;
}

// Rotate matrix about Z axis
D3DMATRIX RotateZMatrix( const float rads )
{
	float	cosine, sine;
	D3DMATRIX   m;

	cosine = (float) cos(rads);
    sine   = (float) sin(rads);

	// y, z ٲ  . (3dsǥ ٸ ...)
    m(0,0) = cosine; m(1,0) = sine;   m(2,0) = 0.0f; m(3,0) = 0.0f;
    m(0,1) = -sine;  m(1,1) = cosine; m(2,1) = 0.0f; m(3,1) = 0.0f;
    m(0,2) = 0.0f;   m(1,2) = 0.0f;   m(2,2) = 1.0f; m(3,2) = 0.0f;
    m(0,3) = 0.0f;   m(1,3) = 0.0f;   m(2,3) = 0.0f; m(3,3) = 1.0f;
    
	return m;
}
