/*VECTOR.C********************************************************************

   Clax: Portable keyframing library
         see inclosed LICENSE.TXT for licensing terms.

         for documentation, refer to CLAX.TXT

   author           : Borzom
   file created     : 29/04/97
   file description : vector math

   revision history :
     v0.10 (29/04/97) Borzom: Initial version.

   notes            :

*****************************************************************************/

#include <math.h>
#include <stdio.h>
#include "clax.h"
#include "claxi.h"


//	create vector (out = [x,y,z]).
void vec_make (float x, float y, float z, c_VECTOR *out)
{
	out->x = x;
	out->y = y;
	out->z = z;
}


// zero vector (out = [0,0,0]).
void vec_zero (c_VECTOR *out)
{
	out->x = 0.0f;
	out->y = 0.0f;
	out->z = 0.0f;
}


//  vector copy (out = a).
void vec_copy (c_VECTOR *a, c_VECTOR *out)
{
	out->x = a->x;
	out->y = a->y;
	out->z = a->z;
}



void vec_print (c_VECTOR *a)
{
/*
  vec_print: print vector on stdout.
*/
  printf ("x: %9.3f y: %9.3f z: %9.3f\n", a->x, a->y, a->z);
}


/*
  vec_add: vector addition (out = a+b).
*/
void vec_add (c_VECTOR *a, c_VECTOR *b, c_VECTOR *out)
{
	c_VECTOR	Temp;

	Temp.x = a->x + b->x;
	Temp.y = a->y + b->y;
	Temp.z = a->z + b->z;

	vec_copy( &Temp, out );
}


//	vector substraction (out = a - b).
void vec_sub (c_VECTOR *a, c_VECTOR *b, c_VECTOR *out)
{
	c_VECTOR temp;

	temp.x = a->x - b->x;
	temp.y = a->y - b->y;
	temp.z = a->z - b->z;

	vec_copy (&temp, out);
}



void vec_mul (c_VECTOR *a, c_VECTOR *b, c_VECTOR *out)
{
/*
  vec_mul: vector multiplication (out = a*b).
*/
  c_VECTOR temp;

  temp.x = a->x * b->x;
  temp.y = a->y * b->y;
  temp.z = a->z * b->z;
  vec_copy (&temp, out);
}


//  vector multiplication by a scalar.
void vec_scale (c_VECTOR *a, float s, c_VECTOR *out)
{
	out->x = a->x * s;
	out->y = a->y * s;
	out->z = a->z * s;
}


/*
  vec_scale: vector multiplication by a scalar.
*/
void vec_rescale (c_VECTOR *a, float s, c_VECTOR *out)
{
	c_VECTOR temp;
	float    len;

	len = (a->x*a->x + a->y*a->y + a->z*a->z);
	if (len == 0.0f) {
	    out->x = 0.0f;
		out->y = 0.0f;
		out->z = 0.0f;
		return;
	}

	s *= (1.0f / len);
	temp.x = a->x * s;
	temp.y = a->y * s;
	temp.z = a->y * s;

	vec_copy (&temp, out);
}

void vec_negate (c_VECTOR *a, c_VECTOR *out)
{
/*
  vec_negate: vector negation (out = -a).
*/
  c_VECTOR temp;

  temp.x = -a->x;
  temp.y = -a->y;
  temp.z = -a->z;
  vec_copy (&temp, out);
}

int32 vec_equal (c_VECTOR *a, c_VECTOR *b)
{
/*
  vec_equal: vector compare.
*/
  return (a->x == b->x && a->y == b->y && a->z == b->z);
}


/*
	vec_length: computes vector magnitude.
*/
float vec_length (c_VECTOR *a)
{
	float len;

	len = (float)sqrt (a->x * a->x + a->y * a->y + a->z * a->z);

	if (len == 0.0f) 
		len = 1.0f;

	return (len);
}



float vec_distance (c_VECTOR *a, c_VECTOR *b)
{
/*
  vec_distance: computes distance between two vectors.
*/
  c_VECTOR v;

  vec_sub (a, b, &v);
  return (vec_length (&v));
}


/*
  vec_dot: computes dot product of two vectors.
*/
float vec_dot (c_VECTOR *a, c_VECTOR *b)
{
	float mag;

	mag = 1.0f / (vec_length(a) * vec_length(b));
	return (a->x*b->x + a->y*b->y + a->z*b->z) * mag;
}



float vec_dotunit (c_VECTOR *a, c_VECTOR *b)
{
/*
  vec_dotunit: computes dot product of two vectors.
*/
  return (a->x*b->x + a->y*b->y + a->z*b->z);
}


/*
  vec_cross: computes cross product of two vectors.
*/
void vec_cross (c_VECTOR *a, c_VECTOR *b, c_VECTOR *out)
{
	out->x = a->y * b->z - a->z * b->y;
	out->y = a->z * b->x - a->x * b->z;
	out->z = a->x * b->y - a->y * b->x;
}


/*
  vec_midpoint: computes middle point of two vectors.
*/
void vec_midpoint (c_VECTOR *a, c_VECTOR *b, c_VECTOR *out)
{
	c_VECTOR temp;

	vec_add (a, b, &temp);
	vec_scale (&temp, 0.5, &temp);
	vec_copy (&temp, out);
}


/*
  vec_lerp: linear interpolation of two vectors.
*/
void vec_lerp (c_VECTOR *a, c_VECTOR *b, float alpha, c_VECTOR *out)
{
	c_VECTOR temp;

#define _LERP(a,l,h) ( (l) + (( (h)-(l) ) * (a) ) )
	temp.x = _LERP (alpha, a->x, b->x);
	temp.y = _LERP (alpha, a->y, b->y);
	temp.z = _LERP (alpha, a->z, b->z);
#undef _LERP

	vec_copy (&temp, out);
}



void vec_combine (c_VECTOR *a, c_VECTOR *b, float as, float bs,
                  c_VECTOR *out)
{
/*
  vec_combine: compute linear combination of two vectors.
*/
  c_VECTOR temp;

  temp.x = (as * a->x) + (bs * b->x);
  temp.y = (as * a->y) + (bs * b->y);
  temp.z = (as * a->z) + (bs * b->z);
  vec_copy (&temp, out);
}


/*
  vec_normalize: vector normalization.
*/
void vec_normalize (c_VECTOR *a, c_VECTOR *out)
{
	float    len;
	//c_VECTOR temp;

	len = vec_length (a);

	if (len == 0.0f) {
		out->x = 0.0f;
		out->y = 0.0f;
		out->z = 0.0f;
		return;
	}

	len = 1.0f / len;
	out->x = a->x * len;
	out->y = a->y * len;
	out->z = a->z * len;
}
