#include "dxc.h"

CDX_Screen::CDX_Screen()
{
	Initialise();
}

CDX_Screen::~CDX_Screen()
{
	Finalise();
}

void CDX_Screen::Initialise(void)
{
	m_DirectDraw = NULL;
	m_FrontBuffer = NULL;
	m_BackBuffer = NULL;
	m_Width = 0;
	m_Height = 0;
	m_Bpp = 0;
	m_Hwnd = NULL;
	m_IsFullScreen = TRUE;
	
}

//////////////////////////////////////////////////////////////////////////////////
// CDX_Screen Finalise
//////////////////////////////////////////////////////////////////////////////////
void CDX_Screen::Finalise(void)
{
	
	SAFE_DELETE(m_BackBuffer);
	SAFE_DELETE(m_FrontBuffer);
	SAFE_RELEASE(m_DirectDraw);
}


//////////////////////////////////////////////////////////////////////////////////
// CDX_Screen CreateDirectDraw
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDX_Screen::CreateDirectDraw(void)
{
	LPDIRECTDRAW directdraw = NULL;
	HRESULT rval;

	
	if(directdraw == NULL)
	{
		// If no hardware found then use a software driver
		rval = DirectDrawCreate(NULL, &directdraw, NULL);
		if(rval != DD_OK) return rval;
	}

	// Get the latest interface
	rval = directdraw->QueryInterface(IID_IDirectDraw4, (LPVOID*)&m_DirectDraw);
	if(rval != DD_OK) return rval;

	SAFE_RELEASE(directdraw);

	// Set the cooperative level flags
	DWORD Mode = DDSCL_NORMAL;

	if(m_IsFullScreen)
		Mode = DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWMODEX;

	// Set the cooperative level
	rval = m_DirectDraw->SetCooperativeLevel(m_Hwnd, Mode);
	if(rval != DD_OK) return rval;

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDX_Screen CreateBuffers
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDX_Screen::CreateBuffers(void)
{
	HRESULT rval;

	m_FrontBuffer = new LPDIRECTDRAWSURFACE;
	m_BackBuffer = new LPDIRECTDRAWSURFACE;

	if(m_IsFullScreen)
	{
		// Set the screen and viewport dimensions
		SetRect(&m_ScreenRect, 0, 0, m_Width, m_Height);
		
		DWORD Mode = 0;

		// Check for Mode 13h
		if((m_Width == 320) && (m_Height == 200) && (m_Bpp == 8))
			Mode = DDSDM_STANDARDVGAMODE;

		// Set the dispay mode
		rval = m_DirectDraw->SetDisplayMode(m_Width, m_Height, m_Bpp, 0, Mode);
		if(rval != DD_OK) return rval;

		// Set the front buffer flags for fullscreen mode
		m_FrontBuffer->m_Desc.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
		m_FrontBuffer->m_Desc.dwBackBufferCount = 1;
		m_FrontBuffer->m_Desc.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
		                                       DDSCAPS_FLIP |
		                                       DDSCAPS_COMPLEX;
		
	}
	else
	{
		// Set the screen and viewport dimensions
		
		GetClientRect(m_Hwnd, &m_ScreenRect);
		ClientToScreen(m_Hwnd, (LPPOINT)&m_ScreenRect.left);
		ClientToScreen(m_Hwnd, (LPPOINT)&m_ScreenRect.right);
		m_Width  = m_ScreenRect.right - m_ScreenRect.left;
		m_Height = m_ScreenRect.bottom - m_ScreenRect.top;

		// Set the front buffer flags for windowed mode
		m_FrontBuffer->m_Desc.dwFlags = DDSD_CAPS;
		m_FrontBuffer->m_Desc.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
	}

	// Create the front buffer
	rval = m_FrontBuffer->Create(this);
	if(rval != DD_OK) return rval;

	if(m_IsFullScreen)
	{
		m_FrontBuffer->m_Caps.dwCaps = DDSCAPS_BACKBUFFER;
		m_FrontBuffer->m_Surface->GetAttachedSurface(
			&m_FrontBuffer->m_Caps,
			&m_BackBuffer->m_Surface);
	}
	else
	{
		m_BackBuffer->m_Desc.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
		m_BackBuffer->m_Desc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;

		if(m_Is3D) m_BackBuffer->m_Desc.ddsCaps.dwCaps |= DDSCAPS_3DDEVICE;

		m_BackBuffer->m_Desc.dwWidth = m_Width;
		m_BackBuffer->m_Desc.dwHeight = m_Height;

		// Create the back buffer
		rval = m_BackBuffer->Create(this);
		if(rval != DD_OK) return rval;

		LPDIRECTDRAWCLIPPER clipper;

		// Create the window clipper
		rval = m_DirectDraw->CreateClipper(0, &clipper, NULL);
		if(rval != DD_OK) return rval;

		rval = clipper->SetHWnd(0, m_Hwnd);
		if(rval != DD_OK) return rval;

		rval = m_FrontBuffer->m_Surface->SetClipper(clipper);
		if(rval != DD_OK) return rval;;
	
		SAFE_RELEASE(clipper);
	}

	return rval;
}


//////////////////////////////////////////////////////////////////////////////////
// CDX_Screen Create
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDX_Screen::Create(HWND hwnd, DWORD width, DWORD height, DWORD bpp, DWORD flags)
{
	HRESULT rval;

	Finalise();
	Initialise();

	// Set some internal variables
	m_Width = width;
	m_Height = height;
	m_Bpp = bpp;
	m_Hwnd = hwnd;

	// What flags were requested?
	if(flags & CDXCREATE_FULLSCREEN) m_IsFullScreen = TRUE;
	
	// Create the DirectDraw instance
	rval = CreateDirectDraw();
	if(rval != DD_OK) goto CREATE_FAILED;

	
	// Create the front and back buffers
	rval = CreateBuffers();
	if(rval != DD_OK) goto CREATE_FAILED;

	goto CREATE_SUCCESS;

CREATE_FAILED:

	Finalise();

CREATE_SUCCESS:

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDX_Screen Flip - swaps the front and back buffers
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDX_Screen::Flip(void)
{
	HRESULT rval;

	if(m_IsFullScreen)
	{
		rval = m_FrontBuffer->m_Surface->Flip(NULL, DDFLIP_WAIT);
	}
	else
	{
		GetClientRect(m_Hwnd, &m_ScreenRect);
		ClientToScreen(m_Hwnd, (LPPOINT)&m_ScreenRect.left);
		ClientToScreen(m_Hwnd, (LPPOINT)&m_ScreenRect.right);

		rval = m_FrontBuffer->m_Surface->Blt(
			&m_ScreenRect,
			m_BackBuffer->m_Surface,
			NULL,
			DDFLIP_WAIT,
			NULL);
	}
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDX_Screen Restore - restores the front buffer if it is lost
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDX_Screen::Restore(void)
{
	HRESULT rval = DD_OK;

  if(m_FrontBuffer)
		if(m_FrontBuffer->m_Surface->IsLost())
			rval = m_FrontBuffer->m_Surface->Restore();

  if(m_BackBuffer)
		if(m_BackBuffer->m_Surface->IsLost())
			rval = m_BackBuffer->m_Surface->Restore();

  
	return rval;
}
