unit DGCCache;

interface
uses sysutils, Forms, Classes, ExtCtrls, DGC;

const
  MaxCache = 255;

type
  ESurfaceFileNotFound = class(Exception);
  ECacheGenFailure = class(Exception);

type
  TDGCCachedSurfaces= class(tComponent)
  private
    fSurfaceList: tStringList;
    fScreen: tDGCScreen;
    fBaseDirectory: string;
    fOnPurge: tNotifyEvent;
    fOldAppIdleHolder: tIdleEvent;
    function GetCount: Integer;
    function InsertSurface(Surface: tDGCSurface; Name: String):Integer;
    function GetSurface(SurfaceName: String): tDGCSurface;
    function LoadSurfaceFromFile(SurfaceName: string): Integer;
    procedure PurgeOldestSurface;
    procedure PurgeWhenIdle(Sender: TObject; var Done: Boolean);
  public
    Constructor Create(AOwner: TComponent); Override;
    Destructor  Destroy; Override;
    procedure InsertNewSurface(var Surface: TDGCSurface;
                               Width,Height: Integer;Name: String);
    property Surface[SurfaceName: String]: TDGCSurface read GetSurface;                           
  published
    property BaseDirectory: String read fBaseDirectory write fBaseDirectory;
    property ImageCount: Integer read GetCount;

    property OnPurge: tNotifyEvent read fOnPurge write fOnPurge;
  end;


implementation

function TDGCCachedSurfaces.GetCount: Integer;
begin
  if fSurfaceList = nil then
    result:= -1
  else
    result:= fSurfaceList.Count;
end;

Constructor TDGCCachedSurfaces.Create(AOwner: TComponent);
var
  I: Integer;
begin
  inherited Create(AOwner);
  //find the screen object in the application
  //dont need to look for a screen when designing
  fScreen:= nil;
  if not(csDesigning in ComponentState) then
    begin
      for I:= 0 to Owner.ComponentCount - 1 do
        if Owner.Components[I] is tDGCScreen then
          begin
            fScreen:= Owner.Components[I] as tDGCScreen;
            Break;
          end;
      if fScreen = Nil then
       raise ECacheGenFailure.Create('Cache Error: Screen not Initialized');
    end;

  //now set the rest
  fSurfaceList:= tStringList.Create;
  fBaseDirectory:= '';
  fOldAppIdleHolder:= Nil;
  with fSurfaceList do
    begin
      Sorted:= True;
      Duplicates:= dupIgnore;
    end;
end;

destructor TDGCCachedSurfaces.Destroy;
var I: integer;
begin
  //if needed, set the OnIdle event back to prevent hanging the app
  if Assigned(fOldAppIdleHolder) then Application.OnIdle:= fOldAppIdleHolder;
  if fSurfaceList <> NIL then //clear out all of the surfaces/images.
    for I:= 0 to fSurfaceList.Count-1 do
      if fSurfaceList.Objects[I] <> nil then fSurfaceList.Objects[I].Free;
  fSurfaceList.Free;
  inherited Destroy;
end;

procedure TDGCCachedSurfaces.InsertNewSurface
          (var Surface: TDGCSurface; Width,Height: Integer; Name: String);
begin
  fScreen.CreateSurface(Surface,Width,Height);
  InsertSurface(Surface, Name);
end;

function TDGCCachedSurfaces.InsertSurface(Surface: tDGCSurface;
                                          Name: String):Integer;
Begin
  //if fSurfaceList.Count >= MaxCache then PurgeOldestSurface;
  if fSurfaceList.Count >= MaxCache then
    begin
      fOldAppIdleHolder:= Application.OnIdle;
      Application.OnIdle:= PurgeWhenIdle;
    end;
  result:= fSurfaceList.AddObject(Name,Surface);
end;

function TDGCCachedSurfaces.GetSurface(SurfaceName: String): tDGCSurface;
var Index: integer;
begin
  if not(fSurfaceList.Find(SurfaceName, Index)) then
    begin
      index:= LoadSurfaceFromFile(SurfaceName);
      if Index = -1 then
        raise ESurfaceFileNotFound.Create(
          'Failed to load ' + SurfaceName +' Not Found')
    end;
  result:= (fSurfaceList.Objects[Index] as tDGCSurface);
end;

procedure TDGCCachedSurfaces.PurgeOldestSurface;
var
  I: integer;
  LImg: integer;
  LDate: double;
begin
  if Assigned(fOnPurge) then fOnPurge(Self);
  LImg:= -1;
  LDate:= double(Time);
  for I:= 0 to fSurfaceList.Count-1 do
    if (fSurfaceList.Objects[I] as tDGCSurface).CacheLevel < LDate then
      begin
        LImg:= I;
        LDate:= (fSurfaceList.Objects[I] as tDGCSurface).CacheLevel;
      end;
  (fSurfaceList.Objects[LImg] as tDGCSurface).Free;
  fSurfaceList.Delete(LImg);
end;


function TDGCCachedSurfaces.LoadSurfaceFromFile(SurfaceName: string): Integer;
var
  IMG: tImage;   //used for hirez graphics from file
  Surface: tDGCSurface;
begin
  try
    try
      IMG:= TImage.Create(nil);
      IMG.Picture.LoadFromFile(fBaseDirectory + SurfaceName);
      //create a new image/surface
      fScreen.CreateSurface(Surface,IMG.Picture.Width,IMG.Picture.Height);
      result:= InsertSurface(Surface, SurfaceName);
      //push the image to the new surface
      With Surface.Canvas do
        begin
          Draw(0,0,IMG.Picture.Graphic);
          Release;
        end;
    except
      on EInOutError do result:= -1;
    end;
  finally
    if IMG <> NIL then IMG.Free;
  end;
end;

procedure TDGCCachedSurfaces.PurgeWhenIdle(Sender: TObject; var Done: Boolean);
begin
  try
    PurgeOldestSurface;
  finally
    Done:= true;
    Application.OnIdle:= fOldAppIdleHolder;
    fOldAppIdleHolder:= nil;
  end;
end;

end.
