{
 The Delpi Games Creator - Beta 6
 --------------------------------
 Copyright 1996,1997 John Pullen, Paul Bearne, Jeff Kurtz

 This unit is part of the freeware Delphi Games Creator. This unit is
 completely free to use for personal or commercial use. The code is
 supplied with no guarantees on performance or stabilibty and must be
 used at your own risk.
}

unit DGCMap;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DsgnIntf;

const
  MapFileVersion = 'ML01';

type
  TDGCMapLibProperty = class(TPropertyEditor)
  public
    procedure Edit; override;
    function GetAttributes : TPropertyAttributes; override;
    function GetValue : string; override;
  end;

  TDGCMapRec = record
    Name  : String[15];
    XSize : Integer;
    YSize : Integer;
    MSize : Integer;
    Data  : PByteArray;
  end;

  TDGCMap = array[0..0] of TDGCMapRec;
  PDGCMap = ^TDGCMap;

  TDGCMaps = class(TPersistent)
  private
    FMaps     : PDGCMap;
    FMapCount : Integer;
  protected
    procedure DefineProperties(Filer: TFiler); override;
    procedure AssignTo(Dest: TPersistent); override;

    procedure LoadFromStream(Stream : TStream);
    procedure SaveToStream(Stream : TStream);

  public
    constructor Create;
    destructor Destroy; override;

    property Maps     : PDGCMap Read FMaps     Write FMaps;
    property MapCount : Integer Read FMapCount Write FMapCount;
    procedure FreeMaps;
    procedure LoadFromFile(Filename : String);
    procedure SaveToFile(Filename : String);
  end;

  TDGCMapLib = class(TComponent)
  private
    FMapLib : TDGCMaps;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property MapLib : TDGCMaps Read FMapLib Write FMapLib;
  end;
procedure Register;

implementation

uses DGCMLO;

constructor TDGCMaps.Create;
begin
   inherited Create;
   FMaps := nil;
   FMapCount := -1;
end;

destructor TDGCMaps.Destroy;
begin
   FreeMaps;
   inherited Destroy;
end;

procedure TDGCMaps.LoadFromStream(Stream : TStream);
var
   Loop : Integer;
   Tag  : array[0..3] of char;
   mapsinfile: integer;
begin
   FreeMaps;
   Stream.ReadBuffer(Tag,4);

   Stream.ReadBuffer(MapsInFile,SizeOf(MapsInFile));
   If MapsInFile >= 0 then
   begin
      FMapCount := MapsInFile;
      ReAllocMem(FMaps, (FMapCount +1) * SizeOf(TDGCMapRec));
      for loop := 0 to MapsInFile do
      begin
        FMaps[Loop].Data := Nil;
        Stream.ReadBuffer(FMaps[loop], SizeOf(FMaps[Loop]));
        GetMem(FMaps[Loop].Data, FMaps[Loop].MSize);
        Stream.ReadBuffer(FMaps[Loop].Data[0], FMaps[Loop].MSize);
      end;
   end;
end;

procedure TDGCMaps.LoadFromFile(Filename : String);
var
   Stream : TFileStream;
begin
   Stream := TFileStream.Create(FileName, fmOpenRead);
   LoadFromStream(Stream);
   Stream.Destroy;
end;

procedure TDGCMaps.SaveToStream(Stream : TStream);
var
   loop : integer;
begin
   Stream.WriteBuffer(MapFileVersion,Length(MapFileVersion));
   Stream.WriteBuffer(FMapCount,SizeOf(FMapCount));

   for loop := 0 to FMapCount do
   begin
      Stream.WriteBuffer(FMaps[Loop], SizeOf(FMaps[Loop]));
      Stream.WriteBuffer(FMaps[Loop].Data[0], FMaps[Loop].MSize);
   end;
end;

procedure TDGCMaps.SaveToFile(FileName: String);
var
   Stream: TFileStream;
begin
   Stream  := TFileStream.Create(FileName, fmCreate);
   try
      SaveToStream(Stream);
   finally
      Stream.Destroy;
   end;
end;

procedure TDGCMaps.FreeMaps;
var
   x : integer;
begin
  If FMapCount >= 0 then
  begin
    For x := 0 to FMapCount do
       FreeMem(FMaps[x].Data, FMaps[x].MSize);
    ReAllocMem(FMaps,0);
    FMapCount := -1;
    FMaps := Nil;
  end;
end;

procedure TDGCMaps.DefineProperties(Filer: TFiler);
   function HasMaps: Boolean;
   begin
      if FMaps <> nil then
         Result := True
      else
          Result := False;
   end;
begin
   inherited DefineProperties(Filer);
   Filer.DefineBinaryProperty('MapLib', LoadFromStream, SaveToStream, HasMaps);
end;

procedure TDGCMaps.AssignTo(Dest: TPersistent);
var
   m: TDGCMaps;
   n: Integer;
begin
   //Copy Header
   if Dest is TDGCMaps then
   begin
      //Remove any maps already loaded
      m := TDGCMaps(Dest);
      m.FreeMaps;
      if MapCount = -1 then exit;
      //Allocate Memory for Maps
      m.MapCount := FMapCount;
      GetMem(m.FMaps, (MapCount+1) * SizeOf(TDGCMapRec));
      if m.FMaps = nil then
         raise Exception.Create('TDGCMaps.AssignTo failed (GetMem)');
      //Copy the data

      for n := 0 to m.MapCount do
      begin
         m.Maps[n].Name  := FMaps[n].Name;
         m.Maps[n].XSize := FMaps[n].XSize;
         m.Maps[n].YSize := FMaps[n].YSize;
         m.Maps[n].MSize := FMaps[n].MSize;
         m.Maps[n].Data  := Nil;
         GetMem(m.Maps[n].Data, m.Maps[n].MSize);
         CopyMemory(m.Maps[n].Data,FMaps[n].Data, m.Maps[n].MSize);
      end;
      exit;
   end;
   inherited AssignTo(Dest);
end;

constructor TDGCMapLib.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FMapLib := TDGCMaps.Create;
end;

destructor TDGCMapLib.Destroy;
begin
   FMapLib.Free;
   inherited Destroy;
end;

function TDGCMapLibProperty.GetAttributes : TPropertyAttributes;
begin
  Result := [ paDialog ];
end;

function TDGCMapLibProperty.GetValue : string;
begin
  Result := '(TDGCMaps)';
end;

procedure TDGCMapLibProperty.Edit;
var
   m : TDGCMaps;
begin
   frmOpenMap := TfrmOpenMap.Create(Application);
   m := TDGCMaps(GetOrdValue);
   frmOpenMap.MapLib.Assign(m);

   if frmOpenMap.ShowModal = mrOk then
   begin
      m.assign(frmOpenMap.MapLib);
      SetOrdValue(LongInt(m));
   end;
   frmOpenMap.Free;
end;

procedure Register;
begin
  RegisterComponents('DGC', [TDGCMapLib]);
end;

end.
