{The Delpi Games Creator - Beta 6
 --------------------------------
 Copyright 1996 John Pullen, Paul Bearne, Jeff Kurtz

 This unit is part of the freeware Delphi Games Creator. This unit is
 completely free to use for personal or commercial use. The code is
 supplied with no guarantees on performance or stabilibty and must be
 used at your own risk.
}

{Changes:
10-Jul-97 Jeff : Changed .Free on streams to .Destroy to release the file
09-Jul-97 Jeff : DrawStretch method added
10-Jun-97 Jeff : DrawTrans method added
06-Jun-97 Jeff : CreateAllBitmaps and FreeAllBitmaps methods added
02-May-97 John : SaveToFile method added to TDGCImageLib component
}

unit DGCILib;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DsgnIntf, BmpUtil, ComCtrls;

const
     ImageLibID = 'IL10';

type
  TBMP = array[0..0] of TBitmap;
  PBMP = ^TBMP;

  //Image Library Storage Class
  TDGCImages = class(TPersistent)
  private
    { Private declarations }
    function GetImageData(Index: Integer): TImgLibImage;
    function GetCount: Integer;
  protected
    { Protected declarations }
    procedure DefineProperties(Filer: TFiler); override;
    procedure AssignTo(Dest: TPersistent); override;
  public
    { Public declarations }
    ImgHeader: TImgLibHeader;
    FImages: PImgLibImageList;
    procedure FreeImages;
    procedure LoadFromStream(Stream: TStream);
    procedure SaveToStream(Stream: TStream);
    procedure SaveToFile(FileName: String);
    property ImageData[Index: Integer]: TImgLibImage read GetImageData;
    property ImageList: PImgLibImageList read FImages;
    property Count: Integer read GetCount;
    property Header: TImgLibHeader read ImgHeader;
    constructor Create;
    destructor Destroy; override;
    procedure LoadFromFile(FileName: String);
  end;

  //Property Editor Class
  TDGCOpenLibProperty = class(TPropertyEditor)
  public
    procedure Edit; override;
    function GetAttributes : TPropertyAttributes; override;
    function GetValue : string; override;
  end;

  //Main Class
  TDGCImageLib = class(TComponent)
  private
    { Private declarations }
    FImages: TDGCImages;
    FFileName: String;
    FBmps : PBMP;
    FAllBitmaps : Boolean;
    FProgressbar : TProgressBar;
    function GetImageCount: Integer;
    procedure SetImages(NewVal: TDGCImages);
    function GetIsFileLibrary: Boolean;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DrawImage(Canvas: TCanvas; X, Y: Integer; Idx: Integer);
    procedure DrawTrans(Canvas: TCanvas; X, Y, Idx, Color: Integer);
    procedure DrawStretch(Canvas: TCanvas; X, Y, Idx, XSize, YSize: Integer);

    procedure CreateAllBitmaps;
    procedure FreeAllBitmaps;
    procedure LoadImageList(IList : TImageList);

    property ImageCount: Integer read GetImageCount;
    procedure LoadFromFile(FileName: String);
    procedure ReLoadFromFile;
    property IsFileLibrary: Boolean read GetIsFileLibrary;
    property ProgressBar : TProgressBar Read FProgressBar Write FProgressBar;
  published
    { Published declarations }
    property Images: TDGCImages read FImages write SetImages;
  end;

//procedure Register;

implementation

uses dgcilo, DGC;

//TDGCImageLib Implementation
//===========================
constructor TDGCImageLib.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FFileName := '';
   FImages := TDGCImages.Create;
end;

destructor TDGCImageLib.Destroy;
begin
//   If FAllBitmaps then
     FreeAllBitmaps;
//   else
     FImages.Free;
   inherited Destroy;
end;

procedure TDGCImageLib.LoadImageList(IList : TImageList);
var
   BmpInfo: PBitmapInfo;
   HeaderSize: Integer;
   ImgHdr: TImgLibImage;
   Palette: TImgLibPalette;
   Bmp: TBitmap;
   loop : integer;

begin
   IList.Clear;
   IList.Width  := Images.ImageData[0].Width;
   IList.Height := Images.ImageData[0].Height;

   if assigned(FProgressBar) then
   begin
      FProgressBar.Min := 0;
      FProgressBar.Max := ImageCount -1;
   end;

   for loop := 0 to ImageCount -1 do
   begin
     If assigned(FProgressBar) then FProgressBar.Position := loop;
     Bmp := TBitmap.Create;
     ImgHdr := Images.ImageData[loop];
     HeaderSize := SizeOf(TBitmapInfo) + (256 * SizeOf(TRGBQuad));
     BmpInfo := AllocMem(HeaderSize);
     //First Get Colours
     Palette := Images.Header.Palette;
     ImgLibPalToBmpInfo(Palette, BmpInfo);
     with BmpInfo^.bmiHeader do
     begin
        biSize := SizeOf(TBitmapInfoHeader);
        biWidth := ImgHdr.Width;
        biHeight := -ImgHdr.Height;
        biPlanes := 1;
        biBitCount := 8; //always convert to 8 bit image
        biCompression := BI_RGB;
        biClrUsed := 0;
        biClrImportant := 0;
     end;
     CreateDIB256(Bmp, BmpInfo, ImgHdr.Bits);
     IList.AddMasked(Bmp,clBlack);
     //CleanUp
     FreeMem(BmpInfo, HeaderSize);
     Bmp.Destroy;
   end;
end;

procedure TDGCImageLib.CreateAllBitmaps;
var
   BmpInfo: PBitmapInfo;
   HeaderSize: Integer;
   ImgHdr: TImgLibImage;
   Palette: TImgLibPalette;
//   Bmp: TBitmap;
   idx : Integer;
begin
   If FAllBitmaps then exit;
   GetMem(FBmps, ImageCount * SizeOf(TBitmap));
   HeaderSize := SizeOf(TBitmapInfo) + (256 * SizeOf(TRGBQuad));
   BmpInfo := AllocMem(HeaderSize);
   Palette := Images.Header.Palette;
   ImgLibPalToBmpInfo(Palette, BmpInfo);

   if assigned(FProgressBar) then
   begin
      FProgressBar.Min := 0;
      FProgressBar.Max := ImageCount -1;
   end;

   for idx := 0 to ImageCount -1 do
   begin
      If assigned(FProgressBar) then FProgressBar.Position := idx;
      FBmps[Idx] := TBitmap.Create;
      ImgHdr := Images.ImageData[idx];
      //First Get Colours
      with BmpInfo^.bmiHeader do
      begin
         biSize := SizeOf(TBitmapInfoHeader);
         biWidth := ImgHdr.Width;
         biHeight := -ImgHdr.Height;
         biPlanes := 1;
         biBitCount := 8; //always convert to 8 bit image
         biCompression := BI_RGB;
         biClrUsed := 0;
         biClrImportant := 0;
      end;
      CreateDIB256(FBmps[Idx], BmpInfo, ImgHdr.Bits);
      //CleanUp
   end;
   FreeMem(BmpInfo, HeaderSize);
   FAllBitmaps := True;
//   FImages.Free;
end;

procedure TDGCImageLib.FreeAllBitmaps;
var
  idx : integer;
begin
   If not FAllBitmaps then exit;
   for idx := 0 to ImageCount -1 do
   begin
     FBmps[Idx].Free;
   end;
   FreeMem(FBmps, ImageCount * SizeOf(TBitmap));
   FAllBitmaps := False;
end;

procedure TDGCImageLib.SetImages(NewVal: TDGCImages);
begin
   FImages.Assign(NewVal);
end;

function TDGCImageLib.GetImageCount: Integer;
begin
   if Assigned(FImages) then
      Result := FImages.Header.ImageCount
   else
      Result := 0;
end;

procedure TDGCImageLib.LoadFromFile(FileName: String);
begin
   FFileName := '';
   If FAllBitmaps then FreeAllBitmaps;
   FImages.LoadFromFile(FileName);
   FFileName := FileName;
   If FAllBitMaps then CreateAllBitmaps;
end;

function TDGCImageLib.GetIsFileLibrary: Boolean;
begin
   Result := FFileName <> '';
end;

procedure TDGCImageLib.ReLoadFromFile;
begin
   if FFileName = '' then
      raise Exception.Create('LoadFromFile must be called first');
   LoadFromFile(FFileName);
end;

procedure TDGCImageLib.DrawImage(Canvas: TCanvas; X, Y: Integer; Idx: Integer);
var
   BmpInfo: PBitmapInfo;
   HeaderSize: Integer;
   ImgHdr: TImgLibImage;
   Palette: TImgLibPalette;
   Bmp: TBitmap;
begin
   If Idx > ImageCount -1 then exit;
   If FAllBitmaps then
   begin
      Canvas.Draw(x, y, FBmps[Idx]);
      exit;
   end;

   Bmp := TBitmap.Create;
   ImgHdr := Images.ImageData[idx];
   HeaderSize := SizeOf(TBitmapInfo) + (256 * SizeOf(TRGBQuad));
   BmpInfo := AllocMem(HeaderSize);
   //First Get Colours
   Palette := Images.Header.Palette;
   ImgLibPalToBmpInfo(Palette, BmpInfo);
   with BmpInfo^.bmiHeader do
   begin
      biSize := SizeOf(TBitmapInfoHeader);
      biWidth := ImgHdr.Width;
      biHeight := -ImgHdr.Height;
      biPlanes := 1;
      biBitCount := 8; //always convert to 8 bit image
      biCompression := BI_RGB;
      biClrUsed := 0;
      biClrImportant := 0;
   end;
   CreateDIB256(Bmp, BmpInfo, ImgHdr.Bits);

   //CleanUp
   FreeMem(BmpInfo, HeaderSize);
   Canvas.Draw(x, y, Bmp);
   Bmp.Destroy;
end;

procedure TDGCImageLib.DrawTrans(Canvas: TCanvas; X, Y, Idx, Color: Integer);
var
   BmpInfo: PBitmapInfo;
   HeaderSize: Integer;
   ImgHdr: TImgLibImage;
   Palette: TImgLibPalette;
   Bmp: TBitmap;
begin
   If Idx > ImageCount -1 then exit;
   If FAllBitmaps then
   begin
      Canvas.BrushCopy(Rect(X,Y,X+FBmps[Idx].Width,Y+FBmps[Idx].Height),FBmps[Idx],
                       Rect(0,0,FBmps[Idx].Width,FBmps[Idx].Height),color);
      exit;
   end;

   Bmp := TBitmap.Create;
   ImgHdr := Images.ImageData[idx];
   HeaderSize := SizeOf(TBitmapInfo) + (256 * SizeOf(TRGBQuad));
   BmpInfo := AllocMem(HeaderSize);
   //First Get Colours
   Palette := Images.Header.Palette;
   ImgLibPalToBmpInfo(Palette, BmpInfo);
   with BmpInfo^.bmiHeader do
   begin
      biSize := SizeOf(TBitmapInfoHeader);
      biWidth := ImgHdr.Width;
      biHeight := -ImgHdr.Height;
      biPlanes := 1;
      biBitCount := 8; //always convert to 8 bit image
      biCompression := BI_RGB;
      biClrUsed := 0;
      biClrImportant := 0;
   end;
   CreateDIB256(Bmp, BmpInfo, ImgHdr.Bits);

   //CleanUp
   FreeMem(BmpInfo, HeaderSize);
   Canvas.BrushCopy(Rect(X,Y,X+Bmp.Width,Y+Bmp.Height),Bmp,
                    Rect(0,0,Bmp.Width,Bmp.Height),color);
   Bmp.Destroy;
end;

procedure TDGCImageLib.DrawStretch(Canvas: TCanvas; X, Y, Idx, XSize, YSize: Integer);
var
   BmpInfo: PBitmapInfo;
   HeaderSize: Integer;
   ImgHdr: TImgLibImage;
   Palette: TImgLibPalette;
   Bmp: TBitmap;
   destrect : TRect;
begin
   If Idx > ImageCount -1 then exit;
   If FAllBitmaps then
   begin
      DestRect := Rect(X,Y,X+XSize,Y+YSize);
      Canvas.StretchDraw(DestRect,FBmps[Idx]);
      exit;
   end;

   Bmp := TBitmap.Create;
   ImgHdr := Images.ImageData[idx];
   HeaderSize := SizeOf(TBitmapInfo) + (256 * SizeOf(TRGBQuad));
   BmpInfo := AllocMem(HeaderSize);
   //First Get Colours
   Palette := Images.Header.Palette;
   ImgLibPalToBmpInfo(Palette, BmpInfo);
   with BmpInfo^.bmiHeader do
   begin
      biSize := SizeOf(TBitmapInfoHeader);
      biWidth := ImgHdr.Width;
      biHeight := -ImgHdr.Height;
      biPlanes := 1;
      biBitCount := 8; //always convert to 8 bit image
      biCompression := BI_RGB;
      biClrUsed := 0;
      biClrImportant := 0;
   end;
   CreateDIB256(Bmp, BmpInfo, ImgHdr.Bits);

   //CleanUp
   FreeMem(BmpInfo, HeaderSize);
   DestRect := Rect(X,Y,X+XSize,Y+YSize);
   Canvas.StretchDraw(DestRect,FBmps[Idx]);
   Bmp.Destroy;
end;

//TDGCImages Implementation
//==========================
constructor TDGCImages.Create;
begin
   inherited Create;
   ZeroMemory(@ImgHeader, SizeOf(ImgHeader));
   FImages := nil;
end;

destructor TDGCImages.Destroy;
begin
   FreeImages;
   inherited Destroy;
end;


procedure TDGCImages.LoadFromStream(Stream: TStream);
var
   n: Integer;
   ImgHdr: TImgLibImage;
begin
   //Read Header
   FreeImages; //Remove current image list if there is one
   Stream.ReadBuffer(ImgHeader, SizeOf(ImgHeader));
   if StrLIComp(PChar(ImageLibID), ImgHeader.Ident, 4) <> 0 then
      raise Exception.Create('Not a valid Image Library File');
   GetMem(FImages, ImgHeader.ImageCount * SizeOF(TImgLibImage));
   if FImages = nil then
      raise Exception.Create('TDGCImage: GemMem Failed for FImages');
   for n := 0 to ImgHeader.ImageCount - 1 do
   begin
      ImgHdr := FImages^[n];
      //Read Image Header
      Stream.ReadBuffer(ImgHdr, SizeOf(TImgLibImage) - SizeOf(PByte));
      //Alocate memory for bits and read bits
      GetMem(ImgHdr.Bits, WidthBytes(ImgHdr.Width) * ImgHdr.Height);
      if ImgHdr.Bits = nil then
         raise Exception.Create('TDGCImage: GemMem Failed for ImgHdr.Bits');
      Stream.ReadBuffer(ImgHdr.Bits^, WidthBytes(ImgHdr.Width) * ImgHdr.Height);
      FImages^[n] := ImgHdr;
  end;
end;

procedure TDGCImages.SaveToStream(Stream: TStream);
var
   n: Integer;
   ImgHdr: TImgLibImage;
begin
   Stream.WriteBuffer(ImgHeader, SizeOf(ImgHeader));
   for n := 0 to ImgHeader.ImageCount - 1 do
   begin
      ImgHdr := FImages^[n];
      //write Image Header
      Stream.WriteBuffer(ImgHdr, SizeOf(TImgLibImage) - SizeOf(PByte));
      //Write image bits
      Stream.WriteBuffer(ImgHdr.Bits^, WidthBytes(ImgHdr.Width) * ImgHdr.Height);
   end;
end;

procedure TDGCImages.DefineProperties(Filer: TFiler);
   function HasImages: Boolean;
   begin
      if FImages <> nil then
         Result := True
      else
          Result := False;
   end;
begin
   inherited DefineProperties(Filer);
   Filer.DefineBinaryProperty('Images', LoadFromStream, SaveToStream, HasImages);
end;

//Free the specified Image from the list box item idx
procedure TDGCImages.FreeImages;
var
   Image: TImgLibImage;
   idx: Integer;
begin
   if FImages = nil then exit;
   for idx := 0 to ImgHeader.ImageCount - 1 do
   begin
      Image := FImages^[idx];
      if Image.Bits <> nil then
         FreeMem(Image.Bits, WidthBytes(Image.Width) * Image.Height);
   end;
   FreeMem(FImages, ImgHeader.ImageCount * SizeOF(TImgLibImage));
   FImages := nil;
   ImgHeader.ImageCount := 0;
end;

procedure TDGCImages.LoadFromFile(FileName: String);
var
   Stream: TFileStream;
begin
   Stream := TFileStream.Create(FileName, fmOpenRead);
   LoadFromStream(Stream);
   Stream.Destroy;
end;

function TDGCImages.GetImageData(Index: Integer): TImgLibImage;
begin
   Result := FImages^[Index];
end;

function TDGCImages.GetCount: Integer;
begin
   Result := ImgHeader.ImageCount;
end;

procedure TDGCImages.AssignTo(Dest: TPersistent);
var
   I: TDGCImages;
   ImageSize: Integer;
   n: Integer;
   ImgHdr: TImgLibImage;
   Ilist: PImgLibImageList;
begin
   //Copy Header
   if Dest is TDGCImages then
   begin
      //Remove any images already loaded
      I := TDGCImages(Dest);
      I.FreeImages;
      if ImgHeader.ImageCount = 0 then
         exit;
      //Move header
      I.ImgHeader := ImgHeader;
      //Allocate Memory for ImageList
      ImageSize := I.ImgHeader.ImageCount * SizeOF(TImgLibImage);
      GetMem(IList, ImageSize);
      if IList = nil then
         raise Exception.Create('TDGCImages.AssignTo failed (GetMem): ' + IntToStr(ImageSize));
      I.FImages := IList;
      //Copy the data
      for n := 0 to I.ImgHeader.ImageCount - 1 do
      begin
         ImgHdr := FImages^[n];
         ImageSize := WidthBytes(ImgHdr.Width) * ImgHdr.Height;
         GetMem(ImgHdr.Bits, ImageSize);
         if ImgHdr.Bits = nil then
            raise Exception.Create('TDGCImages.AssignTo failed');
         Move(FImages^[n].Bits^, ImgHdr.Bits^, ImageSize);
         I.FImages^[n] := ImgHdr;
      end;
      exit;
   end;
   inherited AssignTo(Dest);
end;

procedure TDGCImages.SaveToFile(FileName: String);
var
   f: TFileStream;
begin
   f  := TFileStream.Create(FileName, fmCreate);
   try
      SaveToStream(f);
   finally
      f.Destroy;
   end;
end;

//Propery Editor Implemantation
function TDGCOpenLibProperty.GetAttributes : TPropertyAttributes;
begin
  Result := [ paDialog ];
end;

//TDGCOpenLibProperty Implemenataion
function TDGCOpenLibProperty.GetValue : string;
begin
  Result := '(TDGCImages)';
end;

procedure TDGCOpenLibProperty.Edit;
var
   I: TDGCImages;
begin
   //This bit i'm not too sure about. The following code works
   //but other propert editors do not have to do the assign
   //they just call the GetOrdValue and SetOrdValue methods
   //???????
   frmOpenLib := TfrmOpenLib.Create(Application);
   I := TDGCImages(GetOrdValue);
   frmOpenLib.Images.Assign(I);
   if frmOpenLib.ShowModal = mrOK then
   begin
      SetOrdValue(LongInt(I));
      I.Assign(frmOpenLib.Images);
   end;
   frmOpenLib.Free;
end;

end.
