{The Delpi Games Creator - Beta 6
 --------------------------------
 Copyright 1996,1997 John Pullen, Paul Bearne, Jeff Kurtz

 This unit is part of the freeware Delphi Games Creator. This unit is
 completely free to use for personal or commercial use. The code is
 supplied with no guarantees on performance or stabilibty and must be
 used at your own risk.

NOTES:
 YOU MUST HAVE AT LEAST DIRECT X 3 (or greater) INSTALLED OR THIS COMPONENT WILL
 NOT INSTALL PROPERLY!!!

 MAKE SURE YOU HAVE DPLAYX.DLL (not DPLAY.DLL -- an 'x' on the end)
 IN YOUR WINDOW'S SYSTEM DIRECTORY.

FIXES:
 Sep-05-97: Jeff : CreateSession wasn't clearing the player lists.

 Added a 'Timeout' property for the EnumSessions. It was hard-coded at
 300 for my testing purposes but across the Internet a longer timeout
 value was needed.

 The 'Receive' portion of messaging was only grabbing one message at a
 time when a notification came across. It will now pull all messages in
 the que and send them to the OnXXXXX events. (Thanx Malachy Duffin for
 catching this one)

 Plans for the future:
   1. Clean up the code
   2. Error messages to report in the trace instead of prompts.
   3. Add Group capabilities
   4. Fix any reported bugs
   5. Convert to DirectX3 (DirectPlay2 interface)
   6. Add Lobby support
}

unit DGCPlay;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  dplay;

type
  TServerGUIDS = class(TObject)
    spGUID : PGUID;
  end;

  TDPMessage = record
    dwType    : DWORD;
    dpMessage : array[0..255] of Char;
  end;

  TDPMSGX = record
    dwType    : DWORD;
    dpMessage : array[0..255] of Char;
  end;

  pTDPMessage = ^TDPMessage;

  TDPMessageThread = class(TThread)
  protected
    procedure Execute; Override;
  public
    Constructor Create;
  end;

  //Events
  TDPGameMessage  = Procedure(From: TDPID; MsgType:Integer; GameMsg: Pointer) of Object;
  TDPAddPlayer    = Procedure(ID:TDPID; LongName, ShortName:String) of Object;
  TDPDeletePlayer = Procedure(ID:TDPID; LongName, ShortName:String) of Object;

  TDGCPlay = class(TComponent)
  private
    { Private declarations }
    FProviderNames     : TStringList;
    FNumberOfProviders : Integer;
    FServerGUID        : TGUID;
    FSessionDesc       : TDPSESSIONDESC;
    FGameGUID          : TGUID;
    FGameName          : String;
    FPassword          : String;
    FMaxPlayers        : DWORD;
    FLongName          : String;
    FShortName         : String;
    FUserData          : String;

    FLocalPlayer       : TDPID;
    FPlayerLNames      : TStringList;
    FPlayerSNames      : TStringList;

    FNumberOfSessions  : Integer;
    FSessions          : TStringList;
    FSessionID         : DWORD;

    FOnAddPlayer       : TDPAddPlayer;
    FOnDeletePlayer    : TDPDeletePlayer;
    FOnGameMessage     : TDPGameMessage;
    FDPMessage         : TDPMessage;

    FTimeOut           : Integer;
    FVersion           : String;
    NVersion           : String;

    Procedure ThreadDone(Sender: TObject);
    procedure CreateSession;
    function  GetNumberOfPlayers : DWORD;
  protected
    { Protected declarations }
    procedure   EnumServers;
    procedure   EnumSessions;
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    function    GameMessage: Pointer;

  public
    { Public declarations }
    procedure   EnumPlayers(SessionNumber : DWORD);
    Procedure   OpenProvider(ServerName : String; LocalGame:Boolean);
    Procedure   OpenSession(SessionNumber : DWORD);
    Procedure   SendMessage(ToID: TDPID; var DPMessage);
    Procedure   CloseSession;
    property    NumberOfProviders : Integer     Read FNumberOfProviders;
    Property    ProviderNames     : TStringList Read FProviderNames;
    property    NumberOfPlayers   : DWORD       Read GetNumberOfPlayers;
    property    GameGUID          : TGUID       Read FGameGUID Write FGameGUID;
    property    LocalPlayer       : TDPID        Read FLocalPlayer;
    property    PlayerLNames      : TStringList Read FPlayerLNames;
    property    PlayerSNames      : TStringList Read FPlayerSNames;
    property    NumberOfSessions  : Integer     Read FNumberOfSessions;
    property    Sessions          : TStringList Read FSessions;
    property    SessionID         : DWORD       Read FSessionID;

    property LongName:   String Read FLongName   Write FLongName;
    property ShortName:  String Read FShortName  Write FShortName;
    property UserData:   String Read FUserData   Write FUserData;

    property DPMessage: Pointer Read GameMessage;

  published
    { Published declarations }
    property GameName:   String Read FGameName   Write FGameName;
    property MaxPlayers: DWORD  Read FMaxPlayers Write FMaxPlayers;
    property Password:   String Read FPassword   Write FPassword;

    property OnAddPlayer:    TDPAddPlayer    Read FOnAddPlayer    Write FOnAddPlayer;
    property OnDeletePlayer: TDPDeletePlayer Read FOnDeletePlayer Write FOnDeletePlayer;
    property OnGameMessage:  TDPGameMessage  Read FOnGameMessage  Write FOnGameMessage;
    property TimeOut :       Integer         Read FTimeOut        Write FTimeOut;
    property Version :       String          Read FVersion        Write NVersion;
  end;

//These are the Callback functions for certain DirectPlay API Calls.
function EnumServerCallback(var lpspGUID:TGUID; lpServer:lpSTR; dwMajor:DWORD;
         dwMinor:DWORD; lpContext:Pointer):BOOL; stdcall;
function EnumSessionsCallback( var lpDPSessionDesc: TDPSESSIONDESC ;
      lpContext: Pointer ; var lpdwTimeOut: DWORD ; dwFlags: DWORD ): BOOL ;
      stdcall ;
function EnumPlayersCallback(dpid:TDPID; lpShortName:lpSTR; lpLongName:lpSTR;
         dwFlags:DWORD; lpContext:Pointer):BOOL; stdcall;

procedure ProcessMessage(var Buffer; FromID: TDPID);
procedure Register;
//Global variables used in the callback functions. These functions can not use
//Procedural defined variables ???
var
   gServerGUID      : TGUID;
   gGetGUID         : Boolean;
   gProviderNames   : TStringList;
   gLookupName      : String;

   FDirectPlay      : IDirectPlay;
   FDPMessageHandle : DWORD;
   FDPDoEvent       : Boolean;

   gSessions        : TStringList;
   gPlayerLNames    : TStringList;
   gPlayerSNames    : TStringList;
   gNumberOfPlayers : Integer;

   gOnAddPlayer     : TDPAddPlayer;
   gOnDeletePlayer  : TDPDeletePlayer;
   gOnGameMessage   : TDPGameMessage;

   gFromID          : TDPID;
   gToID            : TDPID;
   gDPMessage       : TDPMessage;

implementation

//______________________________________________________________________________
//This procedure is called when the component is called. This is a good place
//to set default values, etc..
constructor TDGCPlay.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDPDoEvent := False;
  FProviderNames := TStringList.Create;
  gProviderNames := TStringList.Create;
  FPlayerLNames  := TStringList.Create;
  FPlayerSNames  := TStringList.Create;
  gPlayerLNames  := FPlayerLNames;
  gPlayerSNames  := FPlayerSNames;
  FSessions      := TStringList.Create;
  gSessions      := TStringList.Create;
  FMaxPlayers    := 2;
  FTimeOut       := 500;
  FVersion       := 'DGC Beta 6';
  If not (csDesigning in ComponentState) then
     EnumServers;   //Get a list of the available service providers...
end;

//______________________________________________________________________________
//This is called when the program ends -- it clears up the strings lists
destructor TDGCPlay.Destroy;
begin
  inherited Destroy;
  FProviderNames.Free;
  gProviderNames.Free;
  FPlayerLNames.Free;
  FPlayerSNames.Free;
  FSessions.Free;
  gSessions.Free;

  FDPDoEvent := False;
  If FDirectPlay <> Nil Then FDirectPlay.Close;
end;

//______________________________________________________________________________
//This method will close DirectPlay
procedure TDGCPlay.CloseSession;
begin
     FDirectPlay.Close;
end;

//______________________________________________________________________________
function TDGCPlay.GetNumberOfPlayers : DWORD;
begin
  result := gNumberOfPlayers;
end;

//______________________________________________________________________________
//This method is called to generate the list of available service providers.
//The result is stored in FProviderNames
procedure TDGCPlay.EnumServers;
var
  Status : DWORD;
  loop   : Integer;
begin
  FProviderNames.Clear;
  gProviderNames.Clear;
  gGetGUID := False;

  Status := DirectPlayEnumerate(EnumServerCallback, Nil);
  If Status <> DP_OK then
     begin
       ShowMessage(Format('Enum Servers Failed :%x',[Status]));
       exit;
     end;

  FNumberOfProviders := gProviderNames.Count;
  For loop := 0 to FNumberOfProviders -1 do
      FProviderNames.Add(gProviderNames.Strings[loop]);
end;

//______________________________________________________________________________
//This will gather the names of the sessions available on the selected service
//provider.
//FOR NOW : Only games matching GUIDs will be supported...
Procedure TDGCPlay.EnumSessions;
var
   Status   : Integer;
   loop     : Integer;
begin
   ZeroMemory(@FSessionDesc, SizeOf(FSessionDesc));
   With FSessionDesc do
   begin
     dwSize          := SizeOf(FSessionDesc);
     guidSession     := FGameGUID;
     StrPCopy(szPassword, FPassword);
   end;

   FSessions.Clear;
   gSessions.Clear;

   Status := FDirectPlay.EnumSessions(FSessionDesc, FTimeOut, EnumSessionsCallback,
                Nil, DPENUMSESSIONS_ALL);
   If Status <> DP_OK then
      begin

        ShowMessage(Format('EnumSessions Failed :%x',[status]));
        exit;
      end;

   FNumberOfSessions := gSessions.Count;
   For loop := 0 to FNumberOfSessions -1 do
   begin
       FSessions.AddObject(gSessions[loop], TObject(gSessions.Objects[loop]));
   end;
end;

//______________________________________________________________________________
//This method is used to fill the name lists with the names of the players for
//the currently selected session.
procedure TDGCPlay.EnumPlayers(SessionNumber: DWORD);
var
   Status : Integer;
begin
   //Clear existing names...
   FPlayerLNames.Clear;
   FPlayerSNames.Clear;
   gPlayerLNames.Clear;
   gPlayerSNames.Clear;
   FSessionID := SessionNumber;

   //Add Player '0 - All' to name lists
   FPlayerLNames.AddObject('All', TObject(0));
   FPlayerSNames.AddObject('All', TObject(0));
   gNumberOfPlayers := 0;

   Status := FDirectPlay.EnumPlayers(SessionNumber, EnumPlayersCallback, Nil, DPENUMPLAYERS_SESSION);
   If Status <> DP_OK then
      begin
        ShowMessage(Format('EnumPlayers Failed :%x',[status]));
        exit;
      end;
   gNumberOfPlayers := gPlayerLNames.Count;
end;

//______________________________________________________________________________
//This method is used to Create a DirectPlay Session. First creating the DP
//Object, then the Session, then the Local Player.
procedure TDGCPlay.OpenSession(SessionNumber : DWORD);
var
   Status       : Integer;
   FormalName   : Array[0..DPLONGNAMELEN  -1] of Char;
   FriendlyName : Array[0..DPSHORTNAMELEN -1] of Char;

begin
  //Get a list of all the players in this session by calling EnumPlayers
  FSessionID := SessionNumber;
  EnumPlayers(SessionNumber);

  //Now create a remote player for the session selected
  ZeroMemory(@FSessionDesc, SizeOf(FSessionDesc));
  With FSessionDesc do
  begin
     dwSize         := SizeOf(FSessionDesc);
     guidSession    := FGameGUID;
     dwSession      := FSessionID;
     dwFLAGS        := DPOPEN_OPENSESSION;
  end;

  Status := FDirectPlay.Open(FSessionDesc);
  If Status <> DP_OK then
     begin
       ShowMessage(Format('Open Failed :%x',[status]));
       exit;
     end;

  //Now that the Session is opened, create the remote player
  //This sends DPSYS_ADDPLAYER message to other players...
  StrPCopy(FormalName,   FLongName );
  StrPCopy(FriendlyName, FShortName);

  Status := FDirectPlay.CreatePlayer(FLocalPlayer, FriendlyName, FormalName, @FDPMessageHandle);
  If Status <> DP_OK then
     begin
       ShowMessage(Format('Local Player Create Failed :%x',[status]));
       exit;
     end;

  FPlayerLNames.AddObject(FormalName,   TObject(dword(FLocalPlayer)));
  FPlayerSNames.AddObject(FriendlyName, TObject(dword(FLocalPlayer)));
  Inc(gNumberOfPlayers);
  gToID := FLocalPlayer;

  //This will create a Thread that will handle Messages for this player.
  FDPDoEvent := True;
  With TDPMessageThread.Create do
       begin
         gOnAddPlayer    := FOnAddPlayer;
         gOnDeletePlayer := FOnDeletePlayer;
         gOnGameMessage  := FOnGameMessage;
         OnTerminate     := ThreadDone;
       end;
end;

//______________________________________________________________________________
//This method is called to open a selected provider. If LocalGame is TRUE then
//this will call the procedure to CREATE a new game instance, otherwise it
//assumes that it will connect to an existing session and gather the names of
//the Sessions available on that Provider.
procedure TDGCPlay.OpenProvider(ServerName : String; LocalGame : Boolean);
var
   Status : DWORD;
begin
  If FDirectPlay = Nil then
  begin
    gLookupName := ServerName;
    gProviderNames.Clear;
    gGetGUID := True;

    Status := DirectPlayEnumerate(EnumServerCallback, Nil);
    If Status <> DP_OK then
       begin
         ShowMessage(Format('Enum Servers Failed :%x',[Status]));
         exit;
       end;

    FServerGUID := gServerGUID;

    //Create a DirectPlay object using the selected provider.
    Status := DirectPlayCreate(FServerGUID, FDirectPlay,Nil);
    If Status <> DP_OK then
       begin
         ShowMessage(Format('DirectPlay Create Failed :%x',[Status]));
         exit;
       end;
  end;

  If LocalGame = True then
     CreateSession
  else
     EnumSessions;

end;

//______________________________________________________________________________
//This procedure is called when the LocalGame flag of OpenProvider is set to
//True -- This will create a new session and the local player.
procedure TDGCPlay.CreateSession;
var
  Status : DWORD;
  FormalName   : Array[0..DPLONGNAMELEN  -1] of Char;
  FriendlyName : Array[0..DPSHORTNAMELEN -1] of Char;
begin
  //Create (Open) a new session with info provided by the user.
  ZeroMemory(@FSessionDesc, SizeOf(FSessionDesc));
  With FSessionDesc do
  begin
     dwSize         := SizeOf(FSessionDesc);
     guidSession    := FGameGUID;
     dwMaxPlayers   := FMaxPlayers;
     dwFlags        := DPOPEN_CREATESESSION;
     StrPCopy(szSessionName, FGameName);
     StrPCopy(szPassword, FPassword);
     StrPCopy(szUserField  , FUserData);
  end;

  Status := FDirectPlay.Open(FSessionDesc);
  if Status <> DP_OK then
     begin
       ShowMessage(Format('DirectPlay Open Failed :%x',[status]));
       exit;
     end;

  //Now Create the LOCAL (This Computer) player.
  StrPCopy(FormalName,   FLongName );
  StrPCopy(FriendlyName, FShortName);

  Status := FDirectPlay.CreatePlayer(FLocalPlayer, FriendlyName, FormalName, @FDPMessageHandle);
  If Status <> DP_OK then
     begin
       ShowMessage(Format('Local Player Create Failed :%x',[status]));
       exit;
     end;

  //Clear existing names...
  FPlayerLNames.Clear;
  FPlayerSNames.Clear;
  gPlayerLNames.Clear;
  gPlayerSNames.Clear;

  //Add a position in the list for 'ALL' players, id '0'
  FPlayerLNames.AddObject('All', TObject(0));
  FPlayerSNames.AddObject('All', TObject(0));

  //Add the LOCAL (This Machine) player to the players list
  FPlayerLNames.AddObject(FormalName,   TObject(dword(FLocalPlayer)));
  FPlayerSNames.AddObject(FriendlyName, TObject(dword(FLocalPlayer)));

  //Number of ACTUAL players in the game (not counting '0 - All');
  gNumberOfPlayers := 2;
  gToID := FLocalPlayer;


  //This will create a Thread that will handle Messages for this player.
  FDPDoEvent := True;
  With TDPMessageThread.Create do
       begin
         gOnAddPlayer    := FOnAddPlayer;
         gOnDeletePlayer := FOnDeletePlayer;
         gOnGameMessage  := FOnGameMessage;
         OnTerminate     := ThreadDone;
       end;
end;

//______________________________________________________________________________
constructor TDPMessageThread.Create;
begin
  inherited Create(False);
  FreeOnTerminate := True;
end;

//______________________________________________________________________________
procedure TDPMessageThread.Execute;
var
   Status  : Integer;
   BufSize : DWORD;
begin
  While (FDPDoEvent = True) do
  begin
     Status := WaitForSingleObject(FDPMessageHandle, INFINITE);
     Case Status of
     WAIT_TIMEOUT:
        //Does nothing -- no message
     ELSE
       Begin
         ResetEvent(FDPMessageHandle);
         If FDirectPlay <> Nil then
         Begin
            Status := 0;
            While Status = DP_OK do
            begin
              BufSize := SizeOf(gDPMessage);
              Status  := FDirectPlay.Receive(gFromID, gToID, DPRECEIVE_ALL,
                         gDPMessage, BufSize);
              If Status = DP_OK then
              begin
                 ProcessMessage(gDPMessage, gFromID);
              end;
            end;    // End of While Status
         end;       // End of If FDirectPlay
       end;         // End of ELSE
     end;           // End of Case
  end;              // End of While..Begin
end;                // End of Procedure



//______________________________________________________________________________
Procedure ProcessMessage(var buffer; FromID : TDPID);
var
  LongName  : String;
  ShortName : String;
begin
   gDPMessage := TDPMessage(Buffer);
   CASE TDPMSG_GENERIC(Buffer).dwType of
   DPSYS_ADDPLAYER:
      begin
        gPlayerLNames.AddObject(TDPMSG_ADDPLAYER(Buffer).szLongName,
               TObject(TDPMSG_ADDPLAYER(Buffer).DPID));
        gPlayerSNames.AddObject(TDPMSG_ADDPLAYER(Buffer).szShortName,
               TObject(TDPMSG_ADDPLAYER(Buffer).DPID));
        Inc(gNumberOfPlayers);
        If Assigned(gOnAddPlayer) then gOnAddPlayer(TDPMSG_ADDPLAYER(Buffer).DPID,
                                                    TDPMSG_ADDPLAYER(Buffer).szLongName,
                                                    TDPMSG_ADDPLAYER(Buffer).szShortName);
      end;
   DPSYS_DELETEPLAYER:
      Begin
        LongName  := gPlayerLNames.Strings[gPlayerLNames.IndexOfObject(TObject(TDPMSG_DELETEPLAYER(Buffer).DPID))];
        ShortName := gPlayerSNames.Strings[gPlayerSNames.IndexOfObject(TObject(TDPMSG_DELETEPLAYER(Buffer).DPID))];
        gPlayerLNames.Delete(gPlayerLNames.IndexOfObject(
               TObject(TDPMSG_DELETEPLAYER(Buffer).DPID)));
        gPlayerSNames.Delete(gPlayerSNames.IndexOfObject(
               TObject(TDPMSG_DELETEPLAYER(Buffer).DPID)));
        Dec(gNumberOfPlayers);
        If Assigned(gOnDeletePlayer) then gOnDeletePlayer(TDPMSG_DELETEPLAYER(Buffer).DPID,
                                                          LongName, ShortName);
      end;
   else
      begin
        If Assigned(gOnGameMessage) then gOnGameMessage(FromID,
                                         gDPMessage.dwType, @gDPMessage);
      end;
   end; // End CASE DPMSG
end;

//______________________________________________________________________________
Procedure TDGCPlay.SendMessage(ToID: TDPID; var DPMessage);
var
  BufSize : DWORD;
  TempBuf : TDPMessage;
begin
  TempBuf := TDPMessage(DPMessage);
  BufSize := SizeOf(TempBuf);
  FDirectPlay.Send(FLocalPlayer, ToID, DPSEND_HIGHPRIORITY, TempBuf, BufSize);
end;

//______________________________________________________________________________
function TDGCPlay.GameMessage : Pointer;
begin
  result := @gDPMessage;
end;

//______________________________________________________________________________
Procedure TDGCPlay.ThreadDone(Sender: TObject);
begin
  FDPDoEvent := False;
end;

//______________________________________________________________________________
//This is the function that is called from the EnumServers method. It will
//return all of the available servers that a user can use for the means of
//communication.
function EnumServerCallback(var lpspGUID:TGUID; lpServer:lpSTR; dwMajor:DWORD;
         dwMinor:DWORD; lpContext:Pointer):BOOL;stdcall;
begin
  if (gGetGUID = True) and (lpServer = gLookupName) then
     gServerGUID := lpspGUID
  else
  begin
     gProviderNames.Add(lpServer);
  end;
  Result := True;
end;

//______________________________________________________________________________
//This is the function that is called from the EnumSessions procedure. A call to
//this will return all the names of the sessions available.
function EnumSessionsCallBack( var lpDPSessionDesc: TDPSESSIONDESC ;
      lpContext: Pointer ; var lpdwTimeOut: DWORD ; dwFlags: DWORD ): BOOL ;
      stdcall ;
begin
  Try
     begin
       gSessions.AddObject(lpDPSessionDesc.szSessionName, TObject(lpDPSessionDesc.dwSession));
     end;
  except
    begin
      Result := FALSE;
      exit;
    end;
  end;
  result := True;
end;

//______________________________________________________________________________
//This is the function that is called from the EnumPlayers procedure. A call to
//this procedure will return all the players in the sessions quarried.
function EnumPlayersCallback(dpid:TDPID; lpShortName:lpSTR; lpLongName:lpSTR;
         dwFlags:DWORD; lpContext:Pointer):BOOL; stdcall;
begin
     gPlayerLNames.AddObject(lpLongName,  TObject(DWORD(DPID)));
     gPlayerSNames.AddObject(lpShortName, TObject(DWORD(DPID)));
     Result := True;
end;

procedure Register;
begin
  RegisterComponents('DGC', [TDGCPlay]);
end;

end.
