/*
 *      Console:   Logger and console routines
 */


#ifdef _WIN32
#define WIN32_EXTRA_LEAN
#include <windows.h>
#endif


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "sys.hpp"
#include "console.hpp"
#include "memory.hpp"
#include "keys.hpp"
#include "vidmodes.h"
#include "itg/itg.h"    // Graphics library
#include "rgb_conv.hpp"


//      Variables


static char **C_list;          // List of items
static unsigned int *C_tab;    // Table of Y positions, for c_updatelist
static char *contemp;          // text buffer
static char *consingle;        // single line

char *cfa;                     // current function address
char conlen=10;                // Number of lines in the console, up to 20
char conwid=40;                // Number of rows in the console, up to 80
char conlog;                   // Toggle, if we log the console to bootlog.txt
extern char digit[256][8];
static int textcolour=0xffff;

void (*bootmsg)(char *msg, ...); // dynamic logger (call this mainly)
int conx=16,cony=16,old_conx,old_cony,old_conl;
static char console_init=0;

extern int logx,logy,loglen;
extern char logpic[32];
static SPRITE console_backing;
short *logpicture;

extern char no_panic;
extern char SuperPanic;
extern char quickstart;

//      Functions

extern void On();
extern int display_PCX(char *picture,short *screen);
static void C_ycalc();
void C_printit(char *msg);

static void superpanic();
//static int my_strlen(char *a);

// Code

/*
 *      init_logger - start bootlogger
 */

void Init_logger()
{
bootmsg=boot0;          // Set default logger, backup last log
remove("bootlog.prv");
rename("bootlog.txt","bootlog.prv");
}

/*
 *      switch_logger_on - switch to graphical boot logger
 */

void switch_logger_on()
{
boot2("Starting graphics system\n");
On();

boot2("Setting up image\n");
logpicture = (short *)M_get(1,BUFFERSIZE);
display_PCX(logpic,logpicture);

boot2("Starting console\n");

// Store old console size

old_conx = conx;
old_cony = cony;
old_conl = conlen;

// Set new size

conx = logx;
cony = logy;
conlen = loglen;

boot2("console: %d,%d and %d lines\n",conx,cony,conlen);

bootmsg = boot1;

C_ycalc();
C_cls();
}

/*
 *      switch_logger_off - shut down graphical logger
 */

void switch_logger_off()
{
boot2("Switching console to in-game settings\n");

conx = old_conx;
cony = old_cony;
conlen = old_conl;

graflog = 0;
bootmsg = boot2;
C_cls();
C_ycalc();
M_free(logpicture);
}

/*
 *      logger_get_backing - get background for ingame Console
 */

void logger_get_backing()
{
console_backing.get_sprite(conx,cony,swapscreen);       // Get existing stuff
}

/*
 *      boot0 - Text mode logger
 */

void boot0(char *msg, ...)
{
FILE *blt;
char buffer[256];

	va_list ap;
	va_start(ap, msg);

        if(SYS_GET_KEY() == KEY_ESC)
            exit(1);

        vsprintf(buffer,msg,ap);
	SYS_CPRINT(buffer);
	if(buffer[strlen(buffer)-1]=='\n')
   	    SYS_CPRINT("\r");

        if(quickstart)
            {
	    va_end(ap);
            return;
            }

	blt=fopen("bootlog.txt","a");
	if(!blt)
		panic("bootmsg()","Could not append to bootlog.txt",NULL);
	vfprintf(blt,msg,ap);
	fclose(blt);
	va_end(ap);
}

/*
 *      boot1 - Graphical logger, not tested yet
 */

void boot1(char *msg, ...)
{
FILE *blt;
char buffer[256];

	va_list ap;
	va_start(ap, msg);

        if(SYS_GET_KEY() == KEY_ESC)
            {
            V_off();
            exit(1);
            }

        vsprintf(buffer,msg,ap);

        C_printf(buffer);
	C_updatelist();

//	blit(screen,swapscreen);
        update(swapscreen);

        if(quickstart)
            {
	    va_end(ap);
            return;
            }



	blt=fopen("bootlog.txt","a");
	if(!blt)
		panic("bootmsg()","Could not append to bootlog.txt",NULL);
	vfprintf(blt,msg,ap);
	fclose(blt);
	va_end(ap);
}

/*
 *      boot2 - (quiet logger)  Log but don't print it onscreen
 */

void boot2(char *msg, ...)
{
FILE *blt;

if(quickstart)
        return;

	va_list ap;
	va_start(ap, msg);

        blt=fopen("bootlog.txt","a");
        if(!blt)
		panic("bootmsg()","Could not append to bootlog.txt",NULL);
        vfprintf(blt,msg,ap);
        fclose(blt);
	va_end(ap);
}

/*
 *      bug- (like boot2)  Log a bug and set a warning flag
 */

void Bug(char *msg, ...)
{
FILE *blt;
	va_list ap;
	va_start(ap, msg);

/*	if(kbhit())
		if(getch()==27)
			{
			off();
			exit(1);
			}*/

	blt=fopen("bootlog.txt","a");
	if(!blt)
		panic("bootmsg()","Could not append to bootlog.txt",NULL);

        fprintf(blt,"BUG: ");
        vfprintf(blt,msg,ap);
	fclose(blt);

        buggy=1;
	va_end(ap);
}

/*
 *      C_cls - clear the console
 */

void C_cls()
{
for(int ctr=0;ctr<=C_LEN;ctr++)
    C_list[ctr][0]=0;
}

/*
 *  C_printf - Printf on the console and split the string if it won't fit.
 *             I've never done word splits before, so I'm making this up
 *             as I go along.. I presume there is a 'proper' way to do it?
 */

void C_printf(char *msg, ...)
{
char *start;
char *line[128];
int len,ctr,ctr2,lines,col;
char fin;

va_list ap;
va_start(ap, msg);

vsprintf(contemp,msg,ap);          // Temp is now the message
va_end(ap);

if(strlen(contemp)>3072)
    {
    boot2("Line too long:\n%s\n",contemp);
    panic(cfa,"Line too long.","See bootlog.txt for details");
    }

// Sanity Check
len = strlen(contemp)-1;
for(ctr=0;ctr<len;ctr++)
    if((unsigned char)contemp[ctr]>191)
        contemp[ctr]=' ';

// Break lines that are too long by inserting |

col=0;
for(ctr=0;ctr<len;ctr++)
    {
    if(contemp[ctr] == '|')        // Newline, reset column count
        col=0;                  // -1 because it gets incremented later
    if(col==conwid-1)           // Reached a boundary
        {
        ctr2=ctr;
        fin=0;
        do
            {
            if(contemp[ctr2] == '|') // Oh dear.  There weren't any spaces.
                {
                contemp[ctr] = '|';  // Blow the current character away
                fin=1;              // to prevent a failure
                col=0;
                }
            if(contemp[ctr2] == ' ') // Found a space.
                {
                contemp[ctr2] = '|'; // Turn the space to | for newline
                fin=1;
                col=ctr-ctr2;     // Reset column counter to new value
                }
            if(ctr2 == 0)         // Oh no.  We've reached the start
                {
                contemp[ctr]='|';    // Emergency, zap current character
                fin=0;              // to prevent a failure
                col=0;
                }
            ctr2--;
            } while(!fin);
        }
    col++;
    }


// First find carriage returns and decompose into lines

lines=0;
start = &contemp[0];
line[lines++] = start;
line[lines] = start;  // Safeguard
for(ctr=0;ctr<len;ctr++)
    {
    if(contemp[ctr] == '|')
        {
        start=&contemp[ctr+1];
        line[lines++]=start;
        line[lines] = start;  // Safeguard
        }
    if(lines > 128)
        {
        boot2("Too many lines in message:\n%s\n",contemp);
        panic(cfa,"Too many lines in message","See BOOTLOG.TXT");
        }
    }

for(ctr=0;ctr<lines-1;ctr++)
    {
    len = (line[ctr+1] - line[ctr])-1;
    strncpy(consingle,line[ctr],len);
    consingle[len]=0;
    C_printit(consingle);
    C_bumplist();        // Force a linefeed (i.e. not in string)
    }

C_printit(line[lines-1]); // Take care of 'natural' linefeeds
}

/*
 *  C_printit - Raw print on the console
 */

void C_printit(char *msg)
{
if(conlog)
    boot2("PRN: %s",msg);       // If there is a CR, boot2 will know about it

strcat(C_list[conlen],msg);     // Message now in console buffer

if(strchr(msg,'\n') != NULL)    // If there is a CR, act upon it
    C_bumplist();
}

/*
 *  C_delete - erase the current line of text
 */

void C_delete()
{
strcpy(C_list[conlen],"");     // Message now in console buffer
}

/*
 *  C_bumplist - Print a CR on the console
 */

void C_bumplist()
{
char *t;

// We make the list go around in a circular manner, then nail the end one

t=C_list[0];
for(int ctr=0;ctr<conlen;ctr++)
	C_list[ctr]=C_list[ctr+1];
C_list[conlen]=t;
C_list[conlen][0]=0;
if(conlog) boot2("\n");
}

/*
 *  C_updatelist - Display the list onscreen, character by character
 */

void C_updatelist()
{
int xo;
unsigned char *cptr;

if(graflog < 1)
    console_backing.block_put_sprite(conx,cony,swapscreen);
else
    fblit((char *)swapscreen,(char *)logpicture);

for(int ctr=0;ctr<=conlen;ctr++)
	{
	cptr=(unsigned char*)C_list[ctr];
	for(xo=conx;(*cptr)!=0;cptr++)
            I32font(xo+=8,C_tab[ctr],digit[*cptr],textcolour,swapscreen);
	//	    digit[*cptr].cel_put_sprite(xo+=8,C_tab[ctr],swapscreen);
	}
}

/*
 *  Start the console and initialise the list
 */

void C_init()
{
cfa="C_init()";
C_list=(char **)M_get(sizeof(char *),(C_LEN+2));
C_tab=(unsigned int *)M_get(sizeof(int),(C_LEN+2));

console_init = 1;

console_backing.allocate(conwid * 8,(conlen+1) * 8);        // Allocate space

for(int ctr=0;ctr<=C_LEN;ctr++)
	{
	C_list[ctr]=(char *)M_get(1,128);
	C_list[ctr][0]=0;
	}

C_ycalc();      // Set up the Y lookup table

contemp = (char *)M_get(1,4096);
consingle = (char *)M_get(1,256);
}

/*
 *  calculate the Y offsets
 */

void C_ycalc()
{
cfa="C_ycalc()";
for(int ctr=0;ctr<=C_LEN;ctr++)
	C_tab[ctr]=cony+(ctr<<3);
}

/*
 *  Shut down the console and free the memory
 */

void C_term()
{
M_free(contemp);
M_free(consingle);
M_free(C_tab);
for(short ctr=0;ctr<=C_LEN;ctr++)
	M_free(C_list[ctr]);
M_free(C_list);
console_init = 0;
console_backing.free();
}

/*
 *  Set the location on screen of the log
 */

void C_gotoxy(int x,int y)
{
conx = x;
cony = y;
for(int ctr=0;ctr<=C_LEN;ctr++)
	{
	C_list[ctr][0]=0;
	C_tab[ctr]=cony+(ctr<<3);
	}
}

/*
 *  Write a string direct to screen, not via the console
 */

void C_printxy(int x,int y,char *msg)
{
int len;

len=strlen(msg);
for(int ctr=0;ctr<len;ctr++)
    I32font(x+=8,y,digit[msg[ctr]],textcolour,swapscreen);
}


/*
 *  Set the colour for all future text I/O
 */

void C_color(int r,int g,int b)
{
textcolour=MakeRGB(r,g,b);
}


/*
 * Panic - go back to DOS and report the problem
 */

void panic(char *curfunc,char *m,char *n)
{
static int pip=0;

if(no_panic)
    exit(1);

if(InGFX)
	{
	puts("\n");
	puts(m);
	if(n)
		puts(n);
	exit(1);
	}

V_off();
puts(  "PANIC:Ŀ");
printf("                                                                             \r %s\n",m);
if(n)
	printf("                                                                             \r %s\n",n);
printf("                                                                             \r panic call from %s\n",curfunc);
puts(  "\n\n\n");

if(pip)
    {
    puts("Panic Panic");
    exit(1);
    }

pip = 1;        // Safety valve, in case panic panics

boot2("PANIC:Ŀ");boot2("\n");
boot2(m); boot2("\n");
if(n) boot2(n); boot2("\n");
boot2("panic call from %s\n",curfunc);

if(SuperPanic == 1)     // It must be 1, we do NOT want this to happen
    superpanic();       // by accident!

exit(1);
}

/*
 *  S_Panic - 'Safe' panic - for C++ destructors
 */

void S_panic(char *curfunc,char *m,char *n)
{
if(no_panic)
    exit(1);

V_off();

puts(  "S_PANIC:Ŀ");
printf("                                                                             \r %s\n",m);
if(n)
	printf("                                                                             \r %s\n",n);
printf("                                                                             \r panic call from %s\n",curfunc);
puts(  "\n\n\n");

exit(1);
}

/*
 *      Plot - draw the little row of dots
 *
 *             Call Plot(number); to set the maximum length of the row
 *             Then call Plot(0); each cycle of the loop to draw a dot
 */


void Plot(int max)
{
static int ctr,mpos;
ctr++;
if(max)
	{
	mpos=max/20;
	if(mpos<1)
		mpos=1;
	ctr=0;
	}
if(ctr>mpos)
	{
	bootmsg(".");
	ctr=0;
	}
}

void superpanic()
{
printf("Causing an exception...\n");
#ifdef _WIN32

DebugBreak();

#else

asm("int $3");

#endif
}



