//
//      Darkness - The lighting engine
//

#include "console.hpp"
#include "core.hpp"
#include "memory.hpp"
#include "loaders.hpp"
#include "object.hpp"
#include "rgb_conv.hpp"
#include "bitmap.h"
#include "itg/itg.h"

// Defines

// #define WOBBLE       // This makes the lights move slightly like in U7
                        // Unforunately the light boundaries move too..

struct LTAB
    {
    int x,y;
    int light;
    };

// Variables

    // Lookup table for the possible light paths.
    // Each node has three adjacent squares which are lit by the engine.
    // The dark routine lights up (or darkens) each square, and marks it
    // as being used, so that it cannot be brightened twice.
    // This is necessary because there will be overlap between some nodes.
    // That's necessary due to this algorithm...

    static int node[8][4][2]={
                        { {-1,-1},{-2,-2},{-2,-1},{-1,-2}  }, // top-left
                        { {+0,-1},{+0,-2},{-1,-2},{+1,-2}  }, // top
                        { {+1,-1},{+2,-2},{+1,-2},{+2,-1}  }, // top-right
                        { {+1,+0},{+2,+0},{+2,-1},{+2,+1}  }, // right
                        { {+1,+1},{+2,+2},{+2,+1},{+1,+2}  }, // bottom-right
                        { {+0,+1},{+0,+2},{-1,+2},{+1,+2}  }, // bottom
                        { {-1,+1},{-2,+2},{-2,+1},{+1,+2}  }, // bottom-left
                        { {-1,+0},{-2,+0},{-2,-1},{-2,+1}  }  // left
                      };

char *darkmap;          // A screen buffer used to draw the darkness on
char *darkrect;         // A linear buffer for darkmap rendering
SPRITE lightmap;        // The shape of the lightmask we'll use
char *lm[8][8];         // The lightmask split into tiles.
//SPRITE lmcore;        // The middle part of the lightmask (optimisation)
char *lmcore;
int lmw,lmh;            // Width and height of the lightmask in tiles.
int lmw2,lmh2;          // lmw and lmh respectively, but divided by two

static unsigned char darklevel;  // Ambient darkness (0=daylight, 31=pure black)
static unsigned long darklevel4; // the above, but packed for dword block-move

extern int mapx,mapy;           // Current position on the map

// Function pointers to the 15 or 16 bit darkness routine

void (*I32darkmem)(char *d,char *s,int pixels);
void (*I32lightning)(char *d);

// Functions

void Dark_Init();                       // Set up
void Dark_Project();                    // Render
void Dark_Term();                       // Shut Down
extern int blocksLight(int x,int y);    // Casts a 'shadow'?
void SetDarkness(int level);            // Set ambient darkness level
static inline void proj_dark(int x, int y);
static inline void proj_light(int x, int y);

// Code

/*
 *      Dark_Init - set up the lighting engine
 */

void Dark_Init()
{
bootmsg("Allocate Darkmap\n");
darkmap = (char *)M_get(BUFFERSIZE,sizeof(char));

// Load in the light mask

load_CEL(&lightmap,"sprites/lightmap");

// Clear the darkmap screen

//fset(darkmap,0,76800);

// Put the light mask where we can get_sprite it

lightmap.block_put_sprite(0,0,(char *)bg_screen);

// Find size of the light mask in tiles

lmw = lightmap.get_width()/32;
lmh = lightmap.get_height()/32;

// Correct the value (since it rounds down)

if((lmw * 32) < lightmap.get_width())
    lmw++;
if((lmh * 32) < lightmap.get_height())
    lmh++;

lmw2=lmw/2;
lmh2=lmh/2;

// Break the lightmap into tiles

for(int x=0;x<=lmw;x++)
    for(int y=0;y<=lmh;y++)
        {
        lm[x][y]=(char *)M_get(32,32);
        for(int cx=0;cx<32;cx++)
            for(int cy=0;cy<32;cy++)
                lm[x][y][(32*cy)+cx]=bg_screen[(1280*(cy+(y<<5)))+((cx+(x<<5))*2)]&31;
        }

// Now allocate the central hub
// (The inner 3x3 tiles are not subject to solidity checks, so we can just
//  project it always, instead of using tiles and checking.)

lmcore=(char *)M_get(96,96);
for(int cx=0;cx<96;cx++)
    for(int cy=0;cy<96;cy++)
        lmcore[(96*cy)+cx]= bg_screen[(1280*(cy+32))+((cx+32)*2)]&31;

// Set the default light level to daytime
SetDarkness(0);
}

/*
 *      Dark_Project - render the lighting
 */

void Dark_Project()
{
int cx,cy,x,y,ex,ey,ctr;
int startx,starty;
OBJECT *temp;
LTAB light[256];
int lights;

fset(darkmap,darklevel4,76500);      // Clear the darkness map with this level

// Find all lightsources

lights=0;

ey = mapy+VSH+3;
ex = mapx+VSW+3;
startx = mapx-3;
starty = mapy-3;

if(startx<0) startx=0;
if(starty<0) starty=0;

for(cy=starty;cy<ey;cy++)
for(cx=startx;cx<ex;cx++)
    {
    temp = GetRawObjectBase(cx,cy);
    for(;temp;temp=temp->next)              // Traverse the list
        if(temp->flags.on && lights<255)
            if(temp->light)
                {
                // if the object is in the screen region
                if((cx+lmw >= mapx) && (cx-lmw < ex))
                    if((cy+lmh >= mapy) && (cy <= ey))
                        {
                        x = cx-mapx;
                        y = cy-mapy;
                        if(x+lmw>0 && y+lmh>0)
                            if(x-lmw2<VSW && y-lmh2<VSH)
                                {
                                light[lights].light=temp->light;
                                light[lights].x=x;
                                light[lights].y=y;
                                lights++;
                                }
                        }
                }
    }

// Project darksources first

for(ctr=0;ctr<lights;ctr++)
    if(light[ctr].light<0)
        proj_dark(light[ctr].x,light[ctr].y);

// Then project lightsources

for(ctr=0;ctr<lights;ctr++)
    if(light[ctr].light>0)
        proj_light(light[ctr].x,light[ctr].y);
}

/*
 *      Dark_Term - Shut down the lighting
 */

void Dark_Term()
{
lightmap.free();
M_free(darkmap);
}

/*
 *      SetDarkness - Set the current ambient darkness level
 */

void SetDarkness(int r)
{
darklevel = (r>>3);

// This ought to assemble nicely

darklevel4 = darklevel;
darklevel4<<=8; darklevel4+= darklevel;
darklevel4<<=8; darklevel4+= darklevel;
darklevel4<<=8; darklevel4+= darklevel;
}

void proj_dark(int x, int y)
{
int xx,yy,cx,cy;
char dlm[8][8];
int xoffset,yoffset;

#ifdef WOBBLE
xoffset = STARTX+(1-rnd(2));
yoffset = STARTX+(1-rnd(2));
#else
xoffset = STARTX;
yoffset = STARTX;
#endif

// Light central hub square

I32darkspr_inv(((x-1)<<5)+xoffset,((y-1)<<5)+yoffset,96,96,darkmap,lmcore);

// Clear the already-lit array
fset(dlm,0,16);     // (8*8)/4 = 16

cx = x+mapx;
cy = y+mapy;

for(int ctr=0;ctr<8;ctr++)
    {
    // Store for faster access
    xx = (node[ctr][0][0]);
    yy = (node[ctr][0][1]);

    if(!blocksLight(cx+xx,cy+yy))
        {
        xx = (node[ctr][1][0])+2;
        yy = (node[ctr][1][1])+2;
        if(!dlm[xx][yy])
            {
            I32darkspr_inv(((x+xx-2)<<5)+xoffset,((y+yy-2)<<5)+yoffset,32,32,darkmap,lm[xx][yy]);
            dlm[xx][yy]=1;
            }


        xx = (node[ctr][2][0])+2;
        yy = (node[ctr][2][1])+2;
        if(!dlm[xx][yy])
            {
            I32darkspr_inv(((x+xx-2)<<5)+xoffset,((y+yy-2)<<5)+yoffset,32,32,darkmap,lm[xx][yy]);
            dlm[xx][yy]=1;
            }

        xx = (node[ctr][3][0])+2;
        yy = (node[ctr][3][1])+2;
        if(!dlm[xx][yy])
            {
            I32darkspr_inv(((x+xx-2)<<5)+xoffset,((y+yy-2)<<5)+yoffset,32,32,darkmap,lm[xx][yy]);
            dlm[xx][yy]=1;
            }

        }
    }
}

void proj_light(int x, int y)
{
int xx,yy,cx,cy;
char dlm[8][8];
int xoffset,yoffset;

xoffset = STARTX;//+(1-rnd(2));
yoffset = STARTX;//+(1-rnd(2));

// Light central hub square

I32darkspr(((x-1)<<5)+xoffset,((y-1)<<5)+yoffset,96,96,darkmap,lmcore);

// Clear the already-lit array
fset(dlm,0,16);     // (8*8)/4 = 16

cx = x+mapx;
cy = y+mapy;

for(int ctr=0;ctr<8;ctr++)
    {
    // Store for faster access
    xx = (node[ctr][0][0]);
    yy = (node[ctr][0][1]);

    if(!blocksLight(cx+xx,cy+yy))
        {
        xx = (node[ctr][1][0])+2;
        yy = (node[ctr][1][1])+2;
        if(!dlm[xx][yy])
            {
            I32darkspr(((x+xx-2)<<5)+xoffset,((y+yy-2)<<5)+yoffset,32,32,darkmap,lm[xx][yy]);
            dlm[xx][yy]=1;
            }


        xx = (node[ctr][2][0])+2;
        yy = (node[ctr][2][1])+2;
        if(!dlm[xx][yy])
            {
            I32darkspr(((x+xx-2)<<5)+xoffset,((y+yy-2)<<5)+yoffset,32,32,darkmap,lm[xx][yy]);
            dlm[xx][yy]=1;
            }

        xx = (node[ctr][3][0])+2;
        yy = (node[ctr][3][1])+2;
        if(!dlm[xx][yy])
            {
            I32darkspr(((x+xx-2)<<5)+xoffset,((y+yy-2)<<5)+yoffset,32,32,darkmap,lm[xx][yy]);
            dlm[xx][yy]=1;
            }

        }
    }
}


