/*
 *   IRE Filesystem wrappers
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
// MR: added
#ifdef _WIN32
#include <string.h>
#else
#include <strings.h>
#endif

// MR: added
#ifndef _WIN32
#include <unistd.h>
#endif

#include "doslib.hpp"
#include "sys.hpp"
#include "fs.hpp"
#include "memory.hpp"

// Defines

#define MAX_RARS 16
#define USE_RFS         // Comment this out to prevent RAR loader (debugging)

// Variables

static char **RFS_nt;   // I can't remember what this is anymore
static FILE *_Rfp;
static long _Rfiles=0,_Rcur;
static char *_Rname,**_RArgV;
static short _RArgC,Rstep,Rctr;
static struct FileHeader RFH;
static long RFnum=0, Rnum;
static char *RARS[MAX_RARS];
static RFS *RFStab;
static char pip=0;              // Used to exit dangerous panic-in-panic loops

char resdir[128];   // Name of the main directory
extern void panic(char *curfunc,char *m,char *n);
extern void S_panic(char *curfunc,char *m,char *n);

// Functions

static void R_Count();
static void R_Mount();
static void Traverse(void (*Hfunc)());
static void getstep();
void slashes(char *p);

// Code


/*
 *      load::load - initialise the FSW
 */

LOAD::LOAD()    // Load constructor
{
fileptr=NULL;
Xlength=NULL;
Xorigin=NULL;
}

LOAD::~LOAD()    // Load destructor
{
if(pip)         // Panic in panic?
    return;     // Yes: exit the loop

pip = 1;
if(fileptr != NULL)
	S_panic(cfa,"A file was never closed!  Truename is:",truename);
}

/*
 *      load::open_current - open the file only from current directory
 */

void LOAD::open_current(char *fname)    // Load open from current directory
{
char path1[128];
char path2[128];

if(fileptr != NULL)
	panic(cfa,"attempt to reopen unclosed file:",fname);

strcpy(path1,fname);
strlwr(path1);
sprintf(path2,"%s/%s",resdir,fname);
strlwr(path2);

slashes(path1);
slashes(path2);
slashes(fname);

// path is now the full filename

// Try in current directory

fileptr=fopen(fname,"rb");
if(!fileptr)
	panic(cfa,"could not open file from current directory:",fname);

// Got it

strcpy(truename,path2);                         // Physical file is opened
Xlength=::filelength(fileno(fileptr));          // Get it's length
Xorigin=ftell(fileptr);                         // Get is position
return;
}

/*
 *      load::open, open the file from curdir, resdir or RFS
 */

void LOAD::open(char *fname)    // Load open
{
char path1[128];
char path2[128];
int fptr;

// Sanity check
if(fileptr != NULL)
	panic(cfa,"attempt to reopen unclosed file:",fname);

// set path2 to be the full filename

strcpy(path1,fname);
sprintf(path2,"%s/%s",resdir,fname);

strlwr(path1);
strlwr(path2);
slashes(path1);
slashes(path2);

// try and open it


fileptr=fopen(path1,"rb");              // Try current dir first
if(!fileptr)
	fileptr=fopen(path2,"rb");       // Now try in the res directory

if(fileptr)                             // If we have it, log it
    {
    strcpy(truename,path2);
    Xlength=::filelength(fileno(fileptr));
    Xorigin=ftell(fileptr);
    return;
    }

// So far we haven't found it, let's open the RFS and look there.

#ifdef USE_RFS

fptr=RFS_Open(path2);
if(fptr!=-1)
    {
    strcpy(truename,RFStab[fptr].rarname);   // Physical file is the .RAR
    slashes(truename);
    fileptr=fopen(truename,"rb");            // Open it
    if(!fileptr)
        panic(cfa,"could not open RAR archive",truename);
    Xlength=RFStab[fptr].length;             // get length
    Xorigin=RFStab[fptr].offset;             // get start position
    ::fseek(fileptr,Xorigin,SEEK_SET);       // do REAL seek, not load::fseek
    return;
    }

#endif
// Could not find it?  Oh well.  CFA is the function that called load::open

panic(cfa,"could not open file:",fname);
}

/*
 *      load::close, close the file with checks and tidy up
 */

void LOAD::close()
{
// First make sure it's open
if(fileptr)
	fclose(fileptr);
// Then tidy up variables
fileptr=NULL;
Xlength=0;
}

/*
 *      load::getb, read a single byte
 */

short LOAD::getb()
{
if(!fileptr)
	panic(cfa,"read attempt on unopened file!",NULL);
return getc(fileptr);
}

/*
 *      load::getw, read a double byte
 */

short LOAD::getw()
{
short x;
if(!fileptr)
	panic(cfa,"read attempt on unopened file!",NULL);

// Read, untangle endianness

x=getc(fileptr);
x+=(getc(fileptr)<<8);

// Return result

return x;
}

/*
 *      load::getl, read 4 bytes
 */

long LOAD::getl()
{
long x;
if(!fileptr)
	panic(cfa,"read attempt on unopened file!",NULL);

// Read in with Fread and return

fread(&x,1,4,fileptr);
return x;
}

/*
 *      load::read, FSW around fread
 */

void LOAD::read(void *x,long n,long s)
{
if(!fileptr)
	panic(cfa,"read attempt on unopened file!",NULL);
fread(x,n,s,fileptr);
}

/*
 *      load::filelength, return stored length of file
 */

long LOAD::filelength()
{
return Xlength;
}

/*
 *      load::fseek, seek regardless of file type and offset
 */

void LOAD::fseek(long ptr,short Whence)
{
if(Whence==SEEK_SET)
	::fseek(fileptr,Xorigin,SEEK_SET);
if(Whence==SEEK_END)
	::fseek(fileptr,Xorigin+Xlength,SEEK_SET);
::fseek(fileptr,ptr,SEEK_CUR);
}

/*
 *      exist, does it exist
 */

short exist(char *fname)
{
char name[128];
char string[128];

strcpy(name,fname);
strlwr(name);
slashes(name);

if(Gexist(name))               // Check in root directory first
    return 1;

strcpy(string,resdir);          // Check full path
strcat(string,"/");
strcat(string,fname);
strlwr(string);
return(Gexist(string));         // Go down to Gexist
}

/*
 *      Gexist, does it exist really or even in RFS?
 */

short Gexist(char *fname)
{
slashes(fname);
if(!access(fname,0))
	return 1;
if(RFS_Open(fname)!=-1)
	return 1;
return 0;
}


// 	Alternate functions for the 3 primary word sizes

/*
 *	getl - get 32 bits
 */

long getl(FILE *fp)
{
long l;
fread(&l,1,4,fp);
return l;
}

/*
 *	putl - put 32 bits
 */

void putl(long l, FILE *fp)
{
fwrite(&l,1,4,fp);
}



/*
 *	getsh - get 16 bits
 */


short getsh(FILE *fp)
{
short s;
fread(&s,1,2,fp);
return s;
}



/*
 *	putsh - put 16 bits
 */

void putl(short s, FILE *fp)
{
fwrite(&s,1,2,fp);
}


/*
 *	getb - get 8 bits
 */

char getb(FILE *fp)
{
char b;
fread(&b,1,1,fp);
return b;
}


/*
 *	putb - put 8 bits
 */

void putb(char b, FILE *fp)
{
fwrite(&b,1,1,fp);
}


// RFS specific routines

/*
 *	getstep - Calculate size and stepping
 */

static void getstep()
{
Rstep=-1;
do
  {
  Rstep++;
  Rnum = _Rfiles/(Rstep+1);
  } while(Rnum>60);
}

/*
 *      R_Count, get number of files in the RAR
 */

static void R_Count()
{
if(RFH.Method==0x30)
	_Rfiles++;
else
	{
	puts("\nUnknown method:You can only use the STORE compression method in patch files.");
	exit(1);
	}
fseek(_Rfp,RFH.NameSize,SEEK_CUR);
fseek(_Rfp,RFH.PackSize,SEEK_CUR);
}

/*
 *      R_Mount, open the RAR and find the files in it
 */

static void R_Mount()
{
#ifdef USE_RFS

RFS_nt[_Rcur]=(char *)malloc(RFH.NameSize+1);
if(!RFS_nt[_Rcur])
	{
	puts("\nOut of memory");
	exit(1);
	}
fread((char *)RFS_nt[_Rcur],1,RFH.NameSize,_Rfp);
RFS_nt[_Rcur][RFH.NameSize]=0;
RFStab[_Rcur].name=RFS_nt[_Rcur];

// Reverse slashes in filename

for(int ctr=0;ctr<RFH.NameSize;ctr++)
if(RFS_nt[_Rcur][ctr] == '\\')
    RFS_nt[_Rcur][ctr] = '/';

RFStab[_Rcur].length=RFH.PackSize;
RFStab[_Rcur].offset=ftell(_Rfp);
RFStab[_Rcur].rarname=_Rname;

if(Rctr++>=Rstep)
	{
	printf("%c",'.');
	Rctr=0;
	}

fseek(_Rfp,RFH.PackSize,SEEK_CUR);
_Rcur++;
#endif
}

/*
 *      RFS_Mount - Open a RARfile, stat it and mount it
 */


void RFS_Mount(short argc,char *argv[])
{
#ifdef USE_RFS
char str[128];
short ctr,ok;
_Rfiles=0;
_Rcur=0;

_RArgC=argc;
_RArgV=argv;

if(!argv[0])
	{
	puts("\nNo .RARfile was specified.");
	puts("This probably means someone's been screwing with the IRE.INI file.");
	puts("Make sure the line '-file MAINRAR.RAR' is present and correct.");
	exit(1);
	}

// First, traverse the RAR using the COUNT method to stat it

Traverse(R_Count);

if(!_Rfiles)       // If none, abort
    {
    fprintf(stderr,"No files found\n");
    return;
    }

Rnum=_Rfiles;
getstep();

// Got a sensible measure of files, draw a pretty meter

printf("  [");
for(ctr=0;ctr<Rnum;ctr++)
	printf("%c",' ');
printf("]\r  [");

// Alloc enough space for the RFS header table

RFStab=(RFS *)calloc(_Rfiles,sizeof(struct RFS));
if(!RFStab)
	{
	puts("\nOut o' memory");
	exit(1);
	}

// Alloc enough space for the name lookup

RFS_nt=(char **)calloc(_Rfiles,sizeof(char *));
if(!RFS_nt)
	{
	puts("\nOut o' memory");
	exit(1);
	}

// Now log the whole RAR with Mount, fill in the meter as we go

Traverse(R_Mount);
printf("]\n");

// Assume everything will work perfectly..

ok=1;

// List each RAR that we're mounting so the user can see if anything
// went wrong, like the W_init in Doom.
// Also, print warnings and pause if an error did occur

for(ctr=RFnum-1;ctr>=0;ctr--)
	{
	printf("  MOUNT: %s",_RArgV[ctr]);
	if(!Gexist(_RArgV[ctr]))
		{
		printf(" - Warning: Can't open\n");
		ok=0;
		}
	else
		{
/*		if(RAR_Open(_RArgV[ctr],resdir) == -1)
			{
			printf(" - Warning: No '%s' directory.. ignoring this .rar!\n",resdir);
			ok=0;
			}
		else*/
			printf(" - Success\n");
		}
	}

// If everything went well, ok is still 1.

if(!ok)
	{
	puts("\n  There are warnings.. Press any key to continue or ESC to quit.");
        if(SYS_KEY_WAIT() == 27)       // If esc is pressed, wuit
		exit(1);
	}

// Now, make sure we have a scriptfile.

strcpy(str,resdir);
strcat(str,"\\main.txt");
if(RFS_Open(str) == -1)
	if(access(str,0) == -1)
                {
                strcpy(str,"None of the RARfiles contained ");
                strcat(str,resdir);
                strcat(str,"\\main.txt");
		panic("RVFS_Mount()",str,"I could not find one in the current directory either.");
                }

// Finish.
#endif
return;
}

/*
 *      Traverse - Open a RARfile, get all the names, lengths etc
 *                   and fill out the meter as we go along
 */

static void Traverse(void (*Hfunc)())
{
#ifdef USE_RFS
struct MarkHeader RMH;
struct ArchiveHeader RAH;
short ctr;
for(ctr=0;ctr<_RArgC;ctr++)
	{
	_Rname=_RArgV[ctr];
	_Rfp=fopen(_Rname,"rb");
	if(!_Rfp)
		return;
	fread(&RMH,1,sizeof(RMH),_Rfp);	// read header block
	if(strcmp((char *)&RMH.Mark,(char *)"Rar!\x1a\x07"))
		{
		fclose(_Rfp);
                fprintf(stderr,"RAR: Invalid header\n");
		return;
		}
	fread(&RAH,1,sizeof(RAH),_Rfp);
	fseek(_Rfp,RMH.HeadSize-sizeof(RMH),SEEK_CUR);
	for(;fread(&RAH,1,7,_Rfp)==7;)
		{
		switch(RAH.HeadType)
			{
			case FILE_HEAD:
			fseek(_Rfp,-7L,SEEK_CUR);	// back 7 bytes
			fread(&RFH,1,sizeof(RFH),_Rfp);
			Hfunc();
			break;

			default:
			fseek(_Rfp,RAH.HeadSize-7,SEEK_CUR);
			if(RAH.Flags&0x8000)
				fseek(_Rfp,(long)RAH.Reserved,SEEK_CUR);
			}
		}
	}
fclose(_Rfp);
#endif
}

/*
 *      RFS_Open -  Open a file from the RFS
 */

long RFS_Open(char* fname)
{
int len;
#ifdef USE_RFS

if(!_Rfiles)    // If no RARs, don't open
    return -1;

// MR: VC doesn't handle defining a variable in a for statment more than once in a scope
// see ctr var below


// reverse slashes

len = strlen(fname);
// MR: put it in its own scope
{
for(int ctr=0;ctr<len;ctr++)
    if(fname[ctr] == '\\')
        fname[ctr] = '/';
}

// now opening in reverse order...
//for(long ctr=_Rfiles-1;ctr>0;ctr--)

// MR: put it in its own scope
{
for(long ctr=0;ctr<_Rfiles;ctr++)
	if(!stricmp(RFStab[ctr].name,fname))
		return ctr;
}

#endif
return -1;
}

/*
 *      RAR_Open -  Open a RARFILE
 */

long RAR_Open(char *rar, char* fname)
{
#ifdef USE_RFS
for(long ctr=_Rfiles-1;ctr>0;ctr--)
    if(!stricmp(RFStab[ctr].rarname,rar))
        if(!stricmp(RFStab[ctr].name,fname))
	    return ctr;
#endif
return -1;
}

/*
 *      RFS_Add -  Add a RAR file to the RFS list
 */

void RFS_Add(char *fn)
{
#ifdef USE_RFS
RARS[RFnum]=(char *)calloc(strlen(fn)+1,1);
if(!RARS[RFnum])
    panic("oscli()","Oops, out of memory.",NULL);
strcpy(RARS[RFnum],fn);
RFnum++;
return;
#endif
}

void Init_Filesystem()
{
printf("Init_Filesystem();\n");
#ifdef USE_RFS
RFS_Mount((short)RFnum,RARS);
#endif
}

void slashes(char *a)
{
int ctr,len;
len = strlen(a);

for(ctr=0;ctr<len;ctr++)
    if(a[ctr] == '\\')
        a[ctr]='/';
}

