/*
 *  - Objects menu tab
 */


#include <stdio.h>
#include "igui.hpp"
#include "../memory.hpp"
#include "../console.hpp"
#include "../core.hpp"
#include "../fs.hpp"
#include "../script.hpp"
#include "../init.hpp"
#include "../linklist.hpp"
#include "../object.hpp"
#include "../keys.hpp"
#include "menusys.h"

// DEFINES

#define POCKET_X 144
#define POCKET_Y 80
#define POCKET_H 34

// VARIABLES

static int sel_xoff=0;  // Offset from where we started the drag
static int sel_yoff=0;  // so that the dragging is centered on that spot
                        // i.e. we drag it by the point we first clicked on

extern int focus,OB_Id; // GUI variables

extern int MapXpos_Id,MapYpos_Id;       // Map X,Y coordinate boxes
extern char mapxstr[],mapystr[];

extern int l_proj;      // Flag, are layers being displayed?
extern int sx_proj;     // Flag, are other sprites being excluded? 1 = OFF
extern int mapx,mapy;   // Current map position
extern SPRITE transfer; // 256x256 Transfer buffer for screen clipping

extern OBJECT *objsel;  // Object that is currently selected.  Can be NULL

static char last_sprite[32];  // Type of the Last sprite that was created
static int  last_dir=0;
static OBJECT *pocsel=NULL;
int DirU_Id,DirD_Id,DirL_Id,DirR_Id;
int obj_xId,obj_yId,obj_name_Id,obj_pname_Id,obj_oname_Id;

static char tag_str[]="         ";
char obj_x_str[]="      ",obj_y_str[]="      ",obj_name_str[]="                ";
char obj_pname_str[]="                ";
char obj_oname_str[]="                ";
int tag_Id,offset=0,editpocket=0;

static char stats_hp_str[] =    "0          ";
static char stats_dex_str[] =   "0          ";
static char stats_str_str[] =   "0          ";
static char stats_intel_str[] = "0          ";
static char stats_weight_str[] ="0          ";
static char stats_damage_str[] ="0          ";
static char stats_quantity_str[] ="0          ";
static int stats_hp_Id;
static int stats_dex_Id;
static int stats_str_Id;
static int stats_intel_Id;
static int stats_weight_Id;
static int stats_damage_Id;
static int stats_quantity_Id;
static char none[]="-";

static char str32[33];
static char behave_str[33];
static int behave_Id,use_Id,kill_Id,look_Id,stand_Id,hurt_Id,init_Id,talk_Id;
static char edit_mode = 0;

// FUNCTIONS

extern void pause();
extern void Toolbar();
extern void DrawMap(int x,int y,char s_proj,char l_proj,char f_proj);
extern void gen_largemap();
static void Nothing();
static void clipmap();  // Make sure coordinates are within bounds
static void Deal_With_Contents(OBJECT *obj);

void OB_up();           // Panning functions
void OB_down();
void OB_left();
void OB_right();

void OB_QuickInsert();  // For fast replication of known objects
void OB_SlowInsert();   // For choosing and object before creating it
void OB_Edit();         // Choose the type
void OB_Delete();       // Delete it
void OB_Pick();         // Left click on the map to pick one up
void OB_MoveThere();    // Right click on the map to move it faster

void OB_DirU();         // Point the object up
void OB_DirD();         // Point the object down
void OB_DirL();         // Point the object left
void OB_DirR();         // Point the object right

void OB_Dir(int dir);   // Set the current direction, helper

void OB_SetTag();       // Set Tag
void OB_FindTag();      // Find an object with a certain Tag
void OB_FreeTag();      // Get next free tag

void SelectObj(int cx,int cy);  // Do the actual picking up
void OB_Update();             // Update the direction display
static void OB_SetPersonalName();

static void GetMapX();                  // Get map position
static void GetMapY();
static void GetObjX();                  // Get object position
static void GetObjY();
static void OB_debug();
static void OB_Schedule();
static void inSchedule(int x, int y, int lines);
static void OB_SetSchedule();
static void SetSched(int pos);
static void ResetObject();

static void OB_Pocket();
static void inPocket(int x, int y, int lines);
static void PO_Next();
static void PO_Prev();
static void PO_Add();
static void PO_Del();
static void PO_Highlight();
static void PO_In();
static void PO_Out();

extern void Snap();

static void OB_Stats();
static void Stats_Update();
static void SetHp();
static void SetDex();
static void SetStr();
static void SetIntel();
static void SetWeight();
static void SetDamage();
static void SetQuantity();
static void Finish();
static void Resettt();

static void EditBehaviour();
static void Behave_Update();
static void SetBehave();
static void SetUse();
static void SetKill();
static void SetLook();
static void SetStand();
static void SetHurt();
static void SetInit();
static void SetTalk();
static int PickVrm(char *original);

static void SetOwner();
static void OBO_MakePublic();
static void OBO_FromList();
static void OBO_MakeOwner();


// Parts of the OB_Edit browser

void GetOBFromList( int x0, int y0, char *prompt, int listsize, char **list, char *name);
void OB_Thumb(int x,int y,char *name);

// CODE

/*
 * GoFocal - The menu tab function.  This is called from the toolbar
 */

void OB_GoFocal()
{
int temp;               // Used to get button handles

if(focus==2)            // Make sure we don't redraw if the user clicks again
	return;

focus=2;                // This is now the focus
Toolbar();              // Build up the menu system again from scratch
IG_SetFocus(OB_Id);     // And make the button stick inwards

edit_mode = 0;

for(temp=0;temp<32;temp++)
    str32[temp]=' ';
str32[32]=0;
// Set up Map window

IG_BlackPanel(VIEWX-1,VIEWY-1,258,258);
IG_Region(VIEWX,VIEWY,256,256,OB_Pick,NULL,OB_MoveThere);

// Write sprite drawing status

temp = IG_ToggleButton(486,72,"Exclude ON ",Nothing,NULL,NULL,&sx_proj);
IG_SetInText(temp,"Exclude OFF");

temp = IG_ToggleButton(486,100,"Rooftops OFF",Nothing,NULL,NULL,&l_proj);
IG_SetInText(temp,"Rooftops ON ");

// Set up the X and Y map position counter at the top

MapXpos_Id = IG_InputButton(24,40,mapxstr,GetMapX,NULL,NULL);
MapYpos_Id = IG_InputButton(112,40,mapystr,GetMapY,NULL,NULL);

DrawMap(mapx,mapy,1,l_proj,0);                       // Draw the map

IG_TextButton(416,104,__up,OB_up,NULL,NULL);         // Pan up button
IG_TextButton(400,128,__left,OB_left,NULL,NULL);     // Pan left button
IG_TextButton(432,128,__right,OB_right,NULL,NULL);   // Pan right button
IG_TextButton(416,150,__down,OB_down,NULL,NULL);     // Pan down button

IG_AddKey(UP_KEY,OB_up);                             // Pan up key binding
IG_AddKey(DN_KEY,OB_down);                           // Pan down key binding
IG_AddKey(LF_KEY,OB_left);                           // Pan left key binding
IG_AddKey(RT_KEY,OB_right);                          // Pan right key binding

// Create objects

IG_TextButton(300,400,"New Item",OB_SlowInsert,NULL,NULL);
IG_AddKey(INS,OB_QuickInsert);

// Modify objects

IG_TextButton(400,400,"Edit Item",OB_Edit,NULL,NULL);
IG_AddKey(ENTER,OB_Edit);

// Delete objects

IG_TextButton(510,400,"Delete Item",OB_Delete,NULL,NULL);
IG_AddKey(DEL,OB_Delete);

// Object direction facings

DirU_Id = IG_Tab(598,282,__up,OB_DirU,NULL,NULL);
DirD_Id = IG_Tab(598,314,__down,OB_DirD,NULL,NULL);
DirL_Id = IG_Tab(580,297,__left,OB_DirL,NULL,NULL);
DirR_Id = IG_Tab(616,297,__right,OB_DirR,NULL,NULL);
IG_Text(492,304,"DIRECTION:",BLACK);


//IG_Text(128,358,"Tag:",BLACK);
//tag_Id = IG_InputButton(160,350,tag_str,OB_GetTag,NULL,NULL);

IG_TextButton(300,432,"Find tag",OB_FindTag,NULL,NULL);
IG_Text(388,440,"TAG:",BLACK);
tag_Id = IG_InputButton(420,432,tag_str,OB_SetTag,NULL,NULL);
IG_TextButton(510,432,"Next free tag",OB_FreeTag,NULL,NULL);

IG_Text(300,304,"X:",BLACK);
obj_xId = IG_InputButton(324,296,obj_x_str,GetObjX,NULL,NULL);
IG_Text(400,304,"Y:",BLACK);
obj_yId = IG_InputButton(424,296,obj_y_str,GetObjY,NULL,NULL);

IG_Text(300,264,"Object type:",BLACK);
obj_name_Id = IG_InputButton(400,256,obj_name_str,OB_Edit,NULL,NULL);

IG_TextButton(300,368,"Pockets",OB_Pocket,NULL,NULL);
IG_TextButton(372,368,"Statistics",OB_Stats,NULL,NULL);
IG_TextButton(468,368,"Behaviour",EditBehaviour,NULL,NULL);
IG_TextButton(556,368,"RESET",ResetObject,NULL,NULL);
//IG_TextButton(364,368,"Edit Schedule",OB_Schedule,NULL,NULL);

//obj_pname_Id=IG_InputButton(32,432,obj_pname_str,OB_SetPersonalName,NULL,NULL);
//IG_Text(32,420,"Individual name:",BLACK);
obj_pname_Id=IG_InputButton(32,360,obj_pname_str,OB_SetPersonalName,NULL,NULL);
IG_Text(32,348,"Individual name:",BLACK);
obj_oname_Id=IG_InputButton(32,400,obj_oname_str,SetOwner,NULL,OBO_FromList);
IG_Text(32,388,"Object is property of:",BLACK);

// Debugging: perform consistency check

IG_AddKey(KEY_F9,OB_debug);

// Set state variables

pocsel=NULL;
editpocket=0;

OB_Update();

pause();
pause();
}

/*
 *   Nothing - Just draw the map.  Used by toggle-buttons
 */

void Nothing()
{
DrawMap(mapx,mapy,1,l_proj,0);     // Do something anyway
return;
}

/*
 *   OB_up - Scroll upwards
 */

void OB_up()
{
mapy--;
clipmap();
DrawMap(mapx,mapy,1,l_proj,0);
}

/*
 *   OB_down - Scroll downwards
 */

void OB_down()
{
mapy++;
clipmap();
DrawMap(mapx,mapy,1,l_proj,0);
}

/*
 *   OB_left - Scroll to the left
 */

void OB_left()
{
mapx--;
clipmap();
DrawMap(mapx,mapy,1,l_proj,0);
}

/*
 *   OB_right - Scroll to the right
 */

void OB_right()
{
mapx++;
clipmap();
DrawMap(mapx,mapy,1,l_proj,0);
}

/*
 *   OB_SlowInsert - Create an object and ask the user what it should be like
 */

void OB_SlowInsert()
{
objsel=OB_Alloc();              // Allocate the object
objsel->x=mapx+(VSW/2);         // Set reasonable defaults
objsel->y=mapy+(VSH/2);
MoveToMap(objsel->x,objsel->y,objsel);  // Let there be Consistency

sel_xoff = 0;                   // If we drag and drop, no grab offset
sel_yoff = 0;

objsel->name = CHlist[0].name;             // Give it a name
objsel->form = &SQlist[CHlist[0].dir[0]];  // And a shape

OB_Edit();              // Do this to set everything up

if(objsel->flags.quantity)
    objsel->stats->quantity=1;          // Make sure there is at least one
}

/*
 *   OB_QuickInsert - Create an object but assume is is the same as the last
 *                    (kind of drive-thru object creation)
 */

void OB_QuickInsert()
{
int click_x,click_y,ctr;

if(getnum4char(last_sprite) == -1)     // Is it valid?
    {
    OB_SlowInsert();                    // No, do it the slow way
    return;
    }

// First do a quick bounds check, to prevent the creation of sprites
// where the user can't actually see.

click_x = ((x-VIEWX)>>5);       // Work out the position of the mouse
click_y = ((y-VIEWY)>>5);       // in tiles

if(click_x<0 || click_y<0)      // out the top/left of the screen?
    return;                     // Yes, abort
if(click_x>VSW-1 || click_y>VSH-1)  // Out the bottom/right of the screen?
    return;                     // Yes, abort

// Now do the full calculation.

click_x = ((x-VIEWX)>>5)+(mapx); // click_x = (mouse_x_pos/32) + map_x_pos
click_y = ((y-VIEWY)>>5)+(mapy); // click_y = (mouse_y_pos/32) + map_y_pos

objsel=OB_Alloc();              // Allocate the object

objsel->x=click_x;              // Use the current position
objsel->y=click_y;
MoveToMap(objsel->x,objsel->y,objsel);  // Let there be Consistency

if(objsel->flags.quantity)
    objsel->stats->quantity=1;          // Make sure there is at least one

sel_xoff = 0;                   // If we drag and drop, no grab offset
sel_yoff = 0;

OB_Init(objsel,last_sprite);    // Re-evaluate the character
Deal_With_Contents(objsel);     // Create/destroy any contents
OB_Dir(last_dir);               // Set the direction

OB_Update();                    // Sort out the direction display

DrawMap(mapx,mapy,1,l_proj,0);
}

/*
 *   OB_Edit - Ask the user what an existing object should be like
 *             OB_SlowInsert calls this as its GUI.
 *             OB_Edit calls down to GetOBFromList to display the browser
 */

void OB_Edit()
{
char **list;
char Name[32];

// Make sure we have an object

if(!objsel)
	{
	Notify(-1,-1,"No sprite has been selected.","You must pick a sprite before you can modify it.");
	return;
	}

// Make sure it's not a silly object

if(objsel==curmap.object)
	{
	Notify(-1,-1,"Not permitted.",NULL);
	return;
	}

// Allocate a list of possible characters

list=(char **)M_get(CHtot+1,sizeof(char *));

// Set it up, convert to uppercase for Wyber's ordered selection routine.

for(int ctr=0;ctr<CHtot;ctr++)
        {
	list[ctr]=CHlist[ctr].name;
        strupr(list[ctr]);              // Affects the original too
        }

strcpy(Name,objsel->name);      // Set up the default string

qsort(list,CHtot,sizeof(char*),CMP);  // Sort them for the dialog box

// This is the graphical dialog box, taken from DEU.

GetOBFromList( -1,-1, "Choose a character:", CHtot-1, list, Name);

// If the user didn't press ESC instead of choosing, modify the character

if(Name[0])     // It's ok, do it
	{
        OB_Init(objsel,Name);           // Re-evaluate the character
        Deal_With_Contents(objsel);     // Create/destroy any contents
        strcpy(last_sprite,Name);       // Keep this name for reference
	}

free(list);                             // Dispose of the list

if(!editpocket)
    {
    OB_Update();                          // Sort out the direction display
    DrawMap(mapx,mapy,1,l_proj,0);          // Update the map
    }
}

/*
 *   OB_Delete - Remove an object
 */

void OB_Delete()
{
if(!objsel)
	{
	Notify(-1,-1,"No sprite has been selected.","You must pick a sprite before you can delete it.");
	return;
	}

if(objsel==curmap.object)                       // This can't happen
	{
	Notify(-1,-1,"Not Permitted",NULL);     // Get shirty
	return;
	}

DestroyObject(objsel);                          // Destroy the object
objsel=NULL;                                    // Nothing currently selected
DrawMap(mapx,mapy,1,l_proj,0);                  // Draw the map again

sel_xoff = 0;                   // If we drag and drop, no grab offset
sel_yoff = 0;
}

/*
 *   OB_Pick - Drag and drop an object.  This calls down to SelectObj();
 */

void OB_Pick()
{
int click_x,click_y,nx,ny;

click_x = ((x-VIEWX)>>5)+(mapx); // click_x = (mouse_x_pos/32) + map_x_pos
click_y = ((y-VIEWY)>>5)+(mapy); // click_y = (mouse_y_pos/32) + map_y_pos

if(dragflag)                    // If we're dragging an object
	{
        if(!objsel)
            {
            Notify(-1,-1,"Internal error in drag'n'drop code",NULL);
            return;
            }

	nx=click_x + sel_xoff;    // Get clicked position
	ny=click_y + sel_yoff;

	if(nx<0) nx=0;     // Sanity checks.
	if(ny<0) ny=0;

        MoveObject(objsel,nx,ny);       // Move it.

	dragflag=2;                      // Reset drag timer
	}
else                                     // If we're not dragging, or time out
	{
        SelectObj(click_x,click_y);      // Go fish

	if(objsel)                       // If an object was selected
            {
	    dragflag=2;                        // We go into drag mode
            strcpy(last_sprite,objsel->name);  // We'll make more of these
            last_dir = objsel->curdir;

            objsel->form=NULL; // Force direction to change
            OB_SetDir(objsel,objsel->curdir);
            }
	}

DrawMap(mapx,mapy,1,l_proj,0);           // Draw the map again
OB_Update();
}

/*
 *   OB_MoveThere - Move an object to the current square without dragging it
 */

void OB_MoveThere()
{
int click_x,click_y,nx,ny;

click_x = ((x-VIEWX)>>5)+(mapx); // click_x = (mouse_x_pos/32) + map_x_pos
click_y = ((y-VIEWY)>>5)+(mapy); // click_y = (mouse_y_pos/32) + map_y_pos

if(objsel)
    {
    nx=click_x + sel_xoff;    // Get clicked position
    ny=click_y + sel_yoff;

    if(nx<0) nx=0;     // Sanity checks.
    if(ny<0) ny=0;

    gen_largemap();
    MoveObject(objsel,nx,ny);       // Move it.
    DrawMap(mapx,mapy,1,l_proj,0);
    OB_Update();
    }
}

/*
 *   OB_SetTag - Set the TAG Id for an object
 *

void OB_SetTag()
{
int num;

if(!objsel)     // Abort if no object selected
    return;

num=InputIntegerValue(-1,-1,0,65535,objsel->tag);
objsel->tag = num;
OB_Update();

}
*/

/*
 *   OB_FindTag - Find an object with a certain tag
 */

void OB_FindTag()
{
OBJECT *temp;
int tag=0;

// Get the tag to search for

tag=InputIntegerValue(-1,-1,0,65535,tag);

temp = LookInAllPocketsForTag(tag);

// Scan the linked list for that tag

if(!temp)
    {
    Notify(-1,-1,"Could not find an object using that tag number.",NULL);
    return;
    }

if(InPocket(temp))
    {
    Notify(-1,-1,"That item is inside a container.",NULL);
    objsel = LookInAllPocketsFor(temp);
    }
else
    objsel = temp;               // We've found one

mapx=objsel->x-(VSW/2);      // center the object in the view
mapy=objsel->y-(VSH/2);

clipmap();                     // Clip the view to sensible bounds
OB_Update();                   // Update tag number
DrawMap(mapx,mapy,1,l_proj,0); // Project
return;                        // Leave
}


/*
 *   OB_FreeTag - Find first free tag
 */

void OB_FreeTag()
{
OBJECT *temp;
char stemp[32];

// Take a number

for(int ctr=0;ctr<65535;ctr++)
    {
    temp = LookInAllPocketsForTag(ctr);
    if(!temp)
            {
            sprintf(stemp,"tag: %d",ctr);
            Notify(-1,-1,"The next free tag is:",stemp);
            return;
            }

    // If not we keep going round and round until we find one
    }

// Proclaim the impossible

Notify(-1,-1,"Probable System Fault","There are over 65535 tags.  Are they really all used?");
}

/*
 *   OB_SetTag - Set the tag
 */

void OB_SetTag()
{
if(!objsel)
    {
    Notify(-1,-1,"You need to select an object to do this",NULL);
    return;
    }

// Get the new tag number

objsel->tag=InputIntegerValue(-1,-1,0,65535,objsel->tag);

// Update the display

OB_Update();
DrawMap(mapx,mapy,1,l_proj,0);
}

void ResetObject()
{
if(!objsel)
    return;

if(!Confirm(-1,-1,"This will destroy any individual settings for this object.","Do you wish to do this?"))
    return;

OB_Init(objsel,objsel->name);
OB_Update();
DrawMap(mapx,mapy,1,l_proj,0);
}

/* =========================== Auxiliary functions ======================== */

/*
 *   SelectObj - fish for an object on the map where the user has clicked
 */

void SelectObj(int cx,int cy)
{
objsel = NULL;          // Assume nothing was there
sel_xoff = 0;           // No select offset
sel_yoff = 0;

gen_largemap();
objsel = GetObject(cx,cy);      // Go fishing

if(objsel)                      // Caught a fish
    {
    if(objsel->flags.large)             // If it's a big fish, find out
        {                               // what part of the object was
        sel_xoff = objsel->x-cx;        // grabbed, for drag'n'drop purposes
        sel_yoff = objsel->y-cy;
        }
    }

OB_Update();                    // Sort out the direction display
//DrawMap(mapx,mapy,1,l_proj,0);  // Redraw the map with the changes

// Time to go
return;
}

/*
 *    OB_Update - update direction arrows and that stuff
 */

void OB_Update()
{
char temp[13];

if(edit_mode == 1)
    return;

IG_ResetFocus(DirU_Id); // Pull all the buttons out by default
IG_ResetFocus(DirD_Id);
IG_ResetFocus(DirL_Id);
IG_ResetFocus(DirR_Id);

if(!objsel)             // Abort, because no sprite was chosen
    return;

// Now find the right button to push in

switch(objsel->curdir)
    {
    case CHAR_U:
    IG_SetFocus(DirU_Id); // Up is stuck in
    break;

    case CHAR_D:
    IG_SetFocus(DirD_Id); // Down
    break;

    case CHAR_L:
    IG_SetFocus(DirL_Id); // Left
    break;

    case CHAR_R:
    IG_SetFocus(DirR_Id); // Right
    break;
    };

sprintf(tag_str,"%-9d",objsel->tag);
IG_UpdateText(tag_Id,tag_str);

sprintf(obj_x_str,"%-6d",objsel->x);
IG_UpdateText(obj_xId,obj_x_str);

sprintf(obj_y_str,"%-6d",objsel->y);
IG_UpdateText(obj_yId,obj_y_str);

sprintf(obj_name_str,"%-16s",objsel->name);
IG_UpdateText(obj_name_Id,obj_name_str);

sprintf(obj_pname_str,"%-16s",objsel->personalname);
IG_UpdateText(obj_pname_Id,obj_pname_str);

if(!objsel->stats->owner)
    strcpy(obj_oname_str,"everyone        ");
else
    if(!objsel->stats->owner->personalname)
       strcpy(obj_oname_str,"nameless owner! ");
    else
       sprintf(obj_oname_str,"%-16s",objsel->stats->owner->personalname);
IG_UpdateText(obj_oname_Id,obj_oname_str);
}

/*
 *   clipmap - clip the map's coordinates to sensible values
 */

void clipmap()
{
if(mapx<0) mapx=0;
if(mapx>curmap.w-VSW) mapx=curmap.w-VSW;
if(mapy>curmap.h-VSH) mapy=curmap.h-VSH;
if(mapy<0) mapy=0;
}

/*
 *   GetOBFromList - show thumbnails of the sprites you can choose from.
 *                   Called by OB_Edit, calls down to OB_Thumb
 */

void GetOBFromList( int x0, int y0, char *prompt, int listsize, char **list, char *name)
{
InputNameFromListWithFunc( x0, y0, prompt, listsize, list, 5, name, 257, 257, OB_Thumb);
}

/*
 *   OB_Thumb - show thumbnails of the sprites you can choose from.
 *              Called by GetOBFromList
 */

void OB_Thumb(int x,int y,char *name)
{
int num;
fbox2(x,y,256,256,21,swapscreen);

num = getnum4char(name);
if(num != -1)
        {
        fbox2(x,y,256,256,21,bg_screen);
	SQlist[CHlist[num].dir[0]].seq[0]->image.cel_put_sprite(x,y,bg_screen);
        transfer.get_sprite(x,y,bg_screen);
        transfer.block_put_sprite(x,y,swapscreen);
        }
else
	IG_Text(x+4,y+32,"Not found",WHITE);
}

/*
 * OB_Dir - code to set up the object's direction, from the four defaults
 *          (up, down, left, right)
 */

void OB_Dir(int d)
{
IG_ResetFocus(DirU_Id); // All of them are stuck out,
IG_ResetFocus(DirD_Id); // but after we return the appropriate function
IG_ResetFocus(DirL_Id); // will make the right one sink in
IG_ResetFocus(DirR_Id); // If no was sprite selected, this is what we want

if(!objsel)             // Safety valve
    return;

if(objsel->curdir == d) // Are we already pointing upwards?
    return;             // If so, abort

OB_SetDir(objsel,d);    // Set the direction, reset visual form
last_dir = objsel->curdir;

DrawMap(mapx,mapy,1,l_proj,0);  // Redraw the map with the changes
}

/*
 *    OB_DirU - Make the object face Upwards
 */

void OB_DirU()
{
OB_Dir(CHAR_U);
if(!objsel)
    {
    Notify(-1,-1,"No sprite has been selected.","You must pick a sprite before you can modify it.");
    return;
    }
IG_SetFocus(DirU_Id);
}

/*
 *    OB_DirD - Make the object face Down
 */

void OB_DirD()
{
OB_Dir(CHAR_D);
if(!objsel)
    {
    Notify(-1,-1,"No sprite has been selected.","You must pick a sprite before you can modify it.");
    return;
    }
IG_SetFocus(DirD_Id);
}

/*
 *    OB_DirL - Make the object face Left
 */

void OB_DirL()
{
OB_Dir(CHAR_L);
if(!objsel)
    {
    Notify(-1,-1,"No sprite has been selected.","You must pick a sprite before you can modify it.");
    return;
    }
IG_SetFocus(DirL_Id);
}

/*
 *    OB_DirR - Make the object face Right
 */

void OB_DirR()
{
OB_Dir(CHAR_R);
if(!objsel)
    {
    Notify(-1,-1,"No sprite has been selected.","You must pick a sprite before you can modify it.");
    return;
    }
IG_SetFocus(DirR_Id);
}

/*
 *    GetMapX - Ask the user where on the map they want to be
 */

void GetMapX()
{
mapx=InputIntegerValue(-1,-1,0,curmap.w-VSW,mapx);
DrawMap(mapx,mapy,1,l_proj,0);
}

/*
 *    GetMapY - Ask the user where on the map they want to be
 */

void GetMapY()
{
mapy=InputIntegerValue(-1,-1,0,curmap.h-VSH,mapy);
DrawMap(mapx,mapy,1,l_proj,0);
}

/*
 *    GetObjX - Ask the user where on the map they want to be
 */

void GetObjX()
{
int newx;

if(objsel)
    {
    newx=InputIntegerValue(-1,-1,0,curmap.w-VSW,objsel->x);
    MoveObject(objsel,newx,objsel->y);
    }
DrawMap(mapx,mapy,1,l_proj,0);
OB_Update();
}

/*
 *    GetObjY - Ask the user where on the map they want to be
 */

void GetObjY()
{
int newy;
if(objsel)
    {
    newy=InputIntegerValue(-1,-1,0,curmap.h-VSH,objsel->y);
    MoveObject(objsel,objsel->x,newy);
    }
DrawMap(mapx,mapy,1,l_proj,0);
OB_Update();
}

/*
 *    OB_debug - Check map consistency
 */

void OB_debug()
{
char good=1;
int vx,vy,yoff;
OBJECT *temp;
int objects=0,large=0;
char string1[128];
char string2[128];

for(vy=0;vy<curmap.h;vy++)
    {
    yoff=ytab[vy];
    for(vx=0;vx<curmap.w;vx++)
        {
        for(temp=curmap.objmap[yoff+vx];temp;temp=temp->next)
            if(temp)                            // Safety check
                {
                objects++;
                if(temp->x != vx)
                    good=0;
                if(temp->y != vy)
                    good=0;
                if(temp->flags.large)
                    large++;
                }
        }
    }
if(good)
    Notify(-1,-1,"Map OK.","All object locations consistent.");
else
    Notify(-1,-1,"Map Consistency Failed.","Some object locations inconsistent.");

if(curmap.object->next)
    Notify(-1,-1,"Unlinked objects detected",NULL);

sprintf(string1,"There are %d objects\n",objects);
sprintf(string2,"Of these %d are large objects\n",large);
Notify(-1,-1,string1,string2);
}



void Deal_With_Contents(OBJECT *obj)
{
OBJECT *temp;
if(!obj->pocket)
    {
    for(int ictr=0;ictr<objsel->funcs->contents;ictr++)
        {
        if(objsel->funcs->contains[ictr])
            {
            temp = OB_Alloc();
            temp->curdir = temp->dir[0];
            OB_Init(temp,objsel->funcs->contains[ictr]);
            OB_SetDir(temp,1);
            MoveToPocket(temp,objsel);
            }
        }
    }
else
    FreePockets(obj);
}

void OB_Pocket()
{
if(!objsel)
    {
    Notify(-1,-1,"No sprite has been selected.","You must pick a sprite before you can modify it.");
    return;
    }

focus=0;
IG_KillAll();

DrawScreenBox3D(128,64,512,416);

inPocket(POCKET_X,POCKET_Y,POCKET_H);
IG_Region(POCKET_X,POCKET_Y,128,288,PO_Highlight,NULL,NULL);

IG_TextButton(304,80,__up,PO_Prev,NULL,NULL);
IG_TextButton(304,344,__down,PO_Next,NULL,NULL);

IG_TextButton(384,80, "Create object",PO_Add,NULL,NULL);
IG_TextButton(384,112,"Remove object",PO_Del,NULL,NULL);
IG_TextButton(384,144,"Move outside ",PO_Out,NULL,NULL);
IG_TextButton(384,176,"Bring inside ",PO_In,NULL,NULL);
IG_TextButton(384,384,"Finish",OB_GoFocal,NULL,NULL);

IG_AddKey(0x1b,OB_GoFocal);     // ESC to quit this menu
IG_AddKey(0x5d,Snap);           // Add key binding for shift-F10

editpocket=1;
pocsel=NULL;
}

void PO_Prev()
{
offset--;
if(offset<0)
    offset=0;
inPocket(POCKET_X,POCKET_Y,POCKET_H);
}

void PO_Next()
{
offset++;
inPocket(POCKET_X,POCKET_Y,POCKET_H);
}

void PO_Add()
{
OBJECT *temp,*ooe;

if(!objsel)
    {
    Notify(-1,-1,"System error","This should not be able to happen");
    return;
    }

temp=OB_Alloc();              // Allocate the object

MoveToPocket(temp,objsel);

temp->name = CHlist[0].name;             // Give it a name
temp->form = &SQlist[CHlist[0].dir[0]];  // And a shape

// Because OB_Edit is designed to be a callback, with no parameters,
// it uses the global OBJSEL.  This time we want to call it with TEMP,
// so we need to do some shuffling.

ooe = objsel;
objsel = temp;

OB_Edit();              // Do this to set everything up

objsel = ooe;           // Restore objsel to normal

temp->flags.seekstate=temp->flags.fixed;
temp->flags.fixed=1;

if(temp->flags.quantity)
    temp->stats->quantity=1;          // Make sure there is at least one

inPocket(POCKET_X,POCKET_Y,POCKET_H);
}

void PO_Del()
{
OBJECT *temp;

if(!pocsel)
    {
    Notify(-1,-1,"No object has been selected","First highlight the object you want to remove.");
    return;
    }

if(!objsel)
    {
    Notify(-1,-1,"System error","This should not be able to happen");
    return;
    }

LL_Remove(&objsel->pocket,pocsel);
pocsel=NULL;
inPocket(POCKET_X,POCKET_Y,POCKET_H);
delay(50);
}

void PO_Out()
{
if(!objsel)
    {
    Notify(-1,-1,"System error","This should not be able to happen");
    return;
    }
if(!pocsel)
    {
    Notify(-1,-1,"No object has been selected","First highlight the object you want to leave the container.");
    return;
    }
ForceFromPocket(pocsel,objsel,objsel->x,objsel->y);
pocsel=NULL;
inPocket(POCKET_X,POCKET_Y,POCKET_H);
delay(50);
}

void PO_In()
{
OBJECT *temp;
if(!objsel)
    {
    Notify(-1,-1,"System error","This should not be able to happen");
    return;
    }

if(!Confirm(-1,-1,"This will get the object UNDERNEATH the container.","Do you wish to proceed?"))
    return;

temp = GetRawObjectBase(objsel->x,objsel->y);

if(temp == objsel)
    {
    Notify(-1,-1,"There are no objects beneath the container to pick up!",NULL);
    return;
    }

for(;temp->next;temp=temp->next)
    if(temp->next == objsel)
        {
        MoveToPocket(temp,objsel);
        pocsel=NULL;
        inPocket(POCKET_X,POCKET_Y,POCKET_H);
        delay(50);
        return;
        }

Notify(-1,-1,"Something went wrong.",NULL);
}

void inPocket(int x, int y, int lines)
{
OBJECT *temp;
int pos=0,line=0;

DrawScreenBoxHollow(x,y,x+(16*8)+16,y+(lines*8)+16);

for(temp=objsel->pocket;temp;temp=temp->next)
    {
    if(pos>=offset && line<lines)
        if(temp)
            {
            IG_Text(x+8,y+8,temp->name,WHITE);
            if(temp == pocsel)
                IG_Text(x+8,y+8,temp->name,RED);
            y+=8;
            line++;
            }
    pos++;
    }

if(line == 0 && offset == 0)
    IG_Text(x+8,y+8,"EMPTY",WHITE);
}


void PO_Highlight()
{
OBJECT *temp;
int pos=0,line=0,lines=0;
int sx,sy,cy;

sx = POCKET_X;
sy = POCKET_Y;
lines = POCKET_H;
pocsel=NULL;

cy = y-8;

for(temp=objsel->pocket;temp;temp=temp->next)
    {
    if(pos>=offset && line<lines)
        if(temp)
            {
            if(cy>=sy && cy <=(sy+8))
                pocsel=temp;
            sy+=8;
            line++;
            }
    pos++;
    }

inPocket(POCKET_X,POCKET_Y,POCKET_H);
}

void OB_Schedule()
{
if(!objsel)
    {
    Notify(-1,-1,"No sprite has been selected.","You must pick a sprite before you can modify it.");
    return;
    }

focus=0;
IG_KillAll();

DrawScreenBox3D(128,64,512,416);

inSchedule(POCKET_X,POCKET_Y,24);
IG_Region(POCKET_X,POCKET_Y,192,288,OB_SetSchedule,NULL,NULL);
IG_TextButton(384,POCKET_Y+32,"Finish",OB_GoFocal,NULL,NULL);

IG_AddKey(0x1b,OB_GoFocal);     // ESC to quit this menu
IG_AddKey(0x5d,Snap);           // Add key binding for shift-F10

editpocket=1;
}

void inSchedule(int x, int y, int lines)
{
char str[32];
int ctr;

return;

if(!objsel)
    return;

DrawScreenBoxHollow(x,y,x+(24*8)+16,y+(lines*12)+16);

for(ctr=0;ctr<24;ctr++)
    {
    sprintf(str,"%2d:00",ctr);
    IG_Text(x+8,y+8,str,WHITE);

    if(!objsel->schedule[ctr])
        IG_Text(x+56,y+8,"-",WHITE);
    else
        IG_Text(x+56,y+8,objsel->schedule[ctr],WHITE);
    y+=12;
    }
}

void OB_SetSchedule()
{
int ctr;
int sx=POCKET_X+8;
int sy=POCKET_Y+8;

for(ctr=0;ctr<24;ctr++)
    {
    if(x>sx && x<sx+192)
        if(y>sy && y<(sy+12))
            {
            SetSched(ctr);
            inSchedule(POCKET_X,POCKET_Y,24);
            return;
            }
    sy+=12;
    }
}

void SetSched(int pos)
{
#if 0
char **list;
char Name[32];
int tot,cur,ctr;

tot=1;

list=(char **)M_get(tot,sizeof(char *));

// Set it up, convert to uppercase for Wyber's ordered selection routine

cur=0;
list[cur++]="-";
for(ctr=0;ctr<MAX_PATHS;ctr++)
        if(PAlist[ctr])
            {
            list[cur]=PAlist[ctr]->name;
            strupr(list[cur++]);              // Affects the original too
            }

if(objsel->schedule[pos])
    strcpy(Name,objsel->schedule[pos]);      // Set up the default string
else
    strcpy(Name,list[0]);

qsort(list,tot,sizeof(char*),CMP);  // Sort them for the dialog box

// This is the graphical dialog box, taken from DEU.

InputNameFromList( -1,-1, "Choose a Movement Path:", tot-1, list, Name);

// If the user didn't press ESC instead of choosing, modify the character

if(Name[0])     // It's ok, do it
    {
    if(!objsel->schedule[pos])
        objsel->schedule[pos]=M_get(1,16);
    strcpy(objsel->schedule[pos],Name);
    }

free(list);                             // Dispose of the list
#endif
}

/*
 *
 *      EDIT STATISTICS
 *
 */

void OB_Stats()
{
if(!objsel)
    return;
focus=0;
IG_KillAll();

DrawScreenBox3D(32,128,608,320);

setcolor(WHITE);
DrawScreenText(48,160,"Health");
stats_hp_Id =    IG_InputButton(128,152,stats_hp_str,SetHp,NULL,NULL);
DrawScreenText(48,192,"Dexterity");
stats_dex_Id =   IG_InputButton(128,184,stats_dex_str,SetDex,NULL,NULL);
DrawScreenText(48,224,"Strength");
stats_str_Id =   IG_InputButton(128,216,stats_str_str,SetStr,NULL,NULL);
DrawScreenText(48,256,"Intellig.");
stats_intel_Id = IG_InputButton(128,248,stats_intel_str,SetIntel,NULL,NULL);

DrawScreenText(232,160,"Weight");
stats_weight_Id = IG_InputButton(320,152,stats_weight_str,SetWeight,NULL,NULL);
DrawScreenText(232,192,"Damage");
stats_damage_Id = IG_InputButton(320,184,stats_damage_str,SetDamage,NULL,NULL);
DrawScreenText(232,224,"Quantity");
stats_quantity_Id = IG_InputButton(320,216,stats_quantity_str,SetQuantity,NULL,NULL);


IG_TextButton(512,152,"Finish",OB_GoFocal,NULL,NULL);
IG_AddKey(0x1b,OB_GoFocal);     // ESC to quit this menu
IG_AddKey(0x5d,Snap);           // Add key binding for shift-F10

Stats_Update();
}

void Stats_Update()
{
sprintf(stats_hp_str,"%-10d",objsel->stats->hp);
IG_UpdateText(stats_hp_Id,stats_hp_str);

sprintf(stats_dex_str,"%-10d",objsel->stats->dex);
IG_UpdateText(stats_dex_Id,stats_dex_str);

sprintf(stats_str_str,"%-10d",objsel->stats->str);
IG_UpdateText(stats_str_Id,stats_str_str);

sprintf(stats_intel_str,"%-10d",objsel->stats->intel);
IG_UpdateText(stats_intel_Id,stats_intel_str);

sprintf(stats_weight_str,"%-10d",objsel->stats->weight);
IG_UpdateText(stats_weight_Id,stats_weight_str);

sprintf(stats_damage_str,"%-10d",objsel->stats->damage);
IG_UpdateText(stats_damage_Id,stats_damage_str);

sprintf(stats_quantity_str,"%-10d",objsel->stats->quantity);
IG_UpdateText(stats_quantity_Id,stats_quantity_str);
}

void SetHp()
{
objsel->stats->hp = InputIntegerValue(-1,-1,0,0x7fffffff,objsel->stats->hp);
Stats_Update();
}

void SetDex()
{
objsel->stats->dex = InputIntegerValue(-1,-1,0,0x7fffffff,objsel->stats->dex);
Stats_Update();
}

void SetStr()
{
objsel->stats->str = InputIntegerValue(-1,-1,0,0x7fffffff,objsel->stats->str);
Stats_Update();
}


void SetIntel()
{
objsel->stats->intel= InputIntegerValue(-1,-1,0,0x7fffffff,objsel->stats->intel);
Stats_Update();
}

void SetWeight()
{
objsel->stats->weight= InputIntegerValue(-1,-1,0,0x7fffffff,objsel->stats->weight);
Stats_Update();
}

void SetDamage()
{
objsel->stats->damage= InputIntegerValue(-1,-1,-0x7fffffff,0x7fffffff,objsel->stats->damage);
Stats_Update();
}

void SetQuantity()
{
objsel->stats->quantity = InputIntegerValue(-1,-1,-0x7fffffff,0x7fffffff,objsel->stats->quantity);
Stats_Update();
}

void OB_SetPersonalName()
{
if(!objsel)
    return;
InputIString(-1,-1,"Enter the character's individual name:",32,objsel->personalname);
OB_Update();
}

/*
 *
 *      EDIT BEHAVIOUR
 *
 */

//behave

/*
char contains[8][32];
char contents;
// Up to 8 objects can be put in the object when it's first created
*/

void EditBehaviour()
{
focus=0;
IG_KillAll();

if(objsel->behave == -1)
    strcpy(behave_str,none);
else
    strcpy(behave_str,COlist[objsel->behave].name);

DrawScreenBox3D(32,128,608,432);

setcolor(WHITE);
DrawScreenText(48,160,"Initial behaviour:");
behave_Id =    IG_InputButton(224,152,str32,SetBehave,NULL,NULL);
DrawScreenText(48,192,"When used:");
use_Id =   IG_InputButton(224,184,str32,SetUse,NULL,NULL);
DrawScreenText(48,224,"When killed:");
kill_Id =   IG_InputButton(224,216,str32,SetKill,NULL,NULL);
DrawScreenText(48,256,"When looked at:");
look_Id = IG_InputButton(224,248,str32,SetLook,NULL,NULL);

DrawScreenText(48,288,"When stood on:");
stand_Id = IG_InputButton(224,280,str32,SetStand,NULL,NULL);
DrawScreenText(48,320,"When hurt:");
hurt_Id = IG_InputButton(224,312,str32,SetHurt,NULL,NULL);
DrawScreenText(48,352,"When created:");
init_Id = IG_InputButton(224,344,str32,SetInit,NULL,NULL);
DrawScreenText(48,384,"Speech File:");
talk_Id = IG_InputButton(224,376,str32,SetTalk,NULL,NULL);


IG_TextButton(512,152,"Finish",Finish,NULL,NULL);
IG_TextButton(512,184,"Reset ",Resettt,NULL,NULL);
IG_AddKey(0x1b,OB_GoFocal);     // ESC to quit this menu
IG_AddKey(0x5d,Snap);           // Add key binding for shift-F10

Behave_Update();
}

void Behave_Update()
{
if(objsel->behave == -1)
    strcpy(behave_str,none);
else
    strcpy(behave_str,COlist[objsel->behave].name);
IG_UpdateText(behave_Id,behave_str);

if(objsel->funcs->ucache == -1)
    IG_UpdateText(use_Id,none);
else
    IG_UpdateText(use_Id,COlist[objsel->funcs->ucache].name);

if(objsel->funcs->kcache == -1)
    IG_UpdateText(kill_Id,none);
else
    IG_UpdateText(kill_Id,COlist[objsel->funcs->kcache].name);

if(objsel->funcs->lcache == -1)
    IG_UpdateText(look_Id,none);
else
    IG_UpdateText(look_Id,COlist[objsel->funcs->lcache].name);

if(objsel->funcs->scache == -1)
    IG_UpdateText(stand_Id,none);
else
    IG_UpdateText(stand_Id,COlist[objsel->funcs->scache].name);

if(objsel->funcs->hcache == -1)
    IG_UpdateText(hurt_Id,none);
else
    IG_UpdateText(hurt_Id,COlist[objsel->funcs->hcache].name);

if(objsel->funcs->icache == -1)
    IG_UpdateText(init_Id,none);
else
    IG_UpdateText(init_Id,COlist[objsel->funcs->icache].name);

IG_UpdateText(talk_Id,objsel->funcs->talk);
}

void SetBehave()
{
int before = objsel->behave;

if(objsel->behave == -1)
    objsel->behave = PickVrm(NULL);
else
    objsel->behave = PickVrm(COlist[objsel->behave].name);

if(objsel->behave != before)
    objsel->funcs->dirty=1;
Behave_Update();
}

void SetUse()
{
int before=objsel->funcs->ucache;

objsel->funcs->ucache = PickVrm(objsel->funcs->use);
if(objsel->funcs->ucache != -1)
    strcpy(objsel->funcs->use, COlist[objsel->funcs->ucache].name);

if(objsel->funcs->ucache != before)
    objsel->funcs->dirty=1;
Behave_Update();
}

void SetKill()
{
int before=objsel->funcs->kcache;

objsel->funcs->kcache = PickVrm(objsel->funcs->kill);
if(objsel->funcs->kcache != -1)
    strcpy(objsel->funcs->kill, COlist[objsel->funcs->kcache].name);

if(objsel->funcs->kcache != before)
    objsel->funcs->dirty=1;
Behave_Update();
}

void SetLook()
{
int before=objsel->funcs->lcache;

objsel->funcs->lcache = PickVrm(objsel->funcs->look);
if(objsel->funcs->lcache != -1)
    strcpy(objsel->funcs->look, COlist[objsel->funcs->lcache].name);

if(objsel->funcs->lcache != before)
    objsel->funcs->dirty=1;
Behave_Update();
}

void SetStand()
{
int before=objsel->funcs->scache;

objsel->funcs->scache = PickVrm(objsel->funcs->stand);
if(objsel->funcs->scache != -1)
    strcpy(objsel->funcs->stand, COlist[objsel->funcs->scache].name);

if(objsel->funcs->scache != before)
    objsel->funcs->dirty=1;
Behave_Update();
}

void SetHurt()
{
int before=objsel->funcs->hcache;

objsel->funcs->hcache = PickVrm(objsel->funcs->hurt);
if(objsel->funcs->hcache != -1)
    strcpy(objsel->funcs->hurt, COlist[objsel->funcs->hcache].name);

if(objsel->funcs->hcache != before)
    objsel->funcs->dirty=1;
Behave_Update();
}

void SetInit()
{
int before=objsel->funcs->icache;

objsel->funcs->icache = PickVrm(objsel->funcs->init);
if(objsel->funcs->icache != -1)
    strcpy(objsel->funcs->init, COlist[objsel->funcs->icache].name);

if(objsel->funcs->icache != before)
    objsel->funcs->dirty=1;
Behave_Update();
}

void SetTalk()
{
char before[32];

strcpy(before,objsel->funcs->talk);

InputString(-1,-1,"Enter the file with the character's speech (or '-' for none)",32,objsel->funcs->talk);
if(!stricmp(objsel->funcs->talk,"-"))
    objsel->funcs->tcache=-1;
else
    {
    objsel->funcs->tcache=1;
    if(!exist(objsel->funcs->talk))
        Notify(-1,-1,"WARNING:  That file does not exist.","Make sure it's correct...");
    }

if(strcmp(before,objsel->funcs->talk))
    objsel->funcs->dirty=1;
Behave_Update();
}

void Finish()
{
IG_WaitForRelease();
OB_GoFocal();
}

int PickVrm(char *original)
{
char name[128];
char **mlist;
long num=0;

mlist = M_get(COtot+1,sizeof(char *));

// If a null pointer
if(!original)
     original="-";

// If an empty string
if(!original[0])
     original="-";

mlist[num++]="-";
for(long ctr=0;ctr<COtot;ctr++)
    if(COlist[ctr].name)
         mlist[num++]=COlist[ctr].name;

qsort(mlist,num,sizeof(char *),CMP);

strcpy(name,original);
strupr(name);

InputNameFromListWithFunc(-1,-1,"Choose a function to call:",num-1,mlist,5,name,0,0,NULL);
if(!name[0])
    strcpy(name,original);

M_free(mlist);

for(long ctr=0;ctr<COtot;ctr++)
    if(COlist[ctr].name != NULL)
    if(!stricmp(COlist[ctr].name,name))
        return ctr;
return -1;
}

void Resettt()
{
int id;
if(!Confirm(-1,-1,"Reset to default values: are you sure?",NULL))
    return;

id = getnum4char(objsel->name);

if(id == -1)
    {
    Notify(-1,-1,"WTF??",NULL);
    return;
    }

objsel->behave = CHlist[id].behave;
objsel->funcs->ucache= CHlist[id].funcs->ucache;
objsel->funcs->kcache= CHlist[id].funcs->kcache;
objsel->funcs->lcache= CHlist[id].funcs->lcache;
objsel->funcs->scache= CHlist[id].funcs->scache;
objsel->funcs->hcache= CHlist[id].funcs->hcache;
objsel->funcs->icache= CHlist[id].funcs->icache;

if(objsel->funcs->ucache != -1)
    strcpy(objsel->funcs->use, COlist[objsel->funcs->ucache].name);
if(objsel->funcs->kcache != -1)
    strcpy(objsel->funcs->kill, COlist[objsel->funcs->kcache].name);
if(objsel->funcs->lcache != -1)
    strcpy(objsel->funcs->look, COlist[objsel->funcs->lcache].name);
if(objsel->funcs->scache != -1)
    strcpy(objsel->funcs->stand, COlist[objsel->funcs->scache].name);
if(objsel->funcs->hcache != -1)
    strcpy(objsel->funcs->hurt, COlist[objsel->funcs->hcache].name);
if(objsel->funcs->icache != -1)
    strcpy(objsel->funcs->init, COlist[objsel->funcs->icache].name);

strcpy(objsel->funcs->talk,CHlist[id].funcs->talk);
objsel->funcs->dirty=0;
}

/*
 * SetOwner - Enter a special mode to set the object's owner
 */

void SetOwner()
{
int temp;               // Used to get button handles

if(!objsel)
    return;

focus=0;                // This is now the focus
Toolbar();              // Build up the menu system again from scratch


IG_BlackPanel(VIEWX-1,VIEWY-1,258,258);
IG_Region(VIEWX,VIEWY,256,256,OBO_MakeOwner,NULL,NULL);

IG_Text(300,64,"SET OWNER OF THE OBJECT",WHITE);
IG_Text(300,72,"Click on the character who will own it",WHITE);

// Set up the X and Y map position counter at the top

MapXpos_Id = IG_InputButton(24,40,mapxstr,GetMapX,NULL,NULL);
MapYpos_Id = IG_InputButton(112,40,mapystr,GetMapY,NULL,NULL);

DrawMap(mapx,mapy,1,l_proj,0);                       // Draw the map

IG_TextButton(416,104,__up,OB_up,NULL,NULL);         // Pan up button
IG_TextButton(400,128,__left,OB_left,NULL,NULL);     // Pan left button
IG_TextButton(432,128,__right,OB_right,NULL,NULL);   // Pan right button
IG_TextButton(416,150,__down,OB_down,NULL,NULL);     // Pan down button

IG_AddKey(UP_KEY,OB_up);                             // Pan up key binding
IG_AddKey(DN_KEY,OB_down);                           // Pan down key binding
IG_AddKey(LF_KEY,OB_left);                           // Pan left key binding
IG_AddKey(RT_KEY,OB_right);                          // Pan right key binding

IG_TextButton(300,300,"Make Public Property",OBO_MakePublic,NULL,NULL);

IG_TextButton(300,332," Choose from a list ",OBO_FromList,NULL,NULL);

IG_TextButton(300,364,"  Cancel and Abort  ",OB_GoFocal,NULL,NULL);
IG_AddKey(0x1a,OB_GoFocal);

edit_mode = 1;

pause();
pause();
}

void OBO_MakePublic()
{
objsel->stats->owner = NULL;
OB_GoFocal();
}

/*
 *      Find all 'probably sentient' objects and present a list of p-names
 */

void OBO_FromList()
{
OBJECT *a;
char *original;
char name[128];
char **mlist;
long num=0;
int x,y;

if(!objsel)
    return;

// Find the name of the original owner, for a starting point

if(!objsel->stats->owner)
    original="-";
else
    if(!objsel->stats->owner->personalname)
        original="-";
    else
        original=objsel->stats->owner->personalname;

// Count the eligible candidates, which are either sentients or very strange.

for(x=0;x<curmap.w;x++)
    for(y=0;y<curmap.h;y++)
        for(a=GetObjectBase(x,y);a;a=a->next)
            if(a->flags.person && a->personalname)
                num++;

// Allocate space for them

mlist = M_get(num+1,sizeof(char *));

// Set up entry '-' (public property)

num=0;
mlist[num]=M_get(1,4);
strcpy(mlist[num++],"-");

// Gather them into the system and convert to lowercase

num=1;
for(x=0;x<curmap.w;x++)
    for(y=0;y<curmap.h;y++)
        for(a=GetObjectBase(x,y);a;a=a->next)
            if(a->flags.person)
                if(a->personalname)
                    if(strlen(a->personalname)>1)
                        {
                        mlist[num]=M_get(1,16);
                        strcpy(mlist[num],a->personalname);
                        strupr(mlist[num++]); // For the DEU system
                        }

// Sort for the DEU menu system

qsort(mlist,num,sizeof(char *),CMP);

strcpy(name,original);
strupr(name);

// Do the querying

InputNameFromListWithFunc(-1,-1,"Choose the being who owns the object:",num-1,mlist,5,name,0,0,NULL);

// Set back to original owner if user pressed ESC

if(!name[0])
    strcpy(name,original);

// Free the menu, we won't need it anymore.

for(long ctr=0;ctr<num;ctr++)
    M_free(mlist[ctr]);
M_free(mlist);

// If it's just "-", then make it public property and finish

if(!stricmp(name,"-"))
    {
    objsel->stats->owner = NULL;
    OB_GoFocal();
    return;
    }

// If not, search all eligible candidates again, and look for that name.
// Set the pointer when we've got it, and then finish

for(x=0;x<curmap.w;x++)
    for(y=0;y<curmap.h;y++)
        for(a=GetObjectBase(x,y);a;a=a->next)
            if(a->personalname && a->flags.person)
                if(!stricmp(name,a->personalname))
                    {
                    objsel->stats->owner = a;
                    OB_GoFocal();
                    return;
                    }

// Oh dear.  I wonder what went wrong?
// Set it to be public to avoid nasty accident

Notify(-1,-1,"Oh bugger!",NULL);
objsel->stats->owner = NULL;
OB_GoFocal();
}

/*
 *   OBO_MakeOwner - Pick an object, based on OB_Pick
 */

void OBO_MakeOwner()
{
int click_x,click_y,nx,ny;
OBJECT *a;

a = objsel;

click_x = ((x-VIEWX)>>5)+(mapx); // click_x = (mouse_x_pos/32) + map_x_pos
click_y = ((y-VIEWY)>>5)+(mapy); // click_y = (mouse_y_pos/32) + map_y_pos

SelectObj(click_x,click_y);      // Go fish

if(objsel)                       // If an object was selected
    {
    last_dir = objsel->curdir;
    objsel->form=NULL; // Force direction to change
    OB_SetDir(objsel,objsel->curdir);

    if(!objsel->personalname)
        {
        if(Confirm(-1,-1,"The chosen object doesn't have an individual name!","Are you sure you want to do this?"))
           a->stats->owner = objsel;
        }
    else
        a->stats->owner = objsel;
    }

objsel = a;

OB_GoFocal();
}

