//
//      DOS-specific video driver code
//

#include <stdio.h>

#include <dpmi.h>
#include <pc.h>
#include <go32.h>
#include <dos.h>

#include "itg.h"

#ifndef __DJGPP__
#error "This file is for DOS-DJGPP only"
#endif

extern void gen_clut(int bpp);
extern void OhNo(char *s);

unsigned short __SegA000 = 0xb00b;

typedef struct ModeInfoBlock {
  unsigned ModeAttributes;
  unsigned granularity,startseg,farfunc;
  short bscanline;
  short XResolution;
  short YResolution;
  short charpixels;
  unsigned bogus1,bogus2,bogus3,bogus4;
  unsigned PhysBasePtr;
  char bogus[228];
};

extern "C"
       {
       // Banked mode 110 driver
       char *i110_version();
       int i110_on();
       void i110_update();
       void i110_retrieve();

       // Banked mode 111 driver
       char *i111_version();
       int i111_on();
       void i111_update();
       void i111_retrieve();

       // Cirrus mode 110 driver
       char *i110CL_version();
       int i110CL_on();
       void i110CL_update();
       void i110CL_retrieve();

       // Cirrus mode 111 driver
       char *i111CL_version();
       int i111CL_on();
       void i111CL_update();
       void i111CL_retrieve();

       // Linear 110 driver

       char *i4110_version();
       int i4110_on();
       void i4110_update();
       void i4110_retrieve();

       // Linear 111 driver
       char *i4111_version();
       int i4111_on();
       void i4111_update();
       void i4111_retrieve();
       }

static int Generate_Aperture(int mode);
static void Get_Selector();

//
// High-level interface
//



/*
 *  110 - banked 640x480x15
 */

int Init_110()
{
   puts(ITGver);
   puts("  Driver:  110.asm");
   printf("  %s\n",i110_version());

   on       = i110_on;
   update   = i110_update;
   retrieve = i110_retrieve;

   Get_Selector();
   gen_clut(15);
   return 1;
}


/*
 *  111 - banked 640x480x16
 */

int Init_111()
{
   puts(ITGver);
   puts("  Driver:  111.asm");
   printf("  %s\n",i111_version());

   on       = i111_on;
   update   = i111_update;
   retrieve = i111_retrieve;

   Get_Selector();
   gen_clut(16);
   return 1;

}


/*
 *  4110 - flat 640x480x15
 */

int Init_4110()
{
   puts(ITGver);
   puts("  Driver:  4110.asm");
   printf("  %s\n",i4110_version());

   on       = i4110_on;
   update   = i4110_update;
   retrieve = i4110_retrieve;

   if(!Generate_Aperture(0x110))
       return 0;
   gen_clut(15);
   return 1;
}


/*
 *  4111 - flat 640x480x16
 */

int Init_4111()
{
   puts(ITGver);
   puts("  Driver:  4111.asm");
   printf("  %s\n",i4111_version());

   on       = i4111_on;
   update   = i4111_update;
   retrieve = i4111_retrieve;

   if(!Generate_Aperture(0x110))
       return 0;
   gen_clut(16);
   return 1;
}


/*
 *  110CL - 640x480x15 for Cirrus Logic
 */

int Init_110CL()
{
   puts(ITGver);
   puts("  Driver:  110CL.asm");
   printf("  %s\n",i110CL_version());

   on       = i110CL_on;
   update   = i110CL_update;
   retrieve = i110CL_retrieve;

   Get_Selector();
   gen_clut(15);
   return 1;
}


/*
 *  111CL - 640x480x16 for Cirrus Logic
 */

int Init_111CL()
{
   puts(ITGver);
   puts("  Driver:  111CL.asm");
   printf("  %s\n",i111CL_version());

   on       = i111CL_on;
   update   = i111CL_update;
   retrieve = i111CL_retrieve;

   Get_Selector();
   gen_clut(16);
   return 1;
}



//
// Low-level video code
//

/*
 *   Get mode info
 */

ModeInfoBlock *get_mode_info(int mode)
{
static ModeInfoBlock info;
__dpmi_regs r;

/* Use the transfer buffer to store the results of VBE call */
r.x.ax = 0x4F01;
r.x.cx = mode;
r.x.es = __tb / 16;
r.x.di = 0;
__dpmi_int(0x10, &r);

if(r.h.ah) return 0;
dosmemget(__tb, sizeof(ModeInfoBlock), &info);
return &info;
}

/*
 *   Make the linear framebuffer
 */

int Generate_Aperture(int mode)
{
__dpmi_meminfo info;
__dpmi_regs reg;
ModeInfoBlock *mb;
unsigned int ADDR;
int width,height;

mb=get_mode_info(mode);
if (!mb)
    {
    printf("Get VESA mode info failed.  Probable malfunction of video card.\n");
    return 0;
    //OhNo("Get VESA mode info failed.  Probable malfunction of video card.");
    }

if (!(mb->ModeAttributes & 0x80))
    {
    printf("Linear frame buffer not supported for VESA mode 0x%x.\n",mode);
    return 0;
    }

width=mb->XResolution;
height=mb->YResolution;
info.size=width*height*2;
info.address=mb->PhysBasePtr;

if(__dpmi_physical_address_mapping(&info) == -1)
    {
    printf("Physical mapping of address 0x%x failed!\n",mb->PhysBasePtr);
    OhNo("Could not allocate selector.  You may need to reboot.");
    }

ADDR=info.address;		// Updated by above call
printf("  Aperture = 0x%x\n",ADDR);

__SegA000 = __dpmi_allocate_ldt_descriptors(1);
__dpmi_set_segment_base_address(__SegA000,ADDR);
ADDR = 0;	// basis now zero from selector
__dpmi_set_segment_limit(__SegA000,(width*height*2)|0xfff);
printf("  Video selector = 0x%x\n",__SegA000);

if((short)__SegA000 == -1  || (short)__SegA000 == 0)
    OhNo("Could not allocate selector.  You may need to reboot.");
return 1;
}

/*
 *   Get the video selector
 */

void Get_Selector()
{
   puts("  Init Video Selector.. ");
   __SegA000 = __dpmi_segment_to_descriptor(0xa000);

   if((short)__SegA000 == -1  || (short)__SegA000 == 0)
        OhNo("Could not allocate selector.  You may need to reboot.");
   printf("  Video selector = 0x%x\n",__SegA000);
}

