//
//	ITG/640 graphics library core - a C implementation
//      Use the assembly code version whenever possible

//      NOTE: I32cel2 and I32cel3 have only been lightly tested and may break

#ifdef NO_ASM

#include <stdio.h>
#include <stdlib.h>

extern char *I32_clut;  // Colour lookup table
extern char *I32_dlut;  // Darkness lookup table
extern char *I32_llut;  // Light lookup table

#define SCREENSIZE 614400  // screen size
#define SCREENSIZE4 153600 // screen size / 4

#define BAL_15BPP 0x7bdf
#define BAL_16BPP 0xf7df

#define BALANCE15_5050(x,y) {x=x&BAL_15BPP;x>>=1;y=y&BAL_15BPP;y>>=1;}
#define BALANCE16_5050(x,y) {x=x&BAL_16BPP;x>>=1;y=y&BAL_16BPP;y>>=1;}
#define BALANCE15_2575(x,y) {x=x&BAL_15BPP;x>>=1;}
#define BALANCE16_2575(x,y) {x=x&BAL_16BPP;x>>=1;}
#define BALANCE15_7525(x,y) {y=y&BAL_15BPP;y>>=1;}
#define BALANCE16_7525(x,y) {y=y&BAL_16BPP;y>>=1;}

// Switch off the video mode (DOS only)

void off()
{
#ifdef __DJGPP__
asm("movl $3,%eax");
asm("int $0x10");
#endif
}

// Plot a single pixel

void dot(int x,int y, short colour, short *screen)
{
screen[(y*640)+x] = colour;
}

// Plot a line

void hline(int x,int y, int len, short colour, short *screen)
{
short *p;
int ctr;
p = &screen[(y*640)+x];

for(ctr=0;ctr<len;ctr++)
    *p++=colour;
}

// Do a dword memory set (or not if using non-pure implementation)

#ifdef PURE
void fset(long *dest,int character,int number)
{
for(ctr=0;ctr<number;ctr++)
    *dest++=character;
}
#else
void fset(char *dest,int character,int number)
{
memset(dest,character,number<<2);
}
#endif

// Memcpy (accelerated in ASM version)

void fcpy(char *dest,char *src,int number)
{
memcpy(dest,src,number<<2);
}

// screen blit

void fblit(char *dest,char *src)
{
memcpy(dest,src,SCREENSIZE);
}

// get a sprite

void I32get1(int x,int y, int width, int height, short *sprite, short *screen)
{
char *p;
int ctr,width2;

width2=width>>1;
p = (char *)screen+(1280*y)+(x*2);

for(ctr=0;ctr<height;ctr++)
    {
    memcpy(sprite,p,width);
    p+=1280;
    sprite+=width2;
    }
}

// put a sprite, no fancy stuff

void I32put1(int x,int y, int width, int height, short *sprite, short *screen)
{
char *p;
int ctr,width2;

width2=width>>1;
p = (char *)screen+(1280*y)+(x*2);

for(ctr=0;ctr<height;ctr++)
    {
    memcpy(p,sprite,width);
    p+=1280;
    sprite+=width2;
    }
}

// Put a sprite with transparency

void I32cel1(int x,int y, int width, int height, short *sprite, short *screen)
{
char *p;
short d,*q;
int ctr,ctr2;

width>>=1;
p = (char *)screen+(1280*y)+(x*2);

for(ctr=0;ctr<height;ctr++)
    {
    q=(short *)p;
    for(ctr2=0;ctr2<width;ctr2++)
        {
        d = *sprite++;
        if(d)
            *q=d;
        q++;
        }
    p+=1280;
    }
}

// Put a sprite with transparency, and an offset for pre-clipping

void I32cel2(int x,int y, int width, int height, short *sprite, short *screen, int preclip)
{
char *p;
short d,*q;
int ctr,ctr2;

p = (char *)screen+(1280*y)+(x*2);

sprite+=preclip;
for(ctr=0;ctr<height;ctr++)
    {
    q=(short *)p;
    for(ctr2=0;ctr2<width;ctr2++)
        {
        d = *sprite++;
        if(d)
            *q=d;
        q++;
        }
    p+=1280;
    sprite+=preclip;
    }
}

// Put a sprite with transparency, and an offset for post-clipping

void I32cel3(int x,int y, int width, int height, short *sprite, short *screen, int postclip)
{
char *p;
short d,*q;
int ctr,ctr2;

p = (char *)screen+(1280*y)+(x*2);

for(ctr=0;ctr<height;ctr++)
    {
    q=(short *)p;
    for(ctr2=0;ctr2<width;ctr2++)
        {
        d = *sprite++;
        if(d)
            *q=d;
        q++;
        }
    p+=1280;
    sprite+=postclip;
    }
}

// Put a sprite with translucency in 15bpp mode

void I32celt15(int x,int y, int width, int height, short *sprite, short *screen)
{
unsigned short *q;
char *p;
int a,b,c,d,e,f;
int ctr,ctr2;

width>>=1;
p = (char *)screen+(1280*y)+(x*2);

for(ctr=0;ctr<height;ctr++)
    {
    q=(short *)p;
    for(ctr2=0;ctr2<width;ctr2++)
        {
        c = *sprite++;
        f = *q;
        if(c)
            {
            BALANCE15_5050(c,f);
            a = c & 0x7c00;
            b = c & 0x03e0;
            c = c & 0x001f;

            d = f & 0x7c00;
            e = f & 0x03e0;
            f = f & 0x001f;

            a+=d;
            b+=e;
            c+=f;

            if(a>0x7c00)
                a=0x7c00;
            if(b>0x03e0)
                b=0x03e0;
            if(c>0x001f)
                c=0x001f;
            *q=(a+b+c);
            }
        q++;
        }
    p+=1280;
    }
}

// Put a sprite with translucency in 16bpp mode

void I32celt16(int x,int y, int width, int height, short *sprite, short *screen)
{
unsigned short *q;
char *p;
int a,b,c,d,e,f;
int ctr,ctr2;

width>>=1;
p = (char *)screen+(1280*y)+(x*2);

for(ctr=0;ctr<height;ctr++)
    {
    q=(short *)p;
    for(ctr2=0;ctr2<width;ctr2++)
        {
        c = *sprite++;
        f = *q;
        if(c)
            {
            BALANCE16_5050(c,f);
            a = c & 0xf800;
            b = c & 0x03e0;
            c = c & 0x001f;

            d = f & 0xf800;
            e = f & 0x03e0;
            f = f & 0x001f;

            a+=d;
            b+=e;
            c+=f;

            if(a>0xf800)
                a=0xf800;
            if(b>0x07e0)
                b=0x07e0;
            if(c>0x001f)
                c=0x001f;
            *q=(a+b+c);
            }
        q++;
        }
    p+=1280;
    }
}

// Put a sprite of darkness
// This uses a 640x480x8 screen to work with

void I32darkspr(int x,int y, int width, int height, unsigned char *out, unsigned char *in)
{
unsigned char *o;
int ctr,ctr2;

out = out+(y*640)+x;

for(ctr=0;ctr<height;ctr++)
    {
    o=out;
    for(ctr2=0;ctr2<width;ctr2++)
        *o++ = I32_dlut[((*in++)<<8) + *o];
    out+=640;
    }
}

// Put a sprite of light
// This uses a 640x480x8 screen to work with

void I32darkspr_inv(int x,int y, int width, int height, unsigned char *out, unsigned char *in)
{
unsigned char *o;
int ctr,ctr2;

out = (char *)out+(y*640)+x;

for(ctr=0;ctr<height;ctr++)
    {
    o=out;
    for(ctr2=0;ctr2<width;ctr2++)
        *o++ = I32_llut[((*in++)<<8)+*o];
    out+=640;
    }
}

// Transfer darkness map to physical screen, 15bpp
// Doesn't work if I32_clut is used as an array, so we do this instead.

void I32darkmem15(unsigned short *dest, unsigned char *src, int len)
{
int ctr;
for(ctr=0;ctr<len;ctr++)
    *dest++ = *(short *)(((((*src++&0x1f)<<15)+*dest)<<1)+(int)I32_clut);
}

// Transfer darkness map to physical screen, 16bpp
// Same as above, but using 65536 instead of 32768

void I32darkmem16(unsigned short *dest, unsigned char *src, int len)
{
int ctr;
for(ctr=0;ctr<len;ctr++)
    *dest++ = *(unsigned short *)(((((*src++&0x3f)<<16)+*dest)<<1)+(int)I32_clut);
}

// get a sprite, 8bpp for the light system

void I32get2(int x,int y, int width, int height, unsigned char *sprite, unsigned char *screen)
{
unsigned char *p;
int ctr;

p = (unsigned char *)screen+(y*640)+x;

for(ctr=0;ctr<height;ctr++)
    {
    memcpy(sprite,p,width);
    p+=640;
    sprite+=width;
    }
}

// Draw a character of the font on the screen

void I32font(int x,int y, char *character, short colour, short *screen)
{
unsigned short *p;
unsigned char c;
int ctr,ctr2;

p = &screen[(y*640)+x];

for(ctr=0;ctr<8;ctr++)
    {
    c=*character++;
    for(ctr2=0;ctr2<8;ctr2++)
        {
        if(c&128)
            *p=colour;
        c<<=1;
        p++;
        }
    p+=632;
    }
}

// Lightning in 15bpp mode

void I32lightning15(unsigned short *screen)
{
int a,b,c;
int ctr;

for(ctr=0;ctr<65535;ctr++)
        {
        c = *screen;
        a = c & 0x7c00;
        b = c & 0x03e0;
        c = c & 0x001f;

        a<<=1;
        b<<=1;
        c<<=2;

        if(a>0x7c00)
            a=0x7c00;
        if(b>0x03e0)
            b=0x03e0;
        if(c>0x001f)
            c=0x001f;
        *screen++=(a+b+c);
        }
}

// Lightning in 16bpp mode

void I32lightning16(unsigned short *screen)
{
int a,b,c;
int ctr;

for(ctr=0;ctr<65535;ctr++)
        {
        c = *screen;
        a = c & 0xf800;
        b = c & 0x07e0;
        c = c & 0x001f;

        a<<=1;
        b<<=1;
        c<<=2;

        if(a>0xf800)
            a=0xf800;
        if(b>0x07e0)
            b=0x07e0;
        if(c>0x001f)
            c=0x001f;
        *screen++=(a+b+c);
        }
}


#endif		//#ifndef NO_ASM
