//
//      IRE linked list management routines
//

#include <stdlib.h>

#include "doslib.hpp"
#include "core.hpp"
#include "memory.hpp"
#include "script.hpp"
#include "linklist.hpp"
#include "object.hpp"
#include "console.hpp"

// defines

// variables

static char stemp[128];       // Used for string operations
extern char in_editor;        // Which program we are in?
extern char AL_dirty;         // Has the ActiveList changed?

// functions

OBJECT *OB_Alloc();           // Allocate an Object
void OB_Free(OBJECT *ptr);    // Free an Object

void OB_SetDir(OBJECT *objsel,int dir);     // set direction
void OB_SetSeq(OBJECT *objsel,char *name);  // change appearance
void OB_Init(OBJECT *objsel,char *name);  // (re)set an object

int OB_Fore(OBJECT *objsel);   // Move object forwards in the list
int OB_Back(OBJECT *objsel);   // Move object back in the list

void FreePockets(OBJECT *obj);

extern char fullrestore;     // Be less thorough if we're only reloading
extern void Call_VRM(int i);
extern OBJECT *current_object;
extern OBJECT *victim;
OBJLIST *MasterList=NULL;

// code

/*
 *    OB_Alloc()  -  Allocate a new Object in the linked list
 */

OBJECT *OB_Alloc()
{
OBJECT *temp;

temp=(struct OBJECT *)M_get(1,sizeof(OBJECT));
temp->next=NULL;
LL_Add(&curmap.object,temp);
ML_Add(&MasterList,temp);

temp->slen=0;
temp->sdir=0;
temp->sptr=0;
temp->tag=0;
temp->behave=-1;                // Invalid behaviour VRM
temp->flags.on =1;

// Allocate sublayers

temp->personalname=(char *)M_get(32,sizeof(char));
temp->funcs = (FUNCS *)M_get(1,sizeof(FUNCS));
temp->stats = (STATS *)M_get(1,sizeof(STATS));
temp->maxstats = (STATS *)M_get(1,sizeof(STATS));
temp->wield = (OBJECT **)M_get(W_SIZE,sizeof(struct OBJECT *));
temp->schedule = (char **)M_get(24,sizeof(char *));

//for(int ctr=0;ctr<W_SIZE;ctr++)
//    temp->wield[ctr] = NULL;

// Return the address of the new entry

return temp;
}

/*
 *    OB_Free()  -  Destroy the specified entry in the list and relink it
 *                  May take a while
 */

void OB_Free(OBJECT *ptr)
{
// We cannot free the anchor (simplifies the logic)

if(ptr == curmap.object)
	panic("OB_Free()","Attempted to free OB_anchor:",itoa((long)ptr,stemp,16));

// We certainly cannot do this either

if(!ptr)
	panic("OB_Free()","Attempted to free NULL pointer:",NULL);

// First remove the object
LL_Remove(&curmap.object,ptr);

// Then remove any objects inside it
FreePockets(ptr);

// Then Deallocate it
LL_Kill(ptr);
}


/*
 *    LL_Add - Generalised linked-list add
 */

void LL_Add(OBJECT **base,OBJECT *object)
{
OBJECT *temp;

if(!base)
    panic("LL_Add","Attempted to add object to NULL",object->name);

if(!object)
    panic("LL_Add","Attempted to add NULL object to list",NULL);

temp=*base;
object->next = NULL;

// No objects in the list?

if(!temp)
    {
    *base = object;
    return;
    }

// Only one?

if(!temp->next)
    {
    temp->next = object;
    return;
    }

// Any other number

for(;temp->next;temp=temp->next);
temp->next = object;
}


/*
 *    LL_Remove - Generalised linked-list free
 */

void LL_Remove(OBJECT **base,OBJECT *object)
{
OBJECT *temp;
char found;

if(!base)
    panic("LL_Remove","Attempted to remove object from NULL",object->name);

if(!object)
    panic("LL_Remove","Attempted to remove NULL object from list",NULL);

temp=*base;

// Look for it, to make sure it is there.

found=0;
for(;temp;temp=temp->next)
    if(temp == object)
        found = 1;

if(!found)
    panic("LL_Remove","Could not find object in the list",object->name);

temp=*base;

// Is it the first object?

if(temp == object)
    {
    *base = object->next;
    object->next = NULL;
    return;
    }

// Is it the second object?

if(temp->next == object)
    {
    temp->next = object->next;
    object->next = NULL;
    return;
    }

// Any other number

for(;temp->next;temp=temp->next)
    if(temp->next == object)
        {
        temp->next = object->next;
        object->next = NULL;
        return;
        }

panic("LL_Remove","Whoops!  You really messed that up Joe!","The linked list is now a kinked list");
}


/*
 *  LL_Kill - Release the memory held by an object and dependent structs.
 *            This is a low-level operation but has error checking.
 */

void LL_Kill(OBJECT *obj)
{
if(!obj)
    panic("LB_Kill","Object is NULL",NULL);
if(obj->next)
    panic("LL_Kill","Object is part of a linked-list",obj->name);
if(obj->pocket)
    panic("LL_Kill","Object has items in pocket",obj->name);

M_free(obj->stats);
obj->stats = NULL;

M_free(obj->maxstats);
obj->maxstats = NULL;

M_free(obj->funcs);
obj->funcs = NULL;

M_free(obj->wield);
obj->wield = NULL;

for(int ctr=0;ctr<24;ctr++)
    {
    if(obj->schedule[ctr])
        {
        M_free(obj->schedule[ctr]);
        obj->schedule[ctr]=NULL;
        }
    }
M_free(obj->schedule);
obj->schedule=NULL;

// Check all dependencies

CheckDepend(obj);
ML_Del(&MasterList,obj);
if(obj->behave != -1)
    ML_Del(&ActiveList,obj);

if(current_object == obj)
    current_object = NULL;
if(victim== obj)
    victim = NULL;

M_free(obj);
}

/*
 *      ML_Add - Add an object pointer to an object chain
 */

void ML_Add(OBJLIST **m,OBJECT *obj)
{
OBJLIST *t;

if(!obj)
    panic(cfa,"Tried to add NULL to list\n",NULL);

if(!*m)
    {
    *m = (OBJLIST *)M_get(1,sizeof(OBJLIST));
    (*m)->ptr = obj;
    (*m)->next = NULL;
    AL_dirty = 1;
    return;
    }

// Don't add if it's already there

for(t = *m;t;t=t->next)
    if(t->ptr == obj)
         return;

for(t = *m;t->next;t=t->next);
t->next = (OBJLIST *)M_get(1,sizeof(OBJLIST));
t->next->ptr = obj;
t->next->next = NULL;
AL_dirty = 1;
return;
}

/*
 *    ML_Del - Free an object from an object chain
 */

void ML_Del(OBJLIST **m,OBJECT *object)
{
OBJLIST *temp,*t2;
char found;

if(!object)
    panic("ML_Del","Attempted to remove NULL object from list",NULL);

temp=*m;

// Look for it, to make sure it is there.

found=0;
for(;temp;temp=temp->next)
    if(temp->ptr == object)
        {
        found=1;
        break;
        }

if(!found)
    return;

temp=*m;

// Is it the first object?

if(temp->ptr == object)
    {
    *m = temp->next;
    temp->next = NULL;
    M_free(temp);
    AL_dirty = 1;
    return;
    }

// Is it the second object?

if(temp->next->ptr == object)
    {
    t2=temp->next;
    temp->next = temp->next->next;
    t2->next = NULL;
    M_free(t2);
    AL_dirty = 1;
    return;
    }

// Any other number

for(;temp->next;temp=temp->next)
    if(temp->next->ptr == object)
        {
        t2 = temp->next;
        temp->next = temp->next->next;
        t2->next = NULL;
        M_free(t2);
        AL_dirty = 1;
        return;
        }

panic("ML_Del","Oh no!","The chain has become corrupted");
}

void AL_Add(OBJLIST **a,OBJECT *o)
{
if(o->behave == -1)        // It's not active
    return;
ML_Add(a,o);
AL_dirty = 1;
}


/* ========================== Non-core functionality ====================== */

/*
 *    OB_Init()  -  Initialize or re-initialise an Object.
 */

void OB_Init(OBJECT *objsel,char *name)
{
int ctr;
int seq;

ctr = getnum4char(name);                // Find the character's name
if(ctr == -1)
       panic("OB_Init","Could not find object in Section: CHARACTERS!",name);
objsel->name = CHlist[ctr].name;        // Update object's name from static

// Now the object will take on the properties of the new character

// Copy the direction cache
memcpy(objsel->dir,CHlist[ctr].dir,sizeof(CHlist[ctr].dir));

// Set up the stats and function systems
memcpy(objsel->stats,CHlist[ctr].stats,sizeof(STATS));
memcpy(objsel->maxstats,CHlist[ctr].maxstats,sizeof(STATS));
memcpy(objsel->funcs,CHlist[ctr].funcs,sizeof(FUNCS));

// Get the sequence index
seq = objsel->dir[objsel->curdir];

// Set up the animation stats
objsel->sdir = 1;
objsel->slen = SQlist[seq].frames;
objsel->sptr = 0;

// Set up the appearance
objsel->form = &SQlist[seq];
objsel->desc = CHlist[ctr].desc;                // Long text description
objsel->shortdesc = CHlist[ctr].shortdesc;      // Short text description

// Set up the physics
if(!fullrestore)
    {
    objsel->flags = CHlist[ctr].flags;
    objsel->behave = CHlist[ctr].behave;
    objsel->light = CHlist[ctr].light;
    }

ML_Del(&ActiveList,objsel);     // Make sure its gone
AL_Add(&ActiveList,objsel);     // Add it if necessary

objsel->flags.on =1;

    // This was only done on full restore: this was incorrect, since
    // the size can never change.

    objsel->vblockx= CHlist[ctr].vblockx;
    objsel->vblocky= CHlist[ctr].vblocky;
    objsel->vblockw= CHlist[ctr].vblockw;
    objsel->vblockh= CHlist[ctr].vblockh;
    objsel->hblockx= CHlist[ctr].hblockx;
    objsel->hblocky= CHlist[ctr].hblocky;
    objsel->hblockw= CHlist[ctr].hblockw;
    objsel->hblockh= CHlist[ctr].hblockh;

    // Active-Area was not being dealt with correctly

    objsel->VareaX= CHlist[ctr].VareaX;
    objsel->VareaY= CHlist[ctr].VareaY;
    objsel->VareaW= CHlist[ctr].VareaW;
    objsel->VareaH= CHlist[ctr].VareaH;
    objsel->HareaX= CHlist[ctr].HareaX;
    objsel->HareaY= CHlist[ctr].HareaY;
    objsel->HareaW= CHlist[ctr].HareaW;
    objsel->HareaH= CHlist[ctr].HareaH;

// Set up the size

objsel->flags.large=0;  // Assume it's only 1 square

objsel->w    = objsel->form->seq[0]->w;
objsel->h    = objsel->form->seq[0]->h;

// Set up the size in tiles
objsel->mw = objsel->w>>5;
if(objsel->w & 0xf || !objsel->mw)
     objsel->mw++;

objsel->mh = objsel->h>>5;
if(objsel->h & 0xf || !objsel->mh)
     objsel->mh++;

if(objsel->mh>1 || objsel->mw>1)        // If it's bigger than 1 square..
     objsel->flags.large=1;

// If an Active Area has not been set aside, assume it covers the
// whole area

if(objsel->VareaW==0)
    {
    objsel->VareaW=objsel->mw;
    objsel->VareaX=0;
    }
if(objsel->VareaH==0)
    {
    objsel->VareaH=objsel->mh;
    objsel->VareaY=0;
    }
if(objsel->HareaW==0)
    {
    objsel->HareaW=objsel->mw;
    objsel->HareaX=0;
    }
if(objsel->HareaH==0)
    {
    objsel->HareaH=objsel->mh;
    objsel->HareaY=0;
    }

// If a special solid area has not been set aside, assume it covers the
// whole area

if(objsel->vblockw==0)
    {
    objsel->vblockw=objsel->mw;
    objsel->vblockx=0;
    }
if(objsel->vblockh==0)
    {
    objsel->vblockh=objsel->mh;
    objsel->vblocky=0;
    }
if(objsel->hblockw==0)
    {
    objsel->hblockw=objsel->mw;
    objsel->hblockx=0;
    }
if(objsel->hblockh==0)
    {
    objsel->hblockh=objsel->mh;
    objsel->hblocky=0;
    }
}

/*
 *    OB_SetDir()  -  Change the direction that a character is facing
 *                    objsel is the pointer in the list, dir the direction
 *                    it is facing.  See CHAR_L CHAR_R etc..
 */

void OB_SetDir(OBJECT *objsel,int dir)
{
int seq;

if(dir>3)
    {
    Bug("Aargh!  Attempt to set %s to face invalid direction %d!\n",objsel->name,dir);
    return;
    }

//if(dir == objsel->curdir)                       // Not needed
//    return;

// Get the sequence index
seq = objsel->dir[dir];

if(objsel->form == &SQlist[seq])   // abort if we're already the candidate
   return;

   // Set new direction
objsel->curdir = dir;

objsel->form = &SQlist[seq];
objsel->w    = objsel->form->seq[0]->w;
objsel->h    = objsel->form->seq[0]->h;

// Set up the animation stats
objsel->sdir = 1;
objsel->slen = SQlist[seq].frames;
if(objsel->sptr > objsel->slen)
    objsel->sptr = 0;
}

/*
 *    OB_SetSeq()-  Temporarily set the object's display frame to something
 *                  unusual.  The name is the SEQUENCE we select
 */

void OB_SetSeq(OBJECT *objsel,char *seqname)
{
int seq;


// Get the sequence index
seq = getnum4sequence(seqname);
if(seq == -1)
    {
    Bug("Could not find sequence %s in SECTION:SEQUENCES",seqname);
    C_printf("Could not find sequence %s in SECTION:SEQUENCES\n",seqname);
    return;
    }

objsel->form = &SQlist[seq];
objsel->w    = objsel->form->seq[0]->w;
objsel->h    = objsel->form->seq[0]->h;

// Set up the animation stats
objsel->sdir = 1;
objsel->slen = SQlist[seq].frames;
objsel->sptr = 0;


}

/*
 *    OB_Back()  -  Move an object backwards in the list
 */

int OB_Back(OBJECT *objsel)
{
OBJECT *temp;

if(!curmap.object->next)
	return 1;               // Not enough sprites to do this
if(!curmap.object->next->next)
	return 1;               // Not enough sprites to do this

if(!objsel)
	return 2;               // No sprite selected

if(objsel==curmap.object)
	return 3;               // Not permitted

// Look for a link to the selected object two objects away.

for(OBJECT *tt=curmap.object;tt->next->next;tt=tt->next)
	if(tt->next->next == objsel)
		{
		temp=objsel->next;              // Preserve original link
		objsel->next=tt->next;          // link into next part of list
		tt->next=objsel;                // link into the list itself
		objsel->next->next=temp;        // Restore old link
		return 0;                       // Exit
		}
return 0;
}

/*
 *    OB_Fore()  -  Move an object forward in the list
 */

int OB_Fore(OBJECT *objsel)
{
OBJECT *temp;

if(!curmap.object->next)
	return 1;               // Not enough sprites to do this
if(!curmap.object->next->next)
	return 1;               // Not enough sprites to do this

if(!objsel)
	return 2;               // No sprite selected

if(objsel==curmap.object)
	return 3;               // Not permitted

// Look for a link to the selected object two objects away.

for(OBJECT *tt=curmap.object;tt->next->next;tt=tt->next)
	if(tt->next == objsel)
		{
		temp=tt->next->next->next;      // Preserve original link
		tt->next=tt->next->next;        // Change first link
		tt->next->next=objsel;          // Change second link
		objsel->next=temp;              // Change third link
		return 0;
		}
return 0;
}


/*
 * OB_Previous - Pointer to previous object
 */

OBJECT *OB_Previous(OBJECT *a)
{
OBJECT *temp;
for(temp=curmap.object;temp->next;temp=temp->next)
    if(temp->next==a)
        return(temp);
panic("Ah shit","Whoops!  You really messed that up Joe!","The linked list is now a kinked list..");
return NULL;
}


void FreePockets(OBJECT *obj)
{
OBJECT *temp,*next;

// Sanity check
if(!obj)
    panic("FreePockets","NULL pointer encountered",NULL);

// Silently ignore object without pocket
if(!obj->pocket)
    return;

temp = obj->pocket;
while(temp)
    {
    next = temp->next;
    LL_Remove(&obj->pocket,temp);       // Unlink the object
    if(temp->pocket)                    // If it has objects inside
        FreePockets(temp);             // Recurse inside it
    LL_Kill(temp);                      // Free the memory
    temp = next;                        // Go to next object
    };
}


