//
//      Convert from various RGB formats into the one we're using.
//      (15bpp, 16bpp, or possibly 24bpp later on)
//

#include <stdlib.h>
#include "doslib.hpp"
#include "core.hpp"
#include "console.hpp"

// defines

// variables

static char rgb_format;      // 0,1,2 (15,16,24)
extern char *cfa;

extern void (*I32darkmem)(char *d,char *s,int pixels);
extern void (*I32lightning)(char *d);

// functions

int MakeRGB(int r,int g,int b);
void ConvertRGB(char *in,unsigned char *pal,short *out,int length,int div);
void set_RGB_format(int rgbf);


// code

/*
 *  set_RGB_format - choose a colour format
 *                   0 = 15bpp, 1 = 16bpp, 2 = 24bpp
 */

void set_RGB_format(int rgbf)
{
char text[128];
rgb_format = rgbf;

switch(rgbf)
    {
    case 0:
    boot2("Output in 15bpp (5-5-5)\n");
    I32darkmem = I32darkmem15;
    I32lightning= I32lightning15;
    return;

    case 1:
    boot2("Output in 16bpp (5-6-5)\n");
    I32darkmem = I32darkmem16;
    I32lightning= I32lightning16;
    return;

    case 2:
    boot2("Output in 24bpp (8-8-8)\n");
    return;
    }
sprintf(text,"%d",rgbf);
panic(cfa,"set_rgb_format - unknown RGB format number",text);
}

/*
 *  ConvertRGB() - Convert a block of 24-bit colours to appropriate RGB set.
 *                 Calls down to MakeRGB for each pixel
 */

void ConvertRGB(char *in,unsigned char *pal,short *out,int length,int bpp)
{
unsigned char inbyte;
unsigned int outword;
int r,g,b;
char num[10];

itoa(bpp,num,10);

if(bpp != 6 && bpp != 8)
    panic(cfa,"ConvertRGB - input bpp is not 6 or 8.  (from old system, 1 now 6, 3 now 8)",num);

for(int ctr=0;ctr<length;ctr++)
    {
    inbyte=in[ctr];                      // Get the index.
    if(!inbyte)                          // Zero is transparent.
        out[ctr]=0;                      // Make it transparent
    else
        {
        r = pal[inbyte*3];
        g = pal[(inbyte*3)+1];
        b = pal[(inbyte*3)+2];

        if(bpp == 6)                     // If only 6 bpp input, expand to 8
            {
            r<<=2;
            b<<=2;
            g<<=2;
            }

        outword = MakeRGB(r,g,b);        // Get output data
        if(!outword) outword=0x0001;     // Off-black, but not transparent
        out[ctr]=outword;
        }
    }
}

/*
 *  MakeRGB - Get the appropriate native word for the specified RGB levels
 */

int MakeRGB(int r,int g,int b)
{
unsigned int outword=0;

switch(rgb_format)
    {
    case 0:             // 15 bits per pixel, 5-5-5 format
    r>>=3;              // Adjust from 8bpp to 5bpp
    g>>=3;
    b>>=3;

    outword=(r&31);     // Merge output
    outword<<=5;
    outword+=(g&31);
    outword<<=5;
    outword+=(b&31);
    break;

    case 1:
    r>>=3;              // Adjust from 8bpp to 5 or 6 bpp
    g>>=2;
    b>>=3;

    outword=(r&31);     // Merge output
    outword<<=6;
    outword+=(g&63);
    outword<<=5;
    outword+=(b&31);
    break;

    case 2:             // 24 or 32 bpp, no adjust needed
    case 3:
    outword=r;          // Just merge them
    outword<<=8;
    outword+=g;
    outword<<=8;
    outword+=b;
    break;
    }

return(outword);
}

