/*
 *      IT-HE Script Compiler v4
 *
 *      History:
 *      v1 procedures were compiled to VCPU opcodes
 *      v2 Used direct structure access to simplify new features
 *      v3 Switched to DOS-DLL files (using DLX) for higher performance
 *      v4 is part of the IRE boot process, compiling game description files
 *         at startup
 */

#include <stdio.h>
//#include <mem.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "doslib.hpp"
#include "sys.hpp"
#include "textfile.hpp"
#include "itg/itg.h"
#include "core.hpp"
#include "init.hpp"
#include "oscli.hpp"
#include "console.hpp"
#include "sound.hpp"
#include "vidmodes.h"

// defines

#define NO_OF_SECTIONS 16       // Up to 16 separate blocks.  Arbitrary.
#define MAX_LINE_LEN 4096       // Used for buffer arrays

// Variables

char   in_editor = 0;
struct S_POOL   *SPlist;              // Sprite list
struct SEQ_POOL *SQlist;              // Sequence list
struct OBJECT   *CHlist;              // Main list of characters
struct D_TEXT   *DSlist;              // Text descriptions
struct TILE     *TIlist;              // Map tile
struct S_POOL   *RTlist;              // Roof-tile list

extern struct SMTab *wavtab;          // Sound table
extern struct SMTab *modtab;          // Music table

long SPtot;                     // Total number of sprite
long SQtot;                     // Total number of sequence
long COtot;                     // Total number of code
long CHtot;                     // Total number of characters
long DStot;                     // Total number of descriptions
long TItot;                     // Total number of tiles
long RTtot;                     // Total number of roof tiles

long spr_alloc=0;
long seq_alloc=0;
long vrm_alloc=0;
long chr_alloc=0;
long des_alloc=0;
long til_alloc=0;
long mod_alloc=0;
long wav_alloc=0;
long rft_alloc=256;

extern int Songs;
extern int Waves;

static struct SectionMap_S              // Map of the blocks in the script
{
    int start;
    int end;
    char name[32];
} sect_map[NO_OF_SECTIONS];

static int no_of_sections;              // Number of blocks in the file
static char Rbuffer[MAX_LINE_LEN];      // Text buffer for LiSP-alikes

static TF_S script;                     // TextFile class instance
static char *NOTHING="\0";              // This is used instead of NULL
static char *compilename;               // Name of file being compiled

// Functions

void Compile(char *fn);                 // High-level function, compile a file

static void Purify();                   // Clean the text for compiler
static void SeekSect();                 // Map out the blocks

// LiSP and support functions

char *rest(char *input);         // Get pointer to tail of string
char *last(char *input);         // Get pointer to end of tail
char *first(char *input);        // Get COPY of head of string
char *hardfirst(char *line);     // Mutilate original string
                                        // and return just the head

void Strip(char *tbuf);          // Eliminate trailing spaces
int words(char *input);          // Return the number of words

static char isxspace(unsigned char input);      // This is passed a character.
                                                // Is it whitespace?

static void Dump(long lineno,char *error,char *help);  // Report script error

static int getnumber(char *r);        // Get a number (or constant)
static int isnumber(char *r);           // Is the string numeric?
       int getnum4sequence(char *r);    // Get index of animation sequence
       int getnum4VRM(char *name);      // Get index of VRM function
static int getnum4description(char *r); // Get index of text description

// Process the sections

static void Section(char *name);                   // Section dispatcher

static void sprites(long start,long finish);       // Compile sprites
static void sequences(long start,long finish);     // Compile animation sequences
static void characters(long start,long finish);    // Compile characters
static void descriptions(long start,long finish);  // Compile descriptive strings
static void tiles(long start,long finish);         // Compile map tile sequences
static void code(long start,long finish);          // Compile code
static void sounds(long start,long finish);        // Compile sounds
static void music(long start,long finish);         // Compile music
static void rooftiles(long start,long finish);         // Compile music

/*
 *      Compile - Compile the script
 */

void Compile(char *fn)
{
cfa = "Compile";

bootmsg("Compiling script..\n");

bootmsg("  Loading.. ");
script.init(fn);
bootmsg("done.  %ld lines\n",script.lines);

compilename = fn;

bootmsg("  Purifying.. ");
Purify();
bootmsg("done.\n");

bootmsg("  Searching sections.. ");
SeekSect();
bootmsg("done.  %d sections\n",no_of_sections-1);

bootmsg("  Compiling.. \n");

// Keep searching for the right sections until we have compiled them all.

Section("descriptions");
Section("code");
Section("sprites");
Section("sequences");
Section("characters");
Section("tiles");
Section("sounds");
Section("music");
Section("rooftiles");

bootmsg("  Compilation finished.\n");

//script.term();

return;
}

/*
 *      Section - Compile appropriate section, return the number done so far
 */

void Section(char *name)
{
char tbuf[64];
int firstline=0,lastline=0;
char found=0;
int ctr;

for(ctr=0;ctr<no_of_sections;ctr++)
    if(!stricmp(name,sect_map[ctr].name))
        {
        firstline = sect_map[ctr].start;
        lastline = sect_map[ctr].end;
        found=1;
        }

if(!found)
    return;

// Get each end of the section


// Get the section name by stripping off the first element with rest()

strcpy(tbuf,rest(script.line[firstline++]));

// Kill off trailing spaces

Strip(tbuf);

if(!stricmp(name,"descriptions"))
        {
	descriptions(firstline,lastline);
        return;
        }

// Code is compiled second..

if(!stricmp(name,"code"))
        {
	code(firstline,lastline);
        return;
        }

// Sprites are compiled third..

if(!stricmp(name,"sprites"))
        {
	sprites(firstline,lastline);
        return;
        }

// Sequences are compiled fourth..

if(!stricmp(name,"sequences"))
        {
	sequences(firstline,lastline);
        return;
        }

// Characters are compiled fifth..

if(!stricmp(name,"characters"))
        {
	characters(firstline,lastline);
        return;
        }

// Tiles are compiled sixth..

if(!stricmp(name,"tiles"))
        {
	tiles(firstline,lastline);
        return;
        }

// Sounds are compiled seventh..

if(!stricmp(name,"sounds"))
        {
	sounds(firstline,lastline);
        return;
        }

// Music is compiled eighth..

if(!stricmp(name,"music"))
        {
	music(firstline,lastline);
        return;
        }

// RoofTiles are compiled ninth..

if(!stricmp(name,"rooftiles"))
        {
	rooftiles(firstline,lastline);
        return;
        }

Dump(firstline,"Unknown SECTION type.","Valid types are:\r\nSPRITES  SEQUENCES  TILES  CHARACTERS  CODE  DESCRIPTIONS  TILES\r\nSOUNDS  MUSIC  ROOFTILES");
return;
}

/* ======================== Section compilers =========================== */

/*
 *      Sprites - Compile the sprites
 */

void sprites(long start,long finish)
{
long ctr,pos,imagedata;
char *Rptr;
char *line;

// The sprite list is just a simple list, of format:
//
// IDENTIFIER, FILENAME
// IDENTIFIER, FILENAME
//      :    ,    :
//      :    ,    :
//
// Therefore it is sufficient to just get the first two terms and store them.
//

// First, see how many sprites we have.

bootmsg("    Section: sprites\n");

SPtot = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];        // Get the line
	Rptr = first(rest(line));       // If the second word exists,
                                        // then the first word has to as well
        if(Rptr != NOTHING)             // and we assume we have the data
            SPtot++;
        }

bootmsg("      %d sprite entries\n", SPtot);

// Allocate room on this basis

if(in_editor == 2)
    bootmsg("      Allocating space for %d entries\n", spr_alloc);
else
    spr_alloc = SPtot;

SPlist = (S_POOL *)M_get(sizeof(S_POOL),spr_alloc+1);
bootmsg("      %d bytes allocated\n", sizeof(S_POOL)*(spr_alloc+1));

// Now, process the data.  We will never parse this again, so it can be done
// destructively.

bootmsg("      Decoding");
Plot(SPtot);
pos = 0;
imagedata=0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];
	Rptr=first(line);                       // Get first term
	if(Rptr!=NOTHING)                       // If it exists..
		{
		Rptr=first(rest(line));         // Get second term
		if(Rptr!=NOTHING)               // If it exists
			{
                        SPlist[pos].fname = rest(line);      // Get filename
                        Strip(SPlist[pos].fname);            // Kill whitespc

                        // Now get the name as a pointer in the text Block

                        SPlist[pos].name=hardfirst(line);

                        // Initialise sprite entry

                        imagedata += Init_Sprite(pos);

			pos++;  // Go onto the next entry in the sprite list
			}
		}
	}

bootmsg("\n");
bootmsg("      Done.  %d bytes of image data allocated\n",imagedata);
}

/*
 *      Sequences - Compile the sequences
 */

void sequences(long start,long finish)
{
long ctr,pos,spos;
char *line,*Rptr;
long liststart=0,listend=0;
bootmsg("    Section: sequences\n");

SQtot = 0;

//      Sequences are a more complex format, but we can count the number
//      of sequences in the script by looking for the word 'name:'

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];        // Get the line
	Rptr = first(line);             // First see if it's blank
        if(!stricmp(Rptr,"name"))       // then look for the keyword 'Name'
            SQtot++;                    // Got one
        if(!stricmp(Rptr,"quickname"))  // then look for keyword 'QuickName'
            SQtot++;                    // Got one
        }

bootmsg("      %d sequences defined\n", SQtot);

// Allocate room on this basis

if(in_editor != 2)
    seq_alloc = SQtot;  // If not in scripter, use original value of seq_alloc

SQlist = (SEQ_POOL *)M_get(sizeof(SEQ_POOL),seq_alloc+1);

bootmsg("      %d bytes allocated\n", sizeof(SEQ_POOL)*(seq_alloc+1));

// Now, process the data.  We will never parse this again, so it can be done
// destructively.

bootmsg("      Parsing");
Plot(SQtot);
pos = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];
	Rptr=first(line);                       // Get first term
	if(Rptr!=NOTHING)                       // Main parsing is here
		{

        // The 'NAME' clause.  Big.

                if(!stricmp(Rptr,"name"))       // Got the first one
                    {
	            Rptr=first(rest(line));     // Get second term
	            if(Rptr!=NOTHING)           // Make sure its there
                        {
                        // Find the offset in the block

                        SQlist[pos].name = rest(line);      // Get name label
                        Strip(SQlist[pos].name);            // Kill whitespace
                        }
                    else
                        Dump(ctr,"NA: This sequence has no name",NULL);

                    SQlist[pos].frames=0;
		    SQlist[pos].speed=0;
		    SQlist[pos].flags=0;
		    SQlist[pos].x=0;
		    SQlist[pos].y=0;
                    SQlist[pos].overlay=NULL;

                    // First, find the boundary of the framelist
                    // so we can calculate the right number of frames

                    liststart=0;
                    listend=0;

                    long list = ctr+1;  // Skip the NAME: line
                    do
                        {
                        Rptr = first(script.line[list]);
                        if(!liststart)                  // Don't have start
                            if(!stricmp(Rptr,"framelist:"))
                                liststart = list+1;

                        if(!listend)                    // Don't have end
                            if(!stricmp(Rptr,"end"))
                                listend = list-1;

                        if(!stricmp(Rptr,"name") || list == finish)
                            list = -1;                  // Reached the end

                        list++;
                        } while(list>0);                // list = 0 on exit

                    // Now we should have the boundaries.

                    if(!liststart)
                        Dump(ctr,"NA: This sequence has no FRAMELIST: entry!",SQlist[pos].name);

                    if(!listend)
                        Dump(ctr,"NA: This sequence has no END entry!",SQlist[pos].name);

                    // Good.  Now, we'll count the frames

                    for(list = liststart;list<=listend;list++)
                        {
                        Rptr = first(script.line[list]);
                        if(Rptr != NOTHING)
                            SQlist[pos].frames++;
                        }

                    if(!SQlist[pos].frames)
                        Dump(ctr,"NA: This sequence has no frames at all!",SQlist[pos].name);

                    // Now we have the number of frames in the list

                    SQlist[pos].seq = (S_POOL **)M_get(sizeof(S_POOL *),SQlist[pos].frames+1);

                    // Now we've allocated the space.  We're done I think
                    }

        // The 'FRAMELIST' clause.

                if(!stricmp(Rptr,"framelist:"))       // Got the first one
                    {
                    if(!SQlist[pos].name)
                        Dump(pos,"FL: This sequence has no name",NULL);

                    if(!liststart)
                        Dump(ctr,"FL: This sequence has no FRAMELIST: entry!",SQlist[pos].name);

                    if(!listend)
                        Dump(ctr,"FL: This sequence has no END entry!",SQlist[pos].name);

                    spos=0;

                    for(long list = liststart;list<=listend;list++)
                        {
                        Rptr = first(script.line[list]);
                        if(Rptr != NOTHING)
                            {
                            SQlist[pos].seq[spos] = find_spr(Rptr);
                            if(!SQlist[pos].seq[spos++])
                                Dump(ctr,"FL: Could not find this sprite",Rptr);
                            }
                        }
                    }

        // The 'END' clause.

                if(!stricmp(Rptr,"END"))       // End of the sequence
                    {
                    pos++;
                    Plot(0);
                    }

        // The 'QUICKNAME' clause.

        // QN is a fast-track sequence definition for just one frame.
        // Quicknames are of the form:
        //
        //      quickname  sequence_name sprite_name
        //
        // ..and are therefore much easier for things like map tiles
        // which do not usually have a second frame.

                if(!stricmp(Rptr,"quickname")) // Got the first one
                    {
	            Rptr=first(rest(rest(line)));     // Get Third term
	            if(Rptr!=NOTHING)                 // Make sure its there
                        {
                        // For a Quick sequence there is just one frame.
                        SQlist[pos].frames=1;
                        SQlist[pos].seq = (S_POOL **)M_get(sizeof(S_POOL *),SQlist[pos].frames+1);

                        // Now we've allocated the frame, find the sprite
                        SQlist[pos].seq[0] = find_spr(Rptr);
//                        boot2("%s = %p\n",Rptr,SQlist[pos].seq[0]);

                        // Did it work?
                        if(!SQlist[pos].seq[0])
                            Dump(ctr,"QN: Could not find this sprite",Rptr);

                        SQlist[pos].name = hardfirst(rest(line));  // Get name
                        Strip(SQlist[pos].name);            // Kill whitespace

//                        boot2("QN: n:%s s:%d\n",SQlist[pos].name,SQlist[pos].seq[0]);

                        }
                    else
                        Dump(ctr,"QN: QuickName sequences must be defined as:\r\n QUICKNAME sequence_name sprite_name",NULL);

		    SQlist[pos].speed=0;
		    SQlist[pos].flags=0;
		    SQlist[pos].x=0;
		    SQlist[pos].y=0;
                    pos++;
                    Plot(0);
                    }

        // Flags

                if(!stricmp(Rptr,"pingpong"))       // Animation will rewind
                    {
                    SQlist[pos].flags|=1;
		    if(SQlist[pos].frames<2)
			Dump(ctr,"You must not have a pingpong sequence with only one frame!",NULL);
                    }

                if(!stricmp(Rptr,"loop"))             // Animation repeats
                    SQlist[pos].flags|=2;
                if(!stricmp(Rptr,"looped"))
                    SQlist[pos].flags|=2;
                if(!stricmp(Rptr,"loops"))
                    SQlist[pos].flags|=2;

                if(!stricmp(Rptr,"stepped"))          // Animation is stepped
                    SQlist[pos].flags|=4;

        // The 'X' clause.

                if(!stricmp(Rptr,"x"))       // X offset
                    {
                    strcpy(Rbuffer,first(rest(line)));
                    SQlist[pos].x=getnumber(Rbuffer);
                    if(!(isnumber(Rbuffer)))
			Dump(ctr,"The X offset should be a number.",NULL);
                    }

        // The 'Y' clause.

                if(!stricmp(Rptr,"y"))       // Y offset
                    {
                    strcpy(Rbuffer,first(rest(line)));
                    SQlist[pos].y=getnumber(Rbuffer);
                    if(!(isnumber(Rbuffer)))
			Dump(ctr,"The Y offset should be a number.",NULL);
                    }

        // The 'SPEED' clause.

                if(!stricmp(Rptr,"speed"))       // Animation rate
                    {
                    strcpy(Rbuffer,first(rest(line)));
                    SQlist[pos].speed=getnumber(Rbuffer);
                    if(!(isnumber(Rbuffer)))
			Dump(ctr,"The speed should be a number.",NULL);
                    }

        // The 'OVERLAY' clause.

                if(!stricmp(Rptr,"overlay"))       // Overlaid frame
                    {
                    strcpy(Rbuffer,first(rest(line)));
                    SQlist[pos].overlay = find_spr(Rbuffer);
                    // Did it work?
                    if(!SQlist[pos].overlay)
                            Dump(ctr,"OV: Could not find this sprite",Rbuffer);
                    }

                }
        }

bootmsg("\n");
bootmsg("      Done.\n");
}

/*
 *      Characters - Compile the characters
 */

void characters(long start,long finish)
{
int ctr,tmp,pos;
char *line,*Rptr;
char something;         // Something is used for error-checking.
                        // It starts as zero, set when a valid line is found.
                        // If there is no valid line, signal an error

unsigned char polarity,blockx,blocky,blockw,blockh;
// Values for SETSOLID


bootmsg("    Section: characters\n");

CHtot = 0;

//      Count the instances of 'Name' to get the quantity of characters.

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];        // Get the line
	Rptr = first(line);             // First see if it's blank
        if(!stricmp(Rptr,"name"))       // then look for the word 'Name'
            CHtot++;                    // Got one
        }

bootmsg("      %d characters defined\n", CHtot);

// Allocate room on this basis

if(in_editor != 2)
    chr_alloc = CHtot;  // If not in scripter, use original value

CHlist = (OBJECT *)M_get(sizeof(OBJECT),chr_alloc+1);
bootmsg("      %d bytes allocated\n", sizeof(OBJECT)*(chr_alloc+1));

// Now, process the data.  We will never parse this again, so it can be done
// destructively.

bootmsg("      Parsing");
Plot(CHtot);
pos = -1;

for(ctr=start;ctr<=finish;ctr++)
	{
        something=0;                            // Assume nothing is there
        line = script.line[ctr];
	Rptr=first(line);                       // Get first term
	if(Rptr!=NOTHING)                       // Main parsing is here
		{

        // Get the name

                if(!stricmp(Rptr,"name"))       // Got the first one
                    {
	            Rptr=first(rest(line));     // Get second term
	            if(Rptr!=NOTHING)           // Make sure its there
                        {
                        // Find the offset in the block

                        Plot(0);
                        pos ++;                             // Pre-increment
                        CHlist[pos].name = rest(line);      // Get name label

                        Strip(CHlist[pos].name);            // Kill whitespace
                        if(strlen(CHlist[pos].name) > 31)   // Check name length
                            Dump(ctr,"This name exceeds 31 letters",CHlist[pos].name);
                        CHlist[pos].personalname= (char *)M_get(32,1);
                        CHlist[pos].maxstats = (STATS *)M_get(sizeof(STATS),1);
                        CHlist[pos].stats = (STATS *)M_get(sizeof(STATS),1);
                        CHlist[pos].funcs = (FUNCS *)M_get(sizeof(FUNCS),1);
                        CHlist[pos].funcs->ucache=-1;
                        CHlist[pos].funcs->tcache=-1;
                        CHlist[pos].funcs->kcache=-1;
                        CHlist[pos].funcs->lcache=-1;
                        CHlist[pos].funcs->scache=-1;
                        CHlist[pos].funcs->hcache=-1;
                        CHlist[pos].funcs->icache=-1;
                        CHlist[pos].funcs->dirty=0;

                        CHlist[pos].vblockx=0;   // Partly-solid object
                        CHlist[pos].vblocky=0;   // Solidity offset
                        CHlist[pos].vblockw=0;   // and size
                        CHlist[pos].vblockh=0;
                        CHlist[pos].hblockx=0;   // Partly-solid object
                        CHlist[pos].hblocky=0;   // Solidity offset
                        CHlist[pos].hblockw=0;   // and size
                        CHlist[pos].hblockh=0;

                        CHlist[pos].behave = -1;            // No behaviour
                        CHlist[pos].stats->oldbehave = -1;  // No behaviour
                        CHlist[pos].stats->tick=0;
                        CHlist[pos].stats->owner=NULL;

                        strcpy(CHlist[pos].personalname,"-");
                        CHlist[pos].desc = "No description";
                        CHlist[pos].shortdesc = CHlist[pos].desc;
                        // By default the thing is resurrected to be itself
                        strcpy(CHlist[pos].funcs->resurrect,CHlist[pos].name);
			something=1;                        // Got something
                        }
                    else
                        Dump(ctr,"NA: This character has no name",NULL);
                    }

        // Sanity Check

                if(pos == -1)
                        Dump(ctr,"NA: The character's name must come first",NULL);

        // Get the health points

		if(!stricmp(Rptr,"hp"))
			{
			strcpy(Rbuffer,first(rest(line)));
			if(!isnumber(Rbuffer))
				Dump(ctr,"The character's health must be a number.",NULL);
			tmp=getnumber(Rbuffer);
			CHlist[pos].stats->hp=tmp;
			something=1;                        // Got something
			}

		if(!stricmp(Rptr,"dexterity"))
			{
			strcpy(Rbuffer,first(rest(line)));
			if(!isnumber(Rbuffer))
				Dump(ctr,"The character's dexterity must be a number.",NULL);
			tmp=getnumber(Rbuffer);
			CHlist[pos].stats->dex=tmp;
			something=1;                        // Got something
			}

		if(!stricmp(Rptr,"strength"))
			{
			strcpy(Rbuffer,first(rest(line)));
			if(!isnumber(Rbuffer))
				Dump(ctr,"The character's strength must be a number.",NULL);
			tmp=getnumber(Rbuffer);
			CHlist[pos].stats->str=tmp;
			something=1;                        // Got something
			}

		if(!stricmp(Rptr,"intelligence"))
			{
			strcpy(Rbuffer,first(rest(line)));
			if(!isnumber(Rbuffer))
				Dump(ctr,"The character's intelligence must be a number.",NULL);
			tmp=getnumber(Rbuffer);
			CHlist[pos].stats->intel=tmp;
			something=1;                        // Got something
			}

		if(!stricmp(Rptr,"weight"))
			{
			strcpy(Rbuffer,first(rest(line)));
			if(!isnumber(Rbuffer))
				Dump(ctr,"The character's weight must be a number.",NULL);
			tmp=getnumber(Rbuffer);
			CHlist[pos].stats->weight=tmp;
			something=1;                        // Got something
			}

		if(!stricmp(Rptr,"damage"))
			{
			strcpy(Rbuffer,first(rest(line)));
			if(!isnumber(Rbuffer))
				Dump(ctr,"The damage it causes must be a number.",NULL);
			tmp=getnumber(Rbuffer);
			CHlist[pos].stats->damage=tmp;
			something=1;                        // Got something
			}

		if(!stricmp(Rptr,"left"))
			{
			strcpy(Rbuffer,first(rest(line)));
			tmp=getnum4sequence(Rbuffer);
			if(tmp==-1)
				Dump(ctr,"Could not find sequence:",Rbuffer);
			CHlist[pos].dir[CHAR_L]=tmp;
			something=1;                        // Got something
			}

		if(!stricmp(Rptr,"right"))
			{
			strcpy(Rbuffer,first(rest(line)));
			tmp=getnum4sequence(Rbuffer);
			if(tmp==-1)
				Dump(ctr,"Could not find sequence:",Rbuffer);
			CHlist[pos].dir[CHAR_R]=tmp;
			something=1;                        // Got something
			}

		if(!stricmp(Rptr,"up"))
			{
			strcpy(Rbuffer,first(rest(line)));
			tmp=getnum4sequence(Rbuffer);
			if(tmp==-1)
				Dump(ctr,"Could not find sequence:",Rbuffer);
			CHlist[pos].dir[CHAR_U]=tmp;
			something=1;                        // Got something
			}

		if(!stricmp(Rptr,"down"))
			{
			strcpy(Rbuffer,first(rest(line)));
			tmp=getnum4sequence(Rbuffer);
			if(tmp==-1)
				Dump(ctr,"Could not find sequence:",Rbuffer);
			CHlist[pos].dir[CHAR_D]=tmp;
			something=1;                        // Got something
			}

        // Get the description from the section: DESCRIPTION

		if(!stricmp(Rptr,"description"))
			{
			strcpy(Rbuffer,first(rest(line)));

                        if(Rbuffer[0] == '\"')  // Quoted text is direct
                            {
                            CHlist[pos].desc = strchr(line,'\"'); // start
                            CHlist[pos].desc++;            // skip the quote
                            Rptr = strrchr(line,'\"');     // Find last quote
                            if(Rptr)
                                *Rptr = 0;                 // blow quote away
                            }
                        else                    // Look up unquoted text
                            {                   // in the DESCRIPTION bank
			    tmp=getnum4description(Rbuffer);
			    if(tmp==-1)
				Dump(ctr,"Could not find this entry in section: DESCRIPTION",Rbuffer);
                            CHlist[pos].desc=DSlist[tmp].text;  // Get it
                            }
                        something=1;                    // Got something
			}

        // Get the short description from the section: DESCRIPTION

		if(!stricmp(Rptr,"short"))
			{
			strcpy(Rbuffer,first(rest(line)));

                        if(Rbuffer[0] == '\"')  // Quoted text is direct
                            {
                            CHlist[pos].shortdesc = strchr(line,'\"'); // start
                            CHlist[pos].shortdesc++;       // skip the quote
                            Rptr = strrchr(line,'\"');     // Find last quote
                            if(Rptr)
                                *Rptr = 0;                 // blow quote away
                            }
                        else                    // Look up unquoted text
                            {                   // in the DESCRIPTION bank
			    tmp=getnum4description(Rbuffer);
			    if(tmp==-1)
				Dump(ctr,"Could not find this entry in section: DESCRIPTION",Rbuffer);
                            CHlist[pos].shortdesc=DSlist[tmp].text;  // Get it
                            }
                        something=1;                    // Got something
			}

        // Get the character's USE function

		if(!stricmp(Rptr,"IfUsed"))
			{
                        strcpy(CHlist[pos].funcs->use,first(rest(line)));
                        something=1;                    // Got something
			}

        // Get the character's behaviour function

		if(!stricmp(Rptr,"behave") || !stricmp(Rptr,"behaviour")
		   || !stricmp(Rptr,"behavior"))
			{
			strcpy(Rbuffer,first(rest(line)));
                        CHlist[pos].behave = getnum4VRM(Rbuffer);
                        CHlist[pos].stats->oldbehave = CHlist[pos].behave;
                        if(CHlist[pos].behave == -1)
                            Dump(ctr,"Could not find this function in section: CODE",Rbuffer);
                        something=1;                    // Got something
			}

        // Get the character's STAND function

		if(!stricmp(Rptr,"IfTriggered"))
			{
                        strcpy(CHlist[pos].funcs->stand,first(rest(line)));
                        CHlist[pos].flags.trigger = 1;
                        something=1;                    // Got something
			}

        // Get the character's KILLED function

		if(!stricmp(Rptr,"IfKilled") || !stricmp(Rptr,"IfDead"))
			{
                        strcpy(CHlist[pos].funcs->kill,first(rest(line)));
                        something=1;                    // Got something
			}

                // Get the character's LOOK function

		if(!stricmp(Rptr,"IfLooked") || !stricmp(Rptr,"IfLookedAt")
                ||!stricmp(Rptr,"IfLook"))
			{
                        strcpy(CHlist[pos].funcs->look,first(rest(line)));
                        something=1;                    // Got something
			}

        // Get the character's HURT function

		if(!stricmp(Rptr,"IfHurt") || !stricmp(Rptr,"IfDamaged"))
			{
                        strcpy(CHlist[pos].funcs->hurt,first(rest(line)));
                        something=1;                    // Got something
			}

        // Get the character's INIT function

		if(!stricmp(Rptr,"Init") || !stricmp(Rptr,"OnInit"))
			{
                        strcpy(CHlist[pos].funcs->init,first(rest(line)));
                        something=1;                    // Got something
			}

        // Get the character's resurrection object, if special

		if(!stricmp(Rptr,"resurrect") || !stricmp(Rptr,"resurrect_to")
                || !stricmp(Rptr,"resurrect_as"))
			{
                        strcpy(CHlist[pos].funcs->resurrect,first(rest(line)));
                        something=1;                    // Got something
			}

		if(!stricmp(Rptr,"no_resurrect"))
                        {
                        strcpy(CHlist[pos].funcs->resurrect,"-");
                        something=1;                    // Got something
                        }

        // Get the character's light level
        // Invert it to bring in line with my bizarre light physics

		if(!stricmp(Rptr,"light"))
			{
                        CHlist[pos].light = getnumber(first(rest(line)));
                        something=1;                    // Got something
			}

        // Is the character solid only in places?
        // If so, find out which part is solid and how big it is

		if(!stricmp(Rptr,"setsolid")
                 ||!stricmp(Rptr,"subsolid"))
			{
//                        CHlist[pos].flags.solid = 1;  // Make sure it's solid
                        polarity=0;
                        strcpy(Rbuffer,first(rest(line)));
			if(!stricmp(Rbuffer,"H"))
                                polarity=1;
			if(!stricmp(Rbuffer,"V"))
                                polarity=2;
                        if(!polarity)
                                Dump(ctr,"\r\nPartly-solid objects must be declared with a polarity and four numbers:\r\nSETSOLID [H or V] <x offset> <y offset> <width> <height>",Rbuffer);

                        strcpy(Rbuffer,first(rest(rest(line))));
			if(!isnumber(Rbuffer))
                                Dump(ctr,"\r\nPartly-solid objects must be declared with a polarity and four numbers:\r\nSETSOLID [H or V] <x offset> <y offset> <width> <height>",Rbuffer);
                        blockx= getnumber(Rbuffer);

                        strcpy(Rbuffer,first(rest(rest(rest(line)))));
			if(!isnumber(Rbuffer))
                                Dump(ctr,"\r\nPartly-solid objects must be declared with a polarity and four numbers:\r\nSETSOLID [H or V] <x offset> <y offset> <width> <height>",Rbuffer);
                        blocky= getnumber(Rbuffer);

                        strcpy(Rbuffer,first(rest(rest(rest(rest(line))))));
			if(!isnumber(Rbuffer))
                                Dump(ctr,"\r\nPartly-solid objects must be declared with a polarity and four numbers:\r\nSETSOLID [H or V] <x offset> <y offset> <width> <height>",Rbuffer);
                        blockw= getnumber(Rbuffer);

                        strcpy(Rbuffer,first(rest(rest(rest(rest(rest(line)))))));
			if(!isnumber(Rbuffer))
                                Dump(ctr,"\r\nPartly-solid objects must be declared with a polarity and four numbers:\r\nSETSOLID [H or V] <x offset> <y offset> <width> <height>",Rbuffer);
                        blockh= getnumber(Rbuffer);

                        if(polarity == 1)
                            {
                            CHlist[pos].hblockx = blockx;
                            CHlist[pos].hblocky = blocky;
                            CHlist[pos].hblockw = blockw;
                            CHlist[pos].hblockh = blockh;
                            }
                        else
                            {
                            CHlist[pos].vblockx = blockx;
                            CHlist[pos].vblocky = blocky;
                            CHlist[pos].vblockw = blockw;
                            CHlist[pos].vblockh = blockh;
                            }
                        something=1;                    // Got something
			}

        // Is the character smaller than it's physical appearance?
        // If so, find out which part is active and how big it is

		if(!stricmp(Rptr,"setarea")
                 ||!stricmp(Rptr,"setactive")
                 ||!stricmp(Rptr,"setactivearea"))
			{
                        polarity=0;
                        strcpy(Rbuffer,first(rest(line)));
			if(!stricmp(Rbuffer,"H"))
                                polarity=1;
			if(!stricmp(Rbuffer,"V"))
                                polarity=2;
                        if(!polarity)
                                Dump(ctr,"\r\nObjects with Active Areas must be declared with a polarity and four numbers:\r\nSetActiveArea [H or V] <x offset> <y offset> <width> <height>",Rbuffer);

                        strcpy(Rbuffer,first(rest(rest(line))));
			if(!isnumber(Rbuffer))
                                Dump(ctr,"\r\nObjects with Active Areas must be declared with a polarity and four numbers:\r\nSetActiveArea [H or V] <x offset> <y offset> <width> <height>",Rbuffer);
                        blockx= getnumber(Rbuffer);

                        strcpy(Rbuffer,first(rest(rest(rest(line)))));
			if(!isnumber(Rbuffer))
                                Dump(ctr,"\r\nObjects with Active Areas must be declared with a polarity and four numbers:\r\nSetActiveArea [H or V] <x offset> <y offset> <width> <height>",Rbuffer);
                        blocky= getnumber(Rbuffer);

                        strcpy(Rbuffer,first(rest(rest(rest(rest(line))))));
			if(!isnumber(Rbuffer))
                                Dump(ctr,"\r\nObjects with Active Areas must be declared with a polarity and four numbers:\r\nSetActiveArea [H or V] <x offset> <y offset> <width> <height>",Rbuffer);
                        blockw= getnumber(Rbuffer);

                        strcpy(Rbuffer,first(rest(rest(rest(rest(rest(line)))))));
			if(!isnumber(Rbuffer))
                                Dump(ctr,"\r\nObjects with Active Areas must be declared with a polarity and four numbers:\r\nSetActiveArea [H or V] <x offset> <y offset> <width> <height>",Rbuffer);
                        blockh= getnumber(Rbuffer);

                        if(polarity == 1)
                            {
                            CHlist[pos].HareaX = blockx;
                            CHlist[pos].HareaY = blocky;
                            CHlist[pos].HareaW = blockw;
                            CHlist[pos].HareaH = blockh;
                            }
                        else
                            {
                            CHlist[pos].VareaX = blockx;
                            CHlist[pos].VareaY = blocky;
                            CHlist[pos].VareaW = blockw;
                            CHlist[pos].VareaH = blockh;
                            }
                        something=1;                    // Got something
			}

        // Get the flags

                if(!stricmp(Rptr,"solid"))             // Is it solid?
                    {
                    CHlist[pos].flags.solid = 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"blocklight") ||
                   !stricmp(Rptr,"blockslight"))        //Is it solid?
                    {
                    CHlist[pos].flags.blocklight= 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"invisible"))             // Is it invisible?
                    {
                    CHlist[pos].flags.invisible = 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"translucent"))           // Is it translucent?
                    {
                    CHlist[pos].flags.translucent= 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"post_overlay"))     // Are overlays postprojected?
                    {
                    CHlist[pos].flags.overlay= 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"fixed"))             // Can't move/get it?
                    {
                    CHlist[pos].flags.fixed = 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"container"))         // Is a bag or chest?
                    {
                    CHlist[pos].flags.container= 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"wielded"))           // Can it be wielded?
                    {
                    CHlist[pos].flags.wield= 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"tabletop"))           // Can you drop things
                    {                                   // On this solid object
                    CHlist[pos].flags.tabletop= 1;
                    CHlist[pos].flags.solid = 1;        // Make sure it's solid
                    something=1;
                    }

                if(!stricmp(Rptr,"spikeproof"))         // unaffected by triggers
                    {
                    CHlist[pos].flags.spikeproof= 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"person"))          // can't be stuffed in sacks etc
                    {
                    CHlist[pos].flags.person= 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"quantity"))        // Can it be a pile?
                    {
                    CHlist[pos].flags.quantity= 1;
                    something=1;
                    }

        // Get the character's speech file

		if(!stricmp(Rptr,"Conversation"))
			{
                        strcpy(CHlist[pos].funcs->talk,first(rest(line)));
                        CHlist[pos].funcs->tcache=1;
                        something=1;                    // Got something
			}

        // Get the character's default contents

		if(!stricmp(Rptr,"contains"))
			{
                        if(CHlist[pos].funcs->contents<8)
                            strcpy(CHlist[pos].funcs->contains[CHlist[pos].funcs->contents++],first(rest(line)));
                        else
                            Dump(ctr,"This character has more than 8 initial items",Rbuffer);
                        something=1;                    // Got something
			}

                if(!stricmp(Rptr,"fragile"))             // Is it fragile?
                    {
                    CHlist[pos].flags.fragile = 1;
                    something=1;
                    }

// Finish

	        if(!something && !skip_unk)
			Dump(ctr,"Unknown keyword:",Rptr);

                }
        }

/*
		if(!stricmp(Rbuffer,"overlay"))
			{
			strcpy(Rbuffer,first(rest(Sbuffer[ctr])));
			tmp=getnum4char(Rbuffer);
			if(tmp==-1)
				Dump(ctr,"Could not find character called:",Rbuffer);
			for(ctr2=0;ctr2<6;ctr2++)
				if(CHlist[CHtot].overlay[ctr2]==0 && tmp)
					{
					CHlist[CHtot].overlay[ctr2]=tmp;
//					printf("Entered overlay %x to character %s\n",tmp,CHlist[CHtot].name);
					tmp=0;
					}
				if(tmp!=0)
					Dump(ctr,"This character already has 6 overlays.",Rbuffer);
			something=1;
			}
*/

bootmsg("\n");
bootmsg("      Done.\n");
}

/*
 *      Descriptions - Compile the descriptive strings
 */

void descriptions(long start,long finish)
{
long ctr,pos;
char *Rptr;
char *line;

// The description list is just a simple list, of format:
//
// IDENTIFIER, TEXT ... .. .
// IDENTIFIER, TEXT ... .. .
//      :    ,    :
//      :    ,    :
//
// We get the first term as the identifier, and the rest of the line as the
// actual description text.  This compiler is exactly the same as sprites,
// but without calling the sprite loader.
//

// First, see how many we have.

bootmsg("    Section: descriptions\n");

DStot = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];        // Get the line
	     Rptr = first(rest(line));  // If a second word exists,
                                        // then the first word has to as well
        if(Rptr != NOTHING)             // and we assume we have the data
            DStot++;
        }

bootmsg("      %d descriptions\n", DStot);

// Allocate room on this basis

if(in_editor != 2)
    des_alloc = DStot;  // If not in scripter, use original value

DSlist = (D_TEXT *)M_get(sizeof(D_TEXT),des_alloc+1);

bootmsg("      %d bytes allocated\n", sizeof(D_TEXT)*(des_alloc+1));

// Now, process the data.  We will never parse this again, so it can be done
// destructively.

bootmsg("      Parsing");
Plot(DStot);
pos = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];
	Rptr=first(line);                       // Get first term
	if(Rptr!=NOTHING)                       // If it exists..
		{
		Rptr=first(rest(line));         // Get second term
		if(Rptr!=NOTHING)               // If it exists
			{
                        DSlist[pos].text = rest(line);       // Get whole line
                        Strip(DSlist[pos].text);             // Kill whitespc

                        // Now get the name as a pointer in the text Block

                        DSlist[pos].name= hardfirst(line);
                        Strip(DSlist[pos].name);

	                pos++;  // Go onto the next entry in the sprite list
			}
		}
	}

bootmsg("\n");
bootmsg("      Done.\n");
}

/*
 *      Tiles - Compile the tiles
 *              This compiler is based on characters()
 */

void tiles(long start,long finish)
{
int ctr,tmp,pos;
char *line,*Rptr;
char something;         // Something is used for error-checking.
                        // It starts as zero, set when a valid line is found.
                        // If there is no valid line, signal an error

bootmsg("    Section: tiles\n");

TItot = 0;

//      Count the instances of 'Name' to get the quantity of tiles.

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];        // Get the line
	Rptr = first(line);             // First see if it's blank
        if(!stricmp(Rptr,"name"))       // then look for the word 'Name'
            TItot++;                    // Got one
        }

bootmsg("      %d tiles defined\n", TItot);

// Allocate room on this basis
if(in_editor != 2)
    til_alloc = TItot;  // If not in scripter, use original value

TIlist = (TILE *)M_get(sizeof(TILE),til_alloc+1);

bootmsg("      %d bytes allocated\n", sizeof(TILE)*(til_alloc+1));

// Now, process the data.  We will never parse this again, so it can be done
// destructively.

bootmsg("      Parsing");
Plot(TItot);
pos = -1;

for(ctr=start;ctr<=finish;ctr++)
	{
        something=0;                            // Assume nothing is there
        line = script.line[ctr];
	Rptr=first(line);                       // Get first term
	if(Rptr!=NOTHING)                       // Main parsing is here
		{

        // Get the name

                if(!stricmp(Rptr,"name"))       // Got the first one
                    {
	            Rptr=first(rest(line));     // Get second term
	            if(Rptr!=NOTHING)           // Make sure its there
                        {
                        // Find the offset in the block

                        pos ++;                             // Pre-increment
                        Plot(0);
                        TIlist[pos].name = rest(line);      // Get name label

                        Strip(TIlist[pos].name);            // Kill whitespace
                        if(strlen(TIlist[pos].name) > 31)   // Check name length
                            Dump(ctr,"This name exceeds 31 letters",TIlist[pos].name);
			something=1;                        // Got something
                        }
                    else
                        Dump(ctr,"NA: This tile has no name",NULL);
                    }

        // Sanity Check

                if(pos == -1)
                        Dump(ctr,"NA: The tile's name must come first",NULL);

        // Get the sequence of the tile

		if(!stricmp(Rptr,"sequence"))
			{
			strcpy(Rbuffer,first(rest(line)));
			tmp=getnum4sequence(Rbuffer);
			if(tmp==-1)
				Dump(ctr,"Could not find sequence:",Rbuffer);

                        // Get seq. directly
                        TIlist[pos].form=&SQlist[tmp];
                        // Get name of sequence too
                        strcpy(TIlist[pos].seqname,first(rest(line)));

			something=1;                    // Got something
			}

        // Get the description from the section: DESCRIPTION

		if(!stricmp(Rptr,"description"))
			{
			strcpy(Rbuffer,first(rest(line)));

                        if(Rbuffer[0] == '\"')  // Quoted text is direct
                            {
                            TIlist[pos].desc = strchr(line,'\"'); // start
                            TIlist[pos].desc++;            // skip the quote
                            Rptr = strrchr(line,'\"');     // Find last quote
                            if(Rptr)
                                *Rptr = 0;                 // blow quote away
                            }
                        else                    // Look up unquoted text
                            {                   // in the DESCRIPTION bank
			    tmp=getnum4description(Rbuffer);
			    if(tmp==-1)
				Dump(ctr,"Could not find this entry in section: DESCRIPTION",Rbuffer);
                            TIlist[pos].desc=DSlist[tmp].text;  // Get it
                            }
                        something=1;                    // Got something
			}

        // Get the flags

                if(!stricmp(Rptr,"solid"))             // Is it solid?
                    {
                    TIlist[pos].flags.solid = 1;
                    something=1;
                    }

                if(!stricmp(Rptr,"blocklight") ||
                   !stricmp(Rptr,"blockslight"))        //Is it solid?
                    {
                    TIlist[pos].flags.blocklight= 1;
                    something=1;
                    }

// Finish

	        if(!something && !skip_unk)
			Dump(ctr,"Unknown keyword:",Rbuffer);

                }
        }

bootmsg("\n");
bootmsg("      Done.\n");
}

/*
 *      Code - Load in the VRMS
 *           this compiler is based on sprites()
 */

void code(long start,long finish)
{
long ctr,pos;
char *Rptr;
char *line;

// The VRM list is just a simple list, of format:
//
// FUNCTION, FILENAME
// FUNCTION, FILENAME
//      :    ,    :
//      :    ,    :
//
// Therefore it is sufficient to just get the first two terms and store them.
//

// First, see how many VRM functions we have.

bootmsg("    Section: code\n");

COtot = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];        // Get the line
	Rptr = first(rest(line));       // If the second word exists,
                                        // then the first word has to as well
        if(Rptr != NOTHING)             // and we assume we have the data
            COtot++;
        }

bootmsg("      %d virtual runtime modules\n", COtot);

// Allocate room on this basis

if(in_editor != 2)
    vrm_alloc = COtot;  // If not in scripter, use original value
COlist = (VRM *)M_get(sizeof(VRM),vrm_alloc+1);

bootmsg("      %d bytes allocated\n", sizeof(VRM)*(vrm_alloc+1));

// Now, process the data.  We will never parse this again, so it can be done
// destructively.

bootmsg("      Parsing");
Plot(COtot);
pos = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];
	Rptr=first(line);                       // Get first term
	if(Rptr!=NOTHING)                       // If it exists..
		{
		Rptr=first(rest(line));         // Get second term
		if(Rptr!=NOTHING)               // If it exists
			{
                        COlist[pos].fname = rest(line);      // Get filename
                        Strip(COlist[pos].fname);            // Kill whitespc

                        // Now get the name as a pointer in the text Block

                        COlist[pos].name= hardfirst(line);

                        // Initialise this VRM, but not in the editor

                        if(!in_editor)
                            Init_VRM(pos);

			pos++;  // Go onto the next entry in the VRM list
			}
		}
	}

bootmsg("\n");
bootmsg("      Done.\n");
}

/*
 *      Sounds - Load in the Sounds
 *               this compiler is based on sprites()
 */

void sounds(long start,long finish)
{
long ctr,pos;
char *Rptr;
char *line;

// The WAVE list is just a simple list, of format:
//
// NAME, FILENAME
// NAME, FILENAME
//   : ,    :
//   : ,    :
//
// Therefore it is sufficient to just get the first two terms and store them.
//

//if(in_editor) return;

// First, see how many sounds we have.

bootmsg("    Section: sounds\n");


Waves = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];        // Get the line
	Rptr = first(rest(line));       // If the second word exists,
                                        // then the first word has to as well
        if(Rptr != NOTHING)             // and we assume we have the data
            Waves++;
        }

bootmsg("      %d wave files\n", Waves);

// Allocate room on this basis

if(in_editor != 2)
    wav_alloc = Waves;  // If not in scripter, use original value
//wavtab = (SMTab *)M_get(sizeof(SMTab),Waves+1);
wavtab = (SMTab *)M_get(sizeof(SMTab),wav_alloc+1);

bootmsg("      %d bytes allocated\n", sizeof(SMTab)*(Waves+1));

// Now, process the data.  We will never parse this again, so it can be done
// destructively.

bootmsg("      Parsing");
Plot(Waves);
pos = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];
	Rptr=first(line);                       // Get first term
	if(Rptr!=NOTHING)                       // If it exists..
		{
		Rptr=first(rest(rest(line)));   // Get third term
		if(Rptr!=NOTHING)               // If it exists
                        if(!stricmp("nodrift",first(Rptr)))
                            wavtab[pos].nodrift = 1;

		Rptr=first(rest(line));         // Get second term
		if(Rptr!=NOTHING)               // If it exists
			{
                        wavtab[pos].fname = hardfirst(rest(line));      // Get filename
                        Strip(wavtab[pos].fname);            // Kill whitespc

                        // Now get the name as a pointer in the text Block
                        wavtab[pos].name= hardfirst(line);
			pos++;  // Go onto the next entry in the WAV list
			}
		}
	}

bootmsg("\n");
bootmsg("      Done.\n");
}

/*
 *      Music - Load in the music
 *              this compiler is based on sprites()
 */

void music(long start,long finish)
{
long ctr,pos;
char *Rptr;
char *line;

// The MUSIC list is just a simple list, of format:
//
// NAME, FILENAME
// NAME, FILENAME
//   : ,    :
//   : ,    :
//
// Therefore it is sufficient to just get the first two terms and store them.
//

// First, see how many mods we have.

//if(in_editor) return;

bootmsg("    Section: music\n");

Songs = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];        // Get the line
	Rptr = first(rest(line));       // If the second word exists,
                                        // then the first word has to as well
        if(Rptr != NOTHING)             // and we assume we have the data
            Songs++;
        }

bootmsg("      %d music files\n", Songs);

// Allocate room on this basis

if(in_editor != 2)
    mod_alloc = Songs;  // If not in scripter, use original value
modtab = (SMTab *)M_get(sizeof(SMTab),mod_alloc+1);

bootmsg("      %d bytes allocated\n", sizeof(SMTab)*(Songs+1));

// Now, process the data.  We will never parse this again, so it can be done
// destructively.

bootmsg("      Parsing");
Plot(Songs);
pos = 0;

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];
	Rptr=first(line);                       // Get first term
	if(Rptr!=NOTHING)                       // If it exists..
		{
		Rptr=first(rest(line));         // Get second term
		if(Rptr!=NOTHING)               // If it exists
			{
                        modtab[pos].fname = rest(line);      // Get filename
                        Strip(modtab[pos].fname);            // Kill whitespc

                        // Now get the name as a pointer in the text Block

                        modtab[pos].name= hardfirst(line);
			pos++;  // Go onto the next entry in the MOD list
			}
		}
	}

bootmsg("\n");
bootmsg("      Done.\n");
}

/*
 *      rooftiles - Load in sprites to be used as roof tiles
 */

void rooftiles(long start,long finish)
{
long ctr,pos,imagedata;
char *Rptr;
char *line;

// The rooftiles list is just a single list of sprite files:
//
// FILENAME
// FILENAME
//      :
//      :
//
// Therefore it is sufficient to just get the first term and store it.
//

// First, see how many sprites we have.

bootmsg("    Section: rooftiles\n");

RTtot = 1; // 0 is special (no roof tile)

for(ctr=start;ctr<=finish;ctr++)
	{
        line = script.line[ctr];        // Get the line
	Rptr = first(line);             // If the second word exists,
                                        // then the first word has to as well
        if(Rptr != NOTHING)             // and we assume we have the data
            RTtot++;
        }

bootmsg("      %d rooftile sprites\n", SPtot);

// Allocate room on this basis

if(in_editor == 2)
    bootmsg("      Allocating space for %d entries\n", rft_alloc);
else
    rft_alloc = RTtot;

RTlist = (S_POOL *)M_get(sizeof(S_POOL),rft_alloc+1);
bootmsg("      %d bytes allocated\n", sizeof(S_POOL)*(rft_alloc+1));

// Now, process the data.  We will never parse this again, so it can be done
// destructively.

bootmsg("      Decoding");
Plot(RTtot);
pos = 1; // 0 is special (no roof tile)
imagedata=0;

for(ctr=start;ctr<=finish;ctr++)
	{
        // Stuff an integer 1 into a pointer slot to use as a flag later
        RTlist[pos].name=(char *)1L;
        line = script.line[ctr];
        Rptr=first(rest(line));
        if(!stricmp(Rptr,"stand_under"))
              RTlist[pos].name=(char *)0L; //..and again..
	Rptr=first(line);                       // Get first term
	if(Rptr!=NOTHING)                       // If it exists..
		{
                RTlist[pos].fname = hardfirst(line);     // Get filename
                imagedata += Init_RoofTile(pos);
		pos++;  // Go onto the next entry in the sprite list
		}

	}

bootmsg("\n");
bootmsg("      Done.  %d bytes of image data allocated\n",imagedata);
}


/*
 *      Restring - Turn all the static strings from compiled script
 *                 into dynamically-allocated strings (that can be lengthened)
 */

void Restring()
{
int ctr;
char buf[MAX_LINE_LEN];

#define RESTR_MACRO(x) {strcpy(buf,x);\
                       strupr(buf);\
                       x=(char *)M_get(MAX_LINE_LEN,1);\
                       strcpy(x,buf);}

#define I_RESTR_MACRO(x) {strcpy(buf,x);\
                       x=(char *)M_get(MAX_LINE_LEN,1);\
                       strcpy(x,buf);}

for(ctr=0;ctr<SPtot;ctr++)
    {
    RESTR_MACRO(SPlist[ctr].name);
    RESTR_MACRO(SPlist[ctr].fname);
    }

for(ctr=0;ctr<SQtot;ctr++)
    RESTR_MACRO(SQlist[ctr].name);

for(ctr=0;ctr<Songs;ctr++)
    {
    RESTR_MACRO(modtab[ctr].name);
    RESTR_MACRO(modtab[ctr].fname);
    }

for(ctr=0;ctr<CHtot;ctr++)
    {
    RESTR_MACRO(CHlist[ctr].name);
    I_RESTR_MACRO(CHlist[ctr].desc);
    I_RESTR_MACRO(CHlist[ctr].shortdesc);
    }

for(ctr=0;ctr<Waves;ctr++)
    {
    RESTR_MACRO(wavtab[ctr].name);
    RESTR_MACRO(wavtab[ctr].fname);
    }

for(ctr=0;ctr<COtot;ctr++)
    {
    RESTR_MACRO(COlist[ctr].name);
    RESTR_MACRO(COlist[ctr].fname);
    }

for(ctr=0;ctr<TItot;ctr++)
    {
    RESTR_MACRO(TIlist[ctr].name);
    I_RESTR_MACRO(TIlist[ctr].desc);
    }

}

/*
 *      GetVrmName - Return a pointer to the VRM's name string
 */

char *GetVrmName(int num)
{
if(num>0 && num <COtot)
    return COlist[num].name;
return NULL;
}


/*
 *      Getnum4char - Find the index of the character in the CHlist array
 */

int getnum4char(char *name)
{
int ctr;
for(ctr=0;ctr<CHtot;ctr++)
	if(!stricmp(name,CHlist[ctr].name))
		return ctr;
return -1;
}

/*
 *      Getnum4VRM - Find the index of the VRM in the COlist array
 */

int getnum4VRM(char *name)
{
int ctr;
for(ctr=0;ctr<COtot;ctr++)
	if(!stricmp(name,COlist[ctr].name))
		return ctr;
return -1;
}

/*
 *      Getnum4description - Find the index of the description in the array
 */

int getnum4description(char *name)
{
int ctr;
for(ctr=0;ctr<DStot;ctr++)
	if(!stricmp(name,DSlist[ctr].name))
		return ctr;
return -1;
}

/*
 *      Getnum4sprite - Find the index of the sequence in the SPlist array
 */

int getnum4sprite(char *name)
{
int ctr;
for(ctr=0;ctr<SPtot;ctr++)
	if(!stricmp(name,SPlist[ctr].name))
		return ctr;
return -1;
}

/*
 *      Getnum4sequence - Find the index of the sequence in the SQlist array
 */

int getnum4sequence(char *name)
{
int ctr;

for(ctr=0;ctr<SQtot;ctr++)
	if(!stricmp(name,SQlist[ctr].name))
		return ctr;

return -1;

}

/*
 *      Getnumber - Return the numeric value of a string
 *                  Support for constants can be re-added later if needed
 */

int getnumber(char *number)
{
int ctr;

ctr=atoi(number);
return ctr;
}

/*
 *      Isnumber - is the string a number?
 */

int isnumber(char *thing)
{
int ctr,Xc;

Xc=strlen(thing);
for(ctr=0;ctr<Xc;ctr++)
	if((thing[ctr]<'0'||thing[ctr]>'9')&&thing[ctr]!='-'&&thing[ctr]!='+')
		return NULL;
return 1;
}

/*
 *      isXspace - is the character under scrutiny whitespace or not?
 */

char isxspace(unsigned char input)
{
char res;
res=0;
res=isspace(input);
if(input<=32)
	res=1;
return res;
}

/*
 *      rest - A LISP string processing routine.
 *             rest returns all but the first word of a string.
 *             e.g. rest("1 2 3") returns "2 3"
 */

char *rest(char *input)
{
short ctr,len,ptr,ptr2,ptr3;

if(input==NULL)
	return NOTHING;

len=strlen(input);
ptr=-1;
for(ctr=0;ctr<len;ctr++)
	if(!isxspace(input[ctr]))
		{
		ptr=ctr;
		break;
		}
if(ptr==-1)
	return NOTHING;

ptr2=-1;
for(ctr=ptr;ctr<len;ctr++)
	if(isxspace(input[ctr]))
		{
		ptr2=ctr;
		break;
		}

if(ptr2==-1)
	return NOTHING;

ptr3=-1;
for(ctr=ptr2;ctr<len;ctr++)
	if(!isxspace(input[ctr]))
		{
		ptr3=ctr;
		break;
		}
if(ptr3==-1)
	return NOTHING;

return &input[ptr3];
}

/*
 *      first - A LISP string processing routine.
 *              first returns only the first word of a string.
 *              e.g. first("1 2 3") returns "1"
 *
 *              first returns a copy of the string, it does not modify the
 *              original string.  Use hardfirst() if you want this to happen.
 */

char *first(char *input)
{
short ctr,len,ptr;
char tbuf[MAX_LINE_LEN];

if(input==NULL)
	return NOTHING;

strcpy(tbuf,input);
len=strlen(input);
ptr=-1;
for(ctr=0;ctr<len;ctr++)
	if(!isxspace(input[ctr]))
		{
		ptr=ctr;
		break;
		}
if(ptr==-1)
	return NOTHING;

for(ctr=ptr;ctr<len;ctr++)
	if(isxspace(input[ctr]))
		{
		tbuf[ctr]=NULL;
		break;
		}

return &tbuf[ptr];
}

/*
 *      hardfirst - Like 'first' but modifies the original string.
 */

char *hardfirst(char *line)
{
char *ptr;
char *out;

ptr = first(line);

// got the first item, now search for it to find the address

out = strstr(line,ptr);
if(!out)
    panic("Oh no!","The string vanished",ptr);

ptr = rest(out);        // Got beginning of second item
if(ptr != NOTHING)
    *(ptr-1) = 0;       // Wind back, and punch a hole

// We now have the name, but first we must strip off
// all the whitespace or the stricmp will fail later

Strip(out);

return out;
}

/*
 *      last - return last item
 *             e.g. last("1 2 3") returns "3"
 */

char *last(char *a)
{
char tvb[MAX_LINE_LEN];
char tvb2[MAX_LINE_LEN];
char *p,*pr;
strcpy(tvb,a);

// Whittle down the array until the end, but keeping a buffer, pr, one behind

//pr = tvb;
p = a;
do    {
      pr = p;
      p = rest(tvb);
      strcpy(tvb2,p);
      strcpy(tvb,tvb2);
      } while(p != NOTHING);

// pr is now the last entry, or possibly NOTHING.
// Search for its position in the real string

p = strstr(a,pr);

// If the last item is not found, the first item should be the last item

if(!p)
    p = a;

// Return what we found

return p;
}

/*
 *      words - Return the number of words in the sentence
 */

int words(char *a)
{
int rc;
char tvb[MAX_LINE_LEN];
char tvb2[MAX_LINE_LEN];
char *p;
strcpy(tvb,a);
rc = 0;
do    {
      p =rest(tvb);
      strcpy(tvb2,p);
      strcpy(tvb,tvb2);
      rc++;
      } while(p != NOTHING);
return rc;
}

/*
 *      Strip - Shorten a string to remove any trailing spaces
 */

void Strip(char *tbuf)
{
char tvb[MAX_LINE_LEN];
char *tptr;
int before;

// Sanity checks first

if(tbuf == NOTHING || tbuf == NULL)
    return;                     // There is nothing there!  abort

// Check for too short string

if(strlen(tbuf) <= 1)
    return;                     // There is nothing there!  abort

// Check for just a lot of spaces

if(words(tbuf) == 0)            // how many words?
    return;                     // There is nothing there!  abort

// Find first non-space character

for(tptr = tbuf;*tptr == ' ';tptr++);

// Make sure there was something

if(!*tptr)
    return;

// Tptr is now the beginning of the 'pure' string
// We do all the processing on a copy, so we can abort if necessary

strcpy(tvb,tptr);
before = words(tvb);                    // Get number of words before

// Find and erase spaces

if(strchr(last(tvb),' '))
    *(strchr(last(tvb),' ')) =0;

// If the number of words after is DIFFERENT, something wasn't right

if(words(tvb) != before)
    return;                     // Only one item.. don't bother

// Now we copy the modified one onto the real string

strcpy(tbuf,tvb);

// All done
}

/*
 *      Dump - Crash out and show a fragment of the code
 */

void Dump(long lineno,char *error,char *help)
{
char errbuf[1024];
char errstr[1024];
short ctr,sl,c2,c2m8;
long l2,Lctr;
V_off();
printf("\n\n");
SYS_TEXTCOLOR(GREEN);
sprintf(errstr,"Error in %s at line %ld: %s\r\n",compilename,lineno,error);
SYS_CPRINT(errstr);

l2=lineno+5;
if(l2>script.lines)
	l2=script.lines;
SYS_TEXTCOLOR(RED);
for(Lctr=lineno;Lctr<l2;Lctr++)
	if(script.line[Lctr]!=NOTHING&&script.line[Lctr]!=NULL)
	{
	c2=0;
	sl=strlen(script.line[Lctr]);
	for(ctr=0;ctr<sl;ctr++)
		if(script.line[Lctr][ctr]=='\t')
			{
			c2m8=c2%8;
			if(!c2m8)
				c2m8=8;
			for(;c2m8>0;c2m8--)
				errbuf[c2++]=' ';
			}
		else
			errbuf[c2++]=script.line[Lctr][ctr];
	errbuf[c2++]=0;
	sprintf(errstr,"%6ld: %s\r\n",Lctr,errbuf);
        SYS_CPRINT(errstr);
	}
SYS_TEXTCOLOR(GREEN);
if(help)
        {
	sprintf(errstr,"\r\n%s\r\n",help);
        SYS_CPRINT(errstr);
        }
SYS_TEXTCOLOR(LIGHTGRAY);
SYS_CPRINT(" \n");         // Stop the cursor from printing green text
exit(1);
}

/*
 *      Purify - Remove comments from the script to ease compilation
 */

void Purify()
{
int ctr,len,pos;
char *ptr;

// Scan each line for comments

for(ctr=0;ctr<script.lines;ctr++)
    {
    ptr = strchr(script.line[ctr],'#');         // Look for the hash
    if(ptr)
           *ptr=0;                              // make EOL if we found one

    ptr = strchr(script.line[ctr],10);          // Look for the LF
    if(ptr)
           *ptr=0;                              // make EOL if we found one
    }

// Scan each line for whitespace and turn it into space

for(ctr=0;ctr<script.lines;ctr++)
    {
//    strupr(script.line[ctr]);
    len = strlen(script.line[ctr]);    // Got length of line
    for(pos=0;pos<len;pos++)
        if(isxspace(script.line[ctr][pos]))
                script.line[ctr][pos] = ' ';    // Whitespace becomes space

    Strip(script.line[ctr]);
    }


return;
}


/*
 *      SeekSect - Locate each discrete section
 */

void SeekSect()
{
int ctr;
char *ptr;

no_of_sections = 0;

for(ctr=0;ctr<script.lines;ctr++)
	{
//	strupr(script.line[ctr]);               // Convert to uppercase
	strcpy(Rbuffer,script.line[ctr]);       // Get line
	ptr=first(Rbuffer);                     // Get pointer to line
	if(ptr)
		if(!stricmp(ptr,"SECTION:"))
                        {
			sect_map[no_of_sections++].end = ctr-1;
			sect_map[no_of_sections].start = ctr;
                        strcpy(sect_map[no_of_sections].name,first(rest(Rbuffer)));
                        }
        }
sect_map[no_of_sections++].end = script.lines-1;
}

