/************************************************************************
       -= SEER - C-Scripting engine v 0.9a =-
State      : under implementation
Author     : Przemyslaw Podsiadly
             email: ppodsiad@elka.pw.edu.pl
             WWW:   http://home.elka.pw.edu.pl/~ppodsiad
File:seer.h
Desc:main header file
************************************************************************/
#ifndef __SEER_H__

#include <stdarg.h>

#ifdef __cplusplus
extern "C" {
#endif
#ifdef WIN32
//*****************************VC
#ifdef INTERNAL_FILE
 #define EXTERN __declspec(dllexport)
#else
 #define EXTERN __declspec(dllimport)
#endif

#else
//*****************************GCC
 #define EXTERN extern
#endif

/**************************  SeeR INFO ***********************************/
#define SeeR_VERSION          92 /* SeeR_VERSION/100.0 */
#define SeeR_VERSION_STR      "0.92a"
#define SeeR_DATE_STR         "1999-03-07"
#define SeeR_DATE             19990307    /* yyyymmdd */

/*
   Here you can check some information on current state of SeeR.
   e.g.
   printf("%d lines compiled.\n",scSeeR_Info(scInfo_LinesCompiled));
*/
#define scInfo_Version            0 /* Returns SeeR_VERSION */
#define scInfo_Build              1 /* Build version, returns: */
#define scInfo_Build_Release      1 /* SeeR may not debug, but is fastest */
#define scInfo_Build_Debug        2 /* if SeeR may generate debug files */
#define scInfo_LinesCompiled      2 /* Number of lines compiled recently */
int scSeeR_Info(int,...); //these "..." reserved for future parameters


/**************************Data definitions*******************************/
EXTERN char scScript_Title[128],scScript_Author[128],*scScript_SeeR;
EXTERN int (*scProgressCallback)(int percent);
#ifndef __cplusplus
typedef enum bool {false,true} bool;
#endif
typedef char *scScript;

typedef struct scFlags {
        int priority;
        int forkno;//0 if not a forked instance
} scFlags;

typedef struct scInstance {
        scScript code;
        char *data_stack;//here, first 256 ints are registers,then data, then stack
        scFlags flags;
        struct scInstance *forks,*next_fork,*forked;
} scInstance;

typedef struct scProcess{
         scInstance* inst;
         int speed;
         int paused;
         struct scProcess* next;
       } scProcess;

typedef struct scScheduler{
         scProcess* first;
       } scScheduler;

/**************************Errors*****************************************/

enum {
  scOK=0,
  scErr_Parse=1,
  scErr_Declaration=2,
  scErr_CodeGen=3,
  scErr_Operand=4,
  scErr_Cast=5,
  scErr_Runtime=101,
  scErr_Violation=102,
  scErr_Internal=255
};

EXTERN int scErrorNo;
EXTERN char scErrorMsg[255];
EXTERN char scErrorLine[256];
//throw an error connected with scripts:
void scRuntime_Error(char *er, ...);

/**************************Instances**************************************/
scInstance* scCreate_Instance(scScript,char*,...);
void scFree_Instance(scInstance *sci);
scInstance* scGet_This();//returns scActual_Instance
EXTERN scInstance *scActual_Instance;

/**************************Compilation************************************/
scScript scCompile_File(char *);
scScript scCompile_Text(char *);
scScript scLoad_Script(char *);

/**************************Runing and conversation************************/
int scStart_Instance(scInstance *,...);// calls main(...) in script
int scGet_Script_Size(scScript);
int scCall_Instance(scInstance*,int address,...);
int scVCall_Instance(scInstance* inst,int address,int paramc,int *params);
                                 //paramc is size of params in bytes!
int scGet_Symbol(scInstance*,char *);

#define scVar(inst,adr,typ) (*((typ*)((inst)->data_stack+256*4+(adr))))

/**************************Externals**************************************/

void scAdd_Internal_Header(char *,char *);
void scAdd_External_Symbol(char *,void *);
#define scAddExtSym(a) scAdd_External_Symbol(#a,&a);
void scExport_Kernel();

/**************************MULTITASKING**********************************/
scScheduler* scGet_Scheduler();
scScheduler* scSet_Scheduler(scScheduler*);//returns previous scheduler,
                                   //if you pass NULL, it'll create a new one

int scVLaunch_Instance(scInstance* inst,int spd,int address,int paramc,int *params);
                                        //paramc is size of params in bytes!
int scLaunch_Instance(scInstance* inst,int spd,int address,...);
int scContinue_Instances(void);//returns number of instances remaining to run

void scPause_Instance(scInstance*,int);//pause if (int)
void scKill_Instance(scInstance*);

scInstance* scGet_Forked(scInstance*,int);
scInstance* scFork_Instance(scInstance*);
void scKill_Forked_Instances(scInstance*);

/****************Priorities**********************************************/
//can call all imported kernel API function
#define scrights_KERNEL 256
//cannot ---
#define scrights_USER     0
//when no instance is being run:
#define scrights_TOP MAXINT

/****************Status of instance**************************************/
#define scstatus_FREE    0
#define scstatus_RUNNING 1
#define scstatus_PAUSED  2
int scGet_Instance_Status(scInstance *);

/*****************Operating on Actual_Instance***************************/
scInstance* scGet_Actual_Forked(int);
int scGet_Forked_Status(int);
void scKill_My_Forked_Instances();
void scKill_Forked(int);

/*****************Titles and Author info ********************************/
char* scGet_Title(scScript);
char* scGet_Author(scScript);
char* scGet_Instance_Title(scInstance*);

/*********************DEBUGGING******************************************/
void scToggle_Debug(bool debon);
void scOpen_Debug(char *filename);
void scClose_Debug();//also turns debuggin off

/*********************PRIORITIES*****************************************/
EXTERN bool scIgnore_Rights;
void scSet_Priority(scInstance*, int priority_level);
int scGet_Priority(scInstance*);//if NULL returns scNONE
int scGet_Actual_Priority();

//These macros are to check if an instance calling a function containing
//them has at least required priority - if not it'll set
//an error and return rets (if function returns void, rets should be `;')
#define scPriorityGuard(priority,rets) \
        if ((scGet_Actual_Priority()<priority)&&!scIgnore_Rights)\
        {\
        scRuntime_Error("Priority violation at instance `%s'",\
        scGet_Instance_Title(scActual_Instance));\
        scErrorNo=scErr_Violation; return rets;\
        }

#define scKernelOnly(rets) scPriorityGuard(scrights_KERNEL,rets);

#ifdef __cplusplus

/*********************C++ INTERFACE**************************************/
enum {scriptLoad,scriptCompileFile,scriptCompileText};

class seerScript{
 protected:
  scScript script;
 public:
  seerScript(int a,char *text):script(NULL)
    {switch(a)
     {
      case scriptLoad       :script=scLoad_Script(text);break;
      case scriptCompileText:script=scCompile_Text(text);break;
      case scriptCompileFile:script=scCompile_File(text);break;
     }
    }
  ~seerScript()
    {if (script) free(script);}
  int Size() {return scGet_Script_Size(script);}
  char* Title() {return scGet_Title(script);}
  char* Author() {return scGet_Author(script);}
  scScript Get() {return script;}
};

class seerInstance{
 protected:
  scInstance *ins;
 public:
  seerInstance()
    {ins=NULL;}
  seerInstance(scScript sc)
    {ins=scCreate_Instance(sc,"");}
  seerInstance(scInstance* i)
    {ins=i;}
  seerInstance(seerInstance& i) //make fork
    {ins=scFork_Instance(i.ins);}
  ~seerInstance()
    {scFree_Instance(ins);}
  scInstance *Get() {return ins;}
  int Symbol(char *s)
    {return scGet_Symbol(ins,s);}
  int Status()
    {return scGet_Instance_Status(ins);}
  char* Title()
    {return scGet_Instance_Title(ins);}
  int Start()
    {return Call("main");}
  int Create(seerScript& sc)
    {return (int)(ins=scCreate_Instance(sc.Get(),""));}
  int Call(int addr,...)
    {
     va_list va;
     va_start(va,addr);
     return scVCall_Instance(ins,addr,-1,&(va_arg(va,int)));
            //-1 - default number of parameters
    }
  int Call(char *symbol,...)
    {
     va_list va;
     va_start(va,symbol);
     return scVCall_Instance(ins,Symbol(symbol),-1,&(va_arg(va,int)));
    }
  int VCall(int address,int paramc,int *params)
    {
     return scVCall_Instance(ins,address,paramc,params);
    }
  int VLaunch(int spd,int address,int paramc,int *params)
    {
    return scVLaunch_Instance(ins,spd,address,paramc,params);
    }
  int Launch(int spd,int addr,...)
    {
     va_list va;
     va_start(va,addr);
     return scVLaunch_Instance(ins,spd,addr,-1,&(va_arg(va,int)));
    }
  int Launch(int spd,char *symbol,...)
    {
     va_list va;
     va_start(va,symbol);
     return scVLaunch_Instance(ins,spd,Symbol(symbol),-1,&(va_arg(va,int)));
    }
  void Pause(int p)
    {
    scPause_Instance(ins,p);
    }
  void Kill()
    {
    scKill_Instance(ins);
    }
  scInstance* Forked(int i)
    {return scGet_Forked(ins,i);}
  void Kill_Forked()
    {scKill_Forked_Instances(ins);}
};

}
#endif

#define __SEER_H__
#endif
