//
//      Sound/Timer engine wrappers
//

//      Portability notes
//
//      This contains code and data specific to the MIDAS sound library,
//      with some of my own extensions added.
//
//      However, to the game engine it is a black-box module.
//
//      For other platforms, it should be fairly straightforward from
//      the game's point of view.
//
//      The version I used (1.1.2) can be compiled for Linux and Win32,
//      and the modified file should plug in easily.
//
//      For systems other than DOS, the timer mechanism could be different.
//      Just make sure that the animation counter and flags are handled at
//      140Hz, with the animation timer updating at 35Hz.
//


// MR: does not use this
#ifndef _WIN32
#include <unistd.h>     // This is for sleep, in the timer check routine
#endif
#include <stdlib.h>     // atexit(), random
#include <string.h>     // String splicing routines
#ifdef __DJGPP__
#include <pc.h>         // outportb
#endif

    #include "soundsys/midasdll.h"                  // Midas 1.1.2
extern "C" void midasFileBase(long io,long len);           // My crude hack in MIDAS

/*
extern "C"      // This comes first to prevent naming collisions
    {
    #include "soundsys/midasdll.h"                  // Midas 1.1.2
    void midasFileBase(long io,long len);           // My crude hack in MIDAS
    }
*/
#include "doslib.hpp"
#include "sound.hpp"    // Us
#include "memory.hpp"   // Memory system
#include "console.hpp"  // Graphical console and logger
#include "fs.hpp"       // Filesystem routines
#include "vidmodes.h"

// defines

#define TIMER_INT 8     // Reprogram int 8
#define QUEUE_RATE 32   // Refresh sound queue every 32 clock cycles


// Lock routines borrowed from the ALLEGRO library

#define END_OF_FUNCTION(x)    void x##_end() { }
#define LOCK_VARIABLE(x)      _go32_dpmi_lock_data((void *)&x, sizeof(x))
#define LOCK_FUNCTION(x)      _go32_dpmi_lock_code(x, (long)x##_end - (long)x)


// variables

extern char mus_chan;                      // Number of music channels
extern char sfx_chan;                      // Number of effects channels
extern int  driftlevel;                    // Amount of frequency variation

static int sf_volume = 63;
static int mu_volume = 0;

static char string[32];    // For string splicing
static char T_Running=0;   // Is the timer running?
static char S_Running=0;   // Is the sound running?
static MIDASmodule **song; // dynamic array of songs
static MIDASsample *wave;  // dynamic array of samples
static MIDASmodulePlayHandle curplay=0;
int Songs=0;
int Waves=0;
static int pausedata=0;   // Speed of song before pausing, 0 if not paused
static int cursong=-1;

struct SMTab *wavtab;       // Tables of data for each sound and song.
struct SMTab *modtab;       // Filled by script.cc

// functions

void S_Init();              // Set up sound device
void S_Load();              // Load the sound and music
void S_Start();             // Start the mixing kernel
void S_Term();              // Stop the mixing kernel and shut down
void S_SoundVolume();       // Set the Sound volume
void S_MusicVolume();       // Set the Music volume
void S_PlaySample(int s);   // Play a sound sample
void S_PlayModule(int s);   // Play a music module
void S_StopModule();        // Stop the music
char S_Sync();              // Make sure the timer is working

void T_Init();              // Animation timer
void T_Term();

//static void S_Tickhandler(void); // Mixer refresh
static void LoadMods();
static void LoadWavs();
static void MIDASerror(void);                   // Panic with a MIDAS message
static int  GetWavFreq(FILE *fp);               // Get frequency

// Interrupt handler from ALLEGRO
#ifdef __DJGPP__
extern "C" int _install_irq(int num, int (*handler)());
extern "C" void _remove_irq(int num);
#endif
extern "C" int T_Midas_Init();
extern "C" void T_Midas_Term();

// Public code

/*
 *      S_Init - Load in the soundcard settings and set up the device
 */

void S_Init()
{
if(S_Running)          // Safety valve
    return;

// Announce the sound library

bootmsg("S_Init()\n");

bootmsg("  Midas Sound system 1.12i - Copyright 1998 Housemarque\n");

bootmsg("  MIDASstartup\n");

MIDASstartup();

bootmsg("  MIDASconfig\n");

#ifdef __DJGPP__
if(!MIDASloadConfig("audio.cfg"))       // Get it
    if ( MIDASgetLastError())
       {
         V_off();
         boot0("Sound has not been properly configured.\n");
         boot0("Please run SETUP.EXE and try again.\n");
         boot0("The MIDAS sound engine says:");
         InGFX=1;   // Prevent screen clear
         MIDASerror();
         exit(1);
       }
#endif
    /* Decrease the size of buffer used: (not needed in DOS, the default
       buffers there are already much smaller) */
#ifndef __DJGPP__
    MIDASsetOption(MIDAS_OPTION_MIXBUFLEN, 150);
    MIDASsetOption(MIDAS_OPTION_MIXBUFBLOCKS, 4);
#endif    

bootmsg("  StartSys\n");

    /* Initialize MIDAS and start background playback (at 140 polls
       per second): */
    if ( !MIDASinit() )
        MIDASerror();

bootmsg("  Init successful\n");

S_Running=1;                    // Right! Let's go...
}


/*
 *      S_Start - Start the actual sound output
 */

void S_Start()
{
cfa = "S_Start";
boot2("S_Start()\n");

if ( !MIDASstartBackgroundPlay(140) )
    MIDASerror();

/* Open all channels: */
if ( !MIDASopenChannels( mus_chan + sfx_chan ))
    MIDASerror();

/* Allocate a part of the channels for use as automatic sound effect
   channels: */
if ( !MIDASallocAutoEffectChannels(sfx_chan) )
    MIDASerror();

if(!T_Midas_Init())
    MIDASerror();

//mp_volume = 100;
}


/*
 *      S_Term - Stop the sound process
 */

void S_Term()
{
cfa = "S_Term";
if(!S_Running)          // Safety valve
    return;

boot2("S_Term()\n");

// Stop background playing

boot2("stopbgplay\n");   // Shut everything down
if (!MIDASstopBackgroundPlay())
   MIDASerror();

boot2("closechan\n");   // Shut everything down
//MIDAScloseChannels();
MIDASfreeAutoEffectChannels();

boot2("stopsong\n");    // Stop current song
if(cursong != -1)
    {
    MIDASstopModule(curplay);
    cursong = -1;
    }

boot2("close\n");       // Close sound library
if(!MIDASclose())
    MIDASerror();       // Oh my God we're in trouble now..

boot2("closed\n");      // Phew!

S_Running=0;            // We could actually restart the system now
boot2("runstop\n");
}


/*
 *      S_Load - Load in the music and samples.
 */

void S_Load()
{
bootmsg("S_Load()\n");

LoadMods();
LoadWavs();
}

/*
 *      T_Init - Start the timer
 */

void T_Init()
{
cfa = "T_Init";
boot2("T_Init()\n");
T_Running = 1;

//if(!T_Midas_Init())
//    MIDASerror();
}


/*
 *      T_Term - Stop the timer
 */

void T_Term()
{
cfa = "T_Term";
if(!T_Running)
    return;
boot2("T_Term()\n");

T_Midas_Term();
T_Running = 0;

// Set clock to 18hz, just to make sure

#ifdef __DJGPP__
outportb(0x43,0x34);
outportb(0x40,0);
outportb(0x40,0);
#endif

}


/*
 *      S_MusicVolume - Set the music level
 */

void S_MusicVolume(int vol)
{
mu_volume = vol;
if(curplay)
    MIDASsetMusicVolume(curplay,mu_volume>>2);
}


/*
 *      S_SoundVolume - Set the effects level
 */

void S_SoundVolume(int vol)
{
sf_volume = (vol>>2);
}


/*
 *   S_PlayModule - play a music module from the array song[]
 *                  may disrupt the sound effects slightly
 */

void S_PlayModule(char *name)
{
for(int ctr=0;ctr<Songs;ctr++)
    if(!stricmp(modtab[ctr].name,name))
        {
        if(ctr==cursong)
            return;
        if(cursong != -1)
            MIDASstopModule(curplay);
        curplay = MIDASplayModule(song[ctr],sfx_chan);
        cursong = ctr;
        return;
        }
MIDASsetMusicVolume(curplay,0);

Bug("S_PlayModule- could not find song '%s'\n",name);
}

/*
 *   S_PauseModule - Stop music, not SFX
 */

void S_PauseModule()
{
if(pausedata)
    return;
pausedata = mu_volume;
mu_volume = 0;
MIDASsetMusicVolume(curplay,mu_volume>>2);
}

/*
 *   S_UnPauseModule - Restart music
 */

void S_UnPauseModule()
{
if(!pausedata)
    return;
mu_volume = pausedata;
MIDASsetMusicVolume(curplay,mu_volume>>2);
pausedata = 0;
}

/*
 *   S_PlaySample - play a sound sample.
 */

void S_PlaySample(char *name)
{
int freq,num;
for(int ctr=0;ctr<Waves;ctr++)
    if(!stricmp(wavtab[ctr].name,name))
        {
        freq=wavtab[ctr].freq;
        if(!wavtab[ctr].nodrift)
            {
            num = 10 - (rand()%20);
            freq+=(num*driftlevel);
            }
        wavtab[ctr].handle = MIDASplaySample(wave[ctr],MIDAS_CHANNEL_AUTO,0,freq,sf_volume,MIDAS_PAN_MIDDLE);
        if(!wavtab[ctr].handle)
                    MIDASerror();
        return;
        }

Bug("S_PlaySample- could not find sound '%s'\n",name);
}


//
//   Private code hereon
//

void MIDASerror(void)
{
    MIDASclose(); /*!!!!!*/
    panic(cfa,"The MIDAS sound engine says:",MIDASgetErrorMessage(MIDASgetLastError()));
}

/*
 *   LoadMods - Load in the music
 */

void LoadMods()
{
LOAD fp;
cfa = "S_Load - mods";

// Allocate space for the songs

song=(void ***)(MIDASmodule)M_get(Songs,sizeof(MIDASmodule));

//novn=0;

// load in each song

bootmsg("  Loading songs");     // This line is not terminated, for the dots
//bootmsg("[%d]",Songs);

Plot(Songs);    // Work out how many dots to print

for(int ctr=0;ctr<Songs;ctr++)
	{
        sprintf(string,"%s",modtab[ctr].fname);         // Get the filename
        fp.open(string);                                // Open it

        midasFileBase(fp.Xorigin,fp.Xlength);   // Modify MIDAS base
        song[ctr]=(void **)MIDASloadModule(fp.truename);
      //midasFileBase(0,0);                     // Reset base to 0
        if(!song[ctr])
            MIDASerror();

        fp.close();                                     // Clean up
        Plot(0);                                        // Print a dot
	}

bootmsg("\n");  // End the line of dots

}


/*
 *   LoadWavs - Load in the sounds, calls LoadWavData for each file
 */

void LoadWavs()
{
LOAD fp;

cfa = "S_Load - wavs";
// allocate space for the waves

wave=(MIDASsample *)M_get(Waves,sizeof(MIDASsample));

// load in each song

bootmsg("  Loading samples");     // This line is not terminated, for the dots
//bootmsg("[%d]",Waves);

Plot(Waves);    // Work out how many dots to print

for(int ctr=0;ctr<Waves;ctr++)
	{
        sprintf(string,"%s",wavtab[ctr].fname);         // Get the filename
        fp.open(string);                                // Open it

        midasFileBase(ftell(fp.fileptr),fp.Xorigin);                 // Set base
        wave[ctr] = MIDASloadWaveSample(fp.truename,MIDAS_LOOP_NO);
        midasFileBase(0,0);                       // Set base

        if(!wave[ctr])                                  // Check it worked
            {
            boot2("WAVLOAD: Error loading file %s from %s\n",string,fp.truename);
            MIDASerror();                               // Oh well
            }
        wavtab[ctr].freq = GetWavFreq(fp.fileptr);

        fp.close();                                     // Clean up
        Plot(0);                                        // Print a dot
	}

bootmsg("\n");  // End the line of dots
}


/*
 * GetWavFreq(FILE *fp) - Probe a wav file and return the sample rate
 *                        Assumes the wav file is valid, since the load worked
 */

int GetWavFreq(FILE *fp)
{
fseek(fp,24L,SEEK_CUR); // Frequency is at offset 24
return getw(fp);
}

