//
//      Video driver module
//

#include <stdio.h>
#include "console.hpp"
#include "rgb_conv.hpp"
#include "itg/itg.h"

#ifndef __DJGPP__
#define __USE_SDL__
#endif

#ifdef __DJGPP__
extern "C" char *i110_version();      // Info functions, not in itg64.h
extern "C" char *i111_version();
extern "C" char *i4110_version();
extern "C" char *i4111_version();
extern "C" char *i110CL_version();
extern "C" char *i111CL_version();
#endif

#ifdef __USE_SDL__
extern int Init_SDL15D();
extern int Init_SDL15W();
extern int Init_SDL16D();
extern int Init_SDL16W();
extern "C" char *sdl15D_version();      // Info functions, not in itg64.h
extern "C" char *sdl15W_version();
extern "C" char *sdl16D_version();
extern "C" char *sdl16W_version();
extern void sdl_off();
#endif

#ifdef __linux__
extern int Init_Svga15();
extern int Init_Svga16();
extern char *svga15_version();      // Info functions, not in itg64.h
extern char *svga16_version();      // Info functions, not in itg64.h
extern void svga_off();
extern int Init_ggi15();
extern int Init_ggi16();
extern char *ggi15_version();      // Info functions, not in itg64.h
extern char *ggi16_version();      // Info functions, not in itg64.h
extern void ggi15_off();
extern void ggi16_off();
#endif

#ifdef __BEOS__
extern char *be16D_version();
extern int Init_Be16D();
extern void be16D_off();
#endif

#ifdef _WIN32
extern char *win15_version();
extern int Init_Win15();
extern void win15_off();
extern char *win16_version();
extern int Init_Win16();
extern void win16_off();
#endif

// defines

// variables

static int vid_num,vid_mode;

struct videomodetype
   {
   char rgbformat;
   char *bpp;
   int (*init)(void);
   char *(*info)(void);
   void (*off)(void);
   };

static struct videomodetype V_Modes[]=
       {
#ifdef __DJGPP__
       {0,"15",Init_110,i110_version,off},
       {1,"16",Init_111,i111_version,off},
       {0,"15",Init_4110,i4110_version,off},
       {1,"16",Init_4111,i4111_version,off},
       {0,"15",Init_110CL,i110CL_version,off},
       {1,"16",Init_111CL,i111CL_version,off},
#endif
// MR: win32 uses the same stuff currently
#if defined(__linux__) || defined(_WIN32)
       {1,"16",Init_SDL16D,sdl16D_version,sdl_off},
       {1,"16",Init_SDL16W,sdl16W_version,sdl_off},
       {0,"15",Init_SDL15D,sdl15D_version,sdl_off},
       {0,"15",Init_SDL15W,sdl15W_version,sdl_off},
//       {0,"15",Init_ggi15,ggi15_version,ggi15_off},
//       {1,"16",Init_ggi16,ggi16_version,ggi16_off},
//       {0,"15",Init_Svga15,svga15_version,svga_off},
//       {1,"16",Init_Svga16,svga16_version,svga_off},
#endif
#ifdef __BEOS__
	{1,"16",Init_Be16D,be16D_version,be16D_off},
	{1,"16",Init_SDL16W,sdl16W_version,sdl_off},
#endif
// Sentinel value, print obscene number of colours in case of error
       {-1,"BUGGER ALL ",NULL,NULL}
       };

// functions

void V_init();            // Get all known video drivers
void V_query();           // Print textmode listing of known drivers
int  V_on(int mode);      // Switch to the video mode
void V_format(int mode);  // Set RGB format

// code

/*
 *      V_init - build list of drivers
 */

void V_init()
{
vid_num=-1;
for(int ctr=0;V_Modes[ctr].rgbformat!=-1;ctr++)
    vid_num=ctr+1;
}

/*
 *      V_query - display list of video modes using printf
 */

void V_query()
{
printf("Available video modes:\n");
for(int ctr=0;ctr<vid_num;ctr++)
    printf("%2d %sbpp %s\n",ctr,V_Modes[ctr].bpp,V_Modes[ctr].info());
printf("\n\n");
}

/*
 *      V_on - set the video mode
 */

int V_on(int mode)
{
int i;

if(mode<0 || mode>vid_num)
   return 0;

// In text mode, so use printf()

printf("Switching to graphics mode...\n");
boot2( "Switching to graphics mode...\n");

V_Modes[mode].init();
i = on();
vid_mode = mode;

return i;
}

/*
 *      V_format - set RGB format for this mode
 */

void V_format(int mode)
{
if(mode<0 || mode>vid_num)
   return;
printf("RGB format set to %sbpp...\n",V_Modes[mode].bpp);
boot2( "RGB format set to %sbpp...\n",V_Modes[mode].bpp);
set_RGB_format(V_Modes[mode].rgbformat);
}

/*
 *      V_off - shut down video
 */

void V_off()
{
V_Modes[vid_mode].off();
}



