/*
Copyright (C) Matthew 'pagan' Baranowski & Sander 'FireStorm' van Rossen

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifdef WIN32

#include "system.h"
#include "ndictionary.h"
#include "md3gl.h"
#include "md3view.h"
#include "Resource.h"
#include "AFXRES.H"

extern HINSTANCE WinhInstance;
extern NodeSequenceInfo tagMenuList;
HDC  hDC_;

bool sys_rbuttondown = false;
bool sys_lbuttondown = false;
bool sys_mbuttondown = false;


OPENFILENAME *FileOpenDialog(HWND hwnd, WORD wCommand, WORD wNotify, HWND hwndCtrl, int type);
OPENFILENAME *FileSaveDialog(HWND hwnd, WORD wCommand, WORD wNotify, HWND hwndCtrl, int type);

/*
selects a 16 bit color file format, could be higher though it wouldn't work with a voodoo
*/
void SetPixelFormat( HDC hdc)
{
	PIXELFORMATDESCRIPTOR pfd, *ppfd;
    int pixelformat;
    ppfd = &pfd;    

	memset(ppfd, 0, sizeof(PIXELFORMATDESCRIPTOR));
    ppfd->nSize = sizeof(PIXELFORMATDESCRIPTOR);
    ppfd->nVersion = 1;
    ppfd->dwFlags = PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
    ppfd->dwLayerMask = PFD_MAIN_PLANE;
    ppfd->iPixelType = PFD_TYPE_RGBA;
    ppfd->cColorBits = 16;
    ppfd->cDepthBits = 16;
    ppfd->cAccumBits = 0;
    ppfd->cStencilBits = 0;

    pixelformat = ChoosePixelFormat( hdc, ppfd);

    if ( pixelformat == 0) {
        Error("ChoosePixelFormat failed");
    }

    if (ppfd->dwFlags & PFD_NEED_PALETTE) {
       Error ("ChoosePixelFormat needs palette" );
    }

    if (SetPixelFormat( hdc, pixelformat, ppfd) == FALSE) {
        Error("SetPixelFormat failed");
    }
}

/*
creates an OpenGL rendering context and makes it current
*/

void InitOpenGL( HWND hwnd )
{
	HDC hdc = GetDC( hwnd );
	mdview.hdc = hdc;

	SetPixelFormat( hdc );

	HGLRC glrc = wglCreateContext( hdc );
	mdview.glrc = glrc;
	
	if ( glrc == NULL) {	
		Error("Failed on wglCreateContext( HDC  hdc )");            
    }
                
    if (!wglMakeCurrent( hdc, glrc)) {        
        Error("Failed on wglMakeCurrent(..)" );
    }
}

/*
handles WM_CREATE
*/

bool SysOnCreate(HWND hwnd, CREATESTRUCT FAR* lpCreateStruct) 
{ 
	mdview.hwnd = hwnd;
	InitOpenGL( hwnd );
    initialize_glstate();

	RECT rect;
	GetClientRect( hwnd, &rect );
	set_windowSize( rect.right, rect.bottom );
	
	if (getCmdLine() != NULL )
	{
		if (!loadmdl( getCmdLine() )) {
			Debug( "could not load %s", getCmdLine() );
		}
	}
	
    return true;
}

/*
called every pass of the event loop
*/
void SysOnIdle()
{
	animation_loop();
}

/*
renders the model with simple viewing parameters
*/
void SysOnPaint( HWND hwnd ) 
{ 
	PAINTSTRUCT ps;            
		
    BeginPaint(hwnd, &ps);

	wglMakeCurrent( mdview.hdc, mdview.glrc );
	render_mdview();

    SwapBuffers( GetDC(hwnd) );  
    EndPaint(hwnd, &ps);    
} 

/* 
notifies md3view of resize event 
*/
void SysOnSize(HWND hwnd, UINT state, int cx, int cy) 
{ 
	RECT rect;
	GetClientRect( hwnd, &rect );
	set_windowSize( rect.right, rect.bottom );
}

/*
sets quit parameter to break out of the message loop
*/
int  SysOnDestroy(HWND hWnd)
{		
	wglDeleteContext( mdview.glrc );
	mdview.done = true;
	return 0; 	
}

/*
passes control to drag.cpp drag function
*/
void SysOnMouseMove(HWND hwnd, int x, int y, UINT keyFlags) 
{ 
	if (!(sys_rbuttondown || sys_lbuttondown || sys_mbuttondown)) return;

	POINT point;
	GetCursorPos(&point);

	drag( (mkey_enum)keyFlags, point.x, point.y );
}

/*
releases and shows the cursor again
*/
void SysOnRButtonUp(HWND hwnd, int x, int y, UINT flags) 
{ 
	if (!sys_rbuttondown) return;

	ReleaseCapture(); 
	ShowCursor( true ); 
	end_drag( (mkey_enum)flags, x, y );
	sys_rbuttondown = false;
}

/* 
same as above
*/
void SysOnLButtonUp(HWND hwnd, int x, int y, UINT flags) 
{ 
	if (!sys_lbuttondown) return;

	ReleaseCapture();
	ShowCursor( true ); 	
	end_drag( (mkey_enum)flags, x, y );
	sys_lbuttondown = false;
}

/*
on mouse down, hide the cursor and capture it to window
*/
void SysOnRButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags) 
{ 
	POINT point;
	GetCursorPos(&point);
	start_drag( (mkey_enum)keyFlags, point.x, point.y );
	SetCapture( hwnd );
	ShowCursor( false ); 
	sys_rbuttondown = true;
}

/*
*/
void SysOnLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags) 
{ 
	POINT point;
	GetCursorPos(&point);
	start_drag( (mkey_enum)keyFlags, point.x, point.y );
	SetCapture( hwnd );
	ShowCursor( false ); 
	sys_lbuttondown = true;
}


/*
processes command issued from the tag menu
*/
void SysOnTagCommand( HWND hwnd, int id, HWND hwndCtl, UINT codeNotify ) 
{
		int i = 0, tagID = id - ID_TAG_START;
		NodePosition pos;
		GLMODEL_DBLPTR dblptr;
		for (pos=tagMenuList.first() ; pos!=NULL ; pos=tagMenuList.after(pos)) {			
			if (i == tagID) break;
			i++;
		}

#ifdef DEBUG_PARANOID
		if (pos==NULL) Debug("tagMenuList entry not found");
#endif
		dblptr = (GLMODEL_DBLPTR)pos->element();
		
			OPENFILENAME *ofn;
			ofn = FileOpenDialog(hwnd, id, codeNotify, hwndCtl, IDS_FILESTRING );
			if (ofn) 
			{
				char fullName[256];
				strcpy( fullName, ofn->lpstrFile );
				strcat( fullName, ofn->lpstrFileTitle );
			
				if (dblptr != 0) freemdl_fromtag(dblptr);
				loadmdl_totag( fullName, dblptr );
				
				
				InvalidateRect( hwnd, NULL, FALSE );
				
				//leave this here!! without the screen doesn't 
				//refresh properly sometimes!!
				InvalidateRect( mdview.hwnd, NULL, FALSE );
				free( ofn );
			}
			else {
				if (dblptr != 0) {
					freemdl_fromtag( dblptr );
				}
			}

}


/*
processes events from the menu
*/
void SysOnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify) 
{ 

	// process tag menu requests
	if ((id >= ID_TAG_START) && (id < ID_TAG_START+tagMenuList.size())) {
		SysOnTagCommand( hwnd, id, hwndCtl, codeNotify );
		InvalidateRect( mdview.hwnd, NULL, FALSE );
	}
			

	switch(id)
	{
		case ID_FILE_OPEN:
		{
			OPENFILENAME *ofn;
			ofn = FileOpenDialog(hwnd, id, codeNotify, hwndCtl, IDS_FILESTRING );
			if (ofn) 
			{
				char fullName[256];
				strcpy( fullName, ofn->lpstrFile );
				strcat( fullName, ofn->lpstrFileTitle );
			
				free_mdviewdata();
				loadmdl( fullName );
				
				InvalidateRect( hwnd, NULL, FALSE );
				
				//leave this here!! without the screen doesn't 
				//refresh properly sometimes!!
				InvalidateRect( mdview.hwnd, NULL, FALSE );
				free( ofn );
			}
			break;
		}
		case ID_FILE_EXPORTTORAW:
		{
			OPENFILENAME *ofn;
			ofn = FileSaveDialog(hwnd, id, codeNotify, hwndCtl, IDS_RAWFILEFILTER );
			if (ofn) 
			{
				char fullName[256];
				strcpy( fullName, ofn->lpstrFileTitle );
				
			
				write_baseModelToRaw( fullName );
					
				free( ofn );
			}
			break;
		}
		
		case ID_FILE_IMPORTSKIN:
		{
			OPENFILENAME *ofn;
			ofn = FileOpenDialog(hwnd, id, codeNotify, hwndCtl, IDS_SKINFILEFILTER );
			if (ofn) 
			{
				char fullName[256];
				strcpy( fullName, ofn->lpstrFileTitle );
			
				importNewSkin( fullName );
				
				InvalidateRect( hwnd, NULL, FALSE );
				
				//leave this here!! without the screen doesn't 
				//refresh properly sometimes!!
				InvalidateRect( mdview.hwnd, NULL, FALSE );
				free( ofn );
			}
			break;
		}
		case ID_FILE_EXIT:
		{
			mdview.done = true;
			break;
		}
		case ID_ABOUT:
		{
			MessageBox(hwnd, ABOUT_TEXT, "About",MB_OK | MB_ICONINFORMATION);
			break;
		}

		case ID_VIEW_REFRESHTEXTURE:
			refreshTextureRes();
			InvalidateRect( hwnd, NULL, FALSE );
			break;

		case ID_VIEW_WIREFRAME: 
			oglStateWireframe(); 
			InvalidateRect( hwnd, NULL, FALSE );
			break;

		case ID_VIEW_FLATSHADED: 
			oglStateShadedFlat(); 
			InvalidateRect( hwnd, NULL, FALSE );
			break;

		case ID_VIEW_TEXTURED: 
			oglStateFlatTextured(); 
			InvalidateRect( hwnd, NULL, FALSE );
			break;
			
		case ID_VIEW_TEXTUREDSHADED: 
			oglStateShadedTextured();
			InvalidateRect( hwnd, NULL, FALSE );
			break;

		case ID_VIEW_NEAREST:
			mdview.texMode = TEX_FAST;
			setTextureFilter();
			InvalidateRect( hwnd, NULL, FALSE );
			break;

		case ID_VIEW_UNFILTEREDTEXTURE:
			mdview.texMode = TEX_UNFILTERED;
			setTextureFilter();
			InvalidateRect( hwnd, NULL, FALSE );
			break;

		case ID_VIEW_FILTEREDTEXTURE:
			mdview.texMode = TEX_FILTERED;
			setTextureFilter();
			InvalidateRect( hwnd, NULL, FALSE );
			break;

		case ID_ANIMATION_SLOWER:
			mdview.animSpeed *= ANIM_SLOWER;
			break;

		case ID_ANIMATION_FASTER:
			mdview.animSpeed *= ANIM_FASTER;
			break;

		case ID_ANIMATION_STOP:
			mdview.animate = false;
			break;

		case ID_ANIMATION_START:
			mdview.animate = true;
			break;

		case ID_ANIMATION_REWIND:
			rewindAnim();
			InvalidateRect( hwnd, NULL, FALSE );
			break;

		case ID_ANIMATION_INTERPOLATE:
			if (mdview.interpolate) mdview.interpolate = false;
			else mdview.interpolate = true;
			break;
	}
}

void SysOnKeyDown(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags) { 

}
void SysOnKeyUp(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags) { }







/* ---------------------------------------------- dialog code -------------------------------------------- */

/*
saves the directory name after every dialog use
*/

char*	szDirName = new char[256];
void SetDirectory(LPOPENFILENAME lpofn)
{
    lpofn->lpstrFile[lpofn->nFileOffset] = '\0';
    lstrcpy(szDirName, lpofn->lpstrFile);
	lstrcpy(mdview.basepath, lpofn->lpstrFile);
}


/*
handles the file open dialog
*/
OPENFILENAME *FileOpenDialog(HWND hwnd, WORD wCommand, WORD wNotify, HWND hwndCtrl, int type)
{
    OPENFILENAME *ofn = (OPENFILENAME *)malloc(sizeof( OPENFILENAME ));
	memset( ofn, 0, sizeof( OPENFILENAME ) );
    char szFile[256];       // filename string
    char szFileTitle[256];  // file-title string
    char szFilter[256];     // filter string
    char chReplace;         // strparator for szFilter
    int i, cbString;        // integer count variables

    // Retrieve the current directory name and store it in szDirName.

    if (szDirName[0] == '\0')
    {
        GetCurrentDirectory(256,szDirName);
    }

    // Place the terminating null character in the szFile.

    szFile[0] = '\0';

    // Load the filter string from the resource file.

    cbString = LoadString(WinhInstance, type, szFilter, sizeof(szFilter));

    // Add a terminating null character to the filter string.

    chReplace = szFilter[cbString - 1];
    for (i = 0; szFilter[i] != '\0'; i++)
    {
        if (szFilter[i] == chReplace)
            szFilter[i] = '\0';
    }

    // Set the members of the OPENFILENAME structure.

    ofn->lStructSize = sizeof(OPENFILENAME);
    ofn->hwndOwner = hwnd;
    ofn->lpstrFilter = szFilter;
    ofn->nFilterIndex = 1;
    ofn->lpstrFile = szFile;
    ofn->nMaxFile = sizeof(szFile);
    ofn->lpstrFileTitle = szFileTitle;
    ofn->nMaxFileTitle = sizeof(szFileTitle);
    ofn->lpstrInitialDir = szDirName;
    ofn->Flags = OFN_SHOWHELP | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    // Display the Open dialog box.

    if (GetOpenFileName(ofn))
    {        
        SetDirectory(ofn);
		return ofn;
    }
	return NULL;
}

/*
handles the file save dialog
*/
OPENFILENAME *FileSaveDialog(HWND hwnd, WORD wCommand, WORD wNotify, HWND hwndCtrl, int type)
{
    OPENFILENAME *ofn = (OPENFILENAME *)malloc(sizeof( OPENFILENAME ));
	memset( ofn, 0, sizeof( OPENFILENAME ) );
    char szFile[256];       // filename string
    char szFileTitle[256];  // file-title string
    char szFilter[256];     // filter string
    char chReplace;         // strparator for szFilter
    int i, cbString;        // integer count variables

    // Retrieve the current directory name and store it in szDirName.

    if (szDirName[0] == '\0')
    {
        GetCurrentDirectory(256,szDirName);
    }

    // Place the terminating null character in the szFile.

    szFile[0] = '\0';

    // Load the filter string from the resource file.

    cbString = LoadString(WinhInstance, type, szFilter, sizeof(szFilter));

    // Add a terminating null character to the filter string.

    chReplace = szFilter[cbString - 1];
    for (i = 0; szFilter[i] != '\0'; i++)
    {
        if (szFilter[i] == chReplace)
            szFilter[i] = '\0';
    }

    // Set the members of the OPENFILENAME structure.

    ofn->lStructSize = sizeof(OPENFILENAME);
    ofn->hwndOwner = hwnd;
    ofn->lpstrFilter = szFilter;
    ofn->nFilterIndex = 1;
    ofn->lpstrFile = szFile;
    ofn->nMaxFile = sizeof(szFile);
    ofn->lpstrFileTitle = szFileTitle;
    ofn->nMaxFileTitle = sizeof(szFileTitle);
    ofn->lpstrInitialDir = szDirName;
    ofn->Flags = OFN_SHOWHELP | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    // Display the Open dialog box.

    if (GetSaveFileName(ofn))
    {        
        //SetDirectory(ofn);
		return ofn;
    }
	return NULL;
}

#endif