
///////////////////////////////////////////////////////////////////////////////////////
//                                                                                    
//             MATHEMATICS  LIBRARY
//                                                version 0.1
// date: 1999. 9. 15. WED  remarks added       
//       1999. 10.26. TUE: quaternion added , class names changed              
//       1999. 10.27. WED: unary function bug fixed..(reference -> stack value)
//       2000. 4.4  TUE: function reorganized
// author: Jongwhan Lee (leejw51@cubic.re.kr)
//
// ACCURACY: ADJUST MATH_ZERO
// This is template library, because it's template class, there is no
// pre-compiled binaries.
// It's genetically designed, you can use double,float,int or any type
// for vector and matrix
//
// CVec : vector
// CMat : matrix
// CQua : quaternion
// CRot : quaternion rotation
// 
// quaternion
//
//typedef CVec<double> VEC;
//typedef CQua<double> QUA;
//typedef CRot<double> ROT;
//typedef CMat<double> MAT;
//	ROT a(VEC(0,0,0), VEC(-5,-10,1), RAD(50)) ;
//	ROT aa(VEC(2,5,10), VEC(3,2,4), RAD(20));
//	cout<< aa*a*VEC(1,2,2) <<endl;
//	cout<<"******************************\n";
//	cout<<VEC(1,2,2)*Rot(RAD(50), VEC(-5,-10,1))*
//		Rot(RAD(20), VEC(3,2,4))*Trs(VEC(2,5,10));
//                                                                                     
///////////////////////////////////////////////////////////////////////////////////////                                                                                    

#include <iostream>  // input / output stream
#include <iomanip> // manipulator

#ifndef __3D_ANIMATION_MATH__
#define __3D_ANIMATION_MATH__

///////////////////////////////////////////////////////////////////////////////////////
//                                                                                    
//                   GLOBAL CONSTANTS AND DEFINITIONS 
//                               for mathematics                                                                  
//                                                                                     
///////////////////////////////////////////////////////////////////////////////////////
//matrix for 3d graphics
#define MATH_PI 3.1415926535897931
#define MATH_ZERO 0.000001 //precision zero . adjust this value to control the exactness of your program
// be careful, close the bracket like this. (a) 
#define RAD(a) ((a)/180.0*3.1415926535897931) //from degree to radian eg)) RAD(90) = pi/2.0 radian
#define DEG(a) ((a)/3.1415926535897931*180.0) //from radian to degree eg)) DEG(3.141459) = 180 degrees
#define DEG2RAD(a) ((a)/180.0*3.1415926535897931) //from degree to radian eg)) RAD(90) = pi/2.0 radian
#define RAD2DEG(a) ((a)/3.1415926535897931*180.0) //from radian to degree eg)) DEG(3.141459) = 180 degrees


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
// CVec , Matrix Library using STL(standard template library)
// //Usage
// CVec<double> a(1,2,3);
// CVec<double> b(2,5,10);
// CVec<double> c= a * b; //cross product
// c= a % b; //dot product
//
// CMat<double> a(2,2);  // row ,column
// a(0,0)=1, a(0,1)=2;
// a(1,0)=3, a(1,1)=4;
// CMat b=~a; // ~ inverse matrix
// double det=b.Det(); //determinant
// CMat c = a*b;   
// if you need other calculation, add it as global function
//
// Because this is STL, including this head file in your code, then it will be compiled together.
// #include "mathobj.h"
// if there is error, confirm that
// using namespace std;
//
// usage: matrix & vector hybrid calculation
//	CMat<double> a=Identity<double,4,4>()*RotZ(RAD(40));
//
//	CVec<double> b(1,2,3);
//	b*=a;
//	cout<< b;
//
// Arbitrary vector rotation usage>>
//                                                                                                
//	CMat<float> t;                                                                                             
//	CVec<float> a(2.0, 1.0, 0.0);                                                                                             
//	CVec<float> b (3.0, 3.0, 1.0);                                                                                             
//                                                                                          
//	cout<<Trs<float>(-a)* Rot<float>((float)RAD(90.0), b-a) *  Trs<float>(a);
////////////////////////////////////////////////////////////////////////////////////////////////////////////
#include <iostream> //because we're using STL libray, use iostream instead of iostream.h
#include <iomanip> //manipulator
#include <math.h>
#include <vector> //stl library . linked list
using namespace std; //use standard namespace

//class declarations
template <class T> class CMat;
template <class T> class CVec;

///////////////////////////////////////////////////////////////////////////////////////
//                                                                                    
//                   VECTOR                                                                   
//                                                                                     
///////////////////////////////////////////////////////////////////////////////////////
//implementation
//because it's template class, it is delcared and implemented here
template <class T> 
class CVec {
public:
	//data
	T x,y,z;

	//constructors
	CVec(void) {x=y=z=(T)0.0;}
	CVec(T X, T Y, T Z) {x=X, y=Y, z=Z;}
	T Size(void) {
		return sqrt(x*x+ y*y+ z*z);
	}

	//Make Absolute Value
	CVec<T> GetAbs(void) {
		return CVec<T> (fabs(x) , fabs(y) , fabs(z));
	}

	//opengl specific code
	void Put(void) {
		glVertex3f(x,y,z);
	}
	void Move(void) {
		glTranslated(x,y,z);
	}
	void Normal(void) {
		glNormal3d(x,y,z);
	}

	//for bounding boxes
	void MakeMax(CVec& One) {
		if (One.x > x) x=One.x;
		if (One.y > y) y=One.y;
		if (One.z > z) z=One.z;
	}
	void MakeMin(CVec& One) {
		if (One.x < x) x=One.x;
		if (One.y < y) y=One.y;
		if (One.z < z) z=One.z;
	}

	//distance : Distance(VECTOR, VECTOR);
	T Distance(CVec& tmp) {
		return (*this-tmp).Size();
	}

	//read the value
	T& operator () (int p)
	{
		if (p==0) return x;
		else if (p==1) return y;
		else return z;
	}



	// mathematical operation

	//unary function don't use reference
	// - VECTOR
	CVec  operator - (void)
	{
		return CVec(-x,-y,-z);
	}


	//subtract : VECTOR - VECTOR 
	CVec operator- (CVec& tmp)
	{
		return CVec(x - tmp.x , y - tmp.y, z - tmp.z);
	}
	CVec& operator -= (CVec& tmp) {
		*this = *this - tmp;
		return *this;
	}

	//add : VECTOR + VECTOR
	CVec operator+ (CVec& tmp) 
	{
		return CVec(x + tmp.x , y + tmp.y , z + tmp.z );
	}
	CVec& operator += (CVec& tmp) {
		*this = *this + tmp;
		return *this;
	}

	// divide
	CVec operator / (T two) 
	{
		if (fabs(two)<MATH_ZERO) return *this;

		return CVec(x / two, y / two,z / two);
	}
	CVec& operator /= (T tmp) {
		*this = *this / tmp;
		return *this;
	}

	// multiply
	CVec operator * (T tmp) 
	{
		return CVec(x * tmp, y * tmp, z * tmp);
	}

		
	//cross product
	//multiply : VECTOR* VECTOR
	CVec operator * (CVec& tmp) 
	{
		//   i      j(-1)     k
		//  one.x  one.y  one.z 
		//  two.x  two.y  two.z
		return CVec (y*tmp.z - z*tmp.y,
			-(x*tmp.z - z*tmp.x),
			x*tmp.y - y*tmp.x);
	}

	//dot product : VECTOR % VECTOR
	T operator % (CVec& tmp) 
	{
		return x * tmp.x  +  y * tmp.y  + z * tmp.z ;  
	}

	
	//normalize : ~VECTOR
	CVec operator ~ (void) 
	{
		T size= Size();
		if (size<MATH_ZERO) return *this; 
		
		//valid? 
		return *this / size;
	}


	//if operators : VECTOR == VECTOR
	bool operator == (CVec & tmp) 
	{
		if (fabs(x - tmp.x) < MATH_ZERO	&&
			fabs(y - tmp.y) < MATH_ZERO &&
			fabs(z - tmp.z) < MATH_ZERO)
			return true;
		else return false;
	}

	
	// if operator : VECTOR != VECTOR
	int operator != (CVec<T> & tmp)
	{
		return !(*this == tmp);
	}

	

};



/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   multiply
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//multiply : NUMBER * VECTOR
template <class T>
CVec<T> operator * (T one, CVec<T>& two) 
{
	return CVec<T>
		(
		one * two.x,
		one * two.y,
		one * two.z
		);
}

// stream : cout<< VECTOR ;
template <class T>
ostream& operator << (ostream& out, CVec<T>& vec) 
{
	out<<setw(10)<<vec.x << setw(10) << vec.y<< setw(10) << vec.z <<endl;
	return out;
}




///////////////////////////////////////////////////////////////////////////////////////
//                                                                                    
//                   MATRIX                                                                  
//    
//
//
//  usage)) 
//                                                                                    
//    typedef CVec<double> VECTOR;                                                                                  
//    typedef CMat<double> MAT;                                                                               
//    void main(void)                                                                                
//    {                                                                                
//    	VECTOR v(10, 10, 0);                                                                                
//    	MAT a(4,4);                                                                                
//    	MAT b=Trs(VECTOR(0, -10, 0))*RotZ(RAD(90));                                                                                
//    	cout<< v*b;                                                                                
//                                                                                    
//    	Identity(a);                                                                                
//    	cout<<a;                                                                                
//    }                                                                                
//          
// result>>
//  0,10,0
//  1 , 0 , 0 , 0
//  0 , 1 , 0 , 0
//  0 , 0 , 1 , 0
//  0 , 0 , 0 , 1
//                                                                                    
//         
// you had better use typedef for flexible codes.                                                                
//                                                                                     
///////////////////////////////////////////////////////////////////////////////////////
//CMat
template <class T>
class CMat{
public:
	int sx,sy; //size x, size y
	vector<T> m; 
	

	//destructor
	~CMat(void) {
		m.clear();
	}

	
	//constructor
	CMat(int SY, int SX)
	{
		//size
		sy=SY, sx=SX;

		//set up
		int i;
		for (i=0; i< SY*SX; i++) {
			m.push_back((T)0.0);
		}

	}

	CMat(void)
	{
		sx=sy=0;
	}


	//inverse CMat
	T Det();
	T Determinant(CMat& m); //determinant
	CMat Minor(unsigned r, unsigned c); //minor 
	void Print(void);
	double* Set(void); //load to opengl matrix stack



	void ReadGL(double GLMat[16]);

	//load identity matrix
	void Identity(void)
	{
		int i,j;
		//set zero
			for (j=0;j<sy;j++)
			for (i=0;i<sx;i++)
				(*this)(j,i)=(T)0;

		//valid
		if (sx != sy) return;

		//set one
		for (i=0;i<sx;i++) {
			(*this)(i,i)=1;
		}
	}

	//read the value
	T& operator () (int y, int x)
	{
		//valid?
		return m[y*sx + x];
	}


	// mathematical functions
	// MATRIX / NUMBER
	CMat operator / (T tmp)
	{
		int i,j;
		CMat mat=*this; // copy

		// valid?
		if (fabs(tmp)<MATH_ZERO) return *this;

		for (j=0;j<mat.sy;j++)
			for (i=0;i<mat.sx;i++) {
				mat(j,i) /= tmp;
			}

		//done
		return mat;
	}

	//   multiply
	// Matrix * double
	CMat operator* ( T tmp)
	{
		int i,j;
		CMat<T> mat=*this;

		for (j=0;j<mat.sy;j++)
			for (i=0;i<mat.sx;i++) {
				mat(j,i) *= tmp;
			}
		//done
		return mat;
	}

	// multiply
	CMat<T> operator * (CMat<T>& two) 
	{
		//valid?
		//  1x4  4x4
		// sy sx=sy sx
		CMat<T> tmp(sy, two.sx);

		if (sx != two.sy) return tmp;
		int num=sx; //the same number


		T sum; //temporary sum

		int i,x,y;
		for (y=0;y<sy;y++)
		{
			for (x=0;x<two.sx;x++) {

				sum=0;
				for (i=0;i<num;i++) {
					sum += (*this)(y, i) * two(i, x);
				}
				//store
				tmp(y,x) = sum;
			}
		}

		//done
		return tmp;
	}

	// Matrix A/ Matrix B = Matrix A* ~Matrix B
	CMat operator / (CMat & tmp)
	{
		return (*this)*~tmp; //multiply inverse matrix
	}


	
	//inverse CMat
	CMat operator~(void)
	{
		int i,j;
		CMat inverse(sy, sx), minor(sy-1, sx-1);

		//valid?
		if(sy!=sx) return inverse;
		if (fabs(Det())<MATH_ZERO) return inverse; //if determinant is zero, there is no inverse matrix

		for(i=0; i<sy; i++)
			for(j=0; j<sx; j++) {
				minor=Minor(i,j);
				if((i+j)%2)
					inverse(j, i)=-minor.Det();
				else
					inverse(j, i)=minor.Det();
			}

		return inverse/Det();
	}


};


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   Member Functions
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
template <class T>
double* CMat<T>::Set(void)
{
	static double a[16];
	int i;

	//reset
	for (i=0;i<16;i++) a[i]=0;

	if (sx!=4 || sy!=4) return a;//nothing to do

	int x,y;
	int c=0;
	for (y=0;y<sy;y++) {
		for (x=0;x<sx;x++) {
			a[c]=m[y*sx+x];
			c++;
		} //end of for
	} //end of for

	glMultMatrixd(a);


	return a;
	
}

// stream : cout<< VECTOR ;
template <class T>
ostream& operator << (ostream& out, CMat<T>& mat) 
{
	//empty 
	if (mat.m.empty()) return out;

	int x,y;
	for (y=0;y<mat.sy;y++) {
		for (x=0;x<mat.sx;x++) {
			out<<setw(15)<<mat.m[y*mat.sx + x];		
		} //end of for
		out<<endl;

	} //end of for

	return out;
}





template <class T>
void CMat<T>::ReadGL(double GLMat[16]) {
	m.clear();

	//seize memory
	int i,j;

	sx=4,sy=4;
	for (i=0;i<sx*sy;i++)
	m.push_back(0);

	//copy
	int c=0;
	for (i=0;i<4;i++)
		for (j=0;j<4;j++) {
			m[j*sx+i]=GLMat[c];
			c++;
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   divide
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

// NUMBER / Matrix
template <class T>
CMat<T> operator / (T one, CMat<T>& two)
{
	int i,j;
	CMat<T> tmp=two;

	for (j=0;j<two.sy;j++)
		for (i=0;i<two.sx;i++) {
			tmp(j,i) = one / tmp(j,i);
		}

	//done
	return tmp;
}


// double * Matrix
template <class T>
CMat<T> operator* (T one, CMat<T>& two)
{
	int i,j;
	CMat<T> tmp=two;

	for (j=0;j<two.sy;j++)
		for (i=0;i<two.sx;i++) {
			tmp(j,i) = one * tmp(j,i);
		}
	//done
	return tmp;
}


//inverse CMat
template <class T>
T CMat<T>::Det()
{
	return Determinant(*this);
}

//find minor 
template <class T>
CMat<T> CMat<T>::Minor(unsigned r, unsigned c)
{
	int i,j;
	CMat<T> minor(sy-1, sx-1), temp=*this;

	//valid range?
	if(sy==1||sx==1)
		return *this;

	for(i=0; i<sy; i++)
		for(j=c; j<sx-1; j++)
			temp(i,j)=temp(i,j+1);

	for(j=0; j<sx; j++)
		for(i=r; i<sy-1; i++)
			temp(i,j)=temp(i+1,j);

	for(i=0; i<sy-1; i++)
		for(j=0; j<sx-1; j++)
			minor(i,j)=temp(i,j);

	return minor;
}

//use recursion to find the determinant of m
template <class T>
T CMat<T>::Determinant(CMat<T>& m)
{
	int j;
	T cofactor, det=(T)0;
	CMat<T> minor(m.sy-1, m.sx-1);

	if(m.sy==1)
		return m(0, 0);

	for(j=0; j<m.sy; j++) {
		minor=m.Minor(j, 0);
		cofactor=((j%2)==0) ? m(j,0) : -m(j,0);
		det+=cofactor*Determinant(minor);
	}

	return det;
}



//3d rotation
template <class T>
CMat<T> RotZ( T th)
{
	//                                                                                               
	//                                              4 x 4                                            
	//									cos(th)		sin(th)     0		0                                  
	//                                                                                               
	//     1x4             1x4			-sin(th)    cos(th)     0       0                           
	//  [x* y*  z* 1) = [x  y   z 1) *                                                         
	//									0			0			1		0                                                              
	//                                                                                               
	//									0			0			0		1                                                             
	//                                                                                               

	CMat<T> mat(4,4);
	mat(0,0)=cos(th),	mat(0,1)=sin(th),	mat(0,2)=0,		mat(0,3)=0;
	mat(1,0)=-sin(th),	mat(1,1)=cos(th),	mat(1,2)=0,		mat(1,3)=0;
	mat(2,0)=0,			mat(2,1)=0,			mat(2,2)=1,		mat(2,3)=0;
	mat(3,0)=0,			mat(3,1)=0,			mat(3,2)=0,		mat(3,3)=1;
	return mat;
}
template <class T>
CMat<T> RotY(T th)
{
	//                                                                                               
	//                                              4 x 4                                            
	//									cos(th)		0          -sin(th) 0                             
	//                                                                                               
	//     1x4             1x4			0           1           0       0                           
	//  [x* y*  z* 1) = [x  y   z 1) *                                                         
	//									sin(th)		0			cos(th) 0                                                              
	//                                                                                               
	//									0			0			0		1                                                             
	//                                                                                               

	CMat<T> mat(4,4);
	mat(0,0)=cos(th),	mat(0,1)=0,			mat(0,2)=-sin(th),		mat(0,3)=0;
	mat(1,0)=0,			mat(1,1)=1,			mat(1,2)=0,				mat(1,3)=0;
	mat(2,0)=sin(th),	mat(2,1)=0,			mat(2,2)=cos(th),		mat(2,3)=0;
	mat(3,0)=0,			mat(3,1)=0,			mat(3,2)=0,				mat(3,3)=1;
	return mat;
}
template <class T>
CMat<T> RotX(T th)
{
	//                                                                                               
	//                                              4 x 4                                            
	//									1		0			0			0                             
	//                                                                                               
	//     1x4             1x4			0       cos(th)		sin(th)		0                           
	//  [x* y*  z* 1) = [x  y   z 1) *                                                         
	//									0		-sin(th)	cos(th)		0                                                              
	//                                                                                              
	//									0		0			0			1                                                             
	//                                                                                               

	CMat<T> mat(4,4);

	mat(0,0)=1,	mat(0,1)=0,			mat(0,2)=0,			mat(0,3)=0;
	mat(1,0)=0,	mat(1,1)=cos(th),	mat(1,2)=sin(th),	mat(1,3)=0;
	mat(2,0)=0,	mat(2,1)=-sin(th),	mat(2,2)=cos(th),	mat(2,3)=0;
	mat(3,0)=0,	mat(3,1)=0,			mat(3,2)=0,			mat(3,3)=1;

	return mat;
}

//translate
template <class T>
CMat<T> Trs(CVec<T> th)
{
	//                                                                                               
	//                                              4 x 4                                            
	//									1		0			0			0                             
	//                                                                                               
	//     1x4             1x4			0       cos(th)		sin(th)		0                           
	//  [x* y*  z* 1) = [x  y   z 1) *                                                         
	//									0		-sin(th)	cos(th)		0                                                              
	//                                                                                              
	//									0		0			0			1                                                             
	//                                                                                               

	CMat<T> mat(4,4);

	mat(0,0)=1,		mat(0,1)=0,			mat(0,2)=0,			mat(0,3)=0;
	mat(1,0)=0,		mat(1,1)=1,			mat(1,2)=0,			mat(1,3)=0;
	mat(2,0)=0,		mat(2,1)=0,			mat(2,2)=1,			mat(2,3)=0;
	mat(3,0)=th.x,	mat(3,1)=th.y,		mat(3,2)=th.z,			mat(3,3)=1;

	return mat;
}

//hybrid calculations
//matrix should be 4x4
// FINAL = VECTOR * MATRIX * MATRIX
// order: ------->
template <class T>
CVec<T> operator* (CVec<T>& one, CMat<T> & two)
{
	CVec<T> vec;
	CMat<T>		tmp; //temporary matrix

	//matrix setup
	CMat<T> mat(1,4);
	mat(0,0)= one.x, mat(0,1)=one.y, mat(0,2) =one.z, mat(0,3)=1.0;

	//calculation
	// VECTOR ------>
	tmp= mat * two;
	
	//read
	vec.x= tmp(0,0), vec.y= tmp(0,1), vec.z = tmp(0,2);

	return vec;
}

// return type should be CVec<T>& to be effective
template <class T>
CVec<T>& operator*= (CVec<T>& one, CMat<T>& two)
{
	one=one*two;
	return one; //not one*two , because we give not temporary value
}


//get radian from the lx,ly
template <class T>
T GetRad(T lx, T ly)
{
	T th=0.0; //default value

	//            /        |--                                                                           
	//           /         |                                                                           
	//          /          |                                                                           
	//         /           |                                                                           
	//        /            |  ly                                                                               
	//       /             |                                                                              
	//      /th            |                                                                           
	//     +---------      |--
	//         lx                                                                                            
	//  
	//special cases
	if (fabs(lx)<MATH_ZERO && fabs(ly)<MATH_ZERO) { //origin ?
		th=(T)RAD(0);
	}
	else if (fabs(lx)<MATH_ZERO && ly>0) th=(T)RAD(90); //90 degrees
	else if (fabs(lx)<MATH_ZERO && ly<0) th=(T)RAD(270); //270 degrees
	//general cases
	//                  |                                                                            
	//                  |                                                                            
	//                  |                                                                            
	//                2 |  1                                                                          
	//           -------+------                                                                           
	//                3 | 4                                                                           
	//                  |                                                                            
	//                  |                                                                            

	else {
		th=atan(fabs(ly/lx));
		
		if (lx>0 && ly>=0) th=th;
		else if (lx< 0 &&  ly>=0) th= RAD(180) -th;
		else if (lx<0 && ly<= 0) th=RAD(180)+th;
		else if (lx>0 && ly<=0) th= RAD(360) -th;
		}

	//done..
	return th;
}



//rotation for the arbitary vector
//counter clock wise rotation
template <class T>
CMat<T> Rot( CVec<T> Vec, T TH)
{
	//from origin to Vec
	T a=Vec.x;
	T b=Vec.y;
	T c=Vec.z;

	// get correct angle, use GetRad function
	//alpha=asin(b / sqrt(b*b + c*c));
	//alpha2=acos( c / sqrt(b*b+c*c));
	T alpha3= GetRad(c,b);

	//pi= asin(a / sqrt(a*a + b*b + c*c));
	//pi2= acos( sqrt(b*b + c*c) / sqrt(a*a+ b*b+ c*c));
	T pi3= (T)GetRad((T)sqrt(b*b + c*c), (T)a); 


	CMat<T> tmp;
	tmp= RotX<T>(alpha3) * RotY<T>(-pi3) * RotZ(TH) * RotY<T>(pi3) * RotX<T>(-alpha3);

	return tmp;

}

// MATRIX
// FINAL = VECTOR * MATRIX * MATRIX -> PROJECTION

// QUATERNION
// FINAL =  PROJECTION <- QUATERNION * QUATERNION * VECTOR

/////////////////////////////////////////////////////////////////////////////////////////
//
//                                        quaternion
//usage:
//	ROT a(VEC(0,0,0), VEC(-1,-1,-1), RAD(50)) ;
//	cout<< ROT(VEC(0,0,0),VEC(1,1,1),RAD(50))*a*VEC(1,2,2) <<endl;
//	cout<<VEC(1,2,2)*Rot(RAD(50), VEC(-1,-1,-1))*Rot(RAD(50), VEC(1,1,1));
//	cout<< Rot(RAD(50), VEC(-1,-1,-1))*Rot(RAD(50), VEC(1,1,1));
// q= c+ u  , u= xi+ yj + zk
// q`=c` + u` , u`= x'i + y`j + z'k

// x,y,z represents vector.

template <class T>
class CQua {
	public:
		T c,x,y,z; //quarternion = c + xi + yj + zk
		CQua(void) {
			c=x=y=z=0;
		}

		CQua(CVec<T> N, T Th)
		{
			Set(N,Th);			
		}
		void Set(CVec<T> N, T Th) 
		{
			if (N.Size()<MATH_ZERO) return; // cannot be unit vector

			N= ~N; //make a unit vector
			c= cos(Th/2.0);
			x= sin(Th/2.0)* N.x;
			y= sin(Th/2.0)* N.y;
			z= sin(Th/2.0)* N.z;
		}
		CQua(T C, T X, T Y, T Z) {
			c=C;
			x=X;
			y=Y;
			z=Z;
		}
		CMat<T> GetMat(void);
	
		// mathematical operation
		CQua operator - (void)
		{
			return  CQua(-c, -x, -y, -z);
		}

	// mathematical functions
	//dot product
	T operator % (CQua<T>& Two)
	{
		return c * Two.c + 
			   x * Two.x + 
			   y * Two.y + 
			   z * Two.z;

	}
	//multiply
	CQua operator * (T tmp)
	{
		return CQua(c*tmp, x*tmp, y*tmp, z*tmp);
	}
	//divide
	CQua operator / (T tmp)
	{
		if (fabs(tmp)<MATH_ZERO) return *this;
		return CQua(c/tmp, x/tmp, y/tmp, z/tmp);
	}
	
	//find the conjugate
	//inverse is so simple
	CQua operator ~ (void)
	{
		return CQua(c, -x, -y, -z);
	}

	
	//add
	CQua operator + ( CQua<T>& Two)
	{
		return CQua (c + Two.c, 	x + Two.x, y + Two.y, z + Two.z);
	}

	//subtract
	CQua operator - (CQua& Two)
	{
		return CQua (c - Two.c, x - Two.x, y  - Two.y, z - Two.z);
	}

	//multiply
	CQua operator * (CQua& Two)
	{
		CQua<T> tmp;
		tmp.c =  c*Two.c -  x*Two.x -  y*Two.y -  z*Two.z;
		tmp.x =  y*Two.z - Two.y* z +  c*Two.x + Two.c* x;
		tmp.y =  z*Two.x - Two.z* x +  c*Two.y + Two.c* y;
		tmp.z =  x*Two.y - Two.x* y +  c*Two.z + Two.c* z;	
		return tmp;
	}

	//vector multiply
	CVec<T> operator * (CVec<T>& Two)
	{
		CQua<T> tmp;
		tmp.c= 0;// pure quaternion
		tmp.x= Two.x;
		tmp.y= Two.y;
		tmp.z= Two.z;

		CQua<T> res; //result = u + r p ~r , p <- pure quaternion
		res=  (*this) * tmp * ~(*this); // pure quaternion ,too
		
		
		return CVec<T>(res.x, res.y, res.z);
	}


};



template <class T>
CQua<T> operator * ( T Two, CQua<T> One)
{
	One.c *= Two;
	One.x *= Two;
	One.y *= Two;
	One.z *= Two;
	return One;
}


//get matrix from quaternion
template <class T>
CMat<T> CQua<T>::GetMat(void)
{
	CMat<T> m(4,4);
	// 0 row
	m(0,0)= c*c + x*x - y*y - z*z;
	m(1,0)= 2*x*y - 2*c*z;
	m(2,0)= 2*x*z + 2*c*y;
	m(3,0)= 0;

	// 1 row
	m(0,1) = 2*x*y + 2*c*z;
	m(1,1) = c*c - x*x + y*y - z*z;
	m(2,1) = 2*y*z - 2*c*x;
	m(3,1) = 0;

	//2 row
	m(0,2) = 2*x*z - 2*c*y;
	m(1,2) = 2*y*z + 2*c*x;
	m(2,2) = c*c -x*x - y*y + z*z;
	m(3,2) = 0;

	//3 row
	m(0,3) = 0;
	m(1,3) =0;
	m(2,3) = 0;
	m(3,3) = c*c + x*x + y*y + z*z;

	return m;
}

template <class T>
ostream& operator << (ostream&out, CQua<T> tmp) 
{
	out<< setw(10)<<tmp.c << " + "<< 
		setw(10)<< tmp.x <<" i + " <<
		 setw(10)<<tmp.y << " j + " << tmp.z<<" k ";
	return out;
}


// stream : cout<< VECTOR ;
/*template <class T>
ostream& operator << (ostream& out, CQua<T>& One) 
{
	out<<One.c << "+ " << One.x <<"i+ " << One.y <<"j+ " << One.z <<"k " <<endl;
	return out;
}*/


// p->p'  = u + rpR = (u,r) 
// p -> rotation -> translation
// unit quaternion deals with rotation
// CRot can rotate the vector and traslate it!
template <class T>
class CRot { //rotation
public:
	CQua<T> u; // c= 0 <- pure quaternion
	CQua<T> r; // size is 1 <- unit quaternion
	CRot(CQua<T> U, CQua<T> R) {
		u=U, r=R;
	}

	//default constuctor
	CRot(void) {
		u=CQua<T>(0,0,0,0);
		r=CQua<T>(1,0,0,0); // must be unit quaternion to avoid dividing by zero
	}

	CRot GetRot(void) { // return only rotation(CRot)
		return CRot(CQua<T>(0,0,0,0), r);
	}

	CVec<T> GetTrs(void) { // return translatin(vector)
		return CVec<T>(u.x,u.y,u.z);
	}
	
	//only rotation
	CRot (CVec<T> N, T Th) {
		u=CQua<T>(0, 0, 0, 0); //pure quaternion
		r=CQua<T>(1, 0, 0, 0); // unit quaternion
		r.Set(N, Th);
	}

	// P : translation
	// N : rotation around this vector
	// Th : radian
	// order ->  translation <- rotation <- p
	CRot(CVec<T> P, CVec<T> N, T Th) {
		u=CQua<T>(0, P.x, P.y, P.z); //pure quaternion
		r=CQua<T>(1, 0, 0, 0);
		r.Set(N,Th);
	};
	
	CMat<T> GetMat(void) {
		//
		// MATRIX ORDER:  -------->
		// FINAL =        VECTOR * ROTATION * TRANSLATION

		// QUATERNION ORDER:             <---------
		// FINAL =   QUATERNION  *QUATERNION* VECTOR
		return r.GetMat()*Trs(CVec<T>(u.x, u.y, u.z));
	}
	void Set(void) {
		glTranslated( u.x, u.y, u.z);
		r.GetMat().Set();
	}

	// inverse transfrom
	CRot operator ~ (void) 
	{
		CRot<T> tmp;
		tmp.u=  -(~r*u*r); // look minus
		tmp.r= ~r;
		return tmp;
	}

	CRot operator * (CRot<T>& Two)
	{
		CRot<T> tmp;

		tmp.u= u + r*Two.u*~r ; // translation should be processed by rotation

		tmp.r= r*Two.r; // rotation isn't affected by translation
		// it means that roation is computed with rotation

		return tmp;
	}

	//vector multiply
	CVec<T> operator * (CVec<T>& Two)
	{
		CVec<T> res; //result = u + r p ~r , p <- pure quaternion
		res=  CVec<T>(u.x,u.y,u.z) + r*Two; 
		return res;
	}
 
};

// display 
template <class T>
ostream& operator << (ostream& out, CRot<T> rot)
{ 
	out<<"u(pure quaternion) = "<< rot.u << endl<<
		"r(unit quaternion) = " << rot.r ;
	return out;
		 
}




//inverse
//usage
//	ROT a0=	ROT(VEC(1,2,3), VEC(5,-2,-2), RAD(50));
//	ROT a1= ~a0;
//	cout<< a0;
//	cout<< a1;
//	cout<< a1*a0*VEC(10,20,30);



// stream : cout<< VECTOR ;
/*template <class T>
ostream& operator << (ostream& out, CRot<T>& One) 
{
	out<<"u " << One.u ;//pure quaternion
	out<<"r " << One.r ; //unit quaternion
	return out;
}*/

//quaternion example
/*
#include "mathobj.h"

typedef CVec<double> VEC;
typedef CQua<double> QUA;
typedef CRot<double> ROT;
typedef CMat<double> MAT;
 
void main(void)
{   
	VEC a(2,3,5);
	a+=VEC(1,1,1);
	cout<<(~a).Size() <<endl;

	cout<<"********************\n";
	cout<<VEC(2,0,0)*  
		Rot(RAD(90),VEC(-1,-2,-3))*
		Rot(RAD(50),VEC(52, 10,18))<<endl;
	
	cout<< 
		ROT(VEC(0,0,0), VEC(52,10,18), RAD(50))*
		ROT(VEC(0,0,0), VEC(-1,-2,-3), RAD(90))   
		*VEC(2,0,0)
		<<endl;
}*/


#endif