#include "resource.h"
#include "stdafx.h"
#include "modelobj.h"

//global variables.
CModel* GV_pModel=NULL;  //current 3ds file 
CObj* CurObj=NULL; //current object
CMtl* CurMat=NULL; //current material

// exe.cpp : Defines the entry point for the console application.
//


/* --------------------------- 3DSRDR.C -------------------------------

    .3DS file format exerciser v1.4
    by Bezzegh Dnes, aka. DiVeR
    e-mail: diver@inf.bme.hu
    http://www.inf.bme.hu/~diver

    If you are not interested in materials,
    then there is not much new for you. :-(

    Changes since v1.3:

    - Fixed SmooListReader's bug when group>15.
    - New data-chunk type: WORD /CHUNK_WRD/
    - New Material chunks:
	CHUNK_REFMAP	      CHUNK_REFMASK	     CHUNK_SHININESS
	CHUNK_SPECULARMAP     CHUNK_SPECULARMASK     CHUNK_SHSTRENGTH
	CHUNK_OPACITYMAP      CHUNK_OPACITYMASK      CHUNK_TRANSP
	CHUNK_TEXTURE2	      CHUNK_TEXTURE2MASK     CHUNK_TRFALLOFF
	CHUNK_SHININESSMAP    CHUNK_SHININESSMASK    CHUNK_REFLECTBL
	CHUNK_SELFILLUMMAP    CHUNK_SELFILLUMMASK    CHUNK_SELFILLUM
	CHUNK_TEXTUREMASK     CHUNK_BLUR	     CHUNK_SHADETYPE
	CHUNK_BUMPMASK	      CHUNK_TROTATION	     CHUNK_FLAG2SIDE
	CHUNK_USCALE	      CHUNK_VSCALE	     CHUNK_TFLAGS
	CHUNK_UOFFSET	      CHUNK_VOFFSET
	CHUNK_BLACKTINT       CHUNK_WHITETINT
	CHUNK_RTINT	      CHUNK_GTINT	     CHUNK_BTINT
    - New "not so useful" Material chunks:
	CHUNK_FACEMAP	      CHUNK_WIREFRAME	     CHUNK_WIRETHICK
	CHUNK_SOFTEN	      CHUNK_FLAGIN	     CHUNK_FLAGUNIT
	CHUNK_FLAGADD
    - New Light chunks:
	CHUNK_LIGHTR1	      CHUNK_LIGHTR2	     CHUNK_LIGHTMUL
    - Added CHUNK_SHP and CHUNK_LFT. Why not?
    - That's all!

    Did I leave something from material out?

   ------------------------------ v1.3 --------------------------------

    .3DS file format exerciser v1.3.
    by Mats Byggmastar, aka. MRI/Doomsday
    e-mail: mri@penti.sit.fi

    All credits goes to Jare/Iguana as this is just a modified
    version of v1.2 he released.

    You can use the following method to dig up new data:
	- Make a simple .3ds file and convert it with this reader
	  using -dump mode.
	- Go back into 3D Studio and add that little feature to the
	  scene and convert the .3ds file again using -dump mode.
	- Compare the output from both conversions and you'll
	  find the chunk where the data was modified or added.

    "The Unofficial 3dStudio 3DS File Format v1.0" by Jeff Lewis
    is also a great help. I found that file at:

	  www.viewpoint.com/avalon/format_specs.html


    .Mats Byggmastar  15.2.1997  Jakobstad, Finland


   --------------------------- 3DSRDR.C -------------------------------

    .3DS file format exerciser v1.2.
    Written by Javier Arevalo, AKA Jare/Iguana.

    I compile this with Watcom/32, but I guess it should work with
	any compiler and OS combination for which the typedefs are
	valid i.e. any that I know for PCs... Try it and see.
	Oh, and also check the #pragma pack() thing.

    - DISCLAIMER -

    I hope I have not broken any patents or trade secrets by releasing
	this info. This is purely a mind exercise to break into a file
	format that is quite useful to know. As far as I have been told
	a file format is not subject to anything such as copyright or
	patent, so I have done this because I believe I'm allowed to.

    I PLACE THIS FILE IN THE PUBLIC DOMAIN, SO EVERYTHING CONTAINED HERE
	IS TOTALLY FREE FOR YOU TO EXPLORE AND USE. I DISCLAIM ANY AND ALL
	EVENTS COMING OUT OF ANY POSSIBLE USE (OR LACK OF USE) OR EXISTANCE
	OF THIS FILE. I WON'T BE LIABLE FOR ANYTHING RELATED TO THIS FILE,
	OR ANY PRIOR OR FUTURE VERSION OF IT.

    All trademarks mentioned are property of their respective holders.

    - Merits -

    Heavily based on info on the file 3DS_08.TXT by Jim Pitts
      (jp5@ukc.ac.uk)

    Basic material-related stuff digged up by Jare.
    Track info stuff too.

    Thanks to the Egerter brothers of WGT fame and to Walken/Impact studios
	for extra info, Rex Deathstar for support. And definitely to
	Xanthome/Darkzone for you know why. And of course, respect to
	Avatar/Legend Design for being here before all of us.

    For a cool example of actual reading of 3DS files, look no
	further than 3DSCO20.ZIP by Mats Byggmastar aka. MRI. I
	personally prefer using a table-driven modification of this
	code, but both approaches are quite ok and his is much faster
	to write and follow.

    Now only lack is someone to explain how to make use of all this
	stuff i.e. how exactly is data stored, how spline interpolations
	are performed, what are those things called quaternions, etc. And
	also, maybe, dig the rest of the chunks until we are actually able
	to write 3DS files instead of just being bored reading. There's
	lots to do.

    If you decide to work on this further, please make your findings
	public like we have already done, ok? Upload it to
	x2ftp.oulu.fi, THE place for programming info, and/or to
	ftp.cdrom.com. But please PUBLISH it!

    - Change log -

    V 1.2:
	- Added change log to have some idea what's going on.
	- Added pivot point reading inside tracks stuff.
	- Info about spline flags on keyframes.
	- Added face edge visibility info.
	- Finally!! Those flags that mark when the texture is wrapping
	  around inside a face. This happens when you apply spherical
	  or cylindrical coordinates, the faces along the 0 axis don't
	  get proper mapping coords. Someone describe how to fix this?
	- Added -quiet parm, only displays minimal chunk info.
	- Object parent number is stored in CHUNK_TRACKOBJNAME.
	  This makes reference to the node number in CHUNK_OBJNUMBER.
	- Object number changed to unsigned. Parent 65535 means none.
	- Added CHUNK_PRJ and CHUNK_MLI to allow inspecting .PRJ and
	  .MLI files (they're basically the same chunks as .3DS).
	- Added banner to identify myself, and disclaimer for "just in
	  case" possibilities.
	- Corrected possible bug when chunklen == 0 (it was not a
	  chunk).
	- Added several name descriptions of chunks. Use diff to find
	  all the new chunks.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifndef PI
#define PI 3.141592687
#endif

typedef unsigned char  byte;
typedef unsigned short word;
typedef unsigned long  dword;

typedef signed char  sbyte;
typedef signed short sword;
typedef signed long  sdword;

#pragma pack(2)

typedef struct {
    word    id;
    dword   len;
} TChunkHeader, *PChunkHeader;

#pragma pack()

enum {
    CHUNK_RGBF	    = 0x0010,
    CHUNK_RGBB	    = 0x0011,
//    CHUNK_RBGB2     = 0x0012,       // ?? NOT HLS.
    CHUNK_WRD	    = 0x0030,

    CHUNK_PRJ	    = 0xC23D,
    CHUNK_MLI	    = 0x3DAA,
    CHUNK_SHP	    = 0x2D2D,
    CHUNK_LFT	    = 0x2D3D,

    CHUNK_MAIN	    = 0x4D4D,
	CHUNK_OBJMESH	= 0x3D3D,
	    CHUNK_BKGCOLOR  = 0x1200,
	    CHUNK_AMBCOLOR  = 0x2100,
	    CHUNK_OBJBLOCK  = 0x4000,
		CHUNK_TRIMESH	= 0x4100,
		    CHUNK_VERTLIST  = 0x4110,
		    CHUNK_VERTFLAGS = 0x4111,
		    CHUNK_FACELIST  = 0x4120,
		       CHUNK_FACEMAT   = 0x4130,
		    CHUNK_MAPLIST   = 0x4140,
		    CHUNK_SMOOLIST  = 0x4150,
		    CHUNK_TRMATRIX  = 0x4160,
		    CHUNK_MESHCOLOR = 0x4165,
		    CHUNK_TXTINFO   = 0x4170,
		CHUNK_LIGHT	= 0x4600,
		    CHUNK_LIGHTR1  = 0x4659,
		    CHUNK_LIGHTR2  = 0x465A,
		    CHUNK_LIGHTMUL = 0x465B,
		    CHUNK_SPOTLIGHT = 0x4610,
		CHUNK_CAMERA	= 0x4700,
		CHUNK_HIERARCHY = 0x4F00,
	CHUNK_VIEWPORT = 0x7001,
	CHUNK_MATERIAL	       = 0xAFFF,
	    CHUNK_MATNAME      = 0xA000,
	    CHUNK_AMBIENT      = 0xA010,
	    CHUNK_DIFFUSE      = 0xA020,
	    CHUNK_SPECULAR     = 0xA030,
	    CHUNK_SHININESS    = 0xA040,
	    CHUNK_SHSTRENGTH   = 0xA041,
	    CHUNK_TRANSP       = 0xA050,
	    CHUNK_TRFALLOFF    = 0xA052,
	    CHUNK_REFLECTBL    = 0xA053,
	    CHUNK_FLAG2SIDE    = 0xA081,
	    CHUNK_FLAGADD      = 0xA083,
	    CHUNK_SELFILLUM    = 0xA084,
	    CHUNK_WIREFRAME    = 0xA085, // Flag
	    CHUNK_WIRETHICK    = 0xA087, // Float
	    CHUNK_FACEMAP      = 0xA088,   // Flag
	    CHUNK_FLAGIN       = 0xA08A,
	    CHUNK_SOFTEN       = 0xA08C,    // Flag
	    CHUNK_FLAGUNIT     = 0xA08E,  // Very useless
	    CHUNK_SHADETYPE    = 0xA100,
	    CHUNK_TEXTURE      = 0xA200,
	    CHUNK_TEXTUREMASK  = 0xA33E,
	    CHUNK_TEXTURE2     = 0xA33A,
	    CHUNK_TEXTUREMASK2 = 0xA340,
	    CHUNK_REFMAP       = 0xA220,
	    CHUNK_REFMASK      = 0xA34C,
	    CHUNK_OPACITYMAP   = 0xA210,
	    CHUNK_OPACITYMASK  = 0xA342,
	    CHUNK_SPECULARMAP  = 0xA204,
	    CHUNK_SPECULARMASK = 0xA348,
	    CHUNK_SHININESSMAP = 0xA33C,
	    CHUNK_SHININESSMASK= 0xA346,
	    CHUNK_SELFILLUMMAP = 0xA33D,
	    CHUNK_SELFILLUMMASK= 0xA34A,
	    CHUNK_BUMPMAP      = 0xA230,
	    CHUNK_BUMPMASK     = 0xA344,
	    CHUNK_MAPFILE      = 0xA300,
	      CHUNK_TFLAGS     = 0xA351, // Flags
	      CHUNK_BLUR       = 0xA353, // Float
	      CHUNK_USCALE     = 0xA354, // Float
	      CHUNK_VSCALE     = 0xA356, // Float
	      CHUNK_UOFFSET    = 0xA358, // Float
	      CHUNK_VOFFSET    = 0xA35A, // Float
	      CHUNK_TROTATION  = 0xA35C, // Float
	      CHUNK_BLACKTINT  = 0xA360, // Float
	      CHUNK_WHITETINT  = 0xA362, // Float
	      CHUNK_RTINT      = 0xA364, // Float
	      CHUNK_GTINT      = 0xA366, // Float
	      CHUNK_BTINT      = 0xA368, // Float
	CHUNK_KEYFRAMER = 0xB000,
	    CHUNK_AMBIENTKEY	= 0xB001,
	    CHUNK_TRACKINFO = 0xB002,
		CHUNK_TRACKOBJNAME  = 0xB010,
		CHUNK_TRACKPIVOT    = 0xB013,
		CHUNK_TRACKPOS	    = 0xB020,
		CHUNK_TRACKROTATE   = 0xB021,
		CHUNK_TRACKSCALE    = 0xB022,
		CHUNK_TRACKMORPH    = 0xB026,
		CHUNK_TRACKHIDE     = 0xB029,
		CHUNK_OBJNUMBER     = 0xB030,
	    CHUNK_TRACKCAMERA = 0xB003,
		CHUNK_TRACKFOV	= 0xB023,
		CHUNK_TRACKROLL = 0xB024,
	    CHUNK_TRACKCAMTGT = 0xB004,
	    CHUNK_TRACKLIGHT  = 0xB005,
	    CHUNK_TRACKLIGTGT = 0xB006,
	    CHUNK_TRACKSPOTL  = 0xB007,
	    CHUNK_FRAMES    = 0xB008,


};

// ------------------------------------

    // Forward declaration.
void ChunkReader (FILE *f, int ind, long p);

void SkipReader (FILE *f, int ind, long p) {
}

void RGBFReader (FILE *f, int ind, long p) {
    float c[3];
    if (fread(&c, sizeof(c), 1, f) != 1) return;
    TRACE("%*s    Red: %f, Green: %f, Blue: %f\n", ind, "", c[0], c[1], c[2]);
}

void RGBBReader (FILE *f, int ind, long p) {
    byte c[3];
    if (fread(&c, sizeof(c), 1, f) != 1) return;
    TRACE("%*s    Red: %d, Green: %d, Blue: %d\n", ind, "", c[0], c[1], c[2]);
}



void WRDReader (FILE *f, int ind, long p) {
    word value;
    if (fread(&value, sizeof(value), 1, f) != 1) return;
    TRACE("%*s    Value: %d\n", ind, "", value);
}

void FloatReader (FILE *f, int ind, long p) {
    float value;
    if (fread(&value, sizeof(value), 1, f) != 1) return;
    TRACE("%*s    Value: %f\n", ind, "", value);
}

void TextFlagReader (FILE *f, int ind, long p) {
    int i=0;
    byte flag1, flag2;

    TRACE("%*s  ", ind, "");
    if (fread(&flag1, sizeof(flag1), 1, f) != 1) return;
    if (fread(&flag2, sizeof(flag2), 1, f) != 1) return;
    i++;
    TRACE("Source:");
    if((flag1 & 0xc0)==0x80) TRACE("RGB luma tint\n");
    else{
      if((flag1 & 0xc0)==0xc0) TRACE("Alpha tint\n");
      else{
	if((flag2 & 0x02)==0x02) TRACE("RGB tint\n");
	else TRACE("RGB\n");
      }
    }
    TRACE("%*s  ", ind, "");
    TRACE("Filtering:");
    if (flag1 & 0x40) TRACE("Summed arial\n");
    else TRACE("Pyramidal\n");
    TRACE("%*s  ", ind, "");
    TRACE("Other parameters:");
    switch((flag1 & 0x11)){
      case 0x11 : TRACE("Decal"); break;
      case 0x00 : TRACE("Tile"); break;
      case 0x10 :
      case 0x01 : TRACE("Tile & Decal");
    }
    if(flag2 & 1) TRACE(" - Ignoge map alpha");
    if(flag1 & 8) TRACE(" - Negative");
    if(flag1 & 2) TRACE(" - Mirror");
    TRACE("\n%*s  ", ind, "");
    TRACE("%02X ", flag1); TRACE("%02X ", flag2);
    TRACE("\n");
}

void SHTReader (FILE *f, int ind, long p) {
    word value;
    if (fread(&value, sizeof(value), 1, f) != 1) return;
    switch(value){
      case 1: TRACE("%*s    Flat\n", ind, ""); break;
      case 2: TRACE("%*s    Gouraud\n", ind, ""); break;
      case 3: TRACE("%*s    Phong\n", ind, ""); break;
      case 4: TRACE("%*s    Metal\n", ind, ""); break;
    }
}

const char* ASCIIZReader (FILE *f, int ind, long p) {
    int c;
	static char buffer[300];
	int w=0; //pointer

	// Read ASCIIZ name
    while ( (c = fgetc(f)) != EOF && c != '\0')	{
		putchar(c);
		buffer[w++]=c;
	}
    
	TRACE("\"\n");
	buffer[w]=NULL; //close
	return buffer;
}

void ObjBlockReader (FILE *f, int ind, long p) {
//    int c;

	// Read ASCIIZ object name

    
    GV_pModel->m_ObjList.push_back(CObj(ASCIIZReader(f, ind, p)));
	CObj& obj= GV_pModel->m_ObjList.back();

	//TRACE("%*s  Object name <%s> \n", ind, GV_pModel->m_ObjList.back().name);
	TRACE("Object name index: %d  name: %s\n", GV_pModel->m_ObjList.size()-1, obj.name);
	CurObj= &GV_pModel->m_ObjList.back();

	// Read rest of chunks inside this one.
    ChunkReader(f, ind, p);
}

void VertListReader (FILE *f, int ind, long p) {
    word nv;
    float c[3];

    if (fread(&nv, sizeof(nv), 1, f) != 1) return;
    //TRACE("%*sVertices: %d\n", ind, "", nv);
    while (nv-- > 0) {
	if (fread(&c, sizeof(c), 1, f) != 1) return;
	 //TRACE("%*s    X: %f, Y: %f, Z: %f\n", ind, "", c[0], c[1], c[2]);
	 //CurObj->v.push_back(CVertex(c[0], c[2], -c[1]));
	  CurObj->v.push_back(CVertex(c[0], c[1], c[2]));
    }
}

void FaceListReader (FILE *f, int ind, long p) {
    word nv;
    word c[3];
    word flags;

    if (fread(&nv, sizeof(nv), 1, f) != 1) return;
    //TRACE("%*sFaces: %d\n", ind, "", nv);
    while (nv-- > 0) {
	if (fread(&c, sizeof(c), 1, f) != 1) return;
	if (fread(&flags, sizeof(flags), 1, f) != 1) return;
	/*TRACE("%*s  A %d, B %d, C %d, 0x%X:",
	       ind, "", c[0], c[1], c[2], flags);*/

	// add object
	CurObj->f.push_back(CFace());
	CFace& f= CurObj->f.back();
	f.a=c[0];
	f.b=c[1];
	f.c=c[2];
	f.m_pObj= CurObj; // parent object

	

//	  TRACE("%*s	 AB: %d, BC: %d, CA: %d, UWrap %d, VWrap %d\n",
//		 ind, "",
	TRACE(" AB %d BC %d CA %d UWrap %d VWrap %d\n",
	       (flags & 0x04) != 0, (flags & 0x02) != 0, (flags & 0x01) != 0,
	       (flags & 0x08) != 0, (flags & 0x10) != 0);
    }
	// Read rest of chunks inside this one.
    ChunkReader(f, ind, p);
}

//read ambient color
void AmbientReader(FILE* f, int ind, long p) {
	TChunkHeader h;// 6 bytes
	int a=sizeof(h);
	
	if (fread(&h, sizeof(h), 1, f) != 1) return;

	byte c[3];
    if (fread(&c, sizeof(c), 1, f) != 1) return;

	//set
	CurMat->m_fvAmbient[0]=(double)c[0]/255.0;
	CurMat->m_fvAmbient[1]=(double)c[1]/255.0;
	CurMat->m_fvAmbient[2]=(double)c[2]/255.0;
	CurMat->m_fvAmbient[3]=0.5f;
}

//read shininess
void ShininessReader(FILE* f, int ind, long p) {
	TChunkHeader h;// 6 bytes
	int a=sizeof(h);
	
	if (fread(&h, sizeof(h), 1, f) != 1) return;

	word value;
    if (fread(&value, sizeof(value), 1, f) != 1) return;

	//set
	CurMat->m_fShine= (float) value / 100.0;
}
//read ambient color
void SpecularReader(FILE* f, int ind, long p) {
	TChunkHeader h;// 6 bytes
	int a=sizeof(h);
	
	if (fread(&h, sizeof(h), 1, f) != 1) return;

	byte c[3];
    if (fread(&c, sizeof(c), 1, f) != 1) return;
	//set
	CurMat->m_fvSpecular[0]=(double)c[0]/255.0;
	CurMat->m_fvSpecular[1]=(double)c[1]/255.0;
	CurMat->m_fvSpecular[2]=(double)c[2]/255.0;
	CurMat->m_fvSpecular[3]=0.5f;
}
//read ambient color
void DiffuseReader(FILE* f, int ind, long p) {
	TChunkHeader h;// 6 bytes
	int a=sizeof(h);
	
	if (fread(&h, sizeof(h), 1, f) != 1) return;

	byte c[3];
    if (fread(&c, sizeof(c), 1, f) != 1) return;

	//set
	CurMat->m_fvDiffuse[0]=(double)c[0]/255.0;
	CurMat->m_fvDiffuse[1]=(double)c[1]/255.0;
	CurMat->m_fvDiffuse[2]=(double)c[2]/255.0;
	CurMat->m_fvDiffuse[3]=0.5f;
}


void FaceMatReader (FILE *f, int ind, long p) {
//    int c;
    word n, nf;

	// Read ASCIIZ material name
    //TRACE("%*sMaterial name for faces: \"", ind, "");
	char name[300];
	strcpy(name, ASCIIZReader(f, ind, p));
	strcpy(CurObj->mat, name); // give material to object

	ASSERT(GV_pModel); // valid model pointer
	CMtlList& mtls = GV_pModel->m_MatList;
	// find material
	CMtlList::iterator mi=mtls.Find(name);
	ASSERT(mi);
	CMtl& m= *mi; // we got material
	CFaceList&	face=CurObj->f;// facelist to set

    if (fread(&n, sizeof(n), 1, f) != 1) return;
	TRACE("%*sFaces with this material: %d\n", ind, "", n);

    while (n-- > 0) {
	if (fread(&nf, sizeof(nf), 1, f) != 1) return;
		TRACE("%*s    Face %d\n",     ind, "", nf);
		face[nf].m_pMtl= &m;
    }
}

void MapListReader (FILE *f, int ind, long p) {
    word nv;
    float c[2];

    if (fread(&nv, sizeof(nv), 1, f) != 1) return;
    //TRACE("%*sVertices: %d\n", ind, "", nv);
    while (nv-- > 0) {
	if (fread(&c, sizeof(c), 1, f) != 1) return;
	//TRACE("%*s    U: %f, V: %f\n", ind, "", c[0], c[1]);
    }
}

void SmooListReader (FILE *f, int ind, long p) {
    dword s;
    int i;

	//at this point, we have face list..
	//find only one smoothing group..
	CFaceList::iterator fi = CurObj->f.begin();

	//process
    while (ftell(f) < p) {
	if (fread(&s, sizeof(s), 1, f) != 1) return;
	TRACE("%*sSmoothing groups: ", ind, "");

	

	for (i = 0; i < 32; i++) 
    if (s & ((dword)1 << i)) {
			TRACE("%d, ", i + 1);

			
		}
	//store all flags
	fi->s= s;

	fi++;

	TRACE("\n");
    }

}

void TrMatrixReader(FILE *f, int ind, long p) {
    float rot[9];
    float trans[3];
    if (fread(&rot, sizeof(rot), 1, f) != 1) return;
    TRACE("%*sRotation matrix:\n", ind, "");
    TRACE("%*s    %f, %f, %f\n", ind, "", rot[0], rot[1], rot[2]);
    TRACE("%*s    %f, %f, %f\n", ind, "", rot[3], rot[4], rot[5]);
    TRACE("%*s    %f, %f, %f\n", ind, "", rot[6], rot[7], rot[8]);
    if (fread(&trans, sizeof(trans), 1, f) != 1) return;
    TRACE("%*sTranslation matrix: %f, %f, %f\n",
	   ind, "", trans[0], trans[1], trans[2]);


	//copy to the obj
	MAT m(4,4),m2(4,4);
	VECTOR piv(trans[0], trans[1], trans[2]);
	m(0,0)=rot[0],	m(0,1)=rot[1],	m(0,2)=rot[2],	m(0,3)=0;
	m(1,0)=rot[3],	m(1,1)=rot[4],	m(1,2)=rot[5],	m(1,3)=0;
	m(2,0)=rot[6],	m(2,1)=rot[7],	m(2,2)=rot[8],	m(2,3)=0;
	m(3,0)=0,	m(3,1)=0,	m(3,2)=0,	m(3,3)=1;
	m2= ~m;//get inverse matrix


	//process vertices
	CVertexList::iterator vi;
	CVertexList& v= CurObj->v;
	for (vi=v.begin(); vi!=v.end(); vi++) {
		vi->v -= piv; //subtract pivot
		vi->v *= m2; //multiply inverse of transform matrix
	}
  

}

void LightReader(FILE *f, int ind, long p) {
    float c[3];
    if (fread(&c, sizeof(c), 1, f) != 1) return;
    TRACE("%*s    X: %f, Y: %f, Z: %f\n", ind, "", c[0], c[1], c[2]);
	// Read rest of chunks inside this one.
    ChunkReader(f, ind, p);
}

void SpotLightReader(FILE *f, int ind, long p) {
    float c[5];
    if (fread(&c, sizeof(c), 1, f) != 1) return;
    TRACE("%*s    Target X: %f, Y: %f, Z: %f; Hotspot %f, Falloff %f\n",
	   ind, "", c[0], c[1], c[2], c[3], c[4]);
}

void CameraReader(FILE *f, int ind, long p) {
    float c[8];
    if (fread(&c, sizeof(c), 1, f) != 1) return;
    TRACE("%*s    Position: X: %f, Y: %f, Z: %f\n", ind, "", c[0], c[1], c[2]);
    TRACE("%*s    Target: X: %f, Y: %f, Z: %f\n", ind, "", c[3], c[4], c[5]);
    TRACE("%*s    Bank: %f, Lens: %f\n", ind, "", c[6], c[7]);
}

void MatNameReader (FILE *f, int ind, long p) {
//    int c;

	// Read ASCIIZ object name
    TRACE("%*sMaterial name \"", ind, "");
    GV_pModel->m_MatList.push_back(CMtl(ASCIIZReader(f, ind, p)));
	CurMat= &GV_pModel->m_MatList.back();

}

void MapFileReader(FILE *f, int ind, long p) {
//    int c;

	// Read ASCIIZ filename
    TRACE("%*sMap filename \"", ind, "");
    ASCIIZReader(f, ind, p);
}

void FramesReader(FILE *f, int ind, long p) {
    dword c[2];
    if (fread(&c, sizeof(c), 1, f) != 1) return;
    TRACE("%*s    Start: %ld, End: %ld\n",
	   ind, "", c[0], c[1]);

	//read frame start ,end
	GV_pModel->m_nStart=c[0]; //start
	GV_pModel->m_nEnd=c[1]; //end
}

void TrackObjNameReader(FILE *f, int ind, long p) {
//    int c;
    word w[2];
    word parent;

	// Read ASCIIZ name
	CObjList& objlist= GV_pModel->m_ObjList;
	CObjpList& objp= GV_pModel->m_ObjpList; //ojbect pointer list

    TRACE("%*sTrack object name \"", ind, "");
	CurObj=GV_pModel->FindObj(ASCIIZReader(f, ind, p));
	//if it's dummy make a new object
	if (CurObj==NULL) { //it's group name
		objlist.push_back(CObj("dummy"));
		CurObj= & objlist.back();
	}
	//add object pointer..
	//this object pointer list includes dummy object(group)
	objp.push_back(CurObj);

    if (fread(&w, sizeof(w), 1, f) != 1) return;
    if (fread(&parent, sizeof(parent), 1, f) != 1) return;
    TRACE("%*sObject name data: Flags 0x%X, 0x%X, Parent %d\n",
	   ind, "", w[0], w[1], parent);


	//make node at this moment
	CObjNodeList& list= GV_pModel->m_Nodes;
	CObjNode& root= GV_pModel->m_Root;
	//add a new node
	list.push_back(CObjNode());
	CObjNode& new2= list.back(); //this a new node

	//connect
	new2.m_pObj= CurObj; //connect node and object
	CurObj->m_pNode= & new2;
	
	//connect to the proper branch
	if (parent==65535) {
		GV_pModel->m_Root.AddChild(&new2);
	}
	else {
		CObj* obj= objp[parent];//get parent object
		
		ASSERT(obj);
		CObjNode* node= obj->m_pNode; //get node of the object
		ASSERT(node);
		node->AddChild(&new2); 
	}

}

void PivotPointReader(FILE *f, int ind, long p) {
    float pos[3];

    if (fread(&pos, sizeof(pos), 1, f) != 1) return;
    TRACE("%*s  Pivot at X: %f, Y: %f, Z: %f\n",
	   ind, "",
	   pos[0], pos[1], pos[2]);

	CurObj->m_Piv=VECTOR(pos[0], pos[1], pos[2]);

	CVertexList& v=CurObj->v;
	CVertexList::iterator vi;
	for (vi=v.begin(); vi!=v.end(); vi++) {
		vi->v -= CurObj->m_Piv; 
	}
}

    /* Key info flags for position, rotation and scaling:
	Until I know the meaning of each bit in flags I assume all mean
	a following float data.
    */

	// NOTE THIS IS NOT A CHUNK, but A PART OF SEVERAL CHUNKS
void SplineFlagsReader(FILE *f, int ind, word flags) {
    int i;
    float dat;

    for (i = 0; i < 16; i++) {
	static const char *flagnames[] = {
	    "Tension",
	    "Continuity",
	    "Bias",
	    "Ease To",
	    "Ease From",
	};
	if (flags & (1 << i)) {
	    if (fread(&dat, sizeof(dat), 1, f) != 1) return;
	    if (i < sizeof(flagnames)/sizeof(*flagnames))
		TRACE("%*s		%-15s = %f\n",
		       ind, "", flagnames[i], dat);
	    else
		TRACE("%*s		%-15s = %f\n",
		       ind, "", "Unknown", dat);
	}
    }
}

void TrackPosReader(FILE *f, int ind, long p) {
    word n, nf;
    float pos[3];
    word unkown;
    word flags;
    int i;

    for(i=0; i<5; i++) {
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	TRACE("%*sUnknown #%d: 0x%x\n", ind, "", i, unkown);
    }
    if (fread(&n, sizeof(n), 1, f) != 1) return;
    TRACE("%*sPosition keys: %d\n", ind, "", n);
    if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
    TRACE("%*sUnknown: 0x%x\n", ind, "", unkown);
    while (n-- > 0) {
	if (fread(&nf, sizeof(nf), 1, f) != 1) return;
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	if (fread(&flags, sizeof(flags), 1, f) != 1) return;
	TRACE("%*s  Frame %3d: Flags 0x%X\n", ind, "", nf, flags);
	SplineFlagsReader(f, ind, flags);
	if (fread(&pos, sizeof(pos), 1, f) != 1) return;
	TRACE("%*s		X: %f, Y: %f, Z: %f\n",
	       ind, "", pos[0], pos[1], pos[2]);

	 //add position keys
		CurObj->m_PosKeys.push_back(CPosKey(nf, VECTOR(pos[0],pos[1],pos[2])));
    }

	//read the last position
	CurObj->m_Pos=VECTOR(pos[0], pos[1], pos[2]);
}

void TrackRotReader(FILE *f, int ind, long p) {
    word n, nf;
    float pos[4];
    word unkown;
    word flags;
    int i;

	//angle is accumulative
	float angle=0.0f;
	int first=true; //first?
	ROT rot; //rotation quaternion

    for(i=0; i<5; i++) {
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	TRACE("%*sUnknown #%d: 0x%x\n", ind, "", i, unkown);
    }
    if (fread(&n, sizeof(n), 1, f) != 1) return;
    TRACE("%*sRotation keys: %d\n", ind, "", n);
    if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
    TRACE("%*sUnknown: 0x%x\n", ind, "", unkown);
    while (n-- > 0) {
	if (fread(&nf, sizeof(nf), 1, f) != 1) return;
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	if (fread(&flags, sizeof(flags), 1, f) != 1) return;
	TRACE("%*s  Frame %3d: Flags 0x%X\n", ind, "", nf, flags);
	SplineFlagsReader(f, ind, flags);
	if (fread(&pos, sizeof(pos), 1, f) != 1) return;
	
	TRACE("%*s		Angle: %f, X: %f, Y: %f, Z: %f\n",
	       ind, "", pos[0]*180.0/PI, pos[1], pos[2], pos[3]);

	//add angle
	//angle is accumulative..
	//we don't need angle yet
		if (first) { 
			angle= pos[0]; rot=ROT(VECTOR(pos[1], pos[2], pos[3]), pos[0]);
			first=false;
		}
		else {
			//rot is inversed when it's applied to the object 
			// so inversed order here.
 			angle+=pos[0]; rot=rot*ROT(VECTOR(pos[1],pos[2],pos[3]),pos[0]);
		}

	 //add rotatin key
	CurObj->m_RotKeys.push_back(CRotKey(nf, rot));

    }

	CurObj->m_Rot=rot;//get accmulative value
		
}

void TrackScaleReader(FILE *f, int ind, long p) {
    word n, nf;
    float pos[3];
    word unkown;
    word flags;
    int i;

    for(i=0; i<5; i++) {
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	TRACE("%*sUnknown #%d: 0x%x\n", ind, "", i, unkown);
    }
    if (fread(&n, sizeof(n), 1, f) != 1) return;
    TRACE("%*sScale keys: %d\n", ind, "", n);
    if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
    TRACE("%*sUnknown: 0x%x\n", ind, "", unkown);
    while (n-- > 0) {
	if (fread(&nf, sizeof(nf), 1, f) != 1) return;
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	if (fread(&flags, sizeof(flags), 1, f) != 1) return;
	TRACE("%*s  Frame %3d: Flags 0x%X\n", ind, "", nf, flags);
	SplineFlagsReader(f, ind, flags);
	if (fread(&pos, sizeof(pos), 1, f) != 1) return;
	TRACE("%*s	       X: %f, Y: %f, Z: %f\n",
	       ind, "", pos[0], pos[1], pos[2]);

		//add scale keys
		CurObj->m_SclKeys.push_back(CSclKey(nf, VECTOR(pos[0],pos[1],pos[2])));
    }


	CurObj->m_Scl=VECTOR( pos[0], pos[1], pos[2]);
}

void TrackFovReader(FILE *f, int ind, long p) {
    word n, nf;
    float fov;
    word unkown;
    word flags;
    int i;

    for(i=0; i<5; i++) {
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	TRACE("%*sUnknown #%d: 0x%x\n", ind, "", i, unkown);
    }
    if (fread(&n, sizeof(n), 1, f) != 1) return;
    TRACE("%*sFOV keys: %d\n", ind, "", n);
    if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
    TRACE("%*sUnknown: 0x%x\n", ind, "", unkown);
    while (n-- > 0) {
	if (fread(&nf, sizeof(nf), 1, f) != 1) return;
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	if (fread(&flags, sizeof(flags), 1, f) != 1) return;
	TRACE("%*s  Frame %3d: Flags 0x%X\n", ind, "", nf, flags);
	SplineFlagsReader(f, ind, flags);
	if (fread(&fov, sizeof(fov), 1, f) != 1) return;
	TRACE("%*s	       FOV: %f\n",
	       ind, "", fov);
    }
}

void TrackRollReader(FILE *f, int ind, long p) {
    word n, nf;
    float roll;
    word unkown;
    word flags;
    int i;

    for(i=0; i<5; i++) {
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	TRACE("%*sUnknown #%d: 0x%x\n", ind, "", i, unkown);
    }
    if (fread(&n, sizeof(n), 1, f) != 1) return;
    TRACE("%*sRoll keys: %d\n", ind, "", n);
    if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
    TRACE("%*sUnknown: 0x%x\n", ind, "", unkown);
    while (n-- > 0) {
	if (fread(&nf, sizeof(nf), 1, f) != 1) return;
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	if (fread(&flags, sizeof(flags), 1, f) != 1) return;
	TRACE("%*s  Frame %3d: Flags 0x%X\n", ind, "", nf, flags);
	SplineFlagsReader(f, ind, flags);
	if (fread(&roll, sizeof(roll), 1, f) != 1) return;
	TRACE("%*s	       Roll: %f\n",
	       ind, "", roll);
    }
}

void TrackMorphReader(FILE *f, int ind, long p) {
    word n, nf;
    word unkown;
    word flags;
    int i;

    for(i=0; i<5; i++) {
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	TRACE("%*sUnknown #%d: 0x%x\n", ind, "", i, unkown);
    }
    if (fread(&n, sizeof(n), 1, f) != 1) return;
    TRACE("%*sMorph keys: %d\n", ind, "", n);
    if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
    TRACE("%*sUnknown: 0x%x\n", ind, "", unkown);
    while (n-- > 0) {
	if (fread(&nf, sizeof(nf), 1, f) != 1) return;
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	if (fread(&flags, sizeof(flags), 1, f) != 1) return;
	TRACE("%*s  Frame %3d: Flags 0x%X\n", ind, "", nf, flags);
	SplineFlagsReader(f, ind, flags);
	TRACE("%*s		Object name: \"", ind, "");
	ASCIIZReader(f, ind, p);
    }
}

void TrackHideReader(FILE *f, int ind, long p) {
//    word  nf;
	word n;
    word frame;
    word unkown;
//    word flags;
    int i;

    for(i=0; i<5; i++) {
	if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	TRACE("%*sUnknown #%d: 0x%x\n", ind, "", i, unkown);
    }
    if (fread(&n, sizeof(n), 1, f) != 1) return;
    TRACE("%*sHide keys: %d\n", ind, "", n);
    if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
    TRACE("%*sUnknown: 0x%x\n", ind, "", unkown);
    ind += 2;
    while (n-- > 0) {
	if (fread(&frame, sizeof(frame), 1, f) != 1) return;
	TRACE("%*sFrame: %d\n", ind, "", (dword) frame);
	for(i=0; i<2; i++) {
	    if (fread(&unkown, sizeof(unkown), 1, f) != 1) return;
	    TRACE("%*s  Unknown #%d: 0x%x\n", ind, "", i, unkown);
	}
    }
}



void ObjNumberReader(FILE *f, int ind, long p) {
    word n;

    if (fread(&n, sizeof(n), 1, f) != 1) return;
    TRACE("%*sObject number: %d\n", ind, "", n);

	TRACE("Object Number %d / %d \n", n, GV_pModel->m_ObjList.size());
}


// ------------------------------------

struct {
    word id;
    const char *name;
    void (*func)(FILE *f, int ind, long p);
} ChunkNames[] = {
    {CHUNK_RGBF,	"RGB float",	    RGBFReader},
    {CHUNK_RGBB,	"RGB byte",	    RGBBReader},
    {CHUNK_WRD, 	"WORD", 	    WRDReader},

    {CHUNK_PRJ, 	"Project",	    NULL},
    {CHUNK_MLI, 	"Material Library", NULL},
    {CHUNK_SHP, 	"Shape file", NULL},
    {CHUNK_LFT, 	"Loft file", NULL},

    {CHUNK_MAIN,	"Main", 	    NULL},
    {CHUNK_OBJMESH,	"Object Mesh",	    NULL},
    {CHUNK_BKGCOLOR,	"Background color", NULL},
    {CHUNK_AMBCOLOR,	"Ambient color",    NULL},
    {CHUNK_OBJBLOCK,	"Object Block",     ObjBlockReader},
    {CHUNK_TRIMESH,	"Tri-Mesh",	    NULL},
    {CHUNK_VERTLIST,	"Vertex list",	    VertListReader},
    {CHUNK_VERTFLAGS,	"Vertex flag list", SkipReader},
    {CHUNK_FACELIST,	"Face list",	    FaceListReader},
    {CHUNK_MESHCOLOR,	"Mesh color",	    SkipReader},
    {CHUNK_FACEMAT,	"Face material",    FaceMatReader},
    {CHUNK_MAPLIST,	"Mappings list",    MapListReader},
    {CHUNK_TXTINFO,	"Texture info",     SkipReader},
    {CHUNK_SMOOLIST,	"Smoothings",	    SmooListReader},
    {CHUNK_TRMATRIX,	"Matrix",	    TrMatrixReader},
    {CHUNK_LIGHT,	"Light",	    LightReader},
    {CHUNK_LIGHTR1,	"Light Range radius1",	       FloatReader},
    {CHUNK_LIGHTR2,	"Light Range radius2",	       FloatReader},
    {CHUNK_LIGHTMUL,	"Light Multyplity",	       FloatReader},
    {CHUNK_SPOTLIGHT,	"Spotlight",	    SpotLightReader},
    {CHUNK_CAMERA,	"Camera",	    CameraReader},
    {CHUNK_HIERARCHY,	"Hierarchy",	    NULL},

    {CHUNK_VIEWPORT,	"Viewport info",    SkipReader},

    {CHUNK_MATERIAL,	 "Material",		 NULL},
    {CHUNK_MATNAME,	 "Material name",	 MatNameReader},
    {CHUNK_AMBIENT,	 "Ambient color",	  AmbientReader},
    {CHUNK_DIFFUSE,	 "Diffuse color",	 DiffuseReader},
    {CHUNK_SPECULAR,	 "Specular color",	 SpecularReader},
    {CHUNK_SHININESS,	 "Shininess",		 ShininessReader},
    {CHUNK_SHSTRENGTH,	 "Shininess strength",	 NULL},
    {CHUNK_TRANSP,	 "Transparency",	 NULL},
    {CHUNK_TRFALLOFF,	 "Transparency falloff", NULL},
    {CHUNK_REFLECTBL,	 "Reflection blur",	 NULL},
    {CHUNK_SELFILLUM,	 "Self Illumination",	 NULL},
    {CHUNK_FLAG2SIDE,	 "2Side: ON /flag/",	 SkipReader},
    {CHUNK_FLAGADD,	 "Transparency: Add /flag/",		    SkipReader},
    {CHUNK_WIREFRAME,	 "WireFrame: ON /flag/", SkipReader},
    {CHUNK_WIRETHICK,	 "WireFrame thickness",  FloatReader},
    {CHUNK_FLAGIN,	 "Transparency falloff: In /flag/",	    SkipReader},
    {CHUNK_FLAGUNIT,	 "WireFrame thickness: Unit /flag/",	    SkipReader},
    {CHUNK_FACEMAP,	 "Self Illumination: FaceMap /flag/",	    SkipReader},
    {CHUNK_SOFTEN,	 "Shininess: Soften /flag/",	    SkipReader},
    {CHUNK_SHADETYPE,	 "Shadeing type",		    SHTReader},
    {CHUNK_TEXTURE,	 "Texture map", 		    NULL},
    {CHUNK_SPECULARMAP,  "Specular map",		    NULL},
    {CHUNK_OPACITYMAP,	 "Opacity map", 		    NULL},
    {CHUNK_TEXTURE2,	 "Texture map 2",		    NULL},
    {CHUNK_SHININESSMAP, "Shininess map",		    NULL},
    {CHUNK_SELFILLUMMAP, "Self Illumination map",	    NULL},
    {CHUNK_REFMAP,	 "Reflection map",		    NULL},
    {CHUNK_BUMPMAP,	 "Bump map",			    NULL},
    {CHUNK_TEXTUREMASK,  "Texture mask",		    NULL},
    {CHUNK_SPECULARMASK, "Specular mask",		    NULL},
    {CHUNK_OPACITYMASK,  "Opacity mask",		    NULL},
    {CHUNK_TEXTUREMASK2, "Texture2 mask",		    NULL},
    {CHUNK_SHININESSMASK,"Shininess mask",		    NULL},
    {CHUNK_SELFILLUMMASK,"Self Illumination mask",	    NULL},
    {CHUNK_REFMASK,	 "Reflection mask",		    NULL},
    {CHUNK_BUMPMASK,	 "Bump mask",			    NULL},
    {CHUNK_TFLAGS,	 "Texture flags",		    TextFlagReader},
    {CHUNK_BLUR,	 "Texture Blur",		    FloatReader},
    {CHUNK_USCALE,	 "U scale (1/x)",		    FloatReader},
    {CHUNK_VSCALE,	 "V scale (1/x)",		    FloatReader},
    {CHUNK_UOFFSET,	 "U offset",			    FloatReader},
    {CHUNK_VOFFSET,	 "V offset",			    FloatReader},
    {CHUNK_TROTATION,	 "Texture Rotation angle",	    FloatReader},
    {CHUNK_BLACKTINT,	 "Black Tint",			    RGBBReader},
    {CHUNK_WHITETINT,	 "White Tint",			    RGBBReader},
    {CHUNK_RTINT,	 "Red Tint",			    RGBBReader},
    {CHUNK_GTINT,	 "Green Tint",			    RGBBReader},
    {CHUNK_BTINT,	 "Blue Tint",			    RGBBReader},
    {CHUNK_MAPFILE,	 "Map filename",		    MapFileReader},

    {CHUNK_KEYFRAMER,	"Keyframer data",   NULL},
    {CHUNK_AMBIENTKEY,	"Ambient key",	    NULL},
    {CHUNK_TRACKINFO,	"Track info",	    NULL},
    {CHUNK_FRAMES,	"Frames",	    FramesReader},
    {CHUNK_TRACKOBJNAME,"Track Obj. Name",  TrackObjNameReader},
    {CHUNK_TRACKPIVOT,	"Pivot point",	    PivotPointReader},
    {CHUNK_TRACKPOS,	"Position keys",    TrackPosReader},
    {CHUNK_TRACKROTATE, "Rotation keys",    TrackRotReader},
    {CHUNK_TRACKSCALE,	"Scale keys",	    TrackScaleReader},
    {CHUNK_TRACKMORPH,	"Morph keys",	    TrackMorphReader},
    {CHUNK_TRACKHIDE,	"Hide keys",	    TrackHideReader},
    {CHUNK_OBJNUMBER,	"Object number",    ObjNumberReader},

    {CHUNK_TRACKCAMERA, "Camera track", 	    NULL},
    {CHUNK_TRACKCAMTGT, "Camera target track",	    NULL},
    {CHUNK_TRACKLIGHT,	"Pointlight track",	    NULL},
    {CHUNK_TRACKLIGTGT, "Pointlight target track",  NULL},
    {CHUNK_TRACKSPOTL,	"Spotlight track",	    NULL},
    {CHUNK_TRACKFOV,	"FOV track",	    TrackFovReader},
    {CHUNK_TRACKROLL,	"Roll track",	    TrackRollReader},
};

int FindChunk(word id) {
    int i;
    for (i = 0; i < sizeof(ChunkNames)/sizeof(ChunkNames[0]); i++)
	if (id == ChunkNames[i].id)
	    return i;
    return -1;
}

// ------------------------------------

int Verbose = 0;
int Quiet   = 0;
int Dump    = 0;

void ChunkReader(FILE *f, int ind, long p) {
    TChunkHeader h;
    int n, i;
    byte d;
    long pc;

    while (ftell(f) < p) {
	pc = ftell(f);
	if (fread(&h, sizeof(h), 1, f) != 1) return;
	if (h.len == 0) return;
	n = FindChunk(h.id);

	if (n < 0) {
	    if (Verbose) {
		TRACE("%*sUnknown chunk: 0x%04X, offset 0x%lX, size: %d bytes.",
			ind, "", h.id, pc, h.len);
		if (!Dump)
		    TRACE("\n");
	    }
	    if (Dump) {
		fseek(f, pc + 6, SEEK_SET);
		for (i=0; i<h.len-6; i++) {
		    if ((i & 0xf) == 0) TRACE("\n%*s  ", ind, "");
		    if (fread(&d, sizeof(d), 1, f) != 1) return;
		    TRACE("%02X ", d);
		}
		TRACE("\n");
	    } else
		fseek(f, pc + h.len, SEEK_SET);
	} else {
	    if (!Quiet || ChunkNames[n].func == NULL)
		TRACE("%*sChunk type \"%s\", offset 0x%lX, size %d bytes\n",
		       ind, "", ChunkNames[n].name, pc, h.len);
	    pc = pc + h.len;
	    if (ChunkNames[n].func != NULL)
		ChunkNames[n].func(f, ind + 2, pc);
	    else
		ChunkReader(f, ind + 2, pc);
	    fseek(f, pc, SEEK_SET);
	}
	if (ferror(f))
	    break;
    }
}

// ------------------------------------

void Banner(void) {
    TRACE("3DSRDR, 3D Studio formats reader v1.2 by Javier Arevalo AKA Jare/Iguana.\n"
	   "Modified by Mats Byggmastar aka. MRI/Doomsday (v1.3)\n"
	   "& Modified again by Bezzegh Dnes aka. DiVeR /diver@inf.bme.hu/ (v1.4)\n");
}



//////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                           
//                   read 3ds and process quaternion and matrix                                                                         
//                                                                                           
//////////////////////////////////////////////////////////////////////////////////////////////


void CModelList::Read3DS(  char* name) {
	//okay add to the model list (copy it)
	GV_pModel=NULL;  //current 3ds file 
	CurObj=NULL; //current object
	CurMat=NULL; //current material

	push_back(CModel());
	CModel& tmp=back();
	tmp.Read3DS(name);

	// calculate bounding box..
	CalcBox();


	// print..
	char buffer[300];
	sprintf(buffer,"Min:%lf , %lf , %lf \n"
				"Max:%lf , %lf , %lf \n",
				m_vMin.x , m_vMin.y, m_vMin.z,
				m_vMax.x , m_vMax.y, m_vMax.z);
	TRACE(buffer);


}

void CModelList::CalcBox(void)
{
	// valid?
	if (empty()) return;

	// starter
	iterator mi=begin();
	m_vMin= mi->m_vMin;
	m_vMax= mi->m_vMax;

	for (mi=begin(); mi!=end(); mi++)  //model list
	{
		m_vMin.MakeMin(mi->m_vMin);
		m_vMax.MakeMax(mi->m_vMax);
	}
}

//read 3ds file
int CModel::Read3DS( char* name) {


    FILE *f;
    long p;


    f = fopen(name, "rb");
    if (f == NULL) {
	//Banner();
	TRACE("Can't open %s!\n", name);
	return false;
    }


	//start
	//initialize dialog box
	GV_pModel=this;
	

	// Find file size.
    fseek(f, 0, SEEK_END);
    p = ftell(f);
    fseek(f, 0, SEEK_SET);
	// Go!
    ChunkReader(f, 0, p);



	//copy file name
	strcpy(fname, name);




	//post process
	CObjList::iterator oi;
	for (oi=m_ObjList.begin(); oi!= m_ObjList.end(); oi++) {
		oi->UpdateData();
		oi->UpdateNormals();
	}
	//update hierarchy
	Process(&m_Root);

	// get bounding box
	CalcBox();


	//success
	return true;

}


//show models
void CModelList::Show3DS(int i) //index
{
	//CHECK
	int n=size();
	if (i<0 || i>=n) return; // not proper range.. exit.

	iterator oi=begin();
	advance(oi, i);
	oi->Show();
}

//show 3d model
void CModelList::Show(int i)
{
	glDisable(GL_COLOR_MATERIAL);
	Show3DS(i);
	//restore material attributes.
	//set up default value
	float specular[4]={0.0f, 0.0f, 0.0f, 1.0f}; //default value
	float emission[4]={0.0f, 0.0f, 0.0f, 1.0f}; //default value

	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
	glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, emission);
	glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, 0.0f);//default value
	glEnable(GL_COLOR_MATERIAL);
}



