#include "uox3.h"
#include "debug.h"

#define DEBUG_WALKING 0
#define DEBUG_WALK_BLOCK 0

#define DBGFILE "walking.cpp"

const signed char MAX_ITEM_Z_INFLUENCE = 10; // Any item above this height is discarded as being too far away to effect the char
const signed char MAX_ITEM_CLIMB_HEIGHT = 9; // Maximum total height a char could climb (for NPC's since the PC's get blocked by the client)

/*
** Welcome dear reader. I am going to take a stab at rewriting the walking code.
** I know, I know, You've probably heard that a dozen times before, right?
** What can I possibly hope to bring to the table here, where so many others
** before me have toiled in anguish? I will tell you, I am going to be the first
** to document what I have done, so even if I do fail, at least the next poor
** soul can pick up from where I left off instead of having to fight through
** it all from the very beginning.
**
** My approach is going to be this. First I'm going to rip apart the monolithic
** Walking() function into several smaller functions.  You are pretty much guarenteed of
** having bugs in any function thats 800+ lines long.  The advantage of this
** approach will be that, 1) having smaller functions make it easier to 'wrap
** your brain around' as Abaddon likes to say, and 2) itdecreases the chance of
**  side-effects between different pieces of code.  The disadvantage of this
** approach is that some things may be recalculated more than once because the
** values might be required in multiple places. This shouldn't be too much of
** a drawback in reality, because much of the code in the original version just
** recalculated anyways, eg, calls to online() and the like.  Since a grep for
** 'const' throughout the code yields amazingly few uses currently, the new
** code should be no slower than the original version.  During the process of
** pulling it apart, I will rename variables and fix whitespace to improve
** code readability.
**
** Once it has been pulled apart, I'm going to check it into CVS, so it can be
** tested. With no real functional changes made, it should work exactly like it
** did before.  The next step will be to put in a ton of #ifdef DEBUG_WALKING
** printf()s to display what is going on in greater detail. If possible, I'm 
** going to try to have it print out what the map around the character looks like
** complete with Z values, since that seems to be where most of the bugs are
** found.  After that the final step of rewriting/bug fixing should be a ton
** easier.
**
** fur 10/27/1999
*/

/*
** Walking() This function is called whenever we get a message from the client
** to walk/run somewhere.   It is also called by the NPC movement functions in this
** class to make the NPCs move.  The arguments are fairly fixed because we don't
** have a lot of control about what the client gives us.
**
** CHARACTER s - Obviously the character index of the character trying to move.
**
** dir - Which direction the character is trying to move. The first nibble holds
** the cardinal direction.      If the bit 0x80 is set, it means the character is
** running instead of walking.  
**              0: // North
**              1: // Northeast
**              2: // East
**              3: // Southeast
**              4: // South
**              5: // Southwest
**              6: // West
**              7: // Northwest
**
** sequence - This is what point in the walking sequence we are at, this seems to
**            roll over once it hits 256
**
*/
void cMovement::Walking(CHARACTER c, int dir, int sequence)
{
	// sometimes the NPC movement code comes up with -1, for example, if we are following someone
	// and we are directly on top of them
	if (-1 == dir) return;

    UOXSOCKET socket = calcSocketFromChar(c);
    
    if (!VerifySequence(c, socket, sequence))
        return;
    
    if (!CheckWeightFirst(c, socket))
        return;
    
    if (!CheckForFrozen(c, socket))
        return;
    
	// save our original location before we even think about moving
	const short int oldx = chars[c].x;
	const short int oldy = chars[c].y;
	
	// this if assumes that chars[s].dir has no high-bits just lets make sure of it
	// assert((chars[c].dir & 0xFFF0) == 0);
	// this assertion is failing, so either my assumption about it is wrong or there
	// is a bugaboo
	
	// see if we have stopped to turn or if we are moving
	const bool amTurning = (dir&0x0F) != chars[c].dir;
	if (!amTurning)
	{
		if (!CheckForRunning(c, socket, dir))
			return;
		
		if (!CheckForStealth(c, socket))
			return;
		
		/* this is already done in the cNetwork method that calls this, so its redundant here
        ** i'm leaving this in because it might make more sense to have it here because we can
        * call it only in the case of actual movement
        if (chars[c].med) //Morrolan - Meditation
        {
            chars[c].med=0; 
            sysmessage(c, "You break your concentration.");
        }
		*/
		
		// if this was an NPC lets reset their move timer
		// this seems to be an usual place within this function to reset this
		// i guess they can turn a whole lot this way
		if (chars[c].npc) 
		{
			chars[c].npcmovetime=(unsigned int)(uiCurrentTime+(double)(NPCSPEED*CLOCKS_PER_SEC)); //reset move timer
		}
		
		MoveCharForDirection(c, dir);
		
		if (!CrazyXYBlockStuff(c, socket, oldx, oldy, sequence))
			return;
		
		// i actually moved this for now after the z =  illegal_z, in the end of CrazyXYBlockStuff()
		// can't see how that would hurt anything
		if (!CheckForHouseBan(c, socket))
			return;
		
		/*
		** OK AT THIS POINT IT IS NOW OFFICIALLY A LEGAL MOVE TO MAKE, LETS GO FOR IT!
		**
		** That means any bugs concerning if a move was legal must be before this point!
		*/

		// i moved this down after we are certain we are moving
		if( server_data.footSteps )
  			playTileSound( socket );
		
		// since we actually moved, update the regions code
		HandleRegionStuffAfterMove(c, oldx, oldy);            
	}
	else
	{
		//printf("Player is turning in the same spot.\n");
	}
	
	// do all of the following regardless of whether turning or moving i guess
	
	// set the player direction to contain only the cardinal direction bits
	chars[c].dir = (dir&0x0F);
	
	SendWalkToPlayer(c, socket, sequence);
	
	SendWalkToOtherPlayers(c, dir, oldx, oldy);
	
	OutputShoveMessage(c, socket, oldx, oldy);
	
	// keep on checking this even if we just turned, because if you are taking damage
	// for standing here, lets keep on dishing it out. if we pass whether we actually
	// moved or not we can optimize things some
	HandleItemCollision(c, socket, amTurning);
	
	// i'm going ahead and optimizing this, if you haven't really moved, should be
	// no need to check for teleporters and the weather shouldn't change
	if (!amTurning)
	{
		HandleTeleporters(c, socket, oldx, oldy);
        
		HandleWeatherChanges(c, socket);
	}
	
	// i'm afraid i don't know what this does really, do you need to do it when turning??
	HandleGlowItems(c, socket);
	
	// would have already collided, right??
	if (!amTurning && chars[c].npc == 0)
		Magic->GateCollision(c);
	
	// again, don't know if we need to check when turning or not
	checkregion(c); 
	//if (socket==-1) printf("checkregion called for %s region#: %i region-name:%s \n",chars[c].name,chars[c].region,region[chars[c].region].name);
}


// if we have a valid socket, see if we need to deny the movement request because of
// something to do with the walk sequence being out of sync.
bool cMovement::VerifySequence(CHARACTER c, UOXSOCKET socket, int sequence)
{
    if (socket != -1)
    {
        if ((walksequence[socket] + 1 != sequence) && (sequence != 256))
        {
            deny(socket, c, sequence);  
            return false;
        }
        
    }
    return true;
}

// see if we should allow someone to move based upon weight restrictions
bool cMovement::CheckWeightFirst(CHARACTER c, UOXSOCKET socket)
{
    // ok, they must be an alive, non-npc, and not a GM or Counselor to bother
    // checking weight restrictions
    if (!chars[c].dead && !chars[c].npc && !(chars[c].priv&0x01) && !(chars[c].priv&0x80))
    {
				if (!Weight->CheckWeight(c, socket) || (chars[c].stm<3)) // this has to executes, even if k==-1, thus that many !=-1's
        {
            // lets see, we are going to check all the expensive weight stuff first and then check
            // to see if there is a valid socket? that doesnt make sense, if this check 
            // is reallly all encompassing and so cheap to make why don't we check it
            // very first thing?? something to look at in a rewrite
            if (socket != -1)
            {
                sysmessage(socket, "You are too fatigued to move, you are carrying %d stones.", chars[c].weight);
                // very interesting.. we don't send a deny() here, we nuke the
                // the sequence and teleport them. not sure why yet.
                walksequence[socket]=-1;
                teleport(c);
                return false;
            }
            // else ?? what we don't care anymore? i guess NPCs never get
            // fatigued?  this if either needs to come first or there needs
            // to be an else statement here
        }
    }
    return true;
}

// see if someone is frozen and/or casting a spell
bool cMovement::CheckForFrozen(CHARACTER c, UOXSOCKET socket)
{
    // dont put a if socket!=-1 here !!!  i didn't put that comment here
    // but it makes sense because we don't want NPCs walking while frozen
    
    // check to see if someone is frozen/casting a spell
    if (chars[c].priv2&2) // lord binary !!! 
    {                   
        // again, we dont deny() this for some reason, we teleport and then
        // nuke the sequence
        teleport(c);
		// however, we better check for socket != -1 before this stuff
		if (socket != -1)
		{
			// send the appropiate message to them
			if (chars[c].casting)
				sysmessage(socket, "You cannot move while casting.");
			else
				sysmessage(socket, "You are frozen and cannot move.");
			walksequence[socket] = -1;
		}
        return false; //Morrolan
    } 
    return true;
}

bool cMovement::CheckForRunning(CHARACTER c, UOXSOCKET socket, int dir)
// PARAM WARNING: unreferenced paramater socket
{
	// if we are running
	if (dir&0x80)
	{ //AntiChrist -- if running
		// if we are using stealth
		if (chars[c].stealth!=-1) { //AntiChrist - Stealth - stop hiding if player runs
			chars[c].stealth=-1;
			chars[c].hidden=0;
		}
		//Don't regenerate stamina while running
		chars[c].regen2=uiCurrentTime+(server_data.staminarate*CLOCKS_PER_SEC);
		chars[c].running++;
		// if all these things
		if(!chars[c].dead && !chars[c].onhorse && chars[c].running>(server_data.runningstaminasteps)*2)
		{
			//The *2 it's because i noticed that a step(animation) correspond to 2 walking calls
			chars[c].running=0;
			chars[c].stm--;
			updatestats(c,2);
		}
		if( chars[c].war && chars[c].targ != -1 )
		{
			chars[c].timeout=uiCurrentTime+CLOCKS_PER_SEC*2;
		}

	} else {
		chars[c].running=0;
	}                                           
	return true;
}

bool cMovement::CheckForStealth(CHARACTER c, UOXSOCKET socket)
// PARAM WARNING: unreferenced paramater socket
{
	if ((chars[c].hidden)&&(!(chars[c].priv2&8)))
	{
		if(chars[c].stealth!=-1)
		{ //AntiChrist - Stealth
			chars[c].stealth++;
			if(chars[c].stealth>((server_data.maxstealthsteps*chars[c].skill[STEALTH])/1000))
			{
				chars[c].stealth=-1;
				chars[c].hidden=0;
				updatechar( c );
			}
		}
		else
		{
			chars[c].hidden=0;
			updatechar( c );
		}
	}
	return true;
}

// see if a player has tried to move into a house they were banned from
bool cMovement::CheckForHouseBan(CHARACTER c, UOXSOCKET socket)
{
    if ( !chars[c].npc ) // this code is also called from npcs-walking code, so only check for players to cut down lag!
    {
        // check if player is banned from a house - crackerjack 8/12/99
        int i = findmulti(chars[c].x, chars[c].y, chars[c].z);
        if ( i!=-1 ) 
        {
			setserial(c, i, 8);//Set them inside the multi!

            int j=on_hlist(i, chars[c].ser1, chars[c].ser2, chars[c].ser3, chars[c].ser4, NULL);
            if(j==H_BAN) 
            {
                int sx, sy, ex, ey;
                Map->MultiArea(i,&sx,&sy,&ex,&ey);
                chars[c].x = ex;
                chars[c].y = ey+1;
                teleport(c);
                if (socket!=-1)
				{
					sysmessage(socket, "You are banned from that location.");
					// this wasn't resetting walksequence like the others, probably a bug
					walksequence[socket] = -1;
				}
                return false;
            }
        }
    } 
    return true;
}

void cMovement::MoveCharForDirection(CHARACTER c, int dir)
{
	// switch on which direction to travel
	switch(dir&0x0F)
	{
	case '\x00' : chars[c].y--;
		break;
	case '\x01' : { chars[c].x++; chars[c].y--; }
		break;
	case '\x02' : chars[c].x++;
		break;
	case '\x03' : { chars[c].x++; chars[c].y++; }
		break;
	case '\x04' : chars[c].y++;
		break;
	case '\x05' : { chars[c].x--; chars[c].y++; }
		break;
	case '\x06' : chars[c].x--;
		break;
	case '\x07' : { chars[c].x--; chars[c].y--; }
		break;
	default:
		printf("ERROR: Fallout of switch statement without default. walking.cpp, walking()\n"); //Morrolan
		printf("Caused by character %s. dir: %i chars[c].dir&0x0f: %i\n",chars[c].name,chars[c].dir,chars[c].dir&0x0f);
		// if ( socket > -1 ) deny( socket, s, sequence ); // lb, crashfix
		// actually lets make this crash so we can find out who is calling this func with a bogus dir!
		// putting stuff like this in where it can be ignore just hides bugs for awhile, it will come
		// back and bite you in the end. its better to bite the bullet now and find out why its being
		// called with bogus arguments
		//assert(0);
		// damn, its still getting in here as of 11/3/1999
	}
}


// checkout everything we might need to take into account and fill it into the xyblock array
void cMovement::FillXYBlockStuff(CHARACTER c, unitile_st *xyblock, int &xycount)
{
	// get the new coords we are trying to move to
	const short int x = chars[c].x;
	const short int y = chars[c].y;

	int mapid = 0;
//	int mapz = Map->AverageMapElevation(x, y, mapid);
	signed char mapz = Map->AverageMapElevation(x, y, mapid);
	if (mapz != illegal_z)
	{
		land_st land;
		Map->SeekLand(mapid, &land);
		
		xyblock[xycount].type=0;
		xyblock[xycount].basez = mapz;
		xyblock[xycount].id = mapid;
		xyblock[xycount].flag1=land.flag1;
		xyblock[xycount].flag2=land.flag2;
		xyblock[xycount].flag3=land.flag3;
		xyblock[xycount].flag4=land.flag4;
		xyblock[xycount].height=0;
		xyblock[xycount].weight=255;
		xycount++;
	}

	/*
	// first thing is to get the map where we are standing
	map_st map1 = Map->SeekMap0( x, y );
	// if this appears to be a valid land id, <= 2 is invalid
	if (map1.id > 2)
	{
		// get three other nearby maps to decide on an average z?
		map_st map2 = Map->SeekMap0( x + 1, y );
		map_st map3 = Map->SeekMap0( x, y + 1);
		map_st map4 = Map->SeekMap0( x + 1, y + 1);
		
		int testz = 0;
		if (abs(map1.z - map4.z) <= abs(map2.z - map3.z))
		{
			testz = map1.z + map4.z;
		} else {
			testz = map2.z + map3.z;
		}
		
		land_st land;
		Map->SeekLand(map1.id, &land);
		
		xyblock[xycount].type=0;
		xyblock[xycount].basez=testz/2;
		if (testz%2<0) xyblock[xycount].basez--;
		// ^^^ Fix to make it round DOWN, not just in the direction of zero
		// If your compiler does this already (though I dont think any compiler will), comment this out
		xyblock[xycount].id=map1.id;
		xyblock[xycount].flag1=land.flag1;
		xyblock[xycount].flag2=land.flag2;
		xyblock[xycount].flag3=land.flag3;
		xyblock[xycount].flag4=land.flag4;
		xyblock[xycount].height=0;
		xyblock[xycount].weight=255;
		xycount++;
	}
	else
	{
		printf("Uhh.. Someone's walking on something funny..\n");
	}
	*/
	// - Tauriel's region stuff 3/6/99
	const int StartGrid=mapRegions->StartGrid(chars[c].x,chars[c].y);
	const int getcell=mapRegions->GetCell(chars[c].x,chars[c].y);
	
	// - This needs to use a pointer/array to cut down on walking lag...
	unsigned int checkgrid = 0;
	for (int increment = 0; increment < 3; increment++)
	{
		checkgrid = StartGrid + (increment * mapRegions->GetColSize());
		for (int i = 0; i < 3; i++)
		{
			int mapitem = -1;
			int mapitemptr = -1;
			do //check all items in this cell
			{
				mapitemptr = mapRegions->GetNextItem(checkgrid+i, mapitemptr);
				if (mapitemptr == -1) break;
				mapitem = mapRegions->GetItem(checkgrid+i, mapitemptr);
				if (mapitem != -1 && mapitem<1000000)
				{
					if (items[mapitem].id1<0x40)
					{
						if ((items[mapitem].x == x) && (items[mapitem].y == y))
						{
							tile_st tile;
							Map->SeekTile((items[mapitem].id1<<8)+items[mapitem].id2, &tile);
							xyblock[xycount].type=1;
							xyblock[xycount].basez=items[mapitem].z;
							xyblock[xycount].id=(items[mapitem].id1<<8)+items[mapitem].id2;
							xyblock[xycount].flag1=tile.flag1;
							xyblock[xycount].flag2=tile.flag2;
							xyblock[xycount].flag3=tile.flag3;
							xyblock[xycount].flag4=tile.flag4;
							xyblock[xycount].height=tile.height;
							xyblock[xycount].weight=tile.weight;
							xycount++;
						}
					}
					else if (
							(abs(items[mapitem].x- x)<=BUILDRANGE)&&
							(abs(items[mapitem].y- y)<=BUILDRANGE)
							)
					{
						UOXFile *mfile = NULL;
						SI32 length = 0;		// should be SI32, not long
						Map->SeekMulti(((items[mapitem].id1<<8)+items[mapitem].id2)-0x4000, &mfile, &length);
						length=length/MultiRecordSize;
						if (length == -1 || length>=17000000)//Too big... bug fix hopefully (Abaddon 13 Sept 1999)
							//              if (length == -1)
						{
							printf("walking() - Bad length in multi file. Avoiding stall.\n");
							length = 0;
						}
						for (int j=0;j<length;j++)
						{
							st_multi multi;
							mfile->get_st_multi(&multi);
							if (multi.visible && (items[mapitem].x+multi.x == x) && (items[mapitem].y+multi.y == y))
							{
								tile_st tile;
								Map->SeekTile(multi.tile, &tile);
								xyblock[xycount].type=2;
								xyblock[xycount].basez = multi.z + items[mapitem].z;
								xyblock[xycount].id=multi.tile;
								xyblock[xycount].flag1=tile.flag1;
								xyblock[xycount].flag2=tile.flag2;
								xyblock[xycount].flag3=tile.flag3;
								xyblock[xycount].flag4=tile.flag4;
								xyblock[xycount].height=tile.height;
								xyblock[xycount].weight=255;
								xycount++;
							}
						}
					}
				}
			} while (mapitem!=-1);
		} //- end of itemcount for loop
	}

	// now it seems like extra work if we know the player is a real player
	// the client will keep them from walking over anything they shouldn't
	// if we moved this out into the NPC walking code, i would think that
	// would help speed things up a bit
	MapStaticIterator msi(x, y);
	staticrecord *stat;
 	while (stat = msi.Next())
	{
		//printf("staticr[X] type=%d, id=%d\n", 2, stat->itemid);
		tile_st tile;
		msi.GetTile(&tile);
		xyblock[xycount].type=2;
		xyblock[xycount].basez=stat->zoff;
		xyblock[xycount].id=stat->itemid;
		xyblock[xycount].flag1=tile.flag1;
		xyblock[xycount].flag2=tile.flag2;
		xyblock[xycount].flag3=tile.flag3;
		xyblock[xycount].flag4=tile.flag4;
		xyblock[xycount].height=tile.height;
		xyblock[xycount].weight=255;
		xycount++;
	}
/*	// now loop over static items and add those	
	const int x1 = x / 8, x2 = x % 8;
	const int y1 = y / 8, y2 = y % 8;
	int len = Map->StaticInit( x, y );
#ifdef DEBUG_WALKING
			printf("len: %d, baseX: %d, baseY: %d, remX: %d, remY: %d\n", len, x1, y1, x2, y2); 
			printf(" 01234567\n");
			char testmap[9][9];
			memset(testmap, ' ', 9*9);
#endif
	for(int j=0; j < len; j++ )
	{
		staticrecord stat = Map->SeekStatic( x, y );
#ifdef DEBUG_WALKING
		testmap[stat.yoff][stat.xoff] = 'X';
#endif
		if ((stat.xoff==x2)&&(stat.yoff==y2))
		{
			tile_st tile;
			Map->SeekTile(stat.itemid, &tile);
			xyblock[xycount].type=2;
			xyblock[xycount].basez=stat.zoff;
			xyblock[xycount].id=stat.itemid;
			xyblock[xycount].flag1=tile.flag1;
			xyblock[xycount].flag2=tile.flag2;
			xyblock[xycount].flag3=tile.flag3;
			xyblock[xycount].flag4=tile.flag4;
			xyblock[xycount].height=tile.height;
			xyblock[xycount].weight=255;
			xycount++;
		}
	}
#ifdef DEBUG_WALKING
			testmap[y2][x2] = 'O';
			for (int foo = 0; foo < 8; ++foo)
			{
				testmap[foo][8] = '\0';
				printf("%d%s\n", foo, testmap[foo]);
			}
#endif
*/
}
//////////////////////////////////////////////////////////////////////////////
// FUNCTION:    cMovement::CheckWalkable()
// AUTHOR:      Dupois (dupois@home.com)
// DATE:        January 17, 2000
//
// PURPOSE:     Prevent NPC's and PC's from walking over items they are not 
//              supposed to.  For PC's, the client blocks movement without
//              ever sending a walk request to the server (cut down on lag I
//              would guess).
// 
// PARAMETERS:  c         Character's index in chars[]
//              *xyblock  Pointer to a xyblock[] populated by FillXYBlockStuff
//              xycount   Size of xyblock[] array
//
// RETURNS:     signed char Characters new 'z' position or illegal_z if char is blocked
//////////////////////////////////////////////////////////////////////////////
//int cMovement::CheckWalkable( CHARACTER c, unitile_st *xyblock, int xycount )
signed char cMovement::CheckWalkable( CHARACTER c, unitile_st *xyblock, int xycount )
{
  bool  bIsGM             = IsGMBody( c );
  bool  bWalkableItem     = true;
  bool  bClimbing		  = false;
  bool	bSpecialBlockingFound = false;
  int   nWalkableItemCount= 0;
  signed char nNewZ       = illegal_z;
  signed char nItemTop    = illegal_z;
  signed char LastBlockingZ = illegal_z;
  signed char nDelta            = 0;
  int nTotalZHeight	  = 0;

  // Set the base value for nNewZ to the chars old z just in case
//  nNewZ = chars[c].z;

  // Work our way through the blockable item array
  for( int i = 0; i < xycount; i++ )
  {
    // do all of the checking here in here based on the type of object

	  // Calculate the items total height
	  nItemTop = xyblock[i].basez + xyblock[i].height;
    
    // There are different types of statics, MAP(0) and ITEMS(1&2) so handle them differently
    switch( xyblock[i].type )
    {
    case 0: // MAP (STATIC)
	case 1:
	case 2:
      {
#if DEBUG_WALK_BLOCK
        printf("MAP WALKING DEBUG:  type: map-tile char[%i] name=%s, Z=%i\n", c, chars[c].name, chars[c].z);
	      printf("tilenum: %i\n",xyblock[i].id);
        printf("tile Z: %i\n",xyblock[i].basez);
	      printf("Flag1:%x\n", xyblock[i].flag1);
	      printf("Flag2:%x\n", xyblock[i].flag2);
	      printf("Flag3:%x\n", xyblock[i].flag3);
	      printf("Flag4:%x\n", xyblock[i].flag4);
	      printf("Height:%lx\n", xyblock[i].height);
	      printf("Weight:%x\n", xyblock[i].weight);
#endif
		// If the top of the map object is below the max a character can climb up to
		  // calculate the new Z as follows (if it is above the max it is too high above character)
		  if( nItemTop < chars[c].z + MaxZstep )
		{
			// Check if the map object a blockable object before assigning the NewZ value
			// Stops characters from walking on top of mountains and other such things
			//if( bIsGM )
			//{
			//	printf("I am a GM\n" );
			//}
			if( !( xyblock[i].flag1&0x40 ) || (bIsGM) )
			{
				if( nItemTop > nNewZ )
				{
					nNewZ = nItemTop;
				}
			}
			else
			{
				if( nItemTop > LastBlockingZ )
				{
					LastBlockingZ = nItemTop;
					if( LastBlockingZ > chars[c].z )
					{
						// Used all these ifs because switch doesn't support a range (and all these if are easier to modify and read)

						// Display Case Tops
						if( xyblock[i].id > 0x0A9E && xyblock[i].id < 0x0AA6 )
						{
							bSpecialBlockingFound = true;
						}
						// Display Case Tops And Bottoms
						if( xyblock[i].id > 0x0AFC && xyblock[i].id < 0x0B19 )
						{
							bSpecialBlockingFound = true;
						}
						// Tables And Counters And Water Troughs
						if( xyblock[i].id > 0x0B33 && xyblock[i].id < 0x0B45 )
						{
							bSpecialBlockingFound = true;
						}
						// Writing Tables
						if( xyblock[i].id > 0x0B48 && xyblock[i].id < 0x0B4D )
						{
							bSpecialBlockingFound = true;
						}
						// Tables
						if( xyblock[i].id > 0x0B6A && xyblock[i].id < 0x0B91 )
						{
							bSpecialBlockingFound = true;
						}
						// Bamboo And Log Tables
						if( xyblock[i].id > 0x11D5 && xyblock[i].id < 0x11E2 )
						{
							bSpecialBlockingFound = true;
						}
						// Stone Tables
						if( xyblock[i].id > 0x1200 && xyblock[i].id < 0x1207 )
						{
							bSpecialBlockingFound = true;
						}
						// Tables With Cloth Draped Over
						if( xyblock[i].id > 0x1666 && xyblock[i].id < 0x166D )
						{
							bSpecialBlockingFound = true;
						}
						// Bar Tops
						if (xyblock[i].id>0x1910 && xyblock[i].id<0x1914)
						{
							bSpecialBlockingFound = true;
						}

						// Bars
						if (xyblock[i].id>0x1917 && xyblock[i].id<0x1920)
						{
							bSpecialBlockingFound = true;
						}

						// Woodworker's Bench And Cooper's Bench
						if (xyblock[i].id>0x19F0 && xyblock[i].id<0x19FD)
						{
							bSpecialBlockingFound = true;
						}

						// Tables
						if (xyblock[i].id>0x1DA4 && xyblock[i].id<0x1DAD)
						{
							bSpecialBlockingFound = true;
						}

						// Sandstone And Marble Tables
						if (xyblock[i].id>0x1DBA && xyblock[i].id<0x1DC7)
						{
							bSpecialBlockingFound = true;
						}
					}
				}
			}
			#if DEBUG_WALKING
				printf( "nNewZ=%d LastBlockingZ=%d\n", nNewZ, LastBlockingZ );
			#endif
		}

        // Check if the object is a blockable object
        /*if( xyblock[i].flag1 & 0x40 )
		{
          // Check if the map tile is at the same height or greater as the character trying to walk on it,
          // if it isn't then it has no direct effect on the character
          if( ( chars[c].z >= xyblock[i].basez ) && ( !bIsGM ) )
          {
#if DEBUG_WALK_BLOCK
            printf(" MAP BLOCKING MOVEMENT (LEVEL WALK) - chars[%i].name=%s Z=%i!\n", c, chars[c].name, chars[c].z );
#endif
            nNewZ = illegal_z;
          }
        }*/
      }
      break;

    /*case 1: // ITEM (DYNAMIC)
    case 2: // ITEM (STATIC)
      {
#if DEBUG_WALK_BLOCK
        printf( "ITEM #%02i - ID: %i",i,  xyblock[i].id );
        if( xyblock[i].id == 1 ) printf( " (DYNAMIC)\n" );
        else printf( " (STATIC)\n" );
        printf( "           basez: %i\n", xyblock[i].basez );
        printf( "           height: %i\n", xyblock[i].height );
#endif
        // Count how many items are in the tile we want to move to
		// Why, I don't know, seemed like a good idea at the time :)
        nWalkableItemCount++;

        // Check whether or not we already found a blockable dynamic item above the chars
        // current Z and that it is within the MAX_Z_ITEM_INFLUENCE range (if we did then
		// don't bother with the rest since we are already blocked by it
        if( bWalkableItem == true )
        { 
          // Determine if this object is above or below the character
          if( chars[c].z + MaxZstep < nItemTop )
          {
				// Only do this check if its a NPC since the client blocks a Players walk automatically
				// BULLSHIT!! People could use vERY and then walk right through.  This needs to be checked
				// server side.
				if( ( ( nTotalZHeight - chars[c].z ) > MAX_ITEM_CLIMB_HEIGHT )/* && ( chars[c].npc > 0 ) )
				{
					// The total height of all items is too high for an NPC to climb so stop checking
					/*nNewZ = illegal_z;
					bWalkableItem = false;
					continue;
				}

				// The current item is either level with or climbable so set the flag
				// to ignore any items below this point
					bClimbing = true;

				// Item could be climbed so get NewZ, then fall through and check
				// to see if its a blockable object or not.
				// First make sure the item isn't below what we've already climbed on top of.
				if( ( nItemTop > nNewZ ) )
				  nNewZ = nItemTop;
//            }
          }
          else
          {
			  // If we have already started climbing, then ignore items below us
			  if( bClimbing ) continue;
            // CHAR is above the top of the item
            // Determine how far we need to fall
            nDelta = abs(chars[c].z) - abs(nItemTop);

            // Check if item is within Z influence for falling down
			// Note used by may be usefull to stop NPC's from falling too far (eh, let em fall if they want :)
            //if( nDelta > MAX_ITEM_Z_DELTA )
//              continue;

            // We are falling down so figure out how far to fall (take into account that we could have
			// already calculated a different fall height.  Make sure we fall the smallest amount
			// out of all the items below us.
			if( nNewZ < ( chars[c].z - nDelta ) )
				nNewZ = chars[c].z - nDelta;

			// Regardless of if the item is blockable or not.
			// It is below the character and we should fall on top of it
			continue;
          }

          // If we get here we must have passed all of the height checks so all we need to do is
          // check whether this item should block walking
          if( ( xyblock[i].flag1 & 0x40 ) && ( !bIsGM ) )
          {
#if DEBUG_WALK_BLOCK
            // This tile is a blocking tile at the same height as the character so prevent movement
            printf( "ITEM BLOCKING MOVEMENT - chars[%i].name=%s Z=%i\n", c, chars[c].name, chars[c].z );
	          printf( "             ITEM INFO - ID: %i\n",xyblock[i].id);
	          printf( "             ITEM INFO - Flag1:%x\n", xyblock[i].flag1);
	          printf( "             ITEM INFO - Flag2:%x\n", xyblock[i].flag2);
	          printf( "             ITEM INFO - Flag3:%x\n", xyblock[i].flag3);
	          printf( "             ITEM INFO - Flag4:%x\n", xyblock[i].flag4);
            printf( "             ITEM INFO - basez=%i\n", xyblock[i].basez);
            printf( "             ITEM INFO - height=%i\n", xyblock[i].height);
#endif
            bWalkableItem = false;
            nNewZ = illegal_z;
          }
        }
      }
      break;
*/
    default:
      // This shouldn't happen, but just in case we should log it and assume
      // that the check passed so we don't effect anything else
      //printf("UOX3: cMovement::CheckInvalidStaticList INVALID xyblock[%d].type = %d\n", i, xyblock[i].type );
      break;
    }
  }
	if( nNewZ < LastBlockingZ || bSpecialBlockingFound )
	{
		nNewZ = illegal_z;
	}
#if DEBUG_WALKING
	printf( "Z from CheckWalkable=%d\n", nNewZ );
#endif
  return nNewZ;
}


// now if there are any z coordinate bugs, they are probably in this crazy ass function
// here's what i've figured out so far.  we make an array of everything at this square and what
// its z value is, we then loop thru this array to determine how we relate with the items to
// decide what our new z value(s) should be.
//
// after digging though this and mapstuff.cpp a little more, i'm starting to think that the bulk
// of this function really belongs in there.. the trigger stuff should be pulled out of here
// and into HandleItemCollision()


bool cMovement::CrazyXYBlockStuff(CHARACTER c, UOXSOCKET socket, short int oldx, short int oldy, int sequence)
{
#if DEBUG_WALKING
	printf("Enter Crazy(CHAR =%d, SOCKET = %d, oldx = %d, oldy = %d, oldz = %d, x = %d, y = %d)\n",
			c, socket, oldx, oldy, (int) chars[c].z, chars[c].x, chars[c].y);
#endif
	// ok, i've localized these variables
	int xycount=0;
	unitile_st xyblock[XYMAX];

	// get the new coords we are trying to move to
	//const int x = chars[c].x;
	//const int y = chars[c].y;
	FillXYBlockStuff(c, xyblock, xycount);
	
	/*
	** We have filled the xyblock[] with everything pertinant, now its time to loop
	** back over and decide which to use
	*/
#if DEBUG_WALKING
	for (int i = 0; i < xycount; ++i)
	{
	  printf("xyblock[%d] type=%d, id=%d, basez=%d, height=%d\n",
		 i, xyblock[i].type, xyblock[i].id, xyblock[i].basez, xyblock[i].height);
	  if (i != 0)
	    {
	      printf("Flags at num:%d, flag1: %x, flag2: %x, flag3: %x, flag4: %x\n", i,
		 (int) xyblock[i].flag1, (int) xyblock[i].flag2, (int)xyblock[i].flag3, (int)xyblock[i].flag4);
	    }
	}
#endif

	// ok, there is a member z, and there was a local z, which was this supposed to use?
	// this should most definately be handled as a local in the future, but i'm leaving it
	// as the member for now..
	
	z =  illegal_z;
//	const int oldz = chars[c].z;
	const signed char oldz = chars[c].z;
	const bool gmbody = IsGMBody(c);

	// This is to stop NPC's from walking on improper objects (like display cases, etc)
	// that are not caught by anything else.
	z = CheckWalkable( c, xyblock, xycount );
	if( z != illegal_z )
	{
		dispz = z;
	}
	else
	{
		chars[c].x=oldx;
		chars[c].y=oldy;
		if (socket!=-1) deny(socket, c, sequence);
		return false;
	}
	return true;
}

// so we are going to move, lets update the regions
// FYI, Items equal to or greater than 1000000 are considered characters...
void cMovement::HandleRegionStuffAfterMove(CHARACTER c, short int oldx, short int oldy)
{
	// save where we were moving to
	const short int nowx = chars[c].x;
	const short int nowy = chars[c].y;

	// i'm trying a new optimization here, if we end up in the same map cell
	// as we started, i'm sure there's no real reason to remove and readd back
	// to the same spot..
	if (mapRegions->GetCell(oldx, oldy) != mapRegions->GetCell(nowx, nowy))
	{
		// restore our original location and remove ourself
		chars[c].x = oldx;
		chars[c].y = oldy;
		mapRegions->RemoveItem(c+1000000);
		// we have to remove it with OLD x,y ... LB, very important, and he is right!

		// restore the new location and add ourselves
		chars[c].x = nowx;
		chars[c].y = nowy;
	
		mapRegions->AddItem(c+1000000);

		// i suspect the new weather code probably needs something here, so if
		// you walk from one region to another it can adjust the weather, but i
		// could be wrong
	}
#if DEBUG_WALKING
	else
	{
		//printf("Guess what? I didn't change regions.\n");
	}
#endif

	// i'm moving this to the end because the regions shouldn't care what the z was
	chars[c].dispz = dispz;
	chars[c].z = z;
}


// actually send the walk command back to the player and increment the sequence
void cMovement::SendWalkToPlayer(CHARACTER c, UOXSOCKET socket, short int sequence)
{
	if (socket!=-1)
	{
		char walkok[4]="\x22\x00\x01";
		walkok[1]=buffer[socket][2];
		walkok[2]=0x41;
		if (chars[c].hidden)
			walkok[2]=0x00;
		Network->xSend(socket, walkok, 3, 0);

		walksequence[socket] = sequence;
		if (walksequence[socket] == 255)
			walksequence[socket] = 0;
	}
}

// send out our movement to all other players who can see us move
void cMovement::SendWalkToOtherPlayers(CHARACTER c, int dir, short int oldx, short int oldy)
{
	// lets cache these vars in advance
	const int visibleRange = Races->getVisRange( chars[c].race );
	const int newx=chars[c].x;
	const int newy=chars[c].y;

	for (int i=0;i<now;i++)
	{
		// lets see, its much cheaper to call perm[i] first so i'm reordering this
		if ((perm[i]) && (inrange1p(c, currchar[i])))
		{
			if (
				(((abs(newx-chars[currchar[i]].x)==visibleRange )||(abs(newy-chars[currchar[i]].y)== visibleRange )) &&
				((abs(oldx-chars[currchar[i]].x)>visibleRange )||(abs(oldy-chars[currchar[i]].y)>visibleRange ))) ||
				((abs(newx-chars[currchar[i]].x)==visibleRange )&&(abs(newy-chars[currchar[i]].y)==visibleRange ))
				)
			{
				impowncreate(i, c, 1);
			}
			else
				//    if ((abs(newx-chars[currchar[i]].x)<VISRANGE)||(abs(newy-chars[currchar[i]].y)<VISRANGE))
			{
				extmove[1]=chars[c].ser1;
				extmove[2]=chars[c].ser2;
				extmove[3]=chars[c].ser3;
				extmove[4]=chars[c].ser4;
				extmove[5]=chars[c].id1;
				extmove[6]=chars[c].id2;
				extmove[7]=chars[c].x>>8;
				extmove[8]=chars[c].x%256;
				extmove[9]=chars[c].y>>8;
				extmove[10]=chars[c].y%256;
				extmove[11]=chars[c].dispz;
				extmove[12]=dir;
				//     extmove[12]=chars[currchar[c]].dir&0x7F;
				//     extmove[12]=buffer[c][1];
				extmove[13]=chars[c].skin1; //ripper, skin problems bugfix
				extmove[14]=chars[c].skin2;
				if( chars[c].npc && chars[c].runs && chars[c].war ) // Ripper 10-2-99 makes npcs run in war mode or follow :) (Ab mod, scriptable)
					extmove[12]=dir|0x80;
				if (chars[c].war) extmove[15]=0x40; else extmove[15]=0x00;
				if (chars[c].hidden) extmove[15]=extmove[15]|0x80;
				if( chars[c].dead && !chars[c].war ) extmove[15] = extmove[15]|0x80; // Ripper
				if(chars[c].poisoned) extmove[15]=extmove[15]|0x04; //AntiChrist -- thnx to SpaceDog
				//if (chars[c].npcaitype==0x02) extmove[16]=6; else extmove[16]=1;
				int guild, race;
				//chars[i].flag=0x04;       // everyone should be blue on default
				guild=Guilds->Compare( c, currchar[i] );
				race=Races->Compare( c, currchar[i] );
				if( chars[c].kills > repsys.maxkills ) extmove[16]=6;
				else if (guild==1 || race==2)//Same guild (Green)
					extmove[16]=2;
				else if (guild==2 || race==1) // Enemy guild.. set to orange
					extmove[16]=5;
				//                    else if ( !chars[i].npc && ( chars[i].priv&1 || chars[i].priv&80 ) )
				//                            extmove[16] = 7;
				else
				{
					switch(chars[c].flag)
					{//1=blue 2=green 5=orange 6=Red 7=Transparent(Like skin 66 77a)
					case 0x01: extmove[16]=6; break;// If a bad, show as red. 
					case 0x04: extmove[16]=1; break;// If a good, show as blue.
					case 0x08: extmove[16]=2; break; //green (guilds)
					case 0x10: extmove[16]=5; break;//orange (guilds)
					default:extmove[16]=3; break;//grey 
					}
				}
				if (currchar[i] != c ) // fix from homey (NPCs will display right)
					Network->xSend(i, extmove, 17, 0);
			}
		}
	}
}

// see if we should mention that we shove something out of the way
void cMovement::OutputShoveMessage(CHARACTER c, UOXSOCKET socket, short int oldx, short int oldy)
{
	if (socket!=-1)
	{
		// lets cache these vars in advance
		const int visibleRange = Races->getVisRange( chars[c].race );

		const int newx=chars[c].x;
		const int newy=chars[c].y;

		const int StartGrid=mapRegions->StartGrid(chars[c].x,chars[c].y);
		const int getcell=mapRegions->GetCell(chars[c].x,chars[c].y);
		
		int checkgrid = 0;
		for (int increment = 0; increment < 3; increment++)
		{
			checkgrid = StartGrid+(increment*mapRegions->GetColSize());
			for (int a = 0; a < 3; a++)
			{
				int mapitemptr = -1;
				int mapitem = -1;
				int mapchar = -1;
				do //check all items in this cell
				{
					mapchar = mapitem = -1;
					mapitemptr=mapRegions->GetNextItem(checkgrid+a, mapitemptr);
					if (mapitemptr==-1) break;
					mapitem=mapRegions->GetItem(checkgrid+a, mapitemptr);
					if (mapitem>999999)
						mapchar = mapitem-1000000;

					if (mapitem!=-1 && mapitem>=1000000)
					{
						int i=mapchar;
						//printf("DEBUG: Mapchar %i [%i]\n",mapchar,mapitem);
						//Let GMs see logged out players
						if ((online(i)||chars[i].npc)||(chars[c].priv&01) && server_data.showloggedoutpcs )
						{
							if (
								(((abs(newx-chars[i].x)== visibleRange )||(abs(newy-chars[i].y)== visibleRange )) &&
								((abs(oldx-chars[i].x) > visibleRange )||(abs(oldy-chars[i].y)> visibleRange ))) ||
								((abs(newx-chars[i].x)== visibleRange )&&(abs(newy-chars[i].y)== visibleRange ))
								)
							{
								impowncreate(socket, i, 1);
							}
						}
						if (!(
							((chars[c].id1==0x03)&&(chars[c].id2==0xDB)) ||
							((chars[c].id1==0x01)&&(chars[c].id2==0x92)) ||
							((chars[c].id1==0x01)&&(chars[c].id2==0x93)) ||
							(chars[c].priv&0x01) ||
							(chars[c].priv&0x80)
							))
						{
							if (i!=c && (online(i) || chars[i].npc))
							{
								if (chars[i].x==chars[c].x && chars[i].y==chars[c].y && chars[i].z==chars[c].z)
								{
									if (!chars[i].hidden && (!(chars[i].priv&0x10)))
									{
										sprintf(temp, "Being perfectly rested, you shove %s out of the way.", chars[i].name);
										if (socket!=-1) sysmessage(socket, temp);
										chars[c].stm=max(chars[c].stm-4, 0);
										// updatestats(currchar[c], 2); // replaced with:
										updatestats(c, 2);  // arm code
									}
									else if(!(chars[i].priv&1)&&!(chars[i].priv&80))//A normal player (No priv1(Not a gm))
									{
										if (socket!=-1) sysmessage(socket, "Being perfectly rested, you shove something invisible out of the way.");
										chars[c].stm=max(chars[c].stm-4, 0);
										// updatestats(currchar[s], 2); // replaced with:
										updatestats(c, 2);  // arm code
									}
								}
							}
						}
					}
				} while (mapitem!=-1);
			}
		}
	}
}

// handle item collisions, make items that appear on the edge of our sight because
// visible, buildings when they get in range, and if the character steps on something
// that might cause damage
void cMovement::HandleItemCollision(CHARACTER c, UOXSOCKET socket, bool amTurning)
{
	// apparently we don't want NPCs to be affected by any of this stuff,
	// i'm not sure i agree with that yet
	if (socket == -1)
		return;

	// lets cache these vars in advance
	const int visibleRange = Races->getVisRange( chars[c].race );
	const int newx = chars[c].x;
	const int newy = chars[c].y;

	
	// - Tauriel's region stuff 3/6/99
	const int StartGrid = mapRegions->StartGrid(newx, newy);
	const int getcell = mapRegions->GetCell(newx, newy);

	int checkgrid = 0;
	for (int increment = 0; increment < 3; increment++)
	{
		checkgrid = StartGrid + (increment * mapRegions->GetColSize());
		for (int i=0;i<3;i++)
		{
			int mapitemptr=-1;
			int mapitem=-1;
			do //check all items in this cell
			{
				mapitemptr=mapRegions->GetNextItem(checkgrid+i, mapitemptr);
				if (mapitemptr==-1) break;
				mapitem=mapRegions->GetItem(checkgrid+i, mapitemptr);
				if (mapitem!=-1 && mapitem<1000000)
				{
#if DEBUG_WALKING
				  //printf("Checking against Item %s, ID1:%d, ID2:%d\n", items[mapitem].name, items[mapitem].id1, items[mapitem].id2);
#endif
					// split out the x,y,z check so we can use else ifs for faster item id checking
					if ((items[mapitem].id1==0x39 && (items[mapitem].id2==0x96 || items[mapitem].id2==0x8C)))
					{//Fire Field
						if ((items[mapitem].x == newx) && (items[mapitem].y == newy) && (items[mapitem].z==chars[c].z))
						{
							if (!Magic->CheckResist(-1, c, 4))
							{                                               
								Magic->MagicDamage(c, items[mapitem].morex/300);
							}
							soundeffect2(c, 2, 8);
						}
					}
					
					if ((items[mapitem].id1==0x39 && (items[mapitem].id2==0x15 || items[mapitem].id2==0x20)))
					{//Poison field
						if ((items[mapitem].x == newx) && (items[mapitem].y == newy) && (items[mapitem].z==chars[c].z))
						{
							if (!Magic->CheckResist(-1, c, 5))
							{                                               
								Magic->PoisonDamage(c,1);
							}
							soundeffect2(c, 2, 8);
						}
					}
					
					else if ((items[mapitem].id1==0x39 && (items[mapitem].id2==0x79 || items[mapitem].id2==0x67)))
					{//Para Field
						if ((items[mapitem].x == newx) && (items[mapitem].y == newy) && (items[mapitem].z==chars[c].z))
						{
							if (!Magic->CheckResist(-1, c, 6))
							{
								tempeffect(c, c, 1, 0, 0, 0);
							}
							soundeffect2(c, 0x02, 0x04);
						}
					}
					else if (items[mapitem].id1<0x40)
					{
						// look for item triggers, this was moved from CrazyXYBlockStuff()
						if ((items[mapitem].x== newx) && (items[mapitem].y == newy))
						{
							if (items[mapitem].trigger!=0)
							{
								if ((items[mapitem].trigtype==1)&&(!chars[c].dead))
								{
									if (!items[mapitem].disabled)
									{
										triggerwitem(socket,mapitem,1);  //When player steps on a trigger
									}
								}
							}
						}
					}

					// always check for new items if we actually moved
					if (!amTurning)
					{
						// is the item a building on the BUILDRANGE?
						if ((items[mapitem].id1==0x40)&&(items[mapitem].id2>=0x7C)&&(items[mapitem].id2<=0x7E))
						{
							if ((abs(newx-items[mapitem].x)==BUILDRANGE)||(abs(newy-items[mapitem].y)==BUILDRANGE))
							{
								senditem(socket, mapitem);
							}
						}
						// otherwise if the item has just now become visible, inform the client about it
						else
						{
							if ((abs(newx-items[mapitem].x) == visibleRange ) || (abs(newy-items[mapitem].y) == visibleRange ))
							{
								if( ( !items[mapitem].visible ) || ( ( items[mapitem].visible ) && ( chars[currchar[socket]].priv&0x01 ) ) )// we're a GM, or not hidden
									senditem(socket, mapitem);
							}
						}
					}
				}
			} while (mapitem!=-1);
		}
	}
}

void cMovement::HandleTeleporters(CHARACTER c, UOXSOCKET socket, short int oldx, short int oldy)
// PARAM WARNING: unreferenced paramater socket
{
	// now this is one wacky optimization. if we haven't moved don't do this
	// well, we wouldn't be in Walking() if we weren't trying to move!
	if ((chars[c].x!=oldx)||(chars[c].y!=oldy))
	{
		//    /*if (!(chars[c].dead))*/ objTeleporters(c); //morrolan
		if ( chars[c].npc==0)
			objTeleporters( c );   // ripper
		teleporters(c);
	}
}


/********* start of LB's no rain & snow in buildings stuff ***********/
void cMovement::HandleWeatherChanges(CHARACTER c, UOXSOCKET socket)
{
	if (!chars[c].npc && online(c)) // check for being in buildings (for weather) only for PC's
	{
		// ok, this is already a bug, because the new weather stuff doesn't use this global
		// for the weather.
		if (wtype!=0) // check only neccasairy if it rains or snows ...
		{
			int inDungeon = indungeon(c); // dung-check
			bool i = Map->IsUnderRoof(chars[c].x,chars[c].y,chars[c].z); // static check
			// dynamics-check
			int x = Map->DynamicElevation(chars[c].x,chars[c].y,chars[c].z);
			if (x!=illegal_z)
				if (Boats->GetBoat(socket)!=-1)
					x=illegal_z; // check for dynamic buildings except boats
			if (x==1)
				x=  illegal_z; // 1 seems to be the multi-borders
			
			//printf("x: %i\n",x);
			// ah hah! this was a bug waiting to happen if not already, we have overloaded the use of the
			// variable k, which used to hold the socket
			int k = noweather[c];    
			if (inDungeon || i || x!= illegal_z )
				noweather[c] = 1;
			else
				noweather[c] = 0; // no rain & snow in static buildings+dungeons;
			if (k - noweather[c] !=0)
				weather(socket, 0); // iff outside-inside changes resend weather ...
		}
	}
}

void cMovement::HandleGlowItems(CHARACTER c, UOXSOCKET socket)
// PARAM WARNING: unreferenced paramater socket
{
	// i guess things only glow if you are online, i dunno what that means        
	if( online( c ))
	{
		const int serial = chars[c].serial;
		const int serhash = serial%HASHMAX;
		for( int ci = 0; ci < glowsp[serhash].max; ci++ )
		{
			const int i = glowsp[serhash].pointer[ci];
			if( i != -1 )
			{
				if( items[i].free == 0 )
				{
					Items->GlowItem( c, i );
				}
			}
		}
	}
}

// return whether someone is a GM Body
bool cMovement::IsGMBody(CHARACTER c)
{
	if (
        ((chars[c].id1==0x03)&&(chars[c].id2==0xDB)) ||//Gm
        ((chars[c].id1==0x01)&&(chars[c].id2==0x92)) ||//Ghosts
        ((chars[c].id1==0x01)&&(chars[c].id2==0x93))
        ) 
        return true;
    return false;
}

//XXX

char cMovement::UniBlock(CHARACTER c, unitile_st xyblock)
{
    char blocked=0;	//We'll start out not blocked, ohh will that ever change ;-)
	const bool gmbody = IsGMBody(c);
    const signed char oldz = chars[c].z;

    // printf("xyBaseZ: %i, xyHeight: %i, xyType: %i\n", xyblock[num].basez, xyblock[num].height, xyblock[num].type);
    // printf("xyid: %i, oldz: %i, cpriv: %i cpriv2: %i\n", xyblock[num].id, oldz, chars[c].priv, chars[c].priv2);
    // printf("xyflag1: %i, xyflag2: %i, xyflag3: %i, xyflag4: %i\n", xyblock[num].flag1,xyblock[num].flag2,xyblock[num].flag3,xyblock[num].flag4);
    // printf("gmbody: %i, xyweight: %i\n",gmbody,xyblock[num].weight);
    
    /*if((chars[c].account!=-1))
      {
      //        printf("flag1, b1:%i b2:%i b3:%i b4:%i b5:%i b6:%i b7:%i b8:%i\n ",xyblock[num].flag1&0x1,xyblock[num].flag1&0x2,xyblock[num].flag1&0x4,xyblock[num].flag1&0x8,xyblock[num].flag1&0x10,xyblock[num].flag1&0x20,xyblock[num].flag1&0x40,xyblock[num].flag1&0x80);
      //        printf("flag2, b1:%i b2:%i b3:%i b4:%i b5:%i b6:%i b7:%i b8:%i\n ",xyblock[num].flag2&0x1,xyblock[num].flag2&0x2,xyblock[num].flag2&0x4,xyblock[num].flag2&0x8,xyblock[num].flag2&0x10,xyblock[num].flag2&0x20,xyblock[num].flag2&0x40,xyblock[num].flag2&0x80);
      //        printf("flag3, b1:%i b2:%i b3:%i b4:%i b5:%i b6:%i b7:%i b8:%i\n ",xyblock[num].flag3&0x1,xyblock[num].flag3&0x2,xyblock[num].flag3&0x4,xyblock[num].flag3&0x8,xyblock[num].flag3&0x10,xyblock[num].flag3&0x20,xyblock[num].flag3&0x40,xyblock[num].flag3&0x80);
      //        printf("flag4, b1:%i b2:%i b3:%i b4:%i b5:%i b6:%i b7:%i b8:%i\n ",xyblock[num].flag4&0x1,xyblock[num].flag4&0x2,xyblock[num].flag4&0x4,xyblock[num].flag4&0x8,xyblock[num].flag4&0x10,xyblock[num].flag4&0x20,xyblock[num].flag4&0x40,xyblock[num].flag4&0x80);
      }*/
    
    
    if (xyblock.type!=0)
    {
        if (xyblock.height==0) 
            xyblock.height++;
        if (xyblock.id!=1 && xyblock.basez<=oldz+MaxZstep)      // old code was +12
            //                                                          if(( !( xyblock[num].id == 1 )) && ( xyblock[num].basez <= oldz + 12 ))
        {
            if (xyblock.flag1&0x80 && !(chars[c].priv&1 || chars[c].dead))
            {
                blocked=1;
                printf("block 1\n");
            }
            else
            {
                if (gmbody)
                {
                    if ( (xyblock.weight==255 && !(chars[c].priv2&1)) || xyblock.type==2 )
                    {
                        if ( xyblock.weight==255 || xyblock.type==2 )
                        {
                            if (xyblock.flag2&0x04)
                            {
                                if (xyblock.basez<oldz+MaxZstep)                                // this was 3
                                    //                                                          if( xyblock[num].basez < oldz + 3 )
                                {
                                    z = xyblock.basez + xyblock.height;
                                    dispz = xyblock.basez + ( xyblock.height / 2 );
                                }
                                else
                                {
                                    if (chars[c].priv&1 || chars[c].dead || xyblock.flag4&0x20)
                                    {
                                        dispz=z=xyblock.basez;
                                    }
                                    else
                                    {
                                        blocked=1;
                                        printf("block 2\n");
                                    }
                                }
                            }
                            else
                            {
                                if (xyblock.basez+xyblock.height<oldz+MaxZstep) // this was 3
                                    //                                                          if( xyblock[num].basez + xyblock[num].height < oldz + 3 )
                                {
                                    dispz = z = xyblock.basez + xyblock.height;
                                }
                                else
                                {
                                    if ((chars[c].priv&1)||(chars[c].dead)||(xyblock.flag4&0x20))
                                    {
                                        dispz=z=xyblock.basez;
                                    }
                                    else
                                    {
                                        blocked=1;
                                        printf("block 3\n");
                                    }
                                }
                            }
                        }
                    }
                }//if gm body..
                else
                {
                    if ( xyblock.flag1&0x40 && !(chars[c].priv&1))
                    {
                        if (xyblock.basez+xyblock.height> oldz+MaxZstep ) // was if (xyblock.basez + xyblock.height > chars[c].z
                        {
                            blocked=1;
                            printf("block 4 %i+%i>%i\n", xyblock.basez, xyblock.height, chars[c].z );
                        }
                    }
                    else
                    {
                        if ((xyblock.flag2&4))
                        {
                            if (xyblock.basez<oldz+MaxZstep)                                    // used to be 3
                                //                                                                      if( xyblock[num].basez < oldz + 3 )
                            {
                                z=xyblock.basez+xyblock.height;
                                dispz=xyblock.basez+(xyblock.height/2);
                            }
                            else
                            {
                                if (chars[c].priv&1)
                                {
                                    dispz=z=xyblock.basez;
                                }
                                else
                                {
                                    blocked=1;
                                    printf("block 5\n");
                                }
                            }
                        }
                        else
                        {
                            if (xyblock.basez+xyblock.height<oldz+MaxZstep)     // used to be 3
                                //                                                                      if( xyblock[num].basez + xyblock[num].height < oldz + 3 )
                            {
                                dispz=z=xyblock.basez+xyblock.height;
                            }
                            else
                            {
                                if (chars[c].priv&1)
                                {
                                    dispz=z=xyblock.basez;
                                }
                                else
                                {
                                    if (xyblock.flag2&0x20)
                                    {
                                        blocked=1;
                                        printf("block 6\n");
                                    }
                                }//OMG whats going on?! Too many }s AHHHH!!!!!
                            }
                        }
                    }
                }
            }
        }
    }
    else
    {
        if ((xyblock.flag1&0x80 || xyblock.flag1&0x40) && !(chars[c].priv&1))
        {
            if(c==-1) blocked=1;
        }
        else
        {
            if (blocked || (xyblock.basez+xyblock.height<oldz+MaxZstep))                // was 3
                //                              if(( z ==  illegal_z ) || ( xyblock[num].basez + xyblock[num].height < oldz + 3 ))
            {
                dispz=z=xyblock.basez;
                // printf("xyblock Z: %i %i\n", xyblock.basez, xyblock.height );
            }
        }
    }
    if (blocked) 
        printf("returning xy->blocked!\n");
    /*else {
      if( xyblock.basez + xyblock.height < oldz + MaxZstep )
      chars[c].z = z = dispz = xyblock.basez + xyblock.height;
      }*/
    return blocked;
}


void cMovement::CombatWalk(int s) // Only for switching to combat mode
{
    for (unsigned int i=0;i<now;i++)
    {
		// moved perm[i] first since its much faster
        if ((perm[i]) && (inrange1p(s, currchar[i])))
        {
            extmove[1] = chars[s].ser1;
            extmove[2] = chars[s].ser2;
            extmove[3] = chars[s].ser3;
            extmove[4] = chars[s].ser4;
            extmove[5] = chars[s].id1;
            extmove[6] = chars[s].id2;
            extmove[7] = (unsigned char )(chars[s].x>>8);
            extmove[8] = (unsigned char )(chars[s].x%256);
            extmove[9] = (unsigned char )(chars[s].y>>8);
            extmove[10] = (unsigned char )(chars[s].y%256);
            extmove[11] = chars[s].dispz;
            extmove[12] = (unsigned char )(chars[s].dir&0x7F);
            
            extmove[13]=chars[s].skin1; //ripper, skin problems bugfix
            extmove[14]=chars[s].skin2;
            
            
            if (chars[s].war) extmove[15]=0x40; else extmove[15]=0x00;
            if (chars[s].hidden) extmove[15]=extmove[15]|0x80;
            if (chars[s].poisoned) extmove[15]=extmove[15]|0x04; //AntiChrist -- thnx to SpaceDog
            const int guild = Guilds->Compare( s, currchar[i] );
            const int race=Races->Compare( s, currchar[i] );
            if (chars[s].kills > repsys.maxkills ) extmove[16]=6; // ripper
            //if (chars[s].npcaitype==0x02) extmove[16]=6; else extmove[16]=1;
            //chars[i].flag=0x04;       // everyone should be blue on default
            else if (guild==1  || race==2)//Same guild (Green)
                extmove[16]=2;
            else if (guild==2 || race==1) // Enemy guild.. set to orange
                extmove[16]=5;
            //                  else if( !chars[i].npc && ( chars[i].priv&1 || chars[i].priv&80 ) )
            //                          extmove[16] = 7;
            else {
                switch(chars[s].flag)
                {//1=blue 2=green 5=orange 6=Red 7=Transparent(Like skin 66 77a)
                case 0x01: extmove[16]=6; break;// If a bad, show as red. 
                case 0x04: extmove[16]=1; break;// If a good, show as blue.
                case 0x08: extmove[16]=2; break; //green (guilds)
                case 0x10: extmove[16]=5; break;//orange (guilds)
                default:extmove[16]=3; break;//grey 
                }
            }
            
            if (!chars[s].war)
            {
                //                              chars[s].attacker=-1;
                chars[s].targ=-1;
            }
            Network->xSend(i, extmove, 17, 0);
        }
    }
}


void cMovement::NpcWalk(CHARACTER i, int j, int type)   //type is npcwalk mode (0 for normal, 1 for box, 2 for circle)
{
	// sometimes the NPC movement code comes up with -1, for example, if we are following someone
	// and we are directly on top of them
	if (-1 == j) return;

    const short int x = chars[i].x;
    const short int y = chars[i].y;
    const signed char z = chars[i].z;

    // if we are walking in an area, and the area is not properly defined, just don't bother with the area anymore
    if( ((1 == type) && ( chars[i].fx1 == -1 || chars[i].fx2 == -1 || chars[i].fy1 == -1 || chars[i].fy2 == -1 ) ) ||
        ((2 == type) && ( chars[i].fx1 == -1 || chars[i].fx2 == -1 || chars[i].fy1 == -1)))
        // circle's don't use fy2, so don't require them! fur 10/30/1999
    {
        //printf("Rect/circle error!\n" );
        chars[i].npcWander = 2; // Wander freely from now on
        type = 0;
    }

    //Bug Fix -- Zippy
    if (chars[i].priv2&2)
		return;//Frozen - Don't send them al the way to walking to check this, just do it here.
    
    if (chars[i].dir==j)  // If we're moving, not changing direction
    {
        switch(j) // Switch, based on the direction we're going to move.
        {
        case 0: // North
            if ((validNPCMove(x, y-1, z, i))&&(
                (!type)||
                ((type==1)&&(checkBoundingBox(x, y-1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2, chars[i].fy2)))||
                ((type==2)&&(checkBoundingCircle(x, y-1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2))))
                )
                Walking(i, j, 256); // arm code
            break;
        case 1: // Northeast
            if ((validNPCMove(x+1, y-1, z, i) && validNPCMove(x+1, y, z, i) && validNPCMove(x, y-1, z, i))&&(
                (!type)||
                ((type==1)&&(checkBoundingBox(x+1, y-1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2, chars[i].fy2)))||
                ((type==2)&&(checkBoundingCircle(x+1, y-1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2))))
                )
                Walking(i, j, 256); // arm code
            break;
        case 2: // East
            if ((validNPCMove(x+1, y, z, i))&&(
                (!type)||
                ((type==1)&&(checkBoundingBox(x+1, y, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2, chars[i].fy2)))||
                ((type==2)&&(checkBoundingCircle(x+1, y, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2))))
                )
                Walking(i, j, 256); // arm code
            break;
        case 3: // Southeast
            if ((validNPCMove(x+1, y+1, z, i) && validNPCMove(x+1, y, z, i) && validNPCMove(x, y+1, z, i))&&(
                (!type)||
                ((type==1)&&(checkBoundingBox(x+1, y+1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2, chars[i].fy2)))||
                ((type==2)&&(checkBoundingCircle(x+1, y+1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2))))
                )
                Walking(i, j, 256); // arm code
            break;
        case 4: // South
            if ((validNPCMove(x, y+1, z, i))&&(
                (!type)||
                ((type==1)&&(checkBoundingBox(x, y+1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2, chars[i].fy2)))||
                ((type==2)&&(checkBoundingCircle(x, y+1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2))))
                )
                Walking(i, j, 256); // arm code
            break;
        case 5: // Southwest
            if ((validNPCMove(x-1, y+1, z, i) && validNPCMove(x-1, y, z, i) && validNPCMove(x, y+1, z, i))&&(
                (!type)||
                ((type==1)&&(checkBoundingBox(x-1, y+1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2, chars[i].fy2)))||
                ((type==2)&&(checkBoundingCircle(x-1, y+1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2))))
                )
                Walking(i, j, 256); // arm code
            break;
        case 6: // West
            if ((validNPCMove(x-1, y, z, i))&&(
                (!type)||
                ((type==1)&&(checkBoundingBox(x-1, y, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2, chars[i].fy2)))||
                ((type==2)&&(checkBoundingCircle(x-1, y, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2))))
                )
                Walking(i, j, 256); // arm code
            break;
        case 7: // Northwest
            if ((validNPCMove(x-1, y-1, z, i) && validNPCMove(x-1, y, z, i) && validNPCMove(x, y-1, z, i))&&(
                (!type)||
                ((type==1)&&(checkBoundingBox(x-1, y-1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2, chars[i].fy2)))||
                ((type==2)&&(checkBoundingCircle(x-1, y-1, chars[i].fx1, chars[i].fy1, chars[i].fz1, chars[i].fx2))))
                )
                Walking(i, j, 256); // arm code
            break;
        default:
            printf("ERROR: Fallout of switch statement without default. walking.cpp, npcwalk()\n"); //Morrolan
            Walking(i, j, 256); 
        }
    }
    else if (j<8)
        Walking(i, j, 256);
}


//int cMovement::validNPCMove(int x, int y, int z, CHARACTER s)
int cMovement::validNPCMove(short int x, short int y, signed char z, CHARACTER s)
{
    const int getcell=mapRegions->GetCell(x,y);
    int mapitem=-1;
    int mapitemptr=-1;

    chars[s].blocked++;
    do 
    {
        mapitemptr=mapRegions->GetNextItem(getcell, mapitemptr);
        if (mapitemptr==-1) break;
        mapitem=mapRegions->GetItem(getcell, mapitemptr);
        if (mapitem!=-1 && mapitem<1000000)
        {
		    tile_st tile;
            Map->SeekTile((items[mapitem].id1<<8)+items[mapitem].id2, &tile);
            if (items[mapitem].x==x && items[mapitem].y==y && items[mapitem].z+tile.height>z+1 && items[mapitem].z<z+MaxZstep)
            {
                // bugfix found by JustMichael, moved by crackerjack
                // 8/2/99 makes code run faster too - one less loop :)
                if (items[mapitem].id1==0x39 && (items[mapitem].id2==0x46 || items[mapitem].id2==0x56)) return 0;
                if (items[mapitem].id1<=2 || (items[mapitem].id1==3 && items[mapitem].id2<=0xE2)) return 0;
                if (items[mapitem].id1==0x08 && (items[mapitem].id2>0x54 && items[mapitem].id2<0x66)) return 0;
                
                if (items[mapitem].type==12)
                {
                    if (chars[s].npc && (strlen(chars[s].title) > 0 || chars[s].npcaitype != 0))
                    {                            
                        // printf("doors!!!\n");
                        dooruse(-1,mapitem);
                        
                    }                                   
                    chars[s].blocked=0;
                    return 0;
                }
                
            }
        }
    } while (mapitem!=-1);

    // see if the map says its ok to move here
    if (Map->CanMonsterMoveHere(x, y, z))
      {
	chars[s].blocked = 0;
	return 1;
      }
    return 0;
}

unsigned short cMovement::GetYfromDir(int dir, unsigned short y)
{
    int x=0;
    switch(dir)
    {
    case '\x00' : y--;
        break;
    case '\x01' : { x++; y--; }
        break;
    case '\x02' : x++;
        break;
    case '\x03' : { x++; y++; }
        break;
    case '\x04' : y++;
        break;
    case '\x05' : { x--; y++; }
        break;
    case '\x06' : x--;
        break;
    case '\x07' : { x--; y--; }
        break;
    }
    return y;
}

unsigned short cMovement::GetXfromDir(int dir, unsigned short x)
{
    int y=0;
    switch(dir)
    {
    case '\x00' : y--;
        break;
    case '\x01' : { x++; y--; }
        break;
    case '\x02' : x++;
        break;
    case '\x03' : { x++; y++; }
        break;
    case '\x04' : y++;
        break;
    case '\x05' : { x--; y++; }
        break;
    case '\x06' : x--;
        break;
    case '\x07' : { x--; y--; }
        break;
    }
    return x;
}

void cMovement::PathFind(CHARACTER c, unsigned short gx, unsigned short gy)
{
    //This function attempts to find a path 2 tiles ahead of where 
    //the npc is to better walk etc....
    //Pass this function the character that is Walking, and the direction it should try to go.
    if (c<0 || c>=cmem) return;//Invalid character
    
    if (chars[c].pathnum<PATHNUM) return;//They aren't done with their current path yet.
    
    const signed char z=chars[c].z;
    const int no_char=cmem-1;
    
    //printf("Pathfinding: Character at %i,  %i\n",x,y);
    //first lets pikc a direction to try to go in, we'll try this one first, if its no good right away, we'll try the others.
    int d=chardirxyz(c,gx,gy,z);
#if 0
    if (d==1 || d==3 || d==5 || d==7)//the directions are BAD on corners.
    {
        //printf("Diag direction %i...",d);
        int i=rand()%2;//lets change their direction to they avoid the corner.
        if (i==1) d+=1;
        else d-=1;
        
        if (d>7) d-=8;
        else if (d<0) d+=8;
        //printf("Changed to %i [%i]\n",d,i);
    }
#endif
    short int x = GetXfromDir(d,chars[c].x);
    short int y = GetYfromDir(d,chars[c].y);
    
    //printf("    Trying (%i) %i, %i\n",d,x,y);
    
    if (!validNPCMove(x,y,z,c))
    {
        d+=2;//check clockwise 2 dirs
        if (d>7) d-=8;
        x=GetXfromDir(d,chars[c].x);
        y=GetYfromDir(d,chars[c].y);
        if (!validNPCMove(x,y,z,c))
        {
            d-=4;//check counter clockwise 2 dirs 
            if (d<0) d+=8;
            x=GetXfromDir(d,chars[c].x);
            y=GetYfromDir(d,chars[c].y);
            if (!validNPCMove(x,y,z,c))
            {
                d-=2;//check behind them LAST
                if (d<0) d+=8;
                x=GetXfromDir(d,chars[c].x);
                y=GetYfromDir(d,chars[c].y);
                if (!validNPCMove(x,y,z,c))
                {
                    printf("WARNING: Npc %i [%i] appears to be stuck! Pathfinding error!\n",c,chars[c].serial);
                    return;
                }
            }
        }
    }
    chars[c].path[0].x=x;
    chars[c].path[0].y=y;
    
    int i = d;//Lets prefer that we go the direction we are going.
    for (int a = 1; a < PATHNUM; a++)//Now lets find the rest of their path...
    {
        /*x1=x;
          y1=y;
          valid=0;
          i=chardirxyz(c,gx,gy,z);//try the direction of the target first
          
          if (i==1 || i==3 || i==5 || i==7)//the directions are BAD on corners.
          {
          //printf("2nd Diag direction %i...",i);
          b=rand()%2;//lets change their direction to they avoid the corner.
          if (b==1) i+=1;
          else i-=1;
          
          if (i>7) i-=8;
          else if (i<0) i+=8;
          //printf("Changed to %i [%i]\n",i,b);
          }*/
        
        short int x1 = GetXfromDir(i,x);
        short int y1 = GetYfromDir(i,y);
        if (!validNPCMove(x1,y1,z,c))
        {
            i+=2;
            if (i>7) i-=8;
            x1 = GetXfromDir(i, x);
            y1 = GetYfromDir(i, y);
            if (!validNPCMove(x1,y1,z,c))
            {
                i-=4;
                if (i<0) i+=8;
                x1=GetXfromDir(i, x);
                y1=GetYfromDir(i, y);
                if (!validNPCMove(x1,y1,z,c))
                {
                    i-=2;
                    if (i<0) i+=8;
                    x1=GetXfromDir(i, x);
                    y1=GetYfromDir(i, y);
                    if (!validNPCMove(x1,y1,z,c))
                    {
                        printf("Baadddddd error!\n");
                        return;
                    }
                }
            }
        }
        chars[c].path[a].x=x=x1;
        chars[c].path[a].y=y=y1;
        d=i;//the direction we are going so we keep going that direction.
    }
    chars[c].pathnum=0;
    //printf("Pathfinding dump:\n");
    //for (a=0;a<PATHNUM;a++)
    //  printf("    %i)  %i,  %i\n",a+1,chars[c].path[a].x,chars[c].path[a].y);
}


//NEW NPCMOVEMENT ZIPPY CODE STARTS HERE -- AntiChrist meging codes --
void cMovement::NpcMovement(unsigned int currenttime, int i)//Lag fix
{
    register int k;
    
    int j, x,l;
    
    int dnpctime=0;
    if (chars[i].npc && (chars[i].npcmovetime<=currenttime||(overflow)))
    {
        //kolours***************************************(09/19/98)
        //fix for monsters to wander when within player's sphere
        //of influence:
        //              if (chars[i].npcaitype&0x02) 
        //               {
        //                        if (!(chars[i].war)) chars[i].npcWander=2;
        //                }
        //kolours*************************************************
        //if (chars[i].war)
        if (chars[i].war && chars[i].npcWander != 5)
        {       // a very simple pathfinding algorithm
            l=chars[i].attacker;
            if (l!=-1)
            {
                if ((chardist(i, l)>1 || chardir(i, l)!=chars[i].dir))
                {
                    //                          if (online(calcSocketFromChar(l))||chars[l].npc)
                    if( online( l ) || chars[l].npc ) // LB bugkilling, was online( calcSocketFromChar(l) )
                    {
                        /*j=chardir(i, chars[i].attacker);
                          if (chars[i].blocked)
                          {
                          x=0;
                          if (j/2.0!=j/2) x=1;
                          //if (chars[i].blocked<=2) j =chars[i].dir2;
                          if (chars[i].blocked<=2) j=chars[i].dir2=(j-2-x)%8; //works better
                          else
                          {
                          if (rand()%2) j=chars[i].dir2=(j-2-x)%8;
                          else j=chars[i].dir2=(j+2+x)%8;
                          }
                          }
                          NpcWalk( i, j, 0 );*/
                        PathFind(i,chars[l].x,chars[l].y);
                        j=chardirxyz(i,chars[i].path[chars[i].pathnum].x,chars[i].path[chars[i].pathnum].y,chars[i].z);
                        chars[i].pathnum++;
                        Walking(i,j,256);
                    }
                }
            }
        } // end of if l!=-1
        else
        {
            switch(chars[i].npcWander)
            {
            case 0: // No movement
                break;
            case 1: // Follow the follow target
                k=(chars[i].ftarg);
                if (k < 0 || k>=(int)cmem) return;
                if (online(k) || chars[k].npc)
                {
                    if (chardist(i, k)>1 || chardir(i, k)!=chars[i].dir)
                    {
                        PathFind(i,chars[k].x,chars[k].y);
                        j=chardirxyz(i,chars[i].path[chars[i].pathnum].x,chars[i].path[chars[i].pathnum].y,chars[i].z);
                        chars[i].pathnum++;
                        //npcwalk(i,j,0);
                        Walking(i,j,256);
                    }
					// Dupois - Added April 4, 1999
					// Has the Escortee reached the destination ??
					if( ( chars[i].ftarg != -1 ) && ( !chars[k].dead ) && ( chars[i].questDestRegion == chars[i].region ) )
					{
						// Pay the Escortee and free the NPC
						MsgBoardQuestEscortArrive( i, calcSocketFromChar( k ) );
					}
					// End - Dupois
                }
                break;
            case 2: // Wander freely, avoiding obstacles.
                j=rand()%40;
                if (j<8 || j>32) dnpctime=5;
                if (j>7 && j<33) // Let's move in the same direction lots of the time.  Looks nicer.
                    j=chars[i].dir;
                NpcWalk(i,j,0);
                break;
            case 3: // Wander freely, within a defined box
                j=rand()%40;
                if (j<8 || j>32) dnpctime=5;
                if (j>7 && j<33) // Let's move in the same direction lots of the time.  Looks nicer.
                    j=chars[i].dir;
                
                NpcWalk(i,j,1);
                break;
            case 4: // Wander freely, within a defined circle
                j=rand()%40;
                if (j<8 || j>32) dnpctime=5;
                if (j>7 && j<33) // Let's move in the same direction lots of the time.  Looks nicer.
                    j=chars[i].dir;
                NpcWalk(i,j,2);
                break;
            case 5: //FLEE!!!!!!
            {
                k=chars[i].targ;
                //if (!(online(k)||chars[k].npc)) break;
                if (k<0 || (unsigned int)(k)>=cmem) return;
                if (chardist(i, k)<15)
                {
                    j=(chardir(i, k)+4)%8;
                    if (chars[i].blocked)
                    {
                        x=0;
                        if (j/2.0!=j/2) x=1;
                        if (chars[i].blocked<=2) j =chars[i].dir2;
                        else
                        {
                            if (rand()%2) j=chars[i].dir2=(j-2-x)%8;
                            else j=chars[i].dir2=(j+2+x)%8;
                        }
                    }
                    NpcWalk(i,j,0);
                }
                break;
            default:
                break;
                //printf("ERROR: Fallout of switch statement without default [%i]. walking.cpp, npcMovement2()\n",chars[i].npcWander); //Morrolan
            } // break; //Morrolan unnecessary ?
            }
        }
        chars[i].npcmovetime=(unsigned int)(currenttime+double(NPCSPEED*CLOCKS_PER_SEC)); //reset move timer
        //chars[i].npcmovetime=(unsigned int)(currenttime+double(NPCSPEED*CLOCKS_PER_SEC*(1+dnpctime))); //reset move timer
    }
}
