unit MoveText;

interface

uses
   Windows, SysUtils, Messages, Classes, Graphics, Controls, Forms,
   Dialogs, ExtCtrls;

type
  TAlignment = (taCenter, taLeftJustify, taRightJustify);
  TTextStyle = (tsNormal, tsRaised, tsLowered, tsShaddow);
  TMoveDirection = (mdStatic, mdRightToLeft, mdLeftToRight, mdTopToBottom, mdBottomToTop);

  TCustomMoveText = class(TGraphicControl)
  private
    FAlignment: TAlignment;
    FTextStyle: TTextStyle;
    FMoveDirection: TMoveDirection;
    FTimer: TTimer;
    FItems: TStringList;
    FColor: TColor;
    FContinuous: Boolean;
    FFont: TFont;
    FOnBegin, FOnStep, FOnEnd: TNotifyEvent;
    FSteps, FSpeed, FDepth, LineHi, FCurrentStep, FTextWidth,
      FTextHeight, XPos, YPos: Integer;
    procedure SetAlignment(Value: TAlignment);
    procedure SetContinuous(Value: Boolean);
    procedure SetItems(Value: TStringList);
    procedure DataChanged;
    procedure SetTextStyle(Value: TTextStyle);
    procedure SetDirection(Value: TMoveDirection);
    procedure SetSteps(Value: Integer);
    procedure SetSpeed(Value: Integer);
    procedure SetColor(Value: TColor);
    procedure SetFont(Value: TFont);
    procedure SetDepth(Value: Integer);
    procedure SetSizeParams;
    procedure FontChanged(Sender: TObject);
    procedure DoTextOut(ACanvas: TCanvas; X, Y: Integer; AText: string);
  protected
    procedure Paint; override;
    procedure TimerTick(Sender: TObject);
    property Alignment: TAlignment read FAlignment write SetAlignment;
    property Depth: Integer read FDepth write SetDepth default 1;
    property MoveDirection: TMoveDirection read FMoveDirection
      write SetDirection default mdRightToLeft;
    property Items: TStringList read FItems write SetItems;
    property OnBegin: TNotifyEvent read FOnBegin write FOnBegin;
    property OnStep: TNotifyEvent read FOnStep write FOnStep;
    property OnEnd: TNotifyEvent read FOnEnd write FOnEnd;
  public
    property CurrentStep: Integer read FCurrentStep;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ReverseDirection;
    procedure MoveStart(StartingStep: Integer);
    procedure MoveStop;
  published
    property TextStyle: TTextStyle read FTextStyle write SetTextStyle default tsNormal;
    property Steps: Integer read FSteps write SetSteps default 66;
    property Speed: Integer read FSpeed write SetSpeed default 200;
    property Color: TColor read FColor write SetColor default clBtnFace;
    property Continuous: Boolean read FContinuous write SetContinuous;
    property Font: TFont read FFont write SetFont;
  end;

  // TCustomMoveText  ޴ TMoveText ü 
  TMoveText = class(TCustomMoveText)
  published
    property Align;
    property Alignment;
    property Font;
    property MoveDirection;
    property ShowHint;
    property Speed;
    property Steps;
    property Visible;
    property OnBegin;
    property OnStep;
    property OnEnd;
    property Color;
    property Depth;
    property Items;
    property TextStyle;
    property ParentShowHint;
  end;

procedure Register;

implementation

procedure Register;
begin
   RegisterComponents('VCL', [TMoveText]);
end;

// Ʈü 
constructor TCustomMoveText.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle - [csOpaque];
   // ⺻  ؽƮ 
   FItems := TStringList.Create;
   FItems.Add('ؽƮ Դϴ.');

   Width := 200;
   Height := 20;
   FColor := clBtnFace;
   FSteps := 80;
   FCurrentStep := 0;
   FDepth := 1;
   FContinuous := True;
   FTextStyle := tsNormal;
   FAlignment := taCenter;
   // Ʈ  ⺻
   FFont := TFont.Create;
   with FFont do
   begin
      Name := 'ü';
      Size := 10;
      Color := clBlack;
   end;
   FFont.OnChange := FontChanged;
   // ؽƮ  Ÿ̸ ʱȭ
   FTimer := TTimer.Create(Self);
   FSpeed := 100;
   with FTimer do
   begin
      Enabled := False;
      OnTimer := TimerTick;
      Interval := FSpeed;
   end;

   FMoveDirection := mdRightToLeft;
   // ȭ󿡼 ؽƮ ̱ Ѵ.
   SetDirection(FMoveDirection);
end;

// Ʈ ü 
destructor TCustomMoveText.Destroy;
begin
   FItems.Free;
   FTimer.Free;
   FFont.Free;
   inherited Destroy;
end;

// ؽƮ  Ѵ.
procedure TCustomMoveText.SetItems(Value: TStringList);
begin
   if FItems <> Value then
   begin
      FItems.Assign(Value);
      DataChanged;
   end;
end;

// ȭ鿡 ؽƮ Ѵ.
procedure TCustomMoveText.DoTextOut(ACanvas: TCanvas; X, Y: Integer; AText: string);
var
   TextAdjustment: Integer;
begin
   with ACanvas do
   begin
      Font := FFont;
      Brush.Style := bsClear;
      // ؽƮ 
      if FAlignment = taCenter then
        TextAdjustment := Round((FTextWidth / 2) - (TextWidth(AText) / 2))
      else if FAlignment = taRightJustify then
        TextAdjustment := Round(FTextWidth - TextWidth(AText))
      else TextAdjustment := 0;
      //  ؽƮ Ÿ
      case FTextStyle of
  tsRaised: begin
       Font.Color := clBtnHighlight;
       //   ȿ Ÿ ؽƮ
       TextOut(X - FDepth + TextAdjustment, Y - FDepth, AText);
       Font.Color := clBtnShadow;
       //   ȿ Ÿ ؽƮ
       TextOut(X + FDepth + TextAdjustment, Y + FDepth, AText);
      end;
  tsLowered: begin
       Font.Color := clBtnShadow;
       //   ȿ Ÿ ؽƮ
       TextOut(X - FDepth + TextAdjustment, Y - FDepth, AText);
       Font.Color := clBtnHighlight;
       //   ȿ Ÿ ؽƮ
       TextOut(X + FDepth + TextAdjustment, Y + FDepth, AText);
      end;
  tsShaddow: begin
      Font.Color := clBtnShadow;
       //   ȿ Ÿ ؽƮ
      TextOut(X + FDepth + TextAdjustment, Y + FDepth, AText);
      end;
    end;
      Font.Color := FFont.Color;
       //   ؽƮ
     TextOut(X + TextAdjustment, Y, AText);
   end;
end;

// ؽƮ ׷ش.
procedure TCustomMoveText.Paint;
var
   TmpBmp: TBitMap;
   StartXPos, StartYPos, I: Integer;
   PercentDone: Double;
begin
   SetSizeParams;
   TmpBmp := TBitMap.Create;
   try
     TmpBmp.Width := Width;
     TmpBmp.Height := Height;
     with TmpBmp.Canvas do
     begin
        Font := FFont;
        Brush.Color := FColor;
        Brush.Style := bsSolid;
        FillRect(ClipRect);
     end;

     if FTextWidth >= Width then
       XPos := 0
     else
       XPos := (Width - FTextWidth) div 2;
     if FTextHeight >= Height then
       YPos := 0
     else
       YPos := (Height - FTextHeight) div 2;

     if csDesigning in ComponentState then
         PercentDone := 0.5
     else
         PercentDone := FCurrentStep / FSteps;
    // ؽƮ ̴  Ѵ.
    case FMoveDirection of
   mdRightToLeft: begin
        StartYPos := YPos;
        StartXPos := Round((FTextWidth + Width) *
                       (1 - PercentDone)) - FTextWidth;
      end;
   mdLeftToRight: begin
        StartYPos := YPos;
        StartXPos := Round((FTextWidth + Width) *
                        PercentDone) - FTextWidth;
      end;
   mdBottomToTop: begin
        StartXPos := XPos;
        StartYPos := Round((FTextHeight + Height) *
                        (1 - PercentDone)) - FTextHeight;
      end;
   mdTopToBottom: begin
        StartXPos := XPos;
        StartYPos := Round((FTextHeight + Height) *
                         PercentDone) - FTextHeight;
      end;
      else  // ؽƮ  
      begin
         StartXPos := XPos;
         StartYPos := YPos;
      end
     end;
     I := 0;
     // µ ؽƮ 
     while I < FItems.Count do
     begin
        DoTextOut(TmpBmp.Canvas, StartXPos, StartYPos,
                                   FItems.Strings[I]);
        Inc(StartYPos, LineHi);
        Inc(I);
     end;
     Canvas.Draw(0, 0, TmpBmp);
   finally
     TmpBmp.Free;
   end;
end;

// ؽƮ  ε 
procedure TCustomMoveText.SetSizeParams;
var
   S: String;
   I, SWidth: Integer;
   Metrics: TTextMetric;
begin
   // ؽƮ  
   with Canvas do
   begin
      Font := FFont;
      GetTextMetrics(Handle, Metrics);
      LineHi := Metrics.tmHeight + Metrics.tmInternalLeading;
      if FTextStyle in [tsRaised, tsLowered] then
        LineHi := LineHi + 2 * FDepth
      else if FTextStyle in [tsShaddow] then
        LineHi := LineHi + FDepth;
   end;
   FTextWidth := 0;
   I := 0;
   while I < FItems.Count do
   begin
      S := FItems.Strings[I];
      SWidth := Canvas.TextWidth(S);
     if FTextStyle in [tsRaised, tsLowered] then
        SWidth := SWidth + 2 * FDepth
     else if FTextStyle in [tsShaddow] then
        SWidth := SWidth + FDepth;
     if FTextWidth < SWidth then
     FTextWidth := SWidth;
     Inc(I);
   end;
   FTextHeight := LineHi * FItems.Count;
   if FTextWidth >= Width then
     XPos := 0
   else
     XPos := (Width - FTextWidth) div 2;
   if FTextHeight >= Height then
     YPos := 0
   else
     YPos := (Height - FTextHeight) div 2;
end;

// ؽƮ ϰΰ  ΰ Ѵ.
procedure TCustomMoveText.SetContinuous(Value: Boolean);
begin
   if FContinuous <> Value then
   begin
      FContinuous := Value;
      if FMoveDirection <> mdStatic then
      MoveStart(FCurrentStep);
   end;
end;

// ̴ ؽƮ ̵  
procedure TCustomMoveText.SetSteps(Value: Integer);
begin
   if FSteps <> Value then
   begin
      FSteps := Value;
      //  νÿ  ٽñ׸.
      if csDesigning in ComponentState then
      Invalidate;
   end;
end;

// ؽƮ ̵ӵ Ѵ.
procedure TCustomMoveText.SetSpeed(Value: Integer);
begin
   if FSpeed <> Value then
   begin
      if Value > 1000 then Value := 1000
      else if Value < 1 then Value := 1;
      FSpeed := Value;
      if FTimer <> nil then FTimer.Interval := FSpeed;
  end;
end;

//ؽƮ  Ѵ.
procedure TCustomMoveText.SetColor(Value: TColor);
begin
   if FColor <> Value then
   begin
      FColor := Value;
      DataChanged;
   end;
end;

// ؽƮ Ʈ Ǹ
procedure TCustomMoveText.FontChanged(Sender: TObject);
begin
   DataChanged;
end;

// ؽƮ Ʈ Ѵ.
procedure TCustomMoveText.SetFont(Value: TFont);
begin
   if FFont <> Value then
   begin
      FFont.Assign(Value);
      DataChanged;
   end;
end;

// ؽƮ  Ѵ.
procedure TCustomMoveText.MoveStop;
begin
   FTimer.Enabled := False;
end;

// ̴ ؽƮ  Ų.
procedure TCustomMoveText.ReverseDirection;
begin
   if FMoveDirection = mdStatic then Exit;
   FCurrentStep := FSteps - FCurrentStep;
   case FMoveDirection of
     mdLeftToRight: FMoveDirection := mdRightToLeft;
     mdRightToLeft: FMoveDirection := mdLeftToRight;
     mdTopToBottom: FMoveDirection := mdBottomToTop;
     mdBottomToTop: FMoveDirection := mdTopToBottom;
   end;
end;

// Ÿ̸Ӹ Ѵ.
procedure TCustomMoveText.TimerTick(Sender: TObject);
begin
   if not FTimer.Enabled then Exit;
   if (FCurrentStep = 0) and Assigned(FOnBegin) then
      FOnBegin(Self);
   Inc(FCurrentStep);
   Paint;
   if Assigned(FOnStep) then
   FOnStep(Self);
   if FCurrentStep > FSteps then
   begin
      FTimer.Enabled := False;
      if Assigned(FOnEnd) then
         FOnEnd(Self);
      FCurrentStep := 0;
      if FContinuous then
         MoveStart(FCurrentStep);
   end;
end;

//  ؽƮ  Ǹ
procedure TCustomMoveText.DataChanged;
begin
   SetSizeParams;
   Invalidate;
end;

// ؽƮ Ÿ Ѵ.
procedure TCustomMoveText.SetTextStyle(Value: TTextStyle);
begin
   if FTextStyle <> Value then
   begin
      FTextStyle := Value;
      DataChanged;
   end;
end;

// ؽƮ ̴ ġ Ѵ.
procedure TCustomMoveText.SetDirection(Value: TMoveDirection);
begin
   if FMoveDirection <> Value then
    FMoveDirection := Value;
   if FMoveDirection = mdStatic then
    MoveStop
   else
    MoveStart(FCurrentStep);
end;

// ؽƮ Ļ¸ Ѵ.
procedure TCustomMoveText.SetAlignment(Value: TAlignment);
begin
   if FAlignment <> Value then
   begin
      FAlignment := Value;
      DataChanged;
   end;
end;

// ؽƮ  Ѵ.
procedure TCustomMoveText.SetDepth(Value: Integer);
begin
   if FDepth <> Value then
   begin
      FDepth := Value;
      DataChanged;
   end;
end;

// ؽƮ ̵ Ѵ.
procedure TCustomMoveText.MoveStart(StartingStep: Integer);
begin
   if FTimer.Enabled then Exit;
   if (StartingStep >= 0) and (StartingStep <= FSteps) then
   FCurrentStep := StartingStep;
   FTimer.Enabled := True;
end;

end.
