unit TrayIcon;

interface

uses
  Windows, Messages, ShellApi, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, Menus;

const
  WM_CallbackMessage = WM_User + 56;

type
  TTrayIcon = class(TComponent)
  private
    FHandle: HWnd;
    FIconData: TNotifyIconData;
    FIcon: TIcon;
    FHint: string;
    FPopupMenu: TPopupMenu;
    FOnClick: TMouseEvent;
    FOnDblClick: TNotifyEvent;
    FOnHide: TNotifyEvent;
    FOnShow: TNotifyEvent;
    procedure SetHint(const Hint: string); virtual;
    procedure SetIcon(Icon: TIcon); virtual;
    procedure WndProc(var Message: TMessage);
  protected
    procedure DoMenu; virtual;
    procedure Click(Button: TMouseButton); virtual;
    procedure DblClick; virtual;
    procedure EndSession; virtual;
    procedure Changed; virtual;
  public
    constructor Create(Owner: TComponent); override;
    destructor Destroy; override;
    procedure Hide(Sender: TObject); virtual;
    procedure Show(Sender: TObject); virtual;
  published
    property Hint: string read FHint write SetHint;
    property Icon: TIcon read FIcon write SetIcon;
    property PopupMenu: TPopupMenu read FPopupMenu write FPopupMenu;
    property OnClick: TMouseEvent read FOnClick write FOnClick;
    property OnDblClick: TNotifyEvent read FOnDblClick write FOnDblClick;
    property OnHide: TNotifyEvent read FOnHide write FOnHide;
    property OnShow: TNotifyEvent read FOnShow write fOnShow;
  end;

procedure Register;

implementation

constructor TTrayIcon.Create(Owner: TComponent);
begin
  inherited Create(Owner);
  FIcon := TIcon.Create;
  FIcon.Assign(Application.Icon);
  FHandle := AllocateHwnd(WndProc);
  if not (csDesigning in ComponentState) then
  begin
    FillChar(FIconData, SizeOf(FIconData), 0);
    with FIconData do
    begin
      cbSize := SizeOf(FIconData);
      Wnd := FHandle;
      hIcon := Icon.Handle;
      uFlags := NIF_ICON or NIF_MESSAGE;
      uCallbackMessage := WM_CallbackMessage;
    end;
    StrPLCopy(FIconData.szTip, Application.Title, SizeOf(FIconData.szTip) - 1);
    if Application.Title <> '' then
      FIconData.uFlags := FIconData.uFlags or NIF_TIP;
    if not Shell_NotifyIcon(NIM_ADD, @FIconData) then
      raise EOutOfResources.Create('Ʈ    ϴ !');
    Application.OnMinimize := Hide;
    Application.OnRestore := Show;
  end;
end;

destructor TTrayIcon.Destroy;
begin
  FIcon.Free;
  if not (csDesigning in ComponentState) then
    Shell_NotifyIcon(NIM_DELETE, @FIconData);
  inherited Destroy;
end;

procedure TTrayIcon.Changed;
begin
  if not (csDesigning in ComponentState) then
    Shell_NotifyIcon(NIM_MODIFY, @FIconData);
end;

procedure TTrayIcon.WndProc(var Message: TMessage);
begin
  try
    with Message do
      case Msg of
        WM_QueryEndSession: Message.Result := 1;
        WM_EndSession:
          if TWmEndSession(Message).EndSession then EndSession;
        WM_CallbackMessage:
          case Message.lParam of
            WM_LBUTTONDBLCLK: DblClick;
            WM_LBUTTONUP: Click(mbLeft);
            WM_RBUTTONUP: Click(mbRight);
          end;
      else Result := DefWindowProc(FHandle, Msg, wParam, lParam);
    end;
  except
    Application.HandleException(Self);
  end;
end;

procedure TTrayIcon.SetHint(const Hint: string);
begin
  if FHint <> Hint then
  begin
    FHint := Hint;
    StrPLCopy(FIconData.szTip, Hint, SizeOf(FIconData.szTip) - 1);
    if Hint <> '' then
      FIconData.uFlags := FIconData.uFlags or NIF_TIP
    else
      FIconData.uFlags := FIconData.uFlags and not NIF_TIP;
    Changed;
  end;
end;

procedure TTrayIcon.SetIcon(Icon: TIcon);
begin
  if FIcon <> Icon then
  begin
    FIcon.Assign(Icon);
    FIconData.hIcon := Icon.Handle;
    Changed;
  end;
end;

procedure TTrayIcon.DoMenu;
var
  Pt: TPoint;
begin
  if (FPopupMenu <> nil) and not IsWindowVisible(Application.Handle) then
  begin
    GetCursorPos(Pt);
    FPopupMenu.Popup(Pt.X, Pt.Y);
  end;
end;

procedure TTrayIcon.Click(Button: TMouseButton);
var
  MousePos: TPoint;
begin
  GetCursorPos(MousePos);
  DoMenu;
//  if (Button = mbRight) then DoMenu;
  if Assigned(FOnClick) then FOnClick(Self, Button, [], MousePos.X, MousePos.Y);
end;

procedure TTrayIcon.DblClick;
begin
  Show(Self);
  if Assigned(FOnDblClick) then
    FOnDblClick(Self);
end;

procedure TTrayIcon.Hide(Sender: TObject);
begin
  ShowWindow(Application.Handle, SW_Hide);
  if Assigned(FOnHide) then FOnHide(Self);
end;

procedure TTrayIcon.Show(Sender: TObject);
begin
  ShowWindow(Application.Handle, SW_Restore);
  SetForegroundWindow(Application.Handle);
  if Assigned(FOnShow) then FOnShow(Self);
end;

procedure TTrayIcon.EndSession;
begin
  Shell_NotifyIcon(NIM_DELETE, @FIconData);
end;

procedure Register;
begin
  RegisterComponents('Ʈ', [TTrayIcon]);
end;

end.
